/* AGINBSUP.C -- Support code for INBOUND handling.
** By: Sanjay
** Start: 12, August, 1996
** Done: 30, August, 1996
*/

#include "rtrstd.h"

#include "kag.h"
#include "vagstr.h"

#include "agtx.h"
#include "agrx.h"
#include "agpkttyp.h"
#include "agline.h"
#include "agutil.h"
#include "aginbsup.h"
#include "agdebug.h"

#include <wanmgr.h>
#include <serial.h>


/* Local Prototypes */
STATIC enum BOOLEAN is_session_in_line_inbound_list(SESSION_TABLE_ENTRY *sptr_session_entry, LINE_INFO_ENTRY *sptr_line_info);


/* -- CODE ------------------------------------------------------------- */

/* allot_port_to_inbound_if_required()
**	Called when lower level determines that incoming request is not a PPP 
**	packet. We will check to see if anyone is waiting on inbound on this
**	port.If so, this port is alloted to the inbound module and we setup to
**	receive serial packets.
** Returns: TRUE if accepted the port, FALSE if did not.
*/
enum BOOLEAN allot_port_to_inbound_if_required(USHORT port_number)
{
	LINE_VARS_TYPE *sptr_line_vars;
	LINE_INFO_ENTRY *sptr_line_info;
	enum AG_PORT_AVAILABILITY port_status;

	if (ag.enabled == FALSE)
		return FALSE;

	sptr_line_info = &ag.line_info_array[port_number];
	if (sptr_line_info->number_of_inbound_users == 0)
		return FALSE;	

	if (ag.owned_by_module[port_number] != UNKNOWN_MODULE)
		return FALSE;

	port_status = ag_get_port_availability(port_number);
	if (port_status == AG_PORT_DISABLED || port_status == AG_PORT_NOT_ASYNC)
		return FALSE;

	set_wan_port_owner(port_number, OWNED_BY_AG);
	
	ag.fptr_msm_init_phase_callback[port_number] = NULL;

	sptr_line_vars = &sptr_line_info->line_vars;
	sptr_line_vars->current_baud_rate = sptr_line_vars->default_baud_rate;
	sptr_line_vars->current_data_bits = sptr_line_vars->default_data_bits;
	sptr_line_vars->current_parity = sptr_line_vars->default_parity;
	sptr_line_vars->current_stop_bits = sptr_line_vars->default_stop_bits;
	sptr_line_vars->current_max_packet_size = sptr_line_vars->default_max_packet_size;

	/* We cannot yet assoicate a session. So we use NULL now and later
	** associate a session when connection is fully established. Also
	** during the init phases of the inbound modem state machine, the tx
	** function will be different and the rx behaviour will be different
	*/
	if (setup_serial_port_for_use(port_number, NULL,
			serial_tx_complete, serial_rx_callback, &ag.fptr_rx_complete,
			sptr_line_info->line_vars.current_baud_rate,
			sptr_line_info->line_vars.current_data_bits,
			sptr_line_info->line_vars.current_parity,
			sptr_line_info->line_vars.current_stop_bits) == FAIL)
	{
#ifdef DEBUG
		printf("AG: Failed to init port %u for inbound call\n\r", port_number);
#endif /* DEBUG */

		set_wan_port_owner(port_number, OWNED_BY_NONE);
		return FALSE;
	}

	ag.owned_by_module[port_number] = INBOUND_MODULE;

	ag.line_info_array[port_number].inbound_init_on = TRUE;

	return TRUE;
}

/* reset_inbound_input_state()
** 	All guys who want input from serial during the inbound init phase
**	should call below function first.
*/
void reset_inbound_input_state(LINE_INFO_ENTRY *sptr_line_info, enum BOOLEAN echo_input)
{
	sptr_line_info->index_inbound_input_buffer = 0;
	sptr_line_info->inbound_input_valid = FALSE;
	sptr_line_info->echo_inbound_input = echo_input;
}

/* serial_input_during_inbound_init()
**	This function is called when serial input is got during the inbound
**	init phase. During this time, we usually get character by character
**	input. Two things are done here
**	(1) if input is printable, the input is echoed back.
**	(2) the input is put into a input buffer and when '\n' is encountered,
**		the input is treated as a line and placed into appropriate buffers
**		based on the current state of the inbound state machine.
*/
void serial_input_during_inbound_init(LINE_INFO_ENTRY *sptr_line_info, BYTE *sptr_rx_data, USHORT rx_data_size)
{
	char ch;
	USHORT i, input_length;
	char bell_char = '\007';
	char blank_space = ' ';
	char password_char = 'x';


  	/* Ignore keystrokes until previously read input is used */
	if (sptr_line_info->inbound_input_valid == TRUE)
		return;

	for (i = 0; i < rx_data_size; i++)
	{
		ch = *(sptr_rx_data + i);

		/* Backspace */
		if (ch == '\b')
		{
			if (sptr_line_info->index_inbound_input_buffer != 0)
			{
				sptr_line_info->index_inbound_input_buffer -= 1;
				write_serial_port((USHORT)sptr_line_info->line_id, &ch, 1);
				write_serial_port((USHORT)sptr_line_info->line_id, &blank_space, 1);
				write_serial_port((USHORT)sptr_line_info->line_id, &ch, 1);
			}
			else
				write_serial_port((USHORT)sptr_line_info->line_id, &bell_char, 1);
			
			continue;
		}

		/* Are we at end of line? */
		if (ch == '\n' || ch == '\r')
		{
			write_serial_port((USHORT)sptr_line_info->line_id, &ch, 1);

			/* Null terminate the input string */
			sptr_line_info->inbound_input_buffer[sptr_line_info->index_inbound_input_buffer] = '\0';

			input_length = sptr_line_info->index_inbound_input_buffer;
			sptr_line_info->index_inbound_input_buffer = 0;

			switch (sptr_line_info->next_inbound_state)
			{
			case IS_GET_NAME:
				/* Treat input as user name */
				if (input_length > USER_NAME_LENGTH)
					input_length = USER_NAME_LENGTH - 1;
				memcpy(sptr_line_info->inbound_user_name, sptr_line_info->inbound_input_buffer, input_length + 1);
				sptr_line_info->inbound_user_name[USER_NAME_LENGTH - 1] = '\0';
				break;
			case IS_GET_PASSWORD:
				/* Treat input as password */
				if (input_length > PASSWORD_LENGTH)
					input_length = PASSWORD_LENGTH - 1;
				memcpy(sptr_line_info->inbound_user_password, sptr_line_info->inbound_input_buffer, input_length + 1);
				sptr_line_info->inbound_user_password[PASSWORD_LENGTH - 1] = '\0';
				break;
			case IS_GET_PHONE_NUMBER:
				/* Treat input as phone number */
				if (input_length > CALLBACK_NUMBER_LENGTH)
					input_length = CALLBACK_NUMBER_LENGTH - 1;
				memcpy(sptr_line_info->inbound_user_callback_number, sptr_line_info->inbound_input_buffer, input_length + 1);
				sptr_line_info->inbound_user_callback_number[CALLBACK_NUMBER_LENGTH - 1] = '\0';
				break;
			case IS_GET_MENU_INPUT:
				/* In this case, input is in the build buffer itself */
				break;
			}
			/* In any other state, input is ignored */

			sptr_line_info->inbound_input_valid = TRUE;
			continue;
		}

		/* We have limited buffer space to read in the input...so take care */
		if (sptr_line_info->index_inbound_input_buffer == (MAX_INBOUND_INPUT - 1))
		{
			write_serial_port((USHORT)sptr_line_info->line_id, &bell_char, 1);
			continue;
		}

		/* Echo displayable keystrokes */
		if (isprint(ch))
		{
			/* In the IS_GET_MENU_INPUT state, we accept only numerics and
			** the keys 'U', 'D', 'P', 'N'. The latter are not echoed
			*/
			if (sptr_line_info->next_inbound_state == IS_GET_MENU_INPUT)
			{
				if (isdigit(ch))
				{
					write_serial_port((USHORT)sptr_line_info->line_id, &ch, 1);
					sptr_line_info->inbound_input_buffer[sptr_line_info->index_inbound_input_buffer] = ch;
					sptr_line_info->index_inbound_input_buffer += 1;
					continue;
				}

				ch = toupper(ch);
				if ((ch == 'U' || ch == 'D' || ch == 'P' || ch == 'N') &&
						sptr_line_info->index_inbound_input_buffer == 0)
				{
					switch (ch)
					{
					case 'U':
						if (sptr_line_info->top_host_index)
							sptr_line_info->top_host_index--;
						break;
					case 'D':
						if ((sptr_line_info->top_host_index + INBOUND_HOSTS_LIST_SIZE) < sptr_line_info->number_of_inbound_users)
							sptr_line_info->top_host_index++;
						break;
					case 'P':
						if (sptr_line_info->top_host_index < INBOUND_HOSTS_LIST_SIZE)
							sptr_line_info->top_host_index = 0;
						else
							sptr_line_info->top_host_index -= INBOUND_HOSTS_LIST_SIZE;
						break;
					case 'N':
						if ((sptr_line_info->top_host_index + INBOUND_HOSTS_LIST_SIZE) < sptr_line_info->number_of_inbound_users)
							sptr_line_info->top_host_index += INBOUND_HOSTS_LIST_SIZE;
						break;
					}

					sptr_line_info->next_inbound_state = IS_DISPLAY_HOSTS;
				}
				else
					write_serial_port((USHORT)sptr_line_info->line_id, &bell_char, 1);
				continue;
			}
			else if (sptr_line_info->next_inbound_state == IS_GET_NAME ||
					sptr_line_info->next_inbound_state == IS_GET_PASSWORD ||
					sptr_line_info->next_inbound_state == IS_GET_PHONE_NUMBER)
			{
				if (sptr_line_info->echo_inbound_input == FALSE)
					write_serial_port((USHORT)sptr_line_info->line_id, &password_char, 1);
				else
					write_serial_port((USHORT)sptr_line_info->line_id, &ch, 1);

				sptr_line_info->inbound_input_buffer[sptr_line_info->index_inbound_input_buffer] = ch;
				sptr_line_info->index_inbound_input_buffer += 1;
				continue;
			}
		}
	}
	return;
}

void add_inbound_session_to_line(LINE_INFO_ENTRY *sptr_line_info, SESSION_TABLE_ENTRY *sptr_session_entry_to_add)
{
	add_entry_to_list((LINK *)&sptr_line_info->sessions_awaiting_inbound, (LINK *)&sptr_session_entry_to_add->inbound_links);
	sptr_line_info->number_of_inbound_users++;
}

enum BOOLEAN delete_inbound_session_from_line(LINE_INFO_ENTRY *sptr_line_info, SESSION_TABLE_ENTRY *sptr_session_entry_to_delete)
{
	BYTE *sptr_session_link;
	SESSION_TABLE_ENTRY *sptr_session_entry;
	

	sptr_session_link = (BYTE *)get_pointer_to_first_entry_in_list((LINK *)&sptr_line_info->sessions_awaiting_inbound);
	while (sptr_session_link != NULL)
	{
		sptr_session_entry = (SESSION_TABLE_ENTRY *)(sptr_session_link - offsetof(SESSION_TABLE_ENTRY, inbound_links));
		if (sptr_session_entry == sptr_session_entry_to_delete)
		{
			/* This entry could possibly be the one we are looking for.
			** Verify this since the entry as such is dynamically allocated 
			** and deallocted during the uptime of the router.
			*/
			if (sptr_session_entry->session_ID == sptr_session_entry_to_delete->session_ID)
				break;
		}
		sptr_session_link = (BYTE *)get_pointer_to_next_entry_in_list((LINK *)sptr_session_link);
	}

	if (sptr_session_link == NULL)
	{
#ifdef DEBUG
		ag_printf(AG_ALARM_PRINTF, "AG: Freeing session not in inbound list WAN %u\n\r", (USHORT)sptr_line_info->line_id);
#endif /* DEBUG */
		return FALSE;
	}

	delete_entry_from_list((LINK *)&sptr_line_info->sessions_awaiting_inbound, (LINK *)sptr_session_link);

	sptr_line_info->number_of_inbound_users--;
	if (sptr_line_info->number_of_inbound_users == 0)
	{
		/* Since there are no more sessions awaiting inbound calls, we can
		** do some cleanup.
		*/

		delete_line_from_inbound_list(sptr_line_info);

		reset_inbound_state_machine_globals(sptr_line_info);
		sptr_line_info->connect_mode = sptr_line_info->line_vars.line_type;
	}

	return TRUE;
}

/* Following adds to a inbound list, but inbound list actually contains
** only lines that have requested an inbound call, but have not yet
** established one.
*/
void add_line_to_inbound_list(LINE_INFO_ENTRY *sptr_line_info_to_add)
{
	/* When deleting, the following does not necessarily hold true */
	sptr_line_info_to_add->connect_mode = INBOUND_MODEM;
	add_entry_to_list((LINK *)&ag.lines_awaiting_inbound, (LINK *)sptr_line_info_to_add);
}

enum BOOLEAN delete_line_from_inbound_list(LINE_INFO_ENTRY *sptr_line_info_to_delete)
{
	LINE_INFO_ENTRY *sptr_line_info;

	sptr_line_info = (LINE_INFO_ENTRY *)get_pointer_to_first_entry_in_list((LINK *)&ag.lines_awaiting_inbound);
	while (sptr_line_info != NULL && sptr_line_info != sptr_line_info_to_delete)
		sptr_line_info = (LINE_INFO_ENTRY *)get_pointer_to_next_entry_in_list((LINK *)sptr_line_info);

	if (sptr_line_info == NULL)
	{
#ifdef DEBUG
		ag_printf(AG_ALARM_PRINTF, "AG: Freeing line from inbound list though not in the list WAN %u\n\r", (USHORT)sptr_line_info->line_id);
#endif /* DEBUG */
		return FALSE;
	}

	delete_entry_from_list((LINK *)&ag.lines_awaiting_inbound, (LINK *)sptr_line_info);
	return TRUE;
}

SESSION_TABLE_ENTRY *find_matching_user_entry_in_line_inbound_list(LINE_INFO_ENTRY *sptr_line_info, BYTE *user_entry)
{
	BYTE *sptr_session_link;
	SESSION_TABLE_ENTRY *sptr_session_entry;


	if (user_entry == NULL)
		return NULL;

	sptr_session_link = (BYTE *)get_pointer_to_first_entry_in_list((LINK *)&sptr_line_info->sessions_awaiting_inbound);
	while (sptr_session_link != NULL)
	{
		sptr_session_entry = (SESSION_TABLE_ENTRY *)(sptr_session_link - offsetof(SESSION_TABLE_ENTRY, inbound_links));
		/* Here pointer comparison is sufficient perhaps since the user entry
		** will be constant throughout the uptime of the router.
		*/
		if (sptr_session_entry->user_entry == user_entry)
			break;
		sptr_session_link = (BYTE *)get_pointer_to_next_entry_in_list((LINK *)sptr_session_link);
	}

	if (sptr_session_link == NULL)
		return NULL;

	return sptr_session_entry;
}

void remove_session_from_all_inbound_lists(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	BYTE i;
	LINE_INFO_ENTRY *sptr_line_info;
	enum BOOLEAN session_in_atleast_one_list = FALSE;


	for (i = 0; i < ag.number_of_lines; i++)
	{
		sptr_line_info = &ag.line_info_array[i];

		if (delete_inbound_session_from_line(sptr_line_info, sptr_session_entry) == TRUE)
			session_in_atleast_one_list = TRUE;
	}

	if (session_in_atleast_one_list == TRUE)
		ag.stats.total_dialin_users--;
}


void reset_inbound_state_machine_globals(LINE_INFO_ENTRY *sptr_line_info)
{
	sptr_line_info->inbound_init_on = FALSE;
	sptr_line_info->next_inbound_state = IS_CHECK_RING;
	sptr_line_info->inbound_state_timer = 0;
	sptr_line_info->inbound_login_attempts = 0;
	sptr_line_info->menu_line_number = 0;

	sptr_line_info->inbound_input_valid = FALSE;
	sptr_line_info->index_inbound_input_buffer = 0;
	sptr_line_info->inbound_login_attempts = 0;

	sptr_line_info->callback_check_done = FALSE;
	sptr_line_info->callback_hangup_done = FALSE;
}

/* Below function verifies that this session is not in use during the init
** phase of an inbound call.
*/
enum BOOLEAN inbound_says_ok_to_cleanup(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	BYTE i;
	LINE_INFO_ENTRY *sptr_line_info;
	
	/* Basically, if the specified session is on the inbound list of any of
	** the lines, and if that line is in the inbound init phase, then delay
	** the session abortion until the end of the init phase. Otherwise,
	** inbound links could be left in states where CD is high and there are
	** no users.
	*/
	sptr_line_info = &ag.line_info_array[0];
	for (i = 0; i < ag.number_of_lines; i++, sptr_line_info++)
	{
		if (is_session_in_line_inbound_list(sptr_session_entry, sptr_line_info) == TRUE)
			if (sptr_line_info->inbound_init_on == TRUE)
				return FALSE;
	}

	return TRUE;		/* Not in inbound state machine */
}

STATIC enum BOOLEAN is_session_in_line_inbound_list(SESSION_TABLE_ENTRY *sptr_session_entry, LINE_INFO_ENTRY *sptr_line_info)
{
	BYTE *sptr_session_link;
	SESSION_TABLE_ENTRY *sptr_inbound_session_entry;

	sptr_session_link = (BYTE *)get_pointer_to_first_entry_in_list((LINK *)&sptr_line_info->sessions_awaiting_inbound);
	while (sptr_session_link != NULL)
	{
		sptr_inbound_session_entry = (SESSION_TABLE_ENTRY *)(sptr_session_link - offsetof(SESSION_TABLE_ENTRY, inbound_links));
		if (sptr_inbound_session_entry == sptr_session_entry)
		{
			/* This entry could possibly be the one we are looking for.
			** Verify this since the entry as such is dynamically allocated 
			** and deallocted during the uptime of the router.
			*/
			if (sptr_inbound_session_entry->session_ID == sptr_session_entry->session_ID)
				return TRUE;
		}
		sptr_session_link = (BYTE *)get_pointer_to_next_entry_in_list((LINK *)sptr_session_link);
	}

	return FALSE;
}

#if 0
/* Below function verifies that this session is not in use during the init
** phase of an inbound call.
*/
enum BOOLEAN inbound_says_ok_to_cleanup(SESSION_TABLE_ENTRY *sptr_session_entry)
{
	BYTE i;
	LINE_INFO_ENTRY *sptr_line_info;
	SESSION_TABLE_ENTRY *sptr_inbound_session_entry;
	
	
	sptr_line_info = &ag.line_info_array[0];
	for (i = 0; i < ag.number_of_lines; i++, sptr_line_info++)
	{
		if (sptr_line_info->inbound_init_on == FALSE)
			continue;

		sptr_inbound_session_entry = sptr_line_info->sptr_session_entry;
		if (sptr_inbound_session_entry == NULL)
			continue;
		if (sptr_inbound_session_entry->line_in_use != UNKNOWN_LINE)
			continue;
		if (sptr_inbound_session_entry->session_ID != sptr_session_entry->session_ID)
			continue;

		/* If above tests are not successful, the session entry is being used
		** in the inbound state machine. So wait until the state machine 
		** stops moving before deleting the session.
		*/
		return FALSE;		
	}

	return TRUE;		/* Not in inbound state machine */
}
#endif
/* -- END CODE -------------------------------------------------------- */

