/* AGLINE.C -- Code related to manipulating the LINE_INFO_ENTRY and
** related structures.
** By: Sanjay
** Start: 27, July, 1996
** Done: 30, August, 1996
*/

#include "rtrstd.h"

#include <uartmode.h>
#include <serial.h>
#include <msm.h>
#include <wanmgr.h>

#include "kag.h"
#include "vagstr.h"
#include "agpkttyp.h"
#include "agutil.h"
#include "agrx.h"
#include "agtx.h"
#include "agintern.h"
#include "aginbsup.h"
#include "agline.h"

/* Local Structures */
struct baud {
	ULONG baud_rate;
	USHORT baud_divisor;
};

/*Extern Prototype */
extern BYTE *get_a_network_send_packet(USHORT data_size);
extern void free_a_network_send_packet(BYTE *bptr);
extern enum TEST send_network_internal_packet(SESSION_TABLE_ENTRY *sptr_session_entry, UNION_AG_PACKET *sptr_ag_packet, USHORT data_size);
/*Extern Protoype */

/* Local Prototypes */
STATIC char *get_parity_string(BYTE parity);
STATIC ULONG get_baud_from_client_baud_divisor(USHORT baud_divisor);
STATIC USHORT get_client_baud_divisor_from_baud(ULONG baud_rate);
STATIC BYTE get_client_line_format(LINE_VARS_TYPE *sptr_line_vars);
STATIC void modem_init_complete(USHORT port_number, enum BOOLEAN init_successful);

/* Added by Naveen 09/07/1997 ... */
void reset_line_parameters(LINE_INFO_ENTRY *sptr_line_info);
/* ... Added by Naveen */



/* Local Data */
STATIC struct baud baud_rates[] = {
	{50, 2304}, {75, 1536}, {110, 1023}, {135, 853},
	{150, 768}, {300, 384}, {600, 192}, {1200, 96},
	{1800, 72},	{2000, 58}, {2400, 48}, {3600, 36},
	{4800, 24}, {7200, 18}, {9600, 12}, {19200, 6},
	{38400, 3}, {57600, 2}, {115200, 1}
};

/* -- CODE ------------------------------------------------------------- */

enum TEST initialize_line_information()
{
	USHORT i;
	LINE_VARS_TYPE *sptr_line_vars;
	LINE_INFO_ENTRY *sptr_line_info;


	ag.number_of_lines = lsl_control(GET_NUMBER_OF_WAN_PORTS);
	sptr_line_info = ag.line_info_array;

	for (i = 0; i < ag.number_of_lines; i++, sptr_line_info++)
	{
		sptr_line_vars = &sptr_line_info->line_vars;

		sptr_line_info->line_id = (BYTE)i;
		sptr_line_info->line_status = AGLS_LINE_FREE;
		sptr_line_info->cleanup = FALSE;
		sptr_line_info->sptr_session_entry = NULL;

		/* Init un-inited portions of the line parameters */
		sptr_line_vars->line_type = OUTBOUND_MODEM;
		sptr_line_vars->current_baud_rate = sptr_line_vars->default_baud_rate = get_wan_configured_baud_rate(i);
		sptr_line_vars->current_data_bits = sptr_line_vars->default_data_bits = EIGHT_DATA;
		sptr_line_vars->current_parity = sptr_line_vars->default_parity = NONE;
		sptr_line_vars->current_stop_bits = sptr_line_vars->default_stop_bits = ONE_STOP;
		sptr_line_vars->current_max_packet_size = sptr_line_vars->default_max_packet_size;

		/* Following is actually set in minutes via the NVRAM, we convert it
		** here to ticks.
		*/
		sptr_line_vars->idle_time = sptr_line_vars->idle_time * 60 * ag.clock_ticks_per_second;

/*		memset(sptr_line_statistics, 0, sizeof(LINE_INFO_STATS)); */
	}

	return PASS;
}

void change_line_parameters(BYTE line_id, CONNECT_VARS_TYPE *new_parameters)
{
	LINE_INFO_ENTRY *sptr_line_info;
	USHORT port_number = (USHORT) line_id;


	sptr_line_info = &ag.line_info_array[line_id];
	if (new_parameters->max_packet_size != 0xFFFF)
	{
		sptr_line_info->line_vars.current_max_packet_size = modify_serial_rx_buffer_size(port_number, new_parameters->max_packet_size);
		ag_printf(AG_LINE_PRINTF, "AG: Max packet size set to %u, port %u\n\r", sptr_line_info->line_vars.current_max_packet_size, port_number);
	}

	/* Ignore the inter packet delay value -- we don't use it */

	if (new_parameters->idle_time != 0xFFFF)
	{
		sptr_line_info->line_vars.idle_time = new_parameters->idle_time * 60 * ag.clock_ticks_per_second;
		ag_printf(AG_LINE_PRINTF, "AG: Idle time set to set to %u minutes, port %u\n\r", new_parameters->idle_time, port_number);
 	}

	/* Set serial parameters */
	set_baud_rate(line_id, new_parameters->line_parameters.baud_divisor);
	set_line_control(line_id, new_parameters->line_parameters.line_format);
	set_modem_control(line_id, new_parameters->line_parameters.modem_control_byte);

	/* Ignoring the interrupt mask byte */
}

void set_line_status(LINE_INFO_ENTRY *sptr_line_info, BYTE new_status)
{
	sptr_line_info->line_status = new_status;
}

#if 0

/*********** Following function replaced with a macro **************/

/* set_line_status()
**	Determines if the request change in line status is allowed or not. If it
**	is okay, then sets the status to desired status. Returns TRUE if status
**	is changed, else returns FALSE.
*/
enum BOOLEAN set_line_status(LINE_INFO_ENTRY *sptr_line_info, BYTE new_status)
{
	enum BOOLEAN return_value = FALSE;

	switch (sptr_line_info->line_status)
	{
	case AGLS_LINE_FREE:
		if (new_status != AGLS_CLEANING_UP && new_status != AGLS_CALL_COMING_IN)
			return_value = TRUE;
		break;
	case AGLS_LINE_BUSY:
		if (new_status == AGLS_CLEANING_UP)
			return_value = TRUE;
		break;
	case AGLS_BAD_INTERFACE:
		/* Not supported */
		break;
	case AGLS_BAD_LINE:
		if (new_status == AGLS_RESET_MODEM)
			return_value = TRUE;
		break;
	case AGLS_CLEANING_UP:
		if (new_status == AGLS_RESET_MODEM || new_status == AGLS_BAD_LINE ||
				new_status == AGLS_LINE_FREE)
			return_value = TRUE;
		break;
	case AGLS_RESET_MODEM:
		if (new_status == AGLS_BAD_LINE || new_status == AGLS_LINE_FREE || 
				new_status == AGLS_CALLING_BACK)
			return_value = TRUE;
		break;
	case AGLS_CONNECTING:
		if (new_status == AGLS_LINE_BUSY || new_status == AGLS_CLEANING_UP)
			return_value = TRUE;
		break;
	case AGLS_CALL_COMING_IN:
		if (new_status == AGLS_CONNECTING || new_status == AGLS_BAD_LINE || 
				new_status == AGLS_RESET_MODEM || 
				new_status == AGLS_CALLING_BACK || 
				new_status == AGLS_LINE_BUSY)
			return_value = TRUE;
		break;
	case AGLS_CALLING_BACK:
		if (new_status == AGLS_CONNECTING ||
				new_status == AGLS_BAD_LINE || new_status == AGLS_RESET_MODEM ||
				new_status == AGLS_LINE_BUSY)
			return_value = TRUE;
		break;
	}

	if (return_value == TRUE)
		sptr_line_info->line_status = new_status;
#ifdef DEBUG
	else
		DebugBreak();
#endif /* DEBUG */

	return return_value;
}

#endif /* 0 */

BYTE get_EIA_status(USHORT port_number)
{
	BYTE return_value = 0;

#define DSR_SET			0x01
#define CTS_SET			0x02
#define DCD_SET			0x04

	/* Since the box has no means of obtaining the state of the DSR pin,
	** we will return that DSR is set always. The CTS input pin in 
	** programmed such that we cannot read its status. So we return
	** 1 for this pin too.
	*/
	return_value |= DSR_SET;
	return_value |= CTS_SET;

	if (is_dcd_present(port_number)) return_value |= DCD_SET;

	return return_value;

#undef DSR_SET
#undef CTS_SET
#undef DCD_SET
}
/* Added by  Naveen and Sachin  to inform the client about the changes in
	CTS, DSR and CD pin ...*/
BYTE get_cts_status (USHORT port_number)
{
	return (1) ;
}

BYTE get_dsr_status (USHORT port_number)
{
	return (1) ;
}

BYTE get_dcd_status (USHORT port_number)
{
	return (is_dcd_present(port_number)) ;
}


BYTE make_modem_control_byte (BYTE cts, BYTE dsr, BYTE dcd)
{
#define DSR_SET			0x01
#define CTS_SET			0x02
#define DCD_SET			0x04

	BYTE return_value = 0 ;

	if (cts) return_value |= CTS_SET ;
	if (dsr) return_value |= DSR_SET ;
	if (dcd) return_value |= DCD_SET ;
	return return_value;

#undef DSR_SET
#undef CTS_SET
#undef DCD_SET
}
/* ... Naveen and Sachin 20/1/1997 */

void set_connect_vars_info(CONNECT_VARS_TYPE *sptr_connect_vars, LINE_VARS_TYPE *sptr_line_vars)
{
	fill_client_style_line_parameters_info(&sptr_connect_vars->line_parameters, sptr_line_vars);

	sptr_connect_vars->max_packet_size = sptr_line_vars->current_max_packet_size;
/* The following value '2' is assigned by Naveen.P.N. to fix the bug
   that was there when negotiating with WSN ...*/
	sptr_connect_vars->max_inter_packet_delay = sptr_line_vars->max_inter_packet_delay = 2;
/* ... Naveen 20/1/1997*/
	sptr_connect_vars->idle_time = sptr_line_vars->idle_time;
}

enum TEST setup_port_for_dialout(LINE_INFO_ENTRY *sptr_line_info)
{
	USHORT port_number;
	char *modem_init_strings[5];


	port_number = (USHORT)sptr_line_info->line_id;

	set_wan_port_owner(port_number, OWNED_BY_AG);

	ag.fptr_msm_init_phase_callback[port_number] = NULL;

	if (setup_serial_port_for_use(port_number, 
			sptr_line_info->sptr_session_entry, serial_tx_complete, 
			serial_rx_callback, &ag.fptr_rx_complete, 
			sptr_line_info->line_vars.current_baud_rate, 
			sptr_line_info->line_vars.current_data_bits, 
			sptr_line_info->line_vars.current_parity, 
			sptr_line_info->line_vars.current_stop_bits) == FAIL)
	{
#ifdef DEBUG
		printf("AG: Failed to init port %u for dialout\n\r", port_number);
#endif /* DEBUG */

		set_wan_port_owner(port_number, OWNED_BY_NONE);
		return FAIL;
	}

	ag.owned_by_module[port_number] = OUTBOUND_MODULE;

/*  Following condition is commented by Naveen 09/07/1997 
    this because it is not required to reinitialize modem when  we
    establish new connection , but it is required to initialize 
    when we disconnect which is done anyway ... */

	if (/*is_wan_direct_connect(port_number) == FALSE*/ FALSE)
	{
		get_wan_modem_init_strings(port_number, modem_init_strings);

		ag.fptr_msm_init_phase_callback[port_number] = 
				start_modem_init_state_machine(port_number,
				modem_init_strings, (void (*)(USHORT, BYTE *, USHORT))write_serial_port, modem_init_complete);
	}
	else
	{
		modem_init_complete(port_number, TRUE);
	}

	return PASS;
}

/* set_line_control()
**	This function is used to simluate the behaviour of setting the
**	line control register on a 8250 or compatible UART. The 'control_value'
**	is a bit pattern specific to setting the parity, stop bit and data
**	bit settings using the line control register. We map these patters
**	to settings suitable for the QUICC SCC's.
**  NOTE: We will ignore BREAK and DLAB settings as they are unsuitable to
**	be mapped to the QUICC (maybe we can support break but again perhaps
**	MCSI itself does not use these bits).
*/
void set_line_control(USHORT port_number, BYTE control_value)
{
	BYTE data_bits, parity, stop_bits;

	/* See the line control register bit settings to understand the following */
	
	switch (control_value & 0x03)
	{
	case 0:
		data_bits = FIVE_DATA;
		break;
	case 1:
		data_bits = SIX_DATA;
		break;
	case 2:
		data_bits = SEVEN_DATA;
		break;
	case 3:
	default:
		data_bits = EIGHT_DATA;
		break;
	}

	switch (control_value & 0x04)
	{
	default:
	case 0:
		stop_bits = ONE_STOP;
		break;
	case 4:
		stop_bits = TWO_STOP;
		break;
	}

	switch (control_value & 0x18)
	{
	default:
	case 0x00:
	case 0x10:
		parity = NONE;
		break;
	case 0x08:
		parity = ODD;
		break;
	case 0x18:
		parity = EVEN;
		break;
	}

	ag_printf(AG_LINE_PRINTF, "AG: WAN %u: Serial parameters changed to %d data bits, %s parity, %d stop bit(s)\n\r", 
		port_number, ((int)data_bits), get_parity_string(parity), ((int)stop_bits));

	modify_serial_parameters(port_number, data_bits, parity, stop_bits);

	ag.line_info_array[port_number].line_vars.current_data_bits = data_bits;
	ag.line_info_array[port_number].line_vars.current_parity = parity;
	ag.line_info_array[port_number].line_vars.current_stop_bits = stop_bits;
}

STATIC char *get_parity_string(BYTE parity)
{
	switch (parity)
	{
	case ODD:
		return "ODD";
	case EVEN:
		return "EVEN";
	case NONE:
		return "NO";
	}
}

/* set_modem_control()
**	This function is called to control the DTR and RTS signals. The
**  control value will have bits 0 and bit 1 with following information
**	BIT 0 = 1 Set DTR (turn on)
**		  = 0 Reset DTR (turn off)
**	BIT 1 = 1 Set RTS (turn on)
**		  = 0 Reset RTS (turn off)
*/
void set_modem_control(USHORT port_number, BYTE control_value)
{
	if (control_value & 0x01)
	{
		ag_printf(AG_LINE_PRINTF, "AG: WAN %u: Asserting DTR\n\r", port_number);
		set_serial_dtr(port_number);
	}
	else
	{
		ag_printf(AG_LINE_PRINTF, "AG: WAN %u: De-asserting DTR\n\r", port_number);
		reset_serial_dtr(port_number);
	}

	if (control_value & 0x02)
	{
		ag_printf(AG_LINE_PRINTF, "AG: WAN %u: Asserting RTS\n\r", port_number);
		set_serial_rts(port_number);
	}
	else
	{
		ag_printf(AG_LINE_PRINTF, "AG: WAN %u: De-asserting RTS\n\r", port_number);
		reset_serial_rts(port_number);
	}
}

/* set_baud_rate()
**	Called to modify baud rate on the fly. The baud_divisor parameter is
**	as at a MCSI client on a PC where a UART (8250, etc) is being programmed.
**	The divisor for the QUICC is computed differently so we do some
**	translation.
*/
void set_baud_rate(USHORT port_number, USHORT baud_divisor)
{										  
	ULONG baud_rate;

	baud_rate = get_baud_from_client_baud_divisor(baud_divisor);

	ag_printf(AG_LINE_PRINTF, "AG: WAN %u: Resetting baud rate to %lu baud\n\r", port_number, baud_rate);

	modify_serial_baud_rate(port_number, baud_rate);

	ag.line_info_array[port_number].line_vars.current_baud_rate = baud_rate;
}

/* get_baud_from_client_baud_divisor()
**	The baud divisors used by client's are as per PC programming
**	standards. We need to convert to QUICC baud divisors. So we get the
**	baud rate back from the divisor and pass this as a parameter to our
**	serial driver.
*/
STATIC ULONG get_baud_from_client_baud_divisor(USHORT baud_divisor)
{
	int i;


	for (i = 0; i < sizeof(baud_rates) / sizeof(struct baud); i++)
	{
		if (baud_divisor == baud_rates[i].baud_divisor)
			return baud_rates[i].baud_rate;
	}

	ag_printf(AG_ALARM_PRINTF, "AG: Unsupported baud rate request, using 19200\n\r");
	return 19200L;	/* Default to 19200 */
}

void fill_client_style_line_parameters_info(LINE_PARAM_TYPE *sptr_line_parameters, LINE_VARS_TYPE *sptr_line_vars)
{
	sptr_line_parameters->baud_divisor = get_client_baud_divisor_from_baud(sptr_line_vars->current_baud_rate);
	sptr_line_parameters->line_format = get_client_line_format(sptr_line_vars);
	sptr_line_parameters->modem_control_byte = 0;
	sptr_line_parameters->int_mask_byte = 0;
}

USHORT get_client_baud_divisor_from_baud(ULONG baud_rate)
{
	int i;


	for (i = 0; i < sizeof(baud_rates) / sizeof(struct baud); i++)
	{
		if (baud_rate == baud_rates[i].baud_rate)
			return baud_rates[i].baud_divisor;
	}

	ag_printf(AG_ALARM_PRINTF, "AG: Unsupported baud rate, assuming 19200\n\r");

	return 6;		/* Corresponds to 19,200 baud */
}

STATIC BYTE get_client_line_format(LINE_VARS_TYPE *sptr_line_vars)
{
	BYTE line_format = 0;


	switch (sptr_line_vars->current_data_bits)
	{
	case FIVE_DATA:
		line_format |= 0x00;
		break;
	case SIX_DATA:
		line_format |= 0x01;
		break;
	case SEVEN_DATA:
		line_format |= 0x02;
		break;
	case EIGHT_DATA:
		line_format |= 0x03;
		break;
	}

	switch (sptr_line_vars->current_parity)
	{
	case ODD:
		line_format |= 0x08;
		break;
	case EVEN:
		line_format |= 0x18;
		break;
	case NONE:
		line_format |= 0x00;
		break;
	}

	switch (sptr_line_vars->current_stop_bits)
	{
	case ONE_STOP:
		line_format |= 0x00;
		break;
	case TWO_STOP:
		line_format |= 0x04;
		break;
	}

	return line_format;
}

/* freeup_line()
**	This call is made whenever internally we decide to stop the usage of
**	a line by any session. If the reason for freeing up the line is that
**	CD dropped on an inbound call, then only the line if freed, else the
**	network connection is terminated. But all this happens only later after
**	we send a "disconnect" packet to the client end and the disconnect
**	call has posted.
*/
void freeup_line(LINE_INFO_ENTRY *sptr_line_info, BYTE free_reason)
{
	SESSION_TABLE_ENTRY *sptr_session_entry;


	ag_printf(AG_LINE_PRINTF, "AG: Internal disconnect of line %u\n\r", (USHORT) sptr_line_info->line_id);

	sptr_session_entry = sptr_line_info->sptr_session_entry;
	if (sptr_session_entry == NULL || sptr_session_entry->session_status == AG_SESS_ABORTED)
		return;

	send_disconnect_packet(sptr_session_entry, sptr_line_info, free_reason);
	sptr_line_info->line_status = AGLS_CLEANING_UP;
}

STATIC void modem_init_complete(USHORT port_number, enum BOOLEAN init_successful)
{
	LINE_INFO_ENTRY *sptr_line_info;

/* Naveen 05/11/1996 */
   sptr_line_info = &ag.line_info_array[port_number];
   if (sptr_line_info->line_status == AGLS_CONNECTING)
   {
	   modify_serial_rx_buffer_size(port_number, sptr_line_info->line_vars.current_max_packet_size);

      set_line_status(sptr_line_info, AGLS_LINE_BUSY);

	   ag_printf(AG_LINE_PRINTF, "AG: WAN %u is ready for use\n\r", port_number);
	   ag.fptr_msm_init_phase_callback[port_number] = NULL;
   }

/* Naveen 05/11/1996 */
}

void modem_hangup_complete(USHORT port_number)
{
	LINE_INFO_ENTRY *sptr_line_info;

	free_serial_port_from_use(port_number);
	set_wan_port_owner(port_number, OWNED_BY_NONE);
	ag.owned_by_module[port_number] = UNKNOWN_MODULE;

	sptr_line_info = &ag.line_info_array[port_number];

	if (sptr_line_info->number_of_inbound_users != 0)
		add_line_to_inbound_list(sptr_line_info);

	ag.stats.number_of_connections--;
	if (sptr_line_info->connect_mode == INBOUND_MODEM)
		ag.stats.inbound_connections--;
	else
		ag.stats.outbound_connections--;

	set_line_status(sptr_line_info, AGLS_LINE_FREE);
	reset_line_current_statistics(sptr_line_info);
   reset_line_parameters(sptr_line_info);
	sptr_line_info->sptr_session_entry = NULL;

	ag_printf(AG_LINE_PRINTF, "AG: WAN %u is again available for use\n\r", port_number);
}


/* Added By Naveen 09/07/1997 ... */
void reset_line_parameters(LINE_INFO_ENTRY *sptr_line_info)
{
	LINE_VARS_TYPE *sptr_line_vars;

	sptr_line_vars = &sptr_line_info->line_vars;
	sptr_line_vars->current_baud_rate = sptr_line_vars->default_baud_rate;
	sptr_line_vars->current_data_bits = sptr_line_vars->default_data_bits;
	sptr_line_vars->current_parity = sptr_line_vars->default_parity;
	sptr_line_vars->current_stop_bits = sptr_line_vars->default_stop_bits;
	sptr_line_vars->current_max_packet_size = sptr_line_vars->default_max_packet_size;
}
/* ... Added By Naveen 09/07/1997 */


/* Added by Naveen and Sachin to update  Status of CTS,DSR and CD pin...*/
BYTE old_modem_control_byte[MAX_WAN_PORTS]  = {0,0,0};
void update_mcsi_clients_with_line_status ()
{
#define DCD_SET			0x04
	USHORT i;
	BYTE modem_control_byte ;
	USHORT data_size;
	UNION_AG_PACKET *sptr_ag_packet;
	SESSION_TABLE_ENTRY *sptr_session_entry = NULL;
	USHORT port_number ;

	for (i = 0 ; i < ag.number_of_lines ; i++)
	{
		port_number = ag.line_info_array[i].line_id ;
		if (ag.line_info_array[port_number].line_status != AGLS_LINE_BUSY)
			continue ;

		modem_control_byte = make_modem_control_byte (get_cts_status(port_number), get_dsr_status(port_number), get_dcd_status(port_number)) ;
		if (modem_control_byte != old_modem_control_byte[port_number])
		{
			/* Send a control packet to the MCSI or WSN client */

			sptr_session_entry = ag.line_info_array[port_number].sptr_session_entry ;
/* Sachin 25/01/1997 */
			if (sptr_session_entry == NULL)
			{
				return ;
			}
/* Sachin 25/01/1997 */
			data_size = sizeof(BYTE) + sizeof(CONTROL_TYPE);
			sptr_ag_packet = (UNION_AG_PACKET *)get_a_network_send_packet(data_size);
			if (sptr_ag_packet == NULL)
			{
				return ;
			}
	
			sptr_ag_packet->packet_class = AG_CONTROL_PACKET;
			if ((DCD_SET & modem_control_byte) == (DCD_SET & old_modem_control_byte[port_number]))
			{
				sptr_ag_packet->packet_type.control_packet.control_source = AGCS_SET_MODEM_CONTROL;
				sptr_ag_packet->packet_type.control_packet.control_value  = modem_control_byte;
			}
			else
			{
				sptr_ag_packet->packet_type.control_packet.control_source = AGCS_SET_DCD;
				sptr_ag_packet->packet_type.control_packet.control_value  = (modem_control_byte & DCD_SET) ? DCD_SET : 0 ;
			}
			sptr_ag_packet->packet_type.control_packet.channel_number = port_number;

			old_modem_control_byte[port_number] = modem_control_byte ;
	
			change_endian(sptr_ag_packet);
			if (send_network_internal_packet(sptr_session_entry, sptr_ag_packet, data_size) == FAIL)
			{
				free_a_network_send_packet((BYTE *)sptr_ag_packet);
				return ;
			}
		}
	}
#undef DCD_SET			
}
/* ... Naveen and Sachin 20/1/1997 */

/* -- END CODE -------------------------------------------------------- */

