/*--------------------------------------------------------------------------*\

**	Description	: Virtual Server LoopBack Problem Handling File
**	Author		: Sudha
**	Start Date	: 28-Jan-2000	
** End Date		: 07-Feb-2000
**	File Name	: vserlpbk.c
**	Warning		: 
**	Notes			: Most of the functions in this file are duplication of proxy.c
					: file functions. This is for experimentation only. Later we 
					: can optimize it by having a flag in the resp functions in 
					: proxy.c to distinguish between actual proxying & loopback 
					: proxying. Based on the flag value, appropriate flow can be
					: followed.
**	References	:
**	Rev History : 	
		
\*--------------------------------------------------------------------------*/

/* System Includes */

#include <defs.h>
#include <stdio.h>
#include <stdarg.h>
#include <stddef.h>
#include <stdlib.h>
#include <kstart.h>
#include <string.h>
#include <memory.h>
#include <redblack.h>

/* Local Includes */

#include "proxy.h"
#include "vpptpstr.h"
#include "proxyrb.h"
#include "lpbkip.h"
#include "gserlpbk.h" /* extern declarations */

#ifdef __VIRTUAL_SERVER_LOOPBACK__

/* Constants */

/* Global Variables */

/* Static Variables */

/* Function Declarations */

BYTE *check_for_ftp_port_command_in_loopback_request_and_process (UNION_IP_PACKET *uptr_ip_rx_packet, 
	enum BOOLEAN *return_value, IP_PARAMETERS *sptr_ip_parameters, PROXY_SERVER_INFO *sptr_control_connection);
PROXY_SERVER_INFO *get_free_virtual_server_loopback_info_ptr (USHORT original_source_port, 
	BYTE *uptr_rx_packet, IP_PARAMETERS *sptr_ip_parameters, 
	enum BOOLEAN *eptr_ftp_port_command, ULONG local_destination_address);
void process_virtual_server_loopback_request_packet (UNION_IP_PACKET *uptr_ip_rx_packet, 
	IP_PARAMETERS *sptr_ip_parameters, BYTE *proxy_server_info, enum BOOLEAN *ftp_port_command);
enum BOOLEAN check_and_process_if_virtual_server_loopback_request_packet (UNION_IP_PACKET *uptr_ip_rx_packet, 
	IP_PARAMETERS *sptr_ip_parameters, enum BOOLEAN *eptr_size_modified);
void process_the_loopback_response_ftp_packet_after_pasv_commd (UNION_IP_PACKET *uptr_ip_rx_packet, 
	IP_PARAMETERS *sptr_ip_parameters, PROXY_SERVER_INFO *server_descriptor_info, enum BOOLEAN *ftp_pasv_mode);
void process_virtual_server_loopback_response_packet (UNION_IP_PACKET *uptr_ip_rx_packet, 
	IP_PARAMETERS *sptr_ip_parameters, BYTE *proxy_server_info, enum BOOLEAN *ftp_pasv_mode);
enum BOOLEAN is_virtual_server_loopback_response_packet (BYTE *uptr_ip_rx_packet, 
	IP_PARAMETERS *sptr_ip_parameters, enum BOOLEAN *ftp_pasv_mode); 

/*--------------------------------------------------------------------------*\

**	Function Name	: check_for_ftp_port_command_in_loopback_request_and_process()
**	Description 	: If ftp PORT command in virtual server loopback request pkt, 
						: creating client & server descriptor for data connection &
						: associating control connection with this data connection.
**	Parameters		: 1. ptr to the recvd IP packet.
						: 2. ptr to return_value which is of enum BOOLEAN type. This is 
						: 		set to TRUE if ftp PORT command, else FALSE.
						: 3. ptr to ip parameters.
						: 4. ptr to ftp control connection data descriptor.
**	Return Value	: if ftp PORT command, pointer to the addr of newly created 
						: descriptor for data connection.
						: else NULL.
**	Notes				:

\*--------------------------------------------------------------------------*/

BYTE *check_for_ftp_port_command_in_loopback_request_and_process (UNION_IP_PACKET *uptr_ip_rx_packet, 
	enum BOOLEAN *return_value, IP_PARAMETERS *sptr_ip_parameters, PROXY_SERVER_INFO *sptr_control_connection)
{
	BYTE temp_buff[6];
	BYTE *ptr_to_tcp_pkt = NULL;
	USHORT source_port=0;
	ULONG source_ip_address;
	PROXY_SERVER_INFO *ptr_to_proxy_server_info = NULL;
	USHORT ip_total_length = sptr_ip_parameters->total_length;
	USHORT port_command_length, ip_header_length = sptr_ip_parameters->header_length ;
	PROXY_RB_CLIENT_KEY_TYPE client_descriptor_key;
	PROXY_RB_SERVER_KEY_TYPE server_descriptor_key;
	
	ptr_to_tcp_pkt = get_ptr_to_tcp_packet ((BYTE *) uptr_ip_rx_packet, sptr_ip_parameters);
	if ((port_command_length = get_port_command_length (ptr_to_tcp_pkt)) > 35)
	{
		/* It can never be greater */
		*return_value = FALSE;
		return NULL;
	}

	if (port_command_length + ip_header_length + sizeof (TCP_HEADER) != ip_total_length)
	{
		*return_value = FALSE;
		return NULL;
	}

	memcpy (temp_buff,ptr_to_tcp_pkt,4);
	temp_buff[4] = 0;
	if (strcmpi (temp_buff, "Port"))
	{
		*return_value = FALSE;
		return NULL;
	}

	ptr_to_proxy_server_info = (PROXY_SERVER_INFO *) calloc (sizeof (PROXY_SERVER_INFO), 1);
	if (ptr_to_proxy_server_info == NULL)
	{
		proxy_printf ("PROXY_SERVER: Not Enough memory to proxy, Dropping packet\n");
		*return_value = FALSE; 
		return NULL;
	}

	ptr_to_proxy_server_info->outgoing_physical_port =	sptr_control_connection->outgoing_physical_port;

	ptr_to_proxy_server_info->destination_address = sptr_ip_parameters->destination_address;
	ptr_to_proxy_server_info->outgoing_port_address =  get_ip_address (get_secured_lan_port_number());

	ptr_to_proxy_server_info->proxy_idle_timer = 600; /* 600 Secs */
	ptr_to_proxy_server_info->protocol_type = sptr_ip_parameters->protocol;
	ptr_to_proxy_server_info->seq_offset = 0;

	ptr_to_proxy_server_info->ftp_port_command_on = FALSE;
	ptr_to_proxy_server_info->ftp_pasv_mode_on = FALSE;
	ptr_to_proxy_server_info->tftp_packet_detected = FALSE;
	
	ptr_to_proxy_server_info->in_use = TRUE;
	ptr_to_proxy_server_info->connection_closing = FALSE;
	
	ptr_to_proxy_server_info->client_node = NULL;
	ptr_to_proxy_server_info->server_node = NULL;		

	ptr_to_proxy_server_info->outgoing_destination_address = sptr_control_connection->outgoing_destination_address;

	get_ftp_src_port_and_address (ptr_to_tcp_pkt, &source_port, &source_ip_address);

	ptr_to_proxy_server_info->client_ip_address = source_ip_address;
	ptr_to_proxy_server_info->source_port = source_port;
	ptr_to_proxy_server_info->mapped_port = get_a_free_virtual_server_loopback_port(0, source_port);

	client_descriptor_key.client_ip_address = source_ip_address;
	client_descriptor_key.protocol_type = sptr_ip_parameters->protocol;
	client_descriptor_key.application_port = source_port;
	add_entry_to_virtual_server_loopback_client_descriptor_tree ((BYTE *)ptr_to_proxy_server_info, 
		(BYTE *) &client_descriptor_key);

	server_descriptor_key.protocol_type = sptr_ip_parameters->protocol;
	server_descriptor_key.application_port = ptr_to_proxy_server_info->mapped_port ;
	add_entry_to_virtual_server_loopback_server_descriptor_tree ((BYTE *)ptr_to_proxy_server_info, 
		(BYTE *) &server_descriptor_key);

	*return_value = TRUE;
	return ((BYTE *)ptr_to_proxy_server_info);
}

/*--------------------------------------------------------------------------*\

**	Function Name	: get_free_virtual_server_loopback_info_ptr()
**	Description 	: If a new virtual server loopback request packet, creating
						: client & server descriptor. 					
**	Parameters		: 1. actual source port as sent by the client.
						: 2. ptr to recvd IP packet.
						: 3. ptr to IP parameters.
						: 4. ptr to enum BOOLEAN type ftp_port_command, which will be
						:		set to TRUE if ftp PORT command, else FALSE. 
						: 5. local destination addr which is the mapped NAT local address.
**	Return Value	: pointer to PROXY_SERVER_INFO structure - either newly created
						: one or matching existing one.
**	Notes				:

\*--------------------------------------------------------------------------*/

PROXY_SERVER_INFO *get_free_virtual_server_loopback_info_ptr (USHORT original_source_port, 
	BYTE *uptr_rx_packet, IP_PARAMETERS *sptr_ip_parameters, 
	enum BOOLEAN *eptr_ftp_port_command, ULONG local_destination_address)
{
	PROXY_SERVER_INFO *ptr_to_proxy_server_info = NULL;
	ULONG client_ip_address;
	enum BOOLEAN return_value = FALSE;
	PROXY_SERVER_INFO *new_descriptor = NULL;
	USHORT reserved_destination_port = 0, source_port = 0;
	UNION_IP_PACKET *uptr_ip_rx_packet = NULL;

	PROXY_RB_CLIENT_KEY_TYPE client_descriptor_key;
	PROXY_CLIENT_DESCRIPTOR_NODE far *proxy_temp_client_descriptor_list = NULL;

	PROXY_RB_SERVER_KEY_TYPE server_descriptor_key;

	*eptr_ftp_port_command = FALSE;

	uptr_ip_rx_packet = (UNION_IP_PACKET *) uptr_rx_packet;
	client_ip_address = uptr_ip_rx_packet->ip.header.source_ip_address;

	switch (sptr_ip_parameters->protocol)
	{
		case IP_DATA_TCP:
		case IP_DATA_UDP:
			client_descriptor_key.client_ip_address = client_ip_address;
			client_descriptor_key.protocol_type = sptr_ip_parameters->protocol;
			client_descriptor_key.application_port = original_source_port;
			proxy_temp_client_descriptor_list = proxy_rb_client_descriptor_search (sptr_virtual_server_loopback_client_descriptor_tree, 
																client_descriptor_key);

			if (proxy_temp_client_descriptor_list)
			{
				ptr_to_proxy_server_info = (PROXY_SERVER_INFO *) proxy_temp_client_descriptor_list->info.client_descriptor;
			}

			reserved_destination_port = get_tcp_port_number_from_ip_packet ((BYTE *)uptr_ip_rx_packet,
													sptr_ip_parameters, &source_port);

			if ((proxy_temp_client_descriptor_list != NULL) &&
				  (ptr_to_proxy_server_info->client_ip_address == client_ip_address) &&
				  (ptr_to_proxy_server_info->protocol_type == sptr_ip_parameters->protocol) &&
				  (ptr_to_proxy_server_info->source_port == original_source_port))
			{
/* This is to support FTP */
				if (reserved_destination_port == proxy_server.ftp_ctrl_conn_port)
				{
					new_descriptor = (PROXY_SERVER_INFO *) check_for_ftp_port_command_in_loopback_request_and_process 
						(uptr_rx_packet, &return_value, sptr_ip_parameters, ptr_to_proxy_server_info);
				}		

				if (return_value == TRUE)
				{
					new_descriptor->ptr_to_ftp_command_descriptor = ptr_to_proxy_server_info;
					ptr_to_proxy_server_info->ptr_to_ftp_data_descriptor = new_descriptor;

					ptr_to_proxy_server_info->tftp_packet_detected = FALSE;

					new_descriptor->incoming_physical_port = sptr_ip_parameters->rx_port_number;
		
					*eptr_ftp_port_command = TRUE ;
				}
				return (ptr_to_proxy_server_info);
			}
			break;

		default:
			break;
	}
			
	ptr_to_proxy_server_info = (PROXY_SERVER_INFO *) calloc (sizeof (PROXY_SERVER_INFO), 1);
	if (ptr_to_proxy_server_info == NULL)
	{
		proxy_printf ("PROXY_SERVER: Not Enough memory to proxy, Dropping packet\n");	
		return NULL;
	}

	ptr_to_proxy_server_info->ptr_to_ftp_command_descriptor = NULL;
	ptr_to_proxy_server_info->ptr_to_ftp_data_descriptor = NULL;

	ptr_to_proxy_server_info->outgoing_physical_port = sptr_ip_parameters->rx_port_number;

	ptr_to_proxy_server_info->destination_address = sptr_ip_parameters->destination_address;
	ptr_to_proxy_server_info->client_ip_address = uptr_ip_rx_packet->ip.header.source_ip_address ;				
	ptr_to_proxy_server_info->outgoing_port_address =  get_ip_address (get_secured_lan_port_number());

	ptr_to_proxy_server_info->protocol_type = sptr_ip_parameters->protocol;
	ptr_to_proxy_server_info->seq_offset = 0;

	ptr_to_proxy_server_info->ftp_port_command_on = FALSE;
	ptr_to_proxy_server_info->ftp_pasv_mode_on = FALSE;
	ptr_to_proxy_server_info->tftp_packet_detected = FALSE;
	
	ptr_to_proxy_server_info->in_use = TRUE;
	ptr_to_proxy_server_info->connection_closing = FALSE;
	
	ptr_to_proxy_server_info->client_node = NULL;
	ptr_to_proxy_server_info->server_node = NULL;		

	ptr_to_proxy_server_info->outgoing_destination_address = local_destination_address;

	if (sptr_ip_parameters->protocol == IP_DATA_TCP)
	{
		ptr_to_proxy_server_info->proxy_idle_timer = 600; /* 600 Secs */
	}
	else			
	{
		ptr_to_proxy_server_info->proxy_idle_timer = 60; /* 60 Secs */
	}

	if (sptr_ip_parameters->protocol == IP_DATA_ICMP)
	{
		return ptr_to_proxy_server_info;
	}

	ptr_to_proxy_server_info->source_port = original_source_port; 

	ptr_to_proxy_server_info->mapped_port = get_a_free_virtual_server_loopback_port(reserved_destination_port, 
															original_source_port);

/*  For pasv FTP support ... */
	ptr_to_proxy_server_info->destination_port = reserved_destination_port; 
	proxy_rb_pasv_ftp_ctrl_client_descriptor_search (sptr_virtual_server_loopback_client_descriptor_tree, 
		ptr_to_proxy_server_info);	
/* ... For pasv FTP support */

	client_descriptor_key.client_ip_address = ptr_to_proxy_server_info->client_ip_address;
	client_descriptor_key.protocol_type = sptr_ip_parameters->protocol;
	client_descriptor_key.application_port = ptr_to_proxy_server_info->source_port;
	add_entry_to_virtual_server_loopback_client_descriptor_tree ((BYTE *)ptr_to_proxy_server_info, 
		(BYTE *) &client_descriptor_key);

	server_descriptor_key.protocol_type = sptr_ip_parameters->protocol;
	server_descriptor_key.application_port = ptr_to_proxy_server_info->mapped_port;
	add_entry_to_virtual_server_loopback_server_descriptor_tree ((BYTE *)ptr_to_proxy_server_info, 
		(BYTE *) &server_descriptor_key);

	return ptr_to_proxy_server_info;
}

/*--------------------------------------------------------------------------*\

**	Function Name	: process_virtual_server_loopback_request_packet()
**	Description 	: If a virtual server loopback request packet, changing
						: required parameters in recvd IP packet like 
						: 1. source ip addr from client ip addr to secured LAN addr
						: 2. destination addr from valid IP global addr to invalid
						:		NAT mapped IP local addr.
 						: 3. source port to mapped port.	
						: and updating the resp checksum values depending upon protocol.
**	Parameters		: 1. ptr to recvd IP packet.
						: 2. ptr to IP parameters.
						: 3. ptr to matching proxy server info.
						: 4. ptr to enum BOOLEAN type ftp_port_command which indicates
						: 		recvd pkt is ftp PORT command pkt or not.
**	Return Value	: void.
**	Notes				:

\*--------------------------------------------------------------------------*/

void process_virtual_server_loopback_request_packet (UNION_IP_PACKET *uptr_ip_rx_packet, 
	IP_PARAMETERS *sptr_ip_parameters, BYTE *proxy_server_info, enum BOOLEAN *ftp_port_command)
{
	PROXY_SERVER_INFO *ptr_to_proxy_server_info = NULL;
	BYTE *ptr_to_tcp_packet = NULL, *ptr_to_udp_packet = NULL;
	int old_length, new_length ;
	BYTE *sptr_icmp_header = NULL;
	USHORT new_checksum = 0;

	ptr_to_proxy_server_info = (PROXY_SERVER_INFO *) proxy_server_info;

	sptr_ip_parameters->source_address = ptr_to_proxy_server_info->outgoing_port_address;
	sptr_ip_parameters->destination_address = ptr_to_proxy_server_info->outgoing_destination_address;

	uptr_ip_rx_packet->ip.header.source_ip_address = sptr_ip_parameters->source_address;
	uptr_ip_rx_packet->ip.header.destination_ip_address = sptr_ip_parameters->destination_address;

	if (ptr_to_proxy_server_info->protocol_type == ICMP_PROTOCOL)
	{
		uptr_ip_rx_packet->icmp.header.option.echo_message.identifier = 
					ptr_to_proxy_server_info->mapped_icmp_packet_id;					

		sptr_icmp_header = get_ptr_to_icmp_header (uptr_ip_rx_packet, sptr_ip_parameters);		 
		set_icmp_checksum (sptr_icmp_header, 0);
		new_checksum =	calculate_ip_checksum (NULL, 
				(BYTE *)sptr_icmp_header, (USHORT) (sptr_ip_parameters->total_length- sptr_ip_parameters->header_length));
		set_icmp_checksum (sptr_icmp_header, new_checksum);
		return;				
	}

	set_source_port_number ((BYTE *) uptr_ip_rx_packet, sptr_ip_parameters, 
		ptr_to_proxy_server_info->mapped_port, 
		ptr_to_proxy_server_info->protocol_type);
	
	if (ptr_to_proxy_server_info->ftp_port_command_on)
	{
		modify_tcp_seq_number_field ((BYTE *)uptr_ip_rx_packet, sptr_ip_parameters, 
			ptr_to_proxy_server_info->seq_offset);
	}

	if (ptr_to_proxy_server_info->ftp_pasv_mode_on)
	{
		modify_tcp_acknowledge_field ((BYTE *)uptr_ip_rx_packet, sptr_ip_parameters, 
			ptr_to_proxy_server_info->seq_offset);
	}

	if (*ftp_port_command == TRUE)
	{
		ptr_to_proxy_server_info->ftp_port_command_on = TRUE;
		ptr_to_tcp_packet = get_ptr_to_tcp_packet ((BYTE *)uptr_ip_rx_packet, sptr_ip_parameters);
		old_length = get_port_command_length (ptr_to_tcp_packet);
		form_new_port_command (ptr_to_tcp_packet, ptr_to_proxy_server_info->ptr_to_ftp_data_descriptor);
		new_length = get_port_command_length (ptr_to_tcp_packet);
		uptr_ip_rx_packet->ip.header.total_length += (new_length - old_length);
		sptr_ip_parameters->total_length += (new_length - old_length);

		ptr_to_proxy_server_info->seq_offset += new_length - old_length;
	}

	if (ptr_to_proxy_server_info->protocol_type == TCP_PROTOCOL)
	{
		if (ptr_to_proxy_server_info->destination_port == RD_RNR_TAS_TCP_PORT)
		{
			/* we need to see if the data contains the source address in string
			form. If it found replace it with the FW internet port address. */
			ptr_to_tcp_packet = get_ptr_to_tcp_packet ((BYTE *)uptr_ip_rx_packet, sptr_ip_parameters);

			road_runner_rlogin_replace_source_address_if_in_data(
				ptr_to_proxy_server_info->client_ip_address,
				ptr_to_proxy_server_info->outgoing_port_address, ptr_to_tcp_packet);
		}
	}
	else if (ptr_to_proxy_server_info->protocol_type == UDP_PROTOCOL)
	{
		if (ptr_to_proxy_server_info->destination_port == RD_RNR_TAS_UDP_PORT)
		{
/* printf(" Proxy :rd_rnr sock %d\n", RD_RNR_TAS_UDP_PORT); */
			/* we need to see if the data contains the source address in string
			form. If it found replace it with the FW internet port address. */
			ptr_to_udp_packet = get_ptr_to_udp_packet ((BYTE *)uptr_ip_rx_packet, sptr_ip_parameters);

			road_runner_rlogin_replace_source_address_if_in_data(
				ptr_to_proxy_server_info->client_ip_address,
				ptr_to_proxy_server_info->outgoing_port_address, ptr_to_udp_packet);
		}
	}

	if (ptr_to_proxy_server_info->protocol_type == TCP_PROTOCOL)
	{
		update_tcp_checksum_in_packet (uptr_ip_rx_packet, sptr_ip_parameters); 
	}
	else if (ptr_to_proxy_server_info->protocol_type == UDP_PROTOCOL)
	{
		update_udp_checksum_in_packet (uptr_ip_rx_packet, sptr_ip_parameters); 
	}

	return;
}

/*--------------------------------------------------------------------------*\

**	Function Name	: check_and_process_if_virtual_server_loopback_request_packet()
**	Description 	: If packet is rxed from secured LAN & destined to our NAT
						: global address ( static or dynamic ) to our interface addr 
						: or any valid IP address, that should be treated as virtual 
						: server loopback	request packet.
**	Parameters		: 1. ptr to recvd IP packet.
						: 2. ptr to IP parameters.
						: 3. ptr to eptr_size_modified which is used to indicate
						:		whether the pkt is ftp PORT command or not.
						:		If it is ftp PORT command pkt, length of TCP data
						:		might have been changed due to ip address replacement
						:		in TCP data portion. The change in length of pkt needs
						:		to be reflected in IP header also in total length field.
**	Return Value	: if virtual server loopback request pkt, TRUE else FALSE.
**	Notes				:

\*--------------------------------------------------------------------------*/

enum BOOLEAN check_and_process_if_virtual_server_loopback_request_packet (UNION_IP_PACKET *uptr_ip_rx_packet, 
		IP_PARAMETERS *sptr_ip_parameters, enum BOOLEAN *eptr_size_modified)
{
	enum IP_PROTOCOL_VALUE protocol_type;
	enum BOOLEAN is_my_packet = FALSE; 
	USHORT reserved_application_port=0, original_source_port=0;
	ULONG local_destination_address;
	
	PROXY_SERVER_INFO *ptr_to_proxy_info = NULL;
	PROXY_RB_SERVER_KEY_TYPE server_descriptor_key;

/* This is to be included after initing from nvram */
	if (!proxy_server.proxy_server_enabled || !proxy_server.port[sptr_ip_parameters->rx_port_number].proxy_server_enabled)
	{
		proxy_printf ("PROXY_SERVER: proxy server disabled for port %d\n",sptr_ip_parameters->rx_port_number); 
		return FALSE;
	}

	reserved_application_port = get_application_port_number ((BYTE *) uptr_ip_rx_packet, 
											sptr_ip_parameters, &original_source_port, &protocol_type);

	if ((protocol_type != TCP_PROTOCOL) && (protocol_type != UDP_PROTOCOL) && 
				(protocol_type != ICMP_PROTOCOL))
	{
		return FALSE; 
	}

	if (reserved_application_port == INVALID_DESTINATION_PORT && protocol_type != ICMP_PROTOCOL) 
	{
		return FALSE;
	}

	if (check_if_the_packet_is_for_our_NAT_address ((BYTE *)uptr_ip_rx_packet, sptr_ip_parameters, 
		&local_destination_address, &is_my_packet) == TRUE)
	{
		if (is_my_packet == TRUE)
		{
			proxy_printf("Not a loopback request.But for our Telnet/Web Server.\n");
			return FALSE;
		}
		else   
		{
			switch (protocol_type)
			{
				case IP_DATA_ICMP:
					if (is_icmp_echo_request_message (uptr_ip_rx_packet, sptr_ip_parameters))
					{
						ptr_to_proxy_info = get_free_virtual_server_loopback_info_ptr(original_source_port, (BYTE *)uptr_ip_rx_packet, 
													sptr_ip_parameters, eptr_size_modified, local_destination_address);

						if (ptr_to_proxy_info == NULL)
						{
							return FALSE;
						}					

						ptr_to_proxy_info->incoming_physical_port = sptr_ip_parameters->rx_port_number;
						ptr_to_proxy_info->destination_address = sptr_ip_parameters->destination_address;
						ptr_to_proxy_info->proxy_idle_timer = 30; /* 30 secs */
						ptr_to_proxy_info->protocol_type = protocol_type;					

						ptr_to_proxy_info->icmp_packet_id = get_icmp_echo_request_id (uptr_ip_rx_packet, sptr_ip_parameters);

						if (virtual_server_loopback_new_icmp_packet_id == 0)
						{
							virtual_server_loopback_new_icmp_packet_id = 5;
						}
			
						ptr_to_proxy_info->mapped_icmp_packet_id = virtual_server_loopback_new_icmp_packet_id++;
						ptr_to_proxy_info->in_use = TRUE;

						server_descriptor_key.protocol_type = protocol_type;
						server_descriptor_key.application_port = ptr_to_proxy_info->mapped_icmp_packet_id;
						add_entry_to_virtual_server_loopback_server_descriptor_tree ((BYTE *)ptr_to_proxy_info, 
							(BYTE *) &server_descriptor_key);

						process_virtual_server_loopback_request_packet (uptr_ip_rx_packet, sptr_ip_parameters,
							(BYTE *)ptr_to_proxy_info, eptr_size_modified);
					
						return TRUE;
					}	
					break;

				case IP_DATA_TCP:
				case IP_DATA_UDP:
					if (reserved_application_port == INVALID_DESTINATION_PORT)
					{
						proxy_printf ("PROXY :Destination Port is Invalid\n");
						return FALSE;
					}

					ptr_to_proxy_info = get_free_virtual_server_loopback_info_ptr(original_source_port, 
						(BYTE *)uptr_ip_rx_packet, sptr_ip_parameters, eptr_size_modified, local_destination_address);

					if (ptr_to_proxy_info == NULL)
					{
						printf ("PROXY : Not Enough memory to Allocate Descriptor\n"); 
						return FALSE;
					}

					ptr_to_proxy_info->incoming_physical_port = sptr_ip_parameters->rx_port_number;
					ptr_to_proxy_info->destination_port = reserved_application_port;
					ptr_to_proxy_info->in_use = TRUE;

					if ((protocol_type == TCP_PROTOCOL) && 
						(is_close_connection_initiated ((BYTE *) uptr_ip_rx_packet, sptr_ip_parameters) == TRUE))
					{
						proxy_printf ("PROXY : TCP Connection is closed by %08x\n",
									uptr_ip_rx_packet->ip.header.source_ip_address); 
					
						ptr_to_proxy_info->proxy_idle_timer = 30; /* 30 Secs */
						ptr_to_proxy_info->connection_closing = TRUE;
					}

					if (ptr_to_proxy_info->connection_closing == FALSE)
			      {
     				   if (ptr_to_proxy_info->ptr_to_ftp_command_descriptor)
						{
   		      		ptr_to_proxy_info->ptr_to_ftp_command_descriptor->proxy_idle_timer = 600; /* 600 Secs */
						}

						if (protocol_type == TCP_PROTOCOL) 
						{
			   			ptr_to_proxy_info->proxy_idle_timer = 600; /* 600 Secs */
						}
						else 
						{
						   ptr_to_proxy_info->proxy_idle_timer = 60; /* 60 Secs */
						}
     				}
					
					process_virtual_server_loopback_request_packet (uptr_ip_rx_packet, sptr_ip_parameters,
							(BYTE *)ptr_to_proxy_info, eptr_size_modified);
  		
					return TRUE;
					break;

				default:
					break;
			}
		}
	}
	else /* not for our NAT addr */
	{
		/* printf("Not a loopback request.\n"); */
		return FALSE;
	}

	return FALSE;
}

/*--------------------------------------------------------------------------*\

**	Function Name	: process_the_loopback_response_ftp_packet_after_pasv_commd()
**	Description 	: if pasv PORT command pkt, 
						: 1. change the global IP address in TCP data portion to resp 
						: 		mapped invalid local IP address.
						: 2. add a new dynamic NAT mapping for this dynamic pasv port 
						:		application to allow NAT request with this port number.
**	Parameters		: 1. ptr to recvd IP packet.
						: 2. ptr to IP parameters.
						: 3. ptr to server descriptor info which is ctrl connection
						:		descriptor.
						: 4. ptr to enum BOOLEAN type ftp_pasv_mode which is set to
						:		TRUE if pasv PORT command, else FALSE.
**	Return Value	: void.
**	Notes				:

\*--------------------------------------------------------------------------*/

void process_the_loopback_response_ftp_packet_after_pasv_commd (UNION_IP_PACKET *uptr_ip_rx_packet, 
		IP_PARAMETERS *sptr_ip_parameters, PROXY_SERVER_INFO *server_descriptor_info, enum BOOLEAN *ftp_pasv_mode)
{
	BYTE *ptr_to_tcp_packet = NULL;
	int old_length, new_length ;
	BYTE ip_address[20], port_buff[10];
	USHORT number_of_commas=0, buffer_index=0, port_index=0;
	
	USHORT port1, port2, new_port_number;
	PROXY_DYNAMIC_NAT_APP_LIST *ptr_to_nat_app_list = NULL;	

	ptr_to_tcp_packet = get_ptr_to_tcp_packet ((BYTE *)uptr_ip_rx_packet, sptr_ip_parameters);
	old_length = get_port_command_length (ptr_to_tcp_packet);

	if (strstr (ptr_to_tcp_packet, "227 Entering Passive Mode") == NULL)
	{
		*ftp_pasv_mode = FALSE;
		return;
	}

	ptr_to_nat_app_list = (PROXY_DYNAMIC_NAT_APP_LIST *) calloc (sizeof (PROXY_DYNAMIC_NAT_APP_LIST), 1);
	if (ptr_to_nat_app_list == NULL)
	{
		printf ("PROXY : Not Enough memory add application\n");
		*ftp_pasv_mode = FALSE;
		return;
	} 

	ptr_to_nat_app_list->dynamic_server_ip_address = server_descriptor_info->outgoing_destination_address;
	ptr_to_nat_app_list->ip_protocol_type = server_descriptor_info->protocol_type;

	convert_ip_address_to_dot_format (&ip_address[0], server_descriptor_info->destination_address);
	for (buffer_index = 0 ; ip_address[buffer_index] ; buffer_index++)
	{
		if (ip_address[buffer_index] == '.')
		{
			ip_address[buffer_index] = ',';
		}			
	}
			
	for (buffer_index =0; ptr_to_tcp_packet[buffer_index] != 0x0A; buffer_index++)
	{
		if (number_of_commas == 4)
		{
			if((ptr_to_tcp_packet[buffer_index] == 0x0D) || 
				(ptr_to_tcp_packet[buffer_index] == ')'))
			{
				break;
			}

			port_buff[port_index] = ptr_to_tcp_packet[buffer_index];
			port_index++;
			continue;
		}
		if (ptr_to_tcp_packet[buffer_index] == ',')
		{
			number_of_commas++;
		}
	}
	port_buff[port_index] = 0;

	sprintf (&ptr_to_tcp_packet[0], "227 Entering Passive Mode (%s,%s)\x0D\x0A",
			&ip_address[0], &port_buff[0]);

	sscanf (port_buff ,"%hu,%hu",&port1, &port2);		
	
	port1 = port1 & 0x00ff;
	port1 = port1 << 8;
	port2 = port2 &0x00ff;
	new_port_number = (port1 | port2);
	ptr_to_nat_app_list->application_port = new_port_number;

	server_descriptor_info->pasv_port = new_port_number;

	ptr_to_nat_app_list->sptr_forward_link = proxy_server.dynamic_nat_application_list;
	proxy_server.dynamic_nat_application_list = ptr_to_nat_app_list;
	
	new_length = get_port_command_length (ptr_to_tcp_packet);

	uptr_ip_rx_packet->ip.header.total_length += (new_length - old_length);
	sptr_ip_parameters->total_length += (new_length - old_length);

	server_descriptor_info->seq_offset += new_length - old_length;

	*ftp_pasv_mode = TRUE;
	server_descriptor_info->ftp_pasv_mode_on = TRUE;

	return;
}

/*--------------------------------------------------------------------------*\

**	Function Name	: process_virtual_server_loopback_response_packet()
**	Description 	: If a virtual server loopback response packet, changing
						: required parameters in recvd IP packet like 
						: 1. source ip addr from invalid NAT mapped local addr to
						:		valid IP global addr.
						: 2. destination addr from secured LAN addr to client ip
						:		addr.
 						: 3. mapped port to source port ( destination port number),	
						: and updating the resp checksum values depending upon protocol.
**	Parameters		: 1. ptr to recvd IP packet.
						: 2. ptr to IP parameters.
						: 3. ptr to matching proxy server info.
						: 4. ptr to enum BOOLEAN type ftp_pasv_mode which indicates
						: 		recvd pkt is ftp pasv PORT command pkt or not.
**	Return Value	: void.
**	Notes				:

\*--------------------------------------------------------------------------*/

void process_virtual_server_loopback_response_packet (UNION_IP_PACKET *uptr_ip_rx_packet, 
	IP_PARAMETERS *sptr_ip_parameters, BYTE *proxy_server_info, enum BOOLEAN *ftp_pasv_mode)
{
	PROXY_SERVER_INFO *ptr_to_proxy_info = NULL;	
	BYTE *sptr_icmp_header = NULL, *ptr_to_tcp_packet = NULL;
	USHORT new_checksum=0;

	ptr_to_proxy_info = (PROXY_SERVER_INFO *) proxy_server_info;

	sptr_ip_parameters->destination_address = ptr_to_proxy_info->client_ip_address;		
	sptr_ip_parameters->source_address = ptr_to_proxy_info->destination_address;		

	uptr_ip_rx_packet->icmp.ip_header.destination_ip_address = ptr_to_proxy_info->client_ip_address;
	uptr_ip_rx_packet->icmp.ip_header.source_ip_address = ptr_to_proxy_info->destination_address;

	if (sptr_ip_parameters->protocol == ICMP_PROTOCOL)
	{
/* Support for trace route - ICMP TIME EXCEEDED packet */
		sptr_icmp_header = get_ptr_to_icmp_header (uptr_ip_rx_packet, sptr_ip_parameters);		
		if (*sptr_icmp_header == 0x0B) /* ICMP Time Exceeded Type value */
		{
			uptr_ip_rx_packet->icmp.header.option.time_exceeded_message.ip_data.source_ip_address =
				ptr_to_proxy_info->client_ip_address;
			uptr_ip_rx_packet->icmp.header.option.time_exceeded_message.ip_data.destination_ip_address =
				ptr_to_proxy_info->destination_address;
			memcpy ((void *)&uptr_ip_rx_packet->icmp.header.option.time_exceeded_message.data[4], 
				(void *)&ptr_to_proxy_info->icmp_packet_id, sizeof (USHORT));			
		}
		else
		{
			uptr_ip_rx_packet->icmp.header.option.echo_message.identifier = 
				ptr_to_proxy_info->icmp_packet_id;			
		}

		set_icmp_checksum (sptr_icmp_header, 0);
		new_checksum = calculate_ip_checksum (NULL, sptr_icmp_header, 
			(USHORT) (sptr_ip_parameters->total_length - sptr_ip_parameters->header_length));
		set_icmp_checksum (sptr_icmp_header, new_checksum);
		return;				
	}

	if (ptr_to_proxy_info->ftp_port_command_on)
	{
		modify_tcp_acknowledge_field ((BYTE *)uptr_ip_rx_packet, sptr_ip_parameters, 
			ptr_to_proxy_info->seq_offset);
	}

/*  For pasv FTP support ... */
	if (ptr_to_proxy_info->ftp_pasv_mode_on)
	{
		modify_tcp_seq_number_field ((BYTE *)uptr_ip_rx_packet, sptr_ip_parameters, 
			ptr_to_proxy_info->seq_offset);
	}
/* ... For pasv FTP support */

/* sudhir for ftp passive command handling */
	ptr_to_tcp_packet = get_ptr_to_tcp_packet ((BYTE *) uptr_ip_rx_packet, sptr_ip_parameters);
	if (strstr (ptr_to_tcp_packet, "227 Entering Passive Mode") != NULL)
	{
		proxy_printf ("PROXY : Passive port command received\n");
		process_the_loopback_response_ftp_packet_after_pasv_commd (uptr_ip_rx_packet, 
			sptr_ip_parameters, ptr_to_proxy_info, ftp_pasv_mode);
	}

	set_destination_port_number ((BYTE *) uptr_ip_rx_packet, 
		sptr_ip_parameters, ptr_to_proxy_info->source_port,
		ptr_to_proxy_info->protocol_type);			
/* sudhir for ftp passive command handling */
		
	if (sptr_ip_parameters->protocol == TCP_PROTOCOL)
	{
		update_tcp_checksum_in_packet (uptr_ip_rx_packet, sptr_ip_parameters); 
	}
	else if (sptr_ip_parameters->protocol == UDP_PROTOCOL)
	{
		update_udp_checksum_in_packet (uptr_ip_rx_packet, sptr_ip_parameters); 
	}
	
	return;
}

/*--------------------------------------------------------------------------*\

**	Function Name	: is_virtual_server_loopback_response_packet()
**	Description 	: If packet is rxed from secured LAN & destined to our secure 
						: LAN IP address, check if any matching entry present in the 
						: virtual server loopback tree for this packet. We are searching 
						: in the server descriptor tree with server key for the response.	
**	Parameters		: 1. ptr to recvd IP packet.
						: 2. ptr to IP parameters.
						: 3. ptr to enum BOOLEAN type ftp_pasv_mode which indicates
						: 		recvd pkt is ftp pasv PORT command pkt or not.
**	Return Value	: TRUE or FALSE.
**	Notes				:

\*--------------------------------------------------------------------------*/

enum BOOLEAN is_virtual_server_loopback_response_packet (BYTE *uptr_ip_rx_packet, 
	IP_PARAMETERS *sptr_ip_parameters, enum BOOLEAN *ftp_pasv_mode) 
{
	USHORT destination_port = 0, source_port = 0;
	USHORT icmp_packet_id = 0;
	enum IP_PROTOCOL_VALUE protocol_type;
	PROXY_SERVER_INFO *ptr_to_proxy_server_info = NULL;

	PROXY_RB_SERVER_KEY_TYPE server_descriptor_key;
	PROXY_SERVER_DESCRIPTOR_NODE far *proxy_temp_server_descriptor_list = NULL;

	destination_port = get_application_port_number (uptr_ip_rx_packet, sptr_ip_parameters, 
								&source_port, &protocol_type);

	if ((protocol_type != TCP_PROTOCOL) && (protocol_type != UDP_PROTOCOL) && 
		 (protocol_type != ICMP_PROTOCOL))
	{
		return FALSE; 
	}

	if ((destination_port == INVALID_DESTINATION_PORT) && (protocol_type != ICMP_PROTOCOL)) 
	{
		return FALSE;
	}

	if (sptr_ip_parameters->protocol == ICMP_PROTOCOL)
	{
		icmp_packet_id = get_icmp_echo_request_id ((UNION_IP_PACKET *)uptr_ip_rx_packet, sptr_ip_parameters);
		
		server_descriptor_key.protocol_type = sptr_ip_parameters->protocol;
		server_descriptor_key.application_port = icmp_packet_id;
		proxy_temp_server_descriptor_list = proxy_rb_server_descriptor_search (
															sptr_virtual_server_loopback_server_descriptor_tree, 
															server_descriptor_key);

		if (proxy_temp_server_descriptor_list == NULL)
		{
			return FALSE;
		}

		ptr_to_proxy_server_info = (PROXY_SERVER_INFO *) (proxy_temp_server_descriptor_list->info.server_descriptor);
		
		if ((ptr_to_proxy_server_info->mapped_icmp_packet_id == icmp_packet_id)	&& 
			 (ptr_to_proxy_server_info->outgoing_destination_address == sptr_ip_parameters->source_address)	&& 
			 (ptr_to_proxy_server_info->outgoing_port_address == sptr_ip_parameters->destination_address))
		{	
			ptr_to_proxy_server_info->proxy_idle_timer = 1; /* 1 Sec */

			process_virtual_server_loopback_response_packet ((UNION_IP_PACKET *)uptr_ip_rx_packet, 
				sptr_ip_parameters, (BYTE *)ptr_to_proxy_server_info, ftp_pasv_mode);

			return TRUE;
		}
		else
		{
			return FALSE;
		}
  	}

	server_descriptor_key.protocol_type = sptr_ip_parameters->protocol;
	server_descriptor_key.application_port = destination_port;

	proxy_temp_server_descriptor_list = proxy_rb_server_descriptor_search (
														sptr_virtual_server_loopback_server_descriptor_tree, 
														server_descriptor_key);

	if (proxy_temp_server_descriptor_list == NULL)
	{
		return FALSE;
	}

	ptr_to_proxy_server_info = (PROXY_SERVER_INFO *) (proxy_temp_server_descriptor_list->info.server_descriptor);

	if ((ptr_to_proxy_server_info->mapped_port == destination_port) && 
		 (ptr_to_proxy_server_info->protocol_type == sptr_ip_parameters->protocol)	&& 
		 (ptr_to_proxy_server_info->outgoing_port_address == sptr_ip_parameters->destination_address) && 
	 	 (ptr_to_proxy_server_info->outgoing_destination_address == sptr_ip_parameters->source_address))
	{
		if ((protocol_type == TCP_PROTOCOL) && 
			(is_close_connection_initiated (uptr_ip_rx_packet, sptr_ip_parameters) == TRUE)) 
		{
			proxy_printf ("PROXY_SERVER : TCP Connection is closed\n"); 
  			ptr_to_proxy_server_info->connection_closing = TRUE;
			ptr_to_proxy_server_info->proxy_idle_timer = 30; /* 30 Secs */
		}
  		
		if (ptr_to_proxy_server_info->connection_closing == FALSE)
      {
      	if (ptr_to_proxy_server_info->ptr_to_ftp_command_descriptor)
			{
	         ptr_to_proxy_server_info->ptr_to_ftp_command_descriptor->proxy_idle_timer = 600; /* 600 Secs */
			}

			if (protocol_type == TCP_PROTOCOL) 
			{
				ptr_to_proxy_server_info->proxy_idle_timer = 600; /* 600 Secs */
			}
			else
			{
				ptr_to_proxy_server_info->proxy_idle_timer = 60; /* 60 Secs */
			}
      }

		process_virtual_server_loopback_response_packet ((UNION_IP_PACKET *)uptr_ip_rx_packet, 
				sptr_ip_parameters, (BYTE *)ptr_to_proxy_server_info, ftp_pasv_mode);

		return TRUE;
	}
	
	return FALSE;
}

#endif /* __VIRTUAL_SERVER_LOOPBACK__ */
/* --- EOF of vserlpbk.c file --- */
