#include <stdio.h>
#include <stdlib.h>
#include "tftp.h"


enum TEST initialize_tftp_socket_interface (void)
{
	int i, bind_return_value ;
	SOCKADDR_IN local_tftp_socket ;

	if (tftp.socket_interface_initialized == FALSE)
	{
		my_node_ip_address = gethostid() ;
	
		connect_info_table = (CONNECTION_INFO *) malloc(sizeof(CONNECTION_INFO) * (tftp.max_connections + 1)) ;
		if (connect_info_table == NULL)
		{
			return (FAIL) ;
		}
	
		connect_info_table[0].received_packet = (BYTE *) malloc (MAX_TFTP_PACKET_SIZE) ;
		if (connect_info_table[0].received_packet == NULL)
		{
			return (FAIL) ;
		}

		connect_info_table[0].in_use_flag = 1 ;
		connect_info_table[0].socket = socket (AF_INET, SOCK_DGRAM, IPPROTO_UDP) ;
		
		if (connect_info_table[0].socket == FAILED)
		{
			free(connect_info_table) ;
			free(connect_info_table[0].received_packet) ;

			tftp_printf (TFTP_INIT_PRINTF, "\nTFTP Error : Cannot open socket") ;
			return (FAIL) ;
		}

		tftp_printf (TFTP_INIT_PRINTF, "\nTFTP : Socket opened successfully") ;
		local_tftp_socket.sin_family = AF_INET ;
		local_tftp_socket.sin_port = tftp_well_known_port ;
		local_tftp_socket.sin_addr.s_addr = my_node_ip_address ;

		associate_application_with_socket (connect_info_table[0].socket,
		                                 tftp.application_id) ;
		bind_return_value = bind (connect_info_table[0].socket,
		                          (SOCKADDR *) &local_tftp_socket,
										  sizeof(SOCKADDR_IN)) ;
		if (bind_return_value == FAILED)
		{
			free(connect_info_table) ;
			free(connect_info_table[0].received_packet) ;

			tftp_printf (TFTP_INIT_PRINTF, "\nTFTP Error : Bind failed") ;
			return (FAIL) ;
		}
		
		tftp_printf (TFTP_INIT_PRINTF, "\nTFTP : Socket bound successfully") ;
		for (i = 1 ; i <= tftp.max_connections ; i++)
			connect_info_table[i].in_use_flag = 0 ;

		connect_info_table[0].current_state = STATE_LISTENING ;
		connect_info_table[0].in_use_flag = 1 ;

		last_call = get_timer_marker() ;

		tftp.socket_interface_initialized = TRUE ;
	}
	else
	{
		return (FAIL) ;
	}

	return (PASS) ;
}




/*---------------------------------------------------------------------------
Function	: send_data_packet
Synopsis	: sends a packet.
Input 	: Destination IP address
           Destination port
			  socket to send data on
			  Pointer to the information to be sent
			  No. of bytes to be sent
Output	: 0 on success, else error code
---------------------------------------------------------------------------*/
int send_data_packet (ULONG destination_ip_address, int destination_port,
                   int socket, BYTE *info, ULONG length)
{
	SOCKADDR_IN destination ;
	int return_value ;
	BYTE *tftp_send_packet ;

	tftp_send_packet = (BYTE *) malloc (MAX_TFTP_PACKET_SIZE) ;
	if (tftp_send_packet == NULL)
	{
		return -1 ;
	}

	memcpy(tftp_send_packet, info, length) ;

	memset(&destination, 0, sizeof(SOCKADDR_IN)) ;
	destination.sin_family = AF_INET ;
	destination.sin_port = destination_port ;
	destination.sin_addr.s_addr = destination_ip_address ;

	return_value = sendto(socket, tftp_send_packet, length, 0,
	       (SOCKADDR *)&destination, sizeof(SOCKADDR_IN)) ;

	if (return_value != FAILED)
	{
		free (tftp_send_packet) ;
		return 0 ;
	}
	/* If there is a send error the buffer is freed by socket_send_data()
	   or socket_udp_send(). So DO NOT free the buffer */
	return (-1);
}




/*---------------------------------------------------------------------------
Function	: ReceiveData
Synopsis	: reads a packet.
Input 	: Destination IP address
           Destination port
			  socket to send data on
			  Pointer to the information to be sent
			  No. of bytes to be sent
Output	: 0 on success, else error code
---------------------------------------------------------------------------*/
int receive_data(ULONG *source_ip_address, int *source_port, int socket,
                BYTE *packet, int *packet_length)
{
	int add_len = sizeof(SOCKADDR_IN), return_code ;
	SOCKADDR_IN client_address ;

	return_code = recvfrom(socket, packet, MAX_TFTP_PACKET_SIZE, 0,
	                      (SOCKADDR *)&client_address, &add_len) ;
	if (return_code != -1)
	{
		*packet_length = return_code ;
		if (return_code == 0)
			return RECEIVE_FAILED ;
		/* RouterWare's recvfrom() returns 0 if no data is available */
		*source_ip_address = client_address.sin_addr.s_addr ;
		*source_port = client_address.sin_port ;
		return 0 ;
	}
	*packet_length = 0 ;
	return RECEIVE_FAILED ;
}




/*---------------------------------------------------------------------------
Function	: SendAckPacket
Synopsis	: sends an ack packet.
Input 	: Destination IP address
           Destination port
			  socket to send data on
			  sequence number of the packet that is to be ack.ed
Output	: 0 on success, else error code
---------------------------------------------------------------------------*/
int send_ack_packet(ULONG destination_ip_address, int destination_port,
                    int socket, ULONG sequence_number)
{
	SOCKADDR_IN destination ;
	int return_value ;
	BYTE *tftp_send_packet ;

	tftp_send_packet = malloc(MAX_TFTP_PACKET_SIZE) ;
	if (tftp_send_packet == NULL)
	{
		return -1 ;
	}

	memset(&destination, 0, sizeof(SOCKADDR_IN)) ;
	destination.sin_family = AF_INET ;
	destination.sin_port = destination_port ;
	destination.sin_addr.s_addr = destination_ip_address ;

	((TFTP_PACKET *)tftp_send_packet)->opcode = host_to_net_short(OPCODE_ACK) ;
	((TFTP_PACKET *)tftp_send_packet)->message_type.ack.acked_block_number = host_to_net_short((USHORT)sequence_number) ;

	return_value = sendto(socket, tftp_send_packet, 2+2, 0,
	       (SOCKADDR *)&destination, sizeof(SOCKADDR_IN)) ;

	if (return_value != FAILED)
	{
		free (tftp_send_packet) ;
		return 0 ;
	}
	return -1 ;
}




/*---------------------------------------------------------------------------
Function	: SendErrorPacket
Synopsis	: sends an error packet. The destination and the info. to be sent is
        	  present in ErroeBlock
Input 	: ErrorBlock
Output	: 0 on success, else error code
---------------------------------------------------------------------------*/
int send_error_packet(ULONG destination_ip_address, int destination_port,
                    int socket, const BYTE *error_message, USHORT error_code)
{
	SOCKADDR_IN destination ;
	int return_value ;
	BYTE *tftp_send_packet ;

	tftp_send_packet = malloc(MAX_TFTP_PACKET_SIZE) ;
	if (tftp_send_packet == NULL)
	{
		return -1 ;
	}

	memset(&destination, 0, sizeof(SOCKADDR_IN)) ;
	destination.sin_family = AF_INET ;
	destination.sin_port = destination_port ;
	destination.sin_addr.s_addr = destination_ip_address ;

	((TFTP_PACKET *)tftp_send_packet)->opcode = host_to_net_short(OPCODE_ERROR) ;
	((TFTP_PACKET *)tftp_send_packet)->message_type.error.error_number = host_to_net_short(error_code) ;
	strcpy(((TFTP_PACKET *)tftp_send_packet)->message_type.error.error_message, error_message) ;

	return_value = sendto(socket, tftp_send_packet, 2+2+1+strlen(error_message), 0,
	       (SOCKADDR *)&destination, sizeof(SOCKADDR_IN)) ;

	if (return_value != FAILED)
	{
		free (tftp_send_packet) ;
		return 0 ;
	}
	return -1 ;
}

