/*
   dhcp.c : This file contains functions to configure the dhcp server
            through configuration manager 
   Author : Naveen.P.N.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <defs.h>
#include <kstart.h>

#include <incall.h>
#include <cfgmgr.h>

#define FREE_DHCP_SERVER_TAG_LIST 1
#define FREE_CONFIGURATION_MANAGER_TAG_LIST 1

#define PROCESS_FOR_ADD  0
#define PROCESS_FOR_EDIT 1

#define MAX_NUMBER_OF_TAGS 50
#define SIZE_OF_EACH_NODE 200
#define CM_LEASED_TIME_VALUE 51

BYTE CM_DHCP_RANGE_STRING[] = "DHCP Tag Address Range";
BYTE CM_DHCP_EXCLUSION_STRING[] = "DHCP Tag Exclusions";
BYTE CM_DHCP_BIND_STRING[] = "DHCP Tag Bindings";
BYTE CM_DHCP_OPTION_STRING[] = "DHCP Tag Properties";



#if 0 /* Not reqd now */
STRUCT_OF_STRINGS_AND_CORR_FUNCTIONS sptr_string_and_corr_functions[]
{
   {"DHCP Number of Tags", dhcp_set_tag_count}
   {"DHCP Tag Address Range", dhcp_setup_tag_definition}
   {"DHCP Tag Exclusions", dhcp_setup_tag_exclusions}
   {"DHCP Tag Bindings", dhcp_setup_tag_bindings}
   {"DHCP Tag Properties", dhcp_read_tag_options}
   {NULL, NULL}
}
#endif

enum 
{
   CM_IP_ADDRESS_FORMAT,
   CM_MAC_ADDRESS_FORMAT,
   CM_8_BIT_INTEGER_FORMAT,
   CM_16_BIT_INTEGER_FORMAT,
   CM_32_BIT_INTEGER_FORMAT
} CM_DHCP_OPTION_FORMAT;

/*                   Local ProtoTypes ...                              */
BYTE initialise_dhcp_address_range_list();
DHCP_TAG_DEFINITION* cm_build_address_range_list();
BYTE cm_clone_tag_list();
BYTE cm_build_exclusion_list_for_a_tag(USHORT tag_number);
BYTE cm_build_binding_list_for_a_tag(USHORT tag_number);
BYTE cm_build_option_list_for_a_tag(USHORT tag_number);
void cm_free_tag_list(BYTE free_cfgmgr_list);
void cm_free_exclusion_list(USHORT tag_number, BYTE free_cfgmgr_list);
void cm_free_binding_list(USHORT tag_number, BYTE free_cfgmgr_list);
void cm_free_option_list(USHORT tag_number, BYTE free_cfgmgr_list);
void cm_free_exclusion_list_for_a_tag(USHORT tag_number);
void cm_free_binding_list_for_a_tag(USHORT tag_number);
void cm_free_option_list_for_a_tag(USHORT tag_number);
void cm_convert_tag_list_to_cnf_format(PARAMETER_NODE* sptr_last_node);
BYTE cm_form_dhcp_range_string(USHORT tag_number, PARAMETER_NODE** sptr_last_node);
BYTE cm_form_dhcp_exclusion_string(USHORT tag_number, PARAMETER_NODE** sptr_last_node);
BYTE cm_form_dhcp_bindings_string(USHORT tag_number, PARAMETER_NODE** sptr_last_node);
BYTE cm_form_dhcp_options_string(USHORT tag_number, PARAMETER_NODE** sptr_last_node);
void cm_clean_up_existing_dhcp_configuration();
BYTE cm_matched_with_tag_strings(BYTE* lhs_string);
enum CM_DHCP_RANGE_INSERTION_RETURN_TYPE cm_insert_dhcp_address_range(CM_STRUCT_DHCP_RANGE* sptr_range_to_insert);
enum CM_DHCP_RANGE_INSERTION_RETURN_TYPE cm_save_the_range_at_the_tag(CM_STRUCT_DHCP_RANGE* sptr_range_to_insert, USHORT tag_number, BYTE called_for_edit);
enum CM_DHCP_RANGE_INSERTION_RETURN_TYPE cm_validate_dhcp_range(CM_STRUCT_DHCP_RANGE* sptr_range_to_insert);
enum CM_DHCP_RANGE_INSERTION_RETURN_TYPE cm_validate_exclusions(CM_STRUCT_DHCP_RANGE* sptr_range_to_insert);
BYTE cm_range_clashed_with_other_range(BYTE called_for, USHORT tag_number, CM_STRUCT_DHCP_RANGE* sptr_range_to_insert);
enum CM_DHCP_RANGE_INSERTION_RETURN_TYPE cm_add_range_in_tag_list(CM_STRUCT_DHCP_RANGE* sptr_range_to_insert, USHORT tag_number, BYTE called_for_edit);
enum CM_DHCP_RANGE_INSERTION_RETURN_TYPE cm_add_exclusion_to_a_tag(CM_STRUCT_DHCP_RANGE* sptr_range_to_insert, USHORT new_tag_number);
enum CM_DHCP_RANGE_INSERTION_RETURN_TYPE cm_inherit_properties_from_default_tag(USHORT tag_number);
enum CM_DHCP_RANGE_INSERTION_RETURN_TYPE cm_check_binding_of_prev_range(USHORT tag_number, CM_STRUCT_DHCP_RANGE* sptr_range_to_insert);
BYTE ip_address_lies_in_the_range(ULONG ip_address_lower, ULONG ip_address_higher,
                                  ULONG exclude_ip_address_lower, ULONG exclude_ip_address_higher,
                                  ULONG given_ip_address);
enum CM_DHCP_RANGE_DELETION_RETURN_TYPE cm_delete_dhcp_address_range(CM_STRUCT_DHCP_RANGE* sptr_range_to_delete);
BYTE cm_find_the_tag_number_using_range(CM_STRUCT_DHCP_RANGE* sptr_range_to_search, USHORT* tag_number);
enum CM_DHCP_RANGE_DELETION_RETURN_TYPE cm_delete_dhcp_address_range_by_tag_number(USHORT tag_number);
enum CM_DHCP_RANGE_INSERTION_RETURN_TYPE cm_edit_dhcp_address_range(CM_STRUCT_DHCP_RANGE* sptr_old_range, CM_STRUCT_DHCP_RANGE* sptr_new_range);
BYTE cm_get_dhcp_address_range(USHORT tag_number, CM_STRUCT_DHCP_RANGE* sptr_range);
enum CM_DHCP_BIND_RETURN_TYPE cm_bind_mac_and_ip_address(USHORT tag_number, CM_STRUCT_DHCP_BIND* sptr_bind_info);
BYTE cm_get_first_dhcp_binding(USHORT tag_number, CM_STRUCT_DHCP_BIND* sptr_bind_info);
BYTE cm_get_next_dhcp_binding(USHORT tag_number, CM_STRUCT_DHCP_BIND* sptr_bind);
BYTE cm_get_dhcp_binding(USHORT tag_number, BYTE bind_number, CM_STRUCT_DHCP_BIND* sptr_bind_info);
enum CM_DHCP_DELETE_OPTION_RETURN_TYPE cm_delete_dhcp_range_option_accordingly(USHORT tag_number, CM_STRUCT_DHCP_OPTION* sptr_dhcp_option_to_delete, BYTE called_for_delete);


BYTE cm_ip_address_already_bound(BYTE* ip_address);
BYTE cm_mac_address_already_bound(BYTE* mac_address_buffer);

enum CM_DHCP_UNBIND_RETURN_TYPE        cm_unbind_mac_and_ip_address(USHORT tag_number, CM_STRUCT_DHCP_BIND* sptr_unbind_info);
enum CM_DHCP_INSERT_OPTION_RETURN_TYPE cm_insert_dhcp_range_option(USHORT tag_number, CM_STRUCT_DHCP_OPTION* sptr_dhcp_option_to_insert);
enum CM_DHCP_DELETE_OPTION_RETURN_TYPE cm_delete_dhcp_range_option(USHORT tag_number, CM_STRUCT_DHCP_OPTION* sptr_dhcp_option_to_delete);
enum CM_DHCP_INSERT_OPTION_RETURN_TYPE cm_edit_dhcp_range_option(USHORT tag_number, CM_STRUCT_DHCP_OPTION* sptr_dhcp_old_option, CM_STRUCT_DHCP_OPTION* sptr_dhcp_new_option);

BYTE     cm_get_first_dhcp_range_option(USHORT tag_number, CM_STRUCT_DHCP_OPTION* sptr_option);
BYTE     cm_get_dhcp_range_option_by_number(USHORT tag_number, BYTE option_number, CM_STRUCT_DHCP_OPTION* sptr_option);
BYTE     cm_get_dhcp_range_option_by_option(USHORT tag_number, USHORT option, CM_STRUCT_DHCP_OPTION* sptr_option);
BYTE     cm_get_next_dhcp_range_option(USHORT tag_number, CM_STRUCT_DHCP_OPTION* sptr_option);
USHORT   cm_get_total_number_of_ranges();
USHORT   cm_get_total_number_of_bindings_in_a_range(USHORT tag_number);
USHORT   cm_get_number_of_option_values(USHORT option);
void     convert_string(BYTE* original_string, BYTE* converted_string, BYTE option);
void     revert_string(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string, BYTE option);
void     convert_ip_address(BYTE* original_string, BYTE* converted_string);
void     convert_char_string(BYTE* original_string, BYTE* converted_string);
void     convert_32_bit_number(BYTE* original_string, BYTE* converted_string);
void     convert_16_bit_number(BYTE* original_string, BYTE* converted_string);
void     convert_8_bit_number(BYTE* original_string, BYTE* converted_string);
void     revert_ip_address(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string);
void     revert_char_string(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string);
void     revert_32_bit_number(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string);
void     revert_16_bit_number(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string);
void     revert_8_bit_number(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string);
ULONG    dot2ulong(BYTE* ip_address_in_dot_format);
BYTE     validate_dns_name(BYTE* buffer, BYTE max_length, ULONG lower_limit, ULONG upper_limit);
BYTE     validate_integer(BYTE* buffer, BYTE max_digits, ULONG lower_limit, ULONG upper_limit);
BYTE     cm_validate_ip_address(BYTE* buffer, BYTE max_length, ULONG lower_limit, ULONG upper_limit);
BYTE     prev_given_bind(BYTE new_value, BYTE save);
BYTE     prev_given_option(BYTE new_value, BYTE save);
USHORT   divide_by_2_and_round(USHORT value);
BYTE     odd(USHORT value);
BYTE     cm_is_ip_address_excluded(USHORT tag_number, BYTE* ip_address);
void     cm_convert_mac_address_to_string_format(USHORT length, BYTE* mac_address, BYTE* mac_address_in_string_format);
void     get_hex_format_for_ip_address(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string);
void     get_hex_format_char_for_string(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string);
void     get_hex_format_for_32_bit_integer(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string);
void     get_hex_format_for_16_bit_integer(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string);
void     get_hex_format_for_8_bit_integer(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string);
         
/*                   ... Local ProtoTypes                              */

/* Externs... */

extern void    get_lhs_string_of_the_node(PARAMETER_NODE* node, BYTE* lhs_string);
extern BYTE    is_valid_string_to_read_data(BYTE* buffer);
/* ...Externs */

/* -------------------------- Libs ... ----------------------------------*/
CM_OPTION_VALIDATING_TABLE option_validating_table[] = 
{
/*   {1,  validate_ip_address_mask,   0,  0,   0},*/
   {3,  cm_validate_ip_address,        0,  0,   0},
   {6,  cm_validate_ip_address,        0,  0,   0},
   {15, validate_dns_name,       0,  0,   0},
   {22, validate_integer,        5,  576, 0xffff},
   {23, validate_integer,        3,  0,   0xff},
   {26, validate_integer,        5,  68,  0xffff},
   {37, validate_integer,        3,  0,   0xff},
   {51, validate_integer,        10, 600, 0xffffffff},
   {0,  NULL, 0 , 0, 0}
};


CM_OPTION_CONVERSION_TABLE option_conversion_table[] = 
{
   {1,      convert_ip_address,          revert_ip_address   },
   {3,      convert_ip_address,          revert_ip_address   },
   {6,      convert_ip_address,          revert_ip_address   },
   {15,     convert_char_string ,        revert_char_string  },
   {22,     convert_16_bit_number,       revert_16_bit_number},
   {23,     convert_8_bit_number,        revert_8_bit_number },
   {26,     convert_16_bit_number,       revert_16_bit_number},
   {37,     convert_8_bit_number,        revert_8_bit_number },
   {51,     convert_32_bit_number,       revert_32_bit_number},
   {0,      NULL,                    NULL}
};

/* -------------------------- ...Libs  ----------------------------------*/




/* ----------------------------------------------------------------------
 The following function checks whether the pointer sptr_tag_list
 in DHCP_SERVER class is NULL or not , 
   if it is NULL means the dhcp server is diabled hence the linked list is 
      not constructed. In this case we have to construct the linked list.
   if it is not NULL then it means linked list has been formed and we have to
      replicate the existing linked list.
------------------------------------------------------------------------ */

BYTE cm_initialise_dhcp_address_range_list()
{
#if 0
   /* Check whether the linked list is already formed by dhcp server code */
   if (dhcp_server.enabled == FALSE)
   {
      /* Now we have to construct the  address range list */
      sptr_to_start_of_dhcp_address_range = build_address_range_list()
   }
   else
   {
      /* Now we have to duplicate the existing address range list */
      sptr_to_start_of_dhcp_address_range = clone_existing_address_range_list();
   }
#endif

   /* Now we have to duplicate the existing address range list */
   if (!cm_clone_tag_list())
      return FALSE;
   return TRUE;
}

#if 0 /* Not Reqd now */

DHCP_TAG_DEFINITION* cm_build_address_range_list()
{
   PARAMETER_NODE *dhcp_section_node, *next_section;
   BYTE lhs_string[50], index;

   dhcp_section_node = sptr_to_each_section_in_the_configuration[CM_DHCP_SECTION];
   next_section = sptr_to_each_section_in_the_configuration[CM_DHCP_SECTION+1];

   while (dhcp_section_node != next_section)
   {
      /* Get the LHS string */
      get_lhs_string_of_the_node(dhcp_section_node, lhs_string);
      index = 0;

      /* Check whether the string in the library matches this node's
         LHS if so then corr function
      */   
      while (sptr_string_and_corr_functions[index].setup_function != NULL)
      {
         if (strcmpi(sptr_string_and_corr_functions[index].string, lhs_string) == 0)
         {
            if (index == 0)
            {
               /* if the function is to setup number of tags then temporarily
                  enable dhcp server afterwards disable it */
               dhcp_server.enabled = TRUE;
            }

            sptr_string_and_corr_functions[index].validation_function(lhs_string, 0, 0, 0);

            if (index == 0)
            {
               /* just before calling function dhcp server was enabled now it
                  is must to disable it 
                  But after calling the function if dhcp server is disabled then
                  it means to say that there is insuffcient memory and hence
                  return.
               */
               if (dhcp_server.enabled == FALSE)
               {
                  return NULL;
               }
               dhcp_server.enabled = FALSE;
            }
         }

         /* No Match! Goto next string in the lib */
         index ++;
      }

      /* Goto Next Node */
      dhcp_section_node = dhcp_section_node->next;
   }
   return dhcp_server.sptr_tag_list;
}
#endif /* if 0  */

/* The following creates a mirror image of list that is there already in 
   dhcp server module. That list is nothing but tag list with 
   pointer to exclusion list, binding list and option list from each tag
*/
BYTE cm_clone_tag_list()
{
   USHORT tags;

   /* Try to allocate memory for the whole tag list */
   cfgmgr_class.sptr_tag_list = (DHCP_TAG_DEFINITION *) calloc (MAX_NUMBER_OF_TAGS, sizeof (DHCP_TAG_DEFINITION));
   if (cfgmgr_class.sptr_tag_list == NULL)
   {
      return FALSE;
   }
   
   /* Take the memory image of the dhcp server tag list to cfgmgr's tag list
      we will take care of pointers later.
   */
   memcpy(&cfgmgr_class.sptr_tag_list[0], &dhcp_server.sptr_tag_list[0], MAX_NUMBER_OF_TAGS * sizeof (DHCP_TAG_DEFINITION));
   cfgmgr_class.total_number_of_tags = dhcp_server.number_of_tags;


   /* Now we will start taking care of each pointers in each tag */
   for (tags = 0 ; tags < cfgmgr_class.total_number_of_tags; ++tags)
   {

      /* Initialize the pointer to NULL, initially assuming that 
         exclusion list , bind list  and option list are empty
      */
      cfgmgr_class.sptr_tag_list[tags].sptr_exclusion_list = NULL;
      cfgmgr_class.sptr_tag_list[tags].sptr_option_list = NULL;
      cfgmgr_class.sptr_tag_list[tags].sptr_binding_list = NULL;

      if (cm_build_exclusion_list_for_a_tag(tags) == FALSE)
      {
         return FALSE;
      }

      if (cm_build_binding_list_for_a_tag(tags) == FALSE)
      {
         return FALSE;
      }

      if (cm_build_option_list_for_a_tag(tags) == FALSE)
      {
         return FALSE;
      }
   }
   return TRUE;
}


BYTE cm_build_exclusion_list_for_a_tag(USHORT tag_number)
{
   DHCP_EXCLUSION_ENTRY *sptr_exclusion, *sptr_temp_exclusion;

   /* Start building the exclusion list if it exists */
   sptr_exclusion = dhcp_server.sptr_tag_list[tag_number].sptr_exclusion_list;
   while (sptr_exclusion)
   {
      sptr_temp_exclusion = (DHCP_EXCLUSION_ENTRY *) malloc (sizeof (DHCP_EXCLUSION_ENTRY));
      if (sptr_temp_exclusion == NULL)
      {
         /* if there is no memory then it is reqd to remove the whole
            duplicate list and inform the caller that there was
            insufficient memory 
         */
         cm_free_tag_list(FREE_CONFIGURATION_MANAGER_TAG_LIST);
         return FALSE;
      }
      memcpy(sptr_temp_exclusion, sptr_exclusion, sizeof(DHCP_EXCLUSION_ENTRY));

      /* Insert the new Exclusion entry at the start of the list*/
      sptr_temp_exclusion->sptr_next_entry = cfgmgr_class.sptr_tag_list[tag_number].sptr_exclusion_list;
      cfgmgr_class.sptr_tag_list[tag_number].sptr_exclusion_list = sptr_temp_exclusion;

      sptr_exclusion = sptr_exclusion->sptr_next_entry;
   }
   return TRUE;
}

BYTE cm_build_binding_list_for_a_tag(USHORT tag_number)
{
   DHCP_STATIC_BINDING_ENTRY *sptr_bind, *sptr_temp_bind;

   /* Now we start building the binding list */
   sptr_bind = dhcp_server.sptr_tag_list[tag_number].sptr_binding_list;
   while (sptr_bind)
   {
      sptr_temp_bind = (DHCP_STATIC_BINDING_ENTRY *) malloc (sizeof(DHCP_STATIC_BINDING_ENTRY) + sptr_bind->mac_address_length  - 1 );
      if (sptr_temp_bind == NULL)
      {
         /* if there is no memory then it is reqd to remove the whole
            duplicate list and inform the caller that there was
            insufficient memory 
         */
         cm_free_tag_list(FREE_CONFIGURATION_MANAGER_TAG_LIST);
         return FALSE;
      }
      memcpy(sptr_temp_bind, sptr_bind, sizeof(DHCP_STATIC_BINDING_ENTRY) + sptr_bind->mac_address_length - 1);

      /* Insert the new Binding entry at the start of the list*/
      sptr_temp_bind->sptr_next_entry = cfgmgr_class.sptr_tag_list[tag_number].sptr_binding_list;
      cfgmgr_class.sptr_tag_list[tag_number].sptr_binding_list = sptr_temp_bind;

      /* Goto Next binding */
      sptr_bind = sptr_bind->sptr_next_entry;
   }
   return TRUE;
}

BYTE cm_build_option_list_for_a_tag(USHORT tag_number)
{
   DHCP_OPTION_ENTRY *sptr_option, *sptr_temp_option;

   /* Now Start Building the option List */
   sptr_option = dhcp_server.sptr_tag_list[tag_number].sptr_option_list;
   while (sptr_option)
   {
      sptr_temp_option = (DHCP_OPTION_ENTRY *) malloc (sptr_option->option_length + 2 + sizeof (DHCP_OPTION_ENTRY *));
      if (sptr_temp_option == NULL)
      {
         /* if there is no memory then it is reqd to remove the whole
            duplicate list and inform the caller that there was
            insufficient memory 
         */
         cm_free_tag_list(FREE_CONFIGURATION_MANAGER_TAG_LIST);
         return FALSE;
      }
      memcpy(sptr_temp_option, sptr_option, sptr_option->option_length + 2 + sizeof (DHCP_OPTION_ENTRY *));

      /* Insert the new Option entry at the start of the list*/
      sptr_temp_option->sptr_next_option_entry = cfgmgr_class.sptr_tag_list[tag_number].sptr_option_list;
      cfgmgr_class.sptr_tag_list[tag_number].sptr_option_list = sptr_temp_option;

      /* Goto Next Option*/
      sptr_option = sptr_option->sptr_next_option_entry;
   }
   return TRUE;
}

void cm_free_tag_list(BYTE free_cfgmgr_class_list)
{  
   USHORT  tags;
      
   /* Now we will start freeing each list in each tag */
   for (tags = 0 ; tags < cfgmgr_class.total_number_of_tags ; ++tags)
   {
      cm_free_exclusion_list(tags, free_cfgmgr_class_list);
      cm_free_binding_list(tags, free_cfgmgr_class_list);
      cm_free_option_list(tags, free_cfgmgr_class_list);
   }
   free(cfgmgr_class.sptr_tag_list);
   cfgmgr_class.sptr_tag_list = NULL;
}


void cm_free_exclusion_list(USHORT tag_number, BYTE free_cfgmgr_class_list)
{
   DHCP_EXCLUSION_ENTRY *sptr_exclusion, *sptr_temp_exclusion;
   
   /* Start freeing the exclusion list if it exists */
   if (free_cfgmgr_class_list)
      sptr_exclusion = cfgmgr_class.sptr_tag_list[tag_number].sptr_exclusion_list;
   else
      sptr_exclusion = dhcp_server.sptr_tag_list[tag_number].sptr_exclusion_list;

   while (sptr_exclusion)
   {
      sptr_temp_exclusion = sptr_exclusion;
      sptr_exclusion = sptr_exclusion->sptr_next_entry;
      free(sptr_temp_exclusion);
   }
}

void cm_free_binding_list(USHORT tag_number, BYTE free_cfgmgr_class_list)
{
   DHCP_STATIC_BINDING_ENTRY *sptr_bind, *sptr_temp_bind;

   /* Now we start freeing the binding list */
   if (free_cfgmgr_class_list)
      sptr_bind = cfgmgr_class.sptr_tag_list[tag_number].sptr_binding_list;
   else
      sptr_bind = dhcp_server.sptr_tag_list[tag_number].sptr_binding_list;

   while (sptr_bind)
   {
      sptr_temp_bind = sptr_bind;
      sptr_bind = sptr_bind->sptr_next_entry;
      free(sptr_temp_bind);
   }
}

void cm_free_option_list(USHORT tag_number, BYTE free_cfgmgr_class_list)
{
   DHCP_OPTION_ENTRY *sptr_option, *sptr_temp_option;

   /* Now Start Freeing the option List */
   if (free_cfgmgr_class_list)
      sptr_option = cfgmgr_class.sptr_tag_list[tag_number].sptr_option_list;
   else
      sptr_option = dhcp_server.sptr_tag_list[tag_number].sptr_option_list;

   while (sptr_option)
   {
      sptr_temp_option = sptr_option;
      sptr_option = sptr_option->sptr_next_option_entry;
      free(sptr_temp_option);
   }
}



/* I have written a function for this because in future 
   there can be exclusion list of more then one node,
   Currently we are assuming that there will be only one exclusion
*/
void cm_free_exclusion_list_for_a_tag(USHORT tag_number)
{
   free(cfgmgr_class.sptr_tag_list[tag_number].sptr_exclusion_list);
}

/* The Following function frees all the bindings of a range */
void cm_free_binding_list_for_a_tag(USHORT tag_number)
{
   DHCP_STATIC_BINDING_ENTRY *sptr_bind, *sptr_temp_bind;
   
   sptr_bind = cfgmgr_class.sptr_tag_list[tag_number].sptr_binding_list;

   while(sptr_bind)
   {
      sptr_temp_bind = sptr_bind ;
      sptr_bind = sptr_bind->sptr_next_entry;
      free(sptr_temp_bind);
   }
}

/* The Following function frees all the option of a range */
void cm_free_option_list_for_a_tag(USHORT tag_number)
{
   DHCP_OPTION_ENTRY *sptr_option, *sptr_temp_option;
   
   sptr_option = cfgmgr_class.sptr_tag_list[tag_number].sptr_option_list;

   while(sptr_option)
   {
      sptr_temp_option = sptr_option ;
      sptr_option = sptr_option->sptr_next_option_entry;
      free(sptr_temp_option);
   }
}

/*
   This function takes the tag list of the configuration manager and
   converts and each tag and its corr bindings, excludes and options
   to cnf file format.
   This function requires the pointer to the last node of the DHCP
   section in the master-linked-list (This linked list is formed by 
   cfgmgr_class by reading configuration which is there in the flash
   in .cnf format)
*/
void cm_convert_tag_list_to_cnf_format(PARAMETER_NODE* node)
{
   /* Consider each of the tags and its corr. exclusion, binding and options
      and form the string one by one.
   */
   USHORT tags;
   PARAMETER_NODE* sptr_last_node;
   BYTE new_value[4];

   cm_clean_up_existing_dhcp_configuration();

   sptr_last_node = sptr_to_each_section_in_the_configuration[CM_DHCP_SECTION+1];
   sptr_last_node = sptr_last_node->previous;

   for(tags = 0 ; tags < cfgmgr_class.total_number_of_tags ; ++tags)
   {  
      cm_form_dhcp_range_string(tags, &sptr_last_node);
      cm_form_dhcp_exclusion_string(tags, &sptr_last_node);
      cm_form_dhcp_options_string(tags, &sptr_last_node);
      cm_form_dhcp_bindings_string(tags, &sptr_last_node);
   }

   /* Lets do one more thing here Setting the number of tags to correct value */
   sprintf(new_value,"%d\0", cfgmgr_class.total_number_of_tags);
   set_parameter_by_name(CM_DHCP_SECTION, 
                         "DHCP Number of Tags",
                         CM_NO_PORT_PARAMETER_PRESENT,
                         CM_STRING_FORMAT,
                         (void*)new_value);

}

BYTE cm_form_dhcp_range_string(USHORT tag_number, PARAMETER_NODE** sptr_last_node)
{
   PARAMETER_NODE* node;
   BYTE lower_ip_address[CM_IP_ADDRESS_LENGTH], higher_ip_address[CM_IP_ADDRESS_LENGTH], subnet_mask[CM_IP_ADDRESS_LENGTH];

   node = (PARAMETER_NODE*)malloc(SIZE_OF_EACH_NODE);
   if (node == NULL)
   {
      return FALSE;
   }

   /* Convert Lower IP Address to DOT Format */
   ulong_to_dot_format(lower_ip_address, cfgmgr_class.sptr_tag_list[tag_number].lower_ip_address);

   /* Convert Higher IP Address to DOT Format */
   ulong_to_dot_format(higher_ip_address, cfgmgr_class.sptr_tag_list[tag_number].higher_ip_address);

   /* Convert SubNet Mask to DOT Format */
   ulong_to_dot_format(subnet_mask, cfgmgr_class.sptr_tag_list[tag_number].subnet_mask);

   sprintf(node->parameter, "%s = %02d,%s,%s,%s\n\0",  
           CM_DHCP_RANGE_STRING, 
           tag_number,
           lower_ip_address,
           higher_ip_address,
           subnet_mask
          );
   /* Linked List insertion Stuff!! */
   node->next = (*sptr_last_node)->next;
   node->previous = *sptr_last_node;
   (*sptr_last_node)->next = node;

   /* Make the newly inserted node as last node because it was inserted 
      at the end of linked list
   */
   *sptr_last_node = node;

   return(TRUE);
}

BYTE cm_form_dhcp_exclusion_string(USHORT tag_number, PARAMETER_NODE** sptr_last_node)
{
   PARAMETER_NODE* sptr_node;
   BYTE lower_ip_address[CM_IP_ADDRESS_LENGTH], higher_ip_address[CM_IP_ADDRESS_LENGTH];
   DHCP_EXCLUSION_ENTRY* sptr_exclusion;

   sptr_exclusion = cfgmgr_class.sptr_tag_list[tag_number].sptr_exclusion_list;
   while (sptr_exclusion)
   {
      /* Form the String using this exclusion */
      sptr_node = (PARAMETER_NODE*)malloc(SIZE_OF_EACH_NODE);
      if (sptr_node == NULL)
      {
         return FALSE;
      }
      /* Convert Lower Exclusion IP Address to DOT Format */
      ulong_to_dot_format(lower_ip_address, sptr_exclusion->lower_ip_address);

      /* Convert Higher Exclusion IP Address to DOT Format */
      ulong_to_dot_format(higher_ip_address, sptr_exclusion->higher_ip_address);

      sprintf(sptr_node->parameter, "%s = %02d,%s,%s\n\0",
              CM_DHCP_EXCLUSION_STRING,
              tag_number,
              lower_ip_address,
              higher_ip_address
             );

      /* Linked List insertion Stuff!! */
      sptr_node->next = (*sptr_last_node)->next;
      sptr_node->previous = *sptr_last_node;
      (*sptr_last_node)->next = sptr_node;

      /* Make the newly inserted node as last node because it was inserted 
         at the end of linked list
      */
      *sptr_last_node = sptr_node;
      
      /* Goto Next Exclusion if one exists */
      sptr_exclusion = sptr_exclusion->sptr_next_entry;
   }
   return(TRUE);
}

BYTE cm_form_dhcp_bindings_string(USHORT tag_number, PARAMETER_NODE** sptr_last_node)
{
   PARAMETER_NODE* sptr_node;
   BYTE ip_address[CM_IP_ADDRESS_LENGTH], mac_address_in_string_format[50];
   DHCP_STATIC_BINDING_ENTRY* sptr_bind;

   sptr_bind = cfgmgr_class.sptr_tag_list[tag_number].sptr_binding_list;
   while (sptr_bind)
   {
      /* Form the String using this binding*/
      sptr_node = (PARAMETER_NODE*)malloc(SIZE_OF_EACH_NODE);

      if (sptr_node == NULL)
      {
         return FALSE;
      }

      /* Convert IP Address to DOT Format */
      ulong_to_dot_format(ip_address, sptr_bind->ip_address);

      cm_convert_mac_address_to_string_format(sptr_bind->mac_address_length, sptr_bind->mac_address, mac_address_in_string_format);
      sprintf(sptr_node->parameter, "%s = %02d,%02s,%02X,%s\n\0",
              CM_DHCP_BIND_STRING,
              tag_number,
              ip_address,
              sptr_bind->mac_address_length,
              mac_address_in_string_format
             );

      /* Linked List insertion Stuff!! */
      sptr_node->next = (*sptr_last_node)->next;
      sptr_node->previous = *sptr_last_node;
      (*sptr_last_node)->next = sptr_node;

      /* Make the newly inserted node as last node because it was inserted 
         at the end of linked list
      */
      *sptr_last_node = sptr_node;
      
      /* Goto Next Binding if one exists */
      sptr_bind = sptr_bind->sptr_next_entry;
   }
   return(TRUE);
}

BYTE cm_form_dhcp_options_string(USHORT tag_number, PARAMETER_NODE** sptr_last_node)
{
   PARAMETER_NODE* sptr_node;
   DHCP_OPTION_ENTRY* sptr_option;
   BYTE option_in_string_format[100];

   sptr_option = cfgmgr_class.sptr_tag_list[tag_number].sptr_option_list;
   while (sptr_option)
   {
      /* The following code is for SERVER ID option type added by
         DHCP Server Module. We are not suppose to save that in the 
         Flash
      */
      if (sptr_option->option_type == SERVER_ID)
      {
         sptr_option = sptr_option->sptr_next_option_entry;
         continue;   
      }

      /* Form the String using this binding*/
      sptr_node = (PARAMETER_NODE*)malloc(SIZE_OF_EACH_NODE);

      if (sptr_node == NULL)
      {
         return FALSE;
      }

      /* Convert IP Address to DOT Format */
   
/*      revert_string(sptr_option, option_in_string_format, sptr_option->option_type);*/

      switch(sptr_option->option_type)
      {
         case 1:
         case 3:
         case 6:
            get_hex_format_for_ip_address(sptr_option, option_in_string_format);
            break;
         case 15:
            /* It is required to count 00 as the part of the Option Data */
            sptr_option->option_length++;
            get_hex_format_char_for_string(sptr_option, option_in_string_format);
            break;
         case 22:
         case 26:
            get_hex_format_for_16_bit_integer(sptr_option, option_in_string_format);
            break;
         case 23:
         case 37:
            get_hex_format_for_8_bit_integer(sptr_option, option_in_string_format);
            break;
         case 51:
            get_hex_format_for_32_bit_integer(sptr_option, option_in_string_format);
            break;

      };

      sprintf(sptr_node->parameter, "%s = %02d,%02X,%02X,%s\n\0",
              CM_DHCP_OPTION_STRING,
              tag_number,
              sptr_option->option_type,
              sptr_option->option_length,
              option_in_string_format
             );

      /* Linked List insertion Stuff!! */
      sptr_node->next = (*sptr_last_node)->next;
      sptr_node->previous = *sptr_last_node;
      (*sptr_last_node)->next = sptr_node;
		         		      
      /* Make the newly inserted node as last node because it was inserted 
         at the end of linked list
      */
      *sptr_last_node = sptr_node;
      
      /* Goto Next Option if one exists */
      sptr_option = sptr_option->sptr_next_option_entry;
   }
   return(TRUE);
}



/* For Cleaning the current Tags in the master-linked-list i think
   it is better to delete strings of all tags
*/   
void cm_clean_up_existing_dhcp_configuration()
{
   PARAMETER_NODE* sptr_dhcp_section, *sptr_next_section, *sptr_temp;
   BYTE lhs_string[200];

   sptr_dhcp_section = sptr_to_each_section_in_the_configuration[CM_DHCP_SECTION];
   sptr_next_section = sptr_to_each_section_in_the_configuration[CM_DHCP_SECTION+1];

   /* Start searching for string from DHCP section until we reach next 
      section
   */
   while (sptr_dhcp_section != sptr_next_section)
   {
      /* Get the LHS String of the Node */
      get_lhs_string_of_the_node(sptr_dhcp_section, lhs_string);
      if (cm_matched_with_tag_strings(lhs_string))
      {
         /* Save Which node to scan next */
         sptr_temp = sptr_dhcp_section->next;

         /* Remove the Current Node becaue it is a tag string */
         sptr_dhcp_section->previous->next = sptr_dhcp_section->next;
         sptr_dhcp_section->next->previous = sptr_dhcp_section->previous;
         free(sptr_dhcp_section);

         /* Adjust dhcp pointer to point the correct node */
         sptr_dhcp_section = sptr_temp;
         continue;
      }

      /* Point to the next node */
      sptr_dhcp_section = sptr_dhcp_section->next;
   }
}



/* This function returns TRUE if the given lhs string matches any of the
   TAG String (Rangs, Exclusion, Bind or Option)
*/
BYTE cm_matched_with_tag_strings(BYTE* lhs_string)
{
   /* Does the LHS String Match the Range String */
   if (!strcmpi(lhs_string, CM_DHCP_RANGE_STRING))
      return TRUE;

   /* Does the LHS String Match the Exclusion String */
   if (!strcmpi(lhs_string, CM_DHCP_EXCLUSION_STRING))
      return TRUE;

   /* Does the LHS String Match the Bind String */
   if (!strcmpi(lhs_string, CM_DHCP_BIND_STRING))
      return TRUE;

   /* Does the LHS String Match the Option String */
   if (!strcmpi(lhs_string, CM_DHCP_OPTION_STRING))
      return TRUE;

   /* None of them match i.e. this DHCP String is not a Tag String */
   return FALSE;
}



/* This Function gets DHCP Address Range as well as exclusion 
   It first validates LowerIPAddress, UpperIPAddress and SubnetMask 
   then  it validates LowerIPAddressExclusion and UpperIPAddressExclusion
   If both validation are through then 

      * it adds DHCP IP Address Range into the Free Tag Block

      * check for the clash of this range with the other

      * it creates a node for exclusion and "exclusion_pointer" from the
        above block points to this node

      * Inherits all the properties of default Tag i.e. 0 (Zero)
         It is reqd to search tag 0's properties and add those
         properties to newly added Address Range (i.e. Tag)

      * at the end increment the "total_number_of_tags" field in the 
        cfgmgr_class class.
*/


enum CM_DHCP_RANGE_INSERTION_RETURN_TYPE cm_insert_dhcp_address_range(CM_STRUCT_DHCP_RANGE* sptr_range_to_insert)
{
   return(cm_save_the_range_at_the_tag(sptr_range_to_insert, cfgmgr_class.total_number_of_tags, PROCESS_FOR_ADD));
}

enum CM_DHCP_RANGE_INSERTION_RETURN_TYPE cm_save_the_range_at_the_tag(CM_STRUCT_DHCP_RANGE* sptr_range_to_insert, USHORT tag_number, BYTE called_for)
{
   enum CM_DHCP_RANGE_INSERTION_RETURN_TYPE return_value;

   /* Validate the lower ip address , higher ip address and subnet mask */
   return_value = cm_validate_dhcp_range(sptr_range_to_insert);

   if (return_value != CM_DHCP_INSERTION_SUCCESS)
   {
      /* Something is wrong out these parameters let's simply inform the
         caller.
      */
      return return_value;
   }

   /* Validation of the 3 fields are over , Now we have to validate
      exclusion if there is one 
   */
   return_value = cm_validate_exclusions(sptr_range_to_insert);

   if (return_value != CM_DHCP_INSERTION_SUCCESS)
   {
      /* Something went wrong, inform the caller */
      return return_value;
   }

   if (called_for)
   {
      return_value = cm_check_binding_of_prev_range(tag_number, sptr_range_to_insert);
      if (return_value != CM_DHCP_INSERTION_SUCCESS)
      {
         return return_value;
      }
   }

   /* Now Check for the clash of this range with other range (taking
       care of exclusion)
   */
   if (cm_range_clashed_with_other_range(called_for, tag_number, sptr_range_to_insert) == TRUE)
   {
      return CM_DHCP_INSERTION_FAILED_INVALID_RANGE;
   }

   /* Now all Validation Stuffs are over lets continue adding... */
   return_value = cm_add_range_in_tag_list(sptr_range_to_insert, tag_number, called_for);
   
   if (return_value != CM_DHCP_INSERTION_SUCCESS)
   {
      /* Failed to add into tag list inform the caller */
      return return_value;
   }

   /* Adding Range was successful, now we have to add exclusion */
   return_value = cm_add_exclusion_to_a_tag(sptr_range_to_insert, tag_number);
   
   if (return_value != CM_DHCP_INSERTION_SUCCESS)
   {
      /* Failed to add exclusion inform the caller */
      return return_value;
   }

   if (called_for == PROCESS_FOR_ADD)
   {
      /* Now inherit all the properties of the default tag */
      return_value = cm_inherit_properties_from_default_tag(tag_number);

      if (return_value != CM_DHCP_INSERTION_SUCCESS)
      {
         /* Failed to inherit properties from default tag , inform caller */

         /* Free Exclusion List in this tag */
         cm_free_exclusion_list_for_a_tag(tag_number);
         return return_value;
      }
   }
   /* After all these stuffs are success lets increment the total number of
      tags entry in the cfgmgr_class class
   */
   if (called_for == PROCESS_FOR_ADD)
      cfgmgr_class.total_number_of_tags++;

   return CM_DHCP_INSERTION_SUCCESS;
}


enum CM_DHCP_RANGE_INSERTION_RETURN_TYPE cm_validate_dhcp_range(CM_STRUCT_DHCP_RANGE* sptr_range_to_insert)
{
   ULONG lower_ip_address_ulong, higher_ip_address_ulong;

   if (!is_an_ip_address(sptr_range_to_insert->ip_address_lower))
      return CM_DHCP_INSERTION_FAILED_INVALID_LOWER_IP_ADDRESS;

   if (!is_an_ip_address(sptr_range_to_insert->ip_address_higher))
      return CM_DHCP_INSERTION_FAILED_INVALID_HIGHER_IP_ADDRESS;

   if (!is_an_ip_address_mask(sptr_range_to_insert->ip_address_higher, sptr_range_to_insert->net_mask))
      return CM_DHCP_INSERTION_FAILED_INVALID_SUBNET_MASK;
   
   /* Now Check whether both the IP Addresses lie on the same network 
      Please do not confuse with the function name.
      The following function checks whether both the ip address lie on
      same network or not.
   */
   if (local_ip_address_and_remote_ip_address_on_same_network(
                     sptr_range_to_insert->ip_address_lower,
                     sptr_range_to_insert->ip_address_higher,
                     sptr_range_to_insert->net_mask) == FALSE)
      return CM_DHCP_INSERTION_FAILED_INVALID_RANGE;

   /* Lower IP Address should be less then Higher IP Address */
   lower_ip_address_ulong  = dot2ulong(sptr_range_to_insert->ip_address_lower);
   higher_ip_address_ulong = dot2ulong(sptr_range_to_insert->ip_address_higher);
   if (lower_ip_address_ulong > higher_ip_address_ulong)
      return CM_DHCP_INSERTION_FAILED_INVALID_HIGHER_IP_ADDRESS;
      
   return CM_DHCP_INSERTION_SUCCESS;
}

enum CM_DHCP_RANGE_INSERTION_RETURN_TYPE cm_validate_exclusions(CM_STRUCT_DHCP_RANGE* sptr_range_to_insert)
{
   ULONG lower_ip_address_ulong, higher_ip_address_ulong,
         exclude_lower_ip_address_ulong, exclude_higher_ip_address_ulong;

   /* If both the exclusion ip addresses are empty then it means to say that
      there is no exclusion that's why we return with SUCCESS
   */
   if (strlen(sptr_range_to_insert->exclude_ip_address_lower) == 0 &&
       strlen(sptr_range_to_insert->exclude_ip_address_higher) == 0
      )
      return CM_DHCP_INSERTION_SUCCESS;

   if (!is_an_ip_address(sptr_range_to_insert->exclude_ip_address_lower))
   {
      return CM_DHCP_INSERTION_FAILED_INVALID_LOWER_IP_ADDRESS_EXCLUDE;
   }

   if (!is_an_ip_address(sptr_range_to_insert->exclude_ip_address_higher))
   {
      return CM_DHCP_INSERTION_FAILED_INVALID_HIGHER_IP_ADDRESS_EXCLUDE;
   }

   /* Now we shall convert all 4 ip addresses and then compare whether the
      exclusion is within the range or not 
   */
   lower_ip_address_ulong = dot2ulong(sptr_range_to_insert->ip_address_lower);
   higher_ip_address_ulong = dot2ulong(sptr_range_to_insert->ip_address_higher);
   exclude_lower_ip_address_ulong = dot2ulong(sptr_range_to_insert->exclude_ip_address_lower);
   exclude_higher_ip_address_ulong = dot2ulong(sptr_range_to_insert->exclude_ip_address_higher);

   /* Lower IP Address of Exclusion should be greater then or equal to
      Lower IP Address of Range and Higher IP Address of Exclusion should
      less then or equal to Higher IP Address of Range and greater then 
      Lower IP Address of Exclusion
   */
   if (exclude_lower_ip_address_ulong < lower_ip_address_ulong)
      return CM_DHCP_INSERTION_FAILED_INVALID_LOWER_IP_ADDRESS_EXCLUDE;

   if (exclude_higher_ip_address_ulong > higher_ip_address_ulong)
      return CM_DHCP_INSERTION_FAILED_INVALID_HIGHER_IP_ADDRESS_EXCLUDE;

   if (exclude_higher_ip_address_ulong < exclude_lower_ip_address_ulong)
      return CM_DHCP_INSERTION_FAILED_INVALID_HIGHER_IP_ADDRESS_EXCLUDE;

   if ((exclude_higher_ip_address_ulong == exclude_lower_ip_address_ulong) &&
       (higher_ip_address_ulong == lower_ip_address_ulong) && 
       (exclude_lower_ip_address_ulong == lower_ip_address_ulong)
      )
      return CM_DHCP_INSERTION_FAILED_INVALID_EXCLUSION;


   if ((exclude_higher_ip_address_ulong == higher_ip_address_ulong) &&
       (exclude_lower_ip_address_ulong == lower_ip_address_ulong))
      return CM_DHCP_INSERTION_FAILED_INVALID_EXCLUSION;

   return CM_DHCP_INSERTION_SUCCESS;
}

/* The following function checks whether the given range clashes with
   any other range (it also checks whether the range overlaps with 
   exclusion of other range)
*/
BYTE cm_range_clashed_with_other_range(BYTE called_for, USHORT tag_number, CM_STRUCT_DHCP_RANGE* sptr_range_to_insert)
{
   BYTE exclusion_present_in_the_new_entry, exclusion_present_in_the_old_entry;   
   ULONG new_lower_ip_address_ulong, new_higher_ip_address_ulong,
         new_exclude_lower_ip_address_ulong, new_exclude_higher_ip_address_ulong,
         old_lower_ip_address_ulong, old_higher_ip_address_ulong,
         old_exclude_lower_ip_address_ulong, old_exclude_higher_ip_address_ulong;
   USHORT tags;

   /* Convert all the ip address of new entry to ulongs */
   new_lower_ip_address_ulong = dot2ulong(sptr_range_to_insert->ip_address_lower);
   new_higher_ip_address_ulong = dot2ulong(sptr_range_to_insert->ip_address_higher);
   exclusion_present_in_the_new_entry = strlen(sptr_range_to_insert->exclude_ip_address_lower);
   if (exclusion_present_in_the_new_entry)
   {
      new_exclude_lower_ip_address_ulong = dot2ulong(sptr_range_to_insert->exclude_ip_address_lower);
      new_exclude_higher_ip_address_ulong = dot2ulong(sptr_range_to_insert->exclude_ip_address_higher);
   }


   /* Now we have to compare each of these with the exisiting Ranges
   */
   for (tags = 1 ; tags < cfgmgr_class.total_number_of_tags ; ++tags)
   {
      if (called_for == PROCESS_FOR_EDIT)
      {
         if (tags == tag_number)
            continue;
      }
      /* Convert IP Address to Ulongs */
      old_lower_ip_address_ulong  = cfgmgr_class.sptr_tag_list[tags].lower_ip_address;
      old_higher_ip_address_ulong = cfgmgr_class.sptr_tag_list[tags].higher_ip_address;
      exclusion_present_in_the_old_entry = (cfgmgr_class.sptr_tag_list[tags].sptr_exclusion_list == NULL) ? 0 : 1;
      if (exclusion_present_in_the_old_entry)
      {
         old_exclude_lower_ip_address_ulong  = cfgmgr_class.sptr_tag_list[tags].sptr_exclusion_list->lower_ip_address;
         old_exclude_higher_ip_address_ulong = cfgmgr_class.sptr_tag_list[tags].sptr_exclusion_list->higher_ip_address;
      }

      /* If Lower IP Address of Existing Range is greater then Higher IP Address 
         of New Range then it can never Clash */
      if (old_lower_ip_address_ulong > new_higher_ip_address_ulong )
         continue;
      
      /* If Upper IP Address of Existing Range is less then Lower IP Address 
         of New Range then it can never Clash */
      if (old_higher_ip_address_ulong < new_lower_ip_address_ulong)
         continue;

      if (exclusion_present_in_the_new_entry)
      {
         if ((old_lower_ip_address_ulong  >= new_exclude_lower_ip_address_ulong) &&
             (old_higher_ip_address_ulong <= new_exclude_higher_ip_address_ulong)
            )
            continue;
      }

      if (exclusion_present_in_the_old_entry)
      {
         if ((new_lower_ip_address_ulong >= old_exclude_lower_ip_address_ulong) && 
             (new_higher_ip_address_ulong <= old_exclude_higher_ip_address_ulong)
            )
            continue;
      }
      return TRUE;
   }
   return FALSE;
}

enum CM_DHCP_RANGE_INSERTION_RETURN_TYPE cm_add_range_in_tag_list(CM_STRUCT_DHCP_RANGE* sptr_range_to_insert, USHORT tag_number, BYTE called_for)
{
   USHORT new_tag_number = tag_number;
   ULONG lower_ip_address_ulong, higher_ip_address_ulong,
         subnet_mask;

   /* Convert all the IP Addresses to ULONGs */
   lower_ip_address_ulong = dot2ulong(sptr_range_to_insert->ip_address_lower);
   higher_ip_address_ulong = dot2ulong(sptr_range_to_insert->ip_address_higher);
   subnet_mask = dot2ulong(sptr_range_to_insert->net_mask);

   /* Add the new entry to empty slot */
   cfgmgr_class.sptr_tag_list[new_tag_number].lower_ip_address =  lower_ip_address_ulong;
   cfgmgr_class.sptr_tag_list[new_tag_number].higher_ip_address = higher_ip_address_ulong;

   /* While editing do not allow to change the mask */
   if (!called_for)
      cfgmgr_class.sptr_tag_list[new_tag_number].subnet_mask = subnet_mask;

   /* Initialize the pointer to NULL */
   if (called_for == PROCESS_FOR_ADD)
   {
      cfgmgr_class.sptr_tag_list[new_tag_number].sptr_exclusion_list = NULL;
      cfgmgr_class.sptr_tag_list[new_tag_number].sptr_option_list = NULL;
      cfgmgr_class.sptr_tag_list[new_tag_number].sptr_binding_list = NULL;
   }

   /* We did our work in a best way , return success */
   return CM_DHCP_INSERTION_SUCCESS;
}

enum CM_DHCP_RANGE_INSERTION_RETURN_TYPE cm_add_exclusion_to_a_tag(CM_STRUCT_DHCP_RANGE* sptr_range_to_insert, USHORT new_tag_number)
{
   ULONG exclude_lower_ip_address_ulong, exclude_higher_ip_address_ulong;
   DHCP_EXCLUSION_ENTRY *sptr_exclusion;
   BYTE exclusion_present_in_the_new_entry;

   exclusion_present_in_the_new_entry = strlen(sptr_range_to_insert->exclude_ip_address_lower);
   if (exclusion_present_in_the_new_entry)
   {
      exclude_lower_ip_address_ulong = dot2ulong(sptr_range_to_insert->exclude_ip_address_lower);
      exclude_higher_ip_address_ulong = dot2ulong(sptr_range_to_insert->exclude_ip_address_higher);
   }

   if (exclusion_present_in_the_new_entry)
   {
      sptr_exclusion = (DHCP_EXCLUSION_ENTRY *) malloc (sizeof (DHCP_EXCLUSION_ENTRY));
      if (sptr_exclusion == NULL)
         return CM_DHCP_INSERTION_FAILED_INSUFFICIENT_MEMORY;
      sptr_exclusion->lower_ip_address  = exclude_lower_ip_address_ulong;
      sptr_exclusion->higher_ip_address = exclude_higher_ip_address_ulong;
      sptr_exclusion->sptr_next_entry = NULL;
      cfgmgr_class.sptr_tag_list[new_tag_number].sptr_exclusion_list = sptr_exclusion;
   }
   else
      cfgmgr_class.sptr_tag_list[new_tag_number].sptr_exclusion_list= NULL;


   /* We did our work in a best way , return success */
   return CM_DHCP_INSERTION_SUCCESS;
}

/* The Following function is basically inherits all the properties from the
   tag 0 .
   This function just follow the option link in the tag 0 and 
   just copies the properties from it.
*/
enum CM_DHCP_RANGE_INSERTION_RETURN_TYPE cm_inherit_properties_from_default_tag(USHORT tag_number)
{
   DHCP_OPTION_ENTRY *sptr_option, *sptr_temp_option;
   
   /* '0' Means DEFAULT TAG */
   sptr_option = cfgmgr_class.sptr_tag_list[0].sptr_option_list;
   while (sptr_option)
   {
      if (sptr_option->option_type == SERVER_ID)
      {
         sptr_option = sptr_option->sptr_next_option_entry;
         continue;
      }
      sptr_temp_option = (DHCP_OPTION_ENTRY *) malloc (sptr_option->option_length + 2 + sizeof (DHCP_OPTION_ENTRY *));
      if (sptr_temp_option == NULL)
      {
         /* if there is no memory, then it is reqd to remove the
            option list in the new tag
         */
         sptr_option = cfgmgr_class.sptr_tag_list[tag_number].sptr_option_list;
         while (sptr_option)
         {
            sptr_temp_option = sptr_option;
            sptr_option = sptr_option->sptr_next_option_entry;
            free(sptr_temp_option);
         }
         return CM_DHCP_INSERTION_FAILED_INSUFFICIENT_MEMORY;
      }

      /* Insert the new Option entry at the start of the list*/
      memcpy(sptr_temp_option, sptr_option, sptr_option->option_length + 2 + sizeof (DHCP_OPTION_ENTRY *));
      sptr_temp_option->sptr_next_option_entry = cfgmgr_class.sptr_tag_list[tag_number].sptr_option_list;
      cfgmgr_class.sptr_tag_list[tag_number].sptr_option_list = sptr_temp_option;

      /* Goto Next Option*/
      sptr_option = sptr_option->sptr_next_option_entry;
   }
   return CM_DHCP_INSERTION_SUCCESS;   
}


/* The following function checks in the binding list whether ip address
   is matching any of the ip addresses in the binding list and if that
   ip address is not there in the new range then do not allow the new 
   range to take effect
*/
enum CM_DHCP_RANGE_INSERTION_RETURN_TYPE cm_check_binding_of_prev_range(USHORT tag_number, CM_STRUCT_DHCP_RANGE* sptr_range_to_insert)
{
   DHCP_STATIC_BINDING_ENTRY *sptr_bind;
   ULONG lower_ip_address_ulong, higher_ip_address_ulong;
   ULONG exclude_lower_ip_address_ulong, exclude_higher_ip_address_ulong;
   BYTE  exclusion_present_in_the_new_range;
   
   /* Convert all IP Addresses to ULONG Format */
   lower_ip_address_ulong = dot2ulong(sptr_range_to_insert->ip_address_lower);
   higher_ip_address_ulong = dot2ulong(sptr_range_to_insert->ip_address_higher);
   exclusion_present_in_the_new_range = strlen(sptr_range_to_insert->exclude_ip_address_lower);
   if (exclusion_present_in_the_new_range)
   {
      exclude_lower_ip_address_ulong = dot2ulong(sptr_range_to_insert->exclude_ip_address_lower);
      exclude_higher_ip_address_ulong = dot2ulong(sptr_range_to_insert->exclude_ip_address_higher);
   }

   sptr_bind = cfgmgr_class.sptr_tag_list[tag_number].sptr_binding_list;
   while (sptr_bind)
   {
      if (ip_address_lies_in_the_range(lower_ip_address_ulong, higher_ip_address_ulong,
                                       exclude_lower_ip_address_ulong, exclude_higher_ip_address_ulong,
                                       sptr_bind->ip_address) == FALSE
                                       )
         return CM_DHCP_EDITING_FAILED_IP_IS_BOUND_TO_MAC_ADDRESS;
      sptr_bind = sptr_bind->sptr_next_entry;
   }
   return CM_DHCP_INSERTION_SUCCESS;
}

/* The following function checks whether the given ip address lies in a range
   (taking exclusion into consideration) or not.
*/
BYTE ip_address_lies_in_the_range(ULONG ip_address_lower, ULONG ip_address_higher,
                                  ULONG exclude_ip_address_lower, ULONG exclude_ip_address_higher,
                                  ULONG given_ip_address)
{
   if ((given_ip_address >= ip_address_lower && given_ip_address < exclude_ip_address_lower)
        || (given_ip_address <= ip_address_higher && given_ip_address > exclude_ip_address_higher)
      )
      return TRUE;

   return FALSE;
}


enum CM_DHCP_RANGE_DELETION_RETURN_TYPE cm_delete_dhcp_address_range(CM_STRUCT_DHCP_RANGE* sptr_range_to_delete)
{
   USHORT tag_to_delete;
   BYTE return_value;
   
   return_value = cm_find_the_tag_number_using_range(sptr_range_to_delete, &tag_to_delete);

   if (return_value == FALSE)
   {
      return CM_DHCP_DELETION_RANGE_NOT_FOUND;
   }


   /* Do not allow to delete 'default range' */
   if (tag_to_delete == 0)
      return CM_DHCP_DELETION_FAILED_TRYING_TO_DELETE_DEFAULT;

   return (cm_delete_dhcp_address_range_by_tag_number(tag_to_delete));
}

/* The following function gives the range , given the tag number
*/
BYTE cm_find_the_tag_number_using_range(CM_STRUCT_DHCP_RANGE* sptr_range_to_search, USHORT* tag_number)
{
   ULONG lower_ip_address_ulong, higher_ip_address_ulong;
   USHORT tags;

   lower_ip_address_ulong = dot2ulong(sptr_range_to_search->ip_address_lower);
   higher_ip_address_ulong = dot2ulong(sptr_range_to_search->ip_address_higher);
   
   /* We will start searching from begining all the tags */
   for (tags = 0 ; tags < cfgmgr_class.total_number_of_tags ; ++tags)
   {
      if ( (lower_ip_address_ulong  == cfgmgr_class.sptr_tag_list[tags].lower_ip_address) && 
           (higher_ip_address_ulong == cfgmgr_class.sptr_tag_list[tags].higher_ip_address) 
         )
         {
            *tag_number = tags;
            return TRUE;
         }
   }

   /* Failed to search tag in the given list */
   return FALSE;
}


enum CM_DHCP_RANGE_DELETION_RETURN_TYPE cm_delete_dhcp_address_range_by_tag_number(USHORT tag_number)
{

   /* Don't allow to delete 'default range/tag' */
   if (tag_number == 0)
      return CM_DHCP_DELETION_FAILED_TRYING_TO_DELETE_DEFAULT;

   /* Free the exclusion list */
   cm_free_exclusion_list_for_a_tag(tag_number);
   cm_free_option_list_for_a_tag(tag_number);

   /* Now shift the all the blocks after the block to be deleted
      by one block backwards
   */
   memmove(&cfgmgr_class.sptr_tag_list[tag_number], &cfgmgr_class.sptr_tag_list[tag_number+1], (cfgmgr_class.total_number_of_tags - tag_number - 1) * sizeof(DHCP_TAG_DEFINITION));

   cfgmgr_class.total_number_of_tags--;

   return CM_DHCP_DELETION_SUCCESS;
}


enum CM_DHCP_RANGE_INSERTION_RETURN_TYPE cm_edit_dhcp_address_range(CM_STRUCT_DHCP_RANGE* sptr_old_range, CM_STRUCT_DHCP_RANGE* sptr_new_range)
{
   USHORT tag;
   
   BYTE return_value;
   
   return_value = cm_find_the_tag_number_using_range(sptr_old_range, &tag);

   if (tag == 0)
   {
      return CM_DHCP_EDITING_FAILED_TO_CHANGE_DEFAULT;
   }

   if (return_value == FALSE)
   {
      return CM_DHCP_EDITING_FAILED_TO_GET_OLD_RANGE;
   }

   return (cm_save_the_range_at_the_tag(sptr_new_range, tag, PROCESS_FOR_EDIT));
}

BYTE cm_get_dhcp_address_range(USHORT tag_number, CM_STRUCT_DHCP_RANGE* sptr_range)
{

   if (tag_number > cfgmgr_class.total_number_of_tags-1)
      return FALSE;

   ulong_to_dot_format(sptr_range->ip_address_lower ,cfgmgr_class.sptr_tag_list[tag_number].lower_ip_address);
   ulong_to_dot_format(sptr_range->ip_address_higher ,cfgmgr_class.sptr_tag_list[tag_number].higher_ip_address);
   ulong_to_dot_format(sptr_range->net_mask ,cfgmgr_class.sptr_tag_list[tag_number].subnet_mask);

   if (cfgmgr_class.sptr_tag_list[tag_number].sptr_exclusion_list)
   {
      ulong_to_dot_format(sptr_range->exclude_ip_address_lower, cfgmgr_class.sptr_tag_list[tag_number].sptr_exclusion_list->lower_ip_address);
      ulong_to_dot_format(sptr_range->exclude_ip_address_higher, cfgmgr_class.sptr_tag_list[tag_number].sptr_exclusion_list->higher_ip_address);
   }
   else
   {
      sptr_range->exclude_ip_address_lower[0] = 0;
      sptr_range->exclude_ip_address_higher[0] = 0;
   }
   return TRUE;
}


enum CM_DHCP_BIND_RETURN_TYPE cm_bind_mac_and_ip_address(USHORT tag_number, CM_STRUCT_DHCP_BIND* sptr_bind_info)
{
   DHCP_STATIC_BINDING_ENTRY* sptr_new_bind;
   USHORT index;
   BYTE *temp_ptr;
   int ch;

   if (tag_number == 0)
   {
      return CM_DHCP_BINDING_FAILED_CAN_NOT_BIND_DEFAULT_RANGE;
   }

   /* Validate Mac Address and Length */
   
   if (sptr_bind_info->mac_address_length > 10 )
      return CM_DHCP_BINDING_FAILED_INVALID_MAC_ADDRESS_LENGTH;
   if (strlen(sptr_bind_info->mac_address_to_bind) > 2 * sptr_bind_info->mac_address_length)
      return CM_DHCP_BINDING_FAILED_MAC_ADDRESS_GREATER_THAN_LENGTH;
   if (!is_valid_ipx_network_number(sptr_bind_info->mac_address_to_bind))
      return CM_DHCP_BINDING_FAILED_INVALID_MAC_ADDRESS;

   if (!is_an_ip_address(sptr_bind_info->ip_address_to_bind))
      return CM_DHCP_BINDING_FAILED_INVALID_IP_ADDRESS;

   if (cm_ip_address_already_bound(sptr_bind_info->ip_address_to_bind))
      return CM_DHCP_BINDING_FAILED_DUPLICATED_IP_ADDRESS;

   if (cm_mac_address_already_bound(sptr_bind_info->mac_address_to_bind))
      return CM_DHCP_BINDING_FAILED_DUPLICATED_MAC_ADDRESS;

   if (cm_is_ip_address_excluded(tag_number, sptr_bind_info->ip_address_to_bind))
      return CM_DHCP_BINDING_FAILED_IP_ADDRESS_OUT_OF_RANGE;

   sptr_new_bind = (DHCP_STATIC_BINDING_ENTRY *) malloc (sizeof(DHCP_STATIC_BINDING_ENTRY) + sptr_bind_info->mac_address_length - 1);

   if (sptr_new_bind == NULL)
      return CM_DHCP_BINDING_FAILED_INSUFFICIENT_MEMORY;

   sptr_new_bind->ip_address = dot2ulong(sptr_bind_info->ip_address_to_bind);
   sptr_new_bind->mac_address_length = sptr_bind_info->mac_address_length;

/*   strcpy(sptr_new_bind->mac_address, sptr_bind_info->mac_address_to_bind);*/
   
   temp_ptr = sptr_bind_info->mac_address_to_bind;
   for ( index = 0 ; index < sptr_new_bind->mac_address_length ; ++index)
   {
      if (sptr_new_bind->mac_address_length > divide_by_2_and_round(strlen(sptr_bind_info->mac_address_to_bind)) + index)
      {
         sptr_new_bind->mac_address[index] = 0;
      }
      else
      if (odd(strlen(sptr_bind_info->mac_address_to_bind)) && (sptr_new_bind->mac_address_length == divide_by_2_and_round(strlen(sptr_bind_info->mac_address_to_bind)) + index))
      {
         sscanf(temp_ptr, "%2X", &ch);
         sptr_new_bind->mac_address[index] = (BYTE)ch/0x10;
         temp_ptr += 1;
      }
      else
      {
         sscanf(temp_ptr, "%2X", &ch);
         sptr_new_bind->mac_address[index] = (BYTE)ch;
         temp_ptr += 2;
      }
   }

   /* Insert at the front of linked list */
   sptr_new_bind->sptr_next_entry = cfgmgr_class.sptr_tag_list[tag_number].sptr_binding_list; 
   cfgmgr_class.sptr_tag_list[tag_number].sptr_binding_list = sptr_new_bind;
   
   return CM_DHCP_BINDING_SUCCESS;
}

BYTE cm_is_ip_address_excluded(USHORT tag_number, BYTE* ip_address)
{
   ULONG ip_address_ulong;
   DHCP_EXCLUSION_ENTRY *sptr_exclusion;
   
   ip_address_ulong = dot2ulong(ip_address);

   if (ip_address_ulong < cfgmgr_class.sptr_tag_list[tag_number].lower_ip_address
       || ip_address_ulong > cfgmgr_class.sptr_tag_list[tag_number].higher_ip_address)
      return TRUE;

   sptr_exclusion = cfgmgr_class.sptr_tag_list[tag_number].sptr_exclusion_list;
   if (sptr_exclusion == NULL)
      return FALSE;
   else
      if (ip_address_ulong >= sptr_exclusion->lower_ip_address && ip_address_ulong <= sptr_exclusion->higher_ip_address)
         return TRUE;

   return FALSE;
}

/* This function checks in each tag's binding list if the given ip address
   matches or not, if there is a match then that ip address is already 
   bound.
*/
BYTE cm_ip_address_already_bound(BYTE* ip_address)
{
   USHORT tags;
   ULONG ip_address_ulong;
   DHCP_STATIC_BINDING_ENTRY *sptr_bind;

   ip_address_ulong = dot2ulong(ip_address);

   for ( tags = 0 ;  tags < cfgmgr_class.total_number_of_tags ; ++tags)
   {
      sptr_bind = cfgmgr_class.sptr_tag_list[tags].sptr_binding_list;
      while (sptr_bind)
      {
         if (sptr_bind->ip_address == ip_address_ulong)
            return TRUE;
         sptr_bind = sptr_bind->sptr_next_entry;
      }
   }
   return FALSE;
}

/* This funcion is same as above function but checks for mac address binding
   rather then ip address
*/
BYTE cm_mac_address_already_bound(BYTE* mac_address_buffer)
{
   USHORT tags;
   ULONG mac_address, index, mac_address_1;
   DHCP_STATIC_BINDING_ENTRY *sptr_bind;
   BYTE temp_buffer[30], temp_string[10];

#if 0
   temp_ptr = mac_address_buffer;
   for ( index = 0 ; index < mac_address_length ; ++index)
   {
      if (sptr_new_bind->mac_address_length > round(strlen(sptr_bind_info->mac_address_to_bind)/2) + index)
      {
         sptr_new_bind->mac_address[index] = 0;
      }
      else
      if (odd(strlen(sptr_bind_info->mac_address_to_bind)) && (sptr_new_bind->mac_address_length == round(strlen(sptr_bind_info->mac_address_to_bind)/2) + index))
      {
         sscanf(temp_ptr, "%2X", &ch);
         sptr_new_bind->mac_address[index] = (BYTE)ch/10;
         temp_ptr += 1;
      }
      else
      {
         sscanf(temp_ptr, "%2X", &ch);
         sptr_new_bind->mac_address[index] = (BYTE)ch;
         temp_ptr += 2;
      }
   }
#endif
   
   sscanf(mac_address_buffer, "%X", &mac_address);

   for ( tags = 0 ;  tags < cfgmgr_class.total_number_of_tags ; ++tags)
   {
      sptr_bind = cfgmgr_class.sptr_tag_list[tags].sptr_binding_list;
      while (sptr_bind)
      {
         temp_buffer[0] = 0;
         for ( index = 0 ; index < sptr_bind->mac_address_length ; ++index )
         {
            if (sptr_bind->mac_address[index] == 0)
               strcat(temp_buffer, "00");
            else
            {
               sprintf(temp_string, "%02X\0", sptr_bind->mac_address[index]);
               strcat(temp_buffer, temp_string);
            }
         }
         sscanf(temp_buffer, "%X", &mac_address_1);


         if (mac_address == mac_address_1)
            return TRUE;
         sptr_bind = sptr_bind->sptr_next_entry;
      }
   }
   return FALSE;
}

enum CM_DHCP_UNBIND_RETURN_TYPE cm_unbind_mac_and_ip_address(USHORT tag_number, CM_STRUCT_DHCP_BIND* sptr_unbind_info)
{
   ULONG temp_ip_ulong;
   DHCP_STATIC_BINDING_ENTRY* sptr_bind, *sptr_temp_bind;

   temp_ip_ulong = dot2ulong(sptr_unbind_info->ip_address_to_bind);

   sptr_bind = cfgmgr_class.sptr_tag_list[tag_number].sptr_binding_list;

   if (sptr_bind == NULL)
      return CM_DHCP_UNBIND_FAILED;

   sptr_temp_bind = sptr_bind;
   while (sptr_bind)
   {
      if (temp_ip_ulong == sptr_bind->ip_address)
      {
         /* Found the node containing the binding to be removed */
         if (sptr_temp_bind == sptr_bind)      
         {
            cfgmgr_class.sptr_tag_list[tag_number].sptr_binding_list = sptr_bind->sptr_next_entry;
         }  
         else
         {
            sptr_temp_bind->sptr_next_entry = sptr_bind->sptr_next_entry;
         }
         free(sptr_bind);
         return CM_DHCP_UNBIND_SUCCESS;
      }
      sptr_temp_bind = sptr_bind;
      sptr_bind = sptr_bind->sptr_next_entry;
   }
   return CM_DHCP_UNBIND_FAILED;
}


BYTE cm_get_first_dhcp_binding(USHORT tag_number, CM_STRUCT_DHCP_BIND* sptr_bind_info)
{  
   if (cm_get_dhcp_binding(tag_number, 0, sptr_bind_info))
   {
      prev_given_bind(0, TRUE);
      return TRUE;
   }
   else
      return FALSE;
}

BYTE cm_get_next_dhcp_binding(USHORT tag_number, CM_STRUCT_DHCP_BIND* sptr_bind_info)
{
   BYTE prev_bind_number;

   prev_bind_number = prev_given_bind(0, FALSE);

   prev_bind_number++; /* We want next one */

   if (cm_get_dhcp_binding(tag_number, prev_bind_number, sptr_bind_info))
   {
      prev_given_bind(prev_bind_number, TRUE);
      return TRUE;
   }
   else
      return FALSE;
}


BYTE cm_get_dhcp_binding(USHORT tag_number, BYTE bind_number, CM_STRUCT_DHCP_BIND* sptr_bind_info)
{  
   DHCP_STATIC_BINDING_ENTRY* sptr_bind;
   USHORT index;
   BYTE temp_string[50];

   sptr_bind = cfgmgr_class.sptr_tag_list[tag_number].sptr_binding_list;

   if (sptr_bind == NULL)
      return FALSE;
   else
   {
      for ( index = 0 ; index < bind_number ; ++index )
      {
         if (sptr_bind)
            sptr_bind = sptr_bind->sptr_next_entry;
         else
            return FALSE;
      }

      ulong_to_dot_format(sptr_bind_info->ip_address_to_bind, sptr_bind->ip_address);
      sptr_bind_info->mac_address_length = sptr_bind->mac_address_length;

      sptr_bind_info->mac_address_to_bind[0] = 0;
      for ( index = 0 ; index < sptr_bind->mac_address_length ; ++index )
      {
         if (sptr_bind->mac_address[index] == 0)
            strcat(sptr_bind_info->mac_address_to_bind, "00");
         else
         {
            sprintf(temp_string, "%02X\0", sptr_bind->mac_address[index]);
            strcat(sptr_bind_info->mac_address_to_bind, temp_string);
         }
      }

/*      strcpy(sptr_bind_info->mac_address_to_bind, sptr_bind->mac_address);*/
      
      return TRUE;
   }
}

/* The Following function checks whether the option already exists or not
    currently we are supporting only one option value per option
    If it already exists means return with error.
    Otherwise validate the new option value according to option.
    After validation just insert the new option
*/
enum CM_DHCP_INSERT_OPTION_RETURN_TYPE cm_insert_dhcp_range_option(USHORT tag_number, CM_STRUCT_DHCP_OPTION* sptr_dhcp_option_to_insert)
{
   DHCP_OPTION_ENTRY *sptr_option;
   CM_STRUCT_DHCP_OPTION sptr_temp_option;
   USHORT index;
   BYTE converted_option_value[100], *temp_ptr;
   int ch;

   /* Check for Duplication Option */
   if (cm_get_dhcp_range_option_by_option(tag_number, sptr_dhcp_option_to_insert->option, &sptr_temp_option))
   {
      /* Yahh, there is a option already return error */
      return CM_DHCP_INSERTING_OPTION_FAILED_DUPLICATE_ENTRY;
   }
   
   index = 0;

   while (option_validating_table[index].validation_function != NULL)
   {
      if (option_validating_table[index].option == sptr_dhcp_option_to_insert->option)
      {
         if (option_validating_table[index].validation_function(sptr_dhcp_option_to_insert->option_value, option_validating_table[index].max_digits, option_validating_table[index].lower_limit, option_validating_table[index].upper_limit) == FALSE)
            return CM_DHCP_INSERTING_OPTION_FAILED_INVALID_DATA;
      }
      index ++;
   }
  
   convert_string(sptr_dhcp_option_to_insert->option_value, converted_option_value, sptr_dhcp_option_to_insert->option);

   sptr_option = (DHCP_OPTION_ENTRY *) malloc (strlen(converted_option_value) + 2 + sizeof (DHCP_OPTION_ENTRY *));
   if (sptr_option == NULL)
   {
      return CM_DHCP_EDITING_OPTION_FAILED_INSUFFICIENT_MEMORY;
   }

   /* Store new values */
   sptr_option->option_type = sptr_dhcp_option_to_insert->option;
   sptr_option->option_length = strlen(converted_option_value)/2;

   temp_ptr = converted_option_value;
   for ( index = 0 ; index < strlen(converted_option_value)/2 ; ++index)
   {
      sscanf(temp_ptr, "%2X", &ch);
      sptr_option->option_data.byte_array[index] = (BYTE)ch;
      temp_ptr += 2;
   }

    sptr_option->option_data.byte_array[index] = 0;

   /* Insert the new node in the front of the list */
   sptr_option->sptr_next_option_entry = cfgmgr_class.sptr_tag_list[tag_number].sptr_option_list;
   cfgmgr_class.sptr_tag_list[tag_number].sptr_option_list = sptr_option;

   return CM_DHCP_INSERTING_OPTION_SUCCESS; 
}

enum CM_DHCP_DELETE_OPTION_RETURN_TYPE cm_delete_dhcp_range_option(USHORT tag_number, CM_STRUCT_DHCP_OPTION* sptr_dhcp_option_to_delete)
{
   return(cm_delete_dhcp_range_option_accordingly(tag_number, sptr_dhcp_option_to_delete, TRUE));
}

enum CM_DHCP_DELETE_OPTION_RETURN_TYPE cm_delete_dhcp_range_option_accordingly(USHORT tag_number, CM_STRUCT_DHCP_OPTION* sptr_dhcp_option_to_delete, BYTE called_for_delete)
{
   DHCP_OPTION_ENTRY *sptr_option, *sptr_temp_option;
  
   /* If the Option is Leased Time then do not delete */
   if (sptr_dhcp_option_to_delete->option == CM_LEASED_TIME_VALUE && called_for_delete)
      return CM_DHCP_DELETING_OPTION_FAILED_CANNOT_DELETE;

   sptr_temp_option = sptr_option = cfgmgr_class.sptr_tag_list[tag_number].sptr_option_list;

   while (sptr_option)
   {
      if (sptr_option->option_type == sptr_dhcp_option_to_delete->option)
      {
         if (sptr_option == sptr_temp_option)
         {
            cfgmgr_class.sptr_tag_list[tag_number].sptr_option_list = sptr_option->sptr_next_option_entry;
         }
         else
         {  
            sptr_temp_option->sptr_next_option_entry = sptr_option->sptr_next_option_entry;
         }
         free(sptr_option);
         return CM_DHCP_DELETING_OPTION_SUCCESS;
      }

      sptr_temp_option = sptr_option;
      sptr_option = sptr_option->sptr_next_option_entry;
   }
   return CM_DHCP_DELETING_OPTION_FAILED_OPTION_NOT_FOUND;
}

enum CM_DHCP_INSERT_OPTION_RETURN_TYPE cm_edit_dhcp_range_option(USHORT tag_number, CM_STRUCT_DHCP_OPTION* sptr_dhcp_old_option, CM_STRUCT_DHCP_OPTION* sptr_dhcp_new_option)
{
   enum CM_DHCP_INSERT_OPTION_RETURN_TYPE return_value;

   if (cm_delete_dhcp_range_option_accordingly(tag_number, sptr_dhcp_old_option, FALSE) != CM_DHCP_DELETING_OPTION_SUCCESS)
   {
      /* Failed to delete old option value */
      return CM_DHCP_INSERTING_OPTION_FAILED_INVALID_DATA;
   }

   /* Try to insert new option value */
   if ((return_value = cm_insert_dhcp_range_option(tag_number, sptr_dhcp_new_option)) != CM_DHCP_INSERTING_OPTION_SUCCESS)
   {
      /* No! We failed to insert new value , lets store the old value */
      cm_insert_dhcp_range_option(tag_number, sptr_dhcp_old_option);
      return return_value;
   }
   return CM_DHCP_INSERTING_OPTION_SUCCESS;
}

BYTE cm_get_first_dhcp_range_option(USHORT tag_number, CM_STRUCT_DHCP_OPTION* sptr_option)
{
   if (cm_get_dhcp_range_option_by_number(tag_number, 0, sptr_option))
   {
      prev_given_option(0, TRUE);
      return TRUE;   
   }
   else
      return FALSE;
}

BYTE cm_get_next_dhcp_range_option(USHORT tag_number, CM_STRUCT_DHCP_OPTION* sptr_option)
{
   BYTE prev_option;

   prev_option = prev_given_option(0, FALSE);
   prev_option ++;

   if (cm_get_dhcp_range_option_by_number(tag_number, prev_option, sptr_option))
   {
      prev_given_option(prev_option, TRUE);
      return TRUE;   
   }
   else
      return FALSE;
}
/* The following number gives the option value and etc given the 
   number BUT not the option
*/
BYTE cm_get_dhcp_range_option_by_number(USHORT tag_number, BYTE option_number, CM_STRUCT_DHCP_OPTION* sptr_option)
{
   BYTE converted_option_value[100], index, temp_buffer[50], temp_string[50];
   DHCP_OPTION_ENTRY *sptr_temp_option;

   if (cfgmgr_class.sptr_tag_list[tag_number].sptr_option_list == NULL)
      return FALSE;

   sptr_temp_option = cfgmgr_class.sptr_tag_list[tag_number].sptr_option_list;

   for (index = 0 ; index < option_number ; ++index)
   {
      if (sptr_temp_option)
         sptr_temp_option = sptr_temp_option->sptr_next_option_entry;
      else
         return FALSE;
   }

   sptr_option->option = sptr_temp_option->option_type;

   
   temp_buffer[0] = 0;
   for ( index = 0 ; index < sptr_temp_option->option_length ; ++index )
   {
      sprintf(temp_string, "%d\0", sptr_temp_option->option_data.byte_array[index]);
      strcat(temp_buffer, temp_string);
   }
   revert_string(sptr_temp_option, converted_option_value, sptr_option->option);
   strcpy(sptr_option->option_value, converted_option_value);
   sptr_option->position = 1;
   return TRUE;
}

/* The following number gives the option value and etc given the option
*/
BYTE cm_get_dhcp_range_option_by_option(USHORT tag_number, USHORT option, CM_STRUCT_DHCP_OPTION* sptr_option)
{
   BYTE converted_option_value[100];
   DHCP_OPTION_ENTRY *sptr_temp_option;

   if (cfgmgr_class.sptr_tag_list[tag_number].sptr_option_list == NULL)
      return FALSE;

   sptr_temp_option = cfgmgr_class.sptr_tag_list[tag_number].sptr_option_list;

   while (sptr_temp_option)
   {
      if (sptr_temp_option->option_type == option)
         break;
      sptr_temp_option = sptr_temp_option->sptr_next_option_entry;
   }

   if (!sptr_temp_option)
      return FALSE;

   sptr_option->option = sptr_temp_option->option_type;

/*   temp_buffer[0] = 0;
   for ( index = 0 ; index < sptr_temp_option->option_length ; ++index )
   {
      sprintf(temp_string, "%d\0", sptr_temp_option->option_data.byte_array[index]);
      strcat(temp_buffer, temp_string);
   }
*/
   revert_string(sptr_temp_option, converted_option_value, sptr_option->option);
   strcpy(sptr_option->option_value, converted_option_value);
   sptr_option->position = 1;

   return TRUE;
}

/*-----------------------------------------------------------------------*/

USHORT   cm_get_total_number_of_ranges()
{
   return (cfgmgr_class.total_number_of_tags);
}

USHORT   cm_get_total_number_of_bindings_in_a_range(USHORT tag_number)
{
   DHCP_STATIC_BINDING_ENTRY *sptr_bind;
   USHORT number_of_bindings = 0;

   sptr_bind = cfgmgr_class.sptr_tag_list[tag_number].sptr_binding_list;

   if (sptr_bind == NULL)
      return number_of_bindings;

   while (sptr_bind)
   {
      sptr_bind = sptr_bind->sptr_next_entry;
      ++number_of_bindings;
   }
   return number_of_bindings;
}

USHORT cm_get_number_of_option_values(USHORT option)
{
   return 1;
}

void convert_string(BYTE* original_string, BYTE* converted_string, BYTE option)
{  
   BYTE index;

   index = 0;
   while (option_conversion_table[index].convert_function!= NULL)
   {
      if (option_conversion_table[index].option == option)
      {
         option_conversion_table[index].convert_function(original_string, converted_string);
         return;
      }
      index ++;
   }
}


void revert_string(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string, BYTE option)
{  
   BYTE index;

   index = 0;
   while (option_conversion_table[index].revert_function != NULL)
   {
      if (option_conversion_table[index].option == option)
      {
         option_conversion_table[index].revert_function(sptr_option, converted_string);
         return;
      }
      index ++;
   }
}

void convert_ip_address(BYTE* original_string, BYTE* converted_string)
{
   USHORT ip_address_array[4];
	sscanf (original_string,
           "%03hu.%03hu.%03hu.%03hu",
   		  (int *)(&ip_address_array[0]),
           (int *)(&ip_address_array[1]),
		     (int *)(&ip_address_array[2]),
           (int *)(&ip_address_array[3])
          );
   sprintf(converted_string, "%02X%02X%02X%02X\0", (BYTE)ip_address_array[0],
                                                   (BYTE)ip_address_array[1],
                                                   (BYTE)ip_address_array[2],
                                                   (BYTE)ip_address_array[3]);
}


void convert_char_string(BYTE* original_string, BYTE* converted_string)
{
   BYTE index, temp_string[4];

   converted_string[0] = 0;
   for( index = 0 ; index < strlen(original_string) ; ++index)
   {
      sprintf(temp_string, "%02X", original_string[index]);
      strcat(converted_string, temp_string);
   }
}

void convert_32_bit_number(BYTE* original_string, BYTE* converted_string)
{
   ULONG temp_value;
   
   temp_value = atoi(original_string);

   sprintf(converted_string, "%08X", temp_value);
}

void convert_16_bit_number(BYTE* original_string, BYTE* converted_string)
{
   USHORT temp_value;
   
   temp_value = atoi(original_string);

   sprintf(converted_string, "%04X", temp_value);
}

void convert_8_bit_number(BYTE* original_string, BYTE* converted_string)
{
   BYTE temp_value;
   
   temp_value = atoi(original_string);

   sprintf(converted_string, "%02X", temp_value);
}




void revert_ip_address(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string)
{
   BYTE  index, original_string[40], temp_string[30];
   
#if 0  /* Some Junk Code */
	sscanf (original_string,   "%02x", (int *)(&ip_address_array[0]));
   sscanf (original_string+2, "%02x", (int *)(&ip_address_array[1]));
	sscanf (original_string+4, "%02x", (int *)(&ip_address_array[2]));
   sscanf (original_string+6, "%02x", (int *)(&ip_address_array[3]));
#endif 

   original_string[0] = 0;
   for ( index = 0 ; index < sptr_option->option_length ; ++index )
   {
      sprintf(temp_string, "%d\0", sptr_option->option_data.byte_array[index]);
      strcat(original_string, temp_string);
      if (index < sptr_option->option_length-1)
         strcat(original_string, ".");
   }

/*   for ( index  = 0 ; index < strlen(original_string) ; ++index)
   {
      temp_buffer[0] = original_string[index];
      temp_buffer[1] = 0;
      sscanf(temp_buffer, "%X", &temp_value);
      ip_address_array[index] = (BYTE)temp_value;
   }
*/
/*   sprintf(converted_string, "%d.%d.%d.%d\0", (BYTE)ip_address_array[0], (BYTE)ip_address_array[1], (BYTE)ip_address_array[2], (BYTE)ip_address_array[3]);*/
   strcpy(converted_string, original_string);
}

void revert_char_string(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string)
{
#if 0
#define NO_OF_CHARS_PER_BYTE 2

   BYTE index, temp_ptr[5], temp_buffer[50], *original_string;
   ULONG temp_char;

   original_string = sptr_option->option_data.byte_array;
   converted_string[0] = 0;
   for ( index = 0 ; index < strlen(original_string)/2 ; ++index)
   {
      strncpy(temp_ptr, original_string + index * 2, NO_OF_CHARS_PER_BYTE);
      temp_ptr[NO_OF_CHARS_PER_BYTE] = 0;
      sscanf(temp_ptr, "%02X", &temp_char);
      sprintf(temp_buffer, "%c\0", temp_char);
      strcat(converted_string, temp_buffer);
   }
   converted_string[index] = 0;
#endif
   strcpy(converted_string, sptr_option->option_data.byte_array);
}

void revert_32_bit_number(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string)
{
   ULONG temp_value;
   BYTE* original_string;
   
   original_string = sptr_option->option_data.byte_array;
   sscanf(original_string, "%08X", &temp_value);

   sprintf(converted_string, "%lu", sptr_option->option_data._ulong);
}

void revert_16_bit_number(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string)
{
   ULONG temp_value;
   BYTE* original_string;
   
   original_string = sptr_option->option_data.byte_array;
   sscanf(original_string, "%04X", &temp_value);

   sprintf(converted_string, "%d", (USHORT)sptr_option->option_data._ushort);

}

void revert_8_bit_number(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string)
{
   ULONG temp_value;
   BYTE* original_string;
   
   original_string = sptr_option->option_data.byte_array;
   sscanf(original_string, "%02X", &temp_value);

   sprintf(converted_string, "%d\0", sptr_option->option_data._byte);

}

ULONG dot2ulong(BYTE* ip_address_in_dot_format)
{
   USHORT ip_address_array[4];
   
	sscanf (ip_address_in_dot_format,
           "%03hu.%03hu.%03hu.%03hu",
   		  (int *)(&ip_address_array[0]),
           (int *)(&ip_address_array[1]),
		     (int *)(&ip_address_array[2]),
           (int *)(&ip_address_array[3])
          );

   return(convert_4_bytes_to_ulong
                   ((BYTE)ip_address_array[0],
                    (BYTE)ip_address_array[1],
                    (BYTE)ip_address_array[2],
                    (BYTE)ip_address_array[3])
         );
}

#if 0
BYTE validate_ip_address_mask(BYTE* buffer, BYTE max_length, ULONG lower_limit, ULONG upper_limit)
{
   return(is_an_ip_address_mask(buffer));
}
#endif

BYTE cm_validate_ip_address(BYTE* buffer, BYTE max_length, ULONG lower_limit, ULONG upper_limit)
{
   return(is_an_ip_address(buffer));
}

BYTE validate_dns_name(BYTE* buffer, BYTE max_length, ULONG lower_limit, ULONG upper_limit)
{
   return TRUE;
}

BYTE validate_integer(BYTE* buffer, BYTE max_digits, ULONG lower_limit, ULONG upper_limit)
{
   ULONG value;

   if (strlen(buffer) > max_digits)
      return FALSE;

   if (!is_valid_string_to_read_data(buffer))
      return FALSE;

   value = atoi(buffer);

   if ((value < lower_limit) || (value > upper_limit))
      return FALSE;

   return TRUE;
}

BYTE prev_given_bind(BYTE new_value, BYTE save)
{
   static BYTE value = 0;

   if (save)
      value = new_value;
   else
      return value;
}


BYTE prev_given_option(BYTE new_value, BYTE save)
{
   static BYTE value = 0;

   if (save)
      value = new_value;
   else
      return value;
}

USHORT divide_by_2_and_round(USHORT value)
{
   return (value+1)/2;
}

BYTE odd(USHORT value)
{
   return value%2;
}

void cm_convert_mac_address_to_string_format(USHORT length, BYTE* mac_address, BYTE* mac_address_in_string_format)
{
   BYTE temp_string[40];
   USHORT index;
   
   mac_address_in_string_format[0] = 0;
   for ( index = 0 ; index < length ; ++index )
   {
      if (mac_address[index] == 0)
         strcat(mac_address_in_string_format, "00");
      else
      {
         sprintf(temp_string, "%02X\0", mac_address[index]);
         strcat(mac_address_in_string_format, temp_string);
      }
   }
}


void get_hex_format_for_ip_address(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string)
{
   BYTE  index, original_string[40], temp_string[30];
   
   original_string[0] = 0;
   for ( index = 0 ; index < sptr_option->option_length ; ++index )
   {
      sprintf(temp_string, "%02X\0", sptr_option->option_data.byte_array[index]);
      strcat(original_string, temp_string);
   }
   strcpy(converted_string, original_string);
}

void get_hex_format_char_for_string(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string)
{
#define NO_OF_CHARS_PER_BYTE 2

   BYTE index, temp_string[50], *original_string;

   original_string = sptr_option->option_data.byte_array;
   converted_string[0] = 0;
   for ( index = 0 ; index < strlen(original_string) ; ++index)
   {
      sprintf(temp_string, "%02X", original_string[index]);
      strcat(converted_string, temp_string);
   }
   strcat(converted_string ,"00");
}

void get_hex_format_for_32_bit_integer(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string)
{
   sprintf(converted_string, "%08X", sptr_option->option_data._ulong);
}

void get_hex_format_for_16_bit_integer(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string)
{
   sprintf(converted_string, "%04X", (USHORT)sptr_option->option_data._ushort);
}

void get_hex_format_for_8_bit_integer(DHCP_OPTION_ENTRY* sptr_option, BYTE* converted_string)
{
   sprintf(converted_string, "%02X\0", sptr_option->option_data._byte);
}

