#include	"defs.h"

/**************************************************************************/
/* Module to handle the initial transactions to establish the compression */
/* and uncompress the received frames and update the database.            */
/**************************************************************************/

#include <string.h>
#include "fr.h"

#include "bsd.h"

/****************************************************************************/
void check_dlci_mode_1_state (USHORT port_number)
{
	USHORT pvc_index;
	DLCI_LIST_ENTRY *sptr_current_connection_entry;

	pvc_index = 0x0000;

	sptr_current_connection_entry = (DLCI_LIST_ENTRY *) frame_relay.port[port_number].dlci_list.sptr_forward_link;

	while (pvc_index < NUMBER_OF_FRAME_RELAY_PVCS && sptr_current_connection_entry != NULL)
	{
		if (sptr_current_connection_entry->compression == TRUE) {

			if (sptr_current_connection_entry->mode_1_state == DISABLED_STATE) {
				if (is_frame_relay_link_up (port_number) &&
					(sptr_current_connection_entry->mib.frCircuitState == ACTIVE_FRAME_RELAY_CIRCUIT_STATE)) {

					sptr_current_connection_entry->mode_1_state = INITIALIZATION_STATE;
#ifdef DEBUG
					printf ("FRF9: In disabled state\r\n");
#endif
					send_mode_1_request (port_number, sptr_current_connection_entry->dlci);

					sptr_current_connection_entry->got_control_code = MODE1_NO_CODE;
					sptr_current_connection_entry->sent_control_code = MODE1_REQUEST_CODE;

					sptr_current_connection_entry->p_retry_count = 0;
					sptr_current_connection_entry->p_retry_timer = 0;
				}
			}
			else if (sptr_current_connection_entry->mode_1_state == INITIALIZATION_STATE) {
				if (++sptr_current_connection_entry->p_retry_timer == P_RETRY_TIMER) {
					if (++sptr_current_connection_entry->p_retry_count == P_RETRY_COUNT) {
						sptr_current_connection_entry->mode_1_state = DISABLED_STATE;
					}
					else {
#ifdef DEBUG
						printf ("FRF9: In Initialization state\r\n");
#endif
						send_mode_1_request (port_number, sptr_current_connection_entry->dlci);
						sptr_current_connection_entry->sent_control_code = MODE1_REQUEST_CODE;
						sptr_current_connection_entry->p_retry_timer = 0;
					}
				}
			}
			else if (sptr_current_connection_entry->mode_1_state == OPERATION_STATE) {
				check_reset_requests_and_acks (sptr_current_connection_entry->dlci);
			}
		}

		++pvc_index;

		sptr_current_connection_entry	= sptr_current_connection_entry->links.sptr_forward_link;
	}
}

/****************************************************************************/
enum BOOLEAN is_frf9_frame (BYTE *sptr_frame, USHORT *usptr_number_of_bytes_received)
{
	FRF9_DCP_DATA_PDU_HEADER	*sptr_dcp_header;

	sptr_dcp_header = (FRF9_DCP_DATA_PDU_HEADER *)sptr_frame;

	if ((* usptr_number_of_bytes_received) < sizeof(FRF9_DCP_DATA_PDU_HEADER))
		return FALSE;

	if (sptr_dcp_header->control != 0x03)
		return FALSE;

	if (sptr_dcp_header->nlpid != FRF9_NLPID)
		return FALSE;

	return TRUE;
}

/***************************************************************************/
/* If it is a command frame, handle it and return TRUE. Else, if it is a */
/* data frame, decompress the frame and adjust it as a Q.933 frame relay */
/* frame and hand over to the modules to take care of passing to stacks. */
/****************************************************************************/

enum BOOLEAN handle_frf9_frame (USHORT port_number, BYTE **sptr_frame,
		USHORT *usptr_number_of_bytes_received)
{
	USHORT	cur_dlci;
	DLCI_LIST_ENTRY *sptr_dlci_entry;
	FRF9_DCP_DATA_PDU_HEADER	*sptr_dcp_header;
	FRF9_DCPCP_MODE1_PDU_FORMAT	*sptr_dcpcp_pdu;

	sptr_dcp_header = (FRF9_DCP_DATA_PDU_HEADER *)(*sptr_frame);

	if (sptr_dcp_header->dcp_header.control_or_data) {	/* control PDU */
		convert_dlci_to_ushort (&sptr_dcp_header->dlci);

		cur_dlci = sptr_dcp_header->dlci._ushort;

		sptr_dlci_entry = get_dlci_connection_entry (port_number, cur_dlci);
		if (sptr_dlci_entry == NULL)
			return FALSE;

		sptr_dcpcp_pdu = (FRF9_DCPCP_MODE1_PDU_FORMAT *)
								((*sptr_frame) + sizeof(FRF9_DCP_DATA_PDU_HEADER));

		if (sptr_dlci_entry->mode_1_state == OPERATION_STATE) {
			if (sptr_dcpcp_pdu->request_response.code == MODE1_REQUEST_CODE) {
#ifdef DEBUG
				printf ("FRF9: Got Mode_1 request in OPERATION state, on DLCI - %d\r\n", cur_dlci);
#endif

				send_mode_1_request (port_number, sptr_dlci_entry->dlci);

				send_mode_1_response (port_number, sptr_dlci_entry->dlci);

				sptr_dlci_entry->sent_control_code = MODE1_RESPONSE_CODE;

				sptr_dlci_entry->p_retry_count = 0;
				sptr_dlci_entry->p_retry_timer = 0;
			}
			return FALSE;
		}

		if (sptr_dcpcp_pdu->request_response.code == MODE1_REQUEST_CODE) {
#ifdef DEBUG
			printf ("FRF9: Got Mode_1 request on DLCI - %d\r\n", cur_dlci);
#endif
			if (sptr_dlci_entry->got_control_code == MODE1_NO_CODE)
				sptr_dlci_entry->got_control_code = MODE1_REQUEST_CODE;
			send_mode_1_response (port_number, cur_dlci);
			sptr_dlci_entry->sent_control_code = MODE1_RESPONSE_CODE;
		}
		else if (sptr_dcpcp_pdu->request_response.code == MODE1_RESPONSE_CODE) {
#ifdef DEBUG
			printf ("FRF9: Got Mode_1 response on DLCI - %d\r\n", cur_dlci);
#endif
			sptr_dlci_entry->got_control_code = MODE1_RESPONSE_CODE;
		}

		/* check the configuration options */

		if ((sptr_dlci_entry->got_control_code == MODE1_RESPONSE_CODE)
			&& (sptr_dlci_entry->sent_control_code == MODE1_RESPONSE_CODE)) {

#if 1
			printf ("FRF9: Going to OPERATION state on DLCI - %d\r\n", cur_dlci);
#endif
			sptr_dlci_entry->mode_1_state = OPERATION_STATE;

			BSD_initialize_class (sptr_dlci_entry->bsd_context_id);
		}

		return FALSE;
	}

	/* handle the data PDU */
	return (handle_frf9_data_frame (port_number, sptr_frame, usptr_number_of_bytes_received));
}

/****************************************************************************/
enum BOOLEAN handle_frf9_data_frame (USHORT port_number, BYTE **sptr_frame,
		USHORT *usptr_number_of_bytes_received)
{
	USHORT	cur_dlci;
	USHORT	data_len;
	BYTE		*newbuf;
	BYTE		*sptr_frf_frame;
	DLCI_LIST_ENTRY *sptr_dlci_entry;
	FRF9_DCP_DATA_PDU_HEADER	*sptr_dcp_header;
	FRF9_DCP_DATA_PAYLOAD		*sptr_dcp_data_pdu;

	sptr_dcp_header = (FRF9_DCP_DATA_PDU_HEADER *)(*sptr_frame);

	convert_dlci_to_ushort (&sptr_dcp_header->dlci);

	cur_dlci = sptr_dcp_header->dlci._ushort;

	sptr_dlci_entry = get_dlci_connection_entry (port_number, cur_dlci);
	if (sptr_dlci_entry == NULL)
		return FALSE;

	/* if a data packet is received not in operation_state, */
	/* send a reset_request packet to reset the database.   */
	if (sptr_dlci_entry->mode_1_state != OPERATION_STATE) {
#ifdef DEBUG
		printf ("FRF9: Received data packet without reaching OPERATION state on DLCI - %d\r\n", cur_dlci);
#endif

		/* send a reset request packet to the other side */
		send_frf9_reset_request (port_number, cur_dlci);

		return FALSE;
	}

	/* if a reset_request is receieved, reset the compression database send */
	/* a reset_ack and ignore all the data till reset_request is cleared.   */
	if (sptr_dcp_header->dcp_header.reset_request) {
		if (sptr_dlci_entry->sent_reset_ack == FALSE) {
#if 1
			printf ("FRF9: Got reset request on DLCI - %d\r\n", cur_dlci);
#endif
			/* reset the compression database for this DLCI */
			BSD_handle_reset_request_packet (sptr_dlci_entry->bsd_context_id);
		
			/* reset the sequence numbers */
			sptr_dlci_entry->tx_sequence_number = 0;
			sptr_dlci_entry->rx_sequence_number = 0xFF;

			/* send a reset ack packet */
			send_frf9_reset_ack (port_number, cur_dlci);

			sptr_dlci_entry->sent_reset_ack = TRUE;
		}
		return FALSE;
	}
	else {
		sptr_dlci_entry->sent_reset_ack = FALSE;
	}

	/* if we are waiting for reset_ack, ignore all data till ack is received */
	if (sptr_dlci_entry->waiting_for_reset_ack == TRUE) {
		if (sptr_dcp_header->dcp_header.reset_ack) {
#if 1
			printf ("FRF9: Got reset ACK on DLCI - %d\r\n", cur_dlci);
#endif
			sptr_dlci_entry->waiting_for_reset_ack = FALSE;

			sptr_dlci_entry->wait_timer = 0xFFFF;

			/* clear the receiver database */
			BSD_handle_reset_ack_packet (sptr_dlci_entry->bsd_context_id);
		}

		return FALSE;
	}

	sptr_dcp_data_pdu = (FRF9_DCP_DATA_PAYLOAD *)
								((*sptr_frame) + sizeof(FRF9_DCP_DATA_PDU_HEADER));

	/* take care of the sequence number and length stored in the data PDU */
	data_len = (*usptr_number_of_bytes_received) - (sizeof(FRF9_DCP_DATA_PDU_HEADER) + 3);

	/* if the sequence numbers do not match, there is an error, so send */
	/* a reset_request and wait for reset_ack from the other side.      */

	if (sptr_dcp_data_pdu->sequence_number != ((sptr_dlci_entry->rx_sequence_number + 1) & 0xFF)) {
#if 1
		printf ("FRF9: Out of sequence on DLCI - %d, Got - %d, Exp - %d\r\n",
			cur_dlci, sptr_dcp_data_pdu->sequence_number, ((sptr_dlci_entry->rx_sequence_number+1) & 0xFF));
#endif

SendResetRequest:

		/* send a reset request packet to the other side */
		send_frf9_reset_request (port_number, cur_dlci);

		sptr_dlci_entry->wait_timer = 0;

		/* reset the sequence numbers */
		sptr_dlci_entry->tx_sequence_number = 0;
		sptr_dlci_entry->rx_sequence_number = 0xFF;

		sptr_dlci_entry->waiting_for_reset_ack = TRUE;

		return FALSE;
	}

	/* the frame is perfectly fine, now check if it is compressed or not */
	sptr_dlci_entry->rx_sequence_number = sptr_dcp_data_pdu->sequence_number;

	if (sptr_dcp_header->dcp_header.compressed_mode) {

		/* uncompress the data by calling the bsd function */
		newbuf = BSD_decompress_packet (sptr_dlci_entry->bsd_context_id,
			&sptr_dcp_data_pdu->payload[0], &data_len);

		/* if the uncompress fails, free the buffer and send a reset request */
		if ((newbuf == NULL) || (data_len == 0)) {
#if 1
			printf ("FRF9: Decompress failed on DLCI - %d\r\n", cur_dlci);
#endif
			if (newbuf != NULL) {
				/* free the buffer given by the compression module */
				BSD_free_compressed_packet (sptr_dlci_entry->bsd_context_id, newbuf);
			}

			goto SendResetRequest;
		}

		/* if the uncompressed data length doesn't match with the given length */
		/* report it as an error and reset the compression database. */
		if (data_len != sptr_dcp_data_pdu->original_length) {
#if 1
			printf ("FRF9: Error while decompressing. Original len = %d, Expanded to = %d\r\n",
				sptr_dcp_data_pdu->original_length, data_len);
#endif

			/* free the buffer given by the compression module */
			BSD_free_compressed_packet (sptr_dlci_entry->bsd_context_id, newbuf);

			goto SendResetRequest;
		}

#ifdef DEBUG
		printf ("FRF9: Uncompressed from %d to %d on DLCI - %d\r\n",
			((*usptr_number_of_bytes_received) - 8), data_len, cur_dlci);
#endif

		/* copy the decompressed data */
		my_memcpy (&sptr_dcp_data_pdu->payload[0], newbuf, data_len);

		/* free the buffer given by the compression module */
		BSD_free_compressed_packet (sptr_dlci_entry->bsd_context_id, newbuf);

		(*usptr_number_of_bytes_received) = (sizeof(FRF9_DCP_DATA_PDU_HEADER) + 3) + data_len;
	}
	else {
		/* update the current database with the data received */
		BSD_update_database (sptr_dlci_entry->bsd_context_id,
			&sptr_dcp_data_pdu->payload[0], &data_len);
	}

	/* adjust the FRF.9 header and make it look like Q.933 frame */
	/* one byte for NLPID, one for DCP header and one for seq number */

	sptr_frf_frame = (BYTE *)((*sptr_frame) + FRF9_BUFFER_ADJUSTMENT);

	/* copy the Q.933 header */
	*sptr_frf_frame = *(*sptr_frame);
	*(sptr_frf_frame + 1) = *(*sptr_frame + 1);
	*(sptr_frf_frame + 2) = *(*sptr_frame + 2);

	/* adjust the buffer ptr */
	*sptr_frame = sptr_frf_frame;

	/* adjust the number of bytes received */
	(*usptr_number_of_bytes_received) -= FRF9_BUFFER_ADJUSTMENT;

#ifdef DEBUG
	printf ("FRF9: Got %d bytes on DLCI - %d\r\n", *usptr_number_of_bytes_received, cur_dlci);
#endif

	return TRUE;
}

/****************************************************************************/
void reset_compression_on_dlci (USHORT dlci)
{
	DLCI_LIST_ENTRY *sptr_dlci_entry;

	if ((sptr_dlci_entry = get_dlci_connection_entry (0, dlci)) == NULL)
		return;

	if (sptr_dlci_entry->compression == FALSE)
		return;

	sptr_dlci_entry->mode_1_state = DISABLED_STATE;
	sptr_dlci_entry->tx_sequence_number = 0;
	sptr_dlci_entry->rx_sequence_number = 0xFF;
	sptr_dlci_entry->waiting_for_reset_ack = FALSE;
	sptr_dlci_entry->sent_reset_ack = FALSE;
	sptr_dlci_entry->wait_timer = 0xFFFF;
}

/****************************************************************************/
void check_reset_requests_and_acks (USHORT dlci)
{
	DLCI_LIST_ENTRY *sptr_dlci_entry;

	if ((sptr_dlci_entry = get_dlci_connection_entry (0, dlci)) == NULL)
		return;

	if (sptr_dlci_entry->compression == FALSE)
		return;

	if (sptr_dlci_entry->wait_timer != 0xFFFF) {
		sptr_dlci_entry->wait_timer++;

		if (sptr_dlci_entry->wait_timer == WAIT_TIMEOUT) {
			if (sptr_dlci_entry->waiting_for_reset_ack == TRUE) {

				/* send a reset request packet to the other side */
				send_frf9_reset_request (0, dlci);

				sptr_dlci_entry->wait_timer = 0;
			}
		}
	}
}

/****************************************************************************/
void initialize_compression_contexts(void)
{
	USHORT pvc_index = 0;
	DLCI_LIST_ENTRY *sptr_current_connection_entry;

	sptr_current_connection_entry = (DLCI_LIST_ENTRY *) frame_relay.port[0].dlci_list.sptr_forward_link;

	while (pvc_index < NUMBER_OF_FRAME_RELAY_PVCS && sptr_current_connection_entry != NULL)
	{
		if (sptr_current_connection_entry->compression == TRUE) {
			sptr_current_connection_entry->bsd_context_id = BSD_get_context_id();
			if (sptr_current_connection_entry->bsd_context_id == 0xFF) {
				sptr_current_connection_entry->compression = FALSE;
			}
		}

		++pvc_index;

		sptr_current_connection_entry	= sptr_current_connection_entry->links.sptr_forward_link;
	}
}
