/* MSM.C -- Modem State Machine. General purpose code to do two things.
**	1. Initialize a modem by sending a modem init string
**	2. Hangup a modem connection	
** By: Sanjay
*/

/* #define DEBUG */

#ifdef DEBUG
#define STATIC			/* So that map file lists even static function info */
#else
#define STATIC static
#endif

#include "defs.h"
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <kstart.h>
#include <v8022str.h>
#include <vethstr.h>
#include <lslproto.h>

#include "kmsm.h"
#include <msm.h>

/* External Prototypes */
extern void msm_scc_set_dtr_status(ULONG);
extern void msm_start_tx_on_scc2(void);
extern void msm_start_tx_on_scc3(void);
extern void msm_start_tx_on_scc4(void);
extern void msm_stop_tx_on_scc2(void);
extern void msm_stop_tx_on_scc3(void);
extern void msm_stop_tx_on_scc4(void);
extern enum BOOLEAN is_cd_present_on_scc2(void);
extern enum BOOLEAN is_cd_present_on_scc3(void);
extern enum BOOLEAN is_cd_present_on_scc4(void);
extern BYTE *strupr(BYTE *sptr_string);

/* Local Prototypes */
STATIC enum TEST msm_control(enum APPLICATION_CONTROL_OPERATION command, ULONG parameter_0, ULONG parameter_1);
STATIC void msm_timer(void);
STATIC void move_modem_state_machine(USHORT port_number);

STATIC void msm_send_a(USHORT port_number);
STATIC void msm_send_t(USHORT port_number);
STATIC void msm_send_rest(USHORT port_number);
STATIC void msm_wait_for_ok(USHORT port_number);
STATIC void rxed_packet_on_port(USHORT port_number, BYTE *sptr_rx_packet, USHORT rx_number_of_bytes);

STATIC void msm_dtr_down(USHORT port_number);
STATIC void msm_hup_check_cd(USHORT port_number);
STATIC void msm_send_hup_str(USHORT port_number);
STATIC void msm_wait_break(USHORT port_number);
STATIC void msm_end_hup_string(USHORT port_number);

STATIC void msm_wait_send_break(USHORT port_number);

STATIC void lower_dtr(USHORT port_number);
STATIC void raise_dtr(USHORT port_number);
STATIC void start_scc_break(USHORT port_number);
STATIC void stop_scc_break(USHORT port_number);
STATIC enum BOOLEAN is_cd_present(USHORT port_number);
STATIC void write_port(USHORT port_number, BYTE *sptr_buffer, USHORT bytes_to_write);

/* Externals */
extern ULONG LocalIOToMove;

/* Locals */
STATIC MSM_CLASS msm;
STATIC ULONG one_second_timer;
STATIC void (*modem_state_handlers[])(USHORT) = {
	msm_send_a,
	msm_send_t,
	msm_send_rest,
	msm_wait_for_ok,

	msm_dtr_down,
	msm_hup_check_cd,
	msm_send_hup_str,
	msm_wait_break,
	msm_end_hup_string,

	msm_wait_send_break
};

/* -- CODE -------------------------------------------------------------- */

/* initialize_msm_module()
**	Called by main router code to setup this module.
*/
enum TEST initialize_msm_module(ULONG clock_ticks_per_second)
{
	/* Simply register as an application - this is really not an application */

	msm.clock_ticks_per_second = clock_ticks_per_second;

	if ((enum TEST) lsl_control(REGISTER_APPLICATION, "Modem State Machine", MSM_MODULE, msm_timer, msm_control, &msm.application_id) == FAIL)
	{
		printf("MSM: Modem state machine module failed to register properly\n\r");
		return PASS;
	}

#ifdef DEBUG
	printf("MSM: Module registered\n\r");
#endif /* DEBUG */

	msm.msm_timer_enabled = TRUE;
}

STATIC enum TEST msm_control(enum APPLICATION_CONTROL_OPERATION command, ULONG parameter_0, ULONG parameter_1)
{
	switch (command)
	{
	case IS_APPLICATION_ENABLED:
		*((enum BOOLEAN *) parameter_1) = TRUE;
		break;
	default:
		break;
	}

	return (PASS);
}

STATIC void msm_timer()
{
	USHORT port_number;


	if (msm.msm_timer_enabled == FALSE)
		return;

	if (one_second_timer != 0)
		one_second_timer--;

	for (port_number = 0; port_number < MAX_WAN_PORTS; port_number++)
	{
		move_modem_state_machine(port_number);
		if (one_second_timer == 0 && msm.current_state_wait[port_number] != 0)
			msm.current_state_wait[port_number] -= 1;
	}

	if (one_second_timer == 0)
		one_second_timer = msm.clock_ticks_per_second;
}

/* start_modem_init_state_machine()
**	Call this to start initing the modem. Pass the port number to init modem
**	on, the init string to use (assumed prefixed with "AT") and a function 
**	that can be called when init is done. This function gets called with the
**	port number on which init was done and TRUE if init was successful, else
**	FALSE.
**  Among other things, this will try to hangup the modem by lowering the
**	RTS and bringing it up again and also will send a "\r" to stop the
**	modem from continuing dialling.
**	This function will return NULL (if no init string was provided) or a
**	pointer to a function that should be called by the module calling this
**	routine. The way the QUICC works, the calling module will get data
**	received from modem. So during this init phase it should pass 
**  received packets to this returned function.
*/
MSM_CALLBACK start_modem_init_state_machine(USHORT port_number, char **modem_init_strings, void (*fptr_write_port)(USHORT, BYTE *, USHORT), void (*fptr_modem_init_complete)(USHORT, enum BOOLEAN))
{
	int i;
	char init_string_buffer[MODEM_INIT_STRING_LENGTH];


	msm.enabled[port_number] = FALSE;

	if (modem_init_strings == NULL || strlen(*modem_init_strings) < 2)
	{
#ifdef DEBUG
		printf("MSM: No or bad modem init string, port %u\n\r", port_number);
#endif /* DEBUG */

		if (fptr_modem_init_complete)
			fptr_modem_init_complete(port_number, FALSE);
		return NULL;
	}

	msm.enabled[port_number] = TRUE;
	msm.current_state[port_number] = MSM_SEND_A;
	msm.current_state_wait[port_number] = MSM_SEND_A_DELAY;
	for (i = 0; i < 5; i++)
	{
		if (*modem_init_strings != NULL)
		{
			memcpy(init_string_buffer, *modem_init_strings, MODEM_INIT_STRING_LENGTH);
			init_string_buffer[MODEM_INIT_STRING_LENGTH - 1] = '\0';
			convert_controls(msm.modem_init_string[port_number][i], init_string_buffer);
		}
		else
		{
			msm.modem_init_string[port_number][i][0] = '\0';
		}
		modem_init_strings++;
	}
	msm.fptr_modem_init_complete[port_number] = fptr_modem_init_complete;
	msm.fptr_write_function[port_number] = fptr_write_port;
	msm.current_init_string[port_number] = 0;

	lower_dtr(port_number);
	raise_dtr(port_number);

	write_port(port_number, "\r", 1);

#ifdef DEBUG
	printf("MSM: Starting modem init state machine, port %u\n\r", port_number);
#endif /* DEBUG */

	return rxed_packet_on_port;
}

/* start_modem_hangup_state_machine()
**	Call this to start a modem hangup. Pass as parameters to this function,
**	the port on which hangup is to be attempted, the modem hangup string
**	to use and a pointer to a function to callback when hangup is done.
**	Among other things, this function tries to hangup first by pulling down
**	the DTR and bringing it back up again. If modem is hung up, then it
**	calls the callback, else it sends the hangup string and then calls the
**  callback.
*/
void start_modem_hangup_state_machine(USHORT port_number, char *modem_hangup_string, void (*fptr_write_port)(USHORT, BYTE *, USHORT), void (*fptr_modem_hangup_complete)(USHORT))
{
	msm.enabled[port_number] = FALSE;
	
	convert_controls(msm.modem_hangup_string[port_number], modem_hangup_string);
	msm.fptr_modem_hangup_complete[port_number] = fptr_modem_hangup_complete;
	msm.fptr_write_function[port_number] = fptr_write_port;

	msm.current_state[port_number] = MSM_DTR_DOWN;
	msm.current_state_wait[port_number] = MSM_DTR_DOWN_DELAY;

	msm.enabled[port_number] = TRUE;
#ifdef DEBUG
	printf("MSM: Starting modem hangup state machine, port %u\n\r", port_number);
#endif /* DEBUG */
	return;
}

/* send_break()
**	Arranges to send a BREAK sequence. The break character and the number
**	of characters are pre-programmed in the BOOT code. Caller has to
**	make sure that during this time (of one second) other state machines
**	are not called.
*/
void send_break(USHORT port_number)
{
	int temp_sr;


	msm.current_state[port_number] = MSM_WAIT_SEND_BREAK;
	msm.current_state_wait[port_number] = MSM_WAIT_SEND_BREAK_DELAY;

	temp_sr = _GPL();
	_SPL(7);
	start_scc_break(port_number);
	_SPL(temp_sr);
	
	msm.enabled[port_number] == TRUE;
}

STATIC void move_modem_state_machine(USHORT port_number)
{
	if (msm.enabled[port_number] == TRUE)
		modem_state_handlers[msm.current_state[port_number]](port_number);
}

STATIC void msm_send_a(USHORT port_number)
{
	if (msm.current_state_wait[port_number])
		return;

#ifdef DEBUG
	printf("MSM: Current state MSM_SEND_A\n\r");
#endif /* DEBUG */

	write_port(port_number, msm.modem_init_string[port_number][msm.current_init_string[port_number]], 1);

	msm.current_state[port_number] = MSM_SEND_T;
	msm.current_state_wait[port_number] = MSM_SEND_T_DELAY;
}

STATIC void msm_send_t(USHORT port_number)
{
	if (msm.current_state_wait[port_number])
		return;

#ifdef DEBUG
	printf("MSM: Current state MSM_SEND_T\n\r");
#endif /* DEBUG */

	write_port(port_number, &msm.modem_init_string[port_number][msm.current_init_string[port_number]][1], 1);

	msm.current_state[port_number] = MSM_SEND_REST;
	msm.current_state_wait[port_number] = MSM_SEND_REST_DELAY;
}

STATIC void msm_send_rest(USHORT port_number)
{
	char *sptr_rest_of_string;


	if (msm.current_state_wait[port_number])
		return;

#ifdef DEBUG
	printf("MSM: Current state MSM_SEND_REST\n\r");
#endif /* DEBUG */

	sptr_rest_of_string = &msm.modem_init_string[port_number][msm.current_init_string[port_number]][2];
	write_port(port_number, sptr_rest_of_string, strlen(sptr_rest_of_string));

	msm.current_init_string[port_number]++;
	msm.current_state[port_number] = MSM_WAIT_OK;
	msm.current_state_wait[port_number] = MSM_WAIT_OK_DELAY;
}

STATIC void msm_wait_for_ok(USHORT port_number)
{
	if (msm.ok_received[port_number] == TRUE || msm.current_state_wait[port_number] == 0)
	{
		while (msm.current_init_string[port_number] < 5 && 
			msm.modem_init_string[port_number][msm.current_init_string[port_number]][0] == '\0')
		{
			/* Skip null init strings */
			msm.current_init_string[port_number]++;
		}

						
		if (msm.current_init_string[port_number] >= 5)
		{
			msm.enabled[port_number] = FALSE;

			if (msm.fptr_modem_init_complete[port_number])
				msm.fptr_modem_init_complete[port_number](port_number, msm.ok_received[port_number]);

#ifdef DEBUG
			printf("MSM: Last modem init state MSM_WAIT_FOR_OK returning with %d (0 = fail, 1 = success)\n\r", msm.ok_received[port_number]);
#endif /* DEBUG */
			return;
		}
		else
		{
			/* Go to sending the next init string */
			msm.current_state[port_number] = MSM_SEND_A;
			msm.current_state_wait[port_number] = MSM_SEND_A_DELAY;
		}
	}
}

STATIC void rxed_packet_on_port(USHORT port_number, BYTE *sptr_rx_packet, USHORT rx_number_of_bytes)
{
	/* Search for the string "OK". "OK" is assumed not to straddle two
	** receive buffers.
	*/
#ifdef DEBUG
	printf("MSM: Called with received packet (%u bytes)\n\r", rx_number_of_bytes);
#endif /* DEBUG */

	while (rx_number_of_bytes)
	{
		if (*sptr_rx_packet == 'O')
		{
			if ((rx_number_of_bytes > 1) && (*(sptr_rx_packet + 1) == 'K'))
			{
#ifdef DEBUG
				printf("MSM: Received OK string\n\r");
#endif /* DEBUG */
				msm.ok_received[port_number] = TRUE;
				return;
			}
		}
		rx_number_of_bytes--;
		sptr_rx_packet++;
	}
}

/* -- Hangup state machine -- */

STATIC void msm_dtr_down(USHORT port_number)
{
	if (msm.current_state_wait[port_number])
		return;

#ifdef DEBUG
	printf("MSM: Current state MSM_DTR_DOWN\n\r");
#endif /* DEBUG */

	lower_dtr(port_number);

	msm.current_state[port_number] = MSM_HUP_CHECK_CD;
	msm.current_state_wait[port_number] = MSM_HUP_CHECK_CD_DELAY;
}

STATIC void msm_hup_check_cd(USHORT port_number)
{
	if (msm.current_state_wait[port_number])
		return;

#ifdef DEBUG
	printf("MSM: Current state MSM_CHECK_CD\n\r");
#endif /* DEBUG */

	if (is_cd_present(port_number))
	{
		/* Pulling down DTR didn't work. So send the hangup string. */

		raise_dtr(port_number);	/* Bring DTR back up again */

		msm.sptr_command_str[port_number] = msm.modem_hangup_string[port_number];

		msm.current_state[port_number] = MSM_SEND_HUP_STRING;
		msm.current_state_wait[port_number] = MSM_SEND_HUP_STRING_DELAY;
	}
	else
	{
		raise_dtr(port_number);	/* Bring DTR back up again */

		msm.enabled[port_number] = FALSE;

		if (msm.fptr_modem_hangup_complete[port_number])
			msm.fptr_modem_hangup_complete[port_number](port_number);

#ifdef DEBUG
		printf("MSM: Hangup complete\n\r");
#endif /* DEBUG */
	}
}

STATIC void msm_send_hup_str(USHORT port_number)
{
	char *bptr;
	char temp_char_buf[256];
	int temp_sr, bytes_to_write;
	char *break_str = "<BREAK>";


	if (msm.current_state_wait[port_number])
		return;

#ifdef DEBUG
	printf("MSM: Current state MSM_SEND_HUP_STRING\n\r");
#endif /* DEBUG */

	if (msm.sptr_command_str[port_number] == NULL || *msm.sptr_command_str[port_number] == '\0')
	{
		msm.current_state[port_number] = MSM_END_HUP_STRING;
		msm.current_state_wait[port_number] = MSM_END_HUP_STRING_DELAY;
		return;
	}

	if (*msm.sptr_command_str[port_number] == MODEM_COMMAND_PAUSE_CHAR)
	{
		msm.sptr_command_str[port_number]++;
		msm.current_state_wait[port_number] = MODEM_COMMAND_PAUSE_TIME;
		return;			/* Wait in same state for above amount of time */
	}

	strcpy(temp_char_buf, msm.sptr_command_str[port_number]);
	strupr(temp_char_buf);
	bptr = strstr(msm.sptr_command_str[port_number], break_str);
	if (bptr)
	{
		/* Need to issue BREAK command to the SCC */

		temp_sr = _GPL();
		_SPL(7);
		start_scc_break(port_number);
		_SPL(temp_sr);

		msm.sptr_command_str[port_number] = bptr + strlen(break_str);
		msm.current_state[port_number] = MSM_WAIT_BREAK;
		msm.current_state_wait[port_number] = MODEM_COMMAND_BREAK_TIME;
		return;			/* Wait in same state for above amount of time */
	}

	bptr = strchr(msm.sptr_command_str[port_number], MODEM_COMMAND_PAUSE_CHAR);
	if (bptr)
	{
		/* Write all characters upto and not including the pause character */
		bytes_to_write = bptr - msm.sptr_command_str[port_number];
		write_port(port_number, msm.sptr_command_str[port_number], bytes_to_write);
		msm.sptr_command_str[port_number] += bytes_to_write;

		/* Return to the same state */
		return;
	}

	/* Write remaining string to port */
	write_port(port_number, msm.sptr_command_str[port_number], strlen(msm.sptr_command_str[port_number]));

	msm.current_state[port_number] = MSM_END_HUP_STRING;
	msm.current_state_wait[port_number] = MSM_END_HUP_STRING_DELAY;
}

STATIC void msm_wait_break(USHORT port_number)
{
	int temp_sr;

	if (msm.current_state_wait[port_number])
		return;

#ifdef DEBUG
	printf("MSM: Current state sending break\n\r");
#endif /* DEBUG */

	temp_sr = _GPL();
	_SPL(7);

	stop_scc_break(port_number);

	_SPL(temp_sr);

	msm.current_state[port_number] = MSM_SEND_HUP_STRING;
	msm.current_state_wait[port_number] = 0;
}

STATIC void msm_end_hup_string(USHORT port_number)
{
	/* Wait in this state after hangup for sometime */
	if (msm.current_state_wait[port_number] != 0)
		return;

	msm.enabled[port_number] = FALSE;

	if (msm.fptr_modem_hangup_complete[port_number])
		msm.fptr_modem_hangup_complete[port_number](port_number);

#ifdef DEBUG
	printf("MSM: Final hangup state\n\r");
#endif /* DEBUG */
}

/* --------- Break state machine ------------------- */

STATIC void msm_wait_send_break(USHORT port_number)
{
	int temp_sr;

	if (msm.current_state_wait[port_number])
		return;

	temp_sr = _GPL();
	_SPL(7);

	stop_scc_break(port_number);

	_SPL(temp_sr);

	msm.enabled[port_number] = FALSE;
}


/* ----------- Support code ------------- */
STATIC void lower_dtr(USHORT port_number)
{
	switch (port_number)
	{
	case 0:
		LocalIOToMove |= 0x08;
		break;
	case 1:
		LocalIOToMove |= 0x10;
		break;
	case 2:
		LocalIOToMove |= 0x20;
		break;
	}

	msm_scc_set_dtr_status(LocalIOToMove);
}

STATIC void raise_dtr(USHORT port_number)
{
	switch (port_number) 
	{
	case 0:
		LocalIOToMove &= ~0x08;
		break;
	case 1:
		LocalIOToMove &= ~0x10;
		break;
	case 2:
		LocalIOToMove &= ~0x20;
		break;
	}

	msm_scc_set_dtr_status(LocalIOToMove);
}

STATIC void start_scc_break(USHORT port_number)
{
	switch (port_number)
	{
	case 0:
		msm_stop_tx_on_scc2();
		break;
	case 1:
		msm_stop_tx_on_scc3();
		break;
	case 2:
		msm_stop_tx_on_scc4();
		break;
	}
}

STATIC void stop_scc_break(USHORT port_number)
{
	switch (port_number)
	{
	case 0:
		msm_start_tx_on_scc2();
		break;
	case 1:
		msm_start_tx_on_scc3();
		break;
	case 2:
		msm_start_tx_on_scc4();
		break;
	}
}

STATIC enum BOOLEAN is_cd_present(USHORT port_number)
{
	switch (port_number)
	{
	case 0:
		return is_cd_present_on_scc2();
		break;
	case 1:
		return is_cd_present_on_scc2();
		break;
	case 2:
		return is_cd_present_on_scc2();
		break;
	}
}

STATIC void write_port(USHORT port_number, BYTE *sptr_buffer, USHORT bytes_to_write)
{
	msm.fptr_write_function[port_number](port_number, sptr_buffer, bytes_to_write);
}

void convert_controls(char *sptr_destination_string, char *sptr_source_string)
{
	int str_len = 0;

	while (*sptr_source_string)
	{
		sptr_source_string++;
		if (*(sptr_source_string - 1) == '^' && *sptr_source_string != '^')
		{
			if (islower(*sptr_source_string))
				*sptr_destination_string = (*sptr_source_string - 'a' + 1);
			else
				*sptr_destination_string = (*sptr_source_string - 'A' + 1);
			sptr_source_string++;
		}
		else
			*sptr_destination_string = *(sptr_source_string - 1);
		sptr_destination_string++;
		str_len++;
		if (str_len == 63)	/* Max 63 bytes supported */
		{
#ifdef DEBUG
			printf("MSM: Init or hangup string longer than 63 bytes...truncating\n\r");
#endif
			break;
		}
	}
	*sptr_destination_string = '\0';
}

/* -- END CODE ---------------------------------------------------------- */
