/*--------------------------------------------------------------------------
	File        : PPPUDB.C
	Author      : Sachin S Desai
	Date        : 10th May 1996
	Description : Contains code to handle everything related to the user database.
	              It is required to call the initialize_user_database() fn.
	              during initialization and the function check_if_udb_updated()
	              in the foreground.
--------------------------------------------------------------------------*/

#include <udb.h>
#include "..\..\store\boot.h"
#include <flashmgr.h>

#define DEBUG 0
#define VERSION2 "v2.00"

#if DEBUG
void check_compression_algorithm () ;
#endif

extern void *malloc(ULONG) ;
extern void *calloc(ULONG) ;
extern void *memset(void *dest, int init_value, int count) ;
extern void *memcpy(void *dest, void *src, int count) ;
extern char *strcpy(char *dest, char *src) ;
extern int strcmp(char *source, char *dest) ;
extern int strlen(char *str) ;

extern void c_write_to_flash (char *source, char *destination, int size) ;
extern int LZW_expand (unsigned char *input, unsigned char *output, unsigned long total_packed_size) ;
extern long LZ15V_compress (unsigned char *input, unsigned long input_length, unsigned char *output, unsigned long output_buffer_length) ;

extern USHORT update_crc (USHORT acc_crc, BYTE *packet, ULONG length) ;

static enum TEST udb_integrity (void) ;

UDB_HEADER *ptr_user_database_hdr = (UDB_HEADER *) UDB_HEADER_ADDRESS ;
RAS_USER_DATABASE_RECORD *ptr_ras_user_database = (RAS_USER_DATABASE_RECORD *) NULL ;
static USHORT no_of_user_database_records = 0 ;
static enum BOOLEAN check_udb_integrity = FALSE ;
enum BOOLEAN user_database_updated = FALSE ;

/* Added by Ravi on 16 Apr 1999 ... */
extern void update_remote_user_database_structures ();
extern void age_remote_user_database_structures();

/* ... Added by Ravi on 16 Apr 1999 */


int strcmpi (char *s1, char *s2)
{
	char a, b ;
	do
	{
		a = *s1++ ; b = *s2++ ;

		if ((a <= 'z') && (a >= 'a'))
			a -= 'a' - 'A' ;
		if ((b <= 'z') && (b >= 'a'))
			b -= 'a' - 'A' ;
	}
	while ((a == b) && a && b) ;

	return (a - b) ;
}

void clean_udb ()
{
	UDB_HEADER udb_header ;
	USHORT udb_crc = 0xFFFF;
	BYTE   two_zeroes[] = {0, 0} ;
	BYTE temp_buff[1000] ;
	UDB_HEADER *temp_buff_ptr = (UDB_HEADER *) temp_buff ;

/*	udb_header.magic_number = 0x1234 ;
	c_write_to_flash ((BYTE *)&udb_header, (BYTE *)ptr_user_database_hdr, 512) ;*/

	memset (temp_buff_ptr, 0, 512) ;

	temp_buff_ptr->magic_number = 0xABCD ;
	temp_buff_ptr->length = 0L ;
	temp_buff_ptr->no_of_records = 0 ;
	strcpy (&temp_buff_ptr->version[0], VERSION2) ;
	strcpy (&temp_buff_ptr->reserved[0], "USERDEFAULT") ;
	strcpy (&temp_buff_ptr->description[0], "USER DATABASE") ;
	temp_buff_ptr->down_load_address = UDB_DNLD_ADDR;

	udb_crc = update_crc (udb_crc,
								(BYTE *)temp_buff_ptr->down_load_address,
	                     (ULONG)temp_buff_ptr->length) ;
	udb_crc = update_crc (udb_crc, &two_zeroes[0], 2) ;
	temp_buff_ptr->crc = udb_crc;

	c_write_to_flash ((BYTE *)temp_buff_ptr, (BYTE *)ptr_user_database_hdr, 512) ;
	no_of_user_database_records = 0 ;
	check_udb_integrity = FALSE ;
	
	printf ("Cleaned up User database header\n") ;
}

enum TEST initialize_user_database ()
{
	BYTE temp_buff[1000] ;
	UDB_HEADER *temp_buff_ptr = (UDB_HEADER *) temp_buff ;
	USHORT udb_crc = 0xFFFF;
	BYTE   two_zeroes[] = {0, 0} ;

#if Ravi
   /* For proxy Server only */
   clean_udb () ;

	if ((ptr_user_database_hdr->magic_number != 0xABCD) ||
	    (ptr_user_database_hdr->length >= (UDB_MAX_USERS * sizeof (RAS_USER_DATABASE_RECORD)) ||
		 (ptr_user_database_hdr->no_of_records == 0)))
#endif

	if(ptr_user_database_hdr->magic_number != 0xABCD || 
		strcmp(ptr_user_database_hdr->version, VERSION2))
	{
		memset (temp_buff_ptr, 0, 512) ;

		temp_buff_ptr->magic_number = 0xABCD ;
		temp_buff_ptr->length = 0L ;
		temp_buff_ptr->no_of_records = 0 ;
		temp_buff_ptr->crc = 0;
		strcpy (&temp_buff_ptr->version[0], VERSION2) ;
		strcpy (&temp_buff_ptr->reserved[0], "USERDEFAULT") ;
		strcpy (&temp_buff_ptr->description[0], "USER DATABASE") ;
		temp_buff_ptr->down_load_address = UDB_DNLD_ADDR;

		udb_crc = update_crc (udb_crc,
								(BYTE *)temp_buff_ptr->down_load_address,
	                     (ULONG)temp_buff_ptr->length) ;
		udb_crc = update_crc (udb_crc, &two_zeroes[0], 2) ;
		temp_buff_ptr->crc = udb_crc;

		c_write_to_flash ((BYTE *)temp_buff_ptr, (BYTE *)ptr_user_database_hdr, 512) ;
		no_of_user_database_records = 0 ;
		check_udb_integrity = FALSE ;
	
		printf ("User Database Init : Cleaned up User database header\n") ;
		return (FAIL) ; 
	}

	if (!strcmp(ptr_user_database_hdr->reserved, "USERDEFAULT"))
	{
		printf("Setting User Database Defaults ...\n");
		return (PASS) ; 
	}

	check_udb_integrity = TRUE ; 
	user_database_updated = TRUE ;
	return (PASS) ;
}

enum TEST check_if_udb_updated (void)
{
	int expansion_successful ;
	BYTE *ptr_remote_user_database;

	if (user_database_updated == FALSE)
		return (FAIL) ;

	no_of_user_database_records = ptr_user_database_hdr->no_of_records ;

#if 1	/* KVSP */

	if (check_udb_integrity == TRUE)
	{
		check_udb_integrity = FALSE;

		if (udb_integrity() == FAIL)
		{
			printf ("CRC check on user database failed\n") ;
			clean_udb() ; 
			printf ("Cleaned up User Database Header\n") ; 
			user_database_updated = FALSE ;
			return (FAIL) ;
		}
	}
#endif

/* Changed By Ravi on 16 Apr 1999 */
	ptr_remote_user_database = (BYTE *) ptr_user_database_hdr->down_load_address;

	printf("Down load address : %x\n", ptr_remote_user_database); 
	/* This function is present in USERUTIL.C */
  	
	update_remote_user_database_structures(ptr_remote_user_database);
	age_remote_user_database_structures();

#if DEBUG
	check_compression_algorithm () ;
#endif

	user_database_updated = FALSE ;

	return (PASS) ;
}


RAS_USER_DATABASE_RECORD *get_ptr_to_udb_record (char *user_name)
{
	int i ;

	for (i = 0 ; i < no_of_user_database_records ; i++)
	{
		if (strcmpi (ptr_ras_user_database[i].user_name, user_name) == 0)
		{
			return (&ptr_ras_user_database[i]) ;
		}
	}
	return (NULL) ;
}


enum BOOLEAN do_passwords_match (RAS_USER_DATABASE_RECORD *udb_record, char *password)
{
	/*
		If there is no password, return TRUE
	*/
	if (udb_record->password[0] == 0)
		return (TRUE) ;

	if (strcmpi (password, udb_record->password) == 0)
	{
		return (TRUE) ;
	}

	return (FALSE) ;
}


enum BOOLEAN is_dial_in_permitted_for_port (RAS_USER_DATABASE_RECORD *udb_record, char port_number)
{
	ULONG port_permission_mask = 1L ;

	while (port_number--)
		port_permission_mask <<= 1 ;

	if (port_permission_mask & udb_record->dial_in_port_mask)
	{
		return (TRUE) ;
	}

	return (FALSE) ;
}


enum BOOLEAN is_dial_out_permitted_for_port (RAS_USER_DATABASE_RECORD *udb_record, char port_number)
{
	ULONG port_permission_mask = 1 ;

	while (port_number--)
		port_permission_mask <<= 1 ;

	if (port_permission_mask & udb_record->dial_out_port_mask)
		return (TRUE) ;

	return (FALSE) ;
}


enum BOOLEAN is_ipx_permitted (RAS_USER_DATABASE_RECORD *udb_record)
{
	if (udb_record->protocol_mask & 0x02)
	{
		return (TRUE) ;
	}
	return (FALSE) ;
}


enum BOOLEAN is_ip_permitted (RAS_USER_DATABASE_RECORD *udb_record)
{
	if (udb_record->protocol_mask & 0x01)
	{
		return (TRUE) ;
	}
	return (FALSE) ;
}


enum BOOLEAN is_stp_permitted (RAS_USER_DATABASE_RECORD *udb_record)
{
	if (udb_record->protocol_mask & 0x04)
	{
		return (TRUE) ;
	}
	return (FALSE) ;
}


int get_ras_number_of_users ()
{
	return (no_of_user_database_records) ;
}

static enum TEST udb_integrity (void)
{
	USHORT udb_crc = 0xFFFF;
	BYTE   two_zeroes[] = {0, 0} ;

	udb_crc = update_crc (udb_crc,
								(BYTE *)ptr_user_database_hdr->down_load_address,
	                     (ULONG)ptr_user_database_hdr->length) ;
	udb_crc = update_crc (udb_crc, &two_zeroes[0], 2) ;

#if 0
		printf ("UDB: Saved CRC = 0x%x, Calculated CRC = 0x%x\r\n",
		ptr_user_database_hdr->crc, udb_crc);
#endif

	if (udb_crc != ptr_user_database_hdr->crc)
	{
		return (FAIL) ;
	}

	return (PASS) ;
}

BYTE *is_valid_ras_user (char *user_name)
{
	return ((BYTE *)get_ptr_to_udb_record (user_name)) ;
}

enum BOOLEAN has_ras_user_password (BYTE *user_entry)
{
	if (((RAS_USER_DATABASE_RECORD *)user_entry)->password[0])
		return (TRUE) ;
	else
		return (FALSE) ;
}

enum BOOLEAN is_valid_ras_user_password (BYTE *user_entry, char *password_str)
{
	if (strcmpi (password_str, ((RAS_USER_DATABASE_RECORD *)user_entry)->password) == 0)
		return (TRUE) ;
	else
		return (FALSE) ;
}

enum BOOLEAN has_dialout_access (BYTE *user_entry, USHORT port_number)
{
	if (is_dial_out_permitted_for_port ((RAS_USER_DATABASE_RECORD *)user_entry, port_number))
		return (TRUE) ;
	else
		return (FALSE) ;
}

enum BOOLEAN has_dialout_inbound_access (BYTE *user_entry, USHORT port_number)
{
	ULONG port_permission_mask = 1L ;
	RAS_USER_DATABASE_RECORD *udb_record = (RAS_USER_DATABASE_RECORD *)user_entry;

	while (port_number--)
		port_permission_mask <<= 1 ;

	if (port_permission_mask & udb_record->dial_in_port_mask)
	{
		return (TRUE) ;
	}

	return (FALSE) ;
}


char *get_ras_user_name (BYTE *user_entry, char *user_name)
{
	RAS_USER_DATABASE_RECORD *udb_record = (RAS_USER_DATABASE_RECORD *)user_entry;

	return (strcpy (user_name, &udb_record->user_name[0])) ;
}


enum BOOLEAN is_ras_callback_enabled (BYTE *user_entry, USHORT port_number)
{
	RAS_USER_DATABASE_RECORD *udb_record = (RAS_USER_DATABASE_RECORD *)user_entry;

	return ((enum BOOLEAN) udb_record->call_back_enabled) ;
}


enum BOOLEAN is_ras_callback_callback_security (BYTE *user_entry, USHORT port_number)
{
	RAS_USER_DATABASE_RECORD *udb_record = (RAS_USER_DATABASE_RECORD *)user_entry;

	if ((udb_record->call_back_enabled) && (udb_record->call_back_security))
		return (TRUE) ;

	return (FALSE) ;
}



USHORT ras_get_callback_delay (BYTE *user_entry, USHORT port_number)
{
	RAS_USER_DATABASE_RECORD *udb_record = (RAS_USER_DATABASE_RECORD *)user_entry;

	return (udb_record->call_back_delay) ;
}


char *ras_get_callback_number (BYTE *user_entry, USHORT port_number, char *callback_number)
{
	RAS_USER_DATABASE_RECORD *udb_record = (RAS_USER_DATABASE_RECORD *)user_entry;

	return (strcpy (callback_number, &udb_record->call_back_number[0])) ;
}

RAS_USER_DATABASE_RECORD *get_ras_user_entry_from_index (int index)
{
   if (index >= no_of_user_database_records)
         return (NULL) ;

   return (&ptr_ras_user_database[index]) ;
}


/*----------------------------------------------------------------------
	The following functions and variables are for user database
	modifications. Right now, they are used for Telnet configuration.
	They can later be used by any other agent used to configure the
	user database online.
		Any sequence of user database modification requires a call to
	udb_initialize() followed by a sequence of calls to udb_insert(),
	udb_delete() and udb_edit() as the case may be. But for the new
	user database to be actually operational (to be looked at for dial
	out, ras, etc.), a call to udb_update_after_changes() has to be made.
	Following all of these, a call to udb_deinitialize has to be made.
	The whole sequence of operations requires a lot of memory to work
	with. So, it is possible that any of the calls may fail. So, make
	sure you look at the return values of all the functions.
		Further, calls to all these functions form a cycle. When a cycle
	is on, (it is on till a call to udb_deinitialize() is made) a new
	cycle cannot be started.
----------------------------------------------------------------------*/
#define BLOCK_SIZE (1024 * 8)
#define MAX_COMPRESSED_BLOCKS ((sizeof(RAS_USER_DATABASE_RECORD) * UDB_MAX_USERS) / BLOCK_SIZE)

int number_of_ras_users ;
RAS_USER_DATABASE_RECORD **udb_record_ptrs ;
BYTE *new_node ;
enum BOOLEAN udb_handler_inited = FALSE ;

enum UDB_MODIFY udb_initialize ()
{
	int i = 0 ;

	if (udb_handler_inited == TRUE)
		return (INIT_MODIFY_SEQUENCE_ALREADY_ON) ;

	udb_record_ptrs = (RAS_USER_DATABASE_RECORD **)
	                  malloc ((UDB_MAX_USERS + no_of_user_database_records) * sizeof (RAS_USER_DATABASE_RECORD *)) ;

	if (udb_record_ptrs == NULL)
		return (INIT_NO_MEMORY) ;

	new_node = (BYTE *)
	                  malloc ((UDB_MAX_USERS + no_of_user_database_records) * sizeof (BYTE)) ;
	if (new_node == NULL)
		return (INIT_NO_MEMORY) ;

	for (i = 0 ; i < no_of_user_database_records ; i++)
	{
		udb_record_ptrs[i] = &ptr_ras_user_database[i] ;
		new_node[i] = 0 ;
	}

	number_of_ras_users = no_of_user_database_records ;

	udb_handler_inited = TRUE ;
	return (INIT_SUCCESSFUL) ;
}

enum UDB_MODIFY udb_deinitialize ()
{
	if (udb_handler_inited != TRUE)
	{
		printf ("Call to udb_deinitialize() without initializing\n") ;
		return (UDB_HANDLER_NOT_INITED) ;
	}

	free (new_node) ;
	free (udb_record_ptrs) ;

	udb_handler_inited = FALSE ;

	return (DEINIT_SUCCESSFUL) ;
}


int udb_get_number_of_users ()
{
	return (number_of_ras_users) ;
}


RAS_USER_DATABASE_RECORD *udb_get_user_entry_from_index (int index)
{
	if (index >= number_of_ras_users)
		return (NULL) ;

	return (udb_record_ptrs[index]) ;
}


RAS_USER_DATABASE_RECORD *udb_get_user_entry_from_name (char *user_name)
{
	int i ;

	for (i = 0 ; i < number_of_ras_users ; i++)
	{
		if (strcmpi (user_name, &udb_record_ptrs[i]->user_name[0]) == 0)
			return (udb_record_ptrs[i]) ;
	}

	return (NULL) ;
}

/*-------------------------------------------------------------------------
	udb_insert () :
		Allocates memory for the new record.
		Inserts a record in the user database at the correct place.
		The pointers are accordingly adjusted.
-------------------------------------------------------------------------*/
enum UDB_MODIFY udb_insert (RAS_USER_DATABASE_RECORD *sptr_new_record)
{
	RAS_USER_DATABASE_RECORD *sptr_allocated_record ;
	int i, j ;

	if (udb_handler_inited != TRUE)
		return (UDB_HANDLER_NOT_INITED) ;

	if (udb_get_user_entry_from_name (&sptr_new_record->user_name[0]) != NULL)
		return (INSERT_USER_ALREADY_EXISTS) ;

	if (number_of_ras_users >= UDB_MAX_USERS)
		return (INSERT_MAX_COUNT_EXCEEDED) ;

	sptr_allocated_record = (RAS_USER_DATABASE_RECORD *) malloc (sizeof (RAS_USER_DATABASE_RECORD)) ;
	if (sptr_allocated_record == NULL)
	{
		return (INSERT_NO_MEMORY) ;
	}

	*sptr_allocated_record = *sptr_new_record ;

	/* Find the position of the new record */
	switch (number_of_ras_users)
	{
		case (0) :
			i = 0 ;
			break ;

		case (1) :
			if (strcmpi (sptr_allocated_record->user_name, udb_record_ptrs[0]->user_name) < 0)
				i = 0 ;
			else
				i = 1 ;

		default :
			if (strcmpi (sptr_allocated_record->user_name, udb_record_ptrs[0]->user_name) < 0)
				i = 0 ;
			else
			{
				for (j = 1 ; j < number_of_ras_users ; j++)
				{
					if (strcmpi (sptr_allocated_record->user_name, udb_record_ptrs[j]->user_name) > 0)
						continue ;
					else
						break ;
				}
				i = j ;
			}
	}

	for (j = number_of_ras_users-1 ; j >= i ; j--)
	{
		new_node[j+1] = new_node[j] ;
		udb_record_ptrs[j+1] = udb_record_ptrs[j] ;
	}
	new_node[i] = 1 ;
	udb_record_ptrs[i] = sptr_allocated_record ;

	number_of_ras_users++ ;

	return (INSERT_SUCCESSFUL) ;
}


/*-------------------------------------------------------------------------
	udb_delete() :
		Deletes a record from the user database.
		If the memory allocated for the record was 'new', it is freed up.
		The pointers are accordingly adjusted.
-------------------------------------------------------------------------*/
enum UDB_MODIFY udb_delete (char *user_name)
{
	int j, i ;

	if (udb_handler_inited != TRUE)
		return (UDB_HANDLER_NOT_INITED) ;

	for (i = 0 ; i < number_of_ras_users ; i++)
	{
		if (strcmpi (user_name, &udb_record_ptrs[i]->user_name[0]) == 0)
			break ;
	}

	if (i >= number_of_ras_users)
		return (DELETE_USER_NOT_FOUND) ;

	/* i is the index of the record to be deleted */
	if (new_node[i])
	{
		free (udb_record_ptrs[i]) ;
	}

	for (j = i+1 ; j < number_of_ras_users ; j++)
	{
		udb_record_ptrs[j-1] = udb_record_ptrs[j] ;
		new_node[j-1] = new_node[j] ;
	}

	number_of_ras_users-- ;

	return (DELETE_SUCCESSFUL) ;
}


enum UDB_MODIFY udb_edit (RAS_USER_DATABASE_RECORD *sptr_record_to_edit)
{
	enum UDB_MODIFY insert_return_code ;

	if (udb_handler_inited != TRUE)
		return (UDB_HANDLER_NOT_INITED) ;

	if (udb_delete (&sptr_record_to_edit->user_name[0]) == DELETE_USER_NOT_FOUND)
		return (EDIT_RECORD_NOT_FOUND) ;

	insert_return_code = udb_insert (sptr_record_to_edit) ;
	switch (insert_return_code)
	{
		case (INSERT_SUCCESSFUL) :
			return (EDIT_SUCCESSFUL) ;

		case (INSERT_MAX_COUNT_EXCEEDED) :
		case (INSERT_NO_MEMORY) :
		default :
			return (EDIT_NO_MEMORY) ;
	}
}

enum UDB_MODIFY udb_update_after_changes ()
{
	int i ;
	RAS_USER_DATABASE_RECORD *ptr_modified_udb, *next_udb_record ;
	BYTE *compressed_udb, *next_block_to_write, *next_block_to_compress ;
	int comp_size, buffer_left_over ;
	BYTE *fl_ptr_udb = (BYTE *)ptr_user_database_hdr->down_load_address ;
	USHORT udb_crc = (USHORT) 0xFFFF ;
	ULONG udb_length = 0 ;
	UDB_HEADER *udb_hdr ;
	BYTE *temp_ptr, two_zeroes[] = {0,0} ;

	if (udb_handler_inited != TRUE)
		return (UDB_HANDLER_NOT_INITED) ;

	ptr_modified_udb = malloc (sizeof(RAS_USER_DATABASE_RECORD) * number_of_ras_users) ;
	if ((ptr_modified_udb == NULL) && (number_of_ras_users))
	{
		printf ("Not enough memory for new UDB\n") ;
		return (UPDATE_NO_MEMORY) ;
	}

	next_udb_record = ptr_modified_udb ;
	for (i = 0 ; i < number_of_ras_users ; i++)
	{
		*next_udb_record++ = *udb_record_ptrs[i] ;
		if (new_node[i] == TRUE)
			free (udb_record_ptrs[i]) ;
	}

	next_block_to_compress = (BYTE *)ptr_modified_udb ;
	buffer_left_over = sizeof(RAS_USER_DATABASE_RECORD) * number_of_ras_users ;
	compressed_udb = (BYTE *) malloc (buffer_left_over + MAX_COMPRESSED_BLOCKS) ;
	if (compressed_udb == NULL)
	{
		printf ("No memory to compress User database\n") ;
		free (ptr_modified_udb) ;
		return (UPDATE_NO_MEMORY) ;
	}

#if 0
	if (no_of_user_database_records)
#else
	if (ptr_ras_user_database != NULL)
#endif
		free (ptr_ras_user_database) ;

	ptr_ras_user_database = ptr_modified_udb ;
	no_of_user_database_records = number_of_ras_users ;

#if DEBUG
	for (i = 0 ; i < no_of_user_database_records ; i++)
		printf ("User %d : %s\n", i, ptr_ras_user_database[i].user_name) ;
#endif

	/* By now, the user database is in place in RAM */
	/* Now compress it to flash */

	udb_length = 0 ;
	udb_crc = 0xFFFF ;
	fl_ptr_udb = (BYTE *)UDB_ADDRESS ;    

	while (buffer_left_over > 0)
	{
		comp_size = LZ15V_compress (next_block_to_compress, ((buffer_left_over > BLOCK_SIZE) ? BLOCK_SIZE : buffer_left_over),
		            compressed_udb + sizeof (USHORT),
						((buffer_left_over > BLOCK_SIZE) ? BLOCK_SIZE : buffer_left_over) + sizeof(USHORT)) ;
		buffer_left_over -= ((buffer_left_over > BLOCK_SIZE) ? BLOCK_SIZE : buffer_left_over) ;
		*((USHORT *)compressed_udb) = comp_size ;
		comp_size += 2 ;

#if DEBUG
		printf ("Compressed size : %d, original size : %d\n",
		         comp_size, ptr_user_database_hdr->length) ;
		test1 = compressed_udb ;
		test2 = (BYTE *)ptr_user_database_hdr->down_load_address;
		for ( i = 0; i < ptr_user_database_hdr->length ; i++)
		  if (*test1++  != *test2++)
		     break;
		printf ("Mismatch after %d bytes\n", i) ;
#endif

		udb_crc = update_crc (udb_crc, compressed_udb, (ULONG)comp_size) ;
		udb_length += comp_size ;
		next_block_to_write = compressed_udb ;

		/* Now, the buffer is compressed */
		while (comp_size >= 512)
		{
			temp_ptr = malloc (512) ;
			if (temp_ptr == NULL)
			{
				printf ("No memory for flash write\n\r") ;
				break ;
			}
			memcpy (temp_ptr, next_block_to_write, 512) ;

			if (schedule_flash_write (temp_ptr, fl_ptr_udb, 512, NULL, temp_ptr) != 0)
				printf ("Schedule flash write failed\n") ;

			comp_size -= 512 ;
			next_block_to_write += 512 ;
			fl_ptr_udb += 512 ;
		}
		temp_ptr = malloc (512) ;
		if (temp_ptr == NULL)
		{
			printf ("No memory for flash write\n\r") ;
			break ;
		}
		memcpy (temp_ptr, fl_ptr_udb, 512) ;
		memcpy (temp_ptr, next_block_to_write, comp_size) ;

		if (schedule_flash_write (temp_ptr, fl_ptr_udb, 512, NULL, temp_ptr) != 0)
			printf ("Schedule flash write failed\n") ;

		comp_size = 0 ;
	}
	free (compressed_udb) ;
	temp_ptr = malloc (512) ;
	if (temp_ptr == NULL)
	{
		printf ("No memory for flash write\n\r") ;
		return (UPDATE_NO_MEMORY) ;
	}
	memcpy (temp_ptr, (BYTE *)ptr_user_database_hdr, 512) ;
	udb_hdr = (UDB_HEADER *)temp_ptr ;

	udb_crc = update_crc (udb_crc, &two_zeroes[0], 2) ;
	udb_hdr->length = udb_length ;
	udb_hdr->no_of_records = no_of_user_database_records ;
	udb_hdr->crc = udb_crc ;
	udb_hdr->down_load_address = UDB_ADDRESS ;
	strcpy (&udb_hdr->date_stamp[0], "NOT AVAILABLE") ;

#if DEBUG
	printf ("UDB: Calculated CRC = 0x%x\r\n", udb_crc);
	printf ("Length written to Flash is %d\n",udb_length);
#endif

	c_write_to_flash (temp_ptr, (BYTE *)UDB_HEADER_ADDRESS, 512) ;
	free (temp_ptr) ;
	/*
	schedule_flash_write (temp_ptr, (BYTE *)UDB_HEADER_ADDRESS, 512, NULL, temp_ptr) ;
	*/

	printf ("User database updated, number of users : %d\n\r", no_of_user_database_records) ;

	return (UPDATE_SUCCESSFUL) ;
}

#if DEBUG
void check_compression_algorithm ()
{
	USHORT comp_size ;
	BYTE *temp_ptr, *fl_temp_ptr ;
	int i ;
	BYTE *comp_udb = malloc (no_of_user_database_records *
	                         sizeof (RAS_USER_DATABASE_RECORD) + 50) ;

	if (comp_udb == NULL)
		return ;

	comp_size = LZ15V_compress (ptr_ras_user_database,
	                            no_of_user_database_records * sizeof (RAS_USER_DATABASE_RECORD),
	                            comp_udb+2,
										 no_of_user_database_records * sizeof (RAS_USER_DATABASE_RECORD)) ;
	*((USHORT *)comp_udb) = comp_size ;
	comp_size += 2 ;

	temp_ptr = comp_udb ;
	fl_temp_ptr = ptr_user_database_hdr->down_load_address ;
	for (i = 0 ; i < ptr_user_database_hdr->length ; i++)
		if (*temp_ptr++ != *fl_temp_ptr++)
			break ;

	printf ("Mismatch after %d bytes\n", i) ;
	printf ("Flash length : %d, DRAM length : %d\n", ptr_user_database_hdr->length, comp_size) ;

	free (comp_udb) ;
}
#endif

