/*----------------------------------------------------------------------------
	File        : PPPUSER.C
	Author      : S.Srilakshmi
	Date        : 06th July 1999
	Description : Contains code to handle everything related to the user database.
----------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdarg.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>

#include "ppp.h"
#include <redblack.h>
#include <flashmgr.h>
#include "..\..\store\boot.h" 

#include "..\..\applicat\userdata\kuser.h"
#include "..\..\applicat\userdata\vuserstr.h"


UDB_HEADER *sptr_user_database_hdr = (UDB_HEADER *) UDB_HEADER_ADDRESS ;
USHORT deleted_group;
BYTE *current_ptr, *fl_ptr_user_udb;
USER_DATABASE *ptr_to_udb_hdr;
BYTE *uncompressed_udb, *current_uncompressed_ptr;

extern USER_DATABASE_CLASS user_database;
extern GROUP_DATA *sptr_temp_group_data;
extern USER_DATA  *sptr_temp_user_data;
extern ULONG dot2ulong(BYTE *);
extern BYTE is_an_ip_address(BYTE* address);
extern void c_write_to_flash (char *source, char *destination, int size) ;
extern BYTE *my_itoa (long number, BYTE *buf, int radix) ;
extern enum BOOLEAN validate_ip_address (BYTE *);
extern void *current_ptr_to_edit;

enum BOOLEAN is_a_port_number(USHORT port_no)
{
	if((port_no >= 1) && (port_no <= MAX_PORT_NUMBER))
		return TRUE;

	return FALSE;
}

enum BOOLEAN is_valid_ip_address_range(ULONG laddress, ULONG uaddress)
{
	/* To check if both r of same Net */
	if((laddress & 0xffffff00) != (uaddress & 0xffffff00))
		return FALSE;

	if(laddress > uaddress)
		return FALSE;

	return TRUE;
}

enum BOOLEAN validate_filter_entries(FILTER_DATA *ptr_data)
{
	ULONG addr1, addr2;

	switch(ptr_data->header->type)
	{
		case APPLICATION_FILTER :
			if(is_a_port_number(ptr_data->port) == FALSE)
			{
				printf("Invalid Port Number\n");
				return FALSE;
			}
			break;

		case IP_ADDRESS_FILTER :
         if ((is_an_ip_address (&ptr_data->address[0]) == FALSE) || (validate_ip_address ((BYTE *) &ptr_data->address[0]) == FALSE))
			{
				printf("Invalid IP Address\n");
				return FALSE;
			}
			break;

		case DOMAIN_NAME_FILTER :
			if((strlen(ptr_data->address) > 255) || (strlen(ptr_data->address) == 0))
			{
				printf("Invalid Domain Name - Exceeds 255 characters\n");
				return FALSE;
			}
			break;
	}
	return TRUE;
}

enum BOOLEAN validate_group_entry(TEMP_GROUP_PROFILE *ptr_data)
{
	if((strlen(ptr_data->group_name) >= MAX_LENGTH_GROUP_PROFILE_NAME) || (strlen(ptr_data->group_description) >= MAX_LENGTH_GROUP_PROFILE_NAME))
		return FALSE;

	return TRUE;
}

enum BOOLEAN validate_user_entry(TEMP_USER_PROFILE *ptr_data)
{
	if((strlen(ptr_data->user_name) >= MAX_LENGTH_GROUP_PROFILE_NAME))
		return FALSE;

	return TRUE;
}

enum BOOLEAN validate_app_default_entry(TEMP_APP_DEFAULT *ptr_data)
{
	if(is_a_port_number(ptr_data->lower_port) == FALSE)
		return FALSE;

	if(is_a_port_number(ptr_data->upper_port) == FALSE)
		return FALSE;

	if(ptr_data->lower_port > ptr_data->upper_port)
		return FALSE;

	return TRUE;
}

enum BOOLEAN validate_ip_default_entry(TEMP_IP_DEFAULT *ptr_data)
{
	ULONG laddr, uaddr;

	laddr = dot2ulong(ptr_data->lower_address);
	uaddr = dot2ulong(ptr_data->upper_address);

   if ((validate_ip_address ((BYTE *) ptr_data->lower_address) == FALSE)
			|| (validate_ip_address ((BYTE *) ptr_data->upper_address) == FALSE))
		return FALSE;

   if ((is_an_ip_address ((BYTE *) ptr_data->lower_address) == FALSE)
			|| (is_an_ip_address ((BYTE *) ptr_data->upper_address) == FALSE))
		return FALSE;

	if(is_valid_ip_address_range(laddr, uaddr) == FALSE)
		return FALSE;

	return TRUE;
}

enum BOOLEAN check_for_filter_entries(FILTER_DATA *ptr_filter, FILTER_DATA *ptr_data)
{
	ULONG addr1, addr2;

	if(ptr_data->header->type != ptr_filter->header->type)
		return FALSE;		

	switch(ptr_data->header->type)
	{
		case APPLICATION_FILTER :
			if((ptr_filter->protocol == ptr_data->protocol) && (ptr_filter->port == ptr_data->port))
				return TRUE;
			break;

		case IP_ADDRESS_FILTER :
			addr1 = dot2ulong(&ptr_filter->address[0]);
			addr2 = dot2ulong(&ptr_data->address[0]);
			if(addr1 == addr2)
				return TRUE;
			break;

		case DOMAIN_NAME_FILTER :
			if(!strcmp(ptr_filter->address, ptr_data->address))
				return TRUE;
			break;
	}
	return FALSE;
}

/*----------------------------------------------------------------------------
				Funtions For Deleting, Adding, Editing etc. Groups
----------------------------------------------------------------------------*/
void add_entry_to_group_list(GROUP_DATA *ptr_data)
{
	GROUP_DATA *ptr_to_list = user_database.sptr_temp_group_data;

/*	printf("Group : %s\n", ptr_data->data->group_name); */
	if(user_database.sptr_temp_group_data == NULL)
	{
		user_database.sptr_temp_group_data = ptr_data;
		user_database.sptr_temp_group_data->next_node = NULL;
/*		printf("Group : Adding First Node\n"); */
		return;
	}

	while(1)
	{
		if(ptr_to_list->next_node == NULL)
		{
			ptr_to_list->next_node = ptr_data;
			ptr_to_list->next_node->next_node = NULL;
/*			printf("Group : Adding Node\n"); */
			return;
		}
		else
			ptr_to_list = ptr_to_list->next_node;
	}
}		

void delete_next_node_from_group_list (GROUP_DATA *ptr_data)
{
	GROUP_DATA *temp_ptr;
	if(ptr_data == NULL)
		return;
	
	temp_ptr = ptr_data->next_node->next_node;
	deleted_group = ptr_data->next_node->data->group_no;
	free(ptr_data->next_node->data);
	free(ptr_data->next_node);
	ptr_data->next_node = temp_ptr;
}

enum UDB_MODIFY udb_group_insert(TEMP_GROUP_PROFILE *ptr_data)
{
	GROUP_DATA *ptr_to_group_profile, *group;
	TEMP_GROUP_PROFILE *data;

	if(validate_group_entry(ptr_data) == FALSE)
		return ILLEGAL_DATA;

	ptr_to_group_profile = user_database.sptr_temp_group_data;
	data = (TEMP_GROUP_PROFILE *) calloc (sizeof(TEMP_GROUP_PROFILE), 1);
	memcpy(data, ptr_data, sizeof(TEMP_GROUP_PROFILE));

	group = (GROUP_DATA *) calloc (sizeof(GROUP_DATA), 1);
	group->data = data;
	group->count = 0;
	group->filter_action = DEFAULT_ACTION;
	group->prev_node = group->next_node = NULL;
	if(user_database.sptr_temp_group_data == NULL)
	{
		user_database.sptr_temp_group_data = group;
		user_database.num_of_groups = 1;
		return (INSERT_SUCCESSFUL) ;
	}

	while (ptr_to_group_profile != NULL)
	{
		if(!strcmp(ptr_to_group_profile->data->group_name, ptr_data->group_name))
		{
			free (group);
			return (INSERT_USER_ALREADY_EXISTS) ;
		}
		else
		{
			if(ptr_to_group_profile->next_node == NULL)
				break;
			ptr_to_group_profile = ptr_to_group_profile->next_node;
		}
	}
	group->prev_node = ptr_to_group_profile;
	ptr_to_group_profile->next_node = group;
	user_database.num_of_groups++;
	return (INSERT_SUCCESSFUL) ;
}

enum UDB_MODIFY udb_group_delete(TEMP_GROUP_PROFILE *ptr_data)
{
	GROUP_DATA *ptr_to_group_profile, *ptr_to_next_node;

	ptr_to_group_profile = user_database.sptr_temp_group_data;
	if(!strcmp(ptr_to_group_profile->data->group_name,ptr_data->group_name))
	{
		user_database.sptr_temp_group_data = user_database.sptr_temp_group_data->next_node;
		deleted_group = ptr_to_group_profile->data->group_no;
		free(ptr_to_group_profile->data);
		free(ptr_to_group_profile);
		user_database.num_of_groups--;
		return (DELETE_SUCCESSFUL);
	}

	while(ptr_to_group_profile != NULL)
	{
		ptr_to_next_node = ptr_to_group_profile->next_node;
		if(ptr_to_next_node == NULL)
			break;
		
		if(!strcmp(ptr_to_next_node->data->group_name, ptr_data->group_name))
		{
			delete_next_node_from_group_list (ptr_to_group_profile);
			user_database.num_of_groups--;
			return (DELETE_SUCCESSFUL);
		}
		else
			ptr_to_group_profile = ptr_to_group_profile->next_node;
	}
	return (DELETE_USER_NOT_FOUND);
}

enum UDB_MODIFY udb_group_edit(TEMP_GROUP_PROFILE *ptr_data)
{
	GROUP_DATA *ptr_to_group_profile;

	if(validate_group_entry(ptr_data) == FALSE)
		return ILLEGAL_DATA;

	ptr_to_group_profile = user_database.sptr_temp_group_data;

	while (ptr_to_group_profile != NULL)
	{
		if(!strcmp(ptr_to_group_profile->data->group_name, ptr_data->group_name))
		{
			memcpy(ptr_to_group_profile->data, ptr_data, sizeof(TEMP_GROUP_PROFILE));
			return (EDIT_SUCCESSFUL) ;
		}
		else
			ptr_to_group_profile = ptr_to_group_profile->next_node;
	}
	free (ptr_data);
	return (EDIT_RECORD_NOT_FOUND) ;
}

/*----------------------------------------------------------------------------
				Funtions For Deleting, Adding, Editing etc. Users
----------------------------------------------------------------------------*/
void add_entry_to_user_list(USER_DATA *ptr_data)
{
	USER_DATA *ptr_to_list = user_database.sptr_temp_user_data;

/*	printf("User : %s\n", ptr_data->data->user_name); */
	if(user_database.sptr_temp_user_data == NULL)
	{
		user_database.sptr_temp_user_data = ptr_data;
		user_database.sptr_temp_user_data->next_node = NULL;
/*		printf("User : Adding First Node\n"); */
		return;
	}

	while(1)
	{
		if(ptr_to_list->next_node == NULL)
		{
			ptr_to_list->next_node = ptr_data;
			ptr_to_list->next_node->next_node = NULL;
/*			printf("User : Adding Node\n"); */
			return;
		}
		else
			ptr_to_list = ptr_to_list->next_node;
	}
}

void delete_next_node_from_user_list (USER_DATA *ptr_data)
{
	USER_DATA *temp_ptr;
	if(ptr_data == NULL)
		return;
	
	temp_ptr = ptr_data->next_node->next_node;
	free(ptr_data->next_node->data);
	free(ptr_data->next_node);
	ptr_data->next_node = temp_ptr;
}

enum UDB_MODIFY udb_user_insert(TEMP_USER_PROFILE *ptr_data)
{
	USER_DATA *ptr_to_user_profile, *user;
	TEMP_USER_PROFILE *data;

	if(validate_user_entry(ptr_data) == FALSE)
		return ILLEGAL_DATA;

	ptr_to_user_profile = user_database.sptr_temp_user_data;
	data = (TEMP_USER_PROFILE *) calloc (sizeof(TEMP_USER_PROFILE), 1);
	memcpy(data, ptr_data, sizeof(TEMP_USER_PROFILE));

	user = (USER_DATA *) calloc (sizeof(USER_DATA), 1);
	user->data = data;
	user->prev_node = user->next_node = NULL;
	if(user_database.sptr_temp_user_data == NULL)
	{
		user_database.sptr_temp_user_data = user;
		user_database.num_of_users = 1;
		return (INSERT_SUCCESSFUL) ;
	}

	while (ptr_to_user_profile != NULL)
	{
		if(!strcmp(ptr_to_user_profile->data->user_name, ptr_data->user_name))
		{
			free (user);
			return (INSERT_USER_ALREADY_EXISTS) ;
		}
		else
		{
			if(ptr_to_user_profile->next_node == NULL)
				break;
			ptr_to_user_profile = ptr_to_user_profile->next_node;
		}
	}

	user->prev_node = ptr_to_user_profile;
	ptr_to_user_profile->next_node = user;
	user_database.num_of_users++;
	return (INSERT_SUCCESSFUL) ;
}

enum UDB_MODIFY udb_user_delete(TEMP_USER_PROFILE *ptr_data)
{
	USER_DATA *ptr_to_user_profile, *ptr_to_next_node;

	ptr_to_user_profile = user_database.sptr_temp_user_data;
	if(!strcmp(ptr_to_user_profile->data->user_name,ptr_data->user_name))
	{
		user_database.sptr_temp_user_data = user_database.sptr_temp_user_data->next_node;
		free(ptr_to_user_profile->data);
		free(ptr_to_user_profile);
		user_database.num_of_users--;
		return (DELETE_SUCCESSFUL);
	}

	while(ptr_to_user_profile != NULL)
	{
		ptr_to_next_node = ptr_to_user_profile->next_node;
		if(ptr_to_next_node == NULL)
			break;
		
		if(!strcmp(ptr_to_next_node->data->user_name, ptr_data->user_name))
		{
			delete_next_node_from_user_list (ptr_to_user_profile);
			user_database.num_of_users--;
			return (DELETE_SUCCESSFUL);
		}
		else
			ptr_to_user_profile = ptr_to_user_profile->next_node;
	}
	return (DELETE_USER_NOT_FOUND);
}

enum UDB_MODIFY udb_user_edit(TEMP_USER_PROFILE *ptr_data)
{
	USER_DATA *ptr_to_user_profile;

	if(validate_user_entry(ptr_data) == FALSE)
		return ILLEGAL_DATA;

	ptr_to_user_profile = user_database.sptr_temp_user_data;

	while (ptr_to_user_profile != NULL)
	{
		if(!strcmp(ptr_to_user_profile->data->user_name, ptr_data->user_name))
		{
			memcpy(ptr_to_user_profile->data, ptr_data, sizeof(TEMP_USER_PROFILE));
			return (EDIT_SUCCESSFUL) ;
		}
		else
			ptr_to_user_profile = ptr_to_user_profile->next_node;
	}
	free (ptr_data);
	return (EDIT_RECORD_NOT_FOUND) ;
}

/*----------------------------------------------------------------------------
				Funtions For Deleting, Adding, Editing etc. IP Defaults
----------------------------------------------------------------------------*/
void add_entry_to_ip_default_list(IP_DEFAULT_DATA *ptr_data)
{
	IP_DEFAULT_DATA *ptr_to_list = user_database.sptr_temp_ip_default_data;

/*	printf("IP_DFT : %s\n", ptr_data->data->lower_address); */
	if(user_database.sptr_temp_ip_default_data == NULL)
	{
		user_database.sptr_temp_ip_default_data = ptr_data;
		user_database.sptr_temp_ip_default_data->next_node = NULL;
/*		printf("IP_DFT : Adding First Node\n"); */
		return;
	}

	while(1)
	{
		if(ptr_to_list->next_node == NULL)
		{
			ptr_to_list->next_node = ptr_data;
			ptr_to_list->next_node->next_node = NULL;
/*			printf("IP_DFT : Adding Node\n"); */
			return;
		}
		else
			ptr_to_list = ptr_to_list->next_node;
	}
}

void delete_next_node_from_ip_default_list (IP_DEFAULT_DATA *ptr_data)
{
	IP_DEFAULT_DATA *temp_ptr;
	if(ptr_data == NULL)
		return;
	
	temp_ptr = ptr_data->next_node->next_node;
	free(ptr_data->next_node->data);
	free(ptr_data->next_node);
	ptr_data->next_node = temp_ptr;
}

enum UDB_MODIFY udb_ip_default_insert(TEMP_IP_DEFAULT *ptr_data)
{
	IP_DEFAULT_DATA *ptr_to_ip_default_profile, *ip_default;
	TEMP_IP_DEFAULT *data;
	ULONG laddress, uaddress, laddr, uaddr;

	if(validate_ip_default_entry(ptr_data) == FALSE)
		return ILLEGAL_DATA;

	laddr = dot2ulong(ptr_data->lower_address);
	uaddr = dot2ulong(ptr_data->upper_address);

	ptr_to_ip_default_profile = user_database.sptr_temp_ip_default_data;
	data = (TEMP_IP_DEFAULT *) calloc (sizeof(TEMP_IP_DEFAULT), 1);
	memcpy(data, ptr_data, sizeof(TEMP_IP_DEFAULT));

	ip_default = (IP_DEFAULT_DATA *) calloc (sizeof(IP_DEFAULT_DATA), 1);
	ip_default->data = data;
	ip_default->prev_node = ip_default->next_node = NULL;
	if(user_database.sptr_temp_ip_default_data == NULL)
	{
		user_database.sptr_temp_ip_default_data = ip_default;
		user_database.num_of_ipdefaults = 1;
		return (INSERT_SUCCESSFUL) ;
	}

	while (ptr_to_ip_default_profile != NULL)
	{
		laddress = dot2ulong(ptr_to_ip_default_profile->data->lower_address);
		uaddress = dot2ulong(ptr_to_ip_default_profile->data->upper_address);
		
		if((laddr >= laddress && laddr <= uaddress) || (uaddr >= laddress && uaddr <= uaddress))
		{
			free (ip_default);
			return (INSERT_USER_ALREADY_EXISTS) ;
		}

		if((laddress >= laddr && laddress <= uaddr) || (uaddress >= laddr && uaddress <= uaddr))
		{
			free (ip_default);
			return (INSERT_USER_ALREADY_EXISTS) ;
		}

		if(ptr_to_ip_default_profile->next_node == NULL)
			break;
		ptr_to_ip_default_profile = ptr_to_ip_default_profile->next_node;
	}
	ip_default->prev_node =	ptr_to_ip_default_profile;
	ptr_to_ip_default_profile->next_node = ip_default;
	user_database.num_of_ipdefaults++;
	return (INSERT_SUCCESSFUL) ;
}

enum UDB_MODIFY udb_ip_default_delete(TEMP_IP_DEFAULT *ptr_data)
{
	IP_DEFAULT_DATA *ptr_to_ip_default_profile, *ptr_to_next_node;
	ULONG laddress, uaddress, laddr, uaddr;

	laddr = dot2ulong(ptr_data->lower_address);
	uaddr = dot2ulong(ptr_data->upper_address);

	ptr_to_ip_default_profile = user_database.sptr_temp_ip_default_data;
	laddress = dot2ulong(ptr_to_ip_default_profile->data->lower_address);
	uaddress = dot2ulong(ptr_to_ip_default_profile->data->upper_address);
	
	if((laddr == laddress) && (uaddr == uaddress))
	{
		user_database.sptr_temp_ip_default_data = user_database.sptr_temp_ip_default_data->next_node;
		free(ptr_to_ip_default_profile->data);
		free(ptr_to_ip_default_profile);
		user_database.num_of_ipdefaults--;
		return (DELETE_SUCCESSFUL);
	}

	while(ptr_to_ip_default_profile != NULL)
	{
		ptr_to_next_node = ptr_to_ip_default_profile->next_node;
		if(ptr_to_next_node == NULL)
			break;

		laddress = dot2ulong(ptr_to_next_node->data->lower_address);
		uaddress = dot2ulong(ptr_to_next_node->data->upper_address);
			
		if((laddr == laddress) && (uaddr == uaddress))
		{
			delete_next_node_from_ip_default_list (ptr_to_ip_default_profile);
			user_database.num_of_ipdefaults--;
			return (DELETE_SUCCESSFUL);
		}
		else
			ptr_to_ip_default_profile = ptr_to_ip_default_profile->next_node;
	}
	return (DELETE_USER_NOT_FOUND);
}

enum UDB_MODIFY udb_ip_default_edit(TEMP_IP_DEFAULT *ptr_data)
{
	IP_DEFAULT_DATA *ptr_to_ip_default_profile;
	ULONG laddress, uaddress, laddr, uaddr;

	if(validate_ip_default_entry(ptr_data) == FALSE)
		return ILLEGAL_DATA;

	ptr_to_ip_default_profile = user_database.sptr_temp_ip_default_data;
	laddr = dot2ulong(ptr_data->lower_address);
	uaddr = dot2ulong(ptr_data->upper_address);

	while (ptr_to_ip_default_profile != NULL)
	{
    if(current_ptr_to_edit != ptr_to_ip_default_profile)
	 {	
		laddress = dot2ulong(ptr_to_ip_default_profile->data->lower_address);
		uaddress = dot2ulong(ptr_to_ip_default_profile->data->upper_address);
	
		if((laddr >= laddress && laddr <= uaddress) || (uaddr >= laddress && uaddr <= uaddress))
			return (INSERT_USER_ALREADY_EXISTS) ;

		if((laddress >= laddr && laddress <= uaddr) || (uaddress >= laddr && uaddress <= uaddr))
			return (INSERT_USER_ALREADY_EXISTS) ;
	 }

		if(ptr_to_ip_default_profile->next_node == NULL)
			break;
		ptr_to_ip_default_profile = ptr_to_ip_default_profile->next_node;
	}
	return (EDIT_SUCCESSFUL) ;
}
/*----------------------------------------------------------------------------
				Funtions For Deleting, Adding, Editing etc. APP Defaults
----------------------------------------------------------------------------*/
void add_entry_to_app_default_list(APP_DEFAULT_DATA *ptr_data)
{
	APP_DEFAULT_DATA *ptr_to_list = user_database.sptr_temp_app_default_data;

/*	printf("APP_DFT : %d\n", ptr_data->data->lower_port); */
	if(user_database.sptr_temp_app_default_data == NULL)
	{
		user_database.sptr_temp_app_default_data = ptr_data;
		user_database.sptr_temp_app_default_data->next_node = NULL;
/*		printf("APP_DFT : Adding First Node\n"); */
		return;
	}

	while(1)
	{
		if(ptr_to_list->next_node == NULL)
		{
			ptr_to_list->next_node = ptr_data;
			ptr_to_list->next_node->next_node = NULL;
/*			printf("APP_DFT : Adding Node\n"); */
			return;
		}
		else
			ptr_to_list = ptr_to_list->next_node;
	}
}

void delete_next_node_from_app_default_list (APP_DEFAULT_DATA *ptr_data)
{
	APP_DEFAULT_DATA *temp_ptr;
	if(ptr_data == NULL)
		return;
	
	temp_ptr = ptr_data->next_node->next_node;
	free(ptr_data->next_node->data);
	free(ptr_data->next_node);
	ptr_data->next_node = temp_ptr;
}

enum UDB_MODIFY udb_app_default_insert(TEMP_APP_DEFAULT *ptr_data)
{
	APP_DEFAULT_DATA *ptr_to_app_default_profile, *app_default;
	TEMP_APP_DEFAULT *data;
	USHORT lport, uport;

	if(validate_app_default_entry(ptr_data) == FALSE)
		return ILLEGAL_DATA;

	lport = ptr_data->lower_port;
	uport = ptr_data->upper_port;

	ptr_to_app_default_profile = user_database.sptr_temp_app_default_data;
	data = (TEMP_APP_DEFAULT *) calloc (sizeof(TEMP_APP_DEFAULT), 1);
	memcpy(data, ptr_data, sizeof(TEMP_APP_DEFAULT));

	app_default = (APP_DEFAULT_DATA *)	calloc(sizeof(APP_DEFAULT_DATA), 1);
	app_default->data = data;
	app_default->prev_node = app_default->next_node = NULL;
	if(user_database.sptr_temp_app_default_data == NULL)
	{
		user_database.sptr_temp_app_default_data = app_default;
		user_database.num_of_appdefaults = 1;
		return (INSERT_SUCCESSFUL) ;
	}

	while (ptr_to_app_default_profile != NULL)
	{
		if(ptr_to_app_default_profile->data->protocol == ptr_data->protocol)
		{
			if((ptr_to_app_default_profile->data->lower_port >= lport && ptr_to_app_default_profile->data->lower_port <= uport)
				|| (ptr_to_app_default_profile->data->upper_port >= lport && ptr_to_app_default_profile->data->upper_port <= uport))
			{
				free(app_default);
				return (INSERT_USER_ALREADY_EXISTS) ;
			}

			if((lport >= ptr_to_app_default_profile->data->lower_port && lport <= ptr_to_app_default_profile->data->upper_port)
				|| (uport >= ptr_to_app_default_profile->data->lower_port && uport <= ptr_to_app_default_profile->data->upper_port))
			{
				free(app_default);
				return (INSERT_USER_ALREADY_EXISTS) ;
			}
		}

		if(ptr_to_app_default_profile->next_node == NULL)
			break;
		ptr_to_app_default_profile = ptr_to_app_default_profile->next_node;
	}
	app_default->prev_node = ptr_to_app_default_profile;
	ptr_to_app_default_profile->next_node = app_default;
	user_database.num_of_appdefaults++;
	return (INSERT_SUCCESSFUL) ;
}

enum UDB_MODIFY udb_app_default_delete(TEMP_APP_DEFAULT *ptr_data)
{
	APP_DEFAULT_DATA *ptr_to_app_default_profile, *ptr_to_next_node;

	ptr_to_app_default_profile = user_database.sptr_temp_app_default_data;
	if((ptr_to_app_default_profile->data->protocol == ptr_data->protocol) &&
		  (ptr_to_app_default_profile->data->lower_port == ptr_data->lower_port) &&
			 (ptr_to_app_default_profile->data->upper_port == ptr_data->upper_port))
	{
		user_database.sptr_temp_app_default_data = user_database.sptr_temp_app_default_data->next_node;
		free(ptr_to_app_default_profile->data);
		free(ptr_to_app_default_profile);
		user_database.num_of_appdefaults--;
		return (DELETE_SUCCESSFUL);
	}

	while(ptr_to_app_default_profile != NULL)
	{
		ptr_to_next_node = ptr_to_app_default_profile->next_node;
		if(ptr_to_next_node == NULL)
			break;
		
		if((ptr_to_next_node->data->protocol == ptr_data->protocol) &&
		    (ptr_to_next_node->data->lower_port == ptr_data->lower_port) &&
			  (ptr_to_next_node->data->upper_port == ptr_data->upper_port))
		{
			delete_next_node_from_app_default_list (ptr_to_app_default_profile);
			user_database.num_of_appdefaults--;
			return (DELETE_SUCCESSFUL);
		}
		else
			ptr_to_app_default_profile = ptr_to_app_default_profile->next_node;
	}
	return (DELETE_USER_NOT_FOUND);
}

enum UDB_MODIFY udb_app_default_edit(TEMP_APP_DEFAULT *ptr_data)
{
	APP_DEFAULT_DATA *ptr_to_app_default_profile;
	USHORT lport, uport;

	if(validate_app_default_entry(ptr_data) == FALSE)
		return ILLEGAL_DATA;

	lport = ptr_data->lower_port;
	uport = ptr_data->upper_port;

	ptr_to_app_default_profile = user_database.sptr_temp_app_default_data;
	while (ptr_to_app_default_profile != NULL)
	{
    if(current_ptr_to_edit != ptr_to_app_default_profile)
	 {
		if(ptr_to_app_default_profile->data->protocol == ptr_data->protocol)
		{
			if((ptr_to_app_default_profile->data->lower_port >= lport && ptr_to_app_default_profile->data->lower_port <= uport)
				|| (ptr_to_app_default_profile->data->upper_port >= lport && ptr_to_app_default_profile->data->upper_port <= uport))
					return (INSERT_USER_ALREADY_EXISTS) ;

			if((lport >= ptr_to_app_default_profile->data->lower_port && lport <= ptr_to_app_default_profile->data->upper_port)
				|| (uport >= ptr_to_app_default_profile->data->lower_port && uport <= ptr_to_app_default_profile->data->upper_port))
					return (INSERT_USER_ALREADY_EXISTS) ;
		}
	 }	

		if(ptr_to_app_default_profile->next_node == NULL)
			break;
		ptr_to_app_default_profile = ptr_to_app_default_profile->next_node;
	}
	return (EDIT_SUCCESSFUL) ;
}

/*----------------------------------------------------------------------------
				Funtions For Deleting, Adding, Editing etc. Filters
----------------------------------------------------------------------------*/
void add_entry_to_filter_list(FILTER_DATA *ptr_data)
{
	FILTER_DATA *ptr_to_list = user_database.sptr_temp_filter_data;

/*	printf("FILTER : %d\n", ptr_data->header->type); */
	if(user_database.sptr_temp_filter_data == NULL)
	{
		user_database.sptr_temp_filter_data = ptr_data;
		user_database.sptr_temp_filter_data->next_node = NULL;
/*		printf("FILTER : Adding First Node\n"); */
		return;
	}

	while(1)
	{
		if(ptr_to_list->next_node == NULL)
		{
			ptr_to_list->next_node = ptr_data;
			ptr_to_list->next_node->next_node = NULL;
/*			printf("FILTER : Adding Node\n"); */
			return;
		}
		else
			ptr_to_list = ptr_to_list->next_node;
	}
}

void delete_next_node_from_filter_list (FILTER_DATA *ptr_data)
{
	FILTER_DATA *temp_ptr;
	if(ptr_data == NULL)
		return;
	
	temp_ptr = ptr_data->next_node->next_node;
	free(ptr_data->next_node->header);
	free(ptr_data->next_node);
	ptr_data->next_node = temp_ptr;
}

enum UDB_MODIFY udb_filter_insert(FILTER_DATA *ptr_to_filter_data)
{
	TEMP_FILTER_HEADER *header;
	FILTER_DATA *ptr_filter_profile = user_database.sptr_temp_filter_data, *ptr_data;

	if(validate_filter_entries(ptr_to_filter_data) == FALSE)
		return ILLEGAL_DATA;

	header = (TEMP_FILTER_HEADER *) calloc(sizeof(TEMP_FILTER_HEADER), 1);
	if(header == NULL)
	{
		printf("No Enough Memory\n");
		return INSERT_NO_MEMORY;
	}
	memcpy(header, ptr_to_filter_data->header, sizeof(TEMP_FILTER_HEADER));

	ptr_data = (FILTER_DATA *) calloc(sizeof(FILTER_DATA), 1);
	if(ptr_data == NULL)
	{
		printf("No Enough Memory\n");
		return INSERT_NO_MEMORY;
	}

	memcpy(ptr_data, ptr_to_filter_data, sizeof(FILTER_DATA));
	ptr_data->header = header;
	ptr_data->prev_node = ptr_data->next_node = NULL;	
	if(user_database.sptr_temp_filter_data == NULL)
	{
		user_database.sptr_temp_filter_data = ptr_data;
		user_database.num_of_filters = 1;
		return (INSERT_SUCCESSFUL) ;
	}

	while(ptr_filter_profile != NULL)
	{
		if(check_for_filter_entries(ptr_filter_profile, ptr_data))
			return (INSERT_USER_ALREADY_EXISTS) ;

		if(ptr_filter_profile->next_node == NULL)
			break;

		ptr_filter_profile = ptr_filter_profile->next_node;
	}
	ptr_data->prev_node = ptr_filter_profile;
	ptr_filter_profile->next_node = ptr_data;
	user_database.num_of_filters++;
	return (INSERT_SUCCESSFUL) ;
}

enum UDB_MODIFY udb_filter_delete(FILTER_DATA *ptr_data)
{
	FILTER_DATA *ptr_filter_profile, *ptr_to_next_node;
	
	ptr_filter_profile = user_database.sptr_temp_filter_data;
	if(check_for_filter_entries(ptr_filter_profile, ptr_data))
	{
		user_database.sptr_temp_filter_data = user_database.sptr_temp_filter_data->next_node;
		free(ptr_filter_profile->header);
		free(ptr_filter_profile);
		user_database.num_of_filters--;
		return (DELETE_SUCCESSFUL);
	}

	while(ptr_filter_profile != NULL)
	{
		ptr_to_next_node = ptr_filter_profile->next_node;
		if(ptr_to_next_node == NULL)
			break;

		if(check_for_filter_entries(ptr_to_next_node, ptr_data))
		{
			delete_next_node_from_filter_list (ptr_filter_profile);
			user_database.num_of_filters--;
			return (DELETE_SUCCESSFUL);
		}
		else
			ptr_filter_profile = ptr_filter_profile->next_node;
	}
	return (DELETE_USER_NOT_FOUND);
}

void group_initialize()
{
	TEMP_GROUP_PROFILE *temp_group_profile;
	GROUP_DATA *ptr_group_profile;
	int index;

	user_database.num_of_groups = ptr_to_udb_hdr->number_of_items;
	for (index = 0; index < ptr_to_udb_hdr->number_of_items; index++)
	{
		temp_group_profile = (TEMP_GROUP_PROFILE *) current_ptr;

		ptr_group_profile = (GROUP_DATA *) calloc(sizeof(GROUP_DATA),1); 
		if (ptr_group_profile == NULL)
		{
			printf("GROUP: Error Allocating Memory\n");
			return;
		}

		ptr_group_profile->data = (TEMP_GROUP_PROFILE *) calloc(sizeof(TEMP_GROUP_PROFILE),1); 
		if (ptr_group_profile->data == NULL)
		{
			printf("GROUP: Error Allocating Memory\n");
			current_ptr += sizeof(TEMP_GROUP_PROFILE);
			return;
		}

		memcpy(ptr_group_profile->data, temp_group_profile, sizeof(TEMP_GROUP_PROFILE));
		ptr_group_profile->next_node = NULL;
		ptr_group_profile->filter_action = ptr_group_profile->data->allow_sites;

		add_entry_to_group_list(ptr_group_profile); 

		current_ptr += sizeof(TEMP_GROUP_PROFILE);
	}	
}

void user_initialize()
{
	TEMP_USER_PROFILE *temp_user_profile;
	USER_DATA *ptr_user_profile;
	int index;

	user_database.num_of_users = ptr_to_udb_hdr->number_of_items;
	for (index = 0; index < ptr_to_udb_hdr->number_of_items; index++)
	{
		temp_user_profile = (TEMP_USER_PROFILE *) current_ptr;

		ptr_user_profile = (USER_DATA *) calloc(sizeof(USER_DATA),1); 
		if (ptr_user_profile == NULL)
		{
			printf("GROUP: Error Allocating Memory\n");
			return;
		}

		ptr_user_profile->data = (TEMP_USER_PROFILE *) calloc(sizeof(TEMP_USER_PROFILE),1); 
		if (ptr_user_profile->data == NULL)
		{
			printf("GROUP: Error Allocating Memory\n");
			current_ptr += sizeof(TEMP_USER_PROFILE);
			return;
		}

		memcpy(ptr_user_profile->data, temp_user_profile, sizeof(TEMP_USER_PROFILE));
		ptr_user_profile->next_node = NULL;

		add_entry_to_user_list(ptr_user_profile);

		current_ptr += sizeof(TEMP_USER_PROFILE);
	}	
}

void ip_default_initialize()
{
	TEMP_IP_DEFAULT *temp_ip_default;
	IP_DEFAULT_DATA *ptr_ip_default;
	int index;

/* Initialize here for user_database_enabled & default_filter_action */
	user_database.temp_enabled = *(USHORT *)current_ptr;
	current_ptr += sizeof(USHORT);	 
	user_database.temp_action = *(USHORT *)current_ptr;
	current_ptr += sizeof(USHORT);	 

	user_database.num_of_ipdefaults = ptr_to_udb_hdr->number_of_items;
	for (index = 0; index < ptr_to_udb_hdr->number_of_items; index++)
	{
		temp_ip_default = (TEMP_IP_DEFAULT *) current_ptr;

		ptr_ip_default = (IP_DEFAULT_DATA *) calloc(sizeof(IP_DEFAULT_DATA),1);
		if (ptr_ip_default == NULL)
		{
			printf("GROUP: Error Allocating Memory\n");
			return;
		}

		ptr_ip_default->data = (TEMP_IP_DEFAULT *) calloc(sizeof(TEMP_IP_DEFAULT),1);
		if (ptr_ip_default->data == NULL)
		{
			printf("GROUP: Error Allocating Memory\n");
			current_ptr += sizeof(TEMP_IP_DEFAULT);
			return;
		}

		memcpy(ptr_ip_default->data, temp_ip_default, sizeof(TEMP_IP_DEFAULT));
		ptr_ip_default->next_node = NULL;

		add_entry_to_ip_default_list(ptr_ip_default);

		current_ptr += sizeof(TEMP_IP_DEFAULT);
	}	
}

void app_default_initialize()
{
	TEMP_APP_DEFAULT *temp_app_default;
	APP_DEFAULT_DATA *ptr_app_default;
	int index;

	user_database.num_of_appdefaults = ptr_to_udb_hdr->number_of_items;
	for (index = 0; index < ptr_to_udb_hdr->number_of_items; index++)
	{
		temp_app_default = (TEMP_APP_DEFAULT *) current_ptr;

		ptr_app_default = (APP_DEFAULT_DATA *) calloc(sizeof(APP_DEFAULT_DATA),1);
		if (ptr_app_default == NULL)
		{
			printf("GROUP: Error Allocating Memory\n");
			return;
		}

		ptr_app_default->data = (TEMP_APP_DEFAULT *) calloc(sizeof(TEMP_APP_DEFAULT),1);
		if (ptr_app_default->data == NULL)
		{
			printf("GROUP: Error Allocating Memory\n");
			current_ptr += sizeof(TEMP_APP_DEFAULT);
			return;
		}

		memcpy(ptr_app_default->data, temp_app_default, sizeof(TEMP_APP_DEFAULT));
		ptr_app_default->next_node = NULL;

		add_entry_to_app_default_list(ptr_app_default);

		current_ptr += sizeof(TEMP_APP_DEFAULT);
	}	
}

void filter_initialize()
{
	int index;
	TEMP_FILTER_HEADER *filter_header;
	FILTER_DATA *filter_profile;
	
	user_database.num_of_filters = ptr_to_udb_hdr->number_of_items;
	for (index = 0; index < ptr_to_udb_hdr->number_of_items; index++)
	{
		filter_header = (TEMP_FILTER_HEADER *) current_ptr;
		current_ptr += sizeof(TEMP_FILTER_HEADER);	 
		filter_profile = (FILTER_DATA *) calloc (sizeof(FILTER_DATA), 1);
		if(filter_profile == NULL)
		{
			printf("FILTER: Error Allocating Memory\n");
			return;
		}

		filter_profile->header = (TEMP_FILTER_HEADER *) calloc (sizeof(TEMP_FILTER_HEADER), 1);
		if(filter_profile->header == NULL)
		{
			printf("FILTER: Error Allocating Memory\n");
			return;
		}

		filter_profile->next_node = NULL;
		memcpy(filter_profile->header, filter_header, sizeof(TEMP_FILTER_HEADER));
		if(filter_profile->header->type == DOMAIN_NAME_FILTER)
			user_database.num_of_domains++;

		switch(filter_profile->header->type)
		{
			case APPLICATION_FILTER :
				filter_profile->port = *(USHORT *)current_ptr;
				current_ptr += sizeof(USHORT);	 
				filter_profile->protocol = *(USHORT *)current_ptr;
				current_ptr += sizeof(USHORT);	 
				break;
			case IP_ADDRESS_FILTER :
			case DOMAIN_NAME_FILTER :
				memcpy (&filter_profile->address[0], (BYTE *)current_ptr, filter_profile->header->length+1);
				current_ptr += (filter_profile->header->length + 1);
				break;
		}
		add_entry_to_filter_list (filter_profile);
	}
}
	
enum TEST udb_user_initialize()
{
	/* UDB Records from Flash will be writen to temporary Linked List */
	printf("UDB User Initialize\n"); 

	current_ptr = (BYTE *) sptr_user_database_hdr->down_load_address;

/* Added by Sreelu - newaddition */
	if(!strcmp(sptr_user_database_hdr->reserved, "USERDEFAULT"))
	{
/* Set the Default Settings & don't initialise the database fields */
printf("PPPUSER:Setting Defaults\n");
		user_database.temp_enabled = 0;
		user_database.temp_action = 1;
		return;
	}
/* Added by Sreelu - newaddition */

	ptr_to_udb_hdr = (USER_DATABASE *) current_ptr;

	while(ptr_to_udb_hdr->magic_id != INVALID_MAGIC_ID)
	{
		current_ptr += sizeof(USER_DATABASE);
		switch (ptr_to_udb_hdr->magic_id)
		{
			case IP_DEFAULT_MAGIC_ID	:
				  ip_default_initialize();	
				  break;

			case APP_DEFAULT_MAGIC_ID	:
				  app_default_initialize();	
				  break;

			case GROUP_PROFILE_MAGIC_ID :
				  group_initialize();	
				  break;

			case USER_PROFILE_MAGIC_ID :
				  user_initialize();	
				  break;

			case FILTER_PROFILE_MAGIC_ID :
				  user_database.num_of_domains = *(USHORT *)current_ptr;
/*sri 	printf("PPP : Number of Domain Names : %d\n", user_database.num_of_domains); */
				  current_ptr += sizeof(USHORT);	 
				  filter_initialize();	
				  break;

			case DEFAULT_FILTER_MAGIC_ID :   
				  break;

			default :	
				  break;
		}
		ptr_to_udb_hdr = (USER_DATABASE *) current_ptr;
	}
}

void free_user_data_profile()
{
	/* User Data Profile is freeeeeeeeeed */
	USER_DATA *temp_ptr;

	if(user_database.sptr_temp_user_data == NULL)
		return;

	while (user_database.sptr_temp_user_data != NULL)
	{
		temp_ptr = 	user_database.sptr_temp_user_data;
		user_database.sptr_temp_user_data = user_database.sptr_temp_user_data->next_node;
		free(temp_ptr->data);
		free(temp_ptr);
	}
}

void free_group_data_profile()
{
	/* Group Data Profile is freeeeeeeeeed */
	GROUP_DATA *temp_ptr;

	if(user_database.sptr_temp_group_data == NULL)
		return;

	while (user_database.sptr_temp_group_data != NULL)
	{
		temp_ptr = 	user_database.sptr_temp_group_data;
		user_database.sptr_temp_group_data = user_database.sptr_temp_group_data->next_node;
		free(temp_ptr->data);
		free(temp_ptr);
	}
}

void free_ip_default_data_profile()
{
	/* IP Default Data Profile is freeeeeeeeeed */
	IP_DEFAULT_DATA *temp_ptr;

	if(user_database.sptr_temp_ip_default_data == NULL)
		return;

	while (user_database.sptr_temp_ip_default_data != NULL)
	{
		temp_ptr = 	user_database.sptr_temp_ip_default_data;
		user_database.sptr_temp_ip_default_data = user_database.sptr_temp_ip_default_data->next_node;
		free(temp_ptr->data);
		free(temp_ptr);
	}
}

void free_app_default_data_profile()
{
	/* APP Default Data Profile is freeeeeeeeeed */
	APP_DEFAULT_DATA *temp_ptr;

	if(user_database.sptr_temp_app_default_data == NULL)
		return;

	while (user_database.sptr_temp_app_default_data != NULL)
	{
		temp_ptr = 	user_database.sptr_temp_app_default_data;
		user_database.sptr_temp_app_default_data = user_database.sptr_temp_app_default_data->next_node;
		free(temp_ptr->data);
		free(temp_ptr);
	}
}

void free_filter_data_profile()
{
	FILTER_DATA *temp_ptr;

	if(user_database.sptr_temp_filter_data == NULL)
		return;
	
	while(user_database.sptr_temp_filter_data != NULL)
	{
		temp_ptr = user_database.sptr_temp_filter_data;
		user_database.sptr_temp_filter_data = user_database.sptr_temp_filter_data->next_node;
		free(temp_ptr->header);
		free(temp_ptr);
	}
}

enum UDB_MODIFY udb_user_deinitialize ()
{
	/* Ptrs Should be Freeeeeeeeeeeed here */
	printf("UDB User Deinitialize\n");
	free_user_data_profile();	
	free_group_data_profile();	
	free_ip_default_data_profile();	
	free_app_default_data_profile();	
	free_filter_data_profile();
}

void	update_ipdefaults_in_uncompressed_udb()
{
	USER_DATABASE *ptr_to_hdr;
	IP_DEFAULT_DATA *ptr_data;

	ptr_to_hdr = (USER_DATABASE *)	calloc (sizeof(USER_DATABASE), 1);
	ptr_to_hdr->magic_id = IP_DEFAULT_MAGIC_ID;
	ptr_to_hdr->number_of_items = user_database.num_of_ipdefaults;
	ptr_to_hdr->length_of_items = user_database.num_of_ipdefaults * sizeof(TEMP_IP_DEFAULT);

	memcpy(current_uncompressed_ptr, ptr_to_hdr, sizeof(USER_DATABASE));
	current_uncompressed_ptr += sizeof(USER_DATABASE);
	free (ptr_to_hdr);
	memcpy(current_uncompressed_ptr, &user_database.temp_enabled, sizeof(USHORT));
	current_uncompressed_ptr += sizeof(USHORT);
	memcpy(current_uncompressed_ptr, &user_database.temp_action, sizeof(USHORT));
	current_uncompressed_ptr += sizeof(USHORT);

/* Writing IP_DEFAULT Datas */
	ptr_data = user_database.sptr_temp_ip_default_data;
	while(ptr_data != NULL)
	{
		memcpy(current_uncompressed_ptr, ptr_data->data, sizeof(TEMP_IP_DEFAULT));
		ptr_data = ptr_data->next_node;
		current_uncompressed_ptr += sizeof(TEMP_IP_DEFAULT);
	}	 
}
	
void	update_appdefaults_in_uncompressed_udb()
{
	USER_DATABASE *ptr_to_hdr;
	APP_DEFAULT_DATA *ptr_data;

	ptr_to_hdr = (USER_DATABASE *)	calloc (sizeof(USER_DATABASE), 1);
	ptr_to_hdr->magic_id = APP_DEFAULT_MAGIC_ID;
	ptr_to_hdr->number_of_items = user_database.num_of_appdefaults;
	ptr_to_hdr->length_of_items = user_database.num_of_appdefaults * sizeof(TEMP_APP_DEFAULT);

	memcpy(current_uncompressed_ptr, ptr_to_hdr, sizeof(USER_DATABASE));
	current_uncompressed_ptr += sizeof(USER_DATABASE);
	free (ptr_to_hdr);

/* Writing APP_DEFAULT Datas */
	ptr_data = user_database.sptr_temp_app_default_data;
	while(ptr_data != NULL)
	{
		memcpy(current_uncompressed_ptr, ptr_data->data, sizeof(TEMP_APP_DEFAULT));
		ptr_data = ptr_data->next_node;
		current_uncompressed_ptr += sizeof(TEMP_APP_DEFAULT);
	}
}

void	update_groups_in_uncompressed_udb()
{
	USER_DATABASE *ptr_to_hdr;
	GROUP_DATA *ptr_data;

	ptr_to_hdr = (USER_DATABASE *)	calloc (sizeof(USER_DATABASE), 1);
	ptr_to_hdr->magic_id = GROUP_PROFILE_MAGIC_ID;
	ptr_to_hdr->number_of_items = user_database.num_of_groups;
	ptr_to_hdr->length_of_items = user_database.num_of_groups * sizeof(TEMP_GROUP_PROFILE);

	memcpy(current_uncompressed_ptr, ptr_to_hdr, sizeof(USER_DATABASE));
	current_uncompressed_ptr += sizeof(USER_DATABASE);
	free (ptr_to_hdr);

/* Writing GROUP Datas */
	ptr_data = user_database.sptr_temp_group_data;
	while(ptr_data != NULL)
	{
		memcpy(current_uncompressed_ptr, ptr_data->data, sizeof(TEMP_GROUP_PROFILE));
		ptr_data = ptr_data->next_node;
		current_uncompressed_ptr += sizeof(TEMP_GROUP_PROFILE);
	}
}

void	update_users_in_uncompressed_udb()
{
	USER_DATABASE *ptr_to_hdr;
	USER_DATA *ptr_data;

	ptr_to_hdr = (USER_DATABASE *)	calloc (sizeof(USER_DATABASE), 1);
	ptr_to_hdr->magic_id = USER_PROFILE_MAGIC_ID;
	ptr_to_hdr->number_of_items = user_database.num_of_users;
	ptr_to_hdr->length_of_items = user_database.num_of_users * sizeof(TEMP_USER_PROFILE);

	memcpy(current_uncompressed_ptr, ptr_to_hdr, sizeof(USER_DATABASE));
	current_uncompressed_ptr += sizeof(USER_DATABASE);
	free (ptr_to_hdr);

/* Writing USER Datas */
	ptr_data = user_database.sptr_temp_user_data;
	while(ptr_data != NULL)
	{
		memcpy(current_uncompressed_ptr, ptr_data->data, sizeof(TEMP_USER_PROFILE));
		ptr_data = ptr_data->next_node;
		current_uncompressed_ptr += sizeof(TEMP_USER_PROFILE);
	}
}

void	update_filters_in_uncompressed_udb()
{
	USER_DATABASE *ptr_to_hdr;
	FILTER_DATA *ptr_data;

	ptr_to_hdr = (USER_DATABASE *)	calloc (sizeof(USER_DATABASE), 1);
	ptr_to_hdr->magic_id = FILTER_PROFILE_MAGIC_ID;
	ptr_to_hdr->number_of_items = user_database.num_of_filters;
	ptr_to_hdr->length_of_items = user_database.num_of_filters * (sizeof(TEMP_FILTER_HEADER) + 260);

	memcpy(current_uncompressed_ptr, ptr_to_hdr, sizeof(USER_DATABASE));
	current_uncompressed_ptr += sizeof(USER_DATABASE);
	free (ptr_to_hdr);

	memcpy(current_uncompressed_ptr, &user_database.num_of_domains, sizeof(USHORT));
	current_uncompressed_ptr += sizeof(USHORT);

/* Writing USER Datas */
	ptr_data = user_database.sptr_temp_filter_data;
	while(ptr_data != NULL)
	{
		memcpy(current_uncompressed_ptr, ptr_data->header, sizeof(TEMP_FILTER_HEADER));
		current_uncompressed_ptr += sizeof(TEMP_FILTER_HEADER);
		switch(ptr_data->header->type)
		{
			case APPLICATION_FILTER :
				memcpy(current_uncompressed_ptr, &ptr_data->port, sizeof(USHORT));
				current_uncompressed_ptr += sizeof(USHORT);
				memcpy(current_uncompressed_ptr, &ptr_data->protocol, sizeof(USHORT));
				current_uncompressed_ptr += sizeof(USHORT);
				break;
			case IP_ADDRESS_FILTER :
			case DOMAIN_NAME_FILTER	:
				memcpy(current_uncompressed_ptr, &ptr_data->address[0], ptr_data->header->length+1);
				current_uncompressed_ptr += (ptr_data->header->length+1);
				break;
		}
		ptr_data = ptr_data->next_node;
	}
}

void update_invalid_magic_id_in_uncompressed_udb()
{
	USER_DATABASE *ptr_to_hdr;

	ptr_to_hdr = (USER_DATABASE *) calloc (sizeof(USER_DATABASE), 1);
	ptr_to_hdr->magic_id = INVALID_MAGIC_ID;
	ptr_to_hdr->number_of_items = 0;
	ptr_to_hdr->length_of_items = 0;

	memcpy(current_uncompressed_ptr, ptr_to_hdr, sizeof(USER_DATABASE));
	current_uncompressed_ptr += sizeof(USER_DATABASE);
	free (ptr_to_hdr);
}

enum UDB_MODIFY udb_user_update_after_changes ()
{
	/* Create a temp ptr to store complete UDB ...... */
	USHORT udb_crc = (USHORT)	0xFFFF;
	ULONG udb_length = 0 , size, original_size;
	UDB_HEADER *udb_hdr ;
	BYTE *next_block_to_write, *temp_ptr;
	BYTE   two_zeroes[] = {0, 0} ;
	
	udb_length = 0 ;
	udb_crc = 0xFFFF ;
	fl_ptr_user_udb = (BYTE *)sptr_user_database_hdr->down_load_address ;

	size = sizeof(ULONG) + (sizeof(USER_DATABASE) * 6) +
				(sizeof(TEMP_IP_DEFAULT) * user_database.num_of_ipdefaults) +
				(sizeof(TEMP_APP_DEFAULT) * user_database.num_of_appdefaults) +
				(sizeof(TEMP_USER_PROFILE) * user_database.num_of_users)	+
				(sizeof(TEMP_GROUP_PROFILE) * user_database.num_of_groups) +
				sizeof(USHORT) + ((sizeof(TEMP_FILTER_HEADER) + 260) * user_database.num_of_filters);
/* Filter Size + No of Domain Names should be added 
	printf("Total Uncompressed byte size : %d\n", size); */
	
	uncompressed_udb = (BYTE *) calloc (size, 1);
	if(uncompressed_udb == NULL)
	{
		printf ("No memory to uncompress User database\n") ;
		udb_user_deinitialize();
		return (UPDATE_NO_MEMORY);
	}

	current_uncompressed_ptr = uncompressed_udb;
	update_ipdefaults_in_uncompressed_udb();
	update_appdefaults_in_uncompressed_udb();
	update_groups_in_uncompressed_udb();
	update_users_in_uncompressed_udb();
	update_filters_in_uncompressed_udb();
	update_invalid_magic_id_in_uncompressed_udb();
/*....Add for Filters latter */
	
	original_size = current_uncompressed_ptr - uncompressed_udb;
	udb_length = original_size;
	udb_crc = update_crc (udb_crc, uncompressed_udb, (ULONG)original_size) ;
	next_block_to_write = uncompressed_udb ;
	while(original_size > 512)
	{
		temp_ptr = malloc (512);
		if (temp_ptr == NULL)
		{
			printf ("No memory for flash write\n\r") ;
			break ;
		}
		memcpy (temp_ptr, next_block_to_write, 512);

		if (schedule_flash_write (temp_ptr, fl_ptr_user_udb, 512, NULL, temp_ptr) != 0)
			printf ("Schedule flash write failed\n") ;
		original_size -= 512;
		next_block_to_write += 512;
		fl_ptr_user_udb += 512;
	}

	temp_ptr = malloc (512) ;
	if (temp_ptr == NULL)
	{
		printf ("No memory for flash write\n\r") ;
		return (UPDATE_NO_MEMORY) ;
	}
/*	memcpy (temp_ptr, fl_ptr_user_udb, 512) ;*/
	memcpy (temp_ptr, next_block_to_write, original_size) ;

	if (schedule_flash_write (temp_ptr, fl_ptr_user_udb, 512, NULL, temp_ptr) != 0)
		printf ("Schedule flash write failed\n") ;

	free (uncompressed_udb) ;
	original_size = 0 ;
	
	temp_ptr = calloc (512, 1) ;
	if (temp_ptr == NULL)
	{
		printf ("No memory for flash write\n\r") ;
		return (UPDATE_NO_MEMORY) ;
	}
	memcpy (temp_ptr, (BYTE *)sptr_user_database_hdr, 512) ;
	udb_hdr = (UDB_HEADER *)temp_ptr ;

	udb_crc = update_crc (udb_crc, &two_zeroes[0], 2) ;
	udb_hdr->length = udb_length ;
	udb_hdr->no_of_records = 0;
	udb_hdr->crc = udb_crc ;
/*	udb_hdr->down_load_address = UDB_ADDRESS ; 
	printf("UDB ADDRESS : %x", sptr_user_database_hdr->down_load_address); */
	strcpy (&udb_hdr->date_stamp[0], "NOT AVAILABLE") ;

#if 0
	printf ("UDB: Calculated CRC = 0x%x\r\n", udb_crc);
	printf ("Length written to Flash is %d\n",udb_length);
#endif

/*sri printf("UDB Header Address : %x\n", UDB_HEADER_ADDRESS); */
	c_write_to_flash (temp_ptr, (BYTE *)UDB_HEADER_ADDRESS, 512) ;
	free (temp_ptr) ;
	return (UPDATE_SUCCESSFUL) ;
}

void *get_ptr_to_previous_ten_entries(void *ptr_to_list)
{
	int i;
	void *ptr_to_prevs_ten_entry;

	ptr_to_prevs_ten_entry = ptr_to_list;
	for(i = 0; i < 10; i++)
	{
		ptr_to_prevs_ten_entry = (void *)ptr_to_prevs_ten_entry->prev_node;
		if(ptr_to_prevs_ten_entry == NULL)
			return NULL;
	}
	return ptr_to_prevs_ten_entry;
}

void *get_the_last_node(void *ptr_to_list)
{
	while(ptr_to_list != NULL)
	{
		if(!ptr_to_list->next_node)
			break;
		ptr_to_list = (void *)ptr_to_list->next_node;
	}
	return ptr_to_list;
}

/* ------------------------------displays for UDB---------------------------*/
void display_groups_from_list()
{
	GROUP_DATA *sptr_next_group;

	printf ("          Displaying Group Data		              \n") ;

	for (sptr_next_group = user_database.sptr_temp_group_data; sptr_next_group != NULL; 
		  sptr_next_group = sptr_next_group->next_node)
	{
		printf ("\n [%s]  	[%d]", sptr_next_group->data->group_name, sptr_next_group->data->group_no);
	}
	return;		
}

void display_users_from_list()
{
	USER_DATA *sptr_next_user = user_database.sptr_temp_user_data;

	printf ("          Displaying User  Data		             \n ") ;
	if(user_database.sptr_temp_user_data == NULL)
	{
		printf("\n user Data ptr is NULL") ;
		return;			
	}

	while(1)
	{
		printf ("\n [%s] ", sptr_next_user->data->user_name);

		if(sptr_next_user->next_node == NULL)
			break;
		sptr_next_user = sptr_next_user->next_node;
	}
}

void display_filters_from_list()
{
	FILTER_DATA *sptr_next_filter = user_database.sptr_temp_filter_data;

	printf ("          Displaying Filter Data		              \n") ;
	if(user_database.sptr_temp_filter_data == NULL)
	{
		printf("\n Filter Data ptr is NULL") ;
		return;			
	}

	while(1)
	{
		if(sptr_next_filter->header->type == APPLICATION_FILTER)
			printf ("\n [%d]:[%d] - [%x]", sptr_next_filter->port, sptr_next_filter->protocol, sptr_next_filter->header->mask);
		else
			printf ("\n [%s] - [%x]", sptr_next_filter->address, sptr_next_filter->header->mask);
	
		if(sptr_next_filter->next_node == NULL)
			break;
		sptr_next_filter = sptr_next_filter->next_node;
	}
}

void display_ip_defaults_from_list()
{
	IP_DEFAULT_DATA *sptr_next_ip_default = user_database.sptr_temp_ip_default_data;

	printf ("          Displaying IP Default		              \n") ;
	if(user_database.sptr_temp_ip_default_data == NULL)
	{
		printf("\n ip_default Data ptr is NULL") ;
		return;			
	}

	while(1)
	{
		printf ("\n [%s] : [%s]", sptr_next_ip_default->data->lower_address, 
					sptr_next_ip_default->data->upper_address);
		if(sptr_next_ip_default->next_node == NULL)
			break;
		sptr_next_ip_default = sptr_next_ip_default->next_node;
	}
}

void display_app_defaults_from_list()
{
	APP_DEFAULT_DATA *sptr_next_app_default = user_database.sptr_temp_app_default_data;

	printf ("          Displaying APP Default		             \n ") ;
	if(user_database.sptr_temp_app_default_data == NULL)
	{
		printf("\n app_default Data ptr is NULL") ;
		return;
	}

	while(1)
	{
		printf ("\n [%d]  : [%d]", sptr_next_app_default->data->lower_port, 
					sptr_next_app_default->data->upper_port);
		if(sptr_next_app_default->next_node == NULL)
			break;
		sptr_next_app_default = sptr_next_app_default->next_node;
	}
}

#if 0
/*---------------------------Display From end of the Node-------------------*/
void display_groups_from_last_node()
{
	GROUP_DATA *sptr_prev_group;

	printf ("          Displaying Group Data		              \n") ;
	if(user_database.sptr_temp_group_data == NULL)
	{
		printf("\n Group Data ptr is NULL") ;
		return;			
	}

	sptr_prev_group = (GROUP_DATA *) get_the_last_node((void *)user_database.sptr_temp_group_data);
	
	while(1)
	{
		printf ("\n [%s] ", sptr_prev_group->data->group_name);
		if(sptr_prev_group->prev_node == NULL)
			break;
		sptr_prev_group = sptr_prev_group->prev_node;
	}
	return;		
}

void display_users_from_last_node()
{
	USER_DATA *sptr_prev_user;

	printf ("          Displaying User Data		              \n") ;
	if(user_database.sptr_temp_user_data == NULL)
	{
		printf("\n User Data ptr is NULL") ;
		return;			
	}

	sptr_prev_user = (USER_DATA *) get_the_last_node((void *)user_database.sptr_temp_user_data);
	
	while(1)
	{
		printf ("\n [%s] ", sptr_prev_user->data->user_name);
		if(sptr_prev_user->prev_node == NULL)
			break;
		sptr_prev_user = sptr_prev_user->prev_node;
	}
	return;		
}

void display_groups_from_list_in_ten()
{
	GROUP_DATA *sptr_next_group, *prev_ten;
	int i;

	printf ("          Displaying Group Data		              \n") ;
	
	prev_ten = (GROUP_DATA *) get_the_last_node((void *)user_database.sptr_temp_group_data);
	printf("Last Node : %s\n", prev_ten->data->group_name);
	while(1)
	{
		prev_ten = (GROUP_DATA *)get_ptr_to_previous_ten_entries((void *)prev_ten);
		printf("Prev Ten Node : %s\n", prev_ten->data->group_name);
		if(prev_ten == NULL)
			break;

		sptr_next_group = prev_ten;
		for(i=0; i<4; i++)
		{
			printf ("\n [%s] ", sptr_next_group->data->group_name);
			sptr_next_group = sptr_next_group->next_node;
		}
	}
	return;		
}

void generate_new_string(int i, char *istring, char *ostring)
{
	char buf[10];

	i++;
	my_itoa(i, buf, 10);
	strcpy(ostring, istring);
	strcat(ostring, buf);
}

void generate_new_group()
{
	TEMP_GROUP_PROFILE data;
	char new_str[20];
	
	if(!user_database.sptr_temp_group_data)	
		return;

	memcpy(&data, user_database.sptr_temp_group_data->data, sizeof(TEMP_GROUP_PROFILE));
	generate_new_string(user_database.num_of_groups, user_database.sptr_temp_group_data->data->group_name, &new_str[0]);
	strcpy(data.group_name, new_str);
	printf("New group is : %s \n", data.group_name);
	user_database_insert(GROUP_PROFILE_MAGIC_ID, (void *)&data);
}

void generate_new_user()
{
	TEMP_USER_PROFILE data;
	char new_str[20];

	if(!user_database.sptr_temp_user_data)	
		return;

	memcpy(&data, user_database.sptr_temp_user_data->data, sizeof(TEMP_USER_PROFILE));
	generate_new_string(user_database.num_of_users, user_database.sptr_temp_user_data->data->user_name, &new_str[0]);
	strcpy(data.user_name, new_str);
	printf("New user is : %s \n", data.user_name);
	user_database_insert(USER_PROFILE_MAGIC_ID, (void *)&data);
}
#endif

