#include <kstart.h>
#include <stdlib.h>
#include <string.h>

#include "scrtfuns.h"
#include "scrtdefs.h"
#include "scrttype.h"
#include "scrtenum.h"
#include "scrtargs.h"


/*----------------------------------------------------------------------------
	Name		:	UnaryMinusFun											
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles unary minus operator.							
----------------------------------------------------------------------------*/
int UnaryMinusFun (BYTE PortNum)
{
	return (UnaryOperatorFun (Q_UNARY_MINUS, PortNum)) ;
}

/*----------------------------------------------------------------------------
	Name		:	PlusFun													
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles plus operator.									
----------------------------------------------------------------------------*/
int PlusFun (BYTE PortNum)
{
	return (ArithmeticFun (Q_PLUS, PortNum)) ;
}	

		

/*----------------------------------------------------------------------------
	Name		:	MinusFun													
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles minus operator.									
----------------------------------------------------------------------------*/
int MinusFun (BYTE PortNum)		
{
	return (ArithmeticFun (Q_MINUS, PortNum)) ;
}	


/*----------------------------------------------------------------------------
	Name		:	TimesFun													
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles multiplication operator.						
----------------------------------------------------------------------------*/
int TimesFun (BYTE PortNum)		
{
	return (ArithmeticFun (Q_TIMES, PortNum)) ;
}	


/*----------------------------------------------------------------------------
	Name		:	DivFun													
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles division operator.								
----------------------------------------------------------------------------*/
int DivFun (BYTE PortNum)			
{
	return (ArithmeticFun (Q_DIV, PortNum)) ;
}	


/*----------------------------------------------------------------------------
	Name		:	LessOrEqFun												
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles <= operator.									
----------------------------------------------------------------------------*/
int LessOrEqFun (BYTE PortNum)	
{
	return (RelationalFun (Q_LESS_OR_EQUAL, PortNum)) ;
}	


/*----------------------------------------------------------------------------
	Name		:	GreaterOrEqFun											
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles >= operator.									
----------------------------------------------------------------------------*/
int GreaterOrEqFun (BYTE PortNum)
{
	return (RelationalFun (Q_GREATER_OR_EQUAL, PortNum)) ;
}	


/*----------------------------------------------------------------------------
	Name		:	IsEqFun													
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles == operator.									
----------------------------------------------------------------------------*/
int IsEqFun (BYTE PortNum)			
{
	return (RelationalFun (Q_IS_EQUAL, PortNum)) ;
}	


/*----------------------------------------------------------------------------
	Name		:	IsNotEqFun												
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles != operator.									
----------------------------------------------------------------------------*/
int IsNotEqFun (BYTE PortNum)		
{
	return (RelationalFun (Q_IS_NOT_EQUAL, PortNum)) ;
}	


/*----------------------------------------------------------------------------
	Name		:	LessFun													
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles < operator.										
----------------------------------------------------------------------------*/
int LessFun (BYTE PortNum)			
{
	return (RelationalFun (Q_LESS, PortNum)) ;
}	


/*----------------------------------------------------------------------------
	Name		:	GreaterFun												
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles > operator.										
----------------------------------------------------------------------------*/
int GreaterFun (BYTE PortNum)		
{
	return (RelationalFun (Q_GREATER, PortNum)) ;
}	


/*----------------------------------------------------------------------------
	Name		:	NotFun													
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles ~ operator.										
----------------------------------------------------------------------------*/
int NotFun (BYTE PortNum)			
{
	return (UnaryOperatorFun (Q_NOT, PortNum)) ;
}	


/*----------------------------------------------------------------------------
	Name		:	AndFun													
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles && operator.									
----------------------------------------------------------------------------*/
int AndFun (BYTE PortNum)			
{
	return (LogicalFun (Q_AND, PortNum)) ;
}	

int OrFun (BYTE PortNum)			
{
	return (LogicalFun (Q_OR, PortNum)) ;
}	

/*----------------------------------------------------------------------------
	Name		:	MovFun													
	Input		:	None													
	Output 	:	Zero													
	Synopsis	:	Handles mov operator.									
----------------------------------------------------------------------------*/
int MovFun (BYTE PortNum)			
{
	ArgType *a_ptr1 ;
	ArgType *a_ptr2 ;
	ValType *v_ptr ;
	void *arg_addr ;	/*	To remember destination argument address.	*/
	BYTE *ptr ;

	ScriptPortClass[PortNum].CodeIdx ++ ;
	a_ptr1 = (ArgType *) &(ScriptPortClass[PortNum].CodeBuffer[ScriptPortClass[PortNum].CodeAddr + 1]) ;
	a_ptr2 = a_ptr1 + 1 ;

	/*	If a string was moved into the condition variable in a previous	*/
	/*	switch statement, and it is not yet released, free the same.	*/
	/*	This is to prevent memory leaks arising out of the lost reference	*/
	/*	to the value of the condition variable upon subsequent assignment.	*/

	/*	Mod. (24th Sep 94): If a string was moved into the return variable
		in a previous mov statement, and is not yet released, release the 
		same.  As of now there will not be any problems but this will prevent
		bugs being introduced in future.
	*/

	if ((a_ptr2->StorageClass == VAR_CONDITION || 
				a_ptr2->StorageClass == VAR_RETURN) &&
							a_ptr2->Val.SymPtr->SymClass == E_STRING &&
								a_ptr2->Val.SymPtr->Val.StrValue) 
	{
		free (a_ptr2->Val.SymPtr->Val.StrValue) ;
		a_ptr2->Val.SymPtr->Val.StrValue = 0 ;
	}

	v_ptr = GetArgValueAddr (a_ptr2, PortNum) ;

	if (!v_ptr)
	{
		goto error_exit;
	}
	arg_addr = ArgAddr;	/*	Remember argument address.	*/
	switch (a_ptr1->ArgType) 
	{
		case E_INT_CONST :
		case E_INTEGER :
			switch (a_ptr2->ArgType) 
			{
				case E_INT_CONST :
				case E_INTEGER :
					v_ptr->IntValue = GetIntArg (a_ptr1, PortNum) ;
					break ;

				default :
					goto error_exit ;
			}
			break ;

		case E_STRING_CONST :
		case E_STRING :
			switch (a_ptr2->ArgType) 
			{
				case E_STRING_CONST :
				case E_STRING :
					ptr = GetStrArg (a_ptr1, PortNum) ;

					/*	Release previous string, if any.	*/
					/*	This does not apply to temporaries because they	*/
					/*	are killed after usage and StrValue might	*/
					/*	contain garbage becuase their type keeps changing.	*/

					/*	Bug Fixed (24th Sep 94): also check for the symbol
						class in the SymTable info. before freeing.	*/

					if (a_ptr2->StorageClass != VAR_TEMPORARY &&
							((SymType *) arg_addr)->SymClass == E_STRING &&
											v_ptr->StrValue) 
					{
						free (v_ptr->StrValue) ;
						v_ptr->StrValue = 0 ;
					}

					/*	Allocate memory for new string and copy it.	*/
					v_ptr->StrValue = malloc (strlen (ptr) + 1) ;

					if (v_ptr->StrValue)
						strcpy (v_ptr->StrValue, ptr) ;
					else
					{
						script_printf (ALARM_SCRIPT_PRINTF,
								"Memory exhausted, port %d\n\r", PortNum) ;
						goto error_exit ;
					}
					/*	If the first argument is the return value variable	*/
					/*	and is of string type no need to keep that copy;	*/
					/*	so release it.	*/
					if (a_ptr1->StorageClass == VAR_RETURN &&
											a_ptr1->ArgType == E_STRING) 
					{
						v_ptr = GetArgValueAddr (a_ptr1, PortNum) ;
						if (!v_ptr)
						{
							goto error_exit ;
						}
						free (v_ptr->StrValue) ;
						v_ptr->StrValue = 0 ;
					}
					break ;

				default :
					goto error_exit;
			}
			break;

		default :
			goto error_exit;
	}
	#if 1
	/*	If the target is RET_VAR or temporary, patch the type.	*/
	switch (a_ptr2->StorageClass) 
	{
		case VAR_CONDITION :
		case VAR_RETURN :
			((SymType *) arg_addr)->SymClass = a_ptr2->ArgType ;
			break ;

		case VAR_TEMPORARY :
			((ArgType *) arg_addr)->ArgType = a_ptr2->ArgType ;
			break ;
	}
	#endif
	return 0 ;
				 
error_exit :
	return -2 ;
}	


/*----------------------------------------------------------------------------
	Name		:	LabelFun													
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles label operator.									
----------------------------------------------------------------------------*/
int LabelFun (BYTE PortNum)		
{
	ScriptPortClass[PortNum].CodeIdx ++ ;
	return 0 ;
}	

/*----------------------------------------------------------------------------
	Name		:	JmpFun													
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles jmp operator.									
----------------------------------------------------------------------------*/
int JmpFun (BYTE PortNum)			
{
 	ScriptPortClass[PortNum].CodeIdx += 
		*(WORD *) (ScriptPortClass[PortNum].CodeBuffer + ScriptPortClass[PortNum].CodeAddr + 1) ;
	return 0 ;
}	


/*----------------------------------------------------------------------------
	Name		:	JzFun													
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles jz operator.									
--------------------------------------------------------------------------*/
int JzFun (BYTE PortNum)			
{
	return (JzJnzFun (Q_JZ, PortNum)) ;
}	


/*----------------------------------------------------------------------------
	Name		:	JnzFun													
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles jnz operator.									
----------------------------------------------------------------------------*/
int JnzFun (BYTE PortNum)			
{
	return (JzJnzFun (Q_JNZ, PortNum)) ;
}	


/*----------------------------------------------------------------------------
	Name		:	JeqFun													
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles jeq operator.									
----------------------------------------------------------------------------*/
int JeqFun (BYTE PortNum)			
{
	return (JeqJneqFun (Q_JEQ, PortNum)) ;
}	


/*----------------------------------------------------------------------------
	Name		:	JneqFun													
	Input		:	None													
	Output		:	Zero													
	Synopsis	:	Handles jneq operator.									
----------------------------------------------------------------------------*/
int JneqFun (BYTE PortNum)			
{
	return (JeqJneqFun (Q_JNEQ, PortNum)) ;
}	


/*----------------------------------------------------------------------------
	Name		:	PushFun													
	Input		:	None													
	Output		:	Zero													
	Synopsis	:	Handles push operator.									
----------------------------------------------------------------------------*/
int PushFun (BYTE PortNum)			
{
	ArgType *a_ptr1 ;

	if ((ScriptPortClass[PortNum].StackTop + 1) >= ScriptPortClass[PortNum].StackUpperBoundary)	/*	Enough space on stack ?	*/
		goto error_exit ;	 /*	No more stack space.	*/

	a_ptr1 = (ArgType  *) &(ScriptPortClass[PortNum].CodeBuffer[ScriptPortClass[PortNum].CodeAddr + 1]) ;

	ScriptPortClass[PortNum].StackTop->ArgType = a_ptr1->ArgType ;
	ScriptPortClass[PortNum].StackTop->StorageClass = VAR_PARAM ;
	switch (a_ptr1->ArgType) 
	{
		case E_INT_CONST :
			ScriptPortClass[PortNum].StackTop->IntVal =
												GetIntArg (a_ptr1, PortNum) ;
			break ;

		case E_STRING_CONST :
			ScriptPortClass[PortNum].StackTop->StrVal = 
												GetStrArg (a_ptr1, PortNum) ;
			break ;

		case E_INTEGER :
		case E_STRING :
			ScriptPortClass[PortNum].StackTop->ArgVal =
												GetArgValueAddr (a_ptr1, PortNum) ;
			break ;

		default :
			goto error_exit;
	}

	ScriptPortClass[PortNum].StackTop++;
	ScriptPortClass[PortNum].CodeIdx++;
	return 0 ;

error_exit :
	return -2 ;
}	

/*----------------------------------------------------------------------------
	Name		:	CallFun													
	Input		:	None													
	Output 	:	Zero													
	Synopsis	:	Handles call operator.									
----------------------------------------------------------------------------*/
int CallFun (BYTE PortNum)			
{
	ArgType *a_ptr1 ;
	WORD address ;
	WORD arg_count ;
	int result ;

	result = 0 ;
	a_ptr1 = (ArgType *) &(ScriptPortClass[PortNum].CodeBuffer[ScriptPortClass[PortNum].CodeAddr + 1]) ;

	/*	If the function is built-in call the corresponding handler,	*/
	/*	clean up arguments on the stack and proceed to the next	*/
	/*	instruction; otherwise, push the return address and	*/
	/*	transfer control to the beginning of the procedure.	*/
	if (a_ptr1->StorageClass & BUILT_IN_FUNCTION) 
	{
		if (a_ptr1->Handler)
			result = (*a_ptr1->Handler) (PortNum) ;
		/*	Clean up arguments on the stack.	*/
		if (result != DO_NOT_REMOVE_PARAMETERS) 	/* In this case don't remove parameters*/
		{
			arg_count = (WORD) ((ScriptPortClass[PortNum].StackTop - 1)->IntVal) ;
			ScriptPortClass[PortNum].StackTop -= (arg_count + 1);
			ScriptPortClass[PortNum].CodeIdx++;	/*	Mov on to next address.	*/
		}
	}
	else
	{
			/*	Enough space on stack ?	*/
		if ((ScriptPortClass[PortNum].StackTop + 1) >= ScriptPortClass[PortNum].StackUpperBoundary)
			goto error_exit;	/*	No more stack space.	*/
		/*	User defined function, save return address and jump	*/
		/*	to the routine beginning.	*/
		address = (WORD) a_ptr1->IntVal;
		ScriptPortClass[PortNum].StackTop->ArgType = Q_RETURN;	/*	Set arg type to return address.	*/
		ScriptPortClass[PortNum].StackTop->IntVal = (ScriptPortClass[PortNum].CodeIdx + 1);	/*	Push the return address.	*/
		ScriptPortClass[PortNum].StackTop++;
		ScriptPortClass[PortNum].CodeIdx = address;
	}
	return result ;

error_exit :
	return -2 ;
}	


/*----------------------------------------------------------------------------
	Name		:	RetFun													
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles ret operator.									
----------------------------------------------------------------------------*/
int RetFun (BYTE PortNum)			
{
	WORD arg_count ;
	WORD ret_address ;

	PurgeLocalSymbols (PortNum) ;
		/*	No. of parameters on stack.	*/
	arg_count = (WORD) ((ScriptPortClass[PortNum].FramePtr - 4)->IntVal) ;
	ret_address = (WORD) ((ScriptPortClass[PortNum].FramePtr - 3)->IntVal) ;
	ScriptPortClass[PortNum].StackTop = ScriptPortClass[PortNum].FramePtr ;
	ScriptPortClass[PortNum].StackTop -- ;		/*	Restore old temporaries pointer.	*/
	ScriptPortClass[PortNum].TempsPtr = (ArgType *) ScriptPortClass[PortNum].StackTop->ArgVal ;
	ScriptPortClass[PortNum].StackTop -- ;		/*	Restore old frame pointer.	*/
	ScriptPortClass[PortNum].FramePtr = (ArgType *) ScriptPortClass[PortNum].StackTop->ArgVal ; 
	ScriptPortClass[PortNum].StackTop -= (arg_count + 2) ;	/*	Cleanup parameters on stack.	*/

	ScriptPortClass[PortNum].EntryCount -- ;

	ScriptPortClass[PortNum].CodeIdx = ret_address ;
	if (!ScriptPortClass[PortNum].EntryCount)
		return -1 ;	/*	To exit script.	*/
	else
		return 0 ;
}	


/*----------------------------------------------------------------------------
	Name		:	LineNoFun												
	Input		:	None													
	Output		:	Zero													
	Synopsis	:	Handles lineno operator.								
----------------------------------------------------------------------------*/
int LineNoFun (BYTE PortNum)		
{
	ScriptPortClass[PortNum].CurLineNo = (WORD) ScriptPortClass[PortNum].CodeBuffer[ScriptPortClass[PortNum].CodeAddr + 1];
	ScriptPortClass[PortNum].CodeIdx++;
	return 0 ;
}	

/*----------------------------------------------------------------------------
	Name		:	ProcFun													
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles proc operator.									
----------------------------------------------------------------------------*/
int ProcFun (BYTE PortNum)			
{
	if ((ScriptPortClass[PortNum].StackTop + 2) >= ScriptPortClass[PortNum].StackUpperBoundary)	/*	Enough space on stack ?	*/
		goto error_exit ;	/*	No more stack space.	*/
	ScriptPortClass[PortNum].StackTop->ArgVal = ScriptPortClass[PortNum].FramePtr ;	/*	Save the previous frame ptr.	*/

	ScriptPortClass[PortNum].StackTop ++ ;				/*	Adjust stack.	*/
	ScriptPortClass[PortNum].StackTop->ArgVal = ScriptPortClass[PortNum].TempsPtr ;	/*	Save the previos temporaries ptr.	*/

	ScriptPortClass[PortNum].StackTop ++ ;				/*	Adjust stack.	*/
	ScriptPortClass[PortNum].TempsPtr = ScriptPortClass[PortNum].FramePtr = ScriptPortClass[PortNum].StackTop ;	/*	Set new frame pointer.	*/
	ScriptPortClass[PortNum].EntryCount ++ ;	/*	One more proc call.	*/
	ScriptPortClass[PortNum].Scope ++ ;	/*	Raise scope level to adjust stack for locals.	*/
	ScriptPortClass[PortNum].CodeIdx ++ ;
	return 0 ;

error_exit :
	return -2 ;
}	

/*----------------------------------------------------------------------------
	Name		:	EndProcFun												
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles endproc operator.								
----------------------------------------------------------------------------*/
int EndProcFun (BYTE PortNum)		
{
	ScriptPortClass[PortNum].CodeIdx ++ ;
	return 0 ;
}	


/*----------------------------------------------------------------------------
	Name		:	ParamFun													
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles param operator.									
----------------------------------------------------------------------------*/
int ParamFun (BYTE PortNum)		
{
	ScriptPortClass[PortNum].CodeIdx ++ ;
	return 0 ;
}	


/*----------------------------------------------------------------------------
	Name		:	DeclareFun												
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles declare operator.								
----------------------------------------------------------------------------*/
int DeclareFun (BYTE PortNum)		
{
	ArgType *a_ptr1 ;

	a_ptr1 = (ArgType *) &(ScriptPortClass[PortNum].CodeBuffer[ScriptPortClass[PortNum].CodeAddr + 1]) ;
	ScriptPortClass[PortNum].StackTop->ArgType = a_ptr1->ArgType ;	 /* Set param type.	*/
	ScriptPortClass[PortNum].StackTop->StorageClass = VAR_LOCAL ;	/*	Set storage type.	*/

	switch (a_ptr1->ArgType) 
	{
		case E_INTEGER :
			ScriptPortClass[PortNum].StackTop->IntVal = 0 ;
			break ;

		case E_STRING :
			ScriptPortClass[PortNum].StackTop->StrVal = 0 ;
			break ;
	}
	if ((ScriptPortClass[PortNum].StackTop + 1) >= ScriptPortClass[PortNum].StackUpperBoundary)
		goto error_exit ;	 /*	No more stack space.	*/

	ScriptPortClass[PortNum].StackTop ++ ;
	ScriptPortClass[PortNum].CodeIdx ++ ;
	return 0 ;

error_exit :
	return -2 ;
}	

/*----------------------------------------------------------------------------
	Name		:	TempCountFun												
	Input		:	None													
	Output 	:	Zero													
	Synopsis	:	Handles tempcount operator.								
----------------------------------------------------------------------------*/
int TempCountFun (BYTE PortNum)	
{
	WORD count ;

	ScriptPortClass[PortNum].TempsPtr = ScriptPortClass[PortNum].StackTop;	/*	Where temporaries begin on stack.	*/
	count = *((WORD *) &ScriptPortClass[PortNum].CodeBuffer[ScriptPortClass[PortNum].CodeAddr + 1]) ;

	/*	Check if enough space is available on stack.	*/
	if ((ScriptPortClass[PortNum].StackTop + count) >= ScriptPortClass[PortNum].StackUpperBoundary)
		goto error_exit;	/*	No more stack space.	*/

	/*	Now initialize all the temporaries to integer type and zero.	*/
	/*	This is used to check for temporaries assigned string vlaues	*/
	/*	in the procedures and release the same upon abnormal exit from	*/
	/*	the function like stack overrun, out of memory, etc.	*/
	while (count) 
	{
		ScriptPortClass[PortNum].StackTop->StorageClass = VAR_TEMPORARY;	/*	Set variable type.	*/
		ScriptPortClass[PortNum].StackTop->ArgType = E_INTEGER;	/*	Set to integer type.	*/
		ScriptPortClass[PortNum].StackTop->IntVal = 0;	/*	Set value to zero.	*/
		count--;
		ScriptPortClass[PortNum].StackTop ++ ;
	}

	ScriptPortClass[PortNum].CodeIdx ++ ;
	return 0 ;

error_exit :
	return -2 ;
}	

/*----------------------------------------------------------------------------
	Name		:	KillFun													
	Input		:	None													
	Output 	:	Zero													
	Synopsis	:	Handles kill operator.									
----------------------------------------------------------------------------*/
int KillFun (BYTE PortNum)			
{
	ArgType *a_ptr1 ;
	WORD address ;

	address = (WORD) (ScriptPortClass[PortNum].CodeBuffer[ScriptPortClass[PortNum].CodeAddr + 1]) ;

	a_ptr1 = ScriptPortClass[PortNum].TempsPtr + address ;

	if (a_ptr1->ArgType == E_STRING && a_ptr1->StrVal) 
	{
		free (a_ptr1->StrVal) ;
		a_ptr1->StrVal = 0 ;
	}
	ScriptPortClass[PortNum].CodeIdx ++ ;
	return 0 ;
}
	

/*----------------------------------------------------------------------------
	Name		:	UnaryOperatorFun											
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles unary minus and negation operator.				
----------------------------------------------------------------------------*/
int UnaryOperatorFun (BYTE operator, BYTE PortNum)
{
	ArgType *a_ptr1 ;
	ArgType *a_ptr2 ;
	ValType *v_ptr ;
	void *arg_addr ;	/*	To remember destination argument address.	*/
	long int_arg_1 ;

	if (operator != Q_UNARY_MINUS && operator != Q_NOT)
		goto error_exit ;
	ScriptPortClass[PortNum].CodeIdx ++ ;

	a_ptr1 = (ArgType *) &(ScriptPortClass[PortNum].CodeBuffer[ScriptPortClass[PortNum].CodeAddr + 1]) ;
	a_ptr2 = a_ptr1 + 1 ;

	v_ptr = GetArgValueAddr (a_ptr2, PortNum) ;
	if (!v_ptr)
		goto error_exit ;
	arg_addr = ArgAddr ;	/*	Remember argument address.	*/

	switch (a_ptr1->ArgType) 
	{
		case E_INT_CONST :
		case E_INTEGER :
			int_arg_1 = GetIntArg (a_ptr1, PortNum) ;
			switch (operator) 
			{
				case Q_UNARY_MINUS :
					v_ptr->IntValue = -int_arg_1 ;
					break ;

				case Q_NOT :
					v_ptr->IntValue = !int_arg_1 ;
					break ;
			}
			a_ptr2->ArgType = E_INTEGER ;	   /*	Set target type.	*/
			break ;

		default :
			goto error_exit ;
	}
	#if 1
	/*	Destination operand type is the same as source operand type.	*/
	/*	If the target is RET_VAR or temporary, patch the type.	*/
	switch (a_ptr2->StorageClass) 
	{
		case VAR_CONDITION :
		case VAR_RETURN :
			((SymType *) arg_addr)->SymClass = a_ptr1->ArgType ;
			break ;

		case VAR_TEMPORARY :
			((ArgType *) arg_addr)->ArgType = a_ptr1->ArgType ;
			break ;
	}
	#endif
	return 0 ;

error_exit :
	return -2 ;
}


/*----------------------------------------------------------------------------
	Name		:	ArithmeticFun											
	Input		:	operator, arithmetic operator.							
	Output	:	0 if successfully done, else -2.						
	Synopsis	:	Handles arithmetic operators.							
----------------------------------------------------------------------------*/
int ArithmeticFun (BYTE operator, BYTE PortNum)
{
	ArgType *a_ptr1 ;
	ArgType *a_ptr2 ;
	ArgType *a_ptr3 ;
	ValType *v_ptr ;
	void *arg_addr ;	/*	To remember destination argument address.	*/
	long int_arg_1 ;
	long int_arg_2 ;

	ScriptPortClass[PortNum].CodeIdx ++ ;
	a_ptr1 = (ArgType *) &(ScriptPortClass[PortNum].CodeBuffer[ScriptPortClass[PortNum].CodeAddr + 1]);
	a_ptr2 = a_ptr1 + 1 ;
	a_ptr3 = a_ptr1 + 2 ;
	v_ptr = GetArgValueAddr (a_ptr3, PortNum) ;
	if (!v_ptr)
		goto error_exit ;
	arg_addr = ArgAddr ;	/*	Remember argument address.	*/

	switch (a_ptr1->ArgType) 
	{
		case E_INT_CONST :
		case E_INTEGER :
			int_arg_1 = GetIntArg (a_ptr1, PortNum) ;

			switch (a_ptr2->ArgType) 
			{
				case E_INT_CONST :
				case E_INTEGER :
					int_arg_2 = GetIntArg (a_ptr2, PortNum) ;

					switch (operator) 
					{
						case Q_PLUS :
							v_ptr->IntValue = int_arg_1 + int_arg_2 ;
							break ;

						case Q_MINUS :
							v_ptr->IntValue = int_arg_1 - int_arg_2 ;
							break ;

						case Q_TIMES :
							v_ptr->IntValue = int_arg_1 * int_arg_2 ;
							break ;

						case Q_DIV :
							if (!int_arg_2) 
							{
								script_printf (ALARM_SCRIPT_PRINTF,
									"Division by zero, port %d\n\r", PortNum) ;
								goto error_exit ;
							}
							v_ptr->IntValue = int_arg_1 / int_arg_2 ;
							break ;

						default :
							goto error_exit ;
					}
					a_ptr3->ArgType = E_INTEGER ;	 /*	Set target type.	*/
					break ;

				default :
					goto error_exit ;
			}
			break ;

		default :
			goto error_exit;
	}

	#if 1
	/*	If the target is RET_VAR or temporary, patch the type.	*/
	switch (a_ptr3->StorageClass) 
	{
		case VAR_CONDITION :
		case VAR_RETURN :
			((SymType *) arg_addr)->SymClass = a_ptr3->ArgType ;
			break ;

		case VAR_TEMPORARY :
			((ArgType *) arg_addr)->ArgType = a_ptr3->ArgType ;
			break ;
	}
	#endif
	return 0 ;

error_exit :
	return -2 ;
}

/*----------------------------------------------------------------------------
	Name		:	RelationalFun											
	Input		:	operator, relational operator.							
	Output 	:	Zero													
	Synopsis	:	Handles relational operators (<=, >=, ==, !=, <, >).	
----------------------------------------------------------------------------*/
int RelationalFun (BYTE operator, BYTE PortNum)
{
	ArgType *a_ptr1 ;
	ArgType *a_ptr2 ;
	ArgType *a_ptr3 ;
	ValType *v_ptr ;
	void *arg_addr ; 	/*	To remember destination argument address.	*/
	long int_arg_1 ;
	long int_arg_2 ;
	int result ; 

	ScriptPortClass[PortNum].CodeIdx ++ ;

	a_ptr1 = (ArgType  *) &(ScriptPortClass[PortNum].CodeBuffer[ScriptPortClass[PortNum].CodeAddr + 1]) ;
	a_ptr2 = a_ptr1 + 1 ;
	a_ptr3 = a_ptr1 + 2 ;

	v_ptr = GetArgValueAddr (a_ptr3, PortNum) ;
	if (!v_ptr)
		goto error_exit ;

	arg_addr = ArgAddr ;	/*	Remember argument address.	*/

	a_ptr3->ArgType = E_INTEGER ;	/*	Set target type.	*/
	v_ptr->IntValue = 0 ;

	switch (a_ptr1->ArgType)
	{
		case E_INT_CONST :
		case E_INTEGER :
			int_arg_1 = GetIntArg (a_ptr1, PortNum) ;

			switch (a_ptr2->ArgType) 
			{
				case E_INT_CONST :
				case E_INTEGER :
					int_arg_2 = GetIntArg (a_ptr2,PortNum) ;
					switch (operator) 
					{
						case Q_LESS_OR_EQUAL :
							result = int_arg_1 <= int_arg_2 ;
							break ;

						case Q_GREATER_OR_EQUAL :
							result = int_arg_1 >= int_arg_2 ;
							break ;

						case Q_IS_EQUAL :
							result = int_arg_1 == int_arg_2 ;
							break ;

						case Q_IS_NOT_EQUAL :
							result = int_arg_1 != int_arg_2 ;
							break ;

						case Q_LESS :
							result = int_arg_1 < int_arg_2 ;
							break ;

						case Q_GREATER :
							result = int_arg_1 > int_arg_2 ;
							break ;

						default :
							goto error_exit ;
					}
					break ;

				default :
					goto error_exit ;
			}
			break ;

		case E_STRING_CONST :
		case E_STRING :
			switch (a_ptr2->ArgType) 
			{
				case E_STRING_CONST :
				case E_STRING :
					result = strcmp (GetStrArg (a_ptr1, PortNum), 
													GetStrArg (a_ptr2, PortNum)) ;
					switch (operator) 
					{
						case Q_LESS_OR_EQUAL :
							result = result <= 0 ;
							break ;

						case Q_GREATER_OR_EQUAL :
							result = result >= 0 ;
							break ;

						case Q_IS_EQUAL :
							result = result == 0 ;
							break ;

						case Q_IS_NOT_EQUAL :
							result = result != 0 ;
							break ;

						case Q_LESS :
							result = result < 0 ;
							break ;

						case Q_GREATER :
							result = result > 0 ;
							break ;

						default :
							goto error_exit ;
					}
					break ;

				default :
					goto error_exit ;
			}
			break ;

		default :
			goto error_exit ;
	}
	v_ptr->IntValue = result ;

	#if 1
	/*	If the target is RET_VAR or temporary, patch the type.	*/
	switch (a_ptr3->StorageClass)
	{
		case VAR_CONDITION :
		case VAR_RETURN :
			((SymType *) arg_addr)->SymClass = E_INTEGER ;
			break ;

		case VAR_TEMPORARY :
			((ArgType *) arg_addr)->ArgType = E_INTEGER ;
			break ;
	}
	#endif
	return 0 ;

error_exit :
	return -2 ;
}


/*----------------------------------------------------------------------------
	Name		:	LogicalFun												
	Input		:	operator, logical operator.								
	Output	:	Zero													
	Synopsis	:	Handles logical operators (&&, ||).						
----------------------------------------------------------------------------*/

int LogicalFun (BYTE operator, BYTE PortNum)
{
	ArgType *a_ptr1 ;
	ArgType *a_ptr2 ;
	ArgType *a_ptr3 ;
	ValType *v_ptr ;
	void *arg_addr ;	/*	To remember destination argument address.	*/
	long int_arg_1 ;
	long int_arg_2 ;
	int result ;

	if (operator != Q_AND && operator != Q_OR)
		goto error_exit ;

	ScriptPortClass[PortNum].CodeIdx ++ ;

	a_ptr1 = (ArgType *) &(ScriptPortClass[PortNum].CodeBuffer[ScriptPortClass[PortNum].CodeAddr + 1]) ;
	a_ptr2 = a_ptr1 + 1 ;
	a_ptr3 = a_ptr1 + 2 ;

	v_ptr = GetArgValueAddr (a_ptr3, PortNum) ;
	if (!v_ptr)
		goto error_exit ;

	arg_addr = ArgAddr ;	  /*	Remember argument address.	*/

	a_ptr3->ArgType = E_INTEGER ;	 /*	Set target type.	*/
	v_ptr->IntValue = 0 ;

	switch (a_ptr1->ArgType) 
	{
		case E_INT_CONST :
		case E_INTEGER :
			int_arg_1 = GetIntArg (a_ptr1,PortNum) ;

			switch (a_ptr2->ArgType) 
			{
				case E_INT_CONST :
				case E_INTEGER :
					int_arg_2 = GetIntArg (a_ptr2, PortNum) ;

					switch (operator) 
					{
						case Q_AND :
							result = int_arg_1 && int_arg_2 ;
							break ;

						case Q_OR :
							result = int_arg_1 || int_arg_2 ;
							break ;
					}
					v_ptr->IntValue = result ;
					break ;

				default :
					goto error_exit ;
			}
			break ;

		default :
			goto error_exit ;
	}
	#if 1
	/*	If the target is RET_VAR or temporary, patch the type.	*/
	switch (a_ptr3->StorageClass) 
	{
		case VAR_CONDITION :
		case VAR_RETURN :
			((SymType *) arg_addr)->SymClass = E_INTEGER ;
			break ;

		case VAR_TEMPORARY :
			((ArgType *) arg_addr)->ArgType = E_INTEGER ;
			break ;
	}
	#endif
	return 0 ;

error_exit :
	return -2 ;
}


/*----------------------------------------------------------------------------
	Name		:	JzJnzFun													
	Input		:	None													
	Output	:	Zero													
	Synopsis	:	Handles jz and jnz operators.							
----------------------------------------------------------------------------*/
int JzJnzFun (BYTE operator, BYTE PortNum)
{
	ArgType *a_ptr1 ;
	long int_arg_1 ;
	int result ;

	if (operator != Q_JZ && operator != Q_JNZ)
		goto error_exit ;

	a_ptr1 = (ArgType *) &(ScriptPortClass[PortNum].CodeBuffer[ScriptPortClass[PortNum].CodeAddr + 1]) ;

	switch (a_ptr1->ArgType) 
	{
		case E_INT_CONST :
		case E_INTEGER :
			int_arg_1 = GetIntArg (a_ptr1, PortNum) ;

			switch (operator) 
			{
				case Q_JZ :
					result = int_arg_1 == 0 ;
					break ;

				case Q_JNZ :
					result = int_arg_1 != 0 ;
					break ;
			}
			break ;

		default :
			goto error_exit ;
	}
	if (result)		/*	Condition successful, branch.	*/
		ScriptPortClass[PortNum].CodeIdx += *((WORD *) (a_ptr1 + 1)) ;
	else
		ScriptPortClass[PortNum].CodeIdx ++ ;
	return 0 ;

error_exit :
	return -2 ;
}

/*----------------------------------------------------------------------------
	Name		:	JeqJneqFun												
	Input		:	operator, conditional operator.							
	Output	:	Zero													
	Synopsis	:	Handles jeq and jneq operator.							
----------------------------------------------------------------------------*/
int JeqJneqFun (BYTE operator, BYTE PortNum)
{
	ArgType *a_ptr1 ;
	ArgType *a_ptr2 ;
	long int_arg_1 ;
	long int_arg_2 ;
	int result ;

	if (operator != Q_JEQ && operator != Q_JNEQ)
		goto error_exit ;  	/*	Invalid operator.	*/

	a_ptr1 = (ArgType *) &(ScriptPortClass[PortNum].CodeBuffer[ScriptPortClass[PortNum].CodeAddr + 1]) ;
	a_ptr2 = a_ptr1 + 1 ;

	switch (a_ptr1->ArgType) 
	{
		case E_INT_CONST :
		case E_INTEGER :
			int_arg_1 = GetIntArg (a_ptr1, PortNum) ;

			switch (a_ptr2->ArgType) 
			{
				case E_INT_CONST :
				case E_INTEGER :
					int_arg_2 = GetIntArg (a_ptr2, PortNum) ;

					switch (operator) 
					{
						case Q_JEQ :
							result = int_arg_1 == int_arg_2 ; 
							break ;

						case Q_JNEQ :
							result = int_arg_1 != int_arg_2 ;
							break ;
					}
					break ;

				default :
					goto error_exit ;
			}
			break ;

		case E_STRING_CONST :
		case E_STRING :
			switch (a_ptr2->ArgType) 
			{
				case E_STRING_CONST :
				case E_STRING :
					result = strcmp (GetStrArg (a_ptr1, PortNum), 
														GetStrArg (a_ptr2, PortNum)) ;
					switch (operator) 
					{
						case Q_JEQ :
							result = result == 0 ;
							break ;

						case Q_JNEQ :
							result = result != 0 ;
							break ;
					}
					break ;

				default :
					goto error_exit ;
			}
			break ;

		default :
			goto error_exit ;
	}
	if (result)			/*	Condition successful.	*/
		ScriptPortClass[PortNum].CodeIdx += *((WORD *) (a_ptr2 + 1)) ;
	else
		ScriptPortClass[PortNum].CodeIdx ++ ;
	return 0 ;

error_exit :
	return -2 ;
}


