#include <kstart.h>
#include <string.h>

#include "scrtasm.h"
#include "scrtutls.h"
#include "scrtwan.h"


/*----------------------------------------------------------------------------
	Name		:	SendBreak
	Input		:	Wan Port number, Break time (in milli seconds)													
	Output	:	None													
	Synopsis	:  Writes the number of Break characters calculated
					based on the baud rate, number of data bits,
					stop bits, parity bits and the brak time specified
					in milli seconds
					For now there is the limitation of 65536 break characters
	Note     :  Any function calling this routine has to make sure
					that it will wait for the break time before issuing
					the RESTART TRANSMIT command
----------------------------------------------------------------------------*/
void SendBreak (ULONG PortNum, ULONG BreakTime)
{
	ULONG BaudRate ;

	USHORT DataBits ;
	USHORT ParityBits ;
	USHORT StopBits ;
	USHORT StartBits = 1 ;

	USHORT BitsPerChar ;
	ULONG NoOfBrkChars ;

	BaudRate = (ULONG) GetBaudRate (PortNum) ;
	DataBits = GetDataBits (PortNum) ;
	ParityBits = GetParityBits (PortNum) ;
	StopBits = GetStopBits (PortNum) ;
	BitsPerChar = StartBits + DataBits + ParityBits + StopBits ;

	/* Assume lowest baud rate to be 300 baud */
	/* Take care of multiplication and division */
	NoOfBrkChars = 
		 (ULONG) ((ULONG) ((ULONG) (BaudRate / 100) * (ULONG) (BreakTime / 10)) / (ULONG) BitsPerChar) ;

	if (NoOfBrkChars > 65535)
		NoOfBrkChars = 65535 ;

	SetNoOfBrkChars (PortNum, (USHORT) NoOfBrkChars) ;
	WriteToCommandRegister ((ULONG) STOP_TRANSMIT, (ULONG) PortNum) ;
}


int GetBaudRate (ULONG PortNum)
{
	int BaudIndex ;
	int index ;
	BYTE RcvClkSrc ;

	RcvClkSrc = GetBRGClockSource (PortNum) ;
	if (RcvClkSrc == 0xFF)
	{
		return 9600 ;
	}
	BaudIndex = GetBaudRateIndex (RcvClkSrc) ;

	for (index = 0 ; index < AsyncBaudEntries ; index ++)
	{
		if (BRGConfigValues[index] == BaudIndex)
			return BaudRateValues[index] ;
	}
	return 9600 ;  /* In case none of them matches */
}


BYTE *my_itoa (long num, BYTE *buf, int rad) 
{
	int index = 0 ;
	int pos_digit ;
	int i ;
	BYTE temp ;
	BYTE sign = '+' ;

	if (rad <= 1 || rad > 36)
	{
		buf[0] = 0 ;
		return buf ;
	}

	if (num < 0)
	{
	 	sign = '-' ;
		num = -num ;
	}

	while (num)
	{
		pos_digit = num % rad ;
		if (pos_digit > 9)
			pos_digit += 7 ; /* pos_digit += (9 - pos_digit) + 8 */
		buf[index++] = (BYTE) pos_digit + '0' ;
		num /= rad ;
	}

	if (sign == '-' && rad == 10)
		buf[index++] = sign ;
	buf[index] = 0 ;

	for (i = 0 ; i < index / 2 ; i ++)
	{
	 	temp = buf[i] ;
		buf[i] = buf[index - i - 1] ;
		buf[index - i - 1] = temp ;		
	}
	return buf ;
}


/*----------------------------------------------------------------------------
	Name		:	ConvertToUpperCase										
	Input		:	str, mixed case input string.							
	Output	:	None													
	Synopsis	:	This routine converts the input script command line to	
					upper case. Command name and args except string		
					constants are changed.								
----------------------------------------------------------------------------*/
void ConvertToUpperCase (BYTE *str)
{
	for ( ; *str ; str ++)
	{
		if (*str == '"') 
		{	/*	Retain string constants.		*/
			str ++ ;
			while (*str && *str != '"') 
			{
				str ++ ;
			}
		}
		if (*str >= 'a' && *str <= 'z')
			*str -= 'a' - 'A' ;
	}
}

/*----------------------------------------------------------------------------
	Name		:	ConvertToLowerCase										
	Input		:	str, mixed case input string.							
	Output	:	None													
	Synopsis	:	This routine converts the input script command line to	
					lower case. Command name and args except string		
					constants are changed.								
----------------------------------------------------------------------------*/
void ConvertToLowerCase (BYTE *str)
{
	for ( ; *str ; str ++)
	{
		if (*str == '"') 
		{	/*	Retain string constants.		*/
			str ++ ;
			while (*str && *str != '"') 
			{
				str ++ ;
			}
		}
		if (*str >= 'A' && *str <= 'Z')
			*str += 'a' - 'A' ;
	}
}


void SetBaudRate (ULONG PortNum, long BaudRate)
{
	int BaudRateIdx ;
	BYTE RcvClkSource ;

	/* Baud index to be written to BRGC register */
	BaudRateIdx = GetBaudDivisor (BaudRate) ;
	RcvClkSource = GetBRGClockSource (PortNum) ;
	if (RcvClkSource == 0xFF)
		return ;
	WriteBaudToBRGCReg (RcvClkSource, BaudRateIdx) ;
}


BYTE GetBRGClockSource (ULONG PortNum)
{
	DWORD ClkSourceBits ;
	DWORD ThisPortBits ;
	BYTE RcvClkSrc ;
	BYTE TxClkSrc ;

	/* get the clock source bits for all ports */
	ClkSourceBits = ReadSIClkRouteReg() ;

	/* get the particular port's clock source bits */
	ThisPortBits = ClkSourceBits >> (8 * (PortNum + 1)) ;
	ThisPortBits &= 0x3F ;

	/* get the receive clk source */
	RcvClkSrc = (ThisPortBits & 0x38) >> 3 ;
	/* get the transmit clk source */
	TxClkSrc = ThisPortBits & 0x3 ;

	/* rcv and tx sections use different clk source */
	if (RcvClkSrc != TxClkSrc)
	{
		printf ("WAN Port%d using different clock sources for Rx and Tx\n\r", PortNum + 1) ;
		return 0xFF ;
	}

	/* external clk source used */
	if (RcvClkSrc > 3)
	{
		printf ("WAN Port%d using external clock source for Rx and Tx\n\r", PortNum + 1) ;
		return 0xFF ;
	}
	return RcvClkSrc ;
}

void SetMaxIDL (USHORT PortNum, USHORT MaxIDLChars)
{
	switch (PortNum)
	{
		case 0 :
			SetMaxIDL2 (MaxIDLChars) ;
			break ;

		case 1 :
			SetMaxIDL3 (MaxIDLChars) ;
			break ;

		case 2 :
			SetMaxIDL4 (MaxIDLChars) ;
			break ;

		default :
			break ;
	}
}



USHORT GetMaxIDL (USHORT PortNum)
{
	switch (PortNum)
	{
		case 0 :
			return (GetMaxIDL2()) ;

		case 1 :
			return (GetMaxIDL3()) ;

		case 2 :
			return (GetMaxIDL4()) ;

		default :
			return 0 ;
	}

}


void FlushReceiver (USHORT PortNum)
{
	switch (PortNum)
	{
		case 0 :
			FlushSCC2Rx() ;
			return ;

		case 1 :
			FlushSCC3Rx() ;
			return ;

		case 2 :
			FlushSCC4Rx() ;
			return ;

		default :
			return ;
	}
}


void FlushTransmitter (USHORT PortNum)
{
	switch (PortNum)
	{
		case 0 :
			FlushSCC2Tx() ;
			return ;

		case 1 :
			FlushSCC3Tx() ;
			return ;

		case 2 :
			FlushSCC4Tx() ;
			return ;

		default :
			return ;
	}
}


void SetPortCAsOutPut (int PortNum)
{
	switch (PortNum)
	{
		case 0 :
			SetPortCAsOutPutForWAN1() ;
			break ;

		case 1 :
			SetPortCAsOutPutForWAN2() ;
			break ;

		case 2 :
			SetPortCAsOutPutForWAN3() ;
			break ;

		default :
			break ;
	}
}

