/* SERTX.C -- Code handling the transmit part
** By: Sanjay
** Start: 22, July, 1996
** Done: 22, July, 1996
*/

#include <stdio.h>
#include <memory.h>
#include "rtrstd.h"
#include "serial.h"
#include "vserstr.h"
#include "serasm.h"

/* Global Prototypes */
void serial_tx_isr_complete(int port_number);
#ifdef DEBUG
extern void DebugBreak(void);
#endif

/* Local Prototypes */
STATIC void serial_schedule_tx_on_scc(USHORT port_number, SERIAL_RXTX_DESCRIPTOR *sptr_tx_descriptor);

/* Externals */
extern SERIAL_CLASS serial;

/* -- CODE ---------------------------------------------------------------- */

/* serial_tx_isr_complete()
**	Called after a buffer has been transmitted. Called with interrupts 
**	disabled. Will put the current tx descriptor in the free list. Will
**	also schedule another buffer for TX if available.
*/
void serial_tx_isr_complete(int port_number)
{
	SERIAL_RXTX_DESCRIPTOR *sptr_tx_descriptor;


	sptr_tx_descriptor = serial.sptr_current_tx_descriptor[port_number];
	if (sptr_tx_descriptor == NULL)
	{
#ifdef DEBUG
		printf("SERIAL: Strange! Tx interrupt occured without any descriptor\r\n");
#endif /* DEBUG */
		return;
	}

	/* On completion, the buffer used must be freed either by the driver
	** itself or the app. Since we can possibly re-enter the free routines
	** if we do it here, we simply queue up here and call the callbacks or
	** free() in the foreground.
	*/
	add_entry_to_list(&serial.tx_complete_list[port_number], &sptr_tx_descriptor->links);

	/* Schedule another buffer for transmit if available. */
	sptr_tx_descriptor = get_entry_from_list((LINK *)&serial.current_tx_list[port_number]);
	serial.sptr_current_tx_descriptor[port_number] = sptr_tx_descriptor;

	if (sptr_tx_descriptor != NULL)
	{
		serial_schedule_tx_on_scc(port_number, sptr_tx_descriptor);
		serial.statistics[port_number].tx_pending_bytes -= sptr_tx_descriptor->length;
	}
}

STATIC void serial_schedule_tx_on_scc(USHORT port_number, SERIAL_RXTX_DESCRIPTOR *sptr_tx_descriptor)
{
	switch (port_number)
	{
	case 0:
		serial_send_scc2(sptr_tx_descriptor->sptr_buffer, sptr_tx_descriptor->length);
		break;
	case 1:
		serial_send_scc3(sptr_tx_descriptor->sptr_buffer, sptr_tx_descriptor->length);
		break;
	case 2:
		serial_send_scc4(sptr_tx_descriptor->sptr_buffer, sptr_tx_descriptor->length);
		break;
	}
}

enum TEST serial_tx_packet(USHORT port_number, BYTE *sptr_tx_buffer, USHORT number_of_bytes, enum TX_BUFFER_OWNER tx_buffer_owner)
{
	BYTE *bptr;
	USHORT temp_sr;
	USHORT bytes_to_malloc;
	SERIAL_RXTX_DESCRIPTOR *sptr_tx_descriptor;


#ifdef DEBUG
	if (number_of_bytes > 1518)
	{
		DebugBreak();
		return;
	}
#endif

	if (number_of_bytes == 0)
	{
		if (tx_buffer_owner == APPLICATION)
		{
			serial.fptr_tx_complete[port_number](port_number, serial.user_parameter[port_number], sptr_tx_buffer);
		}
	}

	temp_sr = _GPL();
	_SPL(7);

	sptr_tx_descriptor = get_entry_from_list(&serial.free_tx_list[port_number]);
	if (sptr_tx_descriptor == NULL)
	{
		_SPL(temp_sr);
		return FAIL;
	}

	if (tx_buffer_owner == APPLICATION)
	{
		sptr_tx_descriptor->length = number_of_bytes;
		sptr_tx_descriptor->sptr_buffer = sptr_tx_buffer;
		sptr_tx_descriptor->tx_buffer_owner = APPLICATION;
	}
	else
	{
		/* Allocate sufficient space to contain a LINK record at the minimum */
		bytes_to_malloc = (number_of_bytes < sizeof(LINK)) ? sizeof(LINK) : number_of_bytes;
		bptr = (BYTE *)buffer_malloc(bytes_to_malloc);
		if (bptr == NULL)
		{
			_SPL(temp_sr);
			return FAIL;
		}
		memcpy(bptr, sptr_tx_buffer, number_of_bytes);
		sptr_tx_descriptor->length = number_of_bytes;
		sptr_tx_descriptor->sptr_buffer = bptr;
		sptr_tx_descriptor->tx_buffer_owner = DRIVER;
	}

	/* If no packet is currently being transmitted, transmit this, else, 
	** queue it up.
	*/
	if (serial.sptr_current_tx_descriptor[port_number] == NULL)
	{
		serial.sptr_current_tx_descriptor[port_number] = sptr_tx_descriptor;
		serial_schedule_tx_on_scc(port_number, sptr_tx_descriptor);
	}
	else
	{
		add_entry_to_list(&serial.current_tx_list[port_number], (LINK *)&sptr_tx_descriptor->links);
		serial.statistics[port_number].tx_pending_bytes += number_of_bytes;
	}

	_SPL(temp_sr);
	return PASS;
}

/* -- END CODE ------------------------------------------------------------ */

