#include "defs.h"		 
/*---------------------------------------------------------------------------
	file		:	userlog.c
	Auth		:	Md.Imran Naveed
	Date		:	22 Mar '99
	Synopsis :  Logging of important user events such as logon,logoff,invalid 
					login attempts etc. over a period of time.
---------------------------------------------------------------------------*/
#include <stdlib.h>
#include <incall.h>
#include <string.h>

#include <usrlog.h> 
#include "userlog.h"
#include "evntstr.h"

#include "..\applicat\sntp\ksntp.h"
#include "..\applicat\sntp\vsntpstr.h"   /* for getting the time */


/* Globals.....*/
user_log_descriptor_type user_log_descriptors[LOG_MAX_OPENS];
Day *user_log_head_ptr;  	    				/* pointer to first row*/
Day *ptr_to_current_day_record;       		/* pointer to current row */
USER_EVENT *ptr_to_current_event_record;  /* pointer to last added event record */
unsigned long user_log_number_of_msgs;	 		 	/* number of log msgs */
unsigned short user_log_number_of_days;       	/* number of rows */
/* ......Globals */

const char *user_log_msgs[] = {
		"User %s logged on",
		"User %s logged off",
		"User %s identified as defaulter, account locked-up",
		"Invalid login attempts by User %s"
		};
	
/*---------------------------------------------------------------------------
init all pointers and other globals.Also make all log_descriptors invalid.
To be called only once during init time
---------------------------------------------------------------------------*/
void init_user_log(void)
{
	int index;
	 
	user_log_head_ptr = ptr_to_current_day_record = NULL;
	ptr_to_current_event_record = NULL;
	user_log_number_of_msgs = 0;         /* number of log msgs */
	user_log_number_of_days = 0;

	for (index = 0; index < LOG_MAX_OPENS; index++)
	{
		user_log_descriptors[index].valid = 0;
	}
}

/*---------------------------------------------------------------------------
this will delete all messages,make the global pointers null and invalidate
the log_descriptors
---------------------------------------------------------------------------*/
void clean_user_log(void)
{
	int index;
	Day *row_ptr, *temp_row_ptr;
	USER_EVENT *col_ptr, *temp_col_ptr;

	row_ptr = user_log_head_ptr;
	while(row_ptr != NULL)
	{
		col_ptr = row_ptr->first_event;
		while(col_ptr != NULL)
		{
			temp_col_ptr = col_ptr;
			col_ptr = col_ptr->next_event;
			free(temp_col_ptr->user_name);
			free(temp_col_ptr);
		}
		temp_row_ptr = row_ptr;
		row_ptr = row_ptr->next_day;
		free(temp_row_ptr);
	}

	user_log_head_ptr = ptr_to_current_day_record = NULL;
	ptr_to_current_event_record = NULL;
	user_log_number_of_msgs = 0;         
	user_log_number_of_days = 0;
	for (index = 0; index < LOG_MAX_OPENS; index++)
	{
		user_log_descriptors[index].valid = 0;
	}
}

/*---------------------------------------------------------------------------
open_user_log: open log file for read. returns the index of log_descriptor
---------------------------------------------------------------------------*/
int open_user_log(void)
{
	int index = 0;

	for(index = 0; index < LOG_MAX_OPENS; index++)
	{
		if (user_log_descriptors[index].valid == FALSE)
		{
			user_log_descriptors[index].valid = TRUE;
			user_log_descriptors[index].day_ptr = user_log_head_ptr;
			user_log_descriptors[index].event_ptr = user_log_head_ptr->first_event;
			return index;
		}
	}
	return LOG_ERR_MAX_OPEN;
}

/*---------------------------------------------------------------------------
just make the descriptor invalid
---------------------------------------------------------------------------*/
int close_user_log(int log_descriptor)
{
	user_log_descriptors[log_descriptor].valid = FALSE;
}

/*---------------------------------------------------------------------------
get the date and time stamp to be put into the buffer
---------------------------------------------------------------------------*/
static int get_log_date_time_stamp(Day* dptr, USER_EVENT* eptr,char *buffer)
{
	sprintf(buffer, "%02d:%02d:%02d     %02d-%02d-%d     ",eptr->hour,eptr->minute,
							eptr->second,dptr->day,dptr->month,dptr->year);
	return strlen(buffer);
}


/*---------------------------------------------------------------------------
Make space by freeing up some records.makesure that all the read pointers in 
between are invalidated
---------------------------------------------------------------------------*/
static void free_up_event_records(void)
{
	Day *old_head_ptr;
	USER_EVENT *first_eptr;
	int ind,index;

	old_head_ptr = user_log_head_ptr;
	for(index = 0; index < LOG_MAX_FREE_UP_QUANTUM ; index++)
	{
		first_eptr = old_head_ptr->first_event;
		old_head_ptr->first_event = first_eptr->next_event;
		for(ind = 0;ind < LOG_MAX_OPENS;ind++)
		{
			if ((user_log_descriptors[ind].valid == TRUE)	&&
				 (user_log_descriptors[ind].event_ptr == first_eptr))
				 		user_log_descriptors[ind].valid = FALSE; 
		}
		free(first_eptr->user_name);
		free(first_eptr);
		user_log_number_of_msgs--;
		if(!old_head_ptr->first_event)
		{
			if(old_head_ptr->next_day != NULL)
			{
				old_head_ptr = old_head_ptr->next_day;
				free(user_log_head_ptr);
				user_log_number_of_days--;
				user_log_head_ptr = old_head_ptr;
			}
			else
			{
				return;
			}
		}
	}
}



/*---------------------------------------------------------------------------
write_user_log: adds an event record whenever any event is triggered.It first 
checks for the date.If the current date corresponds to the date in the 
ptr_to_current_day_record, then the event is appended at the end of the same
row.If not, then a new row is created and the event is added as the first 
record in that row.
---------------------------------------------------------------------------*/

void write_user_log(EVENT_TYPE_ENUM event_type,BYTE* uname)
{
	SntpDateTime date_time_ptr;
	char time_buf[LOG_TIME_STAMP_LEN],msg_buf[LOG_NORMAL_MSG_LEN];
	EVENT_HEADER event_broadcast_pkt;
	
	GetDateTime(&date_time_ptr); 
	if(user_log_head_ptr == NULL)
	{
		user_log_head_ptr = create_a_new_row_with_one_event_record();
		if(user_log_head_ptr == NULL)
			return;
	}
	else
	{
		if((ptr_to_current_day_record->day != date_time_ptr.Day) ||
			(ptr_to_current_day_record->month != date_time_ptr.Month) ||
			(ptr_to_current_day_record->year != date_time_ptr.Year))
		{
			ptr_to_current_day_record->this_day_completed = TRUE;
			ptr_to_current_day_record = create_a_new_row_with_one_event_record();
			if(ptr_to_current_day_record == NULL)
			 	return;
		}
		else 
		{
			ptr_to_current_event_record->next_event = (USER_EVENT*) malloc (sizeof(USER_EVENT));
			if(ptr_to_current_event_record->next_event == NULL)
				return;
			ptr_to_current_event_record = ptr_to_current_event_record->next_event;
		}
	}
	ptr_to_current_event_record->next_event = NULL;
	ptr_to_current_event_record->event_type = event_type;
	ptr_to_current_event_record->user_name = (BYTE *) malloc (strlen(uname) + 1);
	if(ptr_to_current_event_record->user_name == NULL)
	{
		free(ptr_to_current_event_record);
		return;
	}
	strcpy(ptr_to_current_event_record->user_name,uname); 

	ptr_to_current_event_record->hour   = date_time_ptr.Hour;
	ptr_to_current_event_record->minute = date_time_ptr.Minute;
	ptr_to_current_event_record->second = date_time_ptr.Second;
	ptr_to_current_event_record->read_by_procon = FALSE ;
	user_log_number_of_msgs++;
	if(user_log_number_of_msgs > LOG_MAX_NO_OF_MSGS)
	{
		free_up_event_records();
	}

/* After logging the event,send a broadcast indicating the occurence of this event */
	get_log_date_time_stamp(ptr_to_current_day_record,ptr_to_current_event_record,time_buf);
	sprintf(msg_buf, user_log_msgs[ptr_to_current_event_record->event_type], ptr_to_current_event_record->user_name);
	sprintf(event_broadcast_pkt.event_string ,"%s%s",time_buf,msg_buf);
	event_broadcast_pkt.event_type = USER_EVENT_BDCAST;

	event_broadcast_packets_on_all_ports(&event_broadcast_pkt);
}



/*---------------------------------------------------------------------------
read_user_log: read  "count" number of bytes
input : int log_desc - log descriptor
			char * buffer - buffer to copy read info
			count - max read count;
Algo
	derive from the write_log
	returns count of chars copied into buffer
	advances the read pointers (in log_descriptor) by number of events read.
---------------------------------------------------------------------------*/
int read_user_log(int log_descriptor, char *buffer, int count, int dnld_flag)
{                                                                           
	int max_count,index;
	char time_buf[LOG_TIME_STAMP_LEN],tbuf[LOG_NORMAL_MSG_LEN],msg_buf[LOG_NORMAL_MSG_LEN];
	Day* day_ptr;
	USER_EVENT *event_ptr;

	if ((!user_log_descriptors[log_descriptor].day_ptr) || (!user_log_descriptors[log_descriptor].event_ptr))
		return LOG_NOT_LOGGING;  

	if (user_log_head_ptr == NULL)
		return LOG_NOT_LOGGING;					  

	if (!user_log_descriptors[log_descriptor].valid)
		return LOG_INVALID_DESCRIPTOR;					

	day_ptr = user_log_descriptors[log_descriptor].day_ptr;
	event_ptr = user_log_descriptors[log_descriptor].event_ptr;
	max_count = count;
	*buffer = 0;

	while((day_ptr) && (strlen(buffer) < max_count))
	{
		if(dnld_flag)
		{
			if(event_ptr->read_by_procon)
			{
				event_ptr = event_ptr->next_event;
				if(!event_ptr)
				{
					day_ptr = day_ptr->next_day;
					event_ptr = (day_ptr) ? day_ptr->first_event : NULL ;
				}
				continue;
			}
		}

		get_log_date_time_stamp(day_ptr,event_ptr,time_buf);
		sprintf(msg_buf, user_log_msgs[event_ptr->event_type], event_ptr->user_name);

		sprintf(tbuf,"%s%s%c",time_buf,msg_buf,CHAR_NEW_LINE);
		
		if((strlen(buffer) + strlen(tbuf)) >= max_count)
		{
			break;		/* count should not be exceeded */
		}
		strcat(buffer,tbuf);

		if(dnld_flag)
			event_ptr->read_by_procon = TRUE;

		event_ptr = event_ptr->next_event;
		if(event_ptr == NULL)
		{
			day_ptr = day_ptr->next_day;
			if(day_ptr)
				event_ptr = day_ptr->first_event;
		}
	}
	user_log_descriptors[log_descriptor].day_ptr = day_ptr;
	user_log_descriptors[log_descriptor].event_ptr = event_ptr;

	if ((strlen(buffer) < (max_count-1)) && (event_ptr))
	{  
		for(index=strlen(buffer); index < (max_count-2) ; index++)
		{
			buffer[index] = '#';   /* padding extra space */
		}
		buffer[index++] = CHAR_NEW_LINE;
		buffer[index] = 0;
  	} 
	count = strlen(buffer);
	buffer[count] = CHAR_NEW_LINE;   /* Replace NULL by new line char */
	return (count + 1);
}

/*---------------------------------------------------------------------------
will return the approximate number of bytes and number of messages
present in the log at the moment
---------------------------------------------------------------------------*/
int get_info_user_log(int *approx_total_size, int *approx_number_of_msgs)
{
	*approx_number_of_msgs = user_log_number_of_msgs;
	*approx_total_size = user_log_number_of_msgs * LOG_NORMAL_MSG_LEN;

	return TRUE;
}

/*---------------------------------------------------------------------------
this function will create a node of type Day and one event record, initialise 
it with default values and return a pointer to the node to the caller
---------------------------------------------------------------------------*/
Day* create_a_new_row_with_one_event_record()
{
	Day* new_record_for_day;
	SntpDateTime date_time_ptr;

	if(!ptr_to_current_day_record)
	{
		new_record_for_day = (Day *) malloc (sizeof(Day));
	}
	else
	{
		ptr_to_current_day_record->next_day = (Day *) malloc (sizeof(Day));
		new_record_for_day = ptr_to_current_day_record->next_day ;
	}
	if(new_record_for_day == NULL)
		return NULL;
	new_record_for_day->next_day = NULL;
	new_record_for_day->this_day_completed = FALSE;
	GetDateTime(&date_time_ptr); 
	new_record_for_day->year   = date_time_ptr.Year;
	new_record_for_day->month 	= date_time_ptr.Month;
	new_record_for_day->day 	= date_time_ptr.Day;

	/* create one event record but do not initialise as it will be done by the
		caller */
	new_record_for_day->first_event = (USER_EVENT *) malloc (sizeof(USER_EVENT));
	if(new_record_for_day->first_event == NULL)
	{
		free(new_record_for_day);
		return(NULL);
	}

	/* Update the global variables */
	ptr_to_current_day_record = new_record_for_day;
	ptr_to_current_event_record = new_record_for_day->first_event;
	user_log_number_of_days++;

	return (new_record_for_day);
}

/***********************     last line     ***********************/



