#include	"defs.h"
/*
 * $Log: /IP/RIP.C $
 * 
 * 1     1/23/96 10:52p Titus
 * Revision corresponds to IP Release 1.31
 */
/************************************************************************/
/*	$Modname: rip.c$  $version: 1.15$      $date: 10/25/95$   */
/*
*	$lgb$
1.0 02/02/94 yarran
1.1 02/02/94 yarran IP Initial Release
1.2 02/09/94 yarran Changed header files, added multiple buffer chain for fragmentation and assembly, fixed checksum bug
1.3 02/22/94 yarran Fix i960 compiler error - cast on header.vptr_data.
1.4 02/23/94 yarran Move local variable declaration to top of function.
1.5 02/28/94 yarran Change port flag from USHORT to enum BOOLEAN structure.
1.6 02/28/94 yarran Change route, port flags from bit map to enum BOOLEAN.
1.7 03/01/94 yarran Style changes.
1.8 05/02/94 yarran added rfc1042 changes.
1.9 06/15/94 yarran cosmetic changes.
1.10 09/01/94 ross added BYTE and USHORT_ENUM support.
1.11 10/10/94 ross added rarp, proxy arp, remote access functions
1.12 10/25/94 ross clean up for C++ compiles. Added rwarebuf.h to bottom of meta-include.
1.13 12/20/94 ross
1.14 12/27/94 ross added better table instrumentation via new snmp.
1.15 10/25/95 ross
*	$lge$
*/
/************************************************************************/
/*	Copyright (C) 1989 - 1993 Router Engines, Inc.								*/
/*	Unpublished - rights reserved under the Copyright Laws of the			*/
/*	United States.  Use, duplication, or disclosure by the					*/
/*	Government is subject to restrictions as set forth in					*/
/*	subparagraph (c)(1)(ii) of the Rights in Technical Data and			*/
/*	Computer Software clause at 252.227-7013.										*/
/*	Router Engines, Inc., P.O. Box 3604 Newport Beach, CA 92659				*/
/************************************************************************/
#include	<stdlib.h>
#include <string.h>
#include "ip.h"
/****************************************************************************/
static void	update_existing_route_entry (IP_ROUTE_ENTRY *sptr_route_entry, ULONG gateway_address, USHORT port_number,
	ULONG metric);
static enum TEST create_new_route_entry (IP_ROUTE_ENTRY **ptr_to_sptr_route_entry, ULONG target_ip_address);
static void	update_route_entry (IP_ROUTE_ENTRY *sptr_route_entry, ULONG target_ip_address, BYTE bits, ULONG gateway,
	USHORT port_number, ULONG metric, ROUTE_FLAGS *sptr_flags);
#ifdef __IP__DEBUG__
static void add_route_entry_to_sorted_list (IP_ROUTE_ENTRY *sptr_ip_route_entry_to_add);
#endif /* __IP__DEBUG__ */
/****************************************************************************/
void rip_set_next_regular_update (void)
{
	ip.rip.next_regular_update_time = (USHORT) (ip.rip.minimum_update_time_interval +
		(rand () % ip.rip.maximum_update_time_offset) + 1);
}
/****************************************************************************/
/* This routine is called by ip_router_timer every 30 seconds */

void rip_periodic_timer (void)
{
	IP_ROUTE_ENTRY *sptr_route_entry;
	IP_ROUTE_ENTRY *sptr_next_route_entry;
#ifdef __IP_DEBUG__
	enum BOOLEAN routing_table_is_to_be_printed;

	routing_table_is_to_be_printed = FALSE;
#endif /* __IP__DEBUG__ */

	/* scan all route entries */
#ifdef ADD_IP_DEFAULT_ROUTE_TO_ROUTE_LIST
	sptr_route_entry = get_first_ip_route_entry ();
#else
	sptr_route_entry = (IP_ROUTE_ENTRY *) get_pointer_to_first_entry_in_list ((LINK *) &ip.route_list);
#endif

	while (sptr_route_entry != NULL)
		{
#ifdef ADD_IP_DEFAULT_ROUTE_TO_ROUTE_LIST
		sptr_next_route_entry = get_next_ip_route_entry (sptr_route_entry);
#else
		sptr_next_route_entry = (IP_ROUTE_ENTRY *) get_pointer_to_next_entry_in_list ((LINK *) sptr_route_entry);
#endif

#ifdef RTRERROR
		if ((sptr_route_entry->flags.do_no_aging == FALSE) && (sptr_route_entry->flags.local_interface_route == FALSE)
			&& (sptr_route_entry->flags.route_down == FALSE))
#endif
		if ( (sptr_route_entry->flags.do_no_aging == FALSE) && (sptr_route_entry->flags.local_interface_route == FALSE)
			&& (sptr_route_entry->flags.route_down == FALSE) && (ip.rip.config.route_aging_timeout_value != 0) )
			{
			/*	Advance timer except for static routes or routes to local net/subnet from ports that are up and running.
			 *	If port is down, route_down is set and aging timer should be advanced in order to clean up the route. */

			sptr_route_entry->aging_timer = (USHORT) (sptr_route_entry->aging_timer + ip.rip.periodic_time_counter);

		#ifdef __IP_DEBUG__
			routing_table_is_to_be_printed = TRUE;
		#endif /* __IP__DEBUG__ */
			}

		if ((sptr_route_entry->ipRouteType == INVALID_ROUTE_TYPE) &&
			(sptr_route_entry->aging_timer >= ip.rip.config.route_garbage_collection_timeout_value))
			{

			delete_ip_route_entry (sptr_route_entry);

		#ifdef __IP_DEBUG__
			routing_table_is_to_be_printed = TRUE;
		#endif /* __IP__DEBUG__ */
			}
		else
			{
			if ((sptr_route_entry->ipRouteType != DIRECT_ROUTE_TYPE) && (sptr_route_entry->ipRouteType != INVALID_ROUTE_TYPE) &&
#ifdef RTRERROR
/* Vidy 040696. Due to DOD route_aging_timeout_value could be zero. */
				(sptr_route_entry->aging_timer >= ip.rip.config.route_aging_timeout_value))
#endif
				(sptr_route_entry->aging_timer > ip.rip.config.route_aging_timeout_value))
				{
			#ifdef __IP_DEBUG__
				ip_printf (IP_RIP_PRINTF, "IP/RIP: route to %s via %s aged out\n",
					convert_ip_address_to_dot_format (&ip.print_buffer[0], sptr_route_entry->target),
					convert_ip_address_to_dot_format (&ip.print_buffer_2[0], sptr_route_entry->gateway));
			#endif /* __IP__DEBUG__ */

				/* Should leave this entry in routing table for about 60 seconds then delete it. In the mean time, periodic
				 * broadcast will announce the dead route. The aging timer will keep on increase until it reaches
				 * route_delete_delay_time_value which is 60 seconds above the route_aging_timeout_value. */

				change_route_entry (sptr_route_entry, sptr_route_entry->gateway, INFINITY_METRIC_VALUE,
					sptr_route_entry->port_number);

			#ifdef __IP_DEBUG__
				routing_table_is_to_be_printed = TRUE;
			#endif /* __IP__DEBUG__ */
				}

			/* reset changed flag */

			sptr_route_entry->flags.route_changed = FALSE;
			}

		sptr_route_entry = sptr_next_route_entry;
		}

#ifdef __IP_DEBUG__
	if (routing_table_is_to_be_printed == TRUE)
		{
		ip_print_routing_table ();
		}
#endif /* __IP__DEBUG__ */

	/* now do the broadcast */

	rip_broadcast_updates (FALSE, NO_SUCH_PORT);

#ifdef __IP_DEBUG__
	if (routing_table_is_to_be_printed == TRUE)
		{
		ip_print_routing_table ();
		}
#endif /* __IP__DEBUG__ */
}
/****************************************************************************/
enum BOOLEAN rip_is_it_ok_to_announce_this_route (IP_ROUTE_ENTRY *sptr_route_entry, ULONG destination_address,
	USHORT outgoing_port_number, ULONG *ulptr_netmask)
{

/* Kamal 04/12/1996 */
#if 0
	*ulptr_netmask = ip_get_network_mask (sptr_route_entry->target);
#endif
	*ulptr_netmask = sptr_route_entry->mask ;
/* Kamal 04/12/1996 */

	if ((sptr_route_entry->target & *ulptr_netmask) == (destination_address & ip.port[outgoing_port_number].netmask))
		{
		/* returns true, if the target and the destination are subnets of the same network */

		return (TRUE);
		}
	else if (sptr_route_entry->flags.private_route == TRUE)
		{
		return (TRUE);
		}
	else if (sptr_route_entry->flags.host_route == TRUE)
		{
		/* returns true, if the target is a host on another network */

		return (TRUE);
		}
	else if (sptr_route_entry->target == (sptr_route_entry->target & *ulptr_netmask))
		{
		/* returns true, if the target is another network */

		return (TRUE);
		}

	return (FALSE);
}
/****************************************************************************/
/* This routine is used by RIP, so it does not update the route cache */

IP_ROUTE_ENTRY *rip_search_route_table_with_bits (ULONG destination_ip_address, USHORT bits)
{
	IP_ROUTE_ENTRY *sptr_route_entry;
	ULONG calculated_mask;

	if ((bits == 0x0000)	|| (destination_ip_address == 0x00000000L)) 		/* default route special case */
		{
		if (ip.default_route.port_number != NO_SUCH_PORT)
			{
			return (&ip.default_route);
			}
		else
			{
			return (NULL);
			}
		}

	calculated_mask = 0xffffffffL << (32 - bits);

	destination_ip_address = destination_ip_address & calculated_mask;

#ifdef ADD_IP_DEFAULT_ROUTE_TO_ROUTE_LIST
	sptr_route_entry = get_first_ip_route_entry ();
#else
	sptr_route_entry = (IP_ROUTE_ENTRY *) get_pointer_to_first_entry_in_list ((LINK *) &ip.route_list);
#endif

	while (sptr_route_entry != NULL)
		{
		if (sptr_route_entry->number_of_subnet_mask_bits <= bits)
			{
			if (sptr_route_entry->target == destination_ip_address)
				{
				return (sptr_route_entry);
				}
			}

#ifdef ADD_IP_DEFAULT_ROUTE_TO_ROUTE_LIST
		sptr_route_entry = get_next_ip_route_entry (sptr_route_entry);
#else
		sptr_route_entry = (IP_ROUTE_ENTRY *) get_pointer_to_next_entry_in_list ((LINK *) sptr_route_entry);
#endif
		}

	return (NULL);
}
/****************************************************************************/
/* This routine is used by RIP, so it does not update the route cache. */

IP_ROUTE_ENTRY *rip_search_route_table (ULONG destination_ip_address)
{
	IP_ROUTE_ENTRY *sptr_route_entry;

	if (destination_ip_address == 0x00000000L) 		/* default route special case */
		{
		if (ip.default_route.port_number != NO_SUCH_PORT)
			{
			return (&ip.default_route);
			}
		else
			{
			return (NULL);
			}
		}

#ifdef ADD_IP_DEFAULT_ROUTE_TO_ROUTE_LIST
	sptr_route_entry = get_first_ip_route_entry ();
#else
	sptr_route_entry = (IP_ROUTE_ENTRY *) get_pointer_to_first_entry_in_list ((LINK *) &ip.route_list);
#endif

	while (sptr_route_entry != NULL)
		{
		destination_ip_address = destination_ip_address & sptr_route_entry->mask;

		if (sptr_route_entry->target == destination_ip_address)
			{
			return (sptr_route_entry);
			}

#ifdef ADD_IP_DEFAULT_ROUTE_TO_ROUTE_LIST
		sptr_route_entry = get_next_ip_route_entry (sptr_route_entry);
#else
		sptr_route_entry = (IP_ROUTE_ENTRY *) get_pointer_to_next_entry_in_list ((LINK *) sptr_route_entry);
#endif
		}

	return (NULL);
}
/****************************************************************************/
/* This routine is called by timeout or by receiving RIP response from other routers. */

void change_route_entry (IP_ROUTE_ENTRY *sptr_route_entry, ULONG gateway_address, ULONG metric, USHORT port_number)
{
	USHORT delete_route;

	delete_route = 0x0000;

	/* If changing to different router, we need to add new route and delete old one. If the router is the same,
	 * we need to delete the route if it has become unreachable, or add it back to the table if it had been unreachable. */

	if (sptr_route_entry->gateway != gateway_address)
		{
		/* add a new route entry and delete the old entry */

		if (sptr_route_entry->metric != INFINITY_METRIC_VALUE)
			{
			++delete_route;
			}
		}
	else if (metric == INFINITY_METRIC_VALUE)
		{
		/* route entry age timer expired */

		++delete_route;
		}
	else if (sptr_route_entry->metric == INFINITY_METRIC_VALUE)
		{
		/* The old route has become reachable again */
		}

	if ((sptr_route_entry->flags.local_interface_route == TRUE) && (delete_route > 0x0000))
		{
		/* not a directly connected net anymore, reset the interface flag the gateway address is entered later. */

		sptr_route_entry->flags.local_interface_route = FALSE;

		sptr_route_entry->ipRouteType = INVALID_ROUTE_TYPE;
		}

	update_existing_route_entry (sptr_route_entry, gateway_address, port_number, metric);
}
/****************************************************************************/
static void	update_existing_route_entry (IP_ROUTE_ENTRY *sptr_route_entry, ULONG gateway_address, USHORT port_number,
	ULONG metric)
{
	sptr_route_entry->gateway = gateway_address;
	sptr_route_entry->port_number = port_number;
	sptr_route_entry->metric = (BYTE) metric;
	sptr_route_entry->flags.route_changed = TRUE;
	sptr_route_entry->aging_timer = 0x0000;

	if (metric == INFINITY_METRIC_VALUE)
		{
		sptr_route_entry->ipRouteType = INVALID_ROUTE_TYPE;
		}
}
/****************************************************************************/
/* Add an entry to the IP routing table. Returns 0 on success, -1 on failure
	parameters:
		Target 			IP address prefix
		bits 				Size of target_ip_address address prefix in bits (0-32)
		gateway 			Optional gateway to be reached via interface
		port_number 	Interface to which packet is to be routed
		metric 			Metric for this route entry
		flags 			route entry flags
*/
IP_ROUTE_ENTRY *add_ip_route_entry (ULONG target_ip_address, BYTE bits, ULONG gateway, USHORT port_number, ULONG metric,
	ROUTE_FLAGS *sptr_flags)
{
	IP_ROUTE_ENTRY *sptr_route_entry;

	if (port_number == NO_SUCH_PORT)
		{
		return (NULL);
		}

#ifdef __IP_DEBUG__
	ip_printf (IP_RIP_PRINTF, "IP/RIP: adding route to %s via %s metric %ld port %d bits %d\n",
		convert_ip_address_to_dot_format (&ip.print_buffer[0], target_ip_address),
		convert_ip_address_to_dot_format (&ip.print_buffer_2[0], gateway), metric, port_number, bits);
#endif /* __IP__DEBUG__ */

	if ((bits == 32) && (ip_match_full_address_to_port (target_ip_address) != NO_SUCH_PORT))
		{
		/* Don't accept routes to ourselves */

	#ifdef __IP_ALARM_DEBUG__
		ip_printf (IP_ALARM_PRINTF, "IP/RIP: error: tried to add route to myself %s gateway is %s\n",
			convert_ip_address_to_dot_format (&ip.print_buffer[0], target_ip_address),
			convert_ip_address_to_dot_format (&ip.print_buffer_2[0], gateway));
	#endif /* __IP__ALARM_DEBUG__ */

		return (NULL);
		}

	flush_route_cache ();

	if (bits == 0x00) 																			/* Zero bits refers to the default route */
		{
		sptr_route_entry = &ip.default_route;

#ifdef ADD_IP_DEFAULT_ROUTE_TO_ROUTE_LIST
		update_route_entry (sptr_route_entry, target_ip_address, bits, gateway, port_number, metric, sptr_flags);

		if (create_new_route_entry (&sptr_route_entry, target_ip_address) == FAIL)
			{
			return (NULL);
			}
#endif
		}
	else
		{
		target_ip_address = target_ip_address & (0xffffffffL << (32 - bits)); 	/* Mask off host portion of target_ip_address */

		sptr_route_entry = rip_search_route_table_with_bits (target_ip_address, bits);
		}

	if (sptr_route_entry == NULL)
		{
		if (create_new_route_entry (&sptr_route_entry, target_ip_address) == FAIL)
			{
			return (NULL);
			}
		}

	update_route_entry (sptr_route_entry, target_ip_address, bits, gateway, port_number, metric, sptr_flags);

	/* Srikar, Mar 18, 1997. Added the following initialization for SNMP support */
	sptr_route_entry->metric2 = (ULONG) -1;		/* Unused metric */
	sptr_route_entry->ipRouteInfo = 0;			/* Null object identifier */

	return (sptr_route_entry);
}
/****************************************************************************/
static enum TEST create_new_route_entry (IP_ROUTE_ENTRY **ptr_to_sptr_route_entry, ULONG target_ip_address)
{
	*ptr_to_sptr_route_entry = (IP_ROUTE_ENTRY *) table_malloc (1, sizeof (IP_ROUTE_ENTRY));

	if (*ptr_to_sptr_route_entry == NULL)
		{
	#ifdef __IP_ALARM_DEBUG__
		ip_printf (IP_ALARM_PRINTF, "IP/RIP: table_malloc failed\n");
	#endif /* __IP__ALARM_DEBUG__ */

		return (FAIL);
		}

	(*ptr_to_sptr_route_entry)->use_count = 0x00000000L;
	(*ptr_to_sptr_route_entry)->target = target_ip_address;

	add_entry_to_list ((LINK *) &ip.route_list, (LINK *) *ptr_to_sptr_route_entry);

	return (PASS);
}
/****************************************************************************/
static void	update_route_entry (IP_ROUTE_ENTRY *sptr_route_entry, ULONG target_ip_address, BYTE bits, ULONG gateway,
	USHORT port_number, ULONG metric, ROUTE_FLAGS *sptr_flags)
{
	sptr_route_entry->target = target_ip_address;
	sptr_route_entry->mask = 0xffffffffL << (32 - bits);
	sptr_route_entry->number_of_subnet_mask_bits = bits;
	sptr_route_entry->gateway = gateway;
	sptr_route_entry->metric = (BYTE) metric;
	sptr_route_entry->port_number = port_number;
	sptr_route_entry->aging_timer = 0x0000;

	memcpy (&sptr_route_entry->flags, sptr_flags, sizeof (ROUTE_FLAGS));

	sptr_route_entry->flags.route_changed = TRUE;

	if (sptr_route_entry->number_of_subnet_mask_bits == SUBNET_MASK_NOT_KNOWN)
		{
		sptr_route_entry->flags.no_subnet_mask = TRUE;
		sptr_route_entry->flags.route_down = TRUE;

		ip.rip.need_to_send_icmp_address_mask_request = TRUE;
		}
	else if (sptr_route_entry->number_of_subnet_mask_bits == 32)
		{
		sptr_route_entry->flags.host_route = TRUE;
		}

	if (sptr_route_entry->flags.local_interface_route == TRUE)
		{
		sptr_route_entry->ipRouteType = DIRECT_ROUTE_TYPE;
		}
	else
		{
		sptr_route_entry->ipRouteType = INDIRECT_ROUTE_TYPE;
		}
}
/****************************************************************************/
void delete_ip_route_entry (IP_ROUTE_ENTRY *sptr_route_entry)
{
	if (sptr_route_entry == NULL)
		{
	#ifdef __IP_ALARM_DEBUG__
		ip_printf (IP_ALARM_PRINTF, "IP/RIP: trying to delete non-existant route entry\n");
	#endif /* __IP__ALARM_DEBUG__ */

		return;
		}

	/* TITUS: it is better to flush the cache only if the cached route matches the one that is being deleted */

	flush_route_cache ();

	delete_entry_from_list ((LINK *) &ip.route_list, (LINK *) sptr_route_entry);

	table_free ((void *) sptr_route_entry);
}
/****************************************************************************/
void free_ip_routing_table_entries (void)
{
	IP_ROUTE_ENTRY *sptr_route_entry;
	IP_ROUTE_ENTRY *sptr_route_entry_next;

#ifdef ADD_IP_DEFAULT_ROUTE_TO_ROUTE_LIST
	sptr_route_entry = get_first_ip_route_entry ();
#else
	sptr_route_entry = (IP_ROUTE_ENTRY *) get_pointer_to_first_entry_in_list ((LINK *) &ip.route_list);
#endif

	while (sptr_route_entry != NULL)
		{
#ifdef ADD_IP_DEFAULT_ROUTE_TO_ROUTE_LIST
		sptr_route_entry_next = get_next_ip_route_entry (sptr_route_entry);
#else
		sptr_route_entry_next = (IP_ROUTE_ENTRY *) get_pointer_to_next_entry_in_list ((LINK *) sptr_route_entry);
#endif

		delete_ip_route_entry (sptr_route_entry);

		sptr_route_entry = sptr_route_entry_next;
		}
}
#ifdef __IP__DEBUG__
/****************************************************************************/
static void add_route_entry_to_sorted_list (IP_ROUTE_ENTRY *sptr_ip_route_entry_to_add)
{
	LINKED_LIST_SORT_PARAMETERS linked_list_sort_parameters;

	linked_list_sort_parameters.sptr_list = (LINK *) &ip.route_list;
	linked_list_sort_parameters.sptr_entry_to_add = (LINK *) sptr_ip_route_entry_to_add;

	linked_list_sort_parameters.index[0].offset = offsetof (IP_ROUTE_ENTRY, number_of_subnet_mask_bits);
	linked_list_sort_parameters.index[0].size = sizeof (BYTE);
	linked_list_sort_parameters.index[0]._swap = TRUE;

	linked_list_sort_parameters.index[1].size = 0x0000;

	add_entry_to_sorted_linked_list (&linked_list_sort_parameters);
}
#endif /* __IP__DEBUG__ */
