/*
** TCPACK.C -- Code related to handling incoming ACK packets and updating
**				   send window globals accordingly and arranging for a 
**					retransmit if required.
*/

/* General Notes:
**		There are five things to consider on receiving an ACK
**		1. ACK <= UNA -- ignore this ACK as it is a duplicate one
**		2. ACK > NXT -- an ACK for a segment to the right of the send
**			window -- something we have not yet sent -- how?? -- RFC says
**			send an ACK in response and drop the segment (or queue it?)
**		3. ACK == NXT -- valid ACK acking the full window; stop retransmit
**			timer and update UNA and FILLED
**		4. UNA < ACK < NXT -- valid ACK acking part of the window; keep the 
**			retransmit timer and update only the UNA and FILLED. When the 
**			retransmit timer expires, the yet to be ack-ed portion will be 
**			retransmitted. 
**		5. In cases 3 & 4 above (valid ACK is received), the "current
**			send window" size can be updated if all is okay.
**		6. In cases 3 & 4 above (valid ACK is received) and tcp is currently
**			not in retransmit mode (TCPF_NEEDRTX bit is on), calculate
**			retransmit timeout.
*/

#include "rtrstd.h"
#include "all.h"
#include "tcpextrn.h"

/* tcp_ack()
**		Handles a incoming ACK segment
**	Params:
**		Ptr to connection record for segment
**		Ptr to host format TCP header for segment
** Returns:
**		PASS - to continue with segment processing
**		FAIL - to ditch packet without further ado
** NOTES:
**		ACK should in addition to other things take care of the retransmit
**			queue
*/

enum TEST
tcp_ack(TCP_PER_CONN *ptr_conn_info, TCP_HEADER *tcp_header)
{
	ULONG mrtt;


	if ((tcp_header->ack_num - ptr_conn_info->send_una) <= 0)
	{
#if defined(DEBUG)
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: Duplicate ACK=%X, UNA=%X\n", tcp_header->ack_num, ptr_conn_info->send_una);
#endif /* DEBUG */
		/* DUPE-licate */
		return PASS;				/* though a duplicate, packet is acceptable */
	}

	if ((tcp_header->ack_num - ptr_conn_info->send_nxt) > 0)
	{
#if defined(DEBUG)
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: ACK for pack never sent\n");
#endif /* DEBUG */
		/* ACK for a pack we have not sent */
		send_tcp_control_seg(ptr_conn_info, ptr_conn_info->send_nxt,
				ptr_conn_info->recv_nxt, ACK);

		return FAIL;				/* drop the segment */
	}

	/* Conclusion -- ACK is within send window range */

	/* Clear off elements from the retransmit queue if this ACK ack's 
	** segments in the queue
	*/
	check_effect_of_ack_on_rtx(ptr_conn_info, tcp_header);

	ptr_conn_info->send_una = tcp_header->ack_num;	/* this much ack'ed */

	/* Update the window size */
	if (((ptr_conn_info->send_lwseq - tcp_header->seq_num) < 0) ||
			((ptr_conn_info->send_lwseq == tcp_header->seq_num) &&
			((ptr_conn_info->send_lwack - tcp_header->ack_num) <= 0)))
	{
#if defined(DEBUG)
		if (ptr_conn_info->send_wnd == 0 && tcp_header->win_size > 0)
			tcp_printf(TCP_DEBUG_PRINTF, "TCP: Send window opened...persist timer will be stopped\n");
#endif /* DEBUG */

		ptr_conn_info->send_wnd = tcp_header->win_size; 
		if (tcp_header->win_size > ptr_conn_info->send_max_advert)
		{
			ptr_conn_info->send_max_advert = tcp_header->win_size;
		}

		/* If window size is 0, start off the persist timer */
		if (ptr_conn_info->send_wnd == 0)
		{
#if defined(DEBUG)
			tcp_printf(TCP_DEBUG_PRINTF, "TCP: Zero window advertised\n");
#endif /* DEBUG */
			if (ptr_conn_info->rtx_queue == NULL)
			{
#if defined(DEBUG)
				tcp_printf(TCP_DEBUG_PRINTF, "TCP: Persist timer started\n");
#endif /* DEBUG */
				ptr_conn_info->send_persist_timer = (ptr_conn_info->srtt >> 3) +
												ptr_conn_info->srtt_deviation;
			}
			else
			{
#if defined(DEBUG)
				tcp_printf(TCP_DEBUG_PRINTF, "TCP: Persist timer NOT started assuming retransmit will take care\n");
#endif /* DEBUG */
			}
		}

		ptr_conn_info->send_lwseq = tcp_header->seq_num;
		ptr_conn_info->send_lwack = tcp_header->ack_num;
	}
	else
	{
#if defined(DEBUG)
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: Ignored window advertisement at unsuitable time\n");
#endif /* DEBUG */		
	}

	/* Reestimate retransmit timeout */
	if (((ptr_conn_info->conn_flags & TCPF_RTXON) == 0) &&
			((ptr_conn_info->conn_flags & TCPF_TIMEACK) != 0) &&
			((tcp_header->ack_num - ptr_conn_info->seg_for_rtt_estimate) > 0))
	{
		ptr_conn_info->conn_flags &= ~TCPF_TIMEACK;
		mrtt = get_current_tick_count() - ptr_conn_info->last_seg_sent_tick;
		calc_current_rtx_timeout(ptr_conn_info, mrtt);
	}

	return PASS;
}

/*
** calc_current_rtx_timeout()
**		Calculates the current retransmit timeout value to use using the
**		specified Van Jacobsons method.
** Params:
**		Ptr to connection record for connection
**		Current measured round trip time
**	NOTES:
** Van Jacobsons formula goes as below :
**		err = M - A
**		A = A + g*err
**		D = D + h*(abs(err) - D)
**		RTO = A + 4D
** Where: M - Current measured RTT	(mrtt)
**			 A - Smoothed RTT (rtx_srtt)
**			 D - Smoothed RTT Deviation (rtx_srtt_deviation)
**			 RTO - Retransmit timeout (rtx_timeout)
**			 h = 1/4 (standard value)
**			 g = 1/8 (standard value)
** Since calculations for small values of 'err' can cause loss of info
** (integer maths will treat it as 0), we use a scaled approach as below.
**		err = M - A 
**		AA = AA + err
**		DD = DD + abs(err) - D
**		RTO = A + 4D = AA / 8 + 4 * DD / 4 = AA / 8 + DD
**	Where: 
**		DD = D * 4
**		AA = A * 8
** Also take into account that the timer resolution that we have is 50ms.
** All computation is in "ticks" with each "tick" equal to 50ms in our
** implementation.
*/
void
calc_current_rtx_timeout(TCP_PER_CONN *ptr_conn_info, ULONG mrtt)
{
	long err;


	err = (long)mrtt - (long)(ptr_conn_info->srtt >> 3);
	ptr_conn_info->srtt += err;

	if (err < 0) err = -err;
	ptr_conn_info->srtt_deviation += (err - (ptr_conn_info->srtt_deviation >> 2));
	ptr_conn_info->rtx_timeout = (ptr_conn_info->srtt >> 3) +
															ptr_conn_info->srtt_deviation;


	if (ptr_conn_info->rtx_timeout < tcp.min_rtx_timeout)
	{
		ptr_conn_info->rtx_timeout = tcp.min_rtx_timeout;
		tcp_printf(TCP_PRINTF, "TCP: Retransmit timeout reset to minimum\n");
	}
	if (ptr_conn_info->rtx_timeout > tcp.max_rtx_timeout)
	{
		ptr_conn_info->rtx_timeout = tcp.max_rtx_timeout;
		tcp_printf(TCP_PRINTF, "TCP: Retransmit timeout reset to maximum\n");
	}
}

