/*
** TCPDATA.C -- Code to handle TCP data packets -- properly update
**					 globals and counts, adjust socket queues, etc.
*/

#include "rtrstd.h"
#include "all.h"
#include "tcpextrn.h"

/*
** tcp_data()
**		Called to check if the current packet has any data and if so to 
**		process the data by queueing it up or passing it on to the upper
**		layers. Related globals are setup, delayed ack's are setup, etc.
**		All as per RFC 793.
** Params:
**		Ptr to connection entry for current connection
**		Ptr to host format tcp header for segment that came in
** NOTE:
**		Data will be handled (this function called) only in the LISTEN, 
**			ESTABLISHED, FIN_WAIT_1 and FIN_WAIT_2 states and you can have
**			URG bit set only in these states.
*/

void
tcp_data(TCP_PER_CONN *ptr_conn_info, TCP_HEADER *tcp_header)
{
	BYTE *seg_data;
	ULONG avail_win_size, recv_data_len, recv_wnd_index, i;


	if (IS_SYN_ON(tcp_header))
	{
		/* Execution will come here only in the LISTEN state */
		tcp_header->seq_num++;		/* Skip SYN sequence space in the segment
											** that came in
											*/
	}

	/* We have earlier (immediately on packet reception and validation) 
	** determined that the packet has some data that falls within our 
	** receive window. Now extract that part of the segment that 
	** continues from the last sequence number we have received and 
	** copy from thereon to the limits of our receive window 
	*/

	/* Our special decision being enacted below */
	if ((tcp_header->seq_num - ptr_conn_info->recv_nxt) > 0)
	{
#if defined(DEBUG)
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: Ditching out-of-order data\n");
#endif /* DEBUG */
											/* send an ack indicating what we desire */
		send_tcp_control_seg(ptr_conn_info, ptr_conn_info->send_nxt,
				ptr_conn_info->recv_nxt, ACK);

		/* Also in this case, if the FIN control bit is set, we ignore it
		** and expect a retransmit later -- we could keep the info for 
		** delayed action perhaps, but this is easier.
		*/
		TURN_FIN_OFF(tcp_header);
		return;
	}

	if (ptr_conn_info->seg_len == 0)
		return;

	/* As per specs, URG bit is handled only whenever normal data is
	** handled. So handle URG data in this here function and do our
	** kind of urgent processing (see design docs)
	*/

/*
Satish. 12th March 1997.
Commented the call to tcp_urgent here. tcp_urgent calls the app defined
callback function. The app tries to read the data in the callback itself.
But the data is actually copied to the TCP window later on.
So i have shifted the call to tcp_urgent after copying the data to the
TCP window. No side effects as nothing much is done in tcp_urgent other
than calling app function. 
*/

/*	tcp_urgent(ptr_conn_info, tcp_header); */

	seg_data = ptr_conn_info->seg_data;
	recv_data_len = ptr_conn_info->seg_len;
	if ((tcp_header->seq_num - ptr_conn_info->recv_nxt) < 0)
	{
		/* if part of packet has already been received, copy only from
		** where we need and adjust the amount to copy accordingly
		*/
		seg_data += ptr_conn_info->recv_nxt - tcp_header->seq_num;
		recv_data_len -= ptr_conn_info->recv_nxt - tcp_header->seq_num;
	}

	avail_win_size = ptr_conn_info->recv_buf_size - ptr_conn_info->recv_filled;
	if (avail_win_size == 0)
	{
#if defined(DEBUG)
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: Packet received though receive window is 0 bytes -- junked\n");
#endif /* DEBUG */
		TURN_FIN_OFF(tcp_header);
		return;
	}

	if (recv_data_len > avail_win_size)
	{
		/* We can accomodate only 'avail_win_size' amount of data */
		recv_data_len = avail_win_size;

#if defined(DEBUG)
		if (IS_FIN_ON(tcp_header))
			tcp_printf(TCP_DEBUG_PRINTF, "TCP: FIN detected but turned off\n");
#endif /* DEBUG */
		/* Though FIN is seen in the packet, since we have buffer space to
		** copy only the initial part of the packet, junk the FIN along with
		** the uncopied data
		*/
		TURN_FIN_OFF(tcp_header);
	}
	recv_wnd_index = (ptr_conn_info->recv_first + ptr_conn_info->recv_filled) %
							ptr_conn_info->recv_buf_size;
	for (i = 0; i < recv_data_len; i++, seg_data++)
	{
		(ptr_conn_info->recv_buf)[recv_wnd_index] = *seg_data;
		recv_wnd_index = (recv_wnd_index + 1) % ptr_conn_info->recv_buf_size;
	}

	if (IS_PSH_ON(tcp_header))
	{
		/* Data to be pushed onto app as soon as possible */

		ptr_conn_info->conn_flags |= TCPF_RECVPSH;
	}

	/* Update receive related globals */
	ptr_conn_info->recv_filled += recv_data_len;
	ptr_conn_info->recv_nxt += recv_data_len;


   tcp_urgent(ptr_conn_info, tcp_header);


	/* We will not delay ACK if the PSH bit is set in the incoming
	** segment -- Fix, 19, July, 1996 Sanjay
	*/
	if (IS_PSH_ON(tcp_header))
	{
		send_tcp_control_seg(ptr_conn_info, ptr_conn_info->send_nxt, 
			ptr_conn_info->recv_nxt, ACK);
		ptr_conn_info->conn_flags &= ~TCPF_NEEDACK;
	}
	else
		ptr_conn_info->conn_flags |= TCPF_NEEDACK;
	return;
}

