/* TCPHDR.H -- TCP header format 
*/

#ifndef _TCPHDR_
#define _TCPHDR_

#define FIN  ((USHORT)0x01)
#define SYN  ((USHORT)0x02)
#define RST  ((USHORT)0x04)
#define PSH  ((USHORT)0x08)
#define ACK  ((USHORT)0x10)
#define URG  ((USHORT)0x20)

typedef union {
	struct chl_tag {
#ifndef BIG_ENDIAN
		USHORT codes:6;
		USHORT rsvd:6;
		USHORT hlen:4;
#else /* BIG_ENDIAN */
		USHORT hlen:4;
		USHORT rsvd:6;
		USHORT codes:6;
#endif /* BIG_ENDIAN */
	} code_and_hdrlen;
	USHORT code_and_hdrlen_word;
} CODE_AND_HLEN;

/* EOO = End of options list, NOP = No operation, MSS = Max segment size,
 * WSF = Window scale factor, TS = Time stamp
 */
#define KIND_EOO		((BYTE) 0)
#define KIND_NOP		((BYTE) 1)
#define KIND_MSS		((BYTE) 2)
#define KIND_WSF		((BYTE) 3)
#define KIND_TS		((BYTE) 8)

typedef struct {
	BYTE len;
	USHORT max_seg_size;
} TCP_MSS_OPTION;

typedef struct {
	BYTE len;
	BYTE shift_count;
} TCP_WSF_OPTION;

typedef struct {
	BYTE len;
	ULONG time_stamp_value;
	ULONG time_stamp_echo;
} TCP_TS_OPTION;

typedef struct {
	BYTE kind;
	union {
		TCP_MSS_OPTION opt_MSS;
		TCP_WSF_OPTION opt_WSF;
		TCP_TS_OPTION opt_TS;
	} opt_union;
} TCP_OPTIONS;


typedef struct {
	USHORT src_port;
	USHORT dst_port;
	TCPSEQ seq_num;
	TCPSEQ ack_num;
	CODE_AND_HLEN code_and_hlen;
	USHORT win_size;
	USHORT chk_sum;
	USHORT urg_ptr;
} TCP_HEADER;

/* Following macros are used to check for particular bits being on, setting
** them on or setting them off in the TCP header code field.
** Idea in using these macros is to isolate machine specific code.
** These macros take as parameters a ptr to the TCP header.
*/

#define IS_FIN_ON(thp)		((thp)->code_and_hlen.code_and_hdrlen.codes & FIN)
#define IS_SYN_ON(thp)		((thp)->code_and_hlen.code_and_hdrlen.codes & SYN)
#define IS_RST_ON(thp)		((thp)->code_and_hlen.code_and_hdrlen.codes & RST)
#define IS_PSH_ON(thp)		((thp)->code_and_hlen.code_and_hdrlen.codes & PSH)
#define IS_ACK_ON(thp)		((thp)->code_and_hlen.code_and_hdrlen.codes & ACK)
#define IS_URG_ON(thp)		((thp)->code_and_hlen.code_and_hdrlen.codes & URG)

#define TURN_FIN_ON(thp)	((thp)->code_and_hlen.code_and_hdrlen.codes |= FIN)
#define TURN_SYN_ON(thp)	((thp)->code_and_hlen.code_and_hdrlen.codes |= SYN)
#define TURN_RST_ON(thp)	((thp)->code_and_hlen.code_and_hdrlen.codes |= RST)
#define TURN_PSH_ON(thp)	((thp)->code_and_hlen.code_and_hdrlen.codes |= PSH)
#define TURN_ACK_ON(thp)	((thp)->code_and_hlen.code_and_hdrlen.codes |= ACK)
#define TURN_URG_ON(thp)	((thp)->code_and_hlen.code_and_hdrlen.codes |= URG)

#define TURN_FIN_OFF(thp)	((thp)->code_and_hlen.code_and_hdrlen.codes &= ~FIN)
#define TURN_SYN_OFF(thp)	((thp)->code_and_hlen.code_and_hdrlen.codes &= ~SYN)
#define TURN_RST_OFF(thp)	((thp)->code_and_hlen.code_and_hdrlen.codes &= ~RST)
#define TURN_PSH_OFF(thp)	((thp)->code_and_hlen.code_and_hdrlen.codes &= ~PSH)
#define TURN_ACK_OFF(thp)	((thp)->code_and_hlen.code_and_hdrlen.codes &= ~ACK)
#define TURN_URG_OFF(thp)	((thp)->code_and_hlen.code_and_hdrlen.codes &= ~URG)

#endif

