/*
** TCPSNMP.C -- SNMP code for TCP stack
*/
#include "rtrstd.h"
#include "all.h"
#include "tcpextrn.h"

#include "vtcpsnmp.h"

/* Local prototypes */
static enum TEST tcp_apply_mib_operation_on_current_instance_of_mib_variable(
	void *, MIB_ACCESS_PARAMETERS *,	SNMP_TABLE_ENTRY *);
static enum TEST find_pointer_to_row_in_tcp_mib_table(SNMP_TABLE_ENTRY *,
		MIB_ACCESS_PARAMETERS *, void **);
/* Srikar, Mar 20, 1997. Changed the function prototype, due to change in the object id type from */
/* BYTE to OBJECT_ID */
static enum BOOLEAN get_tcp_mib_table_indices(OBJECT_ID *, USHORT *, enum BOOLEAN *,
		ULONG *, ULONG *, ULONG *, ULONG *);
static TCP_PER_CONN *tcp_calculate_pointer_to_next_row_in_mib_table(
		TCP_PER_CONN *);
/* Srikar, Mar 20, 1997. Changed the function prototype, due to change in the object id type from */
/* BYTE to OBJECT_ID */
static enum TEST build_tcp_table_indices_for_current_instance_of_mib_variable(
	void *, OBJECT_ID *, USHORT *);
static enum BOOLEAN check_if_tcp_mib_table_exists(void);

static enum BOOLEAN check_if_snmp_table_entry_is_valid (SNMP_TABLE_ENTRY *sptr_snmp_table_entry);
/* Srikar, Mar 20, 1997. Changed the function prototype, due to change in the object id type from */
/* BYTE to OBJECT_ID */
static void set_mib_access_parameters (MIB_ACCESS_PARAMETERS *sptr_mib_access_parameters,
	enum MIB_OPERATION mib_operation, char *cptr_mib_string,	BYTE *bptr_variable_value,	USHORT *usptr_size_of_variable_value,
	USHORT *usptr_number_of_subids_in_table_indices, OBJECT_ID *sptr_table_indices, enum BOOLEAN *eptr_end_of_table,
	char **ptr_to_cptr_next_variable_name);
static SNMP_TABLE_ENTRY *get_next_mib_variable (SNMP_TABLE_ENTRY *sptr_snmp_table_entry,
	MIB_ACCESS_PARAMETERS *sptr_mib_access_parameters);
static void	adjust_mib_access_parameters (MIB_ACCESS_PARAMETERS *sptr_mib_access_parameters,
	SNMP_TABLE_ENTRY *sptr_snmp_table_entry);
static void clear_pointer_to_next_variable_name (MIB_ACCESS_PARAMETERS *sptr_mib_access_parameters);
static enum TEST process_scalar_mib_variable (MIB_ACCESS_PARAMETERS *sptr_mib_access_parameters,
	SNMP_TABLE_ENTRY *sptr_snmp_table_entry);
/* Srikar, Mar 20, 1997. Changed the function prototype, due to change in the object id type from */
/* BYTE to OBJECT_ID */
static void	build_indices_for_scalar_mib_variable (OBJECT_ID *sptr_table_indices, USHORT *usptr_number_of_subids_in_table_indices);
/* Srikar, Mar 17, 1997. Added the prototype of function used to compare two entries of the */
/* tcp connection table based on table indices */
static int compare_conn_table_indices(TCP_PER_CONN *ptr_conn_info1, TCP_PER_CONN *ptr_conn_info2);
/* Srikar, Mar 17, 1997. Added the prototype of function used to compare indices with a conn table entry */
static int compare_table_indices_with_an_entry(TCP_PER_CONN *ptr_conn_info1, ULONG local_addr, ULONG local_port,
					ULONG remote_addr, ULONG remote_port);

/*--------------------------------------------------------------------------*/
/* Srikar, Mar 20, 1997. Changed the variable bptr_table_indices to sptr_table_indices and its type */
/* from BYTE * to OBJECT_ID * */
/* Changed the variable usptr_number_of_bytes_in_table_indices to usptr_number_of_subids_in_table_indices */
enum TEST get_tcp_mib_variable(enum MIB_OPERATION mib_operation, char *cptr_mib_string,
	BYTE *bptr_variable_value, USHORT *usptr_size_of_variable_value, USHORT *usptr_number_of_subids_in_table_indices,
	OBJECT_ID *sptr_table_indices, enum BOOLEAN *eptr_end_of_table, char **ptr_to_cptr_next_variable_name)
{
	enum TEST found_variable;
	SNMP_TABLE_ENTRY *sptr_snmp_table_entry;
	MIB_ACCESS_PARAMETERS mib_access_parameters;
	/* Srikar, Mar 25, 1997. Added the following local variable to distinguish between variable not found and get/set failure. */
	enum BOOLEAN variable_found;


#if defined(DEBUG)
	tcp_printf(TCP_PRINTF, "TCP: SNMP: Rx-ed request for mib variable %s\n", cptr_mib_string);
#endif /* DEBUG */
	sptr_snmp_table_entry = &tcp_snmp_table[0];
	set_mib_access_parameters (&mib_access_parameters, mib_operation, cptr_mib_string, bptr_variable_value,
		usptr_size_of_variable_value, usptr_number_of_subids_in_table_indices, sptr_table_indices, eptr_end_of_table,
		ptr_to_cptr_next_variable_name);

	while (check_if_snmp_table_entry_is_valid (sptr_snmp_table_entry) == TRUE)
		{
		if (strcmp (sptr_snmp_table_entry->mib_name, mib_access_parameters.cptr_mib_string) == STRINGS_MATCH)
			{
			if (sptr_snmp_table_entry->sptr_parameters->is_a_table_entry == FALSE)
				{
				found_variable = process_scalar_mib_variable (&mib_access_parameters, sptr_snmp_table_entry);

				return (found_variable);
				}
			else
				{
				/* Srikar, Mar 25, 1997. Added the last parameter. Note : the found_variable is misnomer; instead it */
				/* should have been return_value or so. */
				found_variable = process_tcp_table_mib_variable (&mib_access_parameters, sptr_snmp_table_entry, &variable_found);
				if (variable_found == TRUE &&  found_variable == FAIL)
					return(FAIL);

				if (found_variable == PASS)
					{
					return (found_variable);
					}

				adjust_mib_access_parameters (&mib_access_parameters, sptr_snmp_table_entry);
				}
			}

		sptr_snmp_table_entry = get_next_mib_variable (sptr_snmp_table_entry, &mib_access_parameters);
		}

	clear_pointer_to_next_variable_name (&mib_access_parameters);
	
	return (FAIL);
	
}

static enum BOOLEAN check_if_snmp_table_entry_is_valid (SNMP_TABLE_ENTRY *sptr_snmp_table_entry)
{
	if (sptr_snmp_table_entry->fptr_mib_access_function != NULL)
		{
		return (TRUE);
		}
	else
		{
		return (FALSE);
		}
}
/**********************************************************************************/
/* Srikar, Mar 20, 1997. Changed the variable bptr_table_indices to sptr_table_indices and its type */
/* from BYTE * to OBJECT_ID * */
/* Changed the variable usptr_number_of_bytes_in_table_indices to usptr_number_of_subids_in_table_indices */
static void set_mib_access_parameters (MIB_ACCESS_PARAMETERS *sptr_mib_access_parameters,
	enum MIB_OPERATION mib_operation, char *cptr_mib_string,	BYTE *bptr_variable_value,	USHORT *usptr_size_of_variable_value,
	USHORT *usptr_number_of_subids_in_table_indices, OBJECT_ID *sptr_table_indices, enum BOOLEAN *eptr_end_of_table,
	char **ptr_to_cptr_next_variable_name)
{
	sptr_mib_access_parameters->mib_operation = mib_operation;
	sptr_mib_access_parameters->cptr_mib_string = cptr_mib_string;
	sptr_mib_access_parameters->bptr_variable_value = bptr_variable_value;
	sptr_mib_access_parameters->usptr_size_of_variable_value = usptr_size_of_variable_value;
	sptr_mib_access_parameters->usptr_size_of_table_indices = usptr_number_of_subids_in_table_indices;
	sptr_mib_access_parameters->sptr_table_indices = sptr_table_indices;
	sptr_mib_access_parameters->eptr_end_of_table = eptr_end_of_table;
	sptr_mib_access_parameters->ptr_to_cptr_next_variable_name = ptr_to_cptr_next_variable_name;
}

/**********************************************************************************/
static SNMP_TABLE_ENTRY *get_next_mib_variable (SNMP_TABLE_ENTRY *sptr_snmp_table_entry,
	MIB_ACCESS_PARAMETERS *sptr_mib_access_parameters)
{
	PARAMETER_NOT_USED (sptr_mib_access_parameters);

	++sptr_snmp_table_entry;

	return (sptr_snmp_table_entry);
}
/**********************************************************************************/
static void	adjust_mib_access_parameters (MIB_ACCESS_PARAMETERS *sptr_mib_access_parameters,
	SNMP_TABLE_ENTRY *sptr_snmp_table_entry)
{
	*(sptr_mib_access_parameters->eptr_end_of_table) = sptr_snmp_table_entry->sptr_parameters->is_a_table_entry;

	*(sptr_mib_access_parameters->usptr_size_of_table_indices) = 0x0000;

	++sptr_snmp_table_entry;

	sptr_mib_access_parameters->cptr_mib_string = sptr_snmp_table_entry->mib_name;

	*(sptr_mib_access_parameters->ptr_to_cptr_next_variable_name) = sptr_snmp_table_entry->mib_name;

	*(sptr_mib_access_parameters->usptr_size_of_variable_value) = sptr_snmp_table_entry->size_of_mib_variable;
}
/**********************************************************************************/
static void clear_pointer_to_next_variable_name (MIB_ACCESS_PARAMETERS *sptr_mib_access_parameters)
{
	*(sptr_mib_access_parameters->ptr_to_cptr_next_variable_name) = NULL;

	*(sptr_mib_access_parameters->eptr_end_of_table) = TRUE;
}
/**********************************************************************************/
static enum TEST process_scalar_mib_variable (MIB_ACCESS_PARAMETERS *sptr_mib_access_parameters,
	SNMP_TABLE_ENTRY *sptr_snmp_table_entry)
{
	USHORT arg_not_used;
	/* Srikar, Mar 25, 1997. Added the following local variable to save the return value from the access function. */
	enum TEST return_value;

	arg_not_used = 0x0000;

	*(sptr_mib_access_parameters->usptr_size_of_variable_value) = sptr_snmp_table_entry->size_of_mib_variable;

	/* Srikar, Mar 25, 1997. Save the return value from the access function. */
	return_value = (sptr_snmp_table_entry->fptr_mib_access_function) (arg_not_used, sptr_mib_access_parameters->mib_operation,
		sptr_snmp_table_entry->offset_of_mib_variable,
		sptr_snmp_table_entry->sptr_parameters->address_of_structure_containing_mib_variable,
		(ULONG) arg_not_used,
		sptr_mib_access_parameters->bptr_variable_value, sptr_mib_access_parameters->usptr_size_of_variable_value);

	if (sptr_snmp_table_entry->need_to_swap == TRUE)
		{
		*((ULONG *) sptr_mib_access_parameters->bptr_variable_value) =
			swap_long (*((ULONG *) sptr_mib_access_parameters->bptr_variable_value));
		}

	build_indices_for_scalar_mib_variable (sptr_mib_access_parameters->sptr_table_indices,
		sptr_mib_access_parameters->usptr_size_of_table_indices);

	/* Srikar, Mar 25, 1997. Changed the return value from PASS to return_value. */
	return (return_value);
}
/**********************************************************************************/
/* Srikar, Mar 20, 1997. Changed the variable bptr_table_indices to sptr_table_indices and its type */
/* from BYTE * to OBJECT_ID * */
/* Changed the variable usptr_number_of_bytes_in_table_indices to usptr_number_of_subids_in_table_indices */
static void	build_indices_for_scalar_mib_variable (OBJECT_ID *sptr_table_indices, USHORT *usptr_number_of_subids_in_table_indices)
{
	*sptr_table_indices = 0x00;

	*usptr_number_of_subids_in_table_indices = 0x0001;
}

/* Srikar, Mar 25, 1997. Added the last parameter variable_found to identify if the variable was found or not. */
/* The return value is used to identify if the get/set operation was successful. The last parameter is used to */
/* distinguish the nature of the failure. */
enum TEST
process_tcp_table_mib_variable(MIB_ACCESS_PARAMETERS *sptr_mib_access_parameters,
	SNMP_TABLE_ENTRY *sptr_snmp_table_entry, enum BOOLEAN *variable_found)
{
	void *vptr_row_in_table;

	/* Srikar, Mar 25, 1997. Following initialization added. */
	*variable_found = FALSE;
	if (check_if_tcp_mib_table_exists() == FAIL)
	{
		return FAIL;
	}

	if (find_pointer_to_row_in_tcp_mib_table(sptr_snmp_table_entry, sptr_mib_access_parameters, &vptr_row_in_table) == FAIL)
	{
#if defined(DEBUG)
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: !@#!Bad row indices\n");
#endif /* DEBUG */
		return (FAIL);
	}
	/* Srikar, Mar 25, 1997. Following setting added. */
	*variable_found = TRUE;

	/* Srikar, Mar 25, 1997. Added the check for return value, and return FAIL. */

	if (tcp_apply_mib_operation_on_current_instance_of_mib_variable(vptr_row_in_table, sptr_mib_access_parameters,
		sptr_snmp_table_entry) == FAIL)
		return(FAIL);

	build_tcp_table_indices_for_current_instance_of_mib_variable(vptr_row_in_table,
		sptr_mib_access_parameters->sptr_table_indices,	sptr_mib_access_parameters->usptr_size_of_table_indices);

	return (PASS);
}

static enum TEST
find_pointer_to_row_in_tcp_mib_table(SNMP_TABLE_ENTRY *sptr_snmp_table_entry,
	MIB_ACCESS_PARAMETERS *sptr_mib_access_parameters,	void **ptr_to_vptr_row_in_table)
{
	ULONG tcp_local_addr;
	ULONG tcp_remote_addr;
	ULONG tcp_local_port;
	ULONG tcp_remote_port;
	TCP_PER_CONN *ptr_conn_info;
	enum BOOLEAN received_indices;
	enum BOOLEAN valid_table_indices;


	PARAMETER_NOT_USED (sptr_snmp_table_entry);


#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: Num indices %d\n", *(sptr_mib_access_parameters->usptr_size_of_table_indices));
#endif /* DEBUG */

	valid_table_indices = get_tcp_mib_table_indices(sptr_mib_access_parameters->sptr_table_indices,
		sptr_mib_access_parameters->usptr_size_of_table_indices, &received_indices, 
		&tcp_local_port, &tcp_local_addr, &tcp_remote_port, &tcp_remote_addr);

	if (valid_table_indices == FALSE)
	{
#if defined(DEBUG)
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: !@#!Invalid table indices\n");
#endif /* DEBUG */
		return FAIL;
	}

	*ptr_to_vptr_row_in_table = NULL;
	if (received_indices == FALSE)
	{
#if defined(DEBUG)
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: Rx-ed NO indices\n");
#endif /* DEBUG */
		*ptr_to_vptr_row_in_table = (void *) tcp_calculate_pointer_to_next_row_in_mib_table(NULL);
		if (*ptr_to_vptr_row_in_table == NULL)
		{
#if defined(DEBUG)
			tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: No more ROWS!\n");
#endif /* DEBUG */
			return (FAIL);
		}
		return (PASS);
	}
	else
	{
#if defined(DEBUG)
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: Rx-ed indices\n");
#endif /* DEBUG */
		*ptr_to_vptr_row_in_table = NULL;
		 
		ptr_conn_info = NULL;
		/* Srikar, Mar 17, 1997. Changed the code to test for end condition and for lexicographic
		operation of the get next operator */
		while ((ptr_conn_info = tcp_calculate_pointer_to_next_row_in_mib_table(ptr_conn_info)) != NULL)
		{
			int compare_result;

			compare_result = compare_table_indices_with_an_entry(ptr_conn_info,
											tcp_local_addr, tcp_local_port, tcp_remote_addr, tcp_remote_port);
			if (compare_result == 0)
			{
				*ptr_to_vptr_row_in_table = (void *) ptr_conn_info;
				if (sptr_mib_access_parameters->mib_operation == GET_NEXT_OPERATION)
				{
#if defined(DEBUG)
					tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: Found match on GET_NEXT\n");
#endif /* DEBUG */
					*ptr_to_vptr_row_in_table = (void *) tcp_calculate_pointer_to_next_row_in_mib_table(ptr_conn_info);
					if (*ptr_to_vptr_row_in_table == NULL)
						return FAIL;
				}
				return PASS;
			}
			else if(compare_result > 0)
			{
				 switch(sptr_mib_access_parameters->mib_operation)
				 {
					case GET_OPERATION:
					case SET_OPERATION:
#if defined(DEBUG)
						tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: Found match on GET or SET\n");
#endif /* DEBUG */
						*ptr_to_vptr_row_in_table = NULL;
						return (FAIL);
					case GET_NEXT_OPERATION:
						*ptr_to_vptr_row_in_table = (void *)ptr_conn_info;
						return (PASS);
				 }
			}
		}
	}

	return FAIL;
}

/* Srikar, Mar 17, 1997. Added this function to compare given table indices with an entry */
static int compare_table_indices_with_an_entry(TCP_PER_CONN *ptr_conn_info1, ULONG local_addr, ULONG local_port,
					ULONG remote_addr, ULONG remote_port)
{
	if (ptr_conn_info1->local_addr > local_addr)
		return 1;
	else if (ptr_conn_info1->local_addr < local_addr)
		return -1;
	if (ptr_conn_info1->local_port > local_port)
		return 1;
	else if (ptr_conn_info1->local_port < local_port)
		return -1;
	if (ptr_conn_info1->remote_addr > remote_addr)
		return 1;
	else if (ptr_conn_info1->remote_addr < remote_addr)
		return -1;
	if (ptr_conn_info1->remote_port > remote_port)
		return 1;
	else if (ptr_conn_info1->remote_port < remote_port)
		return -1;
	return 0;
}

enum TEST 
tcp_apply_mib_operation_on_current_instance_of_mib_variable(
	void *vptr_row_in_table, MIB_ACCESS_PARAMETERS *sptr_mib_access_parameters,
	SNMP_TABLE_ENTRY *sptr_snmp_table_entry)
{
	USHORT arg_not_used;
	arg_not_used = 0x0000;

	/* Only the tcpConnState setting is writable. So we make a check for this.*/
	if (sptr_mib_access_parameters->mib_operation == SET_OPERATION)
	{
		if (strcmp(sptr_mib_access_parameters->cptr_mib_string, "tcpConnState") != STRINGS_MATCH)
		{
#if defined(DEBUG)
			tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: Tryin to SET on read-only field\n");
#endif /* DEBUG */
			return FAIL;
		}
	}
	if ((sptr_snmp_table_entry->fptr_mib_access_function) (arg_not_used,
		sptr_mib_access_parameters->mib_operation, sptr_snmp_table_entry->offset_of_mib_variable,
		(ULONG) vptr_row_in_table, (ULONG) arg_not_used, sptr_mib_access_parameters->bptr_variable_value,
		sptr_mib_access_parameters->usptr_size_of_variable_value) == FAIL)
		{
#if defined(DEBUG)
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: Did not find current instance\n");
#endif /* DEBUG */
		return (FAIL);
		}

	*(sptr_mib_access_parameters->usptr_size_of_variable_value) = sptr_snmp_table_entry->size_of_mib_variable;

#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: Current instance found to be sizeof(%d) and of value %d\n", 
				sptr_snmp_table_entry->size_of_mib_variable, *(sptr_mib_access_parameters->bptr_variable_value));
#endif /* DEBUG */

	return (PASS);
}

/* Srikar, Mar 20, 1997. Changed the variable bptr_table_indices to sptr_table_indices and its type */
/* from BYTE * to OBJECT_ID * */
/* Changed the variable usptr_number_of_bytes_in_table_indices to usptr_number_of_subids_in_table_indices */
static enum BOOLEAN
get_tcp_mib_table_indices(OBJECT_ID *sptr_table_indices,
		USHORT *usptr_number_of_subids_in_table_indices,
		enum BOOLEAN *eptr_received_indices, 
		ULONG *tcp_local_port_ptr, ULONG *tcp_local_addr_ptr, 
		ULONG *tcp_remote_port_ptr, ULONG *tcp_remote_addr_ptr)
{
	enum BOOLEAN valid_table_indices;


	valid_table_indices = TRUE;
	*eptr_received_indices = TRUE;

	if (*usptr_number_of_subids_in_table_indices == 0x0000)
	{
#if defined(DEBUG)
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: No indices received\n");
#endif /* DEBUG */
	 	*eptr_received_indices = FALSE;
	}
	/* Srikar, Mar 20, 1997. Changed the code to check if the number of subids recieved is correct */
	/* rather than checking for the number of bytes received */
	else if (*usptr_number_of_subids_in_table_indices == 4 + 1 + 4 + 1)
	{
		USHORT bits = 32;
		
		/* Srikar, Mar 20, 1997. Changed code due to change in object id type */
		*tcp_local_addr_ptr = 0;
		while (bits)
			{
			bits -= 8;
			*tcp_local_addr_ptr |= (0x000000ff & *sptr_table_indices) << bits;
			sptr_table_indices++;
			}
		*tcp_local_port_ptr = (USHORT) *sptr_table_indices;
		sptr_table_indices++;

		bits = 32;
		*tcp_remote_addr_ptr = 0;
		while (bits)
			{
			bits -= 8;
			*tcp_remote_addr_ptr |= (0x000000ff & *sptr_table_indices) << bits;
			sptr_table_indices++;
			}
		*tcp_remote_port_ptr =  (USHORT) *sptr_table_indices;
		sptr_table_indices++;
#if defined(DEBUG)
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: localaddr=%d, localport=%d, remoteaddr=%d, remoteport=%d\n",
				*tcp_local_addr_ptr, *tcp_local_port_ptr, *tcp_remote_addr_ptr, *tcp_remote_port_ptr);
#endif /* DEBUG */
	}
	else
	{
		valid_table_indices = FALSE;
	}

	return (valid_table_indices);
}

/* Srikar, Mar 17, 1997. Changed the function implementation to return the next row in */
/* lexicographic order rather than on logical position in memory */
static TCP_PER_CONN *
tcp_calculate_pointer_to_next_row_in_mib_table(TCP_PER_CONN *ptr_conn_info)
{
	int i;
	TCP_PER_CONN *temp_conn_info_ptr;
	TCP_PER_CONN *next_least_conn_info_ptr;


	temp_conn_info_ptr = tcp.tcp_conn_table;
	next_least_conn_info_ptr = NULL;
	if (ptr_conn_info == NULL)
	{
		for (i = 0; i < tcp.max_tcp_connection; i++)
		{
			if (temp_conn_info_ptr->tcp_state != FREE)
				if (next_least_conn_info_ptr == NULL || compare_conn_table_indices(temp_conn_info_ptr, next_least_conn_info_ptr) < 0)
					next_least_conn_info_ptr = temp_conn_info_ptr;
			temp_conn_info_ptr++;
		}
	}
	else
	{
		for (i = 0; i < tcp.max_tcp_connection; i++)
		{
			if (temp_conn_info_ptr->tcp_state != FREE)
				if(compare_conn_table_indices(temp_conn_info_ptr, ptr_conn_info) > 0 && (next_least_conn_info_ptr == NULL ||
							compare_conn_table_indices(temp_conn_info_ptr, next_least_conn_info_ptr) < 0))
					next_least_conn_info_ptr = temp_conn_info_ptr;
			temp_conn_info_ptr++;
		}
	}
	return next_least_conn_info_ptr;
}

/* Srikar, Mar 17, 1997. Added this function to compare two entries in the tcp connection table */
static int compare_conn_table_indices(TCP_PER_CONN *ptr_conn_info1, TCP_PER_CONN *ptr_conn_info2)
{
 	/* if *ptr_conn_info1 > *ptr_conn_info2 then return +ve, */
	/* else if *ptr_conn_info1 < *ptr_conn_info2 then return -ve */
	/* else return 0(*ptr_conn_info1 == *ptr_conn_info2) */
	if (ptr_conn_info1->local_addr > ptr_conn_info2->local_addr)
		return 1;
	else if (ptr_conn_info1->local_addr < ptr_conn_info2->local_addr)
		return -1;
	if (ptr_conn_info1->local_port > ptr_conn_info2->local_port)
		return 1;
	else if (ptr_conn_info1->local_port < ptr_conn_info2->local_port)
		return -1;
	if (ptr_conn_info1->remote_addr > ptr_conn_info2->remote_addr)
		return 1;
	else if (ptr_conn_info1->remote_addr < ptr_conn_info2->remote_addr)
		return -1;
	if (ptr_conn_info1->remote_port > ptr_conn_info2->remote_port)
		return 1;
	else if (ptr_conn_info1->remote_port < ptr_conn_info2->remote_port)
		return -1;
	return 0;
}

/* Srikar, Mar 20, 1997. Changed the variable bptr_table_indices to sptr_table_indices and its type */
/* from BYTE * to OBJECT_ID * */
/* Changed the variable usptr_number_of_bytes_in_table_indices to usptr_number_of_subids_in_table_indices */
static enum TEST 
build_tcp_table_indices_for_current_instance_of_mib_variable(void *vptr_row_in_table,
	OBJECT_ID *sptr_table_indices, USHORT *usptr_number_of_subids_in_table_indices)
{
	ULONG mask;
	USHORT bits;
	/* Srikar, Mar 20, 1997. Changed the code due to change in the type of object_id from BYTE to OBJECT_ID */
	mask = 0xff000000;
	bits = 24;
	while(mask)
		{
		*sptr_table_indices = (mask & ((TCP_PER_CONN *) vptr_row_in_table)->local_addr) >> bits;
		mask >>= 8;
		bits -= 8;
		sptr_table_indices++;
		}
	*sptr_table_indices = ((TCP_PER_CONN *) vptr_row_in_table)->local_port;
	sptr_table_indices++;

	mask = 0xff000000;
	bits = 24;
	while(mask)
		{
		*sptr_table_indices = (mask & ((TCP_PER_CONN *) vptr_row_in_table)->remote_addr) >> bits;
		mask >>= 8;
		bits -= 8;
		sptr_table_indices++;
		}
	*sptr_table_indices = ((TCP_PER_CONN *) vptr_row_in_table)->remote_port;
	sptr_table_indices++;

	*usptr_number_of_subids_in_table_indices = 4 + 1 + 4 + 1;

#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: build new indices\n");
#endif /* DEBUG */
	return PASS;
}

static enum BOOLEAN 
check_if_tcp_mib_table_exists(void)
{
	return (TRUE);
}

enum TEST
tcp_snmp_get_or_set_conn_state(USHORT port_number, 
	enum MIB_OPERATION mib_operation, ULONG offset, ULONG class, ULONG size,
	void *vptr_mib_value_obtained, USHORT *usptr_length_of_mib_obtained)
{
	ULONG value;
	TCP_PER_CONN *ptr_conn_info;


	value = 0x00000000;

	if (mib_operation == GET_OPERATION || mib_operation == GET_NEXT_OPERATION)
	{
#if defined(DEBUG)
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: Called to GET connstate\n");
#endif /* DEBUG */
		value = *(USHORT *) (class + offset + (size * port_number));
		*(ULONG *) vptr_mib_value_obtained = swap_long (value);
	}
	else if (mib_operation == SET_OPERATION)
	{
#if defined(DEBUG)
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: Called to SET connstate\n");
#endif /* DEBUG */
		value = *(ULONG *) vptr_mib_value_obtained;
		/* Only value allowed is deleteTCB(12). Actually, this check should
		** be done much earlier in the flow of things and a return value of
		** BAD_VALUE should be indicated to the manager. From here, we can
		** only indicate a TOO_BIG error by returning FAIL.
		*/
		if ((USHORT) value != SNMP_DELETE_TCB)
		{
#if defined(DEBUG)
			tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: Bad SET value (%d)\n", value);
#endif /* DEBUG */
			return FAIL;
		}

#if defined(DEBUG)
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: SETting %d\n", value);
#endif /* DEBUG */

		/* If the current state of the entry is LISTEN, we will ignore this
		** request and return PASS. This so that we always can keep listen
		** pending. CLOSED and FREE are already free states. In all other
		** states, we will simply clean up the entry without informing anyone.
		** The SNMP RFC (1213), says that we can send a RST segment on 
		** established connection if we want to. We don't want to for now.
		*/
		ptr_conn_info = (TCP_PER_CONN *) class;
		switch (ptr_conn_info->tcp_state)
		{
		case LISTEN:
#if defined(DEBUG)
			tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: Cannot SET in listen state\n");
#endif /* DEBUG */
			return FAIL;
		case CLOSED:
		case FREE:
			return PASS;
		case SYN_RCVD:
		case ESTAB:
		case CLOSE_WAIT:
		case FIN_WAIT_1:
		case FIN_WAIT_2:
		case CLOSING:
		case LAST_ACK:
		case TIME_WAIT:
			ptr_conn_info->local_addr = 0;
			ptr_conn_info->remote_addr = 0;
			ptr_conn_info->local_port= 0;
			ptr_conn_info->remote_port = 0;
			tcp_quiet_abort(ptr_conn_info);
#if defined(DEBUG)
			tcp_printf(TCP_DEBUG_PRINTF, "TCP: SNMP: SET successful\n");
#endif /* DEBUG */
			break;
		default:
			return FAIL;
		}
	}
	else
		return FAIL;

	return PASS;
}
