/*
** TCPSOCK.C -- Socket library support code for the TCP stack
*/

/* General Notes:
**		All socket calls that potentially can wait will never wait, but will
**			do required processing and return with the "would-block" error.
**			This mechanism is to support RouterWare's style of processing.
**		After a connection is established, any other connection related
**			socket call (send, recv, etc.) will check for TCP related errors
**			and report back the error in the socket error number.
**		Since app on router box will be a server, server behaviour has been
**			checked. Client support is not all that okay. Specially the close.
*/

#include "rtrstd.h"
#include "all.h"
#include "tcpextrn.h"


#define WEB_PORT	80				/* WEB port number */
BYTE_ENUM (TEST) autobindtcp (USER_SOCKET *sptr_user_socket);

USHORT tcp_port_number = 1025;

/*
** socket_tcp()
**		Called in response to a SOCKET call to the sockets API.
*/
BYTE_ENUM (TEST) 
socket_tcp(USER_SOCKET *sptr_user_socket, USHORT protocol)
{
	PARAMETER_NOT_USED (protocol);

	sptr_user_socket->socket_type_of_service = TYPE_TCP;

	return (PASS);
}

/*
** socket_tcp_listen()
**		Socket call to setup for a TCP listen. Allocates one connection
**		record for the purpose and inits the local port and address fields.
** Params:
**		Ptr to a socket record for the socket
**		Mode flags for the listen
** Returns:
**		PASS if all is okay
**		FAIL if all is not okay (in this case, the error code is filled into
**			the socket record's error number field)
*/
BYTE_ENUM (TEST)
socket_tcp_listen(USER_SOCKET *sptr_user_socket, USHORT mode)
{
	TCP_PER_CONN *ptr_conn_info;
	SOCKADDR_IN *ptr_sockaddr_in;


	PARAMETER_NOT_USED(mode);

	ptr_conn_info = 
			(TCP_PER_CONN *) sptr_user_socket->vptr_protocol_control_block;

	if (ptr_conn_info != NULL)
	{
		tcp.socket_fptrs.fptr_set_socket_class_error_number(IS_CONNECTED_ERROR);
		return FAIL;		
	}

	/* Look thro' our list and get a free connection record */
	ptr_conn_info = get_free_tcp_conn_record();
	if (ptr_conn_info == NULL)
	{
		tcp.socket_fptrs.fptr_set_socket_class_error_number(
															TOO_MANY_SOCKETS_OPEN);
		return FAIL;
	}

	ptr_sockaddr_in = (SOCKADDR_IN *)sptr_user_socket->vptr_address;
	
	ptr_conn_info->local_port = ptr_sockaddr_in->sin_port;
	ptr_conn_info->local_addr = ptr_sockaddr_in->sin_addr.s_addr;

	ptr_conn_info->tcp_open = PASSIVE_OPEN;
	tcp_printf(TCP_STATES_PRINTF, "TCP: Moved from FREE to LISTEN state\n", 
			get_state_string(ptr_conn_info->tcp_state));
	ptr_conn_info->tcp_state = LISTEN;

	/* Couple the socket record to the tcp connection record */
	/* so that you can get one from the other if needed */

	ptr_conn_info->socket_record_ptr = sptr_user_socket;
	sptr_user_socket->vptr_protocol_control_block = ptr_conn_info;

	return PASS;
}

/* 
** socket_tcp_connect()
**		Handles socket connect calls. Connect is complicated by checks that
**		have to be made to ensure that parameters are legal. Multiple
**		connections on the same local port should refer to different
**		target sockets, else the connection will be failed.
** Params:
**		Ptr to the socket record for the call
** NOTE:
**		THIS FUNCTION IS NOT IMPLEMENTED AS WE CURRENTLY DO NOT REQUIRE THIS
**		FOR ANY APP
*/
#if 0
BYTE_ENUM (TEST)
socket_tcp_connect(USER_SOCKET *sptr_user_socket)
{
	PARAMETER_NOT_USED(sptr_user_socket);


	tcp.socket_fptrs.fptr_set_socket_class_error_number(
															OPERATION_NOT_SUPPORTED_ERROR);
	return FAIL;
}
#endif

/* sudhir 28/2/97 */

BYTE_ENUM (TEST)
socket_tcp_connect(USER_SOCKET *sptr_user_socket)
{
	TCP_PER_CONN *ptr_conn_info;
	SOCKADDR_IN	*ptr_sockaddr_in;
   TCP_HEADER *tcp_header;
   

	if (sptr_user_socket->vptr_address == NULL)
	{
         
/*
Satish, 17th March 1997
If a explicit bind() is not issued then we assume that the connection
is for a normal tcp port. If you need a reserved tcp port then you
MUST make a bind call.
*/
		if (autobindtcp(sptr_user_socket) == FAIL) 		/* Issue a autobind, if not bound */
		{
	    	tcp.socket_fptrs.fptr_set_socket_class_error_number(ADDRESS_IN_USE_ERROR);
			return(FAIL);
		}
	}

	ptr_conn_info = 
			(TCP_PER_CONN *) sptr_user_socket->vptr_protocol_control_block;

	if (ptr_conn_info != NULL)
	{
		tcp.socket_fptrs.fptr_set_socket_class_error_number(IS_CONNECTED_ERROR);
		return FAIL;		
	}

	/* Look thro' our list and get a free connection record */
	ptr_conn_info = get_free_tcp_conn_record();
	if (ptr_conn_info == NULL)
	{
		tcp.socket_fptrs.fptr_set_socket_class_error_number(
															TOO_MANY_SOCKETS_OPEN);
		return FAIL;
	}
/*
Have a different timeout for connect() call.
This is so that connect will not a loooong time to come out
incase the destination is not found. Let the timer be around
one second to begin with instead of DEF_RTO
Satish. 10th March 1997
*/
	ptr_conn_info->rtx_timeout = DEF_CONNECT_RTO;

	ptr_sockaddr_in = (SOCKADDR_IN *)sptr_user_socket->vptr_address;
	
	ptr_conn_info->local_port = ptr_sockaddr_in->sin_port;
	ptr_conn_info->local_addr = ptr_sockaddr_in->sin_addr.s_addr;

	ptr_conn_info->tcp_open = ACTIVE_OPEN;

	/* Couple the socket record to the tcp connection record */
	/* so that you can get one from the other if needed */

	ptr_conn_info->socket_record_ptr = sptr_user_socket;
	sptr_user_socket->vptr_protocol_control_block = ptr_conn_info;

#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Opening a connection\n");
#endif /* DEBUG */

	ptr_conn_info->init_recv_seq_num = 0;
	ptr_conn_info->recv_nxt = ptr_conn_info->init_recv_seq_num + 1;

	ptr_conn_info->init_send_seq_num = get_current_isn_to_use();
	ptr_conn_info->send_nxt = ptr_conn_info->init_send_seq_num + 1;
	ptr_conn_info->send_una = ptr_conn_info->init_send_seq_num;

	/* Enter remote socket info in connection record */
	ptr_sockaddr_in = (SOCKADDR_IN *)sptr_user_socket->vptr_peer_address;
	
	ptr_conn_info->remote_port = ptr_sockaddr_in->sin_port;
	ptr_conn_info->seg_ip_params.destination_address =
		 ptr_conn_info->remote_addr = ptr_sockaddr_in->sin_addr.s_addr;
					

	if (ptr_conn_info->local_addr == INADDR_ANY) {
		ptr_conn_info->local_addr = ip_get_address_of_outgoing_interface(ptr_conn_info->remote_addr);
	}

	ptr_conn_info->seg_ip_params.source_address = ptr_conn_info->local_addr;

	ptr_conn_info->send_mss = ptr_conn_info->temp_mss;
	ptr_conn_info->send_lwack = ptr_conn_info->init_send_seq_num;

#if defined(DEBUG)
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: Init advert win_size = %d\n",
									tcp_header->win_size);
#endif /* DEBUG */

	/* Change state to SYN_SENT and send a SYN to initiate the handshake */
	ptr_conn_info->tcp_state = SYN_SENT;
	++tcp.mib.tcpActiveOpens;


	send_tcp_control_seg(ptr_conn_info, 
				ptr_conn_info->init_send_seq_num, 
				ptr_conn_info->recv_nxt, SYN );
	add_to_tcp_rtx_queue(ptr_conn_info, 
				ptr_conn_info->init_send_seq_num, 0);

	/* Start a timer on this connection */
	reset_tcp_alive_timer(ptr_conn_info);  

	return (PASS);
}

/*
** socket_tcp_receive()
**		Socket call to receive data from a tcp connection. The socket receive
**		call is supposed to behave like a file-read call returning either
**		the number of bytes requested or a lesser amount (including 0) if
**		EOF is reached (connection is closed). In our case, since we can't
**		block waiting for bytes to be received or the connection to be closed,
**		we will return immediately with a "would-block" error if the requested
**		number of bytes are not there in the receive window and the connection
**		is not yet closed. But in the case of packet reception with PSH bit 
**		set we will fill the receive buffer and return whatever is available 
**		in the receive window. Ofcourse, if the buffer supplied is larger than 
**		the receive window, we will use only the receive window size and 
** 	return that much.
** Params:
**		Ptr to socket record for socket
**		Ptr to buffer to receive the data in
**		Length of the buffer
**		Ptr to a structure to fill in received info (NULL)
**		Length of above structure (NULL)
**		Ptr to a word to know if call succeeded or failed
** Returns:
**		Number of bytes filled in buffer if all is okay
**		0 on EOF or errors
*/
USHORT
socket_tcp_receive(USER_SOCKET *sptr_user_socket, void *vptr_data, 
		USHORT length, SOCKADDR	*sptr_sockaddr, USHORT *usptr_address_length, 
		enum TEST *eptr_error)
{
	USHORT i;
	ULONG win_index;
	BYTE *data_ptr = vptr_data;
	TCP_PER_CONN *ptr_conn_info;


	PARAMETER_NOT_USED(sptr_sockaddr);
	PARAMETER_NOT_USED(usptr_address_length);

	ptr_conn_info = 
			(TCP_PER_CONN *) sptr_user_socket->vptr_protocol_control_block;

	if (ptr_conn_info == NULL)
	{
		tcp.socket_fptrs.fptr_set_socket_class_error_number(
																NOT_CONNECTED_ERROR);
		*eptr_error = FAIL;
		return 0;
	}

	if (ptr_conn_info->conn_error != 0)
	{
		tcp.socket_fptrs.fptr_set_socket_class_error_number(
																ptr_conn_info->conn_error);
		ptr_conn_info->conn_error = 0;
		
		*eptr_error = FAIL;
		return 0;
	}

	/* Connections in some states can receive, others cannot - so make sure */
	switch (ptr_conn_info->tcp_state)
	{
	case CLOSED:
	case LISTEN:
	case SYN_RCVD:
	case SYN_SENT:
	case FIN_WAIT_1:
	case FIN_WAIT_2:
	case CLOSING:
	case TIME_WAIT:
	case LAST_ACK:
		/* Cannot receive in all these states */
		tcp.socket_fptrs.fptr_set_socket_class_error_number(
				NOT_CONNECTED_ERROR);
		*eptr_error = FAIL;
		return 0;
		break;
	case CLOSE_WAIT:
		if (ptr_conn_info->recv_filled == 0)
		{
			/* Allow socket receives in the CLOSE_WAIT state until the rx window
			** is emptied. Then on the next receive call, return the 
			** "end-of-file" error.
			*/
			tcp.socket_fptrs.fptr_set_socket_class_error_number(
																END_OF_FILE_ERROR);
			*eptr_error = FAIL;
			return 0;
   		}
	case ESTAB:
		break;
	}

	if (length > ptr_conn_info->recv_buf_size)
		length = ptr_conn_info->recv_buf_size;	/* will pass only window amt of data */

/*
Satish, March 18th 1997
Always give whatever is present. Do not check for PSH flag.
So the following piece of code is #if 0ed
*/
#if 0
	if ((ptr_conn_info->recv_filled < length) &&
		((ptr_conn_info->conn_flags & TCPF_RECVPSH) == 0))
	{
		/* If insufficient data to return and a "push" was not requested or
		** there is no urgent data, return "would-block" error and 0 bytes.
		*/
		tcp.socket_fptrs.fptr_set_socket_class_error_number(
						WOULD_BLOCK_ERROR);
		*eptr_error = FAIL;
		return 0;
	}
#endif

/*	if (TCPF_RECVPSH & ptr_conn_info->conn_flags)
	{
		if (length >= ptr_conn_info->recv_filled)
		{
			length = ptr_conn_info->recv_filled;
			ptr_conn_info->conn_flags &= ~TCPF_RECVPSH;
		}
	} */

   if (length >= ptr_conn_info->recv_filled)
	{
	    	length = ptr_conn_info->recv_filled;
		ptr_conn_info->conn_flags &= ~TCPF_RECVPSH;	
	}


	/* Sufficient data -- copy the data to user buffer and update receive
	** window globals
	*/
	win_index = ptr_conn_info->recv_first;
	for (i = 0; i < length; i++)
	{
		data_ptr[i] = (ptr_conn_info->recv_buf)[win_index];
		win_index = (win_index + 1) % ptr_conn_info->recv_buf_size;
	}

	/* update globals */
	ptr_conn_info->recv_first = (ptr_conn_info->recv_first + length) % 
											ptr_conn_info->recv_buf_size;
	ptr_conn_info->recv_filled -= length;

	/* Removing data from the receive window would allow more data to
	** be received. i.e. the receive window size should be increased. So
	** cause such an advertisement if okay.
	*/
	if (ptr_conn_info->recv_last_advert == 0 && 
			tcp_recv_wnd_ver2(ptr_conn_info))
	{
		send_tcp_control_seg(ptr_conn_info, ptr_conn_info->send_nxt,
				ptr_conn_info->recv_nxt, ACK);
	}

	*eptr_error = PASS;

   /* sudhir 14/5/97 */
/* Changed by Ravi & Naveen on 12 July 1999 */
   if (!length)
	{
		*eptr_error = FAIL;
      tcp.socket_fptrs.fptr_set_socket_class_error_number (WOULD_BLOCK_ERROR);  
	}

	return length;
}

/*
** socket_tcp_send()
**		Called to perfrom a socket style tcp send operation. As per the 
**		socket rules, send should not return until all is done. But this
**		may cause us to block if the send window has been filled and not yet
**		ack'ed. So we copy as much as is possible and return with a "would-
**		block" error. Note that send actually copies data from the passed
**		buffer to the send window buffer. When TCP actually sends is TCP's
**		problem.
** Params:
**		Ptr to the socket info record
**		Ptr to the buffer to send
**		Length of buffer to send
**		Ptr to socket address information (destination peer address)
**		Ptr to word to return error code in
** NOTE:
**		There is some comment in the socklib sources telling we need to free
**			the passed buffer (which is crazy) after use. We will not do so here.
**			Also the socklib sources take care of verifying the destination 
**			(peer) address.
**		If the passed buffer is larger than the available window buffer space, 
**			we copy only unto the space available and return the number of
**			byte copied.
*/
USHORT
socket_tcp_send(USER_SOCKET *sptr_user_socket, void *vptr_data, 
		USHORT length, SOCKADDR	*sptr_sockaddr, enum TEST *eptr_error)
{
	ULONG win_index;
	USHORT i, avail_space;
	BYTE *data_ptr = vptr_data;
	TCP_PER_CONN *ptr_conn_info;
	enum BOOLEAN should_push = FALSE;


	PARAMETER_NOT_USED(sptr_sockaddr);

#if defined(DEBUG)

		tcp_printf(TCP_DATA_PRINTF, "TCP: sock_send, length %d(%Xh)\n", length, length);
/*
		tcp_printf(TCP_DEBUG_PRINTF, "TCP:");
		for (i = 0; i < length && i < 32 ; i++)
			tcp_printf(TCP_DEBUG_PRINTF, " %02x", *((BYTE *)vptr_data + i));
		tcp_printf(TCP_DEBUG_PRINTF, "\n");
*/

#endif /* DEBUG */

	ptr_conn_info = 
			(TCP_PER_CONN *) sptr_user_socket->vptr_protocol_control_block;

	if (ptr_conn_info == NULL)
	{
		tcp.socket_fptrs.fptr_set_socket_class_error_number(
																NOT_CONNECTED_ERROR);
		*eptr_error = FAIL;
/*
Satish - 8th March 1997
I realise this is a stupid fix. Change the socket library to leave the buffer stuff to the applications.
Because of the shortage of time, this fix:
Check if the port is WEB port. If so dont try and free the buffer. Leave it to the WEB engine
to take care of all these things.
The main problem being the WEB module does not always send a malloc'ed buffer to send. It allocates
one huge buffer and then keeps writing and incrementing write pointers depending on the amount of
data actually written. If a send with a intermediate buffer fails our module tries to free a
stupid pointer and goes for a biiiiig toss.
*/
		if (ptr_conn_info->local_port != WEB_PORT)
   		table_free(vptr_data);
		return 0;
	}

	if (ptr_conn_info->conn_error != 0)
	{
		tcp.socket_fptrs.fptr_set_socket_class_error_number(
																ptr_conn_info->conn_error);
		ptr_conn_info->conn_error = 0;

		*eptr_error = FAIL;

/*
Satish - 8th March 1997
I realise this is a stupid fix. Change the socket library to leave the buffer stuff to the applications.
Because of the shortage of time, this fix:
Check if the port is WEB port. If so dont try and free the buffer. Leave it to the WEB engine
to take care of all these things.
The main problem being the WEB module does not always send a malloc'ed buffer to send. It allocates
one huge buffer and then keeps writing and incrementing write pointers depending on the amount of
data actually written. If a send with a intermediate buffer fails our module tries to free a
stupid pointer and goes for a biiiiig toss.
*/
		if (ptr_conn_info->local_port != WEB_PORT)
   		table_free(vptr_data);
		return 0;
	}

	/* Connections in some states can send , others cannot - so make sure */
	switch (ptr_conn_info->tcp_state)
	{
	case CLOSED:
	case LISTEN:
	case SYN_RCVD:
	case SYN_SENT:
	case FIN_WAIT_1:
	case FIN_WAIT_2:
	case CLOSING:
	case TIME_WAIT:
	case LAST_ACK:
		/* Cannot send in all these states */
		tcp.socket_fptrs.fptr_set_socket_class_error_number(
																NOT_CONNECTED_ERROR);
		*eptr_error = FAIL;

/*
Satish - 8th March 1997
I realise this is a stupid fix. Change the socket library to leave the buffer stuff to the applications.
Because of the shortage of time, this fix:
Check if the port is WEB port. If so dont try and free the buffer. Leave it to the WEB engine
to take care of all these things.
The main problem being the WEB module does not always send a malloc'ed buffer to send. It allocates
one huge buffer and then keeps writing and incrementing write pointers depending on the amount of
data actually written. If a send with a intermediate buffer fails our module tries to free a
stupid pointer and goes for a biiiiig toss.
*/
		if (ptr_conn_info->local_port != WEB_PORT)
   		table_free(vptr_data);
		return 0;
		break;
	case CLOSE_WAIT:
	case ESTAB:
		break;
	}

	avail_space = ptr_conn_info->send_buf_size - (ptr_conn_info->send_filled + 
			ptr_conn_info->send_una_amount);

	if (avail_space == 0)
	{
		/* Try in the future */
		*eptr_error = PASS;
		return 0;
	}

	/* Fit to available buffer space */
	if (length > avail_space)
	{
		length = avail_space;
	}
	else
	{
		/* If send() could be done in one call, push all data to be sent */
		should_push = TRUE;
	}

	win_index = (ptr_conn_info->send_bnext + ptr_conn_info->send_filled) % 
						ptr_conn_info->send_buf_size;
	for (i = 0; i < length; i++)
	{
		ptr_conn_info->send_buf[win_index] = data_ptr[i];
		win_index = (win_index + 1) % ptr_conn_info->send_buf_size;
	}

	/* Update globals */
	if (should_push == TRUE)
	{							  	
		ptr_conn_info->conn_flags |= TCPF_SENDPSH;/* PUSH data out immediately */
	}
	ptr_conn_info->send_filled += length;

	/* Do some persist work if required */
	if (ptr_conn_info->send_wnd == 0 && ptr_conn_info->rtx_queue == NULL &&
			ptr_conn_info->send_persist_timer == 0)
	{
		tcp_persist(ptr_conn_info);
	}

	/* Try sending immediately if possible -- timer will backup */
	tcp_send_fgnd();

	*eptr_error = PASS;
	return length;
}

/* 
** socket_tcp_close()
**		Socket call to tcp to terminate one end of a tcp connection.
** Params:
**		Ptr to the socket information record
**	NOTE:
**		At the router end, close will be called only by a server.
**		This function is a little cunning to accomodate the way "closesocket" 
**			is encoded in RouterWare socket library interface. There, buffers
**			related to the socket will NOT be released if a successful return
**			(PASS) is not indicated here. So irrespective of errors, we will
**			always say PASS.
**		CLOSE will cause the TCP connection to go into the LAST_ACK or the
**			FIN_WAIT_1 states which will need the connection record to stick
**			around for some time. So actual close will be sometime in the
**			future. Timer code will take care of this.
*/
BYTE_ENUM (TEST)
socket_tcp_close(USER_SOCKET *sptr_user_socket)
{
	TCP_PER_CONN *ptr_conn_info;


	ptr_conn_info = 
			(TCP_PER_CONN *) sptr_user_socket->vptr_protocol_control_block;
	
	if (ptr_conn_info == NULL)
	{
		return PASS;
	}

	/* Connections in different states close differently */
	switch (ptr_conn_info->tcp_state)
	{
	/*--------------------------------------------------------------------*/
	case FIN_WAIT_1:
	case FIN_WAIT_2:
	case CLOSING:
	case TIME_WAIT:
		/* CLOSE call will not come in these states */
		return PASS;
		break;
	/*--------------------------------------------------------------------*/
	case LAST_ACK:
		return PASS;
		break;
	/*--------------------------------------------------------------------*/
	case SYN_SENT:
	case CLOSED:
	case LISTEN:
		/* De-couple the socket record from the tcp connection record */
		ptr_conn_info->socket_record_ptr = NULL;
		sptr_user_socket->vptr_protocol_control_block = NULL;

		free_tcp_conn_record(ptr_conn_info);

		return PASS;
		break;
	/*--------------------------------------------------------------------*/
	case SYN_RCVD:
	case ESTAB:
	case CLOSE_WAIT:
		/* All sends should be pushed out. Also a FIN needs to be pushed
		** out and an ACK got in return. All that will take time and the
		** socket will eventually close itself. So app can get out, but
		** the connection record will not be released until later.
		*/
		ptr_conn_info->conn_flags |= TCPF_SENDFIN;
		ptr_conn_info->conn_flags |= TCPF_SENDPSH;

		/* De-couple the socket record from the tcp connection record 
		** so that app can use the socket record again if it so wishes.
		*/
		ptr_conn_info->socket_record_ptr = NULL;
		sptr_user_socket->vptr_protocol_control_block = NULL;

		return PASS;
		break;
	}
	return PASS;
}

/*
** socket_tcp_status()
**		NOT SUPPORTED
*/
BYTE_ENUM (TEST)
socket_tcp_status(USER_SOCKET *sptr_user_socket)
{
	PARAMETER_NOT_USED(sptr_user_socket);

	tcp.socket_fptrs.fptr_set_socket_class_error_number(
														OPERATION_NOT_SUPPORTED_ERROR);
	return FAIL;
}

/*
** socket_tcp_get_state_string()
**		Returns a pointer to a string appropriate for current state of
**		the TCP connection.
*/
char *
socket_tcp_get_state_string(USER_SOCKET *sptr_user_socket)
{
	TCP_PER_CONN *ptr_conn_info;


	ptr_conn_info = 
			(TCP_PER_CONN *) sptr_user_socket->vptr_protocol_control_block;

	if (ptr_conn_info == NULL)
	{
		tcp.socket_fptrs.fptr_set_socket_class_error_number(
												NOT_CONNECTED_ERROR);
		return NULL;
	}

	return get_state_string(ptr_conn_info->tcp_state);
}

/*
** tcp_check_ip_address()
**		Called by common socklib code to check on the protocol address
**		format.
** Params:
**		Ptr to SOCKADDR (socket) format address of peer address
** 	Length of the SOCKADDR struct
*/
BYTE_ENUM (TEST)
tcp_check_ip_address(SOCKADDR *sptr_sockaddr, USHORT sockaddr_length)
{
	SOCKADDR_IN *sptr_sockaddr_in;

	sptr_sockaddr_in = (SOCKADDR_IN *) sptr_sockaddr;

	if ((sptr_sockaddr_in->sin_family != AF_INET) || (sockaddr_length < sizeof (SOCKADDR_IN)))
		{
		return (FAIL);
		}

	return (PASS);
}

/*
** socket_tcp_peek()
** This call is similar to that of socket_tcp_receive(). Only diference here
** is the data is not removed from TCP buffer. 
** Params:
**		Ptr to socket record for socket
**		Ptr to buffer to receive the data in
**		Length of the buffer
**		Ptr to a word to know if call succeeded or failed
** Returns:
**		Number of bytes filled in buffer if all is okay
**		0 on EOF or errors
*/
USHORT
socket_tcp_peek(USER_SOCKET *sptr_user_socket, void *vptr_data, 
		USHORT length, enum TEST *eptr_error)
{
	USHORT i;
	ULONG win_index;
	BYTE *data_ptr = vptr_data;
	TCP_PER_CONN *ptr_conn_info;

	ptr_conn_info = 
			(TCP_PER_CONN *) sptr_user_socket->vptr_protocol_control_block;

	if (ptr_conn_info == NULL)
	{
		tcp.socket_fptrs.fptr_set_socket_class_error_number(
																NOT_CONNECTED_ERROR);
		*eptr_error = FAIL;
		return 0;
	}

	if (ptr_conn_info->conn_error != 0)
	{
		tcp.socket_fptrs.fptr_set_socket_class_error_number(
																ptr_conn_info->conn_error);
		ptr_conn_info->conn_error = 0;
		
		*eptr_error = FAIL;
		return 0;
	}

	/* Connections in some states can receive, others cannot - so make sure */
	switch (ptr_conn_info->tcp_state)
	{
	case CLOSED:
	case LISTEN:
	case SYN_RCVD:
	case SYN_SENT:
	case FIN_WAIT_1:
	case FIN_WAIT_2:
	case CLOSING:
	case TIME_WAIT:
	case LAST_ACK:
		/* Cannot receive in all these states */
		tcp.socket_fptrs.fptr_set_socket_class_error_number(
				NOT_CONNECTED_ERROR);
		*eptr_error = FAIL;
		return 0;
		break;
	case CLOSE_WAIT:
		if (ptr_conn_info->recv_filled == 0)
		{
			/* Allow socket receives in the CLOSE_WAIT state until the rx window
			** is emptied. Then on the next receive call, return the 
			** "end-of-file" error.
			*/
			tcp.socket_fptrs.fptr_set_socket_class_error_number(
																END_OF_FILE_ERROR);
			*eptr_error = FAIL;
			return 0;
		}
	case ESTAB:
		break;
	}

	if (length > ptr_conn_info->recv_buf_size)
		length = ptr_conn_info->recv_buf_size;	/* will pass only window amt of data */

	if ((ptr_conn_info->recv_filled < length) &&
		((ptr_conn_info->conn_flags & TCPF_RECVPSH) == 0))
	{
		/* If insufficient data to return and a "push" was not requested or
		** there is no urgent data, return "would-block" error and 0 bytes.
		*/
		tcp.socket_fptrs.fptr_set_socket_class_error_number(
						WOULD_BLOCK_ERROR);
		*eptr_error = FAIL;
		return 0;
	}

	if (TCPF_RECVPSH & ptr_conn_info->conn_flags)
	{
		if (length >= ptr_conn_info->recv_filled)
		{
			length = ptr_conn_info->recv_filled;
			ptr_conn_info->conn_flags &= ~TCPF_RECVPSH;
		}
	}

	/* Sufficient data -- copy the data to user buffer and update receive
	** window globals
	*/
	win_index = ptr_conn_info->recv_first;
	for (i = 0; i < length; i++)
	{
		data_ptr[i] = (ptr_conn_info->recv_buf)[win_index];
		win_index = (win_index + 1) % ptr_conn_info->recv_buf_size;
	}

	*eptr_error = PASS;
	return length;
}
/*
** socket_tcp_buffer_length()
** Params:
**		Ptr to socket record for socket
** Returns:
**		Number of bytes in buffer
**		0 on or errors
*/
USHORT
socket_tcp_buffer_length(USER_SOCKET *sptr_user_socket,enum TEST *eptr_error)
{
	TCP_PER_CONN *ptr_conn_info;

	ptr_conn_info = 
			(TCP_PER_CONN *) sptr_user_socket->vptr_protocol_control_block;

	if (ptr_conn_info == NULL)
	{
		tcp.socket_fptrs.fptr_set_socket_class_error_number(
																NOT_CONNECTED_ERROR);
		*eptr_error = FAIL;
		return 0;
	}

	if (ptr_conn_info->conn_error != 0)
	{
		tcp.socket_fptrs.fptr_set_socket_class_error_number(
																ptr_conn_info->conn_error);
		ptr_conn_info->conn_error = 0;
		
		*eptr_error = FAIL;
		return 0;
	}
	*eptr_error = PASS;
	return ( ptr_conn_info->recv_filled);	
}

/*
 * Issue an automatic bind of a local address
 */
BYTE_ENUM (TEST)	autobindtcp (USER_SOCKET *sptr_user_socket)
{
	SOCKADDR_IN sptr_local_sockaddr_in;
	ULONG socket_descriptor;

	socket_descriptor = sptr_user_socket->socket_descriptor;

	sptr_local_sockaddr_in.sin_family = AF_INET;
	sptr_local_sockaddr_in.sin_addr.s_addr = INADDR_ANY;

   if ((sptr_local_sockaddr_in.sin_port = get_tcp_port_number(NORMAL_PORT)) == 0)
		return(FAIL);
	bind((int) socket_descriptor, (SOCKADDR *) &sptr_local_sockaddr_in, sizeof (SOCKADDR_IN));
	return(PASS);

}

/* sudhir 26/6/97 
	is_socket_closed is called from telnet dialout.c 
	*/

int is_socket_closed (ULONG socket_descriptor)
{
	USER_SOCKET *sptr_user_socket;
	TCP_PER_CONN *ptr_conn_info;

	sptr_user_socket = (USER_SOCKET *) socket_descriptor;

	ptr_conn_info = (TCP_PER_CONN *) sptr_user_socket->vptr_protocol_control_block;

	if (ptr_conn_info == NULL)
		return (1) ;
	if (ptr_conn_info->conn_error != 0)
		return (1) ;

	switch (ptr_conn_info->tcp_state)
	{
	case CLOSED:
	case LISTEN:
	case SYN_RCVD:
	case SYN_SENT:
	case FIN_WAIT_1:
	case FIN_WAIT_2:
	case CLOSING:
	case TIME_WAIT:
	case LAST_ACK:
	case CLOSE_WAIT:
		return 1 ;

	default :
		return 0 ;
	}
}
