/* NOTE: During the program running, we will be toggling the fail LED on and
**			off to indicate activity. Turn on by writing 0xFE to 0xFF000010 and
**			turn off by writing 0xFF to same address.
** Revisions:
**		17, April, '96 - Sanjay - Replaced call WriteToFlash() with
**							HigherWriteToFlash() to accomodate new kind of flash.
**						   Also introduced a call to DetermineFlashTypes().
*/
#include "types.h"
#include "..\include\boot.h"
#include "bootchg.h"
#include "flutil.h"

/* Local prototypes */
void HigherWriteToFlash(byte *, byte *, int);
int GetNewBootSizeAndOffset(int *, byte **);
void CopyPC_SP(byte *);
int BurnCode(byte *, byte *, dword);
void CorruptCodeHeader(void);
void ResetRouter(void);
void GetBootHeader(BootConfigType *);
void SetBootHeader(BootConfigType *);

/* Globals */
byte LEDOnOffValue;
HeaderType HeaderPkt ;
BootConfigType OldBootHeader, NewBootHeader ;

/* Following globals will be used in determination of flash types */
unsigned short Fl1ProId ;
unsigned short Fl2ProId ;
unsigned long FlProId; /* brindha 17/8/99 from jeyasudha's changes */


/* Steps -
** 1. Read NEWBOOT size from header locations
** 2. Get and save old boot header (to retain password and ethernet address)
** 3. Cause a copy of the NEWBOOT to flash area where original BOOT is 
**		present (absolute 0).
** 4. Redo the boot header with the old password and ethernet address
** 5. Corrupt code header
**	6. Cause a reset of the box
*/

void main()
{
	int i ;
	int NewBootSize ;
	byte *sptr, *dptr ;
	byte *NewBootAddr ;
	byte *BootAbsAddr ;

	DetermineFlashTypes() ;

	LEDOnOffValue = 0xFE ;			/* Turn on value */
	*(dword *) 0xFF000010 = LEDOnOffValue ;
	LEDOnOffValue = ~LEDOnOffValue | 0xFE ;

	if (GetNewBootSizeAndOffset (&NewBootSize, &NewBootAddr) > 0)
	{
		GetBootHeader (&OldBootHeader);
		BootAbsAddr = (byte *) FL_ROMM_START ;
		BurnCode (BootAbsAddr, NewBootAddr, NewBootSize) ;
	}

	GetBootHeader (&NewBootHeader) ;
	NewBootHeader.EthernetAddr_H = OldBootHeader.EthernetAddr_H ;
	NewBootHeader.EthernetAddr_L = OldBootHeader.EthernetAddr_L;

/* brindha on 16/8/99 from Jeyasudha's code for Big Proxy. */
/* All hardware related info need to be copied from old boot header */

	NewBootHeader.NumberOfPorts = OldBootHeader.NumberOfPorts;

	NewBootHeader.Base1 = OldBootHeader.Base1;
	NewBootHeader.Size1 = OldBootHeader.Size1;
	NewBootHeader.Base2 = OldBootHeader.Base2;
	NewBootHeader.Size2 = OldBootHeader.Size2;

	NewBootHeader.ClockSpeed = OldBootHeader.ClockSpeed;
	NewBootHeader.PROMWidth = OldBootHeader.PROMWidth;
	NewBootHeader.ModelNumber = OldBootHeader.ModelNumber;

	NewBootHeader.FlashPromSize = OldBootHeader.FlashPromSize;
	NewBootHeader.RamSize = OldBootHeader.RamSize;
	NewBootHeader.RamStart = OldBootHeader.RamStart;
/* brindha on 16/8/99. */

	dptr = (byte *) NewBootHeader.PassWord ;
	sptr = (byte *) OldBootHeader.PassWord ;
	for (i = 0 ; i < sizeof (NewBootHeader.PassWord) ; i ++, sptr ++, dptr ++)
	{
		*dptr = *sptr ;
	}
	SetBootHeader (&NewBootHeader) ;

	LEDOnOffValue = 0xFF ;			/* Turn off value */
	*(dword *) 0xFF000010 = LEDOnOffValue ;

	CorruptCodeHeader() ;
	ResetRouter() ;
}

/* GetNewBootSizeAndOffset()
**		From the header to this file, read the boot size and location into
**		the passed variables.
*/
int GetNewBootSizeAndOffset (int *PtrNewBootSize, byte **PtrNewBootAddr)
{
	BOOT_CHG_HDR *BootChgCodeHeaderAddr ;

	/* CODE_START_ADDRESS is actually the header location */
	BootChgCodeHeaderAddr = (BOOT_CHG_HDR *) CODE_START_ADDRESS ;

	*PtrNewBootSize = BootChgCodeHeaderAddr->NewBootSize ;
	*PtrNewBootAddr = (unsigned long) BootChgCodeHeaderAddr->NewBootStartAddr +
									(unsigned long) CODE_START_ADDRESS ;

	if (*PtrNewBootSize == 0 || *PtrNewBootAddr == 0)
		return 0 ;
	
	return 1 ;
}

/* BurnCode() : (From Vidy's code for boot)
** 	Copy 'len' number of bytes from flash or dram to flash taking care of
**		flash hassles.
*/
int BurnCode (byte *to, byte *from, dword len)
{
	dword	Offset ;
	word NumOfPages, i ;

	if ((len % FL_PAGE_SIZE))
		NumOfPages = (word) (len / FL_PAGE_SIZE) + 1 ;
	else
		NumOfPages = (word) (len / FL_PAGE_SIZE) ;

	if (NumOfPages) 
	{
		for (i = 0 ; i < (NumOfPages - 1) ; i ++) 
		{
			if (i % 10 == 0)
			{
				/* Toggle the FAIL LED */
				*(dword *) 0xFF000010 = LEDOnOffValue ;
				LEDOnOffValue = ~LEDOnOffValue | 0xFE ;
			}
			Offset = i * FL_PAGE_SIZE ;
			HigherWriteToFlash ((char *) (from + Offset), (char *) (to + Offset), FL_PAGE_SIZE) ;
		}

		/* Last page may not be full */
		Offset = i * FL_PAGE_SIZE ;
		HigherWriteToFlash ((char *) (from + Offset), (char *) (to + Offset), len - Offset) ;
		return 1 ;
	}
	return 0 ;
}

/* CorruptCodeHeader()
**		Write junk into the existing code header in flash so that the new
**		boot that is downloaded will ask for new BIN for Router to be 
**		downloaded when the box is reset.
*/
void CorruptCodeHeader()
{
	int sSize ;
	byte *pByte ;

	pByte = (byte *) &HeaderPkt ;
	sSize = sizeof (HeaderPkt) ;
	while (sSize)
	{
		*pByte = 0;
		pByte ++ ;
		sSize -- ;
	}
	HeaderPkt.MagicNum = NEW_BOOT_MAGIC ;
	BurnCode ((byte *) FL_CODE_HDR, (byte *) &HeaderPkt, sizeof (HeaderPkt)) ;
}

/* ResetRouter()
**		Arrange to cause the router box to re-boot.
*/
void ResetRouter()
{
	Reset() ;
}

/* GetBootHeader()
**		Returns the contents of the boot description record in the flash
*/
void GetBootHeader (BootConfigType *BootRecordPtr)
{
	int i ;
	byte *sptr = (byte *) FL_BOOT_HDR ;
	byte *dptr = (byte *) BootRecordPtr ;


	for (i = 0 ; i < sizeof (BootConfigType) ; i ++, sptr ++, dptr ++)
	{
		*dptr = *sptr ; 	
	}
}

/* SetBootHeader()
**		Writes the contents of the boot description record in the flash
*/
void SetBootHeader (BootConfigType *BootRecordPtr)
{
	BurnCode ((byte *) FL_BOOT_HDR, (byte *) BootRecordPtr, sizeof (BootConfigType)) ;
}


/* HigherWriteToFlash()
**		Function to be called to write a page (<= 512 bytes) to flash PROM.
*/
void HigherWriteToFlash (byte *src, byte *dst, int num)
{
	unsigned long ToFlash ;
	unsigned short FlProdId ;

	if (num > 512 )
		return ;   /* writes lesser than 512 bytes only */

	ToFlash = (unsigned long) dst ;
	ToFlash &= 0x80000 ;

	if (ToFlash)      
		FlProdId = Fl2ProId ;  /* it is go to the second flash */
	else
		FlProdId = Fl1ProId ;

	if (FlProdId == 0x1FA4)
	{    /* AT29C040A  256 bytes write */
		if (num > 256)
		{
			WriteToFlash (src, dst, 256) ;
			WriteToFlash (src + 256, dst + 256, num - 256) ;
		} 
		else
			WriteToFlash (src, dst, num) ;
	}
	else   									 /* AT29C040  512 bytes write */
		WriteToFlash (src, dst, num) ;
}


