// -------------------------------------------------------------------------
// Project Name                 :       MultiRouter Setup for Windows 
// File Name                    :       COM.CPP
// Description                  :       Defines the CMiscCom Class 
// Start Date                   :       8th May 1995
// Author                   	  :       Pravin
// Date Last Modified   		  :       
// Modifications                :       
// -------------------------------------------------------------------------

// -------------------------------------------------------------------------
// Include Files
// -------------------------------------------------------------------------

#include "stdafx.h"
#include "procon.h"
#include "com.h"
#include "msgumain.h"
#include "msguview.h"
#include "dnldprog.h"
#include "cnfwrite.h"
#include "boot.h"

#include "lzvc.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

// -------------------------------------------------------------------------
// Macros
// -------------------------------------------------------------------------
         
// -------------------------------------------------------------------------
// Global Variable Declarations
// -------------------------------------------------------------------------

int idComDev;		//Shared for all instances of CMiscCom.
int CTRL_FLAG;		//Shared for all instances of CMiscCom and used in Maindlg.cpp

int	ExpectingReplyPacket = FALSE;
int nacks=0;


static unsigned int crctab[] = 
{ 0x0000,  0x1189,  0x2312,  0x329b, 0x4624, 0x57ad, 0x6536, 0x74bf, 
0x8c48, 0x9dc1, 0xaf5a, 0xbed3, 0xca6c, 0xdbe5, 0xe97e, 0xf8f7, 
0x1081, 0x0108, 0x3393, 0x221a, 0x56a5, 0x472c, 0x75b7, 0x643e, 
0x9cc9, 0x8d40, 0xbfdb, 0xae52, 0xdaed, 0xcb64, 0xf9ff, 0xe876, 
0x2102, 0x308b, 0x0210, 0x1399, 0x6726, 0x76af, 0x4434, 0x55bd, 
0xad4a, 0xbcc3, 0x8e58, 0x9fd1, 0xeb6e, 0xfae7, 0xc87c, 0xd9f5, 
0x3183, 0x200a, 0x1291, 0x0318, 0x77a7, 0x662e, 0x54b5, 0x453c, 
0xbdcb, 0xac42, 0x9ed9, 0x8f50, 0xfbef, 0xea66, 0xd8fd, 0xc974, 
0x4204, 0x538d, 0x6116, 0x709f, 0x0420, 0x15a9, 0x2732, 0x36bb, 
0xce4c, 0xdfc5, 0xed5e, 0xfcd7, 0x8868, 0x99e1, 0xab7a, 0xbaf3, 
0x5285, 0x430c, 0x7197, 0x601e, 0x14a1, 0x0528, 0x37b3, 0x263a, 
0xdecd, 0xcf44, 0xfddf, 0xec56, 0x98e9, 0x8960, 0xbbfb, 0xaa72, 
0x6306, 0x728f, 0x4014, 0x519d, 0x2522, 0x34ab, 0x0630, 0x17b9, 
0xef4e, 0xfec7, 0xcc5c, 0xddd5, 0xa96a, 0xb8e3, 0x8a78, 0x9bf1, 
0x7387, 0x620e, 0x5095, 0x411c, 0x35a3, 0x242a, 0x16b1, 0x0738, 
0xffcf, 0xee46, 0xdcdd, 0xcd54, 0xb9eb, 0xa862, 0x9af9, 0x8b70, 
0x8408, 0x9581, 0xa71a, 0xb693, 0xc22c, 0xd3a5, 0xe13e, 0xf0b7, 
0x0840, 0x19c9, 0x2b52, 0x3adb, 0x4e64, 0x5fed, 0x6d76, 0x7cff, 
0x9489, 0x8500, 0xb79b, 0xa612, 0xd2ad, 0xc324, 0xf1bf, 0xe036, 
0x18c1, 0x0948, 0x3bd3, 0x2a5a, 0x5ee5, 0x4f6c, 0x7df7, 0x6c7e, 
0xa50a, 0xb483, 0x8618, 0x9791, 0xe32e, 0xf2a7, 0xc03c, 0xd1b5, 
0x2942, 0x38cb, 0x0a50, 0x1bd9, 0x6f66, 0x7eef, 0x4c74, 0x5dfd, 
0xb58b, 0xa402, 0x9699, 0x8710, 0xf3af, 0xe226, 0xd0bd, 0xc134, 
0x39c3, 0x284a, 0x1ad1, 0x0b58, 0x7fe7, 0x6e6e, 0x5cf5, 0x4d7c, 
0xc60c, 0xd785, 0xe51e, 0xf497, 0x8028, 0x91a1, 0xa33a, 0xb2b3, 
0x4a44, 0x5bcd, 0x6956, 0x78df, 0x0c60, 0x1de9, 0x2f72, 0x3efb, 
0xd68d, 0xc704, 0xf59f, 0xe416, 0x90a9, 0x8120, 0xb3bb, 0xa232, 
0x5ac5, 0x4b4c, 0x79d7, 0x685e, 0x1ce1, 0x0d68, 0x3ff3, 0x2e7a, 
0xe70e, 0xf687, 0xc41c, 0xd595, 0xa12a, 0xb0a3, 0x8238, 0x93b1, 
0x6b46, 0x7acf, 0x4854, 0x59dd, 0x2d62, 0x3ceb, 0x0e70, 0x1ff9, 
0xf78f, 0xe606, 0xd49d, 0xc514, 0xb1ab, 0xa022, 0x92b9, 0x8330, 
0x7bc7, 0x6a4e, 0x58d5, 0x495c, 0x3de3, 0x2c6a, 0x1ef1, 0x0f78}; 

/////////////////////////////////////////////////////////////////////////////
// CMiscCom

CMiscCom::CMiscCom()
{
	Pkt = new BYTE[1024 * 2];  //To hold packet of maximum size 1024
	if(Pkt == NULL)
		AfxMessageBox(MSG_COM_NO_MEM, MB_OK | MB_ICONEXCLAMATION);
    
    lpStat = new COMSTAT;
    if(lpStat == NULL)
    	AfxMessageBox(MSG_COM_NO_MEM, MB_OK | MB_ICONEXCLAMATION);

   DateStamp = new char[25];

	TimeOut = FALSE ;
	UDBActualLen = 0 ;
}

CMiscCom::~CMiscCom()
{                       
// Deallocate the memory allocated in the constructor.
	delete [] Pkt;
	delete lpStat;
   delete [] DateStamp;
}

/////////////////////////////////////////////////////////////////////////////
//  CMiscCom functions

// -------------------------------------------------------------------------
// Function				:	CMiscCom::SetDateStamp()
// Arguments         :	None.
// Synopsis          :	Sets the current date for Date stamping.
// Returns           :	None.
// Globals Affected  :	DateStamp[].
// -------------------------------------------------------------------------
void CMiscCom::SetDateStamp()
{
   char szBuf[5] ;
   time_t curtime ;
   time (&curtime) ;
   CTime ttime (curtime) ;
   int tempint = ttime.GetMonth() ;

	// Convert the month from integer to string format.
   switch (tempint)
   {
      case 1 :
         strcpy (DateStamp, "Jan.") ;
         break ;

      case 2 :
         strcpy (DateStamp, "Feb.") ;
         break ;

      case 3 :
         strcpy (DateStamp, "March") ;
         break ;
			 
      case 4 :
         strcpy (DateStamp, "April") ;
         break ;

      case 5 :
         strcpy (DateStamp, "May") ;
         break ;    

      case 6 :
         strcpy (DateStamp, "June") ;
         break ;

      case 7 :
         strcpy (DateStamp, "July") ;
         break ;

      case 8 :
         strcpy (DateStamp, "August") ;
         break ;

      case 9 :
         strcpy (DateStamp, "Sept.") ;
         break ;

      case 10 :
         strcpy (DateStamp, "Oct.") ;
         break ;

      case 11 :
         strcpy (DateStamp, "Nov.") ;
         break ;

      case 12 :
         strcpy (DateStamp, "Dec.") ;
         break ;

      default :
         break ;
   }

   strcat (DateStamp, " ") ;
   tempint = ttime.GetDay() ;
   _itoa (tempint, szBuf, 10) ;
   strcat (DateStamp, szBuf) ;
   strcat (DateStamp, ", ") ;
   tempint = ttime.GetYear() ;
   _itoa (tempint, szBuf, 10) ;
   strcat (DateStamp, szBuf) ;
}

// -------------------------------------------------------------------------
// Function				:	CMiscCom::updatecrc
// Arguments         :	BYTE
// Synopsis          :	updates the global CODE_CRC for the value of BYTE.
// Returns           :	None.
// Globals Affected  :	CODE_CRC.
// -------------------------------------------------------------------------
void CMiscCom::updatecrc(BYTE ch)
{
        CODE_CRC = updcrc (ch, CODE_CRC) ;
}


void CMiscCom::FlushReadBuf()
{
	FlushComm (idComDev, 1) ;    // Clear the Rx queue
}

// -------------------------------------------------------------------------
// Function				:	CMiscCom::Open_Comm()
// Arguments         :	None.
// Synopsis          :	Opens COM port and initializes as specified in the 
//								OwnIniFile.
// Returns           :	TRUE if successful.
// Globals Affected  :	idComDev.
// -------------------------------------------------------------------------
BOOL CMiscCom::Open_Comm()
{
	DCB dcb ;            
	char szBuf[20] ;
	
	GetPrivateProfileString (LPortSectionHeader, (LPCSTR) "Select Port",
				(LPCSTR) "COM1", (LPSTR) szBuf, sizeof (szBuf),
					(LPCSTR) OwnIniFile) ;
#if 1
	idComDev = OpenComm (szBuf, 4096, 2048) ;
#else
	idComDev = OpenComm (szBuf, 1024, 2048) ;
#endif
	if(idComDev < 0)
	{
		AfxMessageBox (MSG_COM_ERR_OPEN, MB_OK | MB_ICONEXCLAMATION) ;
		return FALSE ;
	}	                     

   char TempBuf[8] ;
	GetPrivateProfileString (LPortSectionHeader, (LPCSTR) "Baud Rate",
				(LPCSTR) "19200", (LPSTR) TempBuf, sizeof (TempBuf),
					(LPCSTR) OwnIniFile) ;

   if (!strcmp (TempBuf, "57600"))
      strcat (szBuf, ":19200,n,8,1") ;		//BuildCommDCB does not take 57600.
   else
   {
      strcat (szBuf, ":") ;
      strcat (szBuf, TempBuf) ;
      strcat (szBuf, ",n,8,1") ;
   }
	int err = BuildCommDCB (szBuf, &dcb) ;
	if (err < 0)
	{
		AfxMessageBox (MSG_COM_ERR_DCB, MB_OK | MB_ICONEXCLAMATION) ;
		CloseComm (idComDev) ;
		return FALSE ;
	}
   if (!strcmp (TempBuf, "57600"))
   	dcb.BaudRate = 57600 ;
	err = SetCommState (&dcb) ;
	if (err < 0)
	{
		AfxMessageBox (MSG_COM_ERR_INIT, MB_OK | MB_ICONEXCLAMATION) ;
		CloseComm (idComDev) ;
		return FALSE ;
	}
	FlushComm (idComDev, 0) ; //Clear the Tx queue
	FlushComm (idComDev, 1) ; //Clear the Rx queue
	return TRUE ;			
}

void CMiscCom::Close_Comm()
{
	CloseComm (idComDev) ;			// Close the COM port.
	idComDev = -1 ;
}


// -------------------------------------------------------------------------
// Function				:	CMiscCom::little_endian
// Arguments         :	num (DWORD).
// Synopsis          :	Convert a DWORD from 86 to 68 format and vice-versa.
// Returns           :	Result of the conversion.
// Globals Affected  :	None.
// -------------------------------------------------------------------------
DWORD CMiscCom::little_endian (DWORD num)
{
	DWORD num2 ;
	BYTE *str ;
	BYTE *str2 ;
	str = (BYTE *) &num ;
	str2 = (BYTE *) &num2 ;
	if ((int) sizeof (num) == (int) 4)
	{
		str2[3] = str[0] ;
		str2[2] = str[1] ;
		str2[1] = str[2] ;
		str2[0] = str[3] ;
	}
	else
	{ 
		str2[0] = str[1] ;
		str2[1] = str[0] ;
	}
	return num2 ;
}


// -------------------------------------------------------------------------
// Function				:	CMiscCom::ltl_endian_word
// Arguments         :	num (WORD).
// Synopsis          :	Convert a WORD from 86 to 68 format and vice-versa.
// Returns           :	Result of the conversion.
// Globals Affected  :	None.
// -------------------------------------------------------------------------
WORD CMiscCom::ltl_endian_word (WORD num)
{
	WORD num2 ;
	BYTE *str2, *str1 ;

	str1 = (BYTE *) &num ;
	str2 = (BYTE *) &num2 ;
	str2[0] = str1[1] ;
	str2[1] = str1[0] ;
	return num2 ;
}


// -------------------------------------------------------------------------
// Function				:	CMiscCom::snd_char
// Arguments         :	ch (char).
// Synopsis          :	Send the character.
// Returns           :	None.
// Globals Affected  :	None.
// -------------------------------------------------------------------------
void CMiscCom::snd_char (char ch)
{
	if (idComDev < 0)
		return ;
	GetCommError (idComDev, lpStat) ;
  	while (lpStat == NULL)
  		GetCommError (idComDev, lpStat) ;
	while ((lpStat->cbOutQue) != 0) 
	{
		GetCommError (idComDev, lpStat) ;
		while (lpStat == NULL)
      GetCommError (idComDev, lpStat) ;
	}
  	int err = WriteComm (idComDev, &ch, 1) ;
  	if (abs (err) != 1)	
  		AfxMessageBox (MSG_COM_ERR_WRITE, MB_OK | MB_ICONEXCLAMATION) ;
}	// End of snd_char


// -------------------------------------------------------------------------
// Function				:	CMiscCom::fill
// Arguments         :	ch (BYTE).
// Synopsis          :	Quote the special character. And fill in the packet.
// Returns           :	None.
// Globals Affected  :	Pkt[] , Pindex.
// -------------------------------------------------------------------------
void CMiscCom::fill (BYTE ch)
{
	switch ((BYTE) ch)
	{ 
		case ((BYTE) FRAME_DELIM) : /*-----ch='~'-----*/
			Pkt[Pindex ++] = (BYTE) 0x7d ;
			Pkt[Pindex ++] = (BYTE) 0x5e ;
			break ;

		case ((BYTE) 0x7d) : 
			Pkt[Pindex ++] = (BYTE) 0x7d ;
			Pkt[Pindex ++] = (BYTE) 0x5d ;
			break ;

		case ((BYTE) 0x13) : 
			Pkt[Pindex ++] = (BYTE) 0x7d ;
			Pkt[Pindex ++] = (BYTE) 0x33 ;
			break ;

		case ((BYTE) 0x11) : 
			Pkt[Pindex ++] = (BYTE) 0x7d ;
			Pkt[Pindex ++] = (BYTE) 0x31 ;
			break ;

		case ((BYTE) 0x93) : 
			Pkt[Pindex ++] = (BYTE) 0x7d ;
			Pkt[Pindex ++] = (BYTE) (0x93 ^ 0x20) ;
			break ;

		case ((BYTE) 0x91) : 
			Pkt[Pindex ++] = (BYTE) 0x7d ;
			Pkt[Pindex ++] = (BYTE) (0x91 ^ 0x20) ;
			break ;

		default :
			Pkt[Pindex ++] = (BYTE) ch ;
			break ;
	}
}
// -------------------------------------------------------------------------
// Function				:	CMiscCom::snd_packet
// Arguments         :	packet to be sent, its size.
// Synopsis          :	Sends a packet with BridgeID.
// Returns           :	None.
// Globals Affected  :	Pkt[] , Pindex.
// -------------------------------------------------------------------------
void CMiscCom::snd_packet (BYTE *pkt_data, int pkt_size)
{
   BYTE BridgeID[] = "CFP" ;
   Pindex = 0 ;
	WORD crc = 0xFFFF ;
   Pkt[Pindex ++] = (BYTE) FRAME_DELIM ;
   for (int j = 0 ; j < 4 ; j ++)
   {
      fill (BridgeID[j]) ;
	  	crc = updcrc (BridgeID[j], crc) ;
   }                               
	
   j = 0 ;
   while (j < pkt_size)
   {
      fill (pkt_data[j]) ;
      crc = updcrc (pkt_data[j], crc) ;
      j ++ ;
   }
   crc = updcrc (0, updcrc (0, crc)) ;
   fill ((BYTE) ((crc >> 8) & 0xff)) ; /*---send Ist BYTE of crc----*/
   fill ((BYTE) (crc & 0xff)) ;        /*---send IInd BYTE of crc---*/  
   Pkt[Pindex ++] = (BYTE) FRAME_DELIM ;	   /*---send quote character----*/
  		
  	GetCommError (idComDev, lpStat) ;
  	while (lpStat == NULL)
  		GetCommError (idComDev, lpStat) ;
	while ((lpStat->cbOutQue) != 0) 
	{
		GetCommError (idComDev, lpStat) ;
		while (lpStat == NULL)
	      GetCommError (idComDev, lpStat) ;
	}

  	int err = WriteComm (idComDev, Pkt, Pindex) ;
  	if (abs (err) != Pindex)	
  		AfxMessageBox (MSG_COM_ERR_WRITE, MB_OK | MB_ICONEXCLAMATION) ;
} 

// -------------------------------------------------------------------------
// Function				:	CMiscCom::snd_dnld_packet
// Arguments         :	packet to be sent, its size.
// Synopsis          :	Sends a packet without BridgeID.
// Returns           :	None.
// Globals Affected  :	Pkt[] , Pindex.
// -------------------------------------------------------------------------
void CMiscCom::snd_dnld_packet (BYTE *pkt_data, int pkt_size)
{
   Pindex = 0 ;
   int crc = 0xFFFF ;
   Pkt[Pindex ++] = (BYTE) FRAME_DELIM ;  /* Start of Packet */
	  
   int j = 0 ;
   while (j < pkt_size)
   {
      fill (pkt_data[j]) ;
      crc = updcrc (pkt_data[j], crc) ;
      j ++ ;
   }
   crc = updcrc (0, updcrc (0, crc)) ;
   fill ((BYTE) ((crc >> 8) & 0xFF)) ; /*---send Ist BYTE of crc----*/
   fill ((BYTE) (crc & 0xFF)) ;        /*---send IInd BYTE of crc---*/  
   Pkt[Pindex ++] = (BYTE) FRAME_DELIM ; 	   /*---End of Packet----*/
  		
  	GetCommError (idComDev, lpStat) ;
  	while (lpStat == NULL)
  		GetCommError (idComDev, lpStat) ;
	while ((lpStat->cbOutQue) != 0) 
	{
		GetCommError (idComDev, lpStat) ;
		while (lpStat == NULL)
	      GetCommError (idComDev, lpStat) ;
	}
  	int err = WriteComm (idComDev, Pkt, Pindex) ;
  	if (abs (err) != Pindex)	
  		AfxMessageBox (MSG_COM_ERR_WRITE, MB_OK | MB_ICONEXCLAMATION) ;
} 

// -------------------------------------------------------------------------
// Function				:	CMiscCom::snd_abort_pkt
// Arguments         :	None.
// Synopsis          :	Sends an abort packet when user aborts downloading Code.
// Returns           :	None.
// Globals Affected  :	None.
// -------------------------------------------------------------------------
void CMiscCom::snd_abort_pkt()
{
	DnLdPktHdrType abort_pkt ;
	abort_pkt.PacketType = PT_ABORT ;			// 3
	abort_pkt.SeqNum = (BYTE) SEQ ;
	snd_dnld_packet ((BYTE *) &abort_pkt, sizeof (abort_pkt)) ;
}

BOOL CMiscCom::IsDCDPresent()
{
	LPINT	lpMSR ;
	
	if (idComDev < 0)
		return FALSE ;

	lpMSR = (LPINT) SetCommEventMask (idComDev, EV_RLSD | EV_DSR | EV_CTS);
	lpMSR = (LPINT) ((LPBYTE) lpMSR + 35) ;
	return (*lpMSR & DCD_8250) ;
}

// -------------------------------------------------------------------------
// Function				:	CMiscCom::GetRS232()
// Arguments         :	None.
// Synopsis          :	Gets RS232 signals.
// Returns           :	BYTE containing information regarding RS232 signals.
// Globals Affected  :	None.
// -------------------------------------------------------------------------
BYTE CMiscCom::GetRS232()
{
	LPINT	lpMSR ;
	
	if (idComDev < 0)
		return 0 ;

	lpMSR = (LPINT) SetCommEventMask (idComDev, EV_RLSD | EV_DSR | EV_CTS) ;
	lpMSR = (LPINT) ((LPBYTE) lpMSR + 35) ;
	return (*lpMSR) ;
}

// -------------------------------------------------------------------------
// Function				:	CMiscCom::rcv_char
// Arguments         :	pointer to a BOOL.
// Synopsis          :	Gets a char from COM port if its there before the 
//								BOOL becomes TRUE.
// Returns           :	character read from the COM port.
// Globals Affected  :	None.
// -------------------------------------------------------------------------

// added if (Timer) check to take care of avoiding loops in
// rcv_char call from timer
//
char CMiscCom::rcv_char (BOOL *Timer)
{     
	DCB FAR *dcb = new __far DCB ;

	if (idComDev < 0)
		return FALSE ;

   if (GetCommState (idComDev, dcb))
      AfxMessageBox (MSG_COM_ERR_READ_STAT, MB_OK) ;
  	GetCommError (idComDev, lpStat) ;
	if (Timer)
	{
	  	while ((lpStat == NULL) && (! *Timer))
  			GetCommError (idComDev, lpStat) ;
	}
	else
	{
	  	while (lpStat == NULL)
  			GetCommError (idComDev, lpStat) ;
	}

	while ((lpStat->cbInQue == 0) && (! *Timer))
	{               
		MSG msg ;
		while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
		{
			if (msg.message == WM_QUIT)
			{
				*Timer = 1 ;
				PostQuitMessage (0) ;
				return NULL ;
			}
			TranslateMessage (&msg) ;
			DispatchMessage (&msg) ;
		}
		GetCommError (idComDev, lpStat) ;
		while ((lpStat == NULL) && (! *Timer))
			GetCommError (idComDev, lpStat) ;
	}

	char c ;
	if (lpStat->cbInQue)
		ReadComm (idComDev, &c, 1) ;
	delete dcb ;
	return c ;
}

// -------------------------------------------------------------------------
// Function				:	CMiscCom::rcv_char
// Arguments         :	pointer to a BOOL.
// Synopsis          :	Gets a char from COM port if its there before the 
//								BOOL becomes TRUE.
// Returns           :	character read from the COM port.
// Globals Affected  :	None.
// -------------------------------------------------------------------------
BOOL CMiscCom::timer_rcv_char (char FAR *ch)
{     
                                
	DCB FAR *dcb = new __far DCB ;

	if (idComDev < 0)
		return FALSE ;

   if (GetCommState (idComDev, dcb))
      AfxMessageBox (MSG_COM_ERR_READ_STAT, MB_OK) ;

  	GetCommError (idComDev, lpStat) ;

	if (lpStat->cbInQue)
	{
		ReadComm (idComDev, ch, 1) ;
		delete dcb ;
		return TRUE ;
	}
	else 
	{
		delete dcb ;
		return FALSE ;
	}
}

// -------------------------------------------------------------------------
// Function				:	CMiscCom::rcv_packet
// Arguments         :	packet to be received , its size.
// Synopsis          :	Receives a packet with BridgeID.
// Returns           :	CRC value of the packet / zero if no packet.
// Globals Affected  :	None.
// -------------------------------------------------------------------------
int CMiscCom::rcv_packet (BYTE *pkt_data, int pkt_size, BOOL *Timer)
{
   int k = 0 ;
	WORD crc = 0xFFFF ;
	char ch ;
	ExpectingReplyPacket = TRUE ;

	while ((CTRL_FLAG != 1) && (! *Timer)) 
	{
		MSG msg ;
		while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
		{
			if (msg.message == WM_QUIT) 
			{
				*Timer = 1 ;
				PostQuitMessage (0) ;
				return NULL ;
			}
			TranslateMessage (&msg) ;
			DispatchMessage (&msg) ;
		}
	}

	if (! *Timer)
	{
		while (k < 4)
		{  /*----receive BridgeId-----*/
			ch = rcv_char (Timer) ;
			if (ch == (BYTE) 0x7d)
			{
			 	ch = rcv_char (Timer) ;
		   	ch = (BYTE) ((int) ch ^ 0x20) ;  /*---unquote ch----*/
			}
			crc = updcrc (ch, crc) ;
			k ++ ;
		}

		k = 0 ;
		while (k < pkt_size) 
		{ /*----receive the data packet-----*/
			ch = rcv_char (Timer) ; 
			/* Start Frame delimiter is detected in the timer routine */
			/* so this is the End Frame delimiter */
			if (ch == (BYTE) FRAME_DELIM)
				break ; 
			if ((ch == (BYTE) RES_TYPE) && (k == 0)) 
			{
				/*---result type packet-----*/
				pkt_size = 4 ;
			}
			/*--------crc=updcrc(ch, crc);---------*/
			if (ch == (BYTE) 0x7d)
			{
				ch = rcv_char (Timer) ;
				pkt_data[k] = (BYTE) ((int) ch ^ 0x20) ;  /*---unquote ch----*/
				/*----crc=updcrc(pkt_data[i], crc);----*/
			}
			else
				pkt_data[k] = ch ;
			crc = updcrc (pkt_data[k], crc) ;
			k ++ ;
		}

		ch = (BYTE) 0 ;
		if (k == pkt_size)
			ch = rcv_char (Timer) ;  
		while ((ch != (BYTE) FRAME_DELIM) && !*Timer)
		{  /*--while received character is not 7e--*/
			if (ch == (BYTE) 0x7d)
			{
				ch = rcv_char (Timer) ;
				ch = (BYTE) ((int) ch ^ 0x20) ;  /*---unquote ch----*/
			}
			crc = updcrc (ch, crc) ;
			ch = rcv_char (Timer) ;
		}  

		CTRL_FLAG = 0 ;
		ExpectingReplyPacket = FALSE ;
		return crc ;
	}
	ExpectingReplyPacket = FALSE ;
	CTRL_FLAG = 0 ;
	return 0 ;			
}

// -------------------------------------------------------------------------
// Function				:	CMiscCom::rcv_dnld_packet
// Arguments         :	packet to be received , its size.
// Synopsis          :	Receives a packet without BridgeID.
// Returns           :	CRC value of the packet / zero if no packet.
// Globals Affected  :	None.
// -------------------------------------------------------------------------
int CMiscCom::rcv_dnld_packet (BYTE *pkt_data, int pkt_size, BOOL *Timer)
{
   int k = 0 ;
   int crc = 0xffff ;
   char ch ;

	ExpectingReplyPacket = TRUE ;

	while ((CTRL_FLAG != 1) && (! *Timer))
	{
		MSG msg ;
		while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE)) 
		{
			if (msg.message == WM_QUIT) 
			{
				*Timer = 1 ;
				PostQuitMessage (0) ;
				return NULL ;
			}
			TranslateMessage (&msg) ;
			DispatchMessage (&msg) ;
		}
	}
	if (! *Timer)
	{
	   k = 0 ;
	   while (k < pkt_size) /*----receive the data packet-----*/
	   { 
			ch = rcv_char (Timer) ; 
			if (ch == (BYTE) FRAME_DELIM)
				break ; 
			if (ch == (BYTE) 12 && k == 0)
			{
				/*---result type packet-----*/
				pkt_size = 4 ;
			}
			/*--------crc=updcrc(ch, crc);---------*/
			if (ch == (BYTE) 0x7d)
			{
				ch = rcv_char (Timer) ;
				pkt_data[k] = (BYTE) ((int) ch ^ 0x20) ;  /*---unquote ch----*/
				crc = updcrc (pkt_data[k], crc) ;
			}
			else
			{
				pkt_data[k] = ch ;
				crc = updcrc (pkt_data[k], crc) ;
			}
			k ++ ;
		}
		ch = (BYTE) 0 ;
		if (k == pkt_size)
			ch = rcv_char (Timer) ;  
		while (ch != (BYTE) FRAME_DELIM)
		{ /*--while received character is not 7e--*/
		
			if (ch == (BYTE) 0x7d)
			{
				ch = rcv_char (Timer) ;
				ch = (BYTE) ((int) ch ^ 0x20) ;  /*---unquote ch----*/
			}
			crc = updcrc (ch, crc) ;
			ch = rcv_char (Timer) ;
		}  
		CTRL_FLAG = 0 ;
		ExpectingReplyPacket = FALSE ;
		return crc ;
	}
	CTRL_FLAG = 0 ;
	ExpectingReplyPacket = FALSE ;
	return 0 ;
}


// -------------------------------------------------------------------------
// Function				:	CMiscCom::cnf_dnld
// Arguments         :	pointer to BOOL , File to download.
// Synopsis          :	Updates the configuration download progress, calls
//							:	send_data with the packet to be downloaded.
// Returns           :	None.
// Globals Affected  :	None.
// -------------------------------------------------------------------------
void CMiscCom::cnf_dnld (BOOL *Timer, CString FileName, BOOL EnableCtrl)
{
	CStdioFile input ;
   if (!input.Open (FileName, CFile::modeRead | CFile::typeText))
   {
     AfxMessageBox (MSG_COM_ERR_FILE_OPEN, MB_OK | MB_ICONINFORMATION) ;
      return ;
   }

   DWORD size = input.GetLength() ;
   ComprCodeSize = size ;

	AfxGetApp()->LoadCursor (IDC_WAIT) ;
	AfxGetApp()->DoWaitCursor (1) ;

   CCNFWrite *p_cWriteCNF = new (CCNFWrite) ;
   CHorzGauge *wndHorzGauge = new (CHorzGauge) ;

   char m_szTempBuf[10] ;
   wsprintf ((LPSTR) m_szTempBuf, (LPSTR) "%-9ld%s", size, "") ;
   p_cWriteCNF->m_Length = m_szTempBuf ;
   p_cWriteCNF->m_Version = "v0.99" ;

	// Get the current date.
   p_cWriteCNF->m_Date = DateStamp ; 
   p_cWriteCNF->UpdateData (FALSE) ;
                
   CRect rc ;
	(p_cWriteCNF->GetDlgItem (IDC_STATIC_RECT))->GetWindowRect (&rc) ;
	p_cWriteCNF->ScreenToClient (&rc) ;
   wndHorzGauge->Create ((CWnd *) p_cWriteCNF, rc) ;
   wndHorzGauge->SetGaugeInfo ("", 0, 100, GU_ELEVATED, GU_DEPRESSED,
                     TRUE, FALSE, FALSE, RGB (0xff, 0xff, 0xff), 
                        RGB (0x80,0x80, 0x80), RGB (0, 0, 0xff), 
                            RGB (0xc0, 0xc0, 0xc0), RGB (0, 0, 0)) ;
   wndHorzGauge->UpdateProgress (0) ;

	char sBuf[1025] ;
	UINT nReadCount ;
	SEQ = 1 ;
	nReadCount = input.Read (sBuf, 1024) ;
	while ((nReadCount != 0) && !ABORT)
	{
      DWORD pos = input.GetPosition() ;
		if (size)
	      wndHorzGauge->UpdateProgress ((pos * 100) / size) ;
      for (UINT k = 0 ; k < nReadCount ; k ++)
         updatecrc (sBuf[k]) ;
		if (send_data ((BYTE *) sBuf, nReadCount, Timer, EnableCtrl) < 0) 
         break ;
		nReadCount = input.Read (sBuf , 1024) ;
	}

   input.Close() ;     					// Vidy added
   wndHorzGauge->DestroyGauge() ;
   delete wndHorzGauge ;	   	
   delete p_cWriteCNF ;
	AfxGetApp()->DoWaitCursor (0) ;
}


// -------------------------------------------------------------------------
// Function				:	CMiscCom::send_data
// Arguments         :	packet to be sent, its size and pointer to BOOL.
// Synopsis          :	Sends a packet with its sequence num.
// Returns           :	0 if successful, else -1.
// Globals Affected  :	None.
// -------------------------------------------------------------------------
int CMiscCom::send_data (BYTE *str, UINT count, BOOL *Timer, BOOL EnableCtrl)
{
	struct 
	{	  
	   DnLdPktHdrType hdr ;
   	BYTE data[1024] ;
	} dnld_dt ;
	DnLdPktHdrType hdr_type ;
	
	UINT l = 0 ;
	while (l < count)
	{
		dnld_dt.data[l] = str[l] ;
		l ++ ;
	}	
	dnld_dt.hdr.PacketType = PT_DATA ;		// 2
	dnld_dt.hdr.SeqNum = (BYTE) SEQ ;
   
	hdr_type.PacketType = PT_NACK ;		// 5

	while ((hdr_type.PacketType != PT_ACK) && (! *Timer))		// 6
	{	
		MSG msg ;
		while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
		{
			if (msg.message == WM_QUIT) 
			{
				*Timer = 1 ;
				PostQuitMessage (0) ;
				return NULL ;
			}
			TranslateMessage (&msg);
			DispatchMessage (&msg);
		}

		if (EnableCtrl)
		{
			FlushComm (idComDev, 1) ; // Clear the Rx queue
			snd_dnld_packet ((BYTE *) &dnld_dt,
							sizeof (DnLdPktHdrType) + count) ;
			if (rcv_dnld_packet2 ((BYTE *) &hdr_type, sizeof (hdr_type), Timer))
			{
				continue ;
			}
		}
		else
		{
			snd_dnld_packet ((BYTE *) &dnld_dt,
						sizeof (DnLdPktHdrType) + count) ;
			if (rcv_dnld_packet ((BYTE *) &hdr_type, sizeof (hdr_type), Timer))
			{
				continue ;
			}
		}
		if (! *Timer)
		{
			switch (hdr_type.PacketType)
			{
				case PT_ABORT :			// 3
					AfxMessageBox (MSG_COM_ROUTER_ABORT, MB_OK) ;
					ABORT = TRUE ;	
	            if (CTRL_FLAG == 1)   
						CTRL_FLAG = 0 ;	
					return -1 ;	
	            break;   
						
				case PT_NACK : 		// 5
					break ;

				default :             											
					break ;	
			}
		}
		else
		{
         AfxMessageBox (MSG_COM_NO_ROUTER_RESPONSE, MB_OK) ;
         ABORT = TRUE ;
         return -1 ;
      }
	}	
	SEQ ++ ;
   SEQ = SEQ % 128 ;
   return 0 ;
}

// -------------------------------------------------------------------------
// Function				:	CMiscCom::snd_eof_packet
// Arguments         :	pointer to a BOOL, EnableCtrl Flag.
// Synopsis          :	Sends an eof packet.
// Returns           :	None.
// Globals Affected  :	Pkt[] , Pindex.
// -------------------------------------------------------------------------
void CMiscCom::snd_eof_packet (BOOL *Timer, BOOL EnableCtrl)
{
   typedef struct
	{
   	DWORD	CodeLength ;  		/* Valid length of compressed code */
   	WORD CRC ;					/* 16 bit CRC value */
      BYTE Compression ;
   	BYTE Reserved[21] ;		/* for future use */
   } DnLdEOFType ;

   typedef struct
   {
   	DnLdPktHdrType hdr ;			 
      DnLdEOFType eoftype ;
   } dnld_eof_type ;
   
   dnld_eof_type dnld_eof ;
   DnLdPktHdrType hdr_type ;

   dnld_eof.hdr.PacketType = PT_EOF ;		// 4	init eof packet
   dnld_eof.hdr.SeqNum = (BYTE) SEQ ;
   updatecrc ((BYTE) 0) ;
   updatecrc ((BYTE) 0) ;
   dnld_eof.eoftype.CRC = ltl_endian_word (CODE_CRC) ;
   dnld_eof.eoftype.CodeLength = little_endian ((DWORD) ComprCodeSize) ;
   dnld_eof.eoftype.Compression = (BYTE) 1 ;
   snd_dnld_packet ((BYTE *) &dnld_eof, sizeof (dnld_eof)) ;
	if (EnableCtrl) 
	   rcv_dnld_packet2 ((BYTE *) &hdr_type, sizeof (hdr_type), Timer) ;
	else
	   rcv_dnld_packet ((BYTE *) &hdr_type, sizeof (hdr_type), Timer) ;
}


// -------------------------------------------------------------------------
// Function				:	CMiscCom::LZWcompress
// Arguments         :	pointer to a BOOL, Code File name.
// Synopsis          :	Downloads the specified code file after compression,
//								updates the progress.
// Returns           :	Number of bytes sent out.
// Globals Affected  :	
// -------------------------------------------------------------------------
//
// Though called LZWcompress() for compatibility with old code, this will
// introduce a new compression method -- LZW with variable number of bits
//
int CMiscCom::LZWcompress (BOOL *Timer, CString CodeFileName, BOOL EnableCtrl, BOOL Compress)
{
	char m_szTempBuf[10] ;
	int NameLength, Location ;

	BOOL bIsUDB = FALSE ;
	if (CodeFileName.CompareNoCase (UDBUnCompressedDBaseFile) == 0)
		bIsUDB = TRUE ;

	// Check and open file, etc...
	CStdioFile input ;

	if (!input.Open (CodeFileName, CFile::modeRead | CFile::typeBinary))
	{
      AfxMessageBox (MSG_COM_ERR_FILE_OPEN, MB_OK | MB_ICONINFORMATION) ;
      return 0;
	}
	DWORD InSize = input.GetLength() ;

	CDnldProg *p_cDnldProg = new (CDnldProg) ;

	if (bIsUDB)
	{
		p_cDnldProg->m_AbortButton.EnableWindow (FALSE) ;
		p_cDnldProg->m_AbortButton.ShowWindow (SW_HIDE) ;
		p_cDnldProg->SetWindowText ("Downloading User Data Base") ;
	}
	
	wsprintf ((LPSTR) m_szTempBuf, (LPSTR) "%-9ld%s", InSize, "") ;
	p_cDnldProg->m_Length = m_szTempBuf ;
	

	if (!bIsUDB)
	{
		NameLength = CodeFileName.GetLength() ;
		Location = CodeFileName.ReverseFind ('\\') ;
		p_cDnldProg->m_FileName = CodeFileName.Right (NameLength - Location - 1) ;
	}
	
	p_cDnldProg->UpdateData (FALSE) ;

  	CRect rc ;
	CHorzGauge *wndHorzGauge = new (CHorzGauge) ;        
	(p_cDnldProg->GetDlgItem (IDC_STATIC_RECT))->GetWindowRect (&rc) ;
	p_cDnldProg->ScreenToClient (&rc) ;

	wndHorzGauge->Create ((CWnd *) p_cDnldProg, rc) ;
	wndHorzGauge->SetGaugeInfo ("",  0, 100, GU_ELEVATED, GU_DEPRESSED,
    	TRUE, FALSE, FALSE, RGB (0xff, 0xff, 0xff), 
	    	RGB (0x80,0x80, 0x80), RGB (0, 0, 0xff), 
				RGB (0xc0, 0xc0, 0xc0), RGB (0, 0, 0)) ;
	wndHorzGauge->UpdateProgress (0) ;


	// While not EOF, read 8K blocks of memory from file to compress and
	// pass them to the compression routine. When the compression routine
	// returns, send the compressed bytes out in 1024 byte chunks (if the
	// last chunk is not multiple of 1024, it is not sent and is kept for the
	// next cycle). On each 1024 send, the progress display is updated.
	// Special handling is done to detect end of file.

	DWORD cTotalCompressed = 0 ;
	BOOL bFileEnd = FALSE ;
	const int iReadBufSize = 8 * 1024 ;
	BYTE *abyReadBuf = new BYTE[iReadBufSize] ;		// 8K file read buffer
	const int iCompBufSize = 10 * 1024 ;
	BYTE *abyCompBuf = new BYTE[iCompBufSize] ;		// 10K compress buffer
											  //  (larger than read buf just in case)
	BYTE *pbySendBuf ;

	int iReadAmt ;
	int iCompSize ;										// Compressed size
	int iSizeToSend ;									// Temporaries

	if (abyReadBuf == NULL || abyCompBuf == 0)
	{
      AfxMessageBox (MSG_COM_NO_MEM, MB_OK | MB_ICONINFORMATION) ;
      return 0 ;
	}

	bFileEnd = FALSE ;
	iSizePartOfPrev = 0 ;
	ComprCodeSize = 0 ;
	CODE_CRC = 0xFFFF ;
	SEQ = 1 ;

	while (bFileEnd == FALSE)
	{
		if (p_cDnldProg->Abort)		// User Abort!!!
		{
			snd_abort_pkt() ;
			ABORT = ABORT || p_cDnldProg->Abort ;
			break ;
		}
	   ABORT = ABORT || p_cDnldProg->Abort ;
		TRY
		{
        	iReadAmt = input.Read(abyReadBuf, iReadBufSize) ;
			if (iReadAmt < iReadBufSize)
				bFileEnd = TRUE ;
			if (iReadAmt == 0)
				break ;
		}
		CATCH(CFileException, e)
		{
        	AfxMessageBox (MSG_COM_ERR_FILE_EXCEPTION, MB_OK) ;
			break ;
		}
		END_CATCH

		// Leave space in the destination buffer to fill in the size the
		// read buffer was compressed to.
		if (Compress)
		{
			iCompSize = LZ15VCompressBuf (abyReadBuf, iReadAmt,
						abyCompBuf + sizeof (iCompBufSize), iCompBufSize) ;
			if (iCompSize == 0 || iCompSize == iReadAmt)		  
			{
				*((int *) abyCompBuf) = ltl_endian_word ((WORD) -iReadAmt) ;
				memcpy ((BYTE *) (abyCompBuf + (DWORD) sizeof (iReadAmt)),
										abyReadBuf, iReadAmt) ;
			}
			else
			{
				*((int *) abyCompBuf) = ltl_endian_word (iCompSize) ;
			}
			iSizeToSend = iCompSize + sizeof (iCompSize) ;
			pbySendBuf = abyCompBuf ;
		}
		else
		{
			pbySendBuf = abyReadBuf ;
			iSizeToSend = iReadAmt ;
		}

		if (bIsUDB) // If it is User data base
		{
			if (SendAwayUDB (bFileEnd, pbySendBuf, iSizeToSend,
														Timer, EnableCtrl) == 0)
			{
				break ;
			}
		}
		else
		{
			if (SendAway (bFileEnd, pbySendBuf, iSizeToSend,
														Timer, EnableCtrl) == 0)
			{
				break;
			}
		}

		cTotalCompressed += iReadAmt;
		// Visually display activity
		if (InSize)
			wndHorzGauge->UpdateProgress((cTotalCompressed * 100) / InSize);
	}
   ABORT = ABORT || p_cDnldProg->Abort ;
	if (CTRL_FLAG == 1)
		CTRL_FLAG = 0 ;

	// Visually display activity
	if (InSize)
		wndHorzGauge->UpdateProgress((cTotalCompressed * 100) / InSize);

	delete [] abyReadBuf;
	delete [] abyCompBuf;

	wndHorzGauge->DestroyGauge();
	delete wndHorzGauge;
	delete p_cDnldProg;

	return 1;
}

int CMiscCom::SendAway (BOOL bFileEnd, BYTE *pbySendBuf, int iSizeToSend, BOOL *Timer, BOOL EnableCtrl)
{
	int i ;

	// Send out data in 1024 chunks
	BYTE *dnld_data = new BYTE[MAGIC_SIZE] ;
	if (dnld_data == NULL)
	{
      AfxMessageBox (MSG_COM_NO_MEM, MB_OK | MB_ICONINFORMATION) ;
      return 0 ;
	}

	while (1)
	{
		// If we don't have atleast MAGIC_SIZE bytes to send out,
		// don't send. But send if at end of input file.
		if ((iSizePartOfPrev + iSizeToSend) < MAGIC_SIZE)
		{
			// !!Loop termination will occur within this conditional only!!

			if (bFileEnd == FALSE)		// Not at end of file
			{
				// Copy over the compressed bytes to a temporary buffer to
				// be sent in the future
				for (i = iSizePartOfPrev ; i < iSizePartOfPrev + iSizeToSend ;
														i ++, pbySendBuf ++)
				{
					abyPartOfPrevBuf[i] = *pbySendBuf;
				}
				iSizePartOfPrev += iSizeToSend;
			}
			else						// At end of file...send what remains
			{
				for (i = 0 ; i < iSizePartOfPrev ; i++)
				{
					dnld_data[i] = abyPartOfPrevBuf[i] ;
					updatecrc (dnld_data[i]) ;
				}

				for (; i < iSizeToSend + iSizePartOfPrev; i++, pbySendBuf++)
				{
					dnld_data[i] = *pbySendBuf;
					updatecrc(dnld_data[i]);
				}
				ComprCodeSize += iSizeToSend + iSizePartOfPrev;
				if (send_data (dnld_data, iSizeToSend + iSizePartOfPrev,
																	Timer, EnableCtrl) < 0)
				{
					// Download Aborted!!!
					delete [] dnld_data;
					return 0;
				}
				iSizePartOfPrev = 0 ;
				iSizeToSend = 0 ;
			}
			break ;		// Out of the infinite loop
		}

		// If some non-multiple of 1024 bytes of data is left over
		// from previous cycle, copy over that portion first to the
		// send buffer.
		for (i = 0; i < iSizePartOfPrev; i++)
		{
			dnld_data[i] = abyPartOfPrevBuf[i];
			updatecrc(dnld_data[i]);
		}
		// Pad out the remainder of the 1024 byte send buffer with bytes
		// from the new compressed buffer
		for (; i < MAGIC_SIZE;i++, pbySendBuf++)
		{
			dnld_data[i] = *pbySendBuf;
			updatecrc(dnld_data[i]);
		}
		iSizeToSend -= (MAGIC_SIZE - iSizePartOfPrev);
		if (iSizeToSend < 0)
	      AfxMessageBox(MSG_COM_NO_MEM, MB_OK | MB_ICONINFORMATION);
		iSizePartOfPrev = 0;
		ComprCodeSize += MAGIC_SIZE;
		if (send_data(dnld_data, MAGIC_SIZE, Timer, EnableCtrl) < 0)
		{
			// Download Aborted!!!
			delete [] dnld_data;
			return 0;
		}
	}
	delete [] dnld_data;
	return 1;
}

void CMiscCom::snd_mdm_command (BYTE *pkt_data, int pkt_size)
{
  	GetCommError (idComDev, lpStat) ;
  	while (lpStat == NULL)
  		GetCommError (idComDev, lpStat) ;
	while ((lpStat->cbOutQue) != 0) 	
   {
		GetCommError (idComDev, lpStat) ;
		while (lpStat == NULL)
			GetCommError (idComDev, lpStat) ;
	}
  	int err = WriteComm (idComDev, pkt_data, pkt_size) ;
  	if (abs (err) != pkt_size)	
  		AfxMessageBox (MSG_COM_ERR_WRITE, MB_OK | MB_ICONEXCLAMATION) ;
}	


BOOL CMiscCom::rcv_mdm_response (BYTE *mdm_resp, BOOL *Timer)
{
	BYTE resp[50] ;
	int i = 0 ;

	int reqd_len = strlen ((char *) mdm_resp) ;

	while (! *Timer)
	{
		resp[i ++] = rcv_char (Timer) ;
		if (i > reqd_len)
		{
			if (search_substr (resp, mdm_resp, i - 1))
				return TRUE ;
		}
		if (i >= 50)
			return FALSE ;
	}
	return FALSE ;
}


BOOL CMiscCom::search_substr (BYTE *resp_str, BYTE *reqd_resp, int resp_len)
{
	int j ;

	if (!resp_len)
		return FALSE ;
	int reqd_len = strlen ((char *) reqd_resp) ;
	for (int i = 0 ; i <= (resp_len - reqd_len) ; i ++) 
   {
		for (j = 0 ; j < reqd_len ; j ++) 
      {
			if (resp_str[i + j] != reqd_resp[j])
				break ;
		}
		if (j == reqd_len)
			return TRUE ;
	}
	return FALSE ;
}

/************************************************************************
 * Routine	:	ConvertControls
 * Input	:	destination and source pointers
 * Return	:	pointer to destination
 * Note		:	the '^' character given in a modem setup string is 
 *				interpreted by this routine. it puts the modified string
 *				in the destination. A pointer to the destination is
 *				returned
 ************************************************************************/
void	CMiscCom::ConvertControls(BYTE *dst_str, BYTE *src_str)
{
	while (*src_str) {
		if (*src_str == '^') {
			src_str++;
			if (*src_str)
				if (*src_str == '^')
					*dst_str++ = *src_str++;
				else
					*dst_str++ = (BYTE) (*src_str++ - 'A' + 1);
		}
		else {
			*dst_str++ = *src_str++;
		}
	}
	*dst_str = 0;
}


void CMiscCom::hangup_mdm (char *hangup_string, BOOL *Timer)
{
	DCB dcb ;
	/* Drop DTR for some time */
	GetCommState (idComDev, &dcb) ;
	dcb.fDtrDisable = 1 ;
	SetCommState (&dcb) ;

	while (!*Timer)
	{
		MSG msg ;
		while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE)) 
		{
			if (msg.message == WM_QUIT) 
			{
				*Timer = 1 ;
				PostQuitMessage (0) ;
				return ;
			}
			TranslateMessage (&msg) ;
			DispatchMessage (&msg) ;
		}
	}

	/* DTR Up again */
	dcb.fDtrDisable = 0 ;
	SetCommState (&dcb) ;
			
	snd_mdm_command ((BYTE *) hangup_string, strlen (hangup_string)) ;
}


// -------------------------------------------------------------------------
// Function				:	CMiscCom::rcv_packet2
// Arguments         :	packet to be received , its size.
// Synopsis          :	Receives a packet with BridgeID.
// Returns           :	CRC value of the packet / zero if no packet.
// Globals Affected  :	None.
// -------------------------------------------------------------------------
int CMiscCom::rcv_packet2 (BYTE *pkt_data, int pkt_size, BOOL *Timer)
{
	/*---while the received character is not 7e or not timed out do------*/
 	while ((rcv_char (Timer) != (BYTE) FRAME_DELIM) && (! *Timer)) ;

   int k = 0 ;
	int crc = 0xffff ;
	char ch ;

	if (! *Timer)
	{
		while (k < 4)  /* First will be Bridge ID */
		{  /*----receive BridgeId-----*/  
			ch = rcv_char (Timer) ;
			if (ch == (BYTE) 0x7d)
			{
			 	ch = rcv_char (Timer) ;
		   	ch = (BYTE) ((int) ch ^ 0x20) ;  /*---unquote ch----*/
			}
			crc = updcrc (ch, crc) ;
			k ++ ;
		}

		k = 0 ;
		while (k < pkt_size) 
		{ /*----receive the data packet-----*/
			ch = rcv_char (Timer) ; 
			if (ch == (BYTE) FRAME_DELIM)
				break ; 
			if ((ch == (BYTE) RES_TYPE) && (k == 0)) 
			{
				/*---result type packet-----*/
				pkt_size = 4 ;
			}
			/*--------crc=updcrc(ch, crc);---------*/
			if (ch == (BYTE) 0x7d)
			{
				ch = rcv_char (Timer) ;
				pkt_data[k] = (BYTE) ((int) ch ^ 0x20) ;  /*---unquote ch----*/
				/*----crc=updcrc(pkt_data[i], crc);----*/
			}
			else
				pkt_data[k] = ch ;
			crc = updcrc (pkt_data[k], crc) ;
			k ++ ;
		}

		ch = (BYTE) 0 ;
		if (k == pkt_size)
			ch = rcv_char (Timer) ;  
		while ((ch != (BYTE) FRAME_DELIM) && ! *Timer)
		{   /*--while received character is not 7e--*/
			if (ch == (BYTE) 0x7d)
			{
				ch = rcv_char (Timer) ;
				ch = (BYTE) ((int) ch ^ 0x20) ;  /*---unquote ch----*/
			}
			crc = updcrc (ch, crc) ;
			ch = rcv_char (Timer) ;
		}  
		return crc ;
	}
	return 0 ;			
}

// -------------------------------------------------------------------------
// Function				:	CMiscCom::rcv_dnld_packet2
// Arguments         :	packet to be received , its size.
// Synopsis          :	Receives a packet without BridgeID.
// Returns           :	CRC value of the packet / zero if no packet.
// Globals Affected  :	None.
// -------------------------------------------------------------------------
int CMiscCom::rcv_dnld_packet2 (BYTE *pkt_data, int pkt_size, BOOL *Timer)
{
   int k = 0 ; 
   int crc = 0xffff ;
   char ch ;
 	
   /*---while the recevied character is not 7e or not timed out do------*/
 	while ((rcv_char (Timer) != (BYTE) FRAME_DELIM) && (! *Timer)) ;

	if (! *Timer)
	{
	    k = 0 ;
	    while (k < pkt_size) /*----receive the data packet-----*/
	    { 
			ch = rcv_char (Timer) ; 
			if (ch == (BYTE) FRAME_DELIM)
				break ; 
			if ((ch == (BYTE) 12) && (k == 0))
			{
				/*---result type packet-----*/
				pkt_size = 4 ;
			}
			/*--------crc=updcrc(ch, crc);---------*/
			if (ch == (BYTE) 0x7d)
			{
				ch = rcv_char (Timer) ;
				pkt_data[k] = (BYTE) ((int) ch ^ 0x20) ;  /*---unquote ch----*/
				crc = updcrc (pkt_data[k], crc) ;
			}
			else
			{
				pkt_data[k] = ch ;
				crc = updcrc (pkt_data[k], crc) ;
			}
			k ++ ;
		}
		ch = (BYTE) 0 ;
		if (k == pkt_size)
			ch = rcv_char (Timer) ;  
		while (ch != (BYTE) FRAME_DELIM)  
		{  /*--while received character is not 7e--*/
			if (ch == (BYTE) 0x7d)
			{
				ch = rcv_char (Timer) ;
				ch = (BYTE) ((int) ch ^ 0x20) ;  /*---unquote ch----*/
			}
			crc = updcrc (ch, crc) ; 
			ch = rcv_char (Timer) ;
		}  
		return crc ;
	}
	return 0 ;
}


int CMiscCom::send_data_online (BYTE *str, UINT count, BOOL *Timer, BOOL EnableCtrl)
{
	UINT l = 0 ;

	OnLineFlashWriteType UDBWriteType ;
	
	typedef struct {
		BYTE RtnPktType ;		
		BYTE Dummy ;
		UINT buffer_length ;
		DWORD down_load_address ;
	} UDBRtnRespType ;
	UDBRtnRespType hdr_type ;

	UDBPktCRC = 0xFFFF ;
	while (l < count)
	{
		UDBWriteType.buffer[l] = str[l] ;
		UDBPktCRC = updcrc (UDBWriteType.buffer[l], UDBPktCRC) ;
		l ++ ;
	}	
	UDBWriteType.TargetAction = ONLINE_FLASH_WRITE ;		// 4

	UDBWriteType.DnLdAddress = UDBDnldAddress ;
	UDBWriteType.DnLdAddress += PrevComprCodeSize ;
	UDBWriteType.DnLdAddress = little_endian (UDBWriteType.DnLdAddress) ;
	PrevComprCodeSize = ComprCodeSize ;

	UDBWriteType.Length = ltl_endian_word (UDBActualLen) ; 

	UDBWriteType.CRC = ltl_endian_word (UDBPktCRC) ;

	hdr_type.RtnPktType = 0xFF ;

	while ((hdr_type.RtnPktType != ONLINE_FLASH_WRITE) && (! *Timer))		// 6
	{	
		MSG msg ;
		while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
		{
			if (msg.message == WM_QUIT) 
			{
				*Timer = 1 ;
				PostQuitMessage (0) ;
				return NULL ;
			}
			TranslateMessage (&msg);
			DispatchMessage (&msg);
		}

		if (EnableCtrl)
		{
			FlushComm (idComDev, 1) ; // Clear the Rx queue
			snd_packet ((BYTE *) &UDBWriteType,	sizeof (UDBWriteType)) ;
			if (rcv_packet ((BYTE *) &hdr_type, sizeof (hdr_type), Timer))
			{
				continue ;
			} 
			if (hdr_type.down_load_address != UDBWriteType.DnLdAddress)
				continue ; 
		}

		if (! *Timer)
		{
			switch (hdr_type.RtnPktType)
			{
				case PT_ABORT :			// 3
					AfxMessageBox (MSG_COM_ROUTER_ABORT, MB_OK) ;
					ABORT = TRUE ;	
	            if (CTRL_FLAG == 1)   
						CTRL_FLAG = 0 ;	
					return -1 ;	
	            break;   
						
				case PT_NACK : 		// 5
					break ;

				case ONLINE_FLASH_WRITE :
					break ;

				default :             			
					break ;	
			}
		}
		else
		{
         AfxMessageBox (MSG_COM_NO_ROUTER_RESPONSE, MB_OK) ;
         ABORT = TRUE ;
         return -1 ;
      }
	}	

	if (hdr_type.RtnPktType != ONLINE_FLASH_WRITE)
	{
	} 

   return 0 ;
}

int CMiscCom::SendAwayUDB (BOOL bFileEnd, BYTE *pbySendBuf, int iSizeToSend, BOOL *Timer, BOOL EnableCtrl)
{
	int i ;

	// Send out data in 512 chunks
	BYTE *dnld_data = new BYTE[UDB_SENT_ONCE_COM] ;
	if (dnld_data == NULL)
	{
      AfxMessageBox (MSG_COM_NO_MEM, MB_OK | MB_ICONINFORMATION) ;
      return 0 ;
	}

	while (1)
	{
		// If we don't have atleast UDB_SENT_ONCE_COM bytes to send out,
		// don't send. But send if at end of input file.
		if ((iSizePartOfPrev + iSizeToSend) < UDB_SENT_ONCE_COM)
		{
			// !!Loop termination will occur within this conditional only!!

			if (bFileEnd == FALSE)		// Not at end of file
			{
				// Copy over the compressed bytes to a temporary buffer to
				// be sent in the future
				for (i = iSizePartOfPrev ; i < iSizePartOfPrev + iSizeToSend ;
														i ++, pbySendBuf ++)
				{
					abyPartOfPrevBuf[i] = *pbySendBuf;
				}
				iSizePartOfPrev += iSizeToSend;
			}
			else						// At end of file...send what remains
			{
				for (i = 0 ; i < iSizePartOfPrev ; i++)
				{
					dnld_data[i] = abyPartOfPrevBuf[i] ;
					updatecrc (dnld_data[i]) ;
				}

				for (; i < iSizeToSend + iSizePartOfPrev; i++, pbySendBuf++)
				{
					dnld_data[i] = *pbySendBuf;
					updatecrc (dnld_data[i]) ;
				}
				ComprCodeSize += iSizeToSend + iSizePartOfPrev ;
				UDBActualLen = iSizeToSend + iSizePartOfPrev ;
				if (send_data_online (dnld_data, iSizeToSend + iSizePartOfPrev,
																	Timer, EnableCtrl) < 0)
				{
					// Download Aborted!!!
					delete [] dnld_data;
					return 0;
				}
				iSizePartOfPrev = 0 ;
				iSizeToSend = 0 ;
			}
			break ;		// Out of the infinite loop
		}

		// If some non-multiple of 512 bytes of data is left over
		// from previous cycle, copy over that portion first to the
		// send buffer.
		for (i = 0; i < iSizePartOfPrev; i++)
		{
			dnld_data[i] = abyPartOfPrevBuf[i] ;
			updatecrc (dnld_data[i]) ;
		}
		// Pad out the remainder of the 512 byte send buffer with bytes
		// from the new compressed buffer
		for (; i < UDB_SENT_ONCE_COM;i++, pbySendBuf++)
		{
			dnld_data[i] = *pbySendBuf;
			updatecrc (dnld_data[i]) ;
		}
		iSizeToSend -= (UDB_SENT_ONCE_COM - iSizePartOfPrev);
		if (iSizeToSend < 0)
	      AfxMessageBox(MSG_COM_NO_MEM, MB_OK | MB_ICONINFORMATION);
		iSizePartOfPrev = 0;
		ComprCodeSize += UDB_SENT_ONCE_COM;
		UDBActualLen = UDB_SENT_ONCE_COM ;
		if (send_data_online (dnld_data, UDB_SENT_ONCE_COM, Timer, EnableCtrl) < 0)
		{
			// Download Aborted!!!
			delete [] dnld_data;
			return 0;
		}
	}
	delete [] dnld_data;
	return 1;
}





#if 1


int CMiscCom::receive_buffer (char FAR *buffer, int buffer_length)
{     
	DCB FAR *dcb = new __far DCB ;
	int buffer_read ;

	if (idComDev < 0)
		return 0 ;

   if (GetCommState (idComDev, dcb))
      AfxMessageBox (MSG_COM_ERR_READ_STAT, MB_OK) ;

  	GetCommError (idComDev, lpStat) ;

	if (lpStat->cbInQue)
	{
		buffer_read = ReadComm (idComDev, buffer, buffer_length) ;
		delete dcb ;
		if (buffer_read < 0)
			return 0 ;
		else
			return buffer_read ;
	}
	else 
	{
		delete dcb ;
		return 0 ;
	}
}

#endif
