/*------------------------------------------------------------------------*/
/* LZVC.C -- LZW Compression with variable number of bits.
** From code by Hemanth M.C. (Windows 95 Remote Control Project)
*/
#include <stdlib.h>
#include "lzv.h"
#include "lzvc.h"

#ifdef __cplusplus
extern "C" {
#endif 
/* Globals */
static unsigned next_code;
static unsigned next_bump_code;
static unsigned current_code_bits;
static unsigned char decode_stack[TABLE_SIZE];
static struct dictionary *dict_array[TABLE_BANKS];

/* Local Prototypes */
unsigned OutputBits(struct BitBufIndex *buffer, unsigned bits, unsigned bit_length);
int InitializeStorage(void);
void ReleaseStorage(void);
void InitializeDictionary(void);
unsigned find_child_node(unsigned parent_code, unsigned child_character);
WORD LZ15VCompressBuf(BYTE *lpInputData, WORD wInputLen, BYTE *lpCompData, WORD wCompLen);

unsigned OutputBits(struct BitBufIndex *buffer, unsigned bits, unsigned bit_length)
{

	while(buffer->bit_offset + bit_length > 8) {
		if (buffer->length == 0)
			return 0;

		bit_length -= 8 - buffer->bit_offset;
		if (buffer->bit_offset != 0)
			*buffer->index |= (unsigned char)(bits >> bit_length);
		else
			*buffer->index  = (unsigned char)(bits >> bit_length);
		buffer->index++;
		buffer->length--;
		buffer->bit_offset = 0;
		bits &= (1 << bit_length) - 1;
	}
	if (buffer->length == 0)
		return 0;
	if (buffer->bit_offset != 0)
		*buffer->index |= (unsigned char)(bits << (8 - bit_length) );
	else
		*buffer->index  = (unsigned char)(bits << (8 - bit_length) );
	buffer->bit_offset += bit_length;
	
	return 1;
}
 
 
int InitializeStorage()
{
	unsigned i;
	
	for (i = 0; i < TABLE_BANKS; i++) {
		dict_array[i] = (struct dictionary *) calloc(256, sizeof(struct dictionary));
		if (dict_array[i] == (struct dictionary *)0)
			return 0;
	}
	return 1;
}


void ReleaseStorage()
{
	unsigned i;
	
	for (i = 0; i < TABLE_BANKS; i++)
		free(dict_array[i]);
}


void InitializeDictionary()
{
	unsigned i;

	for (i = 0; i < TABLE_SIZE; i++)
		DICT(i).code_value = UNUSED;
	next_code = FIRST_CODE;

	current_code_bits = 9;
	next_bump_code = 511;
}


unsigned find_child_node(unsigned parent_code, unsigned child_character)
{
	unsigned index;
	unsigned offset;
	
	index = ( child_character << (BITS - 8) ) ^ parent_code;
	if (index == 0)
		offset = 1;
	else if (index < TABLE_SIZE)
		offset = TABLE_SIZE - index;
	for (;;) {
		if (DICT(index).code_value == UNUSED)
			return index;
		if (DICT(index).parent_code== parent_code &&
			DICT(index).character  == child_character)
			return index;
		if (index >= offset)
			index -= offset;
		else
			index += TABLE_SIZE - offset;
	}
}

WORD LZW15VCompressBuf(BYTE *lpInputData, WORD wInputLen, BYTE *lpCompData, WORD wCompLen)
{
	unsigned index;
	unsigned wOrgLen;
	unsigned character;
	unsigned string_code;
	unsigned input_count;
	const unsigned char *input_ix;
	struct BitBufIndex output_buf;


	input_ix = (unsigned char *)lpInputData;
	input_count = wInputLen;
	if (InitializeStorage() == 0)
		return 0;
	InitializeDictionary();

	output_buf.index = (unsigned char *)lpCompData;
	output_buf.length= wOrgLen = ( (wCompLen <= wInputLen) ? wCompLen : wInputLen );
	output_buf.bit_offset = 0;

	if (input_count == 0)
		string_code = END_OF_STREAM;
	else {
		string_code = *input_ix++;
		input_count--;
	}
	while(input_count > 0) {
		character = *input_ix++;
		input_count--;

		index = find_child_node(string_code, character);		
		if (DICT(index).code_value != UNUSED)
			string_code = DICT(index).code_value;
		else {
			DICT(index).code_value = next_code++;
			DICT(index).parent_code= string_code;
			DICT(index).character  = (unsigned char) character;

			if (OutputBits(&output_buf, string_code, current_code_bits) == 0)
				break;
			string_code = character;
			if (next_code > MAX_CODE) {
				if (OutputBits(&output_buf, FLUSH_CODE, current_code_bits) == 0)
					break;
				InitializeDictionary();
			} else if (next_code > next_bump_code) {
				if (OutputBits(&output_buf, BUMP_CODE, current_code_bits) == 0)
					break;
				current_code_bits++;
				next_bump_code = (next_bump_code << 1) | 1;
			}
		}
	}
	ReleaseStorage();
	if (output_buf.length != 0) {
		if (OutputBits(&output_buf, string_code, current_code_bits) != 0)
			if (OutputBits(&output_buf, END_OF_STREAM, current_code_bits) != 0)
				return wOrgLen - output_buf.length + 1;
	}
	if (wCompLen < wInputLen)
		return 0;
	return 0;
}

#ifdef __cplusplus
}
#endif 

