/*------------------------------- LZW15V.C ----------------------------------*
  	Program 	: LZW 9-15 bit (variable) data compression module
  	Author		: M.C.Hemanth
  	Date		: February 2, 1996.

	Description:

		Code largely copied from "The Data Compression Book" by Mark Nelson.
	
		This is the LZW module which implements a powerful version of the LZW 
		algorithm. This version of the program expands the maximum code size
		to 15 bits. It starts encoding with 9 bit codes, working its way up in
		bit size only as necessary. It flushes the dictionary when done.

		InputBits & OutputBits have been modified to suit our need for memory
			buffer compression.

 *---------------------------------------------------------------------------*/

#include <stdlib.h>
#include "types.h"
#include "compress.h"

#define BITS			15
#define MAX_CODE 		( ((unsigned)1 << BITS) - 1 )
#define TABLE_SIZE		((unsigned)35023)
#define TABLE_BANKS		( (TABLE_SIZE >> 8) + 1 )
#define END_OF_STREAM	256
#define BUMP_CODE		257
#define FLUSH_CODE		258
#define FIRST_CODE		259
#define UNUSED			((unsigned)-1)

#pragma warning(disable:4711)

struct dictionary
{
	unsigned code_value;
	unsigned parent_code;
	unsigned char character;
};

struct BitBufIndex {
	unsigned char *index;
	unsigned length;
	unsigned bit_offset;
};

static unsigned char decode_stack[TABLE_SIZE];
static unsigned next_code;
static unsigned current_code_bits;
static unsigned next_bump_code;

static char error_status[100];

static struct dictionary *dict_array[TABLE_BANKS];
#define DICT(i)	dict_array[i>>8][i & 0xFF]

#if 0
void FatalError(const char *fn_name, const char *error)
{
	if (error_status[0] == '\0')
	{
		printf("%s: %s", fn_name, error);
		exit(1);
	}

	_asm int 3			/* Perpetual break point */
}
#endif

#if 0
BYTE *LZ15VGetError()
{
	if (error_status[0] == '\0')
		return NULL;
	else
		return error_status;
}
#endif


unsigned OutputBits(struct BitBufIndex *buffer, unsigned bits, unsigned bit_length)
{
#ifdef _DEBUG
	const char fn_name[] = "OutputBits";
	if (bits & ~( (1 << bit_length) - 1 )) {
		FatalError(fn_name, "Error in input parameter:bits");
		return 0;
	}
#endif

	while(buffer->bit_offset + bit_length > 8) {
		if (buffer->length == 0)
			return 0;

		bit_length -= 8 - buffer->bit_offset;
		if (buffer->bit_offset != 0)
			*buffer->index |= (unsigned char)(bits >> bit_length);
		else
			*buffer->index  = (unsigned char)(bits >> bit_length);
		buffer->index++;
		buffer->length--;
		buffer->bit_offset = 0;
		bits &= (1 << bit_length) - 1;
	}
	if (buffer->length == 0)
		return 0;
	if (buffer->bit_offset != 0)
		*buffer->index |= (unsigned char)(bits << (8 - bit_length) );
	else
		*buffer->index  = (unsigned char)(bits << (8 - bit_length) );
	buffer->bit_offset += bit_length;
	
	return 1;
}
 
 
unsigned InputBits(struct BitBufIndex *buffer, unsigned bit_length)
{
	unsigned bits, byte_length;

#ifdef _DEBUG
	const char fn_name[] = "InputBits";
	if (bit_length > 16) {
		FatalError(fn_name, "'bit_length' too high");
		return 0;
	}
	if (buffer->bit_offset > 16) {
		FatalError(fn_name, "'buffer' not initialized");
		return 0;
	}
#endif

	bits = 0;
	while(buffer->bit_offset + bit_length > 8) {
		byte_length = 8 - buffer->bit_offset;
		bits <<= byte_length;
		bits |= *buffer->index & ( (1 << byte_length) - 1 );
		bit_length -= byte_length;
		buffer->bit_offset = 0;
		buffer->index++;
		buffer->length--;
	}
	bits <<= bit_length;
	bits |= (*buffer->index >> (8 - bit_length));
	buffer->bit_offset += bit_length;
	return bits;
}


int InitializeStorage()
{
	const char fn_name[] = "Initialize Storage";
	unsigned i;
	
	for (i = 0; i < TABLE_BANKS; i++) {
		dict_array[i] = calloc(256, sizeof(struct dictionary));
		if (dict_array[i] == NULL)
//			FatalError(fn_name, "Out of memory");
			return 0 ;
	}

	error_status[0] = '\0';
	return 1 ;
}


void ReleaseStorage()
{
	unsigned i;
	
	for (i = 0; i < TABLE_BANKS; i++)
		free(dict_array[i]);
}


void InitializeDictionary()
{
	unsigned i;

	for (i = 0; i < TABLE_SIZE; i++)
		DICT(i).code_value = UNUSED;
	next_code = FIRST_CODE;

	current_code_bits = 9;
	next_bump_code = 511;
}


unsigned decode_string(unsigned count, unsigned code)
{
	const char fn_name[] = "decode_string";
	while (code > 255) {
		decode_stack[count++] = DICT(code).character;
		code = DICT(code).parent_code;
	}
	decode_stack[count++] = (unsigned char)code;
	return count;
}


unsigned find_child_node(unsigned parent_code, unsigned child_character)
{
	const char fn_name[] = "find_child_node";
	unsigned index;
	unsigned offset;
	
	index = ( child_character << (BITS - 8) ) ^ parent_code;
	if (index == 0)
		offset = 1;
	else if (index < TABLE_SIZE)
		offset = TABLE_SIZE - index;
	else
//		FatalError(fn_name, "index invalid");
		return 0 ;
	for (;;) {
		if (DICT(index).code_value == UNUSED)
			return index;
		if (DICT(index).parent_code== parent_code &&
			DICT(index).character  == child_character)
			return index;
		if (index >= offset)
			index -= offset;
		else
			index += TABLE_SIZE - offset;
	}
}


WORD LZ15VCompressBuf(BYTE *lpInputData, WORD wInputLen, BYTE *lpCompData, WORD wCompLen)
{
	const char fn_name[] = "LZ15VCompressBuf";
	unsigned character;
	unsigned string_code;
	unsigned index;

	unsigned wOrgLen;
	const unsigned char *input_ix;
	unsigned input_count;
	struct BitBufIndex output_buf;

	input_ix = (unsigned char *)lpInputData;
	input_count = wInputLen;
	if (!InitializeStorage())
		return 0 ;
//	if (LZ15VGetError())
//		return 0;
	InitializeDictionary();

	output_buf.index = (unsigned char *)lpCompData;
	output_buf.length= wOrgLen = ( (wCompLen <= wInputLen) ? wCompLen : wInputLen );
	output_buf.bit_offset = 0;

	if (input_count == 0)
		string_code = END_OF_STREAM;
	else {
		string_code = *input_ix++;
		input_count--;
	}
	while(input_count > 0) {
		character = *input_ix++;
		input_count--;

		index = find_child_node(string_code, character);		
		if (DICT(index).code_value != UNUSED)
			string_code = DICT(index).code_value;
		else {
			DICT(index).code_value = next_code++;
			DICT(index).parent_code= string_code;
			DICT(index).character  = (unsigned char) character;

			if (OutputBits(&output_buf, string_code, current_code_bits) == 0)
				break;
			string_code = character;
			if (next_code > MAX_CODE) {
				if (OutputBits(&output_buf, FLUSH_CODE, current_code_bits) == 0)
					break;
				InitializeDictionary();
			} else if (next_code > next_bump_code) {
				if (OutputBits(&output_buf, BUMP_CODE, current_code_bits) == 0)
					break;
				current_code_bits++;
				next_bump_code = (next_bump_code << 1) | 1;
			}
		}
	}
	ReleaseStorage();
	if (output_buf.length != 0) {
		if (OutputBits(&output_buf, string_code, current_code_bits) != 0)
			if (OutputBits(&output_buf, END_OF_STREAM, current_code_bits) != 0)
				return wOrgLen - output_buf.length + 1;
	}
	if (wCompLen < wInputLen)
//		FatalError(fn_name, "Compress buffer full");
		return 0;
}


WORD LZ15VExpandBuf(BYTE *lpCompData, WORD wCompLen, BYTE *lpOutputData, WORD wOutputLen)
{
	const char fn_name[] = "LZ15VExpandBuf";
	unsigned new_code;
	unsigned old_code;
	unsigned character;
	unsigned count;
	struct BitBufIndex input_buf;
	unsigned char *output_ix;
	unsigned output_count;

	output_ix = lpOutputData;
	output_count = wOutputLen;
	input_buf.index = (unsigned char *)lpCompData;
	input_buf.length= wCompLen;
	input_buf.bit_offset = 0;
	
	if (!InitializeStorage())
//	if (LZ15VGetError())
		return 0;

	for (;;) {
		InitializeDictionary();
		old_code = InputBits(&input_buf, current_code_bits);
		if (old_code == END_OF_STREAM)
			goto ReturnOutputLength;
		character = old_code;
		*output_ix++ = (unsigned char)old_code;
		output_count--;
		for (;;) {
			new_code = InputBits(&input_buf, current_code_bits);
			if (new_code == END_OF_STREAM)
				goto ReturnOutputLength;
			if (new_code == FLUSH_CODE)
				break;
			if (new_code == BUMP_CODE) {
				current_code_bits++;
				continue;
			}
			if (new_code >= next_code) {
				decode_stack[0] = (unsigned char)character;
				count = decode_string(1, old_code);
			} else
				count = decode_string(0, new_code);
			character = decode_stack[count - 1];
			while (count > 0 && output_count > 0) {
				*output_ix++ = decode_stack[--count];
				output_count--;
			}
			if (count > 0) {
//				FatalError(fn_name, "Output buffer full");
				goto ReturnError;
			}
			DICT(next_code).parent_code = old_code;
			DICT(next_code).character   = (unsigned char)character;
			next_code++;
			old_code = new_code;
		}
	}

ReturnError:
	ReleaseStorage();
	return 0;

ReturnOutputLength:
	ReleaseStorage();
	return wOutputLen - output_count;
}
