/*--------------------------------------------------------------------------*/
/*	File		:	CHECK.C													*/
/*	Purpose		:	This file contains consistency check routines.			*/
/*	Package		:	MultiExpress Ver 2.00.									*/
/*	Authors		:	S. Narasimhan.											*/
/*	Date		:	April 6, 1992.											*/
/*--------------------------------------------------------------------------*/

#include	"windows.h"

#include	"errcons.h"
#include	"scrtcons.h"
#include	"scrtmesg.h"

#include	"scrttype.h"
#include	"scrtdata.h"
#include	"scrtfuns.h"

#include	<string.h>

/*--------------------------------------------------------------------------*/
/*	Name		:	CheckAssignment											*/
/*	Input		:	sym_ptr, pointer to symbol to be assigned,				*/
/*					expr_ptr, pointer to expression node.					*/
/*	Output		:	true if assignment is compatible, else false.			*/
/*	Synopsis	:	Checks for assignment statement compatibility.			*/
/*--------------------------------------------------------------------------*/

int		CheckAssignment(SymInfoType *sym_ptr, ExprNodeType *expr_ptr)
{
	int		result;
	int		fn_ret_value;	/*	Return value of function.	*/

	result = true;
	switch (sym_ptr->SymClass) {
		case Q_INTEGER :
			switch (expr_ptr->ExprType) {
				case E_INTEGER :
					break;
				case E_REAL :
					break;
				default :
					Error(ERR_TYPE_CONFLICT);
					break;
			}
			break;
		case Q_REAL :
			switch (expr_ptr->ExprType) {
				case E_INTEGER :
					break;
				case E_REAL :
					break;
				default :
					Error(ERR_TYPE_CONFLICT);
					break;
			}
			break;
		case Q_STRING :
			switch (expr_ptr->ExprType) {
				case E_STRING :
					break;
				default :
					Error(ERR_TYPE_CONFLICT);
					break;
			}
			break;
		case Q_PROC :
			if (sym_ptr->ProcInfoPtr)
				fn_ret_value = sym_ptr->ProcInfoPtr->RetValue;
			else
				fn_ret_value = Q_VOID;	/*	Assume void type.	*/
			switch (fn_ret_value) {
				case Q_INTEGER :
					switch (expr_ptr->ExprType) {
						case E_INTEGER :
						case E_REAL :
							break;
						default :
							Error(ERR_TYPE_CONFLICT);
							result = false;
							break;
					}
					break;
				case Q_REAL :
					switch (expr_ptr->ExprType) {
						case E_INTEGER :
						case E_REAL :
							break;
						default :
							Error(ERR_TYPE_CONFLICT);
							result = false;
							break;
					}
					break;
				case Q_STRING :
					switch (expr_ptr->ExprType) {
						case E_STRING :
							break;
						default :
							Error(ERR_TYPE_CONFLICT);
							result = false;
							break;
					}
					break;
				case Q_VOID :
					Error(ERR_ASSIGNMENT_TO_VOID);
					result = false;
					break;
			}
			if (result)
				sym_ptr->Status |= FUNCTION_ASSIGNED;
			break;
	}
	return(result);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	CheckUndefinedLabels									*/
/*	Input		:	None													*/
/*	Output		:	true if no undefined labels, else false.				*/
/*	Synopsis	:	Checks for used but undefined labels.					*/
/*--------------------------------------------------------------------------*/

int		CheckUndefinedLabels(SymInfoType *sym_ptr)
{
	int		result;

	result = false;
	if (sym_ptr) {
		if (sym_ptr->SymClass == Q_LABEL &&
								!(sym_ptr->SymType & LABEL_DEFINED)) {
			Error(ERR_LABEL_NOT_DECLARED);
			result = true;
		}
		result |= CheckUndefinedLabels(sym_ptr->Llink);
		result |= CheckUndefinedLabels(sym_ptr->Rlink);
	}
	return(result);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	CheckForwardProcedures									*/
/*	Input		:	None													*/
/*	Output		:	true if each forward defined procedure body has been	*/
/*						given, else false.									*/
/*	Synopsis	:	Checks for whose body has not been given.				*/
/*--------------------------------------------------------------------------*/

int		CheckForwardProcedures(SymInfoType *sym_ptr)
{
	int		result;

	result = false;
	if (sym_ptr) {
		if (sym_ptr->SymClass == Q_PROC &&
							(sym_ptr->Status & FORWARD_DEFINED) &&
							!(sym_ptr->Status & PROC_BODY_DEFINED)) {
			Error(ERR_PROC_BODY_NOT_FOUND);
			result = true;
		}
		result |= CheckForwardProcedures(sym_ptr->Llink);
		result |= CheckForwardProcedures(sym_ptr->Rlink);
	}
	return(result);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	CompatibleArgs											*/
/*	Input		:	arg_type, argument type,								*/
/*					expr_type, expression type.								*/
/*	Output		:	true if compatible, else false.							*/
/*	Synopsis	:	Checks argument definition and call expression types	*/
/*						for compatibility.									*/
/*--------------------------------------------------------------------------*/

int		CompatibleArgs(int arg_type, int expr_type)
{
	if (arg_type == Q_INTEGER && expr_type == E_INTEGER)
		return(true);
	if (arg_type == Q_REAL && (expr_type == E_INTEGER || expr_type == E_REAL))
		return(true);
	if (arg_type == Q_STRING && expr_type == E_STRING)
		return(true);
	return(false);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	CheckFunctionArgs										*/
/*	Input		:	sym_ptr, function name symbol table pointer,			*/
/*					arg_list, pointer to argument list.						*/
/*	Output		:	true if arguments match, else false.					*/
/*	Synopsis	:	Checks function call with defintion for arguments.		*/
/*--------------------------------------------------------------------------*/

int		CheckFunctionArgs(SymInfoType *sym_ptr, CallArgType *arg_list)
{
	DefnArgType	*formal_arg_ptr;
	CallArgType	*a_ptr;
	int		result;

	result = true;
	if (sym_ptr->ProcInfoPtr)		/*	Proc info available ?	*/
		formal_arg_ptr = sym_ptr->ProcInfoPtr->ArgList;
	else
		formal_arg_ptr = 0;		/*	Assume no arguments.	*/
	a_ptr = arg_list;	/*	None of the arguments should be void type.	*/
	while (a_ptr) {
		if (a_ptr->ExprPtr->ExprType == E_VOID) {
			Error(ERR_VOID_IN_EXPR);
			result = false;
		}
		a_ptr = a_ptr->NextArgPtr;
	}
	while (formal_arg_ptr && arg_list) {
		/*	Do not check type compatibility for built-in functions	*/
		/*	with don't care types.	*/
		if (!(sym_ptr->Status & BUILT_IN_FUNCTION) ||
											formal_arg_ptr->ArgClass) {
			/*	Check for type compatibility.	*/
			if (!CompatibleArgs(formal_arg_ptr->ArgClass,
											arg_list->ExprPtr->ExprType)) {
				Error(ERR_ARG_TYPES_DIFFER);
				result = false;
			}
		}
		/*	Check for call by reference parameter.	*/
		if (formal_arg_ptr->Attribute == Q_VAR &&
						!(arg_list->ExprPtr->Attribute & E_LVALUE)) {
			Error(ERR_NO_LVALUE_FOR_EXPR);
			result = false;
		}
		formal_arg_ptr = formal_arg_ptr->NextArgPtr;
		arg_list = arg_list->NextArgPtr;
	}
	if (formal_arg_ptr) {
		Error(ERR_FEW_ARGS_FOR_FN);
		result = false;
	}
	if (arg_list && (sym_ptr->Status &
								(BUILT_IN_FUNCTION | OPTIONAL_ARGS)) !=
								(BUILT_IN_FUNCTION | OPTIONAL_ARGS)) {
		/*	Actual arguments are more, and the function is not a built	*/
		/*	in one with optional arguments.	*/
		Error(ERR_TOO_MANY_ARGS_FOR_FN);
		result = false;
	}
	return(result);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	ResolveLabels											*/
/*	Input		:	c_ptr, pointer to procedure code beginning.				*/
/*	Output		:	0 if successful, else -1.								*/
/*	Synopsis	:	Resolves user defined label offsets.					*/
/*--------------------------------------------------------------------------*/

int		ResolveLabels(CodeTypeFptr c_ptr)
{
	word	count;
	SymInfoType	*sym_ptr;

	count = 0;	/*	Start counting instructions.	*/
	while (c_ptr) {
		if (c_ptr->Operator == Q_LABEL_DEFN) {	/*	Label declaration.	*/
			sym_ptr = FindSymbol(c_ptr->Arg1);
			if (sym_ptr)	/*	User defined label.	*/
				sym_ptr->Status = count;
		}
		c_ptr = c_ptr->NextCode;
		count++;		/*	One more instruction.	*/
	}
	return(0);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	GetLabelOffset											*/
/*	Input		:	name, label name.										*/
/*	Output		:	offset of label if found, else -1.						*/
/*	Synopsis	:	Gets the label offset in the procedure code.			*/
/*--------------------------------------------------------------------------*/

int		GetLabelOffset(byte *name)
{
	word	count;
	SymInfoType	*sym_ptr;
	CodeTypeFptr	c_ptr;

	/*	If it is not a generated label, look in symbol table for offset.	*/
	if (*name != '$' && (sym_ptr = FindSymbol(name))) {
		return(sym_ptr->Status);
	}
	else {
		c_ptr = ProcCode;
		count = 0;
		while (c_ptr) {
			if (c_ptr->Operator == Q_LABEL_DEFN && !strcmp(c_ptr->Arg1, name))
				return(count);
			c_ptr = c_ptr->NextCode;
			count++;
		}
	}
	return(-1);		/*	Label not found, error !	*/
}

/*--------------------------------------------------------------------------*/
/*	Name		:	GetResultType											*/
/*	Input		:	operator, function to be performed,						*/
/*					type1, type of first operand,							*/
/*					type2, type of second operand.							*/
/*	Output		:	type of result, if valid combination, else -1.			*/
/*	Synopsis	:	Computes the result type of arithmetic operations.		*/
/*--------------------------------------------------------------------------*/

int		GetResultType(byte operator, byte type1, byte type2)
{
	byte	result_type;	/*	Type of result.	*/

	switch (operator) {
		case Q_PLUS :
		case Q_MINUS :
		case Q_TIMES :
		case Q_DIV :
			switch (type1) {
				case E_INT_CONST :
				case E_INTEGER :
					switch (type2) {
						case E_INT_CONST :
						case E_INTEGER :
							result_type = E_INTEGER;	/*	Target type.	*/
							break;
						case E_REAL_CONST :
						case E_REAL :
							result_type = E_REAL;	/*	Target type.	*/
							break;
						default :
							goto error_exit;
					}
					break;
				case E_REAL_CONST :
				case E_REAL :
					switch (type2) {
						case E_INT_CONST :
						case E_INTEGER :
						case E_REAL_CONST :
						case E_REAL :
							result_type = E_REAL;	/*	Target type.	*/
							break;
						default :
							goto error_exit;
					}
					break;
				default :
					goto error_exit;
			}
			break;
		case Q_LESS_OR_EQUAL :
		case Q_GREATER_OR_EQUAL :
		case Q_LESS :
		case Q_GREATER :
		case Q_IS_EQUAL :
		case Q_IS_NOT_EQUAL :
			switch (type1) {
				case E_INT_CONST :
				case E_INTEGER :
				case E_REAL_CONST :
				case E_REAL :
					switch (type2) {
						case E_INT_CONST :
						case E_INTEGER :
						case E_REAL_CONST :
						case E_REAL :
							result_type = E_INTEGER;	/*	Target type.	*/
							break;
						default :
							goto error_exit;
					}
					break;
				case E_STRING_CONST :
				case E_STRING :
					switch (type2) {
						case E_STRING_CONST :
						case E_STRING :
							result_type = E_INTEGER;	/*	Target type.	*/
							break;
						default :
							goto error_exit;
					}
					break;
				default :
					goto error_exit;
			}
			break;
		case Q_AND :
		case Q_OR :
			switch (type1) {
				case E_INT_CONST :
				case E_INTEGER :
					switch (type2) {
						case E_INT_CONST :
						case E_INTEGER :
							result_type = E_INTEGER;	/*	Target type.	*/
							break;
						default :
							goto error_exit;
					}
					break;
				default :
					goto error_exit;
			}
			break;
		default :
			goto error_exit;
	}
	return(result_type);

	error_exit :
	return(-1);
}
