/*--------------------------------------------------------------------------*/
/*	File		:	EXPR.C													*/
/*	Purpose		:	This file contains expression related routines.			*/
/*	Package		:	MultiExpress Ver 2.00.									*/
/*	Authors		:	S. Narasimhan.											*/
/*	Date		:	April 6, 1992.											*/
/*--------------------------------------------------------------------------*/

#ifdef WIN
#include	"windows.h"
#endif

#include	"errcons.h"
#include	"scrtcons.h"
#include	"scrtmesg.h"

#include	"scrttype.h"
#include	"scrtdata.h"
#include	"scrtfuns.h"

#include	<string.h>
#include	<stdlib.h>

/*--------------------------------------------------------------------------*/
/*	Name		:	Pexpression												*/
/*	Input		:	None													*/
/*	Output		:	name, pointer to name of temporary holding expr value.	*/
/*	Synopsis	:	Reduces <expression>									*/
/*--------------------------------------------------------------------------*/

int		Pexpression(ExprNodeType **expr_ptr)
{
	#define	SYM_STACK_SIZE	100
	#define	MAX_EXP_ERRORS	10
	#define	Q_DOLLAR		0x100

	ExprNodeType	*sym_stack[SYM_STACK_SIZE];	/*	Sym stack for parsing.	*/
	ExprNodeType	*node;
	SymInfoType	*sym_ptr;	/*	Symbol table ptr to id.	*/
	CallArgType	*arg_ptr;	/*	Pointer to arg list of function.	*/
	CallArgType	**arg_ptr_ptr;
	int		last;	/*	Index of stack top.	*/
	int		idx;
	int		new_top;
	int		top_stack;	/*	Terminal symbol on top of stack.	*/
	byte	end_of_input;	/*	Indicates end of input.	*/
	int		token;	/*	Input token.	*/
	int		a, b;	/*	Tokens on stack while reducing.	*/
	byte	precedence;
	byte	error_list[MAX_EXP_ERRORS];	/*	error list.	*/
	byte	error_count;
	int		expr_type;		/*	expr type.	*/

	error_count = 0;
	TempVarCount = 0;	/*	Reset temporaries count.	*/
	for (idx = 0; idx < SYM_STACK_SIZE; idx++)
		sym_stack[idx] = 0;	/*	Initialize symbol stack.	*/
	if (Token != Q_ID && Token != Q_INT_CONST && Token != Q_REAL_CONST &&
			Token != Q_STRING_CONST && Token != Q_NOT &&
			Token != Q_UNARY_MINUS && Token != Q_MINUS && Token != Q_LP) {
		if (error_count < MAX_EXP_ERRORS)
			error_list[error_count++] = ERR_ILL_FORMED_EXPRESSION;
		goto error_exit;
	}
	/*	Push sentinel value to mark stack bottom.	*/
#ifndef WIN
	sym_stack[0] = node = (ExprNodeType *) malloc(sizeof(ExprNodeType));
#else
	sym_stack[0] = node = (ExprNodeType *) LocalAlloc(LMEM_FIXED, sizeof(ExprNodeType));
#endif
	if (!node) {
		Error(ERR_OUT_OF_MEMORY);
		goto error_exit;
	}
	node->TokClass = Q_DOLLAR;
	node->Operator = 0;
	node->Attribute = 0;
	node->ExprType = E_UNKNOWN;
	node->Name = 0;
	node->Llink = node->Rlink = 0;
	node->ArgList = 0;
	last = 0;
	while (1) {
		end_of_input = (byte) (Token == Q_IF || Token == Q_THEN ||
				Token == Q_ELSE || Token == Q_ENDIF || Token == Q_WHILE ||
				Token == Q_DO || Token == Q_ENDWHILE || Token == Q_FOR ||
				Token == Q_TO || Token == Q_DOWNTO || Token == Q_STEP ||
				Token == Q_ENDFOR || Token == Q_SWITCH || Token == Q_CASE ||
				Token == Q_DEFAULT || Token == Q_ENDCASE ||
				Token == Q_ENDSWITCH || Token == Q_GOTO || Token == Q_PROC ||
				Token == Q_ENDPROC || Token == Q_RETURN || Token == Q_VAR ||
				Token == Q_INTEGER || Token == Q_REAL || Token == Q_STRING ||
				Token == Q_SEMICOLON);
		/*	Get top terminal symbol on stack.	*/
		for (idx = last; idx >= 0; idx--) {
			top_stack = sym_stack[idx]->TokClass;
			if (ValidExprTerminal(top_stack))
				break;
		}
		if (top_stack == Q_DOLLAR && end_of_input)
			break;		/*	String accepted, done.	*/
		if (!end_of_input && !ValidExprTerminal(Token)) {
			if (error_count < MAX_EXP_ERRORS)
				error_list[error_count++] = ERR_INVALID_EXPR_TOKEN;
			goto error_exit;
		}
		token = Token;	/*	Get the input token.	*/
		/*	Constants are treated synonymous with id for precedence check.	*/
		if (Token == Q_INT_CONST || Token == Q_REAL_CONST ||
										Token == Q_STRING_CONST)
			token = Q_ID;
		if (top_stack == Q_INT_CONST || top_stack == Q_REAL_CONST ||
										top_stack == Q_STRING_CONST)
			top_stack = Q_ID;
		if (top_stack == Q_DOLLAR)
			precedence = L;	/*	shift.	*/
		else
			if (end_of_input)
				if (top_stack == Q_LP) /*	unmatched '(', error.	*/
					precedence = E1;
				else
					precedence = G;	/*	reduce.	*/
			else
				precedence = Precedence[top_stack][token];
		if (precedence == L || precedence == E) {	/*	Shift.	*/
#ifndef WIN
			node = (ExprNodeType *) malloc(sizeof(ExprNodeType));
#else
			node = (ExprNodeType *) LocalAlloc(LMEM_FIXED, sizeof(ExprNodeType));
#endif
			if (!node) {
				Error(ERR_OUT_OF_MEMORY);
				goto error_exit;
			}
			sym_stack[++last] = node;
			node->TokClass = Token;	/*	Put actual token.	*/
			node->Operator = 0;
			node->Attribute = 0;	/*	lvalue/rvalue not known.	*/
			if (Token == Q_ID || Token == Q_INT_CONST ||
						Token == Q_REAL_CONST || Token == Q_STRING_CONST) {
#ifndef WIN
				node->Name = malloc(strlen(TokenStr) + 1);
#else
				node->Name = (byte *) LocalAlloc(LMEM_FIXED, strlen(TokenStr) + 1);
#endif
				if (!node->Name) {
					Error(ERR_OUT_OF_MEMORY);
					goto error_exit;
				}
				strcpy(node->Name, TokenStr);
			}
			else
				node->Name = 0;
			node->Llink = 0;
			node->Rlink = 0;
			node->ArgList = 0;
			switch (Token) {
				case Q_INT_CONST :
					node->ExprType = E_INTEGER;
					break;
				case Q_REAL_CONST :
					node->ExprType = E_REAL;
					break;
				case Q_STRING_CONST :
					node->ExprType = E_STRING;
					break;
				default :
					node->ExprType = E_INVALID;
					break;
			}
			GetToken();
			if (Token == -1) {
				if (error_count < MAX_EXP_ERRORS)
					error_list[error_count++] = ERR_UNEXPECTED_EOF;
				goto error_exit;
			}
			continue;
		}
		if (precedence != G) {
			if (error_count < MAX_EXP_ERRORS) {
				switch (precedence) {
					case E1 : idx = ERR_EXPR_RP_EXPECTED; break;
					case E2 : idx = ERR_BIN_OP_RP_UOP_EXPECTED; break;
					case E3 : idx = ERR_BIN_OP_RP_LP_EXPECTED; break;
					case E4 : idx = ERR_BIN_OP_RP_ID_EXPECTED; break;
					case E5 : idx = ERR_BIN_OP_ID_ID_EXPECTED; break;
				}
				error_list[error_count++] = (byte) idx;
			}
			goto error_exit;	/*	Ill formed expression.	*/
		}
		idx = last;	/*	Initialize to stack top.	*/
		while (1) {	/*	Handle on top of stack, reduce.	*/
			for (; idx >= 0; idx--) {
				a = sym_stack[idx]->TokClass;
				if (ValidExprTerminal(a))
					break;
			}
			/*	a is the terminal to pop from stack.	*/
			new_top = idx;
			idx--;	/*	Pop the stack.	*/
			/*	Find the terminal on top of stack.	*/
			for (; idx >= 0; idx--) {
				b = sym_stack[idx]->TokClass;
				if (b == Q_DOLLAR || ValidExprTerminal(b))
					break;
			}
			/*	b is the terminal on top of stack.	*/
			/*	Constants are synonymous with id for precedence check.	*/
			if (a == Q_INT_CONST || a == Q_REAL_CONST || a == Q_STRING_CONST)
				a = Q_ID;
			if (b == Q_INT_CONST || b == Q_REAL_CONST || b == Q_STRING_CONST)
				b = Q_ID;
			if (b == Q_DOLLAR)
				precedence = L;
			else
				precedence = Precedence[b][a];
			if (precedence == L)
				break;	/*	Reduce handle on top of stack.	*/
		}
		/*	Now reduce symbols between new_top and last including any	*/
		/*	surrounding non-terminals.	*/
		for (idx = new_top; idx <= last; idx++) {
			token = sym_stack[idx]->TokClass;
			if (ValidExprTerminal(token))
				break;
		}
		if (token == Q_RP || !ValidExprTerminal(token)) {
			if (error_count < MAX_EXP_ERRORS)
				error_list[error_count++] = ERR_ILL_FORMED_EXPRESSION;
			goto error_exit;
		}
#ifndef WIN
		node = (ExprNodeType *) malloc(sizeof(ExprNodeType));
#else
		node = (ExprNodeType *) LocalAlloc(LMEM_FIXED, sizeof(ExprNodeType));
#endif
		if (!node) {
			Error(ERR_OUT_OF_MEMORY);
			goto error_exit;
		}
#ifndef WIN
		node->Name = malloc(MAX_TOKEN_LEN);
#else
		node->Name = (byte *) LocalAlloc(LMEM_FIXED, MAX_TOKEN_LEN);
#endif
		if (!node->Name) {
#ifndef WIN
			free((byte *) node);
#else
			LocalFree((ExprNodeType *) node);
#endif
			Error(ERR_OUT_OF_MEMORY);
			goto error_exit;
		}
		node->ArgList = 0;
		if (token == Q_ID && idx == last)
			strcpy(node->Name, sym_stack[idx]->Name);
		else
			GenerateTempVar(node->Name);
		node->TokClass = Q_EXPRESSION;
		switch (token) {
			case Q_ID :
				/*	Either id or a function call is being reduced.	*/
				if (idx != last && sym_stack[idx + 1]->TokClass != Q_LP) {
					if (error_count < MAX_EXP_ERRORS)
						error_list[error_count++] = ERR_ILL_FORMED_EXPRESSION;
					goto error_exit;
				}
				if (!(sym_ptr = FindSymbol(sym_stack[idx]->Name)))
					if (error_count < MAX_EXP_ERRORS)
						error_list[error_count++] = ERR_ID_NOT_DECLARED;
				if (idx == last) {	/*	id is being reduced.	*/
					if (sym_ptr->SymClass == Q_PROC) {
						node->Operator = Q_FN_CALL;
						node->Attribute = E_RVALUE;	/*	Only rvalue.	*/
						if (sym_ptr->ProcInfoPtr)
							expr_type = sym_ptr->ProcInfoPtr->RetValue;
						else
							expr_type = Q_VOID;	/*	Assume void type.	*/
					}
					else {
						node->Operator = Q_ID;
						/*	Expr has both lvalue and rvalue.	*/
						node->Attribute = (E_LVALUE | E_RVALUE);
						expr_type = sym_ptr->SymClass;
					}
					node->Llink = 0;
					node->Rlink = sym_stack[idx];
					sym_stack[idx] = node;
				}
				else {	/*	Fn call being reduced.	*/
					if (sym_ptr->SymClass != Q_PROC)
						if (error_count < MAX_EXP_ERRORS)
							error_list[error_count++] = ERR_ID_NOT_PROC;
					if (sym_ptr->ProcInfoPtr && !(sym_ptr->Status &
									(BUILT_IN_FUNCTION | OPTIONAL_ARGS)) &&
									!sym_ptr->ProcInfoPtr->ArgList &&
									error_count < MAX_EXP_ERRORS)
						error_list[error_count++] = ERR_FN_HAS_NO_ARGUMENTS;
					node->Operator = Q_FN_CALL;
					node->Attribute = E_RVALUE;	/*	Only rvalue.	*/
					if (sym_ptr->ProcInfoPtr)
						expr_type = sym_ptr->ProcInfoPtr->RetValue;
					else
						expr_type = Q_VOID;
					node->Llink = 0;
					node->Rlink = sym_stack[idx];
					ReleaseNodes(sym_stack[idx + 1]); /*	Free LP node.	*/
					sym_stack[idx + 1] = 0;	/*	Invalidate entry.	*/
					new_top = idx;
					sym_stack[new_top] = node;
					idx += 2;	/*	Point to first expression.	*/
					arg_ptr_ptr = &node->ArgList;
					while (1) {
						if (idx >= last) {
							if (error_count < MAX_EXP_ERRORS)
								error_list[error_count++] =
										ERR_ILL_FORMED_EXPRESSION;
							goto error_exit;
						}
						if (sym_stack[idx]->TokClass != Q_EXPRESSION) {
							if (error_count < MAX_EXP_ERRORS)
								error_list[error_count++] =
										ERR_ILL_FORMED_EXPRESSION;
							goto error_exit;
						}
						/*	Add the argument expr to list.	*/

#ifndef WIN
						arg_ptr = (CallArgType *) malloc(sizeof(CallArgType));
#else
						arg_ptr = (CallArgType *) LocalAlloc(LMEM_FIXED, sizeof(CallArgType));
#endif
						if (!arg_ptr) {
							Error(ERR_OUT_OF_MEMORY);
							goto error_exit;
						}
						arg_ptr->ExprPtr = sym_stack[idx];
						arg_ptr->NextArgPtr = 0;
						*arg_ptr_ptr = arg_ptr;
						arg_ptr_ptr = &arg_ptr->NextArgPtr;
						sym_stack[idx] = 0;	/*	Invalidate stack entry.	*/
						idx++;	/*	Should point to , or end of input.	*/
						if (idx == last)
							break;	/*	Reached top of stack.	*/
						if (sym_stack[idx]->TokClass == Q_COMMA) {
							/*	Release comma node here.	*/
							ReleaseNodes(sym_stack[idx]);
							sym_stack[idx] = 0;	/*	Invalidate stack entry.	*/
						}
						else {
							if (error_count < MAX_EXP_ERRORS)
								error_list[error_count++] =
										ERR_ILL_FORMED_EXPRESSION;
							goto error_exit;
						}
						idx++;	/*	Move to next expression.	*/
					}
					if (sym_stack[idx]->TokClass == Q_RP) {
						/*	Release RP node here.	*/
						ReleaseNodes(sym_stack[idx]);
						sym_stack[idx] = 0;	/*	Invalidate stack entry.	*/
					}
					else {
						/*	Some error in the algorithm !	*/
						if (error_count < MAX_EXP_ERRORS)
							error_list[error_count++] =
									ERR_ILL_FORMED_EXPRESSION;
						goto error_exit;
					}
					last = new_top;	/*	Reset the new stack top.	*/
				}
				if (sym_ptr->SymClass == Q_PROC)
					CheckFunctionArgs(sym_ptr, node->ArgList);
				switch (expr_type) {
					case Q_INTEGER :
						node->ExprType = E_INTEGER;
						break;
					case Q_REAL :
						node->ExprType = E_REAL;
						break;
					case Q_STRING :
						node->ExprType = E_STRING;
						break;
					case Q_VOID :
						node->ExprType = E_VOID;
						break;
					default :
						node->ExprType = E_INVALID;
						break;
				}
				break;
			case Q_INT_CONST :
			case Q_REAL_CONST :
			case Q_STRING_CONST :
				if (idx > last) {
					if (error_count < MAX_EXP_ERRORS)
						error_list[error_count++] = ERR_ILL_FORMED_EXPRESSION;
					goto error_exit;	/*	Sym stack consistent ?	*/
				}
				node->Operator = (byte) token;
				node->Attribute = E_RVALUE;	/*	Only rvalue.	*/
				switch (token) {
					case Q_INT_CONST :
						node->ExprType = E_INTEGER;
						break;
					case Q_REAL_CONST :
						node->ExprType = E_REAL;
						break;
					case Q_STRING_CONST :
						node->ExprType = E_STRING;
						break;
				}
				node->Llink = 0;
				node->Rlink = sym_stack[last];
				sym_stack[last] = node;
				break;
			case Q_NOT :
			case Q_UNARY_MINUS :
				if (idx + 1 != last ||
							sym_stack[idx + 1]->TokClass != Q_EXPRESSION) {
					if (error_count < MAX_EXP_ERRORS)
						error_list[error_count++] = ERR_ILL_FORMED_EXPRESSION;
					goto error_exit;	/*	Sym stack consistent ?	*/
				}
				node->Operator = (byte) token;
				node->Attribute = E_RVALUE;	/*	Only rvalue.	*/
				expr_type = sym_stack[idx + 1]->ExprType;
				switch (expr_type) {
					case E_INTEGER :
						node->ExprType = E_INTEGER;
						break;
					case E_REAL :
						if (token == Q_NOT)
							node->ExprType = E_INTEGER;
						else
							node->ExprType = E_REAL;
						break;
					case E_STRING :
						if (error_count < MAX_EXP_ERRORS)
							error_list[error_count++] = ERR_TYPE_CONFLICT;
						node->ExprType = E_INVALID;
						break;
					case E_VOID :
						if (error_count < MAX_EXP_ERRORS)
							error_list[error_count++] = ERR_VOID_IN_EXPR;
						node->ExprType = E_INVALID;
						break;
					default :
						node->ExprType = E_INVALID;
						break;
				}
				node->Llink = 0;
				node->Rlink = sym_stack[idx + 1];
									/*	Free operator node.	*/
#ifndef WIN
				free((byte *) sym_stack[idx]);
#else
				LocalFree((ExprNodeType *) sym_stack[idx]);
#endif
				sym_stack[idx] = node;
				sym_stack[idx + 1] = 0;	/*	Invalidate E stack entry.	*/
				last = idx;	/*	Reset the new stack top.	*/
				break;
			case Q_PLUS :
			case Q_MINUS :
			case Q_TIMES :
			case Q_DIV :
			case Q_LESS_OR_EQUAL :
			case Q_GREATER_OR_EQUAL :
			case Q_LESS :
			case Q_GREATER :
			case Q_IS_EQUAL :
			case Q_IS_NOT_EQUAL :
			case Q_AND :
			case Q_OR :
				if (idx <= 1 || idx + 1 != last ||
							sym_stack[idx - 1]->TokClass != Q_EXPRESSION ||
							sym_stack[idx + 1]->TokClass != Q_EXPRESSION) {
					if (error_count < MAX_EXP_ERRORS)
						error_list[error_count++] = ERR_ILL_FORMED_EXPRESSION;
					goto error_exit;	/*	Sym stack consistent ?	*/
				}
				node->Operator = (byte) token;
				node->Attribute = E_RVALUE;	/*	Only rvalue.	*/
				node->Llink = sym_stack[idx - 1];
				node->Rlink = sym_stack[idx + 1];
				if (node->Llink->ExprType != E_INVALID &&
									node->Rlink->ExprType != E_INVALID) {
					node->ExprType = (byte) FixExprType(token,
							node->Llink->ExprType, node->Rlink->ExprType);
					if (node->ExprType == E_INVALID &&
											error_count < MAX_EXP_ERRORS)
						error_list[error_count++] = ERR_TYPE_CONFLICT;
				}
				else
					node->ExprType = E_INVALID;
											/*	Free operator node.	*/
#ifndef WIN
				free((byte *) sym_stack[idx]);
#else
				LocalFree((ExprNodeType *) sym_stack[idx]);
#endif
				sym_stack[idx - 1] = node;
				/*	Invalidate the E node entries on stack.	*/
				sym_stack[idx] = sym_stack[idx + 1] = 0;
				last = idx - 1;	/*	Reset the new stack top.	*/
				break;
			case Q_LP :
				if (idx + 1 > last ||
							sym_stack[idx + 1]->TokClass != Q_EXPRESSION) {
					if (error_count < MAX_EXP_ERRORS)
						error_list[error_count++] = ERR_ILL_FORMED_EXPRESSION;
					goto error_exit;	/*	Sym stack consistent ?	*/
				}
				node->Operator = Q_LP;
				node->Attribute = sym_stack[idx + 1]->Attribute;
				node->ExprType = sym_stack[idx + 1]->ExprType;
				node->Llink = 0;
				node->Rlink = sym_stack[idx + 1];
				ReleaseNodes(sym_stack[idx]);	/*	Free LP node.	*/
				ReleaseNodes(sym_stack[idx + 2]);	/*	Free RP node.	*/
				sym_stack[idx] = node;
				sym_stack[idx + 1] = 0;	/*	Invalidate E stack entry.	*/
				sym_stack[idx + 2] = 0;	/*	Invalidate RP stack entry.	*/
				last = idx;	/*	Reset the new stack top.	*/
				break;
		}
	}
	if (error_count)
		goto error_exit;
	if (!last) {
		if (error_count < MAX_EXP_ERRORS)
			error_list[error_count++] = ERR_NULL_EXPR_GIVEN;
		goto error_exit;	/*	Sym stack consistent ?	*/
	}
	if (last == 1 && sym_stack[1]->TokClass == Q_EXPRESSION)
		*expr_ptr = sym_stack[1];
	else {
		*expr_ptr = 0;
		goto error_exit;
	}
	for (idx = 0; idx < SYM_STACK_SIZE; idx++) {
		if (idx != 1 && sym_stack[idx]) {
			ReleaseNodes(sym_stack[idx]);
			sym_stack[idx] = 0;
		}
	}
	return(0);

	error_exit :
	for (idx = 0; idx < SYM_STACK_SIZE; idx++) {
		if (sym_stack[idx]) {
			ReleaseNodes(sym_stack[idx]);
			sym_stack[idx] = 0;
		}
	}
	for (idx = 0; idx < (int) error_count; idx++)
		Error(error_list[idx]);
	/*	Skip until a FOLLOW(expression) or EOF occurs.	*/
	while (Token != Q_THEN && Token != Q_ELSE && Token != Q_DO &&
			Token != Q_SEMICOLON && Token != Q_TO && Token != Q_DOWNTO &&
			Token != Q_STEP && Token != -1)
		GetToken();
	*expr_ptr = 0;
	return(-1);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	ValidExprTerminal										*/
/*	Input		:	token, token to check for validity.						*/
/*	Output		:	true/false.												*/
/*	Synopsis	:	If input token is a valid symbol of expression, true	*/
/*						is returned, else false is returned.				*/
/*--------------------------------------------------------------------------*/

byte	ValidExprTerminal(int token)
{
	if (token >= Q_UNARY_MINUS && token <= Q_STRING_CONST)
		return(true);
	return(false);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	ReleaseNodes											*/
/*	Input		:	Pointer to root of expression tree.						*/
/*	Output		:	None.													*/
/*	Synopsis	:	Releases all the nodes in an expression tree.			*/
/*--------------------------------------------------------------------------*/

void	ReleaseNodes(ExprNodeType *node)
{
	CallArgType	*a_ptr1;
	CallArgType	*a_ptr2;

	if (node) {
		a_ptr1 = node->ArgList;
		while (a_ptr1) {
			ReleaseNodes(a_ptr1->ExprPtr);
			a_ptr2 = a_ptr1->NextArgPtr;
#ifndef WIN
			free((byte *) a_ptr1);
#else
			LocalFree((CallArgType *) a_ptr1);
#endif
			a_ptr1 = a_ptr2;
		}
		if (node->Llink)
			ReleaseNodes(node->Llink);
		if (node->Rlink)
			ReleaseNodes(node->Rlink);
		if (node->Name)
#ifndef WIN
			free(node->Name);
#else
			LocalFree((byte *) node->Name);
#endif
#ifndef WIN
		free((byte *) node);
#else
		LocalFree((ExprNodeType *) node);
#endif
	}
}

/*--------------------------------------------------------------------------*/
/*	Name		:	GenerateTempVar											*/
/*	Input		:	Pointer to temporary variable name.						*/
/*	Output		:	None.													*/
/*	Synopsis	:	Copies a temporary variable name.						*/
/*--------------------------------------------------------------------------*/

void	GenerateTempVar(byte *name_ptr)
{
	strcpy(name_ptr, "$T");
	name_ptr += 2;
	/*	Keep track of maximum no. of temporaries generated within a	*/
	/*	function to reserve space for the temporaries.	*/
	if (TotalTempCount < TempVarCount)
		TotalTempCount = TempVarCount;
	itoa(TempVarCount, name_ptr, 10);
	TempVarCount++;
}

/*--------------------------------------------------------------------------*/
/*	Name		:	FixExprType												*/
/*	Input		:	operator, binary operator,								*/
/*					e_type1, type of left side expression,					*/
/*					e_type2, type of right side expression.					*/
/*	Output		:	resultant expression type.								*/
/*	Synopsis	:	Given the binary operator and types of operands,		*/
/*						computes the resultant expression type.				*/
/*--------------------------------------------------------------------------*/

int		FixExprType(int operator, byte e_type1, byte e_type2)
{
	int		e_type;

	switch (operator) {
		case Q_PLUS : case Q_MINUS : case Q_TIMES : case Q_DIV :
			switch (e_type1) {
				case E_INTEGER :
					switch (e_type2) {
						case E_INTEGER : e_type = E_INTEGER; break;
						case E_REAL : e_type = E_REAL; break;
						default : e_type = E_INVALID; break;
					}
					break;
				case E_REAL :
					switch (e_type2) {
						case E_INTEGER : case E_REAL :
							e_type = E_REAL; break;
						default : e_type = E_INVALID; break;
					}
					break;
				default :
					e_type = E_INVALID; break;
					break;
			}
			break;
		case Q_LESS_OR_EQUAL : case Q_GREATER_OR_EQUAL :
		case Q_IS_EQUAL : case Q_IS_NOT_EQUAL :
		case Q_LESS : case Q_GREATER :
			switch (e_type1) {
				case E_INTEGER : case E_REAL :
					switch (e_type2) {
						case E_INTEGER : case E_REAL :
							e_type = E_INTEGER; break;
						default : e_type = E_INVALID; break;
					}
					break;
				case E_STRING :
					switch (e_type2) {
						case E_STRING : e_type = E_INTEGER; break;
						default : e_type = E_INVALID; break;
					}
					break;
				default :
					e_type = E_INVALID; break;
					break;
			}
			break;
		case Q_AND : case Q_OR :
			switch (e_type1) {
				case E_INTEGER :
					switch (e_type2) {
						case E_INTEGER : e_type = E_INTEGER; break;
						default : e_type = E_INVALID; break;
					}
					break;
				default :
					e_type = E_INVALID; break;
			}
			break;
	}
	return(e_type);
}
