/*---------------------------------------------------------------------------
*	File		:	EMLNRTNS.C
*	Purpose		:	Emulation funstions of terminal.
*	Package		:	MultiExpress (Windows) - Version 2.00
*	Authors		:	Vidy
*	Date		:	21st December 1991.
*---------------------------------------------------------------------------*/

#include	<windows.h>

#include	"main.h"
#include	"mew.h"
#include	"session.h"

#include	"emlntype.h"
#include	"emlncons.h"
#include	"emlndata.h"
#include	"emlnfuns.h"
#include	"windisp.h"
#include	"wstdio.h"
#include	"myprint.h"
#include	"status.h"
#include	"fonts.h"
#include	"mewlog.h"
#include	"kbd.h"
#include	"term.h"

BYTE	RecdChar;				// Current recived char being processed
BYTE	ScrollTopMargin;		// for Set scroll region command
BYTE	ScrollBottomMargin;		// for Set Scroll region command

BYTE	ForceFormFeed = FALSE;	// force a form feed at end of printing
extern	BYTE	ToAdvanceCursor;

/*--------------------------------------------------------------------------*/
/*	Name		:	Emulation												*/
/*	Input		:	buf, pointer to buffer									*/
/*					cnt, no. of characters in buffer.						*/
/*	Output	:	None													*/
/*	Synopsis	:	Performs terminal emulation.							*/
/*--------------------------------------------------------------------------*/
void Emulation (BYTE FAR *buf, WORD cnt)
{
	WORD idx ;
	BYTE continuation ;
	StateType *state_ptr ; // StateType in emlntype.h


//	if (!StateTable) 		// 7/1/98
	{		/*	Default dumb tty emulation. */
		while (cnt--) 
		{		// StateTable in emlndata.c
			RecdChar = *buf++ ;
			EchoChar() ;
		}
		return ;
	}

	while (cnt--) 
	{
		continuation = FALSE ;			/*	Force input char bounds check.	*/
		RecdChar = *buf ;
		while (TRUE) 
		{
			state_ptr = StateTable ;
			for (idx = 0; idx < NoOfStates; idx++, state_ptr++) 
			{
				// CurrentState = 0 in InitEmulation-> emuln.c
				if (state_ptr->State == CurrentState && (continuation ||
						(state_ptr->LoCh <= RecdChar &&
									state_ptr->HiCh >= RecdChar))) 
				{
					CurrentState = state_ptr->NextState;
					if (state_ptr->Hdlr)
						(*(state_ptr->Hdlr))();
					continuation = state_ptr->Transition;
					break;
				}
			}
			if (idx < NoOfStates) 
			{		/*	State found, reset.				*/
				if (!continuation)		/*	Check for a continuation state.	*/
					break ;
			}
			else 
			{		/*	State not found, reset.			*/
				CurrentState = 0;
				ResetArgs();
				break;					/*	Assume a break state.			*/
			}
		}
		buf++;
	}
}

/*--------------------------------------------------------------------------*/
/*	Name		:	InvalidateDisplay										*/
/*	Input		:	start and end indices of dispmem (inclusive)			*/
/*	Output		:	None													*/
/*	Synopsis	:	When display mem is invalid between startline and end	*/
/*					line this routine invalidates the corresponding are in	*/
/*					terminal window and calls update window to repaint.		*/
/*	NOTE		:	Startline must be <= Endline.							*/
/*--------------------------------------------------------------------------*/
void	InvalidateDisplay(BYTE Startline, BYTE Endline)
{
	RECT	rc;

	if ( (Endline >= WinOrgY) && (Startline <= (WinOrgY + MaxRowIndx))) {
		/* there is some area to be invalidated */
		/* get top value in character units */
		rc.top = (Startline <= WinOrgY) ? 0 : Startline - WinOrgY;

		/* get bottom value in character units */
		rc.bottom = (WinOrgY + MaxRowIndx <= Endline) ? MaxRowIndx + 1 :
							Endline - WinOrgY + 1;

		rc.left = 0;		/* left will always start from the first char	*/
		/* get right value in character units */
		rc.right = (WinOrgX + MaxClmIndx <= CharsInLine) ? MaxClmIndx + 1 :
							CharsInLine - WinOrgX + 1;
		/* convert all stuff into pixel units */
		rc.top *= nStdioCharHeight;
		rc.bottom *= nStdioCharHeight;
		rc.right *= nStdioCharWidth;

		InvalidateRect(hTermWnd, (LPRECT) &rc, FALSE);
		UpdateWindow(hTermWnd);
	}
}

/*--------------------------------------------------------------------------*/
/*	Name		:	ClearLines												*/
/*	Input		:	start and end indices of dispmem (inclusive)			*/
/*	Output		:	None													*/
/*	Synopsis	:	Called when lines from Startline and Endline are to be	*/
/*					cleared. It updates display window also.				*/
/*	NOTE		:	Startline must be <= Endline.							*/
/*--------------------------------------------------------------------------*/
void	ClearLines(BYTE Startline, BYTE Endline)
{
	register char *charptr, *attrptr;
	WORD	i, lineindx;

	for (lineindx = Startline; lineindx <= Endline; lineindx++) {
		charptr = GetStdioLine(lineindx);		/* get pointer to the line	*/
		attrptr = charptr + ScrnBufSize;		/* Get pointer to attribs	*/
		for (i = 0; i <= MaxClmIndx; i++) {
			*charptr++ = ' '; *attrptr++ = CurAttrib; /* reset char, attrib	*/
		}
	}
}

/*--------------------------------------------------------------------------*/
/*	Name		:	EmlnScrolWinDn											*/
/*	Input		:	Number of lines to go down, start line, end line		*/
/*	Output		:	None													*/
/*	Synopsis	:	Scrolls display data down, from startline to endline by	*/
/*					number of lines, and fills new lines with blank.		*/
/*--------------------------------------------------------------------------*/
void	EmlnScrollWinDn(BYTE nlines, BYTE Startline, BYTE Endline)
{
	BYTE	srcindx, dstindx;
	BYTE	*ptrdst, *ptrsrc, *attrptrdst, *attrptrsrc;
	BYTE	i, j;

	if (nlines >= (BYTE) (Endline - Startline + 1)) {
		/* all we have to do is to blank all possible lines inside, no copy */
		nlines = (BYTE) (Endline - Startline + 1);
		goto BlankLines;
	}

	/* First finish copying chars and attribs of lines shifted down */
	for (srcindx = Endline - nlines, dstindx = Endline;
			1;		/* do for ever, breaks when Startline is copied */
				dstindx--, srcindx--) {
		ptrdst = GetStdioLine(dstindx); ptrsrc = GetStdioLine(srcindx);
		attrptrdst = ptrdst + ScrnBufSize; attrptrsrc = ptrsrc + ScrnBufSize;
		for (i = 0; i <= MaxClmIndx; i++) {
			*ptrdst++ = *ptrsrc++; *attrptrdst++ = *attrptrsrc++;
		}
		if (srcindx == Startline)		// stop when src indx is start line
			break;
	}
	/* now blanken the new lines added */
BlankLines:
	for (i = 0, dstindx = Startline; i < nlines; i++, dstindx++) {
		ptrdst = GetStdioLine(dstindx);
		attrptrdst = ptrdst + ScrnBufSize;
		for (j = 0; j <= MaxClmIndx; j++) {
			*ptrdst++ = ' '; *attrptrdst++ = CurAttrib;
		}
	}
	/* redisplay the scrolled area */
	InvalidateDisplay(Startline, Endline);
	
}

/*--------------------------------------------------------------------------*/
/*	Name		:	EmlnScrolWinUp											*/
/*	Input		:	Number of lines to go down, start line, end line		*/
/*	Output		:	None													*/
/*	Synopsis	:	Scrolls display data up, from endline to startline by	*/
/*					number of lines, and fills new lines with blank.		*/
/*--------------------------------------------------------------------------*/
void	EmlnScrollWinUp(BYTE nlines, BYTE Startline, BYTE Endline)
{
	BYTE	srcindx, dstindx;
	BYTE	*ptrdst, *ptrsrc, *attrptrdst, *attrptrsrc;
	BYTE	i, j;

	if (nlines >= (BYTE) (Endline - Startline + 1)) {
		/* all we have to do is to blank all possible lines inside, no copy */
		nlines = (BYTE) (Endline - Startline + 1);
		goto BlankLines;
	}

	/* First finish copying chars and attribs of lines shifted down */
	for (dstindx = Startline, srcindx = Startline + nlines;
			srcindx <= Endline;
				dstindx++, srcindx++) {
		ptrdst = GetStdioLine(dstindx); ptrsrc = GetStdioLine(srcindx);
		attrptrdst = ptrdst + ScrnBufSize; attrptrsrc = ptrsrc + ScrnBufSize;
		for (i = 0; i <= MaxClmIndx; i++) {
			*ptrdst++ = *ptrsrc++; *attrptrdst++ = *attrptrsrc++;
		}
	}
BlankLines:
	/* now blanken the new lines added */
	for (i = 0, dstindx = Endline; i < nlines; i++, dstindx--) {
		ptrdst = GetStdioLine(dstindx);
		attrptrdst = ptrdst + ScrnBufSize;
		for (j = 0; j <= MaxClmIndx; j++) {
			*ptrdst++ = ' '; *attrptrdst++ = CurAttrib;
		}
	}
	/* redisplay the scrolled area */
	InvalidateDisplay(Startline, Endline);

}

/*--------------------------------------------------------------------------*/
/*	Name		:	ResetArgs												*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Resets the argument list.								*/
/*--------------------------------------------------------------------------*/

void	ResetArgs(void)
{
	while (EmlnArgCnt) {
		EmlnArgs[EmlnArgCnt] = 0;
		EmlnArgCnt--;
	}
	EmlnArgs[0] = 0;
}

/*--------------------------------------------------------------------------*/
/*	Name		:	EchoChar												*/
/*	Input		:	character to be echoed.									*/
/*	Output		:	None													*/
/*	Synopsis	:	Echoes the character given.								*/
/*--------------------------------------------------------------------------*/
void	EchoChar( void )
{
	BYTE	col;
	BYTE	len;

	RecdChar |= GraphicsAdd;
	if ( ! SuppressEcho) {
		switch (RecdChar) {
		case CR_ASCII :
			WriteChar(RecdChar); // +137 windisp.c
			if (PbkEntry.TrmParams.AddLFtoCR)	/*	CR/LF translation.		*/
				WriteChar(LF_ASCII);
			break;
		case LF_ASCII :
			if (!PbkEntry.TrmParams.StripLF)
				WriteChar(RecdChar);
			break;
		case TAB_ASCII :
			col = (BYTE) nCurX;
			len = (BYTE) (PbkEntry.TrmParams.TabStops - col %
												PbkEntry.TrmParams.TabStops);
			EmlnArgs[0] = len;
			CursorRight();
			break;
		case 0x0c :						/*	Form feed, clear and home.		*/
			SetCursorPosition(0);
			EmlnClearScreen();
			break;
		default :
			WriteChar(RecdChar);
			break;
		}
	}
	if (PrintLogging)
		PrintChar(RecdChar);
	if (FileLogging)
		WriteToLogFile();

	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	CursorUp												*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Moves the cursor up by one row.							*/
/*--------------------------------------------------------------------------*/

void	CursorUp(void)
{
	if (!EmlnArgs[0])
		EmlnArgs[0]++;
	if (EmlnArgs[0] >= (WORD)(nCurY + 1))
		nCurY = 0;
	else 
		nCurY -= (BYTE)EmlnArgs[0];
RemoveCursor();				// will redisplay after some time
DisplayCursor();
//	ReDrawCursor();
//ToAdvanceCursor = FALSE;
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	CursorDown												*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Moves the cursor down by one row.						*/
/*--------------------------------------------------------------------------*/

void	CursorDown(void)
{
	if (!EmlnArgs[0])
		EmlnArgs[0]++;
	nCurY += (BYTE)EmlnArgs[0];
	if (nCurY > MaxRowIndx)
		nCurY = MaxRowIndx;				/* can't be more than MaxRowIndx */
RemoveCursor();				// will redisplay after some time
DisplayCursor();
	//ReDrawCursor();
//ToAdvanceCursor = FALSE;
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	CursorRight												*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Moves the cursor right by one column.					*/
/*--------------------------------------------------------------------------*/

void	CursorRight(void)
{
	if (!EmlnArgs[0])
		EmlnArgs[0]++;
	nCurX += (BYTE)EmlnArgs[0];
	if (nCurX > MaxClmIndx)
		nCurX = MaxClmIndx;				/* can't be more than MaxClmIndx */
RemoveCursor();				// will redisplay after some time
DisplayCursor();
//	ReDrawCursor();
//ToAdvanceCursor = FALSE;
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	CursorLeft												*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Moves the cursor left by one column.					*/
/*--------------------------------------------------------------------------*/

void	CursorLeft(void)
{
	if (!EmlnArgs[0])
		EmlnArgs[0]++;
//	if (EmlnArgs[0] >= (WORD)(nCurX + 1))
	if (EmlnArgs[0] > (WORD) nCurX )
		nCurX = 0;
	else 
		nCurX -= (BYTE) EmlnArgs[0];
RemoveCursor();				// will redisplay after some time
DisplayCursor();
//	ReDrawCursor();
	ToAdvanceCursor = FALSE;
	ResetArgs();
}

/***************************************************************************
*	Routine	:	InHorizLine
*	Input	:	None
*	Return	:	TRUE or FALSE
*	Synopsis:	Checks if the display cursor postition line falls thru the
*				window. returns TRUE if so. Please note that this doesn't
*				necessarily mean that the cursor is visible 
***************************************************************************/
BOOL	InHorizLine (void)
{
	if ( (nCurY >= WinOrgY) && (nCurY < (WinOrgY + RowsInWin)) ) 
		return TRUE;
	return FALSE;
}

/*--------------------------------------------------------------------------*/
/*	Name		:	DelToBeginningOfLine									*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Erases characters from beginning of line to	current		*/
/*					cursor position (including cursor position).			*/
/*--------------------------------------------------------------------------*/

void	DelToBeginningOfLine(void)
{
	WORD	i;
	register	char *charptr, *attrptr;
	RECT	rcTmp;
	BYTE	cells;

	/* put blanks in the character cells and update the window and display */
	charptr = GetStdioLine(nCurY);
	attrptr = charptr + ScrnBufSize;
	
	for(i = 0; i <= nCurX; i++) {
		*charptr++ = ' ';					/* put space and inc pointer	*/
		*attrptr++ = CurAttrib;				/* normal at for erased cells	*/
	}

	if (InHorizLine() && (nCurX >= WinOrgX)) {
		/* There are some cells to be cleared */
		rcTmp.top = (nCurY - WinOrgY) * nStdioCharHeight;
		rcTmp.bottom = rcTmp.top + nStdioCharHeight; /* One char height */
		rcTmp.left = 0;								/* Clear from left most */
		cells = (BYTE) (nCurX - WinOrgX + 1);		/* total cells to go */
		cells = min (cells, ClmsInWin);			/* can't be more than this */
		rcTmp.right = cells * nStdioCharWidth;		/* convert to DC */

		InvalidateRect(hTermWnd, (LPRECT) &rcTmp, FALSE);
		UpdateWindow(hTermWnd);
	}

	ResetArgs();
}


/*--------------------------------------------------------------------------*/
/*	Name		:	DelToEndOfLine											*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Erases characters from current cursor position to end	*/
/*					of line (including cursor position).					*/
/*--------------------------------------------------------------------------*/

void	DelToEndOfLine(void)
{
	WORD	i;
	register	char *charptr, *attrptr;
	RECT	rcTmp;
	BYTE	cells;

	if (ToAdvanceCursor) {
		ToAdvanceCursor = FALSE;
		ResetArgs();
		return;
	}
	/* put blanks in the character cells and update the window and display */
	charptr = GetStdioLine(nCurY) + nCurX;
	attrptr = charptr + ScrnBufSize;

	for(i = nCurX; i <= MaxClmIndx; i++) {		/* last cell is left off */
		*charptr++ = ' ';						/* put space and inc pointer */
		*attrptr++ = CurAttrib;						/* put space and inc pointer */
	}

	if (InHorizLine() && (nCurX < (WinOrgX + ClmsInWin))) {
		/* There are some cells to be cleared */
		rcTmp.top = (nCurY - WinOrgY) * nStdioCharHeight;
		rcTmp.bottom = rcTmp.top + nStdioCharHeight; /* One char height */
		rcTmp.left = (WinOrgX >= nCurX) ? 0 : (nCurX - WinOrgX);
		rcTmp.left *= nStdioCharWidth;		/* Convert the units */

		/* Calculation of .right value */
		cells = min (ClmsInWin, (CharsInLine - WinOrgX));
		rcTmp.right = cells * nStdioCharWidth;

		InvalidateRect(hTermWnd, (LPRECT) &rcTmp, FALSE);
		UpdateWindow(hTermWnd);
	}
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	ClearScreen												*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Clears the screen. Attributes are reset to NormAttrib	*/
/*--------------------------------------------------------------------------*/

void	ClearScreen(void)
{
	register	WORD	i;

	for (i = 0; i < ScrnBufSize; i++) {
	    sScreenBuff[i] = ' ';			/* Set all the chars to blank */
		sAttribBuff[i] = CurAttrib;	/* Set attribute to Normal */
	}

	sDispOrig = sScreenBuff;			/* reset the display origin */
	WinOrgX = WinOrgY = 0;

	InvalidateRect (hTermWnd, NULL, FALSE);	/* Invalidate whole area */
	UpdateWindow(hTermWnd);		/* repaint the window */
	SetCursorPosition(0);			/* reset the X and Y position of cur */

	/* Update the Thumb positions */
	SetScrollPos (hTermWnd, SB_HORZ, WinOrgX, TRUE);
	SetScrollPos (hTermWnd, SB_VERT, WinOrgY, TRUE);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	EmlnClearScreen											*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Clears the screen. Attributes are set CurAttrib.		*/
/*--------------------------------------------------------------------------*/

void	EmlnClearScreen(void)
{

	ClearLines(0, MaxRowIndx);			/* This is faster way of clearing	*/
	InvalidateDisplay(0, MaxRowIndx);	/* redisp reset lines	*/

	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	ComputeArg												*/
/*	Input		:	input character.										*/
/*	Output		:	None													*/
/*	Synopsis	:	Compute the Arg based on input character.				*/
/*--------------------------------------------------------------------------*/

void	ComputeArg(void)
{
	EmlnArgs[EmlnArgCnt] = (BYTE) (EmlnArgs[EmlnArgCnt] * 10 + RecdChar - '0');
}

/*--------------------------------------------------------------------------*/
/*	Name		:	AdvanceArg												*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Advances arguemnt count to evauluate next arg.			*/
/*--------------------------------------------------------------------------*/

void	AdvanceArg(void)
{
	if (EmlnArgCnt < MAX_EMLN_ARGS - 1)
		EmlnArgCnt++;
}

/*--------------------------------------------------------------------------*/
/*	Name		:	EmlnSetCursor											*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Sets cursor at Row(in arg0) and Col(in arg1) .								*/
/*--------------------------------------------------------------------------*/

void	EmlnSetCursor(void)
{
	WORD	clm;
										/*	If params not given, assume 1.	*/
	if (!EmlnArgs[0])
		EmlnArgs[0]++;
	EmlnArgs[0] += (BYTE) (EmlnVT100originMode ? ScrollTopMargin : 0);
	clm = EmlnArgs[1];
	if (!clm)
		clm++;
	if (EmlnArgs[0] > LinesInScreen)
		EmlnArgs[0] = LinesInScreen;
	if (clm > CharsInLine)
		clm = CharsInLine;
	ToAdvanceCursor = FALSE;
	SetCursorPosition(((EmlnArgs[0] - 1) << 8) + (clm - 1));
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	ClearFromCursor											*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Clears from cursor position upto the end of screen.		*/
/*					This includes cursor position also						*/
/*--------------------------------------------------------------------------*/

void	ClearFromCursor(void)
{

	DelToEndOfLine();					/* Clear till end of current line	*/
	if (nCurY < MaxRowIndx) {
		ClearLines((BYTE) (nCurY + 1), MaxRowIndx);
		InvalidateDisplay((BYTE) (nCurY + 1), MaxRowIndx); /* redisp lines	*/
	}
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	BeginReverseVideo										*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Begins stand out mode.									*/
/*--------------------------------------------------------------------------*/
void	BeginReverseVideo(void)
{
	BYTE	fgcolor, bgcolor;

	fgcolor = (BYTE) (CurAttrib & 0x07);
	bgcolor = (BYTE) ((CurAttrib >> 4) & 0x07);

	/* Inter change the background color and foreground color */
	SetTermBgFg(TERM_COLOR_FG, bgcolor);
	SetTermBgFg(TERM_COLOR_BG, fgcolor);
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	EndReverseVideo											*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Ends stand out mode.									*/
/*--------------------------------------------------------------------------*/
void	EndReverseVideo(void)
{
	BeginReverseVideo();		/* will toggle the fg and bg colors */
}

/*--------------------------------------------------------------------------*/
/*	Name		:	ShowHexValue											*/
/*	Input		:	ch, input character.									*/
/*	Output		:	None													*/
/*	Synopsis	:	Shows the hex value of the input character.				*/
/*--------------------------------------------------------------------------*/
void	ShowHexValue(void)
{
	BYTE	c1, c2;

	c1 = RecdChar;

	c2 = (BYTE) (c1 / 16);
	RecdChar = (BYTE) (c2 < 10 ? c2 + '0' : c2 + 'A' - 10);
	EchoChar();
	c2 = (BYTE) (c1 % 16);
	RecdChar = (BYTE) (c2 < 10 ? c2 + '0' : c2 + 'A' - 10);
	EchoChar();
	RecdChar = c1;
}

/*--------------------------------------------------------------------------*/
/*	Name		:	EraseChar												*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Erases the character at the current cursor position by	*/
/*						overwriting it with a space.						*/
/*--------------------------------------------------------------------------*/

void	EraseChar(void)
{
	BYTE	*srcptr, *dstptr, *lstptr;
	RECT	rcTmp;
	BYTE	cells;

	if (!EmlnArgs[0])					/*	If no arg given, assume 1.		*/
		EmlnArgs[0]++;

	if (EmlnArgs[0] > (BYTE)(MaxClmIndx - nCurX + 1))	/* max meaningful value */
		EmlnArgs[0] = MaxClmIndx - nCurX + 1;


	lstptr = srcptr = dstptr = GetStdioLine(nCurY);
	dstptr += nCurX;
	lstptr += MaxClmIndx;
	srcptr = dstptr + EmlnArgs[0];

	for (;srcptr <= lstptr; srcptr++, dstptr++)
		*dstptr = *srcptr;

	while (EmlnArgs[0]--)
		*dstptr++ = ' ';

	if (InHorizLine() && (nCurX < (WinOrgX + ClmsInWin))) {
		/* There are some cells to be cleared */
		rcTmp.top = (nCurY - WinOrgY) * nStdioCharHeight;
		rcTmp.bottom = rcTmp.top + nStdioCharHeight; /* One char height */
		rcTmp.left = (WinOrgX >= nCurX) ? 0 : (nCurX - WinOrgX);
		rcTmp.left *= nStdioCharWidth;		/* Convert the units */

		/* Calculation of .right value */
		cells = min (ClmsInWin, (CharsInLine - WinOrgX));
		rcTmp.right = cells * nStdioCharWidth;

		InvalidateRect(hTermWnd, (LPRECT) &rcTmp, FALSE);
		UpdateWindow(hTermWnd);
	}
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	DisplayCharacter										*/
/*	Input		:	ch, input char											*/
/*	Output		:	None													*/
/*	Synopsis	:	Displays input character as is without translating.		*/
/*--------------------------------------------------------------------------*/

void	DisplayCharacter(void)
{
	DispChar(RecdChar, CurAttrib, 1);
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	AdvanceCursor											*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Advances the cursor. If at end of line moves to next	*/
/*						line. If at end of screen scrolls screen.			*/
/*--------------------------------------------------------------------------*/

void	AdvanceCursor(void)
{
	nCurX++;						/* increment cursor X position */
	if (nCurX >MaxClmIndx) {		/* has to wrap around */
		nCurX = 0;					/* reset to first column */
		nCurY++;					/* increment cursor Y position */
		if (nCurY > MaxRowIndx) {	/* needs to Scroll */
			nCurY--;				/* cursor remains in same Y Position */
			ScrlDispBuff();
		}
	}
//RemoveCursor();				// will redisplay after some time
DisplayCursor();
//	ReDrawCursor();					/* display cursor at new position */
}

/*--------------------------------------------------------------------------*/
/*	Name		:	EraseInLine												*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Erases line(s) from current line by overwriting with	*/
/*					blanks and sets cursor at beginning of current line.	*/
/*--------------------------------------------------------------------------*/
void	EraseInLine(void)
{
	BYTE	nrows, EndLine;

	if (RecdChar == 'K')
		EmlnArgs[0] = 1;	/*	Fixed for ANSI; ESC [ 2 K command.	*/
	else
		if (!EmlnArgs[0])
			EmlnArgs[0]++;
	nrows = (BYTE) (EmlnArgs[0] - 1);

	// compute the last line index
	EndLine = (MaxRowIndx <= nCurY + nrows) ?  MaxRowIndx : nCurY + nrows;

	ClearLines(nCurY, EndLine);
	InvalidateDisplay(nCurY, EndLine);	/* redisp reset lines	*/
//	nCurX = 0;						/* reset the cursor to the first column */
	ResetArgs();
//RemoveCursor();				// will redisplay after some time
DisplayCursor();
//	ReDrawCursor();					/* display cursor at new position */
}

/*--------------------------------------------------------------------------*/
/*	Name		:	BeginHighlight											*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	*/
/*--------------------------------------------------------------------------*/

void	BeginHighlight(void)
{
	CurAttrib |= 0x08;		/* turn on the highlight */
	SetTextColor(hTermDC, ColorTable[FORE_GROUND(CurAttrib)]);
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	EndHighlight											*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:															*/
/*--------------------------------------------------------------------------*/

void	EndHighlight(void)
{
	CurAttrib &= ~0x08;		/* turn on the highlight */
	SetTextColor(hTermDC, ColorTable[FORE_GROUND(CurAttrib)]);
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	BeginBlink												*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	*/
/*--------------------------------------------------------------------------*/

void	BeginBlink(void)
{
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	EndBlink												*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:															*/
/*--------------------------------------------------------------------------*/

void	EndBlink(void)
{
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	ClearTermBg												*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Clear the terminal area background.						*/
/*--------------------------------------------------------------------------*/
void	ClearTermBg(void)
{
	register	WORD	i;

	for (i = 0; i < ScrnBufSize; i++)
		sAttribBuff[i] = CurAttrib;		/* Set the attribute also */
}
/*--------------------------------------------------------------------------*/
/*	Name		:	SetToNormalAttribute									*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	THIS IS CALLED ONLY BY EMULATION MACHINE				*/
/*--------------------------------------------------------------------------*/

void	SetToNormalAttribute(void)
{
	/* Set all the attributes and colors to normal */
	CurAttrib = NormAttrib;

	/* Set the color of the DC */
	SetBkColor(hTermDC, ColorTable[BACK_GROUND(CurAttrib)]);
	SetTextColor(hTermDC, ColorTable[FORE_GROUND(CurAttrib)]);
	ResetArgs();
}
/*--------------------------------------------------------------------------*/
/*	Name		:	SetDislNormAtt
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:															*/
/*--------------------------------------------------------------------------*/

void	SetDispNormAtt(void)
{
	/* Set all the attributes and colors to normal */
	CurAttrib = NormAttrib;

	/* Set the color of the DC */
	SetBkColor(hTermDC, ColorTable[BACK_GROUND(CurAttrib)]);
	SetTextColor(hTermDC, ColorTable[FORE_GROUND(CurAttrib)]);
}

/*--------------------------------------------------------------------------*/
/*	Name		:	ClearUptoCursor											*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Clears screen from top to current position.				*/
/*					This includes current cursor position					*/
/*--------------------------------------------------------------------------*/

void	ClearUptoCursor(void)
{
	if (nCurY > 0) {
		ClearLines( 0, (BYTE) (nCurY - 1));				/* clear display	*/
		InvalidateDisplay(0, (BYTE) (nCurY - 1));	/* redisp reset lines	*/
	}
	DelToBeginningOfLine();
	ResetArgs();
}
/*--------------------------------------------------------------------------*/
/*	Name		:	SaveCursorPosition										*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Saves the current cursor position (not nestable).		*/
/*--------------------------------------------------------------------------*/

void	SaveCursorPosition(void)
{
	EmlnCursorPosition = GetCursorPosition();
	EmlnAttribute = CurAttrib;			/*	Save attribute.	*/
	EmlnVT100charSet = VT100charSet;		/*	Save character set.	*/
	EmlnVT100_G0charSet = VT100_G0charSet;	/*	Save G0 character set.	*/
	EmlnVT100_G1charSet = VT100_G1charSet;	/*	Save G1 character set.	*/
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	RestoreCursorPosition									*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Restores cursor position from the saved one.			*/
/*--------------------------------------------------------------------------*/

void	RestoreCursorPosition(void)
{
	SetCursorPosition(EmlnCursorPosition);
	CurAttrib = EmlnAttribute;			/*	Restore attribute.	*/
	VT100charSet = EmlnVT100charSet;		/*	Restore character set.	*/
	VT100_G0charSet = EmlnVT100_G0charSet;	/*	Restore G0 character set.	*/
	VT100_G1charSet = EmlnVT100_G1charSet;	/*	Restore G1 character set.	*/

SetTextColor(hTermDC, ColorTable[FORE_GROUND(CurAttrib)]);
SetBkColor(hTermDC, ColorTable[BACK_GROUND(CurAttrib)]);
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	SaveCursorType											*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Saves the current cursor type (not nestable).			*/
/*--------------------------------------------------------------------------*/

void	SaveCursorType(void)
{
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	RestoreCursorType										*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Restores current cursor type from the saved one.		*/
/*--------------------------------------------------------------------------*/

void	RestoreCursorType(void)
{
	ResetArgs();
}

/****************************************************************************
routine	:	RefreshScreen(Void)
Synopsis	:	Redraw the screen, after a change in colors
****************************************************************************/
void	RefreshScreen(void)
{
	InvalidateRect (hTermWnd, NULL, FALSE);	/* Invalidate whole area */
	UpdateWindow(hTermWnd);			/* repaint the window */
}

/*--------------------------------------------------------------------------*/
/*	Name		:	NewLine													*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Forces a new line in the screen.						*/
/*--------------------------------------------------------------------------*/

void	MoveToNextLine(void)
{
	WriteChar(LF_ASCII);
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	PrintScreen												*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Prints the screen.										*/
/*--------------------------------------------------------------------------*/

void	PrintScreen(void)
{
	BYTE	*ptr;
	BYTE	i, count;
	BOOL	print_was_on;

	/* If printer is not On call toggle Printer */
	print_was_on = PrintLogging;

	if (!print_was_on)			// start printer if not on
		if (!TogglePrinter())
			return;

	/* Now call the print buffer routine for all the lines */
	for (i = 0; i <= MaxRowIndx; i++) {
		ptr = GetStdioLine(i);
		for (count = 0; count < CharsInLine; count++)
			PrintChar(*ptr++);
		PrintChar(CR_ASCII);
		PrintChar(LF_ASCII);
	}
	if (!print_was_on)			// stop printer if it was not on
		TogglePrinter();
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	ComputeOffsetArg										*/
/*	Input		:	input character.										*/
/*	Output		:	None													*/
/*	Synopsis	:	Compute the Arg based on input character as offset.		*/
/*						Used in IBM 3101 emulation.							*/
/*--------------------------------------------------------------------------*/

void	ComputeOffsetArg(void)
{
	EmlnArgs[EmlnArgCnt] = (BYTE) (RecdChar - ' ' + 1);
}

/***************************************************************************
*	Routine	:	SetANSIcolors(void)
*	Input	:	
*	Return	:	
*	Synopsis:	Handles all the Esc[#m commands for ANSI color emulation
***************************************************************************/
void	SetANSIcolors(void)
{

	BYTE	idx;
	BYTE	fgcolor, bgcolor;
	WORD	arg;

	for (idx = 0; idx <= EmlnArgCnt; idx++) {
		arg = EmlnArgs[idx];
		if (arg <= 7) {
			switch (arg) {
				case 0 :
					SetDispNormAtt();
					break;
				case 1 :
					CurAttrib |= 0x08;		/* turn on the highlight */
					SetTextColor(hTermDC, ColorTable[FORE_GROUND(CurAttrib)]);
//					BeginBold();
					break;
				case 5 :
//					BeginUnderScore();
					break;
				case 7 :
					fgcolor = (BYTE) (CurAttrib & 0x07);
					bgcolor = (BYTE) ((CurAttrib >> 4) & 0x07);

					/* Inter change the background color and foreground color */
					SetTermBgFg(TERM_COLOR_FG, bgcolor);
					SetTermBgFg(TERM_COLOR_BG, fgcolor);
					break;
			}
		}
		else {
			switch (arg) {
				case 30 :			/*	Black foreground.				*/
				case 31 :			/*	Red foreground.					*/
				case 32 :			/*	Green foreground.				*/
				case 33 :			/*	Yellow foreground.				*/
				case 34 :			/*	Blue foreground.				*/
				case 35 :			/*	Magneta foreground.				*/
				case 36 :			/*	Cyan foreground.				*/
				case 37 :			/*	White foreground.				*/
					SetTermBgFg (TERM_COLOR_FG, (BYTE)(arg - 30));
					break;
						
				case 40 :			/*	Black background.				*/
				case 41 :			/*	Red background.					*/
				case 42 :			/*	Green background.				*/
				case 43 :			/*	Yellow background.				*/
				case 44 :			/*	Blue background.				*/
				case 45 :			/*	Magneta background.				*/
				case 46 :			/*	Cyan background.				*/
				case 47 :			/*	White background.				*/
					SetTermBgFg (TERM_COLOR_BG, (BYTE)(arg - 40));
					break;
			}
		}
	}
	ResetArgs();
}

/***************************************************************************
*	Routine	:	SetTermBgfg()
*	Input	:	
*	Return	:	
*	Synopsis:	Sets the background and fore ground colors of the terminal
***************************************************************************/
void	SetTermBgFg(BYTE flag, BYTE color_index)
{
	if (flag == TERM_COLOR_FG) {
		CurAttrib &= 0xF8;
		CurAttrib |= color_index;
		SetTextColor(hTermDC, ColorTable[FORE_GROUND(CurAttrib)]);

	} else {		/* back ground color change */
		CurAttrib &= 0x8F;
		CurAttrib |= (color_index << 4);
		SetBkColor(hTermDC, ColorTable[BACK_GROUND(CurAttrib)]);
	}
}

/***************************************************************************
 *	Routine	:	TurnOnPrintRedirection
 *	Input	:	None
 *	Return	:	None
 *	Synopsis:	Turns on print in terminal mode.
 ***************************************************************************/
void
TurnOnPrintRedirection (void)
{
	SuppressEcho = TRUE;
	if ( ! PrintLogging)
		if ( !TogglePrinter())
			return;
	PrintLogging = TRUE;
	ResetArgs();
}

/***************************************************************************
 *	Routine	:	TurnOffPrintRedirection
 *	Input	:	None
 *	Return	:	None
 *	Synopsis:	Turns off print in terminal mode.
 ***************************************************************************/
void
TurnOffPrintRedirection (void)
{
	SuppressEcho = FALSE;
	if ( PrintLogging)
		if ( ! TogglePrinter())
			return;
	PrintLogging = FALSE;
	ResetArgs();
}

/***************************************************************************
 *	Routine	:	TurnOnNumlock
 *	Input	:	None
 *	Return	:	None
 *	Synopsis:	Turns on numlock in terminal mode.
 ***************************************************************************/
void
TurnOnNumlock (void)
{
	HandleKbdLeds(LED_NUMLOCK, LIGHT_ON);	
	ResetArgs();
}

/***************************************************************************
 *	Routine	:	TurnOffNumlock
 *	Input	:	None
 *	Return	:	None
 *	Synopsis:	Turns off numlock in terminal mode.
 ***************************************************************************/
void
TurnOffNumlock (void)
{
	HandleKbdLeds(LED_NUMLOCK, LIGHT_OFF);	
	ResetArgs();
}

/***************************************************************************
 *	Routine	:	InsertLine
 *	Input	:	None
 *	Return	:	None
 *	Synopsis:	Inserts line(s) before the current line. Pushes from current
 *				line so many lines down.
 ***************************************************************************/
void
InsertLine (void)
{
	if (nCurY < ScrollTopMargin || nCurY > ScrollBottomMargin) {
		ResetArgs(); return;		// return if cursor not in scroll window
	}
	if (!EmlnArgs[0])				/* Assume 1 if no arguments are given */
		EmlnArgs[0]++;

	EmlnScrollWinDn ((BYTE) EmlnArgs[0], nCurY, ScrollBottomMargin);

	nCurX = 0;			/* put cursor on 0th column of the current line		*/
//RemoveCursor();				// will redisplay after some time
DisplayCursor();
//	ReDrawCursor();
	ResetArgs();
}

/***************************************************************************
 *	Routine	:	EmlnScrollUp
 *	Input	:	None
 *	Return	:	None
 *	Synopsis:	Scrolls lines after the current line.
 ***************************************************************************/
void
EmlnScrollUp (void)
{

	if (!EmlnArgs[0])		/* Assume 1 if no arguments are given */
		EmlnArgs[0]++;

	EmlnScrollWinUp ((BYTE) EmlnArgs[0], ScrollTopMargin, ScrollBottomMargin);

	ResetArgs();
}

/***************************************************************************
 *	Routine	:	TurnOnGraphics
 *	Input	:	None
 *	Return	:	None
 *	Synopsis:	Turns on ANSI graphics. Sets the eigth bit from the
 *				following characters before showing them.
 ***************************************************************************/
void
TurnOnGraphics (void)
{
	GraphicsAdd = 0x80;
	ResetArgs();
}

/***************************************************************************
 *	Routine	:	TurnOffGraphics
 *	Input	:	None
 *	Return	:	None
 *	Synopsis:	Turns off graphics in terminal mode.
 ***************************************************************************/
void
TurnOffGraphics (void)
{
	GraphicsAdd = 0;
	ResetArgs();
}

/***************************************************************************
 *	Routine	:	InsertChar
 *	Input	:	None
 *	Return	:	None
 *	Synopsis:	Inserts character at the current cursor position by moving
 *				characters after the cursor to the right.
 ***************************************************************************/
void
InsertChar (void)
{
	BYTE	*srcptr, *dstptr, *lstptr;
	RECT	rcTmp;
	BYTE	cells;

	if (!EmlnArgs[0])		/* Assume 0 if no arguments are given */
		EmlnArgs[0]++;
	if (EmlnArgs[0] > (BYTE)(MaxClmIndx - nCurX + 1))	/* max meaningful value */
		EmlnArgs[0] = MaxClmIndx - nCurX + 1;

	lstptr = srcptr = dstptr = GetStdioLine(nCurY);
	lstptr += nCurX;
	dstptr += MaxClmIndx;
	srcptr = dstptr - EmlnArgs[0];

	for (;lstptr <= srcptr; srcptr--, dstptr--)
		*dstptr = *srcptr;
	while (EmlnArgs[0]--)
		*lstptr++ = ' ';

	if (InHorizLine() && (nCurX < (WinOrgX + ClmsInWin))) {
		/* There are some cells to be cleared */
		rcTmp.top = (nCurY - WinOrgY) * nStdioCharHeight;
		rcTmp.bottom = rcTmp.top + nStdioCharHeight; /* One char height */
		rcTmp.left = (WinOrgX >= nCurX) ? 0 : (nCurX - WinOrgX);
		rcTmp.left *= nStdioCharWidth;		/* Convert the units */

		/* Calculation of .right value */
		cells = min (ClmsInWin, (CharsInLine - WinOrgX));
		rcTmp.right = cells * nStdioCharWidth;

		InvalidateRect(hTermWnd, (LPRECT) &rcTmp, FALSE);
		UpdateWindow(hTermWnd);
	}

	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	ANSIdeviceStatReq										*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Sends response for the ANSI device status request.		*/
/*--------------------------------------------------------------------------*/

void	ANSIdeviceStatReq(void)
{
	WORD	position;
	WORD	len;
	BYTE	str[10];	/*	Enough for the response string.	*/

	/*	Response string is : ESC [ row ; col R	*/
	/*	row and col are 1 relative.	*/
	position = GetCursorPosition();
	strcpy(str, "\x1b[");
	//	itoa((position >> 8) + 1, str + 2, 10);
	wsprintf(str+2, "%d", (position >> 8) + 1);
	len = strlen(str);
	str[len] = ';';
	//	itoa(((position + 1) & 0x00ff), &str[len + 1], 10);
	wsprintf(&str[len+1],"%d",(position + 1) & 0xff);

	len = strlen(str);
	str[len++] = 'R';
	str[len] = 0;
	WriteNchars( (LPSTR) str, len);	// write to port, no echo
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	VT100terminalStatReq									*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Sends response for the VT100 terminal status request.	*/
/*--------------------------------------------------------------------------*/

void	VT100terminalStatReq(void)
{
	static	BYTE	VT100termStatResp[] = "\x1b[0n";

	/*	Response string is : ESC [ 0 n , write to port, no echo	*/
	WriteNchars( (LPSTR) VT100termStatResp, strlen(VT100termStatResp));
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	VT100deviceQuery										*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Sends response for the VT100 device status request.		*/
/*--------------------------------------------------------------------------*/

void	VT100deviceQuery(void)
{
	static	BYTE	VT100deviceStatResp[] = "\x1b[?1;2c";

	/*	Response string is : ESC [ ? 1 ; 2 c, write to port no echo	*/
	WriteNchars((LPSTR)VT100deviceStatResp, strlen(VT100deviceStatResp));
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	VT100printerStatReq										*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Sends response for the VT100 printer status request.	*/
/*--------------------------------------------------------------------------*/

void	VT100printerStatReq(void)
{
	static	BYTE	VT100prnStatResp[] = "\x1b[?11n";
	int			ready;

	/*	Response string is : ESC [ ? 1 0 n if printer ready,	*/
	/*	else ESC [ ? 1 1 n.	*/
	ready = (PrintLogging) ? 1 : 0;
	VT100prnStatResp[4] = (BYTE) ('1' - ready);

	WriteNchars( (LPSTR)VT100prnStatResp, strlen(VT100prnStatResp));
			// write to port no echo
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	MoveToPrevLine											*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Moves cursor to previous line in same column. If cursor	*/
/*						is already at first line screen is scrolled down	*/
/*						and cursor is not moved (physically).				*/
/*--------------------------------------------------------------------------*/

void	MoveToPrevLine(void)
{
	if (nCurY == ScrollTopMargin) {
		EmlnScrollWinDn(1, ScrollTopMargin, ScrollBottomMargin);
	} else {
		nCurY--;
//RemoveCursor();				// will redisplay after some time
DisplayCursor();
//		ReDrawCursor();
	}
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	VT100printControl										*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Turns the printing on/off depending on the argument.	*/
/*						Turn on printing  : ESC [ ? 5 i						*/
/*						Turn off printing : ESC [ ? 4 i						*/
/*--------------------------------------------------------------------------*/

void	VT100printControl(void)
{
	switch (EmlnArgs[0]) {
		case 4 :
			if (ForceFormFeed)		/* force form feed, before turing off	*/
				PrintChar(0x0C);
			if ( PrintLogging)
				if ( !TogglePrinter())
					break;
			PrintLogging = FALSE;
			break;
		case 5 :
			if ( ! PrintLogging)	/* turn print on if not on */
				if ( !TogglePrinter())
					break;
			PrintLogging = TRUE;
			break;
	}
	ResetArgs();
}

#define	CUR_UP				0x4800		/*	Cursor up.		*/
#define	CUR_DN				0x5000		/*	Cursor down.	*/
#define	CUR_LEFT			0x4b00		/*	Cursor left.	*/
#define	CUR_RIGHT			0x4d00		/*	Cursor right.	*/

#define	GREY_CUR_UP			0x48e0		/*	Grey cursor up.		*/
#define	GREY_CUR_DN			0x50e0		/*	Grey cursor down.	*/
#define	GREY_CUR_LEFT		0x4be0		/*	Grey cursor left.	*/
#define	GREY_CUR_RIGHT		0x4de0		/*	Grey cursor right.	*/
	
/*--------------------------------------------------------------------------*/
/*	Name		:	SetVT100termParams										*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Sets VT100 terminal emulation parameters.				*/
/*		ESC [ ? 1 h  : select cursor keys to generate control functions.	*/
/*		ESC [ ? 3 h  : set screen to 132 column mode.						*/
/*		ESC [ ? 5 h  : set screen background to reverse video.				*/
/*		ESC [ ? 6 h  : set origin mode to scroll top margin.				*/
/*		ESC [ ? 7 h  : turn on auto wrap.									*/
/*		ESC [ ? 18 h : force form feed on print termination.				*/
/*--------------------------------------------------------------------------*/

void	SetVT100termParams(void)
{
	#define	CTRL_SEQ_UP		"\x1bOA"
	#define	CTRL_SEQ_DN		"\x1bOB"
	#define	CTRL_SEQ_RIGHT	"\x1bOC"
	#define	CTRL_SEQ_LEFT	"\x1bOD"

	BYTE	idx;
	BYTE	*expn;

	idx = 0;
	do {
		switch (EmlnArgs[idx]) {
			case 1 :	/*	Select control functions.	*/
				/*	If an expansion defined, and of same length	*/
				/*	overwrite it. Not a clean way, but...	*/
				if ((expn = GetExpansion(CUR_UP, KeyExpansion)) &&
							strlen(expn) == sizeof(CTRL_SEQ_UP) - 1)
					strcpy(expn, CTRL_SEQ_UP);
				if ((expn = GetExpansion(GREY_CUR_UP, KeyExpansion)) &&
							strlen(expn) == sizeof(CTRL_SEQ_UP) - 1)
					strcpy(expn, CTRL_SEQ_UP);
				if ((expn = GetExpansion(CUR_DN, KeyExpansion)) &&
							strlen(expn) == sizeof(CTRL_SEQ_DN) - 1)
					strcpy(expn, CTRL_SEQ_DN);
				if ((expn = GetExpansion(GREY_CUR_DN, KeyExpansion)) &&
							strlen(expn) == sizeof(CTRL_SEQ_DN) - 1)
					strcpy(expn, CTRL_SEQ_DN);
				if ((expn = GetExpansion(CUR_LEFT, KeyExpansion)) &&
							strlen(expn) == sizeof(CTRL_SEQ_LEFT) - 1)
					strcpy(expn, CTRL_SEQ_LEFT);
				if ((expn = GetExpansion(GREY_CUR_LEFT, KeyExpansion)) &&
							strlen(expn) == sizeof(CTRL_SEQ_LEFT) - 1)
					strcpy(expn, CTRL_SEQ_LEFT);
				if ((expn = GetExpansion(CUR_RIGHT, KeyExpansion)) &&
							strlen(expn) == sizeof(CTRL_SEQ_RIGHT) - 1)
					strcpy(expn, CTRL_SEQ_RIGHT);
				if ((expn = GetExpansion(GREY_CUR_RIGHT, KeyExpansion)) &&
							strlen(expn) == sizeof(CTRL_SEQ_RIGHT) - 1)
					strcpy(expn, CTRL_SEQ_RIGHT);
				break;
			case 3 :	/*	Switch to 132 column mode.	*/
				ChangeScreenWidth(132);
//				UpdateStatusBar(COL_FLD_IDX);
				break;
			case 5 :	/*	Set screen background to reverse video.	*/
			#if 0
				ptr = sAttribBuff;
				for (i = 0; i < ScrnBufSize; i++)
					*ptr++ = 0x70;
				InvalidateDisplay(0, MaxRowIndx);
			#endif
				break;
			case 6 :	/*	Set home position in scrolling region.	*/
				EmlnVT100originMode = 1;
						/*	Set cursor at home position.	*/
				SetCursorPosition(ScrollTopMargin << 8);
				break;
			case 7 :	/*	Turn on auto wrap.	*/
				PbkEntry.TrmParams.AutoMargin = TRUE;
				break;
			case 18 :	/*	Force form feed on print termination.	*/
				ForceFormFeed = TRUE;
				break;
		}
		idx++;	/*	Move onto next argument.	*/
	} while (idx <= EmlnArgCnt);
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	ResetVT100termParams									*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Sets VT100 terminal emulation parameters.				*/
/*		ESC [ ? 1 l  : select cursor keys to generate ANSI saequences.		*/
/*		ESC [ ? 3 l  : set screen to 80 column mode.						*/
/*		ESC [ ? 5 l  : set screen background to normal video.				*/
/*		ESC [ ? 6 h  : set origin mode to screen top.						*/
/*		ESC [ ? 7 l  : turn off auto wrap.									*/
/*		ESC [ ? 18 l : no form feed on print termination.					*/
/*--------------------------------------------------------------------------*/

void	ResetVT100termParams(void)
{
	#define	ANSI_SEQ_UP		"\x1b[A"
	#define	ANSI_SEQ_DN		"\x1b[B"
	#define	ANSI_SEQ_RIGHT	"\x1b[C"
	#define	ANSI_SEQ_LEFT	"\x1b[D"

	BYTE	idx;
	BYTE	*expn;

	idx = 0;
	do {
		switch (EmlnArgs[idx]) {
			case 1 :	/*	Select ANSI sequences.	*/
				/*	If an expansion defined, and of same length	*/
				/*	overwrite it. Not a clean way, but...	*/
				if ((expn = GetExpansion(CUR_UP, KeyExpansion)) &&
							strlen(expn) == sizeof(ANSI_SEQ_UP) - 1)
					strcpy(expn, ANSI_SEQ_UP);
				if ((expn = GetExpansion(GREY_CUR_UP, KeyExpansion)) &&
							strlen(expn) == sizeof(ANSI_SEQ_UP) - 1)
					strcpy(expn, ANSI_SEQ_UP);
				if ((expn = GetExpansion(CUR_DN, KeyExpansion)) &&
							strlen(expn) == sizeof(ANSI_SEQ_DN) - 1)
					strcpy(expn, ANSI_SEQ_DN);
				if ((expn = GetExpansion(GREY_CUR_DN, KeyExpansion)) &&
							strlen(expn) == sizeof(ANSI_SEQ_DN) - 1)
					strcpy(expn, ANSI_SEQ_DN);
				if ((expn = GetExpansion(CUR_LEFT, KeyExpansion)) &&
							strlen(expn) == sizeof(ANSI_SEQ_LEFT) - 1)
					strcpy(expn, ANSI_SEQ_LEFT);
				if ((expn = GetExpansion(GREY_CUR_LEFT, KeyExpansion)) &&
							strlen(expn) == sizeof(ANSI_SEQ_LEFT) - 1)
					strcpy(expn, ANSI_SEQ_LEFT);
				if ((expn = GetExpansion(CUR_RIGHT, KeyExpansion)) &&
							strlen(expn) == sizeof(ANSI_SEQ_RIGHT) - 1)
					strcpy(expn, ANSI_SEQ_RIGHT);
				if ((expn = GetExpansion(GREY_CUR_RIGHT, KeyExpansion)) &&
							strlen(expn) == sizeof(ANSI_SEQ_RIGHT) - 1)
					strcpy(expn, ANSI_SEQ_RIGHT);
				break;
			case 3 :	/*	Switch to 80 column mode.	*/
				ChangeScreenWidth (80);
//				UpdateStatusBar(COL_FLD_IDX);
				break;
			case 5 :	/*	Set screen background to normal video.	*/
			#if 0
				/*	If we support this it does not work with the vttest	*/
				/*	program on the unix host.	*/
				ptr = sAttribBuff;
				for (i = 0; i < ScrnBufSize; i++)
					*ptr++ = NormAttrib;
				InvalidateDisplay(0, MaxRowIndx);
			#endif
				break;
			case 6 :	/*	Set home position to top left corner.	*/
				EmlnVT100originMode = 0;
				SetCursorPosition(0);	/*	Set cursor at top left.	*/
				break;
			case 7 :	/*	Turn off auto wrap.	*/
				PbkEntry.TrmParams.AutoMargin = FALSE;
				break;
			case 18 :	/*	No form feed on print termination.	*/
				ForceFormFeed = FALSE;
				break;
		}
		idx++;	/*	Move onto next argument.	*/
	} while (idx <= EmlnArgCnt);
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	SetVT100LocalEcho										*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Turns the printing on/off depending on the argument.	*/
/*						Turn on local echo  : ESC [ 1 2 h					*/
/*						Turn off local echo : ESC [ 1 2 l					*/
/*--------------------------------------------------------------------------*/

void	SetVT100LocalEcho(void)
{
	if (EmlnArgs[0] == 12) {
		PbkEntry.TrmParams.EchoOn = TRUE;
//		UpdateStatusBar(ECHO_FLD_IDX);
	}
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	ResetVT100LocalEcho										*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Turns the printing on/off depending on the argument.	*/
/*						Turn on local echo  : ESC [ 1 2 h					*/
/*						Turn off local echo : ESC [ 1 2 l					*/
/*--------------------------------------------------------------------------*/

void	ResetVT100LocalEcho(void)
{
	if (EmlnArgs[0] == 12) {
		PbkEntry.TrmParams.EchoOn = FALSE;
//		UpdateStatusBar(ECHO_FLD_IDX);
	}
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	SetVT100CharSet_0										*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Sets the VT100 G0 character set.						*/
/*						UK character set as G0  : ESC ( A					*/
/*						US character set as G0 : ESC ( B					*/
/*						Special character set as G0 : ESC ( 0				*/
/*--------------------------------------------------------------------------*/

void	SetVT100CharSet_0(void)
{
	switch (RecdChar) {
		case 'A' :	/*	Select UK character set.	*/
			VT100_G0charSet = VT100_UKcharSet;
			break;
		case 'B' :	/*	Select US character set.	*/
			VT100_G0charSet = VT100_UScharSet;
			break;
		case '0' :	/*	Select special character set.	*/
			VT100_G0charSet = VT100_SpecialCharSet;
			break;
	}
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	SetVT100CharSet_1										*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Sets the VT100 G1 character set.						*/
/*						UK character set as G1  : ESC ) A					*/
/*						US character set as G1 : ESC ) B					*/
/*						Special character set as G1 : ESC ) 0				*/
/*--------------------------------------------------------------------------*/

void	SetVT100CharSet_1(void)
{
	switch (RecdChar) {
		case 'A' :	/*	Select UK character set.	*/
			VT100_G1charSet = VT100_UKcharSet;
			break;
		case 'B' :	/*	Select US character set.	*/
			VT100_G1charSet = VT100_UScharSet;
			break;
		case '0' :	/*	Select special character set.	*/
			VT100_G1charSet = VT100_SpecialCharSet;
			break;
	}
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	SelectVT100CharSet_0									*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Selects VT100 G0 char set G0 as active character set.	*/
/*--------------------------------------------------------------------------*/

void	SelectVT100CharSet_0(void)
{
	VT100charSet = 0;
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	SelectVT100CharSet_1									*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Selects VT100 G1 char set G0 as active character set.	*/
/*--------------------------------------------------------------------------*/

void	SelectVT100CharSet_1(void)
{
	VT100charSet = 1;
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	VT100EchoChar											*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Echoes the Vt100 character given.						*/
/*--------------------------------------------------------------------------*/

void	VT100EchoChar(void)
{
	BYTE	col, len;


	if (RecdChar < 0x80)	/*	Only chars 0 - 0x7f are translated.	*/
		if (!VT100charSet)
			RecdChar = VT100_G0charSet[RecdChar];
		else
			RecdChar = VT100_G1charSet[RecdChar];

	if (!RecdChar)
		return;

	if (!SuppressEcho) {
		switch (RecdChar) {
			case CR_ASCII :
				WriteChar(RecdChar);
				if (PbkEntry.TrmParams.AddLFtoCR)		//	CR/LF translation.	
					WriteChar(LF_ASCII);
				break;
			case LF_ASCII :
				if (!PbkEntry.TrmParams.StripLF)
					WriteChar(RecdChar);
				break;
			case TAB_ASCII :
				col = (BYTE) nCurX;
				len = (BYTE) (PbkEntry.TrmParams.TabStops - col %
											PbkEntry.TrmParams.TabStops);
				EmlnArgs[0] = len;
				if (EmlnVT100insertMode)
					InsertChar();
				EmlnArgs[0] = len;
				CursorRight();
				break;
			case 0x0c :						/*	Form feed, clear and home.		*/
				SetCursorPosition(0);
				EmlnClearScreen();
				break;
			default :
				/*	If insert mode is set, create space by moving the	*/
				/*	characters to the right and write the character.	*/
				if (EmlnVT100insertMode) {
					EmlnArgs[0] = 1;
					InsertChar();
				}
				WriteChar(RecdChar);
				break;
		}
	}
	if (PrintLogging)
		PrintChar(RecdChar);
	if (FileLogging)
		WriteToLogFile();
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	VT100cancelSeq											*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Echoes the Vt100 character given.						*/
/*--------------------------------------------------------------------------*/

void	VT100cancelSeq(void)
{
	WriteChar(0xb0);	/*	VT100 cancel character.	*/
	ResetArgs();
	CurrentState = 0;
}

/*--------------------------------------------------------------------------*/
/*	Name		:	DeleteLine												*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Erases lines from the active line and scrolls up by		*/
/*						the number of lines deleted.						*/
/*--------------------------------------------------------------------------*/

void	DeleteLine(void)
{
	
	if (!EmlnArgs[0])
		EmlnArgs[0]++;
	EmlnScrollWinUp ((BYTE) EmlnArgs[0], ScrollTopMargin, ScrollBottomMargin);
	nCurX = 0;
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	SetVT100scrollRegion									*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Sets the scroll region for VT100 emulation.				*/
/* NOTE: top margin and bottom margin are 0 relative
/*--------------------------------------------------------------------------*/

void	SetVT100scrollRegion(void)
{
	BYTE	tmpbottom, tmptop;
	WORD	SecondArg;

	/*	Assume defaults if not given.	*/
	if (!EmlnArgs[0])
		EmlnArgs[0]++;
	SecondArg = EmlnArgs[1];
	if (!SecondArg)
		SecondArg = (BYTE) LinesInScreen;

	if (EmlnArgs[0] < (WORD) LinesInScreen)
		tmptop = (BYTE) EmlnArgs[0];
	else
		tmptop = 1;
	if (SecondArg <= (WORD) LinesInScreen)
		tmpbottom = (BYTE) SecondArg;
	else
		tmpbottom = (BYTE) LinesInScreen;
	if (tmptop <= tmpbottom ) {
		ScrollTopMargin = (BYTE) (tmptop - 1);	// we use 0 based indices
		ScrollBottomMargin = (BYTE)(tmpbottom - 1);// we use 0 based indices
	}
	SetCursorPosition((BYTE) (EmlnVT100originMode ? ScrollTopMargin : 0) << 8);
	ResetArgs();
}
/*--------------------------------------------------------------------------*/
/*	Name		:	MoveToLineBegin											*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Puts the cursor at the first column of the current row.	*/
/*--------------------------------------------------------------------------*/

void	MoveToLineBegin(void)
{
	WriteChar(CR_ASCII);
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	StripHighBit											*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Strips the high bit of the received character.			*/
/*--------------------------------------------------------------------------*/

void	StripHighBit(void)
{
	RecdChar &= 0x7f;
}

/*--------------------------------------------------------------------------*/
/*	Name		:	CheckVT100screenAlignment								*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Checks the screen alignment by filling the screen with	*/
/*						E's (except the status line, if shown).				*/
/*--------------------------------------------------------------------------*/

void	CheckVT100screenAlignment(void)
{
	BYTE	row;

	for (row = 0; row < (BYTE) MaxRowIndx; row++) {
		SetCursorPosition(row << 8);
		DispChar('E', Attribute, CharsInLine);
	}
	SetCursorPosition(0);
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	SetVT100insertMode										*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Sets the VT100 terminal to insert mode.					*/
/*--------------------------------------------------------------------------*/

void	SetVT100insertMode(void)
{
	EmlnVT100insertMode = 1;	/*	Set VT100 insert mode on.	*/
	ResetArgs();
}

/*--------------------------------------------------------------------------*/
/*	Name		:	ResetVT100insertMode									*/
/*	Input		:	None													*/
/*	Output		:	None													*/
/*	Synopsis	:	Sets the VT100 terminal to insert mode.					*/
/*--------------------------------------------------------------------------*/

void	ResetVT100insertMode(void)
{
	EmlnVT100insertMode = 0;	/*	Set VT100 insert mode off.	*/
	ResetArgs();
}

/* -----------------------   Last  Line  ---------------------------------- */

