/****************************************************************************
 * Filename   : print.c
 * Purpose    : This file contains the functions to handle the printing
 *				and page setting.
 ****************************************************************************/

#include <windows.h>
#include <drivinit.h>
#include	<commdlg.h>
#include <stdlib.h>

#include "mew.h"
#include	"session.h"
#include	"main.h"
#include	"myprint.h"
#include	"file.h"
#include	"emlntype.h"
#include	"emlncons.h"
#include	"emlndata.h"
#include	"emlnfuns.h"
#include	"time.h"
#include	"status.h"
#include	"emlnfuns.h"
#include	"dlgbox.h"
#include	"looks.h"
#include	"ctl3d.h"

HGLOBAL	hPrinterDM;			/* the current printer settings */
BYTE 	PrintDriver[10];	/* the device driver for the current printer */
BYTE	PrintDevice[60];		/* the type of the current printer */
BYTE 	PrintOutput[10];		/* the current printer port */
BYTE	PrintedSomeThing;
BYTE	PrintLogging = FALSE;	/* is remote data logged on printer ? */

/* the default page-layout is as follows */
PageLayout	DefPageLayout = {
	"",					/* (was "&r&d") current date justified to the right */
	"",					/* (was "&cPage &p") centre justified with page number */
	19, "0.75",			/* 0.75" (19 mm) top margin */
	19, "0.75",			
	19, "0.75",			/* 0.75" (19 mm) bottom margin */
	19, "0.75",			
	1,					/* draft quality is on */
};

static	char	*PrnError = "Printer Error";
PRINTDLG	PDStruct;

/****************************************************************************
 * Function   : OpenPrinter (HWND, LPSTR)
 * Synopsis   : opens the printer DC, initialises the dialog boxes and
 *				returns the DC.
 ****************************************************************************/
HDC
OpenPrinter (HWND hWnd)
{
	HDC		hPr;

	if (! (hPr = GetPrinterDC(hWnd))) {
		DispMsgBox(hWnd, MSG_PRNOPEN, PrnError, MB_OK | MB_ICONHAND);
    	return NULL;
	}
	return (hPr);
}

/****************************************************************************
 * Function   : ClosePrinter (HDC)
 * Synopsis   : closes the printer and deletes the DC.
 ****************************************************************************/
void
ClosePrinter (HDC hPr)
{
	DeleteDC (hPr);						/* delete the printer DC */
}

/****************************************************************************
 * Function   : GetPrinterDC (HWND)
 * Synopsis   : Gets the printer DC for the active printer.
 ****************************************************************************/
HANDLE
GetPrinterDC(HWND hWnd)
{
	LPSTR	printerdm;
	HDC		hDC;

	if ( ! PrintDriver[0])
		if ( ! PrinterSetup(hWnd))
			return NULL;

	if (hPrinterDM) {	/* if printer settings changed, use our DM struct */
		printerdm = GlobalLock(hPrinterDM);
   		hDC = CreateDC(PrintDriver, PrintDevice, PrintOutput, printerdm);
		GlobalUnlock(hPrinterDM);
		return hDC;
	}
	else {
   		return (CreateDC(PrintDriver, PrintDevice,
							PrintOutput, (LPSTR)NULL));
	}
}

void	FillUpPrintDevNames(LPDEVNAMES lpDevNames)
{
	LPSTR	NamePtr;

	NamePtr = (LPSTR)lpDevNames + sizeof(DEVNAMES);
	lstrcpy ((LPSTR) NamePtr, (LPCSTR) PrintDriver) ;
	lpDevNames->wDriverOffset = (DWORD)NamePtr - (DWORD)lpDevNames;

	NamePtr = NamePtr + lstrlen (NamePtr) + 1 ;
	lstrcpy ((LPSTR) NamePtr, (LPCSTR) PrintDevice);
	lpDevNames->wDeviceOffset = (DWORD)NamePtr - (DWORD)lpDevNames;

	NamePtr = NamePtr + lstrlen(NamePtr) + 1 ;
	lstrcpy ((LPSTR) NamePtr, (LPCSTR) PrintOutput);
	lpDevNames->wOutputOffset = (DWORD)NamePtr - (DWORD)lpDevNames;
}

/****************************************************************************
 * Function	: PrinterSetup (HWND)
 * Synopsis	: This function shows the printer dialog box and allows the
 *			  user to change the settings.
 ****************************************************************************/
int
PrinterSetup (HWND hWnd)
{
	LPSTR	Names;
	LPDEVNAMES lpDevNames;
	HGLOBAL	hDevStruct = NULL;

	memset(&PDStruct, 0, sizeof(PRINTDLG));
	PDStruct.lStructSize = sizeof(PRINTDLG);
	PDStruct.hwndOwner = hWnd;
	PDStruct.Flags = PD_PRINTSETUP;

	/* If we have a printer selected, Show this */
	if (PrintDriver[0]) {
		if ( !(hDevStruct = GlobalAlloc(GHND, 200)))
			return FALSE;
		lpDevNames = (LPDEVNAMES) GlobalLock(hDevStruct);
		FillUpPrintDevNames(lpDevNames);

		GlobalUnlock(hDevStruct);
		PDStruct.hDevNames = hDevStruct;
		PDStruct.hDevMode = hPrinterDM;
	}
  	if ( DLGShowPrintDlgBox(&PDStruct)) {
		hPrinterDM = PDStruct.hDevMode;
		Names = GlobalLock(PDStruct.hDevNames);
		/* Copy Device mode */
		lpDevNames = (LPDEVNAMES)Names;

		lstrcpy(PrintDriver, Names + lpDevNames->wDriverOffset);
		lstrcpy(PrintDevice, Names + lpDevNames->wDeviceOffset);
		lstrcpy(PrintOutput, Names + lpDevNames->wOutputOffset);

		GlobalUnlock(PDStruct.hDevNames);
		GlobalFree(PDStruct.hDevNames);
		return TRUE;
	}
	if (hDevStruct)
		GlobalFree(hDevStruct);
	return FALSE;
}

/****************************************************************************
 * Function	: PageSetup (HWND)
 * Synopsis	: This function shows the page setup dialog box and allows the
 *			  user to change the settings.
 ****************************************************************************/
int
PageSetup (HWND hWnd)
{
	InvokeDialog (hWnd, "PAGESETUPDLG", PageSetupDlgProc);
	return TRUE;
}

/****************************************************************************
 * Function	: PageSetupDlg(HWND, unsigned, WORD, LONG)
 * Synopsis	: This dialog box is to help user setup the page layout
 ****************************************************************************/
int FAR PASCAL
PageSetupDlgProc(HWND hDlg, unsigned msg, WORD wParam, LONG lParam)
{

    switch(msg) {


        case WM_INITDIALOG:
			DLGCenterDialog(hDlg);

			SetDlgItemText (hDlg, IDC_PG_HEADER, (LPSTR)DefPageLayout.Header);
			SetDlgItemText (hDlg, IDC_PG_FOOTER, (LPSTR)DefPageLayout.Footer);
			SetDlgItemText (hDlg, IDC_PG_MAR_LT, (LPSTR)DefPageLayout.LeftMrgStr);
			SetDlgItemText (hDlg, IDC_PG_MAR_RT, (LPSTR)DefPageLayout.RightMrgStr);
			SetDlgItemText (hDlg, IDC_PG_MAR_TP, (LPSTR)DefPageLayout.TopMrgStr);
			SetDlgItemText (hDlg, IDC_PG_MAR_BM, (LPSTR)DefPageLayout.BottomMrgStr);
			if (DefPageLayout.DraftQuality)
				CheckDlgButton (hDlg, IDC_PG_DRAFT, TRUE);
			return FALSE;

        case WM_COMMAND:
            switch (wParam) {
				case IDC_PG_DRAFT :
					if (IsDlgButtonChecked(hDlg, wParam))
						CheckDlgButton (hDlg, wParam, FALSE);
					else
						CheckDlgButton (hDlg, wParam, TRUE);
					break;
        case IDOK:
					GetDlgItemText (hDlg, IDC_PG_HEADER,
								(LPSTR)DefPageLayout.Header, 80);
					GetDlgItemText (hDlg, IDC_PG_FOOTER,
								(LPSTR)DefPageLayout.Footer, 80);
					GetDlgItemText (hDlg, IDC_PG_MAR_LT,
								(LPSTR)DefPageLayout.LeftMrgStr, 6);
					DefPageLayout.LeftMargin =
								(int)(atof(DefPageLayout.LeftMrgStr) * 25.4);
					GetDlgItemText (hDlg, IDC_PG_MAR_TP,
								(LPSTR)DefPageLayout.TopMrgStr, 6);
					DefPageLayout.TopMargin =
								(int)(atof(DefPageLayout.TopMrgStr) * 25.4);
					GetDlgItemText (hDlg, IDC_PG_MAR_RT,
								(LPSTR)DefPageLayout.RightMrgStr, 6);
					DefPageLayout.RightMargin =
								(int)(atof(DefPageLayout.RightMrgStr) * 25.4);
					GetDlgItemText (hDlg, IDC_PG_MAR_BM,
								(LPSTR)DefPageLayout.BottomMrgStr, 6);
					DefPageLayout.BottomMargin =
								(int)(atof(DefPageLayout.BottomMrgStr) * 25.4);
					if (IsDlgButtonChecked(hDlg, IDC_PG_DRAFT))
						DefPageLayout.DraftQuality = 1;
					else
						DefPageLayout.DraftQuality = 0;
               EndDialog(hDlg, IDOK);
               return (TRUE);
          case IDCANCEL:
                EndDialog(hDlg, IDCANCEL);
                return (TRUE);
            }
            break;
	}
    return (FALSE);
}

/*##########################################################################*/
/*-------------------- FUNCTIONS FOR PRINTING OUT --------------------------*/
/*##########################################################################*/

HDC		hPrDC;					/* printer DC */
BOOL	PrintAbort = FALSE;		/* set when printing needs to be aborted */
FARPROC	lpPrAbortProc;			/* Instance of printer abort procedure */

int		PrLineNum=0;			/* current line number */
int		PrPageNum=1;			/* current page number */
int		PrX=0, PrY=0;			/* X & Y offsets on the print page */
int		LineSpace;				/* space between lines */
int		LinesPerPage;			/* # of lines fit in one page */
int		TopMarginLines;			/* # of lines left for top margin */
int		BottomMarginLines;		/* # of line left for bottom margin */
int		LeftMarginWidth;		/* left margin width in # of pixels */
BOOL	PrPageEnd = FALSE;		/* variable to indicate if we reached page end */
POINT	PrPhysPageSize;			/* printer physical page size */

/****************************************************************************
 * Function	: StartPrinter (void)
 * Synopsis	: Opens the selected printer, returns TRUE on success else FALSE.
 ****************************************************************************/
int
StartPrinter (void)
{
	int	Ret;

	if (PrintLogging)
		return TRUE;
	if (! (hPrDC = OpenPrinter(hMainWnd)))
		return FALSE;

	PrintAbort = FALSE;
	PrintedSomeThing = FALSE;

	lpPrAbortProc = MakeProcInstance ((FARPROC)PrintAbortProc, hInst);
	Escape(hPrDC, SETABORTPROC, NULL, (LPSTR)(long)lpPrAbortProc, (LPSTR) NULL);

	ComputePrinterVars (hPrDC);

	/* initiate the print job */
	if ((Ret = Escape(hPrDC, STARTDOC, 9, "MEW Print", (LPSTR)NULL)) < 0) {
		CheckPrintError (Ret);
   		FreeProcInstance (lpPrAbortProc);
		ClosePrinter (hPrDC);
		return FALSE;
	}

	PrintLogging = TRUE;
	return TRUE;
}

/****************************************************************************
 * Function	: StopPrinter (void)
 * Synopsis	: Closes the opened printer, if any.
 ****************************************************************************/
int
StopPrinter (void)
{
	if (! PrintLogging)
		return TRUE;

	/* Print all the data, give a form feed and end the print job */
	if (PrintedSomeThing) {
		Escape (hPrDC, NEWFRAME, 0, (LPSTR)NULL, (LPSTR)NULL);
//		while(!PrPageEnd)
//			PrintChar(CR_ASCII);
	}
	Escape (hPrDC, ENDDOC, 0, (LPSTR)NULL, (LPSTR)NULL);

	PrintLogging = FALSE;
   	FreeProcInstance (lpPrAbortProc);
	ClosePrinter (hPrDC);
	return TRUE;
}

/****************************************************************************
 * Function	: PrintBuffer (LPSTR, WORD)
 * Synopsis	: Prints the data in the buffer and returns the actual number of
 *			  characters printed. -1 if some error occurs.
 ****************************************************************************/
int
PrintBuffer (LPSTR buf, WORD len)
{
	int i;
	BYTE	was_printing = PrintLogging;

	if (!PrintLogging)
		StartPrinter();

	for (i=0; (i < (int)len) && (! PrintChar(buf[i])); i++);
	if ( ! was_printing)
		StopPrinter();

	return(i);
}

/****************************************************************************
 * Function	: PrintChar (BYTE)
 * Synopsis	: Prints one character. Returns 0 on success, -1 on failure.
 ****************************************************************************/
int
PrintChar (BYTE ch)
{
	int		Ret;
	char	FmtStr[132];

	if (! PrintLogging)
		return (0);
	PrintedSomeThing = TRUE;

	if (PrPageEnd) {	/* print the footer */
		if (DefPageLayout.Footer[0]) {
			PrX = LeftMarginWidth +
				GetFormattedStr(hPrDC, DefPageLayout.Footer,
					FmtStr, PrPageNum, GetDeviceCaps(hPrDC, HORZRES));
			PrY = PrPhysPageSize.y - ((BottomMarginLines)  * LineSpace);
			TextOut (hPrDC, PrX, PrY, (LPSTR)FmtStr, strlen(FmtStr));
		}

		/* increment the page-number and go to the next page */
		PrPageNum++;
		if ((Ret = Escape(hPrDC, NEWFRAME, 0, (LPSTR)NULL, (LPSTR)NULL)) < 0) {
			CheckPrintError (Ret);
			TogglePrinter ();
			return (-1);
		}

		PrPageEnd = FALSE;
		PrLineNum = 0;
	}

	if (! PrLineNum) {	/* print the header */
		if (DefPageLayout.Header[0]) {
			PrX = LeftMarginWidth +
				GetFormattedStr(hPrDC, DefPageLayout.Header,
					FmtStr, PrPageNum, GetDeviceCaps(hPrDC, HORZRES));
			PrY = (TopMarginLines >> 1) * LineSpace;
			TextOut (hPrDC, PrX, PrY, (LPSTR)FmtStr, strlen(FmtStr));
		}
		/* init the X & Y offsets for printing the data */
		PrLineNum = TopMarginLines + 1;
		PrX = LeftMarginWidth;
		PrY = PrLineNum * LineSpace;
	}

	switch (ch) {
#if 0
		case LF_ASCII:
			if (ApplStrt.PbkEntry.TrmParams.StripLF)
				break;
	PrintLF:
			PrLineNum++;
			if (PrLineNum == (LinesPerPage - BottomMarginLines)) {
				PrPageEnd = TRUE;
				break;
			}
			PrY = PrLineNum * LineSpace;
			break;
		case CR_ASCII:
			PrX = LeftMarginWidth;
			if (ApplStrt.PbkEntry.TrmParams.AddLFtoCR)
				goto PrintLF;
			break;
#else
		case LF_ASCII:
				break;
	PrintLF:
			PrLineNum++;
			if (PrLineNum == (LinesPerPage - BottomMarginLines - 1)) {
				PrPageEnd = TRUE;
				break;
			}
			PrY = PrLineNum * LineSpace;
			break;
		case CR_ASCII:
			PrX = LeftMarginWidth;
			goto PrintLF;
#endif
		default:
			TextOut (hPrDC, PrX, PrY, (LPSTR)&ch, 1);
			PrX += (WORD)GetTextExtent (hPrDC, (LPSTR)&ch, 1);
			break;
	}
	return(0);
}

/****************************************************************************
 * Function	: TogglePrinter (void)
 * Synopsis	: Depending on the print status it either opens or
 *			  closes the printer.
 ****************************************************************************/
int
TogglePrinter(void)
{
	char	buf[100];
	HCURSOR	hSaveCursor;			/* current cursor */


	hSaveCursor = SetCursor (hHourGlass);
	if (PrintLogging) {
		if (StopPrinter()) {
			PrintLogging = FALSE;
			strcpy (buf, "&Print                       Alt+P");
			ModifyMenu (GetMenu(hMainWnd), IDM_FILE_PRINT, 
				MF_BYCOMMAND|MF_STRING, IDM_FILE_PRINT, (LPSTR)buf);
		}
	}
	else {
		if (StartPrinter()) {
			PrintLogging = TRUE;
			strcpy (buf, "&Print Abort              Alt+P");
			ModifyMenu (GetMenu(hMainWnd), IDM_FILE_PRINT, 
				MF_BYCOMMAND|MF_STRING, IDM_FILE_PRINT, (LPSTR)buf);
		}
	}
	UpdateStatusBar (PRINT_FLD_IDX);
	tbUpdateToolbarButton(IDTB_PRINT);
	SetCursor (hSaveCursor);
	return (TRUE);
}

/****************************************************************************
 * Function	: PrintAbortProc (HDC, int)
 * Synopsis	: Processes messages and yields control to print manager.
 ****************************************************************************/
int FAR PASCAL
PrintAbortProc(HDC hPr, int Code)
{
#if 0
	MSG msg;

	while (!PrintAbort && PeekMessage(&msg, NULL, NULL, NULL, PM_REMOVE)) {
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}
	return (!PrintAbort);
#endif
	if (Code == SP_OUTOFDISK)
		return FALSE;
	return TRUE;
}

/****************************************************************************
 * Function	: ComputePrinterVars (HDC)
 * Synopsis	: Computes the printer variables.
 ****************************************************************************/
void
ComputePrinterVars (HDC hPr)
{
	int		PaperHeight;
	TEXTMETRIC	TextMetric;

	GetTextMetrics (hPr, &TextMetric);
	LineSpace = TextMetric.tmHeight + TextMetric.tmExternalLeading;

	Escape (hPr, GETPHYSPAGESIZE, NULL, (LPSTR)NULL, (LPSTR)&PrPhysPageSize);
	LinesPerPage = PrPhysPageSize.y / LineSpace;

	PaperHeight = GetDeviceCaps (hPr, VERTSIZE);
	TopMarginLines = (DefPageLayout.TopMargin * LinesPerPage) / PaperHeight;
	BottomMarginLines = (DefPageLayout.BottomMargin * LinesPerPage) / PaperHeight;

	LeftMarginWidth = DefPageLayout.LeftMargin * GetDeviceCaps(hPr, LOGPIXELSX) / 25;

	PrX = LeftMarginWidth;
	PrY = (TopMarginLines >> 1) * LineSpace;
	PrLineNum = 0;
	PrPageNum = 1;
	return;
}

#define	LEFT_JUSTIFY	1
#define	RIGHT_JUSTIFY	2
#define	CENTER_JUSTIFY	3

/****************************************************************************
 * Function	: GetFormattedStr ()
 * Synopsis	: Formats the header or footer strings and returns the printable
 *			  string in FmtStr variable. Returns the number of blanks to be
 *			  printed before printing the actual string.
 ****************************************************************************/
WORD
GetFormattedStr (HDC hDc, LPSTR Str, LPSTR FmtStr, int PageNum, WORD PageWidth)
{
	TIME	cTime;
	DATE	cDate;
	LPSTR	FmtBuf;
	BYTE	tmp[32];
	WORD	FmtStrWidth;
	WORD	MarginWidth;
	WORD	BlankOffset = 0;
	int		Justify = LEFT_JUSTIFY;

	FmtBuf = FmtStr;
	while (*Str) {
		if (*Str == '&') {
			switch (*++Str) {
				case 'd' :	/* insert current date */
					GetDate (&cDate);
					FormatAsciiDate (&cDate, tmp, DATE_EXPAND);
					strcpy (FmtBuf, tmp);
					FmtBuf += strlen(tmp);
					break;
				case 'p' :	/* insert the current page number */
					//itoa (PageNum, tmp, 10);
					wsprintf(tmp,"%u",PageNum);
					strcpy (FmtBuf, tmp);
					FmtBuf += strlen(tmp);
					break;
				case 'l' :	/* by default left margin justified */
					break;
				case 'r' :	/* right margin justified */
					Justify = RIGHT_JUSTIFY;
					break;
				case 'c' :	/* center justified */
					Justify = CENTER_JUSTIFY;
					break;
				case 't' :
					GetTime (&cTime);
					FormatAsciiTime (&cTime, tmp, TIME_AMPM);
					strcpy (FmtBuf, tmp);
					FmtBuf += strlen(tmp);
					break;
				default:
					*FmtBuf++ = '&';
					*FmtBuf++ = *Str;
					break;
			}
			Str++;
		}
		else
			*FmtBuf++ = *Str++;
	}
	*FmtBuf++ = NULL;

	if ((Justify == RIGHT_JUSTIFY) || (Justify == CENTER_JUSTIFY)) {
		FmtStrWidth = (WORD)GetTextExtent (hDc, (LPSTR)FmtStr, strlen(FmtStr));
		MarginWidth = (WORD)(GetDeviceCaps (hDc, LOGPIXELSX) *
					(DefPageLayout.LeftMargin+DefPageLayout.RightMargin) / 25);
		BlankOffset = PageWidth - MarginWidth - FmtStrWidth;
		if (Justify == CENTER_JUSTIFY)
			BlankOffset >>= 1;	/* divide by 2 */
	}
	return (BlankOffset);
}

static char *PrintErr[] = {
	"General Error",
	"Cancelled from Program",
	"Cancelled from Print Manager",
	"Out of disk space",
	"Out of memory space",
};

int
CheckPrintError (short EscRet)
{
	if (EscRet >= 0)
		return TRUE;
	if ((EscRet & SP_NOTREPORTED) == 0)
		return FALSE;
	MessageBox (hMainWnd, PrintErr[~EscRet], PrnError, MB_OK | MB_ICONHAND);
	return FALSE;
}

