/*----------------------------------------------------------------------------
*	File		:	SCROLL.C
*	Purpose		:	Scroll buffer related routines for MEW
*	Package		:	MultiExpress (Windows) - Version 2.00
*	Authors		:	Vidy
*	Date		:	5th feb 1992.
*---------------------------------------------------------------------------*/

#include	<windows.h>
#include	<commdlg.h>

#include	"main.h"
#include	"mew.h"
#include	"scroll.h"
#include	"wstdio.h"
#include	"term.h"
#include	"search.h"
#include	"file.h"
#include	"session.h"
#include	"status.h"
#include	"time.h"
#include	"select.h"
#include	"windisp.h"
#include	"looks.h"
#include	"dlgbox.h"

#define	REV_DEF_COLOR	0x07		/* to be moved to main.h later */

BYTE	SearchPattern[MAX_SRCH_PAT_LEN + 1];
BYTE	*ScrlMsgHeader = "Scroll Mode";

/* the following vars are independent of scroll buffer lock */
HANDLE	hScrollMem;					/* Handle to Global Scroll memory */
BYTE	ScrollLock = FALSE;			/* TRUE if we are in scroll mode */
WORD	ScrollBuffSize;				/* MUST be a power of TWO */
WORD	NumLinesInScrlBuff;			/* Number of lines in scroll buffer */
WORD	ScrlHead;					/* Beginnig of scrl buffer data */
WORD	ScrlTail;					/* End of actual scrl buff data */
WORD	ScrlTmpTail;				/* end of scrl buff + current screen */

/* Following vars are valid in scroll mode only */
BYTE	FAR	**ScrollIndexTable;		/* line index array ptr */
BYTE	FAR	*pScrlBuf;				/* pointer to the scroll buffer */
BYTE	FAR	*pScrlBufEnd;			/* pointer to end byte */
BYTE	FAR	*pScrlAttBuf;			/* Scroll attributes buffer */
BYTE	FAR	*SearchPtr;				/* far pointer to global scrl buf */

WORD	NumLinesBeforeScroll;		/* temporary to store num lines */
int		TermThumbX, TermThumbY;		/* The terminal Thumb positions */
BYTE	TermWinOrgX, TermWinOrgY;	/* Terminal Window origins */
WORD	ScrlDispIndx;				/* index into ScrollIndexTable where
									current display memory starts */

/***************************************************************************
*	Routine	:	ResetScrollBuffer (void)
*	Input	:	
*	Return	:	
*	Synopsis:	Set the scroll buffer contents to 0
***************************************************************************/
void	ResetScrollBuffer(void)
{
	ScrlHead = ScrlTail = ScrlTmpTail = 0;
	NumLinesInScrlBuff = 0;			/* reset the number of lines to 0 */
	TextSelected = FALSE;			// say no text buffer selected
}

/***************************************************************************
Synop : return TRUE if the parameter passed in is a power of 2
****************************************************************************/
BOOL	IsWordPowerOfTwo(WORD val)
{
	int		word_size = sizeof(WORD) * 8;	// number of bits
	BOOL	found_a_hi_bit = FALSE;

	if ( !val)
		return TRUE;		// 0 is a power of 2
	while (word_size--) {
		if (found_a_hi_bit) {
			if (val & 1)
				return FALSE;
		} else {
			if (val & 1)
				found_a_hi_bit = TRUE;
		}
		val >>= 1;
	}
	return TRUE;
}
/***************************************************************************
*	Routine	:	SetScrollBuffSize(WORD newsize)
*	Input	:	newsize - size of meomeory to be allocated for scroll buff
*	Return	:	Returns TRUE if enough memory is present. else FALSE.
*	Synopsis:	Release the earlier scroll buffer area. allocate new area.
*				Reinit the pointers. return FALSE if no mem
***************************************************************************/
BOOL	SetScrollBuffSize(WORD newsize)
{
	if ( ! IsWordPowerOfTwo (newsize))
		return FALSE;

	if (hScrollMem)
		GlobalFree(hScrollMem);			/* free hScrollMem if Any */

	hScrollMem = NULL;
	NumLinesInScrlBuff = 0;
	ScrlHead = ScrlTail = ScrlTmpTail = 0;
	ScrollBuffSize = 0;

	if (!newsize)
		return TRUE;
	if ( (hScrollMem = GlobalAlloc (GMEM_MOVEABLE,
									(2 * (DWORD) newsize))) == NULL)
		return FALSE;
	ScrollBuffSize = newsize;
	return TRUE;
}
/**************************************************************************
Lock the Global mem used for storing the scroll data
**************************************************************************/
BOOL	LockScrollMem(void)
{
	if (pScrlBuf)
		return FALSE;								// already locked
	pScrlBuf = GlobalLock(hScrollMem);				/* start of scroll data */
	if ( ! pScrlBuf)
		return FALSE;
	pScrlBufEnd = pScrlBuf + ScrollBuffSize - 1;	/* end of scroll data */
	pScrlAttBuf = pScrlBuf + ScrollBuffSize;		/* start of attribs	*/
	return TRUE;
}

/**************************************************************************
Unlock the global memory used for storing scroll data.
**************************************************************************/
BOOL	UnlockScrollMem(void)
{
	if (pScrlBuf)
		GlobalUnlock(hScrollMem);
	pScrlBuf = NULL;
	return TRUE;
}																				

/***************************************************************************
*	Routine	:	EnterScrollMode(void)
*	Input	:	
*	Return	:	TRUE if enough mem is available to create the index table.
*				otherwise FALSE.
*	Synopsis:	Routine called to enter scroll mode.
*				Prepares for scroll display. updates the ScrollBuffer by
*				temporarily adding the display area. Stores the terminal
*				parameters to restore later when scroll mode ends
***************************************************************************/
BOOL	EnterScrollMode(void)
{
	WORD	i;
	BYTE	*dptr;

	if (!hScrollMem)
		return FALSE;

	ScrollLock = TRUE;			/* here after we are in scroll mode */

	/* NumLinesTemp to get the current number of lines */
	NumLinesBeforeScroll = NumLinesInScrlBuff;

	/* add the screen data to the temp region */
	ScrlTmpTail = ScrlTail;

	/* add all the lines in the display to the scroll buffer temporarily */
	for (i = 0, dptr = sDispOrig; i < LinesInScreen; i++) {
		SaveLineToScrlBuff (dptr);
		dptr += CellsInLine;
		if (dptr > sScreenBuffLast)			/* disp memory wraps around.	*/
			dptr = sScreenBuff;
	}

	/* Lock the scroll memory */
	LockScrollMem();

	/* Create the ScrollIndexTable for getting the index array to lines */
	if ( ! (CreateScrollIndexTable()) ) {
		ScrollLock = FALSE;
		UnlockScrollMem();
		return FALSE;
	}
	/* Set the value of index into the ScrollBufferIndexTable where
	the current display memory starts */
	ScrlDispIndx = NumLinesInScrlBuff - LinesInScreen;

	/* Hide the cursor */
	LoosingFocus();

	/* Terminal window origins wrt to the display memory */
	TermWinOrgX = WinOrgX;
	TermWinOrgY = WinOrgY;

	/* Get and store the thumb positions for later restoration */
	TermThumbX = GetScrollPos (hTermWnd, SB_HORZ);
	TermThumbY = GetScrollPos (hTermWnd, SB_VERT);

	/* set the thumb range for the current scroll region */
	SetupWinMatrix();

	/* posotion the thumbs correctly to match the scroll position */
	SetScrollPos (hTermWnd, SB_HORZ, WinOrgX, TRUE);	/* x pos same */
	SetScrollPos (hTermWnd, SB_VERT,
		NumLinesInScrlBuff - LinesInScreen + WinOrgY, TRUE);	/* y pos */

	ModifyMenu (GetMenu(hMainWnd), IDM_EDIT_SCROLL, 
			MF_BYCOMMAND|MF_STRING, IDM_EDIT_SCROLL,
				(LPSTR)"Exit &Scroll     Alt+Scrl");

	CurState |= MSK_SCROLL_MODE;
	UpdateStatusInBar ();
	tbUpdateToolbarButton(IDTB_SCROLL);
	tbUpdateToolbar();

	SearchPtr = ScrollIndexTable [ScrlDispIndx + WinOrgY];
	PatternBright = FALSE;
	PatternFound = FALSE;
	TextSelOn = FALSE;		/* Text buffer-being-selected falg */

	if (TextSelected) {		/* we may have to set the display */
		if ( (SelEndLineIndex < (ScrlDispIndx + WinOrgY)) ||
				(SelStartLineIndex > (ScrlDispIndx + WinOrgY + RowsInWin) ) )
			return TRUE;		/* selection not in display */
		SetDisplayFromIdx(ScrlDispIndx);
		UpdateWindow(hTermWnd);
	}
	HandleKbdLeds(LED_SCROLLLOCK, LIGHT_ON);
	return TRUE;
}

/***************************************************************************
*	Routine	:	ExitScrollMode(void)
*	Input	:	
*	Return	:	
*	Synopsis:	retrieves the terminal parameters before entering scroll mode.
*				sets the terminal window back to the original position.
***************************************************************************/
void	ExitScrollMode(void)
{
	BYTE	save_sel;

	/* Unlock the ScrollLock */
	ScrollLock = FALSE;

	ModifyMenu (GetMenu(hMainWnd), IDM_EDIT_SCROLL, 
					MF_BYCOMMAND|MF_STRING, IDM_EDIT_SCROLL,
							(LPSTR)"Enter &Scroll   Alt+Scrl");
	/* Setup the window matrix back */
	SetupWinMatrix();

	/* set the thumb positions back */
	SetScrollPos (hTermWnd, SB_HORZ, TermWinOrgX, TRUE);	/* x pos  */
	SetScrollPos (hTermWnd, SB_VERT, TermWinOrgY, TRUE);	/* y pos  */

	/* Terminal window origins wrt to the display memory */
	WinOrgX = TermWinOrgX;
	WinOrgY = TermWinOrgY;

	ScrlDispIndx = NumLinesBeforeScroll;

	/* save the TextSelected status before putting the screen back */
	save_sel = TextSelected;
	TextSelected = FALSE;

	/* put the display into the old area */
	SetDisplayFromIdx (ScrlDispIndx);

	/* save the TextSelected status before putting the screen back */
	TextSelected = save_sel;

	/* Set the number of lines in the scroll buffer back */
	NumLinesInScrlBuff -= LinesInScreen;

	/* release the ScrolIndexTable */
	MemFree ( (LPSTR) ScrollIndexTable);
	ScrollIndexTable = NULL;

	/* position the thumbs to the old place */
	SetScrollPos (hTermWnd, SB_HORZ, TermWinOrgX, TRUE);
	SetScrollPos (hTermWnd, SB_VERT, TermWinOrgY, TRUE);

	/* Update the invalidated terminal window */
	UpdateWindow(hTermWnd);

	SettingFocus();

	/* unhide the cursor */
	DisplayCursor ();
	CurState &= ~MSK_SCROLL_MODE;
	HandleKbdLeds(LED_SCROLLLOCK, LIGHT_OFF);
	UpdateStatusInBar ();
	tbUpdateToolbarButton(IDTB_SCROLL);
	tbUpdateToolbar();
	UnlockScrollMem();				/* Unlock global mem for Scroll buffer */
	TextSelected = FALSE;			// text selectioned valid only in scroll
}

/***************************************************************************
*	Routine	:	SaveLineToScrlBuff (BYTE *ptr)
*	Input	:	lineptr - pointer to the line data in dispmem
*	Return	:	
*	Synopsis:	Saves the top line of Terminal into the scroll buffer
*				line is scanned from right end and all blanks are discarded
*	Note	:	Called from the Terminal emulation when a scroll of top
*				line occurs. Also from EnterScrollMode for temp addition.
***************************************************************************/
void	SaveLineToScrlBuff (BYTE *lineptr)
{
	WORD	len;

	if ( ! hScrollMem)				/* no scroll buffer */
		return;
	len = CharsInLine;				/* store the entire line. */
	if ( !LockScrollMem())			/* Lock the scroll memory first */
		return;

	if (GetScrollBufferSpace() < (len + 5))
		MakeRoomInScrollBuffer(len);
	CopyLineToScrollBuffer (lineptr, len);
	UnlockScrollMem();				/* Unlock scroll mem after copy */
}

/***************************************************************************
*	Routine	:	CreateScrollIndexTable(void)
*	Input	:	
*	Return	:	returns FALSE if no mem for scrollindextable.
*	Synopsis:	Scans the scroll buffer and forms the ScrollIndexTable
***************************************************************************/
BOOL	CreateScrollIndexTable(void)
{
	WORD	offset;
	WORD	index;

	if ( (ScrollIndexTable = (BYTE FAR **) MemAlloc
					(NumLinesInScrlBuff * sizeof (BYTE FAR *))) == NULL)
		return FALSE;
	offset = ScrlHead;
	ScrollIndexTable[0] = pScrlBuf + offset;
	for (index = 1; ; index++) {
		/* goto the next EOL char */
		while (*(pScrlBuf + offset++) != SCROLL_EOL_CHAR)
			offset &= (ScrollBuffSize - 1);

	  	offset &= (ScrollBuffSize - 1);
		if (offset == ScrlTmpTail)			/* scanning is over */
			break;
		ScrollIndexTable[index] = pScrlBuf + offset;
	}
}

/***************************************************************************
*	Routine	:	GetScrollBufferSpace(void)
*	Input	:	
*	Return	:	Free space available in the ScrollBuffer
*	Synopsis:	Tell the amount of free space in ScrollBuffer
***************************************************************************/
WORD	GetScrollBufferSpace(void)
{
	WORD	tail;

	tail = (ScrollLock) ? ScrlTmpTail : ScrlTail;
	if (tail >= ScrlHead)
		return (ScrollBuffSize - (tail - ScrlHead) - 1);
	else
		return (ScrlHead - tail - 1);		/* one space to be unfilled */
}

/***************************************************************************
*	Routine	:	MakeRoomInScrlBuff(WORD nchars)
*	Input	:	nchars how much min space is needed
*	Return	:	
*	Synopsis:	Called if space is not available for storing a line
*	NOTE	:	CAN BE CALLED ONLY IF NO ROOM !!! will goof if 
*				number of chars in the buffer is less than nchars
***************************************************************************/
void	MakeRoomInScrollBuffer(WORD nchars)
{
	WORD	offset;
	WORD	count;

	nchars += SCROLL_ROOM_EXTRA;		/* to reduce frequency */

	/* advance the pointer and keep checking if we are loosing any line */
	for (offset = ScrlHead, count = 0; count < nchars; count++) {
		if (*(pScrlBuf + offset) == SCROLL_EOL_CHAR) {
			/* are we chopping a line ? */
			NumLinesInScrlBuff--;
			if (TextSelected) {
				if ( SelStartLineIndex) {
					SelStartLineIndex--;		/* indices in temp table */
					SelEndLineIndex--;
				} else {
					TextSelected = FALSE;
				}
			}
		}
		offset++;
		offset &= (ScrollBuffSize - 1);
	}
	/* Now we got atleast the required space. Align the buffer to
	start at the beginning of a line */
	while (* (pScrlBuf + offset++) != SCROLL_EOL_CHAR)
		offset &= (ScrollBuffSize - 1);		// wrap the index
	offset &= (ScrollBuffSize - 1);			// wrap at the end also

	NumLinesInScrlBuff--;					// reduce the line count

	ScrlHead = offset;						/* set the head to new value */
}

/***************************************************************************
*	Routine	:	CopyLineToScrollBuffer(BYTE *line, WORD count)
*	Input	:	line - pointer to line in DispMem, count - count of chars
*	Return	:	
*	Synopsis:	Copies count number of chars to the Scroll buffer
*				Assumes that enough space is there
***************************************************************************/
void	CopyLineToScrollBuffer (BYTE *line, WORD count)
{
	WORD	tail;

	tail = (ScrollLock) ? ScrlTmpTail : ScrlTail;	/* Which pointer	*/

	while (count--) {
		/* copy the character	*/
		*(pScrlBuf + tail) = *line;
		/* copy attribute	*/
		*(pScrlAttBuf + tail++)  = *(line++ + ScrnBufSize);
		tail &= (ScrollBuffSize - 1);
	}
	/* put the end of line character */
	*(pScrlBuf + tail++) = SCROLL_EOL_CHAR;
		tail &= (ScrollBuffSize - 1);

	if (ScrollLock)						/* update the tail pointer	*/
		ScrlTmpTail = tail;
	else
		ScrlTail = tail;
	NumLinesInScrlBuff++;				/* One more line in scroll buffer	*/
}

/***************************************************************************
*	Routine	:	SetDisplayFromIdx(WORD indx)
*	Input	:	indx - the thumb position index
*	Return	:	
*	Synopsis:	Sets the display starting with line whose index in scroll
*				buffer table is passed. 
***************************************************************************/
void	SetDisplayFromIdx(WORD indx)
{
	WORD	i;

	/*
	display origin can't go more than (NumLinesInScrlBuff - LinesInScreen)
	*/
	if(indx + LinesInScreen < NumLinesInScrlBuff)
		ScrlDispIndx = indx;
	else
		ScrlDispIndx = NumLinesInScrlBuff - LinesInScreen;

	sDispOrig = sScreenBuff;		/* it is easier to reset it */
	for (i = 0; i < LinesInScreen; i++)
		CopyIndxLineFromScrlBuff((BYTE)i);

	/* Invalidate the term window since it doesn't correlate with dispmem */
	InvalidateRect(hTermWnd, NULL, FALSE);
}

/***************************************************************************
*	Routine	:	ScrollDisplayUp (WORD nlines)
*	Input	:	nlines - Number of lines to go up
*	Return	:	
*	Synopsis:	Scrolls the display up nlines
***************************************************************************/
BOOL	ScrollDisplayUp (WORD nlines)
{
	DWORD	old_bkgnd;
	BOOL	ret_val = FALSE;

	while (nlines--) {
		if ( ! WinOrgY) {
			if ( !GetUperLineToDispMem() )
				return ret_val;			/* we are on topmost line already */
		} else {
			WinOrgY--;		   				/* New Y origin of window	*/
		}
		ret_val = TRUE;
		old_bkgnd = SetBkColor(hTermDC, ColorTable[DEF_COLOR]);
		/* issue a block scroll down */
		ScrollWindow(hTermWnd, 0, nStdioCharHeight,
										NULL, (LPRECT) &(TermWindow));
		SetBkColor(hTermDC, old_bkgnd);

//		UpdateWindow(hTermWnd);
	}
	return ret_val;
}

/***************************************************************************
*	Routine	:	ScrollDisplayDown (WORD nlines)
*	Input	:	nlines - number of lines to go down
*	Return	:	
*	Synopsis:	takes the display nlines down
***************************************************************************/
BOOL	ScrollDisplayDown (WORD nlines)
{
	DWORD	old_bkgnd;
	BOOL	ret_val = FALSE;

	while (nlines--) {
		if ( (WinOrgY + RowsInWin) >= LinesInScreen) {
			if ( ! GetBottomLineToDispMem() )
				return ret_val;			/* we are at the bottom most line */
		} else {
			WinOrgY++;		/* new window origin wrt disp mem */
		}

		ret_val = TRUE;
		old_bkgnd = SetBkColor(hTermDC, ColorTable[DEF_COLOR]);
		/* issue a block scroll up */
		ScrollWindow(hTermWnd, 0, - nStdioCharHeight,
									NULL, (LPRECT) &(TermWindow));
		SetBkColor(hTermDC, old_bkgnd);

//		UpdateWindow(hTermWnd);
	}
	return ret_val;
}

/***************************************************************************
*	Routine	:	ScrollInY (WORD command, WORD scrl_pos)
*	Input	:	command - type of scroll, scrl_pos - thump position val
*	Return	:	
*	Synopsis:	This routine does the mapping of the ScrollBuffer to the
*				display memory and to the window. Data from ScrollBuffer is
*				first brought into the display memory and then the window
*				is displayed. This means that display memory is always 
*				in sync to the display memory. This way StdioPaint itself
*				is used for painting even in scroll mode also.
***************************************************************************/
void	ScrollInY (WORD command, WORD thumb_pos)
{
	WORD	cur_pos;
	WORD	newscrlwinorg;		/* the orign of dispwin in scroll buffer */
	
	cur_pos = ScrlDispIndx + WinOrgY;		/* current display position */
	switch (command) {

		case 	SB_LINEUP :				/* One more line on top */
			ScrollDisplayUp (1);
			break;

		case 	SB_LINEDOWN :				/* one more line on bottom */
			ScrollDisplayDown (1);
			break;

		case 	SB_PAGEUP :					/* one page up */
			/* calculate the newscrlwinorg */
			if ((ScrlDispIndx + WinOrgY) > RowsInWin) {
				/* we can move on full window up */
				newscrlwinorg = ScrlDispIndx + WinOrgY - RowsInWin;
			} else {
				/* we can maximum move to the first line */
				newscrlwinorg = 0;
			}
			/* if new origin is current origin return */
			if (newscrlwinorg == (ScrlDispIndx + WinOrgY))
				return;
			
			/* Check if we need to bring any lines to display mem */
			if (newscrlwinorg < ScrlDispIndx)
				SetDisplayFromIdx (newscrlwinorg);

			/* compute the new WinOrgY */
			WinOrgY = (BYTE) (newscrlwinorg - ScrlDispIndx);

			/* invalidate the entire window */
			InvalidateRect(hTermWnd, NULL, FALSE);	/* invalidate the entire area */
			break;

		case 	SB_PAGEDOWN :				/* one page down */
			/* calculate the new possible display origin wrt to ScrollIndx */
			newscrlwinorg = ScrlDispIndx + WinOrgY + RowsInWin;

			/* new orgin can't exceed NumLinesInScrlBuff - RowsInWin */
			newscrlwinorg = min (newscrlwinorg,
					(NumLinesInScrlBuff - RowsInWin) );

			/* if newscrlwinorg is same as current value return */
			if (newscrlwinorg == (ScrlDispIndx + WinOrgY))
				return;

			/* do we have to adjust the display buffer */
			if ( (newscrlwinorg + RowsInWin) >
							(ScrlDispIndx + LinesInScreen)) {
				/* SetDisplay.. takes care of adjusting the orig to max */
				SetDisplayFromIdx(newscrlwinorg);
			}
			/* compute the new WinOrgY */
			WinOrgY = (BYTE) (newscrlwinorg - ScrlDispIndx);

			/* invalidate the entire window */
			InvalidateRect(hTermWnd, NULL, FALSE);	/* invalidate the entire area */
			break;

		case 	SB_THUMBPOSITION :					/* Absolute positioning	*/
			if (thumb_pos == cur_pos)		/* do we have to move at all */
				return;					/* if not return immediately */
			
			if (thumb_pos > cur_pos) {	/* display need to go down */
				/* if the new position is not too far, better scroll */
				if ( (thumb_pos - cur_pos) < (WORD)(RowsInWin >> 1) ) {
					ScrollDisplayDown (thumb_pos - cur_pos);
					break;				/* that is all */
				}
				/* we could't scroll. So set the new display buffer
					and the new value of WinOrgY */

				/* Check if we have to bring lines into display buffer */
				if ((thumb_pos + RowsInWin) > (ScrlDispIndx + LinesInScreen))
					SetDisplayFromIdx (thumb_pos);

				/*calculate the new WinOrgY */
				WinOrgY = (BYTE) (thumb_pos - ScrlDispIndx);
			} else {			/* we need to take the display upwards */
				/* if the new position is not too far, better scroll */
				if ((cur_pos - thumb_pos) < (WORD) (RowsInWin >> 1) ) {
					ScrollDisplayUp (cur_pos - thumb_pos);
					break;			/* that is all */
				}
				/* we could't scroll. So set the new display buffer
					and the new value of WinOrgY */

				/* Check if we have to bring lines into display buffer */
				if (thumb_pos < ScrlDispIndx)
					SetDisplayFromIdx (thumb_pos);

				/*calculate the new WinOrgY */
				WinOrgY = (BYTE) (thumb_pos - ScrlDispIndx);
			}
			/* invalidate the entire window */
			InvalidateRect(hTermWnd, NULL, FALSE);	/* invalidate the entire area */
			break;

			break;
		default :
			return;
	}
	UpdateWindow(hTermWnd);
}

/***************************************************************************
*	Routine	:	UpadateScrollThumbs(void)
*	Input	:	
*	Return	:	
*	Synopsis:	Set the Scroll thumbs to according to the current WinOrgXYs
***************************************************************************/
void	UpdateScrollThumbs(void)
{
	/* Update the Thumb position */
	SetScrollPos (hTermWnd, SB_HORZ, WinOrgX, TRUE);
	SetScrollPos (hTermWnd, SB_VERT, ScrlDispIndx + WinOrgY, TRUE);
}

/***************************************************************************
*	Routine	:	CopyIndxLineFromScrlBuff(BYTE indx)
*	Input	:	Index of the line in display buffer to be copied
*	Return	:	
*	Synopsis:	A line is copied from the scroll buffer to the display buffer
***************************************************************************/
void	CopyIndxLineFromScrlBuff (BYTE index)
{
	BYTE	*dispchptr;			/* pointer to char in display mem */
	BYTE	*dispatptr;			/* pointer to attribute in display mem */
	BYTE	FAR *scrlchptr;			/* pointer to char in scroll buffer */
	BYTE	FAR *scrlatptr;			/* pointer to attrib in scroll buffer */
	WORD	indx;
	WORD	indxwrtscrl;		/* index of line wrt to scroll buffer org */
	BYTE	selection;
	WORD	selstartlineindex;		/* Selection start index */
	WORD	selendlineindex;		/* Selection End index in Scrl*/
	BYTE	selstartoffset;			/* offset to start char in start line */
	BYTE	selendoffset;			/* offset of end char in the end line */
	
	dispchptr = GetStdioLine(index);
	dispatptr = dispchptr + ScrnBufSize;
	scrlchptr = ScrollIndexTable [ ScrlDispIndx + index ];
	scrlatptr = scrlchptr + ScrollBuffSize;
	indxwrtscrl = ScrlDispIndx + index;

#define		NOT_SELECTED	0
#define		ALL_SELECTED	1
#define		LEFT_SELECTED	2
#define		RIGHT_SELECTED	3
#define		PART_SELECTED	4
#define		SWAP_NIBBLE(x)		((x >> 4) | (x << 4))

	if (TextSelected || TextSelOn) {
		if (HeadTailPos() == PNT_1_AFTER) {
			selstartlineindex = SelEndLineIndex;
			selendlineindex = SelStartLineIndex;
			selstartoffset = SelEndOffset;
			selendoffset = SelStartOffset;
		} else {
			selstartlineindex = SelStartLineIndex;
			selendlineindex = SelEndLineIndex;
			selstartoffset = SelStartOffset;
			selendoffset = SelEndOffset;
		}
		if ( (indxwrtscrl < selstartlineindex) || (indxwrtscrl > selendlineindex) )
			selection = NOT_SELECTED;
		else if( (indxwrtscrl>selstartlineindex) && (indxwrtscrl<selendlineindex))
			selection = ALL_SELECTED;
		else if ((indxwrtscrl == selstartlineindex) && (indxwrtscrl != selendlineindex))
			selection = RIGHT_SELECTED;
		else if ((indxwrtscrl == selendlineindex) && (indxwrtscrl != selstartlineindex))
			selection = LEFT_SELECTED;
		else
			selection = PART_SELECTED;
	} else {
		selection = NOT_SELECTED;		/* no chars are selected */
	}

	for ( indx = 0; (*scrlchptr != SCROLL_EOL_CHAR) && 
								(indx <= MaxClmIndx); indx++) {
		*dispchptr = *scrlchptr;			/* copy the character */
		if (selection == NOT_SELECTED)
			*dispatptr = *scrlatptr;
		else if (selection == ALL_SELECTED)
			*dispatptr = (BYTE) SWAP_NIBBLE (*scrlatptr);
		else if (selection == RIGHT_SELECTED)
			*dispatptr = (indx < selstartoffset) ?
				(BYTE) *scrlatptr : (BYTE) SWAP_NIBBLE(*scrlatptr);
		else if (selection == LEFT_SELECTED)
			*dispatptr = (indx > selendoffset) ?
				(BYTE) *scrlatptr : (BYTE) SWAP_NIBBLE(*scrlatptr);
		else				/* partlty selected */
			*dispatptr = ((indx < selstartoffset) || (indx > selendoffset)) ?
				(BYTE) *scrlatptr : (BYTE) SWAP_NIBBLE(*scrlatptr);

		dispchptr++;
		dispatptr++;
		scrlchptr++;
		scrlatptr++;

		if (scrlchptr > pScrlBufEnd) {
			scrlchptr = pScrlBuf;
			scrlatptr = pScrlBuf + ScrollBuffSize;
		}
	}
	while (indx <= MaxClmIndx) {			/* Clear the rest of the line	*/
		*dispchptr++ = ' ';
		if (selection == NOT_SELECTED)
			*dispatptr = DEF_COLOR;
		else if (selection == ALL_SELECTED)
			*dispatptr = REV_DEF_COLOR;
		else if (selection == RIGHT_SELECTED)
			*dispatptr = (indx < selstartoffset) ?
				(BYTE) DEF_COLOR : (BYTE) REV_DEF_COLOR;
		else if (selection == LEFT_SELECTED)
			*dispatptr = (indx > selendoffset) ?
				(BYTE) DEF_COLOR : (BYTE) REV_DEF_COLOR;
		else				/* partlty selected */
			*dispatptr = ((indx < selstartoffset) || (indx > selendoffset)) ?
				(BYTE) DEF_COLOR : (BYTE) REV_DEF_COLOR;
		dispatptr++;
		indx++;
	}
}

/***************************************************************************
*	Routine	:	GetUpperLineToDispMem(void)
*	Input	:	None
*	Return	:	TRUE if a line is brought into Disp mem. FALSE otherwise
*	Synopsis:	"moves" the display memory origin wrt the scroll buffer
*				one line up.
***************************************************************************/
BOOL	GetUperLineToDispMem(void)
{
	if ( ! ScrlDispIndx)
		return FALSE;			/* We are already on the upper most line	*/
	ScrlDispIndx--;

	/* take the sDispOrig to one line back */
	sDispOrig -= CellsInLine;

	/* does it go out of screen buffer area */
	if (sDispOrig < sScreenBuff)
		sDispOrig = sScreenBuffLast;		/* Dispplay origin crosses over */

	CopyIndxLineFromScrlBuff(0);
	return TRUE;
}

/***************************************************************************
*	Routine	:	GetBottomLineToDispMem(void)
*	Input	:	
*	Return	:	TRUE if a new line is brought-in. FALSE otherwise
*	Synopsis:	Scrolls the Display memory one more line down
***************************************************************************/
BOOL	GetBottomLineToDispMem (void)
{
	BYTE	*lastlineptr;

	if (ScrlDispIndx + LinesInScreen >= NumLinesInScrlBuff)
		return FALSE;							/* no more lines at bottom	*/
	ScrlDispIndx++;

	sDispOrig += CellsInLine;				/* increment display mem ptr	*/
	if (sDispOrig > sScreenBuffLast)		/* does it wrap around ? 		*/
		sDispOrig = sScreenBuff;

	lastlineptr = sDispOrig + CellsInLine * MaxRowIndx;
	if (lastlineptr > sScreenBuffLast) {	/* check for wrap around */
		lastlineptr = sScreenBuff + (lastlineptr - sScreenBuffLast);
		lastlineptr -= CellsInLine;			/* reduce one line length */
	}

	/* copy the last line to display memory */
	CopyIndxLineFromScrlBuff(MaxRowIndx);
	return TRUE;
}

/***************************************************************************
*	Routine	:	ResetSearchPtr(void)
*	Input	:	
*	Return	:	
*	Synopsis:	This will reset the search ptr to the beginnig or the
*				top most line currently in the display window.
***************************************************************************/
void	ResetSearchPtr(void)
{
	SearchPtr = ScrollIndexTable [ScrlDispIndx + WinOrgY];
}

/***************************************************************************
*	Routine	:	ScrollTimeKeyHandler (BYTE key)
*	Input	:	
*	Return	:	
*	Synopsis:	Handles the key board commands during Scroll mode
***************************************************************************/
void	ScrollTimeKeyHandler (HWND hWnd, BYTE key)
{
	switch (key) {

//		case ASCII_ESCAPE :
//			ExitScrollMode();
//			break;

		case VK_PRIOR :
			ScrollInY (SB_PAGEUP, 0);
			ResetSearchPtr();
			break;
		case VK_NEXT :
			ScrollInY (SB_PAGEDOWN, 0);
			ResetSearchPtr();
			break;
		case VK_END :
			ScrollInY (SB_THUMBPOSITION, NumLinesInScrlBuff - RowsInWin);
			ResetSearchPtr();
			break;
		case VK_HOME :
			ScrollInY (SB_THUMBPOSITION, 0);
			ResetSearchPtr();
			break;
		case VK_UP :
			ScrollInY (SB_LINEUP, 0);
			ResetSearchPtr();
			break;
		case VK_DOWN :
			ScrollInY (SB_LINEDOWN, 0);
			ResetSearchPtr();
			break;
		case VK_LEFT :
			ScrollX(SB_LINEUP, 0);
			ResetSearchPtr();
			break;
		case VK_RIGHT :
			ScrollX(SB_LINEDOWN, 0);
			ResetSearchPtr();
			break;

		case 's' :
		case 'S' :
			if (InvokeDialog (hTermWnd, (LPSTR) "SCROLLSRCH", ScrlSrchDlgProc))
				FindPattern();
			break;

		case 'n' :
		case 'N' :
			FindPattern();
			break;

		default :
			MessageBeep(0);
			break;
	}
}

/***************************************************************************
*	Routine	:	ScrlSrchDlgProc (HWND, WORD, WORD, LONG )
*	Input	:	
*	Return	:	TRUE if user selected a pattern for search, FALSE otherwise
*	Synopsis:	Handles the search dialog during scroll mode
***************************************************************************/
int FAR PASCAL
ScrlSrchDlgProc (HWND hDlg, WORD message, WORD wParam, LONG lParam)
{
	int		count;

	switch (message) {

		case WM_INITDIALOG :
			DLGCenterDialog(hDlg);

			/* Set the Phone number, if any,  here */
			if (SearchPattern[0])
				SetDlgItemText ( hDlg, IDC_SCRL_PATTERN, SearchPattern);
			CheckRadioButton (hDlg, IDC_SCRL_TOS, IDC_SCRL_TSB,
						IDC_SCRL_TOS);
			/* by default search starts at the begging line of display window */
			SearchPtr = ScrollIndexTable [ScrlDispIndx + WinOrgY];
			return FALSE;

		case WM_COMMAND :
			switch (wParam) {
				case IDC_SCRL_TOS :
				case IDC_SCRL_TSB :
					if (wParam == IDC_SCRL_TOS)
						SearchPtr = ScrollIndexTable[ScrlDispIndx + WinOrgY];
					else
						SearchPtr = pScrlBuf + ScrlHead;
					CheckRadioButton (hDlg, IDC_SCRL_TOS,
									IDC_SCRL_TSB, wParam);
					return TRUE;

				case IDOK :
					/* Get the pattern to be searched */
					count = GetDlgItemText(hDlg, IDC_SCRL_PATTERN,
						(LPSTR)SearchPattern, MAX_SRCH_PAT_LEN);
					SearchPattern[count] = NULL;
					if (count)
						EndDialog(hDlg, TRUE);
					else
						EndDialog(hDlg, FALSE);		/* no pattern */
					return TRUE;
				case IDCANCEL :
					EndDialog(hDlg, FALSE);
					return TRUE;
				default :
					return FALSE;
			}
	}
	return FALSE;		/* Did not process the command. So return FALSE		*/
}
/************************    Last line ************************************/

