/****************************************************************************
*	File		:	SELECT.C
*	Purpose		:	Scroll buffer select related routines for MEW
*	Package		:	MultiExpress (Windows) - Version 2.00
*	Authors		:	Vidy
*	Date		:	27th MAR 1992.
****************************************************************************/

#include	"windows.h"

#include	"main.h"
#include	"mew.h"
#include	"select.h"
#include	"wstdio.h"
#include	"scroll.h"

/* following are globals in text selection */
BYTE	TextSelOn = FALSE;		/* Text selection is on */
BYTE	TextSelected = FALSE;	/* Boolean set if text is selected */

/* all following "*Index" are indices in ScrollIndexTable */
/* These values doesn't make sense outside scroll mode */
WORD	SelStartLineIndex;		/* Selection start index */
WORD	SelEndLineIndex;		/* Selection End index in Scrl*/
WORD	SelCurLineIndex;		/* index of the current selection line */

BYTE	SelStartOffset;			/* offset to start char in start line */
BYTE	SelEndOffset;			/* offset of end char in the end line */
BYTE	SelCurOffset;			/* offset of current char in current line */

WORD	SelLines;				/* number of lines of selected text */

WORD	RepeaterId = FALSE;		/* Auto repeater for mouse position outside */
FARPROC	lpSelectRepeater;
BYTE	InRepeater = FALSE;


/***************************************************************************
*	Routine	:	ComparePos()
*	Input	:	two indices and offsets of points in scroll buffer
*	Return	:	PNT_1_BEFORE, PNT_1_AFTER or PNT_1_EQUAL
*	Synopsis:	compares two positions in the scroll buffer
***************************************************************************/
int		ComparePos(WORD head_ind, WORD pos_ind, BYTE head_off, BYTE pos_off)
{
	if (head_ind < pos_ind)
		return PNT_1_BEFORE;
	if (head_ind == pos_ind) {
		if (head_off < pos_off)
			return PNT_1_BEFORE;
		else if (head_off == pos_off)
			return PNT_1_EQUAL;
	}
	return PNT_1_AFTER;
}

/***************************************************************************
*	Routine	:	HeadTailPos (void)
*	Input	:	
*	Return	:	PNT_1_BEFORE, PNT_1_AFTER ot PNT_1_EQUAL
*	Synopsis:	Compare the head and tail positions
***************************************************************************/
int		HeadTailPos(void)
{
	return ComparePos(SelStartLineIndex, SelEndLineIndex,
										SelStartOffset, SelEndOffset);
}

/***************************************************************************
*	Routine	:	HeadCurPos(void)
*	Input	:	
*	Return	:	PNT_1_BEFORE, PNT_1_AFTER ot PNT_1_EQUAL
*	Synopsis:	Compares the head and current positions
***************************************************************************/
int		HeadCurPos(void)
{
	return ComparePos(SelStartLineIndex, SelCurLineIndex,
										SelStartOffset, SelCurOffset);
}

/***************************************************************************
*	Routine	:	CurTailPos(void)
*	Input	:	
*	Return	:	PNT_1_BEFORE, PNT_1_AFTER ot PNT_1_EQUAL
*	Synopsis:	Compares the current and tail positions
***************************************************************************/
int		CurTailPos(void)
{
	return ComparePos(SelCurLineIndex, SelEndLineIndex,
										SelCurOffset, SelEndOffset);
}

/***************************************************************************
*	Routine	:	RevAtrib (BYTE index, BYTE from, BYTE to)
*	Input	:	
*	Return	:	
*	Synopsis:	Reverses Attributes in memory and invalidates window area
*				if displayed.
*	NOTE	:
*		All coordinates are WRT the display buffer (NOT display window)
***************************************************************************/
void	RevAtrib(BYTE index, BYTE from, BYTE to)
{
	BYTE	count;
	BYTE	*ptr;
	BYTE	ch;
	RECT	inv_rect;

	to = min (MaxClmIndx, to);

	ptr = GetStdioLine(index);
	ptr += (from + ScrnBufSize);
	for (count = (BYTE) (to - from + 1); count; count--, ptr++) {
		ch = *ptr;
		*ptr = (BYTE) ((ch << 4) | (ch >> 4));
	}
	/* invalidate any display area under this */
	if ((WinOrgY + RowsInWin) < index)	/* display is above this line */
		return;
	if (index < WinOrgY)
		return;				/* display is below this line */
	if (from > (WinOrgX + ClmsInWin))	/* display at left side */
		return;
	if (to < WinOrgX)	/* display is at right side */
		return;
	/* now there is something to be redrawn */
	inv_rect.top = (index - WinOrgY) * nStdioCharHeight;
	inv_rect.bottom = inv_rect.top + nStdioCharHeight;
	inv_rect.left = (from - WinOrgX) * nStdioCharWidth;
	inv_rect.right = (to - WinOrgX) * nStdioCharWidth + nStdioCharWidth;
	inv_rect.right = min (inv_rect.right, TermWindow.right);
	InvalidateRect(hTermWnd, &inv_rect, TRUE);
}

/***************************************************************************
*	Routine	:	EraseSelection(void)
*	Input	:	None
*	Return	:	None
*	Synopsis:	Given a Selection range and a display window instance, this
*				routine will deselect the buffer and set the display
*				back to normal.
***************************************************************************/
void	EraseSelection(void)
{
	WORD	lines;
	BOOL	first_line_passed = FALSE;
	BYTE	dispindex;

	if (!TextSelected)
		return;

	if (SelEndLineIndex < ScrlDispIndx)	/* selection before scrn */
		return;

	if (SelStartLineIndex > (ScrlDispIndx + LinesInScreen))
		return;			/* Selection after Screen */

	if (SelStartLineIndex == SelEndLineIndex) {	/* single line selection */
		RevAtrib( (BYTE) (SelStartLineIndex - ScrlDispIndx),
				SelStartOffset, SelEndOffset);
		return;
	}
	lines = 0;
	if (SelStartLineIndex < ScrlDispIndx) {	/* Skip all the lines wich are above */
		while (SelStartLineIndex + lines < ScrlDispIndx)
			lines++;
		dispindex = 0;
	} else {		/* first line is in disp mem */
		dispindex = (BYTE) (SelStartLineIndex - ScrlDispIndx);
		RevAtrib( dispindex, SelStartOffset, CharsInLine);
		dispindex++;
		lines++;
	}
	for (; (dispindex < LinesInScreen) &&
			( (SelStartLineIndex + lines) <= SelEndLineIndex);
											lines++, dispindex++) {
		if ((SelStartLineIndex +lines) == SelEndLineIndex)	/* This is last line */
			RevAtrib (dispindex, 0, SelEndOffset);
		else
			RevAtrib (dispindex, 0, CharsInLine); 
	}
	UpdateWindow (hTermWnd);
}

/***************************************************************************
*	Routine	:	StartSelection (DWORD mouse_xy)
*	Input	:	the mouse position in pixel coordinates
*	Return	:	
*	Synopsis:	called when selection of buffer starts, ie when left button
*				is pressed.
*	NOTE	: 	This routine is called whenever the left button is pressed
*				down in the scroll mode. It selects the character on which
*				the mouse currently is.
***************************************************************************/
void	StartSelection(DWORD mouse_xy)
{
	int		mouse_x, mouse_y;
	BYTE	mwin_x, mwin_y;			/* position in the win in char base */

	/* Erase the previous selection if any */
	if (TextSelected)
		EraseSelection();

	RepeaterId = 0;

	/* Get the mouse coordinates wrt to the window */
	mouse_y = (int)HIWORD (mouse_xy);
	mouse_x = (int)LOWORD (mouse_xy);

	/* calculate the mouse coordinates wrt character base */
	mwin_x = (BYTE) (mouse_x / nStdioCharWidth);
	mwin_y = (BYTE) (mouse_y / nStdioCharHeight);

	/* reverse attribute of the first char selected */
	RevAtrib ((BYTE) (WinOrgY + mwin_y), (BYTE) (WinOrgX + mwin_x),
					(BYTE) (WinOrgX + mwin_x));
	UpdateWindow(hTermWnd);

	/* Set the Selection Start and End indices and offsets */
	SelStartLineIndex = SelEndLineIndex = ScrlDispIndx + WinOrgY + mwin_y;
	SelStartOffset = SelEndOffset = WinOrgX + mwin_x;

	TextSelOn = TRUE;
	TextSelected = FALSE;

	SetCapture(hTermWnd);		/* capture the mouse movements */
	lpSelectRepeater = MakeProcInstance (SelectRepeater, hInst);
}

/***************************************************************************
*	Routine	:	EndSelection(void)
*	Input	:	None
*	Return	:	None
*	Synopsis:	called when selection ends, ie when left button is released
***************************************************************************/
void	EndSelection(void)
{
	WORD	tmpindx;
	BYTE	tmpoffset;

	TextSelOn = FALSE;
	TextSelected = TRUE;

	if (RepeaterId) {
		KillTimer(NULL, RepeaterId);
		RepeaterId = 0;
	}
	/* if the final selection is reverse, make forward by exchainging
	Start and End indices and offsets */
	if (HeadTailPos() == PNT_1_AFTER) {
		tmpindx = SelStartLineIndex;
		SelStartLineIndex = SelEndLineIndex;
		SelEndLineIndex = tmpindx;
		tmpoffset = SelStartOffset;
		SelStartOffset = SelEndOffset;
		SelEndOffset = tmpoffset;
	}
		
	/* calculate the number of lines selected */
	SelLines = SelEndLineIndex - SelStartLineIndex + 1;

	ReleaseCapture();		/* Release the capture of mouse */
	FreeProcInstance (lpSelectRepeater);

}

/***************************************************************************
*	Routine	:	MinusY(void)
*	Input	:	None
*	Return	:	None
*	Synopsis:	
***************************************************************************/
void	MinusY(void)
{
	if (SelEndLineIndex < (ScrlDispIndx + WinOrgY))
		return;

	if (SelEndLineIndex == (ScrlDispIndx + WinOrgY))
		if (SelEndOffset == 0)
			if (! ScrollDisplayUp(1)) /* no more chars to select */
				return;	

	SelCurLineIndex = ScrlDispIndx + WinOrgY;	/* Select till the top most line */
	SelCurOffset = 0;				/* select upto the first char */

	RevLastToCurLines();
	RevMiddleChar();
	SelEndOffset = SelCurOffset;
	SelEndLineIndex = SelCurLineIndex;
	return;
}

/***************************************************************************
*	Routine	:	
*	Input	:	None
*	Return	:	None
*	Synopsis:	
***************************************************************************/
void	PlusY(void)
{
	if (SelEndLineIndex > (ScrlDispIndx + WinOrgY + RowsInWin - 1))
		return;

	if (SelEndLineIndex == (ScrlDispIndx + WinOrgY + RowsInWin - 1))
		if ( SelEndOffset >= MaxClmIndx)
			if (!ScrollDisplayDown(1))		/* no more chars to select */
				return;

	SelCurLineIndex = ScrlDispIndx + WinOrgY + RowsInWin - 1;
	SelCurOffset = MaxClmIndx;

	RevLastToCurLines();
	RevMiddleChar();
	SelEndOffset = SelCurOffset;
	SelEndLineIndex = SelCurLineIndex;
}

/***************************************************************************
*	Routine	:	
*	Input	:	None
*	Return	:	None
*	Synopsis:	
***************************************************************************/
void	MinusX(void)
{
	ScrollRight(1);

	if ( SelEndOffset) {
		SelCurLineIndex = SelEndLineIndex;
		SelCurOffset = WinOrgX;
		RevAtrib ((BYTE) (SelEndLineIndex - ScrlDispIndx),
					SelCurOffset, SelEndOffset);
		RevMiddleChar();
		SelEndOffset = SelCurOffset;
	}
}

/***************************************************************************
*	Routine	:	PlusX(void)
*	Input	:	None
*	Return	:	None
*	Synopsis:	Called when the cursor is out of the right side of disp win
***************************************************************************/
void	PlusX(void)
{
	ScrollLeft(1);

	if (SelEndOffset < MaxClmIndx) {
		SelCurLineIndex = SelEndLineIndex;
		SelCurOffset = (BYTE) (WinOrgX + ClmsInWin - 1);
		RevAtrib ((BYTE) (SelEndLineIndex - ScrlDispIndx),
					SelEndOffset, SelCurOffset);
		RevMiddleChar();
		SelEndOffset = SelCurOffset;
	}
}

/***************************************************************************
*	Routine	:	RevPoint()
*	Input	:	None
*	Return	:	None
*	Synopsis:	Given a point in the scroll buffer, show the character in
*				reverse attributes
***************************************************************************/
void	RevPoint (WORD index, BYTE offset)
{
	if ((index < ScrlDispIndx) || (index >= (ScrlDispIndx + LinesInScreen))) {
//		MessageBox(hTermWnd, "Error point not displayed", "", MB_OK);
		return;		/* not in the display memory. an error condition */
	}
	RevAtrib ((BYTE) (index - ScrlDispIndx), offset, offset);
}

/***************************************************************************
*	Routine	:	RevLastToCurLines(void)
*	Input	:	None
*	Return	:	None
*	Synopsis:	Reverse the attributes from the last selected char to the
*				current selection.
***************************************************************************/
void	RevLastToCurLines(void)
{
	int		nlines;
	int		index;

	if (SelCurLineIndex == SelEndLineIndex)	{	/* if end and cur are in same line */
		if (CurTailPos() == PNT_1_BEFORE)
			RevAtrib((BYTE) (SelCurLineIndex - ScrlDispIndx),
				SelCurOffset, SelEndOffset);
		else
			RevAtrib((BYTE) (SelCurLineIndex - ScrlDispIndx),
				SelEndOffset, SelCurOffset);
		return;
	}
	/* if cur and last are not in same lines, finish first and last lines */
	if (CurTailPos() == PNT_1_BEFORE) {
		/* reverse chars from cur pos to end of line */
		RevAtrib ((BYTE) (SelCurLineIndex - ScrlDispIndx),
					SelCurOffset, MaxClmIndx);
		RevAtrib ((BYTE) (SelEndLineIndex - ScrlDispIndx),
					0, SelEndOffset);
		nlines = SelEndLineIndex - SelCurLineIndex - 1;
		index = SelCurLineIndex - ScrlDispIndx + 1;

	} else {
		/* reverse chars from cur pos to end of line */
		RevAtrib ((BYTE) (SelCurLineIndex - ScrlDispIndx), 0, SelCurOffset);
		RevAtrib ((BYTE) (SelEndLineIndex - ScrlDispIndx),
												SelEndOffset, MaxClmIndx);
		nlines = SelCurLineIndex - SelEndLineIndex - 1;
		index = SelEndLineIndex - ScrlDispIndx + 1;
	}
	for (; nlines; nlines--, index++)
		RevAtrib((BYTE) index, 0, MaxClmIndx);
}

/***************************************************************************
*	Routine	:	RevMiddleChar(void)
*	Input	:	None
*	Return	:	None
*	Synopsis:	Reverses the attributes of the character among start, end
*				and currnet, whichever is in middle position.
***************************************************************************/
void	RevMiddleChar(void)
{
	int		head_tail_pos;

	/* now we will have to reverse either start or end or current point
	depending on which is in the middle */

	if ((head_tail_pos = HeadTailPos()) == PNT_1_BEFORE) {	/* Head is before */
		if (CurTailPos() == PNT_1_AFTER)
			RevPoint(SelEndLineIndex, SelEndOffset);
		else if (HeadCurPos() == PNT_1_BEFORE)
			RevPoint(SelCurLineIndex, SelCurOffset);
		else
			RevPoint(SelStartLineIndex, SelStartOffset);
	} else if (head_tail_pos == PNT_1_EQUAL) {	/* head and tail points are same */
		RevPoint(SelStartLineIndex, SelStartOffset);
	} else {							/* Head is after tail */
		if (CurTailPos() == PNT_1_BEFORE)
			RevPoint(SelEndLineIndex, SelEndOffset);
		else if (HeadCurPos() == PNT_1_BEFORE)
			RevPoint(SelStartLineIndex, SelStartOffset);
		else
			RevPoint(SelCurLineIndex, SelCurOffset);
	}
}

/***************************************************************************
*	Routine	:	NormalXY(void)
*	Input	:	None
*	Return	:	None
*	Synopsis:	callled when new mouse positionis within the display window.
***************************************************************************/
void	NormalXY(void)
{
	RevLastToCurLines();	/* reverse the selection of last to cur points */

	RevMiddleChar();
}

/***************************************************************************
*	Routine	:	SelectRepeater()
*	Input	:	None
*	Return	:	None
*	Synopsis:	This is the timer function to simulate repeating selection
*				when mouse is dragged out of the display window.
***************************************************************************/
WORD FAR PASCAL 
SelectRepeater (HWND hWnd, WORD wMsg, int ID, DWORD dwTime)
{
	POINT		cur_pos;
	RECT		win_pos;
	long		mouse_pos;
	long 		x, y;
	
	_asm {
		mov		al, TRUE
		xchg	al, InRepeater
		or		al, al
		jz		NotBusy
	}
	return 0;

NotBusy:
	if (RepeaterId) {
		KillTimer(NULL, RepeaterId);
		RepeaterId = 0;
	}
	if ( ! TextSelOn)
		goto exit_point;

	/* Compute the mouse position */
	/* first get the cursor position */
	GetCursorPos(&cur_pos);
	GetWindowRect (hTermWnd, (LPRECT) &win_pos);

	y = cur_pos.y - win_pos.top;
	x = cur_pos.x - win_pos.left;
	mouse_pos = ((y << 16) & 0xFFFF0000) | ( x  & 0xFFFF);

	DisplaySelection(mouse_pos);

exit_point:
	InRepeater = FALSE;
	return TRUE;				/* Text selection has ended */
}

/***************************************************************************
*	Routine	:	DisplaySelection(DWORD position)
*	Input	:	The encoded x y position of the mouse. (-)ve means it is
*				on left side or top side of the terminal window
*	Return	:	
*	Synopsis:	This routine is called when text selection is on in scroll
*				mode.
***************************************************************************/
void	DisplaySelection(DWORD position)
{
	int		mouse_x, mouse_y;
	BYTE	mwin_x, mwin_y;			/* position in the win in char base */
	BOOL	mouse_out;

	/* Get the mouse coordinates wrt to the window */
	mouse_y = (int)((position >> 16) & 0xFFFF);
	mouse_x = (int)(position & 0xFFFF);

	mouse_out = TRUE;			/* assume mouse_out = TRUE first */
	if (mouse_y < 0) {
		MinusY();
	} else if (mouse_y >= TermWindow.bottom) {
		PlusY();
	} else if (mouse_x < 0) {
		MinusX();
	} else if ((mouse_x >= TermWindow.right) ||
		(CharsInLine < (BYTE) (WinOrgX + mouse_x / nStdioCharWidth)) ) {
		PlusX();
	} else {
		mouse_out = FALSE;		/* mouse is within the window. */
	}
	if (mouse_out) {		// we need to set the repeater timer
		if ( !RepeaterId)
			RepeaterId = SetTimer ( NULL, NULL, SEL_REPEAT_TIMER,
												lpSelectRepeater);
		goto gout;
	}

	/* once here means the current mouse position is within the win */
	/* calculate the mouse coordinates wrt character base */
	mwin_x = (BYTE) (mouse_x / nStdioCharWidth);
	mwin_y = (BYTE) (mouse_y / nStdioCharHeight);
	
	SelCurLineIndex = ScrlDispIndx + WinOrgY + mwin_y;
	SelCurOffset = WinOrgX + mwin_x;

	if ((SelCurLineIndex == SelEndLineIndex) && (SelCurOffset == SelEndOffset))
		return;

	NormalXY();

	SelEndLineIndex = SelCurLineIndex;
	SelEndOffset = SelCurOffset;

gout:
	UpdateWindow(hTermWnd);
}

/***************************    Last   line  *****************************/
