/*----------------------------------------------------------------------------
*	File		:	WSTDIO.C
*	Purpose		:	Terminal window related routines (Scrolls, WindProc)
*	Package		:	MultiExpress (Windows) - Version 2.00
*	Authors		:	Vidy
*	Date		:	21st December 1991.
*---------------------------------------------------------------------------*/

#include	<windows.h>
#include	"mew.h"
#include "term.h"
#include	"session.h"
#include	"main.h"

#include	"wstdio.h"
#include	"windisp.h"
#include	"emlnfuns.h"
#include	"emlncons.h"
#include	"emlntype.h"
#include	"emlndata.h"
#include	"scroll.h"
#include	"search.h"
#include	"fnkey.h"
#include	"select.h"
#include	"fonts.h"
#include	"status.h"
#include	"kbd.h"
#include	"time.h"
#include	"looks.h"
#include	"clrsetup.h"

extern	HANDLE	hInst;	 		/* application instance handle */
extern	HWND	hMainWnd;		/* main window handle */
extern	HWND	hTermWnd;		/* terminal window handle */
extern	HCURSOR	hCurCursor;		/* current cursor */
extern	HCURSOR	hIBeam;			/* I-Beam cursor inside term */
extern	RECT	TermWndRect;	/* terminal window dimensions */
extern	BOOL	InMenu;			/* informs if we are in a menu selection */

/*********   for testing ******/
extern	void	WritePort(BYTE wParam);
extern	void	ReadAndDisp(void);
extern	void	DownComPort(void);
extern	void	LoosingFocus(void);

BYTE	LineStatus = 0;

/**************************************************************************
*	Declarations
***************************************************************************\

/*	The Geometrical parameters of the Screen  */

BYTE	LinesInScreen = DEF_LINES_IN_SCREEN; /* Number of lines per screen */
BYTE	CharsInLine = DEF_CHARS_IN_LINE; /* Number of chars in a line */
BYTE	CellsInLine;	/* Space for a NULL also */
BYTE	MaxClmIndx;		/* Maximum char index in a line (max nCurX val) */
BYTE	MaxRowIndx;		/* Maximum Line Index in Screen (max nCurY val) */
RECT	TermWindow;		/* The actual window area we use */
WORD	ScrnBufSize;	/* = CellsInLine * LinesInScreen */

char	*sScreenBuff;	/* Circular Array of characters on TTY */
						/* this remains fixed after the memory allocation */
char	*sAttribBuff;	/* The attribute buffer space */
char	*sScreenBuffLast;	/* for computaional convenience */
							/* = sScreenBuff + (CellsInLine * LinesInScreen); */

char	*sDispOrig;		/* Current Display origin */
						/* this keeps changing as the screen scrolls */
BYTE	nCurX;			/* Current cursor position wrt DispOrig */
BYTE	nCurY;

DWORD	StdiobkRGB;						/* background color */
DWORD	StdiotextRGB;					/* text color */
#define Stdio_FONT SYSTEM_FIXED_FONT	/* font used for display */
HFONT	hOldFont;
BOOL	bStdioQuit;
BOOL	bInited = FALSE;
BOOL	InFocus = TRUE;
BOOL	Opened = FALSE;

/*	Display geometry related definitions */
BYTE	WinOrgX;			/*	Origin of display window wrt sDispBuffer */
BYTE	WinOrgY;			/* Changes when the user scrolls the display */
BYTE	RowsInWin;			/* Changes when resizing occurs or a differnt */
BYTE	ClmsInWin;			/* font is selected	*/
BYTE	MinXShift;			/* Optimal min value of shift in X dir */
short	nStdioCharWidth;	/* font details */
short	nStdioCharHeight; 	/* width and height of Stdio font chars */
BYTE	CtrlFlag, ShiftFlag, AltFlag;

BYTE	CurAttrib;			/* Current attributes */
BYTE	NormAttrib;			/* Normal terminal attributes */

char	BlankLine[] = "                                                 \
                                                                         \
                                                                      ";

/***************************************************************************
*	Routine	:	SetupWinMatrix
*	Input	:	
*	Return	:	
*	Synopsis:	This routine is finds out the number of rows and columns
*				That can be displayed in the current size of the window.
*				(updates RowsInWin and ClmsInWin). It avoids lines being
*				partly displayed.
*	NOTE	:	Called whenever the display size chages or the font
*				size chages
***************************************************************************/
void	SetupWinMatrix ( void)
{
	RECT	rcClient;
	WORD	height;
	WORD	width;
	BYTE	oldclms, oldrows;
	int		new_win_x, new_win_y;

	if (!hTermWnd)
		return;

	oldclms = ClmsInWin;
	oldrows = RowsInWin;
	/* Get the Present Client area size of the display window */
	GetClientRect(hTermWnd, (LPRECT) &rcClient);
	height = rcClient.bottom - rcClient.top;
	width = rcClient.right - rcClient.left;

	/* Calculate How many complete rows can be displayed */
	RowsInWin = (BYTE) ((height - BOTTOM_MARGIN) / nStdioCharHeight);
	RowsInWin = min (RowsInWin, LinesInScreen);

	/* calculate how many columns can  be displayed */
	ClmsInWin = (BYTE) (width  / nStdioCharWidth);
	ClmsInWin = min (ClmsInWin, CharsInLine);

	/* This is the actual size window we will be scrolling */
	TermWindow.top = rcClient.top;
	TermWindow.left = rcClient.left;
	TermWindow.right = ClmsInWin * nStdioCharWidth;
	TermWindow.bottom = RowsInWin * nStdioCharHeight;

	/* Set the Scroll range */
//	SetScrollRange (hTermWnd, SB_HORZ, 0, CharsInLine - 1, FALSE);
	SetScrollRange (hTermWnd, SB_HORZ, 0, CharsInLine - ClmsInWin, FALSE);

	if (ScrollLock) 
		SetScrollRange (hTermWnd, SB_VERT, 0,
						NumLinesInScrlBuff - RowsInWin, FALSE);
	else
		SetScrollRange (hTermWnd, SB_VERT, 0,
						LinesInScreen - RowsInWin, FALSE);

	if (Opened) {	/* We need to know if the number of rows or colums
		are increasing. If so we may need to update the origins and
		redraw the screen. This is needed because we don't allow any
		rows after the last line to be displayed */
		if (nCurY < RowsInWin)
			new_win_y = 0;
		else 
			new_win_y = nCurY - (RowsInWin >> 1);

		/* WinOrgY can't be more than (LinesInScreen - RowsInWin) */
		new_win_y = min (new_win_y, (int)(LinesInScreen - RowsInWin));

		if (nCurX <= ClmsInWin)
			new_win_x = 0;
		else
			new_win_x = (nCurX - (ClmsInWin >> 1));

		MoveWinOrg ((BYTE) new_win_x, (BYTE) new_win_y);
	}
	/* Setup the MinXShift variable, it may depends on ClmsInWin */
	MinXShift = XSHIFT_QUANTUM;
}

/***************************************************************************
*	Routine	:	CharOutWin
*	Input	:	BYTE ch to out
*	Return	:	
*	Synopsis:	If Cursor is visible put the char
*				if not cursor is not visible and FollowCursor is set
*				reposition the window and display the char
***************************************************************************/
void	CharOutWin (BYTE ch)
{
	if (RowsInWin == 0 || ClmsInWin == 0)	/* you just can't display	*/
		return;

	if (CursorVisible()) {						/* Cursor is visible */
		TextOut (hTermDC, (nCurX - WinOrgX) * nStdioCharWidth,
					(nCurY - WinOrgY) * nStdioCharHeight, (LPSTR)&ch, 1);
	} else if (PbkEntry.TrmParams.FollowCursor) {		/* Cursor not visible */
		AdjustWin();					/* We need to follow cursor */
		TextOut (hTermDC, (nCurX - WinOrgX) * nStdioCharWidth,
					(nCurY - WinOrgY) * nStdioCharHeight, (LPSTR)&ch, 1);
	}
}

/***************************************************************************
*	Routine	:	CursorVisible
*	Input	:	None
*	Return	:	TRUE or FALSE
*	Synopsis:	Check if the Cursor is visible in the window
***************************************************************************/
int		CursorVisible(void)
{
	if (nCurX >= WinOrgX && nCurX < (WinOrgX + ClmsInWin)
		&& nCurY >= WinOrgY &&	nCurY < (WinOrgY + RowsInWin) )
		return TRUE;
	return FALSE;
}

/***************************************************************************
*	Routine	:	AdjustWin
*	Input	:	None
*	Return	:	None
*	Synopsis:	This routine is called when the cursor is not visible and
*				The cursor has to be followed when characters are drawn.
*				It adjusts the WinOrgX and WinOrgY variables and displays
*				the new area.
***************************************************************************/
void	AdjustWin(void)
{
	BYTE	min_shift;			/* Minimum shift needed to show the cursor */
	BYTE	xshift;				/* actual shift we make in X direction */
	BYTE	yshift;				/* actual shift we make in Y direction */
	BYTE	redraw;			/* should we redraw the screen or scroll */
	BYTE	shift_right, shift_up;		/* scroll direction */
	BYTE	new_x, new_y;

	/* first calculate the amount of shift and direction */
	/* along X axis */
	if (nCurX < WinOrgX) {						/* We need to shift left */
		shift_right = FALSE;
		min_shift = WinOrgX - nCurX; 			/* minimum shift needed */
		xshift = min_shift + MinXShift;			/* optimum shift */
		xshift = min (WinOrgX, xshift);			/* not beyond first colum */
		if(xshift > (BYTE)(ClmsInWin >> 2))		/* Is it more than half-  */
			redraw = TRUE;						/* better redraw screen */
		new_x = WinOrgX - xshift;			/* new X origin */
	} else if ( nCurX >= (WinOrgX + ClmsInWin)) {	/* shift to right */
		shift_right = TRUE;
		min_shift = (BYTE) (nCurX - (WinOrgX + ClmsInWin) + 1);
		xshift = min_shift + MinXShift;
		if(xshift > (BYTE) (ClmsInWin >> 2))	/* Is it more than half-  */
			redraw = TRUE;						/* better redraw screen */
		new_x = WinOrgX + xshift;				/* new X origin */
	} else {							/* no movement in the X direction */
		xshift = 0;
		redraw = FALSE;
		new_x = WinOrgX;
	}
	/* along Y axis */							
	if (nCurY < WinOrgY)	{					/* Upward shift needed */
		shift_up = TRUE;
		yshift = WinOrgY - nCurY;				/* minimu shift needed */
		if(yshift > (BYTE) (RowsInWin >> 2))	/* Is it more than half-  */
			redraw = TRUE;						/* better redraw screen */
		new_y = WinOrgY - yshift;
	} else if (nCurY >= (WinOrgY + RowsInWin)) {	/* Shift downwards */
		shift_up = FALSE;
		yshift = (BYTE) (nCurY - (WinOrgY+RowsInWin) + 1);/* shift needed */
		if(yshift > (BYTE)(RowsInWin >> 2))		/* Is it more than half-  */
			redraw = TRUE;						/* better redraw screen */
		new_y = WinOrgY + yshift;
	} else {						/* no movement in the Y direction */
		yshift = 0;
		redraw = FALSE;
		new_y = WinOrgY;
	}
	/* If redraw is set to TRUE we need to use MoveWinOrg call */
	if (redraw) {
		MoveWinOrg(new_x, new_y);
		return;						/* it is over. don't fall thru */
	}
	if (xshift) {				/* do we need to scroll in X direction */
		if (shift_right) {		/* which direction to shift */
			ScrollLeft(xshift);	/* scrolling left means moving right !! */
		} else {
			ScrollRight(xshift);
		}
	}
	if (yshift) {				/* do we need to scroll in Y direction */
		if (shift_up) {			/* which direction to go */
			ScrollDown(yshift);	/* Scroll down takes the view area up !! */
		} else {
			ScrollUp(yshift);
		}
	}
}

/***************************************************************************
*	Routine	:	ScrollLeft
*	Input	:	count number of columns to Scroll
*	Return	:	TURE if done. FALSE otherwise
*	Synopsis:	ScrollLeft takes the visible area of the display to the
*				right side of the display.
*	Note	:	Routine must be called only after verifying that there is
*				enough area of the screen to scroll. It doesn't check
*				for the argument while scrolling.
***************************************************************************/
int		ScrollLeft(BYTE count)
{
	DWORD	old_bkgnd;

	if (WinOrgX >= MaxClmIndx)
		return FALSE;

	if ((WinOrgX + count) >= CharsInLine)
		count = (BYTE) (CharsInLine - WinOrgX - 1);

	if ( ! ScrollLock && InFocus) {
		CurHideCnt++;
		HideCaret (hTermWnd);
	}
	WinOrgX += count;				/* This is the new X origin */

	/* we need to save the bakground color and set it to white */
//	old_bkgnd = SetBkColor(hTermDC, ColorTable [DEF_BKGND]);
	old_bkgnd = SetBkColor(hTermDC, ColorTable [ColorSetup.TermBg]);
	/* issue a block scroll */
	ScrollWindow(hTermWnd, -(count * nStdioCharWidth), 0,
			   	NULL, (LPRECT) &TermWindow);
	/* restore back ground after scroll */
	SetBkColor(hTermDC, old_bkgnd);

	if ( !ScrollLock && CursorVisible())
	   	ReDrawCursor();

	/* Update the Thumb position */
	SetScrollPos (hTermWnd, SB_HORZ, WinOrgX, TRUE);

	UpdateWindow(hTermWnd);		/* This will force a paint message */
	return TRUE;
}

/***************************************************************************
*	Routine	:	ScrollRight
*	Input	:	count number of columns to scroll
*	Return	:	TRUE if done. FALSE otherwise
*	Synopsis:	ScrollLeft takes the visible area of the display to the
*				left side of the display.
***************************************************************************/
int		ScrollRight(BYTE count)
{
	DWORD	old_bkgnd;

	if (count > WinOrgX)			/* Can't scroll this much */
		count = WinOrgX;						/* just return */

	if (!count)
		return FALSE;

	if ( !ScrollLock && InFocus) {
		CurHideCnt++;
		HideCaret (hTermWnd);
	}
	WinOrgX -= count;		/* This is the new X origin */

	/* we need to save the bakground color and set it to white */
//	old_bkgnd = SetBkColor(hTermDC, ColorTable [DEF_BKGND]);
	old_bkgnd = SetBkColor(hTermDC, ColorTable [ColorSetup.TermBg]);
	/* issue a block scroll */
	ScrollWindow(hTermWnd, (count * nStdioCharWidth), 0,
							NULL, (LPRECT) &TermWindow);
	/* restore back ground after scroll */
	SetBkColor(hTermDC, old_bkgnd);
	
	if ( !ScrollLock && CursorVisible())
		ReDrawCursor();
	/* Update the Thumb position */
	SetScrollPos (hTermWnd, SB_HORZ, WinOrgX, TRUE);

	UpdateWindow(hTermWnd);
	return TRUE;
}

/***************************************************************************
*	Routine	:	ScrollUp
*	Input	:	count number of columns to Scroll
*	Return	:	TRUE if done. FALSE otherwise
*	Synopsis:	ScrollUp takes the visible area of the display to the
*				lower side of the display.
*	Note	:	Routine must be called only after verifying that there is
*				enough area of the screen to scroll. It doesn't check
*				for the argument while scrolling.
***************************************************************************/
int		ScrollUp(BYTE count)
{
	DWORD	old_bkgnd;

	if ((WinOrgY + RowsInWin) >= LinesInScreen)
		return FALSE;		/* nothing to be shown more */
	if ( ! ScrollLock && InFocus) {
		CurHideCnt++;
		HideCaret (hTermWnd);
	}
	WinOrgY += count;		/* This is the new Y origin */

	/* we need to save the bakground color and set it to white */
//	old_bkgnd = SetBkColor(hTermDC, ColorTable [DEF_BKGND]);
	old_bkgnd = SetBkColor(hTermDC, ColorTable [ColorSetup.TermBg]);
	/* issue a block scroll */
	ScrollWindow(hTermWnd, 0, -(count * nStdioCharHeight),
						NULL, (LPRECT) &TermWindow);
	/* restore back ground after scroll */
	SetBkColor(hTermDC, old_bkgnd);
	
	if ( ! ScrollLock && CursorVisible())
		ReDrawCursor();
	/* Update the Thumb position */
	SetScrollPos (hTermWnd, SB_VERT, WinOrgY, TRUE);

	UpdateWindow(hTermWnd);
	return TRUE;
}

/***************************************************************************
*	Routine	:	ScrollDown
*	Input	:	count number of columns to Scroll
*	Return	:	TRUE if done. FALSE otherwise
*	Synopsis:	ScrollDown takes the visible area of the display to the
*				upper side of the display memory.
*	Note	:	Routine must be called only after verifying that there is
*				enough area of the screen to scroll. It doesn't check
*				for the argument while scrolling.
***************************************************************************/
int		ScrollDown(BYTE count)
{
	DWORD	old_bkgnd;

	if (count > WinOrgY)			/* We can't scroll this much down */
		count = WinOrgY;			// was WinOrgY - bug fixed on 15th Oct
	if (!count)
		return FALSE;

	if ( ! ScrollLock && InFocus) {
		CurHideCnt++;
		HideCaret (hTermWnd);
	}
	WinOrgY -= count;				/* New Y origin of window */

	/* we need to save the bakground color and set it to white */
//	old_bkgnd = SetBkColor(hTermDC, ColorTable [DEF_BKGND]);
	old_bkgnd = SetBkColor(hTermDC, ColorTable [ColorSetup.TermBg]);
	/* issue a block scroll */
	ScrollWindow(hTermWnd, 0, (count * nStdioCharHeight),
						NULL, (LPRECT) &TermWindow);
	/* restore back ground after scroll */
	SetBkColor(hTermDC, old_bkgnd);
	
	if ( ! ScrollLock && CursorVisible())
		ReDrawCursor();
	/* Update the Thumb position */
	SetScrollPos (hTermWnd, SB_VERT, WinOrgY, TRUE);

	UpdateWindow(hTermWnd);
	return TRUE;
}

/***************************************************************************
*	Routine	:	MoveWinOrg
*	Input	:	BYTE new_x, BYTE new_y
*	Return	:	
*	Synopsis:	Moves the window wrt the display memory to the new origin.
***************************************************************************/
void	MoveWinOrg(BYTE new_x, BYTE new_y)
{

	if ( !ScrollLock && InFocus) {
		CurHideCnt++;
		HideCaret (hTermWnd);
	}
	WinOrgX = new_x;	/* first update the window origin with new values */
	WinOrgY = new_y;

	InvalidateRect(hTermWnd, NULL, FALSE);	/* invalidate the entire area */

	/* Update the Thumb position */
	SetScrollPos (hTermWnd, SB_HORZ, WinOrgX, TRUE);
	SetScrollPos (hTermWnd, SB_VERT, WinOrgY, TRUE);

	UpdateWindow(hTermWnd);
	if ( ! ScrollLock && CursorVisible())
		ReDrawCursor();
}

/***************************************************************************
*	Routine	:	ScrollX
*	Input	:	wParam SB_TYPE, lParam position
*	Return	:	
*	Synopsis:	This processes the Scroll messages for the window
***************************************************************************/
void	ScrollX (WORD cmnd, BYTE scrl_pos)
{
	BYTE	diff;

	switch (cmnd) {
		case 	SB_LINEUP :					/* one column left */
			ScrollRight(1);
			break;
		case 	SB_LINEDOWN :				/* one column right */
			ScrollLeft(1);
			break;

		case 	SB_PAGEUP :					/* one page left */
			if (WinOrgX >= ClmsInWin)
				MoveWinOrg ( (BYTE) (WinOrgX - ClmsInWin), WinOrgY);
			else 
				ScrollRight(WinOrgX);		/* move to left most column */
			break;

		case 	SB_PAGEDOWN :				/* one page right */
			if ((CharsInLine - (WinOrgX + ClmsInWin)) >= ClmsInWin)
				MoveWinOrg ((BYTE) (WinOrgX + ClmsInWin), WinOrgY);
			else
				ScrollLeft (ClmsInWin);
			break;

		case 	SB_THUMBPOSITION :			/* Absolute positioning */
			if (scrl_pos == WinOrgX)
				return;
			if (scrl_pos > WinOrgX) {	/* need to go right */
				diff = scrl_pos - WinOrgX;
				if (diff > (BYTE) (ClmsInWin >> 2))
					MoveWinOrg (scrl_pos, WinOrgY);
				else
					ScrollLeft (diff);
			} else {					/* need to go left */
				diff = WinOrgX - scrl_pos;
				if (diff > (BYTE) (ClmsInWin >> 2))
					MoveWinOrg (scrl_pos, WinOrgY);
				else
					ScrollRight (diff);
			}
			break;
		default :
			return;
	}
}

/***************************************************************************
*	Routine	:	ScrollY (WORD cmnd)
*	Input	:	
*	Return	:	
*	Synopsis:	
***************************************************************************/
void	ScrollY (WORD cmnd, BYTE scrl_pos)
{
	BYTE	diff;

	switch (cmnd) {
		case 	SB_LINEUP :					/* One more line on top */
			ScrollDown(1);
			break;
		case 	SB_LINEDOWN :				/* one more line on bottom */
			ScrollUp(1);
			break;
		case 	SB_PAGEUP :					/* one page to top */
			if (WinOrgY > RowsInWin)
				MoveWinOrg (WinOrgX, (BYTE) (WinOrgY - RowsInWin));
			else
				ScrollDown (WinOrgY);
			break;
		case 	SB_PAGEDOWN :				/* one page to bottom */
			if ( WinOrgY + (RowsInWin << 1) <= (int) LinesInScreen)
				MoveWinOrg (WinOrgX, (BYTE) (WinOrgY + RowsInWin));
			else 
				ScrollUp ( (BYTE) (LinesInScreen - (WinOrgY + RowsInWin)));
			break;

		case 	SB_THUMBPOSITION :			/* Absolute positioning */
			if (scrl_pos == WinOrgY)
				return;
			if (scrl_pos > WinOrgY) {	/* need to go down */
				diff = scrl_pos - WinOrgY;
				if (diff > (BYTE) (RowsInWin >> 2))
					MoveWinOrg (WinOrgX, scrl_pos);
				else
					ScrollUp (diff);
			} else {					/* need to go up */
				diff = WinOrgY - scrl_pos;
				if (diff > (BYTE) (RowsInWin >> 2))
					MoveWinOrg (WinOrgX, scrl_pos);
				else
					ScrollDown (diff);
			}
			break;
		default :
			return;
	}
}

extern	LOGFONT		TerminalLogFont;
/***************************************************************************
*	Routine	:	SetupStdioDC(void)
*	Input	:	
*	Return	:	
*	Synopsis:	Sets up the "hTermDC" for the window
***************************************************************************/
void SetupStdioDC(void)
{
	HFONT	hFont;
    TEXTMETRIC Metrics;

	if (hTermDC) {				/* This is a request for change */
		ReleaseDC(hTermWnd, hTermDC);
		hTermDC = NULL;
	}

	hTermDC = GetDC(hTermWnd);				/* init the Global TERMINAL DC */

    /* set origin to top of the window */
    SetViewportOrg(hTermDC, 0, 0);

    SetMapMode(hTermDC, MM_TEXT);

    /* Set the background mode to opaque, and select the font. */
    SetBkMode(hTermDC, OPAQUE);
	RatifyFontVals();
	if ( TestFontFixedPitch(&TerminalLogFont) < 0) {
		lstrcpy(TerminalLogFont.lfFaceName, "System");
		TerminalLogFont.lfHeight = DEF_TERM_CHAR_HEIGHT;
		TerminalLogFont.lfWidth = DEF_TERM_CHAR_WIDTH;
	}
	hFont = CreateFontIndirect(&TerminalLogFont);
    hOldFont = SelectObject (hTermDC, hFont);
	DeleteObject(hOldFont);

    /* get the text metrics for the font we are using */
    GetTextMetrics(hTermDC, &Metrics);

    /* calculate the height and width of the font  */
    nStdioCharWidth = Metrics.tmMaxCharWidth;
    nStdioCharHeight = Metrics.tmHeight + Metrics.tmExternalLeading;

	SetupWinMatrix();	/* Called to update the window matrix */
}

/***************************************************************************
*	Routine	:	GetNextRowPtr
*	Input	:	character pointer
*	Return	:	returns pointer to the next row same column
*	Synopsis:	This routine gets the pointer to the next display row
*				in the same column.
***************************************************************************/
char *GetNextRowPtr (char *disptr)
{
	char *tmptr ;				/* local temp ptr */

	tmptr = disptr ;
	tmptr += CellsInLine ;
	if (tmptr >= (sScreenBuffLast + CellsInLine))	/* wraps around */
//		tmptr = sScreenBuff + ((WORD) disptr - (WORD) sScreenBuffLast) ;
		tmptr = sScreenBuff + (disptr - sScreenBuffLast) ;

	return tmptr ;
}

/***************************************************************************
*	Routine	:	GetStdioLine(short)
*	Input	:	index of the line in the display memory
*	Return	:	character pointer
*	Synopsis:	Gets the address of the beginning of a line in the display
*				memory.
***************************************************************************/
char *GetStdioLine (short ndx)
{
	char *tmpptr ;

	tmpptr = sDispOrig + ndx * CellsInLine ;
	if(tmpptr > sScreenBuffLast)
	{
		/* buffer has rolled over */
//		tmpptr = sScreenBuff + ((WORD)tmpptr - (WORD)sScreenBuffLast);
		tmpptr = sScreenBuff + (tmpptr - sScreenBuffLast) ;
		/* buffer index starts from 0 (not 1) */
		tmpptr -= CellsInLine ;
	}
	return tmpptr ;
}

/***************************************************************************
*	Routine	:	
*	Input	:	
*	Return	:	
*	Synopsis:	
***************************************************************************/
void IconicPaint(HWND hWnd)
{
	/* */
}

/***************************************************************************
*	Routine	:	CheckAttribContinuity
*	Input	:	
*	Return	:	
*	Synopsis:	Checks for the continuity of the attributes.
***************************************************************************/
BYTE	CheckAttribContinuity( BYTE *ptr, BYTE max_count)
{
	BYTE	cnt;

	for(cnt = 1; (cnt < max_count) && (*ptr == *(ptr + 1)); ptr++, cnt++);
	return cnt;
}

/***************************************************************************
*	Routine	:	StdioPaint(HWND)
*	Input	:	
*	Return	:	
*	Synopsis:	The I/O window paint procedure.  Draws necessary text in 
*				the window.	
***************************************************************************/
void StdioPaint(HWND hWnd )   
{
    char *psLine, *psAttrib;
    int	i;
    PAINTSTRUCT ps;
    RECT rcOrg, rcUpdate;
    int nVPaintBeg, nVPaintEnd, nHPaintBeg, nHPaintEnd;
	WORD	xoffset;
	WORD	paintlen;
	BYTE	writelen, incpaint;
	int		resid;
	DWORD	old_bkgnd, old_fgnd;

    BeginPaint( hWnd, (LPPAINTSTRUCT)&ps );

    rcUpdate = ps.rcPaint;
	rcOrg = ps.rcPaint;

	/* Convert into character units */
	rcUpdate.top = rcUpdate.top / nStdioCharHeight;
	rcUpdate.bottom = (rcUpdate.bottom + nStdioCharHeight) / nStdioCharHeight;
	rcUpdate.left = rcUpdate.left / nStdioCharWidth;
	rcUpdate.right = (rcUpdate.right + nStdioCharWidth) / nStdioCharWidth;

    nVPaintBeg = rcUpdate.top;
    nVPaintEnd = rcUpdate.bottom;
    nHPaintBeg = rcUpdate.left;
    nHPaintEnd = rcUpdate.right;

	psLine = GetStdioLine(nVPaintBeg + WinOrgY) + nHPaintBeg + WinOrgX;
	xoffset = nHPaintBeg * nStdioCharWidth;
	psAttrib = psLine + ScrnBufSize;	/* this points to attribute */

	/* Calculate the length of terminal data to repaint */
	if ((WinOrgX + nHPaintBeg)  >= (int)CharsInLine) {
		/* no information to be diplayed */
		paintlen = 0;		/* We need to tackle background here */
	} else if ((WinOrgX + ClmsInWin) <= CharsInLine) {
		/* Entire invalid area has to be painted */
		paintlen = (WORD) (nHPaintEnd - nHPaintBeg + 1);
		paintlen = (WORD) min (paintlen, (WORD)(ClmsInWin - nHPaintBeg));
	} else {
		/* The invalidated area crosses the last column in the terminal */
		paintlen = CharsInLine - (WinOrgX + nHPaintBeg);
	}
	/* calculate the rest invalid are length */
	resid = nHPaintEnd - nHPaintBeg + 1 - paintlen;

    /* display the lines that need to be drawn */
    for(i = nVPaintBeg; i <= nVPaintEnd; i++) {
		if (i >= (int) RowsInWin) {
			old_bkgnd = SetBkColor (hTermDC,
//						ColorTable[ BACK_GROUND(CurAttrib)]);
						ColorTable[ ColorSetup.TermBg]);
			TextOut (hTermDC, nHPaintBeg * nStdioCharWidth,
				i * nStdioCharHeight, BlankLine, nHPaintEnd - nHPaintBeg);
			SetBkColor (hTermDC, old_bkgnd);
			continue;
		}
#if 0
		if ((i + WinOrgY) >= (int) LinesInScreen) {
			old_bkgnd = SetBkColor (hTermDC,
						ColorTable[ BACK_GROUND(CurAttrib)]);
			TextOut (hTermDC, nHPaintBeg * nStdioCharWidth,
				i * nStdioCharHeight, BlankLine, nHPaintEnd - nHPaintBeg);
			SetBkColor (hTermDC, old_bkgnd);
			continue;			
		}
#endif
		/* now check the background color and write text */
		for (incpaint = (BYTE) 0; incpaint < (BYTE)paintlen;) {
			/* how many chars have the same attributes */
			writelen = CheckAttribContinuity((BYTE *)(psAttrib + incpaint),
										(BYTE)((BYTE)paintlen - incpaint));
			/* set the atrribute to the new one */
			old_bkgnd = SetBkColor (hTermDC,
						ColorTable[ BACK_GROUND(psAttrib[incpaint]) ]);
			old_fgnd = SetTextColor (hTermDC,
						ColorTable[ FORE_GROUND(psAttrib[incpaint]) ]);
			/* write so much of text */
		    TextOut(hTermDC, xoffset + incpaint * nStdioCharWidth, i * nStdioCharHeight,
								psLine + incpaint, writelen);
			/* increment the counter */
			incpaint += writelen;
			/* set the colors back */
			SetBkColor (hTermDC, old_bkgnd);
			SetTextColor (hTermDC, old_fgnd);
		}

		if (resid > 0) {
			old_bkgnd = SetBkColor (hTermDC,
//						ColorTable[ BACK_GROUND(CurAttrib)]);
						ColorTable[ ColorSetup.TermBg]);
			TextOut (hTermDC, xoffset + paintlen * nStdioCharWidth,
				i * nStdioCharHeight, BlankLine, resid);
			SetBkColor (hTermDC, old_bkgnd);
		}
		psLine = GetNextRowPtr(psLine);		/* pointer to screen data */
		psAttrib = psLine + ScrnBufSize;	/* pointer to attrib */
    }
	DrawCursorIfInView();
	if (ScrollLock)
		UpdateScrollThumbs();

    EndPaint( hWnd, (LPPAINTSTRUCT)&ps );
}

/***************************************************************************
*	Routine	:	InitTermParams
*	Input	:	The new screen height and width
*	Return	:	TRUE if success else FALSE
*	Synopsis:	Updates the global variables related to screen geometry.
*				If no screen memory was already allocated, sufficient
*				memory is allocated and inited.
*				If the Screen parameters are changed and screen memory,
*				already allocated, the screen memory is released,
*				reallocated and initialised.
*				Display origin and cursor position are reset
*	NOTE	:	Calling this with the currnet params will reinit the SCREEN
***************************************************************************/
int	InitTermParams( BYTE nlines, BYTE nchars)
{
	BYTE	flag = FALSE;
	WORD	i, memsize;
	int		oldsize, newsize;

	newsize = (nchars + 1) * nlines * 2;
	oldsize = (CharsInLine + 1) * LinesInScreen * 2;
	if (newsize > oldsize)		/* make sure we can get enough mem */
		if (LocalCompact(0) < (WORD) (2 * ( nchars + 1) * nlines))
			return FALSE;	/* return FALSE if we don't have mem */

	/* Check if the Geometrical parameters have changed */
	if ( (LinesInScreen != nlines) || (CharsInLine != nchars) )  {
		flag = TRUE;
		LinesInScreen = nlines;		/* update the number of lines */
		CharsInLine = nchars;		/*	update the number of chars in line */
	}
	CellsInLine = (BYTE) (nchars + 1);	/* Space for a NULL also */
	MaxClmIndx = (BYTE) (nchars - 1);	/* Maximum char index in a line (max nCurX val) */
	MaxRowIndx = (BYTE) (nlines - 1);	/* Maximum Line Index in Screen (max nCurY val) */
	memsize = CellsInLine * LinesInScreen;

	if (flag) {							/* needs reallocate display buffer */
        if (sScreenBuff) {              /* needs to release memory */
            MemFree( (LPSTR) sScreenBuff);
            sScreenBuff = NULL;                 /* prepare for realloc */
		}
	}
	if (!sScreenBuff) {
		/* *sScreenBuff - Circular Array of characters on TTY */
		/* this remains fixed after the memory allocation */
		/* Allocate memory for the terminal chars and attributes */
		if ((sScreenBuff = (char *)	MemAlloc(2 * memsize)) == 0) {
			/* no memory available */
			return(FALSE);
		}
	}
	sAttribBuff = sScreenBuff + memsize;		/* start address of attrib	*/
	ScrnBufSize = memsize;

	NormAttrib = (BYTE) (ColorSetup.TermBg << 4 | ColorSetup.TermFg);

	/* Set the color of the DC */
	SetDispNormAtt();

	for (i = 0; i < memsize; i++) {
	    sScreenBuff[i] = ' ';			/* Set all the chars to blank */
		sAttribBuff[i] = CurAttrib;			/* Set the attribute also */
	}
	/*	*sScreenBuffLast - doesn't change - for computional convenience */
	/* points to the beginning of the last line in the buffer */
	/* this need not be the last line on display */
	sScreenBuffLast = sScreenBuff + (CellsInLine * (LinesInScreen - 1));

	ScrollTopMargin = 0;		/* needed for scroll stuff */
	ScrollBottomMargin = LinesInScreen - 1;

	sDispOrig = sScreenBuff;	/* Init Display origin */
								/* this changes as the screen scrolls */
	nCurX = nCurY = 0;			/* Init cursor position wrt DispOrig */
	ResetScrollBuffer();		/* This also is needed */
	return(TRUE);
}

/***************************************************************************
*	Routine	:	InitStdio(HWND)
*	Input	:	
*	Return	:	
*	Synopsis:	Inits some vars used. May be removed.
***************************************************************************/
void InitStdio(HWND hWnd)
{
    HDC hDC;
    TEXTMETRIC Metrics;

    /* get the text metrics for the font we are using */
    hDC = GetDC(hWnd);
    GetTextMetrics(hDC,&Metrics);
    ReleaseDC(hWnd,hDC);

    /* calculate the height and width of the font  */
    nStdioCharWidth = Metrics.tmMaxCharWidth;
    nStdioCharHeight = Metrics.tmHeight + Metrics.tmExternalLeading;

    /* get the background and forground colors we are going to use */
    StdiobkRGB = GetSysColor(COLOR_WINDOW); /* background color */
    StdiotextRGB = GetSysColor(COLOR_WINDOWTEXT); /* text color */

    bInited = TRUE;
}

/***************************************************************************
*	Routine	:	stdioInit(HANDLE)
*	Input	:	Instance handle
*	Return	:	TRUE or FALSE
*	Synopsis:	Registers the window class.
***************************************************************************/
BOOL stdioInit(HANDLE hInstance)
{
    WNDCLASS pStdioClass;

    if(hInstance == NULL) return FALSE;

    /* create the stdio window */

    pStdioClass.hCursor        = NULL;
    pStdioClass.lpszClassName  = (LPSTR)"Stdio";
    pStdioClass.hbrBackground  = NULL;	// GetStockObject (BLACK_BRUSH);
    pStdioClass.hInstance      = hInstance;
    pStdioClass.style		    = CS_OWNDC | CS_HREDRAW | CS_VREDRAW ;
    pStdioClass.lpfnWndProc    = StdioWndProc;
	pStdioClass.hIcon          = NULL;
    pStdioClass.lpszMenuName   = (LPSTR) NULL;

    if (!RegisterClass( (LPWNDCLASS)&pStdioClass ))
        /* Initialization failed. */
        /* Windows will automatically deallocate all allocated memory. */
        return FALSE;


    return TRUE;
}
extern	WORD
	FAR PASCAL CursorTimer (HWND hWnd, WORD wMsg, int nIDEvent, DWORD dwTime);
extern	FARPROC	CursorTimerFn;

/***************************************************************************
*	Routine	:	wopen(HWND, BOOL)
*	Input	:	
*	Return	:	Handle to the window created
*	Synopsis:	Create a default style stdio window. If bQuit is TRUE, 
				PostQuitMessage will be called when the window is closed. 
				Therefore, the stdio window can be used for the main 
				application window.
***************************************************************************/
HWND wopen(HWND hWndParent, BOOL bQuit)
{
    /* if window already created, return handle */
    if(hTermWnd != NULL) return hTermWnd;

    hTermWnd = CreateWindow((LPSTR)"Stdio",
                	(LPSTR)NULL,
					WS_CHILD | WS_VISIBLE
					/*| WS_THICKFRAME */| WS_CLIPSIBLINGS
					| WS_HSCROLL | WS_VSCROLL,
					TermWndRect.left + TERM_BORDER,
					TermWndRect.top + TERM_BORDER,
					(TermWndRect.right - TermWndRect.left - (TERM_BORDER << 1)),
					(TermWndRect.bottom - TermWndRect.top - (TERM_BORDER << 1)),
                    (HWND)hWndParent,
					IDC_TERM_WND,	/* child control ID */
                    (HANDLE)hInst, 
                    (LPSTR)NULL        
             	);
    if(hTermWnd == NULL) return FALSE;
    ShowWindow(hTermWnd, SW_SHOW);

	/* Init the window origins wrt the display buffer */
	WinOrgX = WinOrgY = 0;

	/* Get the Term DC, Get the cont and wind parameters */
	SetupStdioDC(); // +311 windisp.c

	CursorTimerFn = MakeProcInstance (CursorTimer, hInst);
	/* Create and display the cursor */
	CreateTermCursor(0);

    bStdioQuit = bQuit;
	Opened = TRUE;
    return hTermWnd;
}

/***************************************************************************
*	Routine	:	wclose
*	Input	:	None
*	Return	:	None
*	Synopsis:	Destroys the cursor, resets the DC. called from StdioWndProc
***************************************************************************/
void	wclose(void)
{
	Opened = FALSE;
	/* destroy the font created by us */
    hOldFont = SelectObject (hTermDC, GetStockObject(SYSTEM_FIXED_FONT));
	DeleteObject(hOldFont);
			
	FreeProcInstance (CursorTimerFn);
	MemFree ((LPSTR) sScreenBuff);		/* Release the disp memory */
	DestroyTermCursor();							/* release the caret */
	ReleaseDC(hTermWnd, hTermDC);
	hTermDC = NULL;							/* Reset this to NULL	*/
	PurgeTermInfo();
	/* destroy stdio data */
	hTermWnd = NULL;
	MemFree ((LPSTR) TrmList);	/* Free the mem for terminal list */
}

extern	BYTE *TmpStr;
extern	BYTE	RepeaterId;

BYTE		CheckLnStatTimer;
#define		LN_STAT_TICKS		( 1000 / TERM_POLL_TIMER )

int FAR PASCAL TermShowSignals (BYTE linestat)
{

	if (LineStatus != linestat) {		/* update LineStatus */
		LineStatus = linestat;
		if (hSigWnd)
			InvalidateRect (hSigWnd, NULL, FALSE);
	}
	return 1;
}

/***************************************************************************
*	Routine	:	StdioWndProc(HWND, WORD, WORD, LONG)
*	Input	:	
*	Return	:	
*	Synopsis:	Process the messages for the Terminal window. Exported in
*				the .DEF file
***************************************************************************/
long FAR PASCAL StdioWndProc( hWnd, message, wParam, lParam )
HWND hWnd;
unsigned message;
WORD wParam;
LONG lParam;
{
	BYTE	linestat;
	MINMAXINFO	FAR *lpmmi;
	BOOL	dcd_present;

    switch (message) {
		case WM_CREATE:
			/* initialize stdio variables */
			ConTime.hour = 0; ConTime.minute = 0; ConTime.second = 0;
			InitStdio(hWnd);
 			break;

		case WM_MOUSEMOVE:
			if (ScrollLock  && TextSelOn) {	/* in scroll mode text sel on */
				if ( ! RepeaterId)			// otherwise repeater wiil do
					DisplaySelection (lParam);
			} else  
				SetCursor (hIBeam);
			break;

		case WM_LBUTTONDOWN :	/* in scroll mode this starts a selection	*/
			if (ScrollLock)
				StartSelection(lParam);
			else
				return DefWindowProc( hWnd, message, wParam, lParam );
			break;

		case WM_LBUTTONUP :		/* in scroll mode this ends a selection		*/
			if (ScrollLock)
				EndSelection();
			else 
				return DefWindowProc( hWnd, message, wParam, lParam );
			break;


		case WM_SYSCOLORCHANGE:
			/* if the colors have been changed in the control panel, */
			/* we need to change also. */
			StdiobkRGB = GetSysColor(COLOR_WINDOW); /* background color */
			StdiotextRGB = GetSysColor(COLOR_WINDOWTEXT); /* text color */
			return DefWindowProc( hWnd, message, wParam, lParam );
			break;

		case WM_GETMINMAXINFO:
			if(!bInited) InitStdio(hWnd);

			/* restrict to NO SIZING */							
			lpmmi = (MINMAXINFO FAR *)lParam;
			lpmmi->ptMinTrackSize.x = TermWndRect.right - TermWndRect.left - (TERM_BORDER << 1);
			lpmmi->ptMaxTrackSize.x = TermWndRect.right - TermWndRect.left - (TERM_BORDER << 1);
			lpmmi->ptMinTrackSize.y = TermWndRect.bottom - TermWndRect.top - (TERM_BORDER << 1);
			lpmmi->ptMaxTrackSize.y = TermWndRect.bottom - TermWndRect.top - (TERM_BORDER << 1);
			break;

		case WM_PAINT:
			/* repaint the Stdio window */
			/* even ScrollLock should be able to use the same paint routine */
			StdioPaint(hWnd);
			break;

		case WM_SIZE :
			/* a sizing has occured */
			SetupWinMatrix();		/* recalculate the rows and columns */
			break;

		case WM_CHAR :
			if (ScrollLock)
				break;
			/* if key expanded don't send it again (WM_KEYDOWN has done it) */
//			if (IsExpandedKey(wParam, lParam))
//				break;
			WritePort ((BYTE) wParam); // +278 term.c
			break;

		case WM_TIMER :
			if (ScrollLock || InMenu)
				break;
			ReadAndDisp(); // +226 term.c
			break;

		case WM_KILLFOCUS :		/* we need to free the caret */
			if (ScrollLock) {
				if (PatternBright)			/* put matched pattern to normal */
					DisplayPattern (FALSE);
				break;
			}
			if(Opened) {
				InFocus = FALSE;
				LoosingFocus();
			}
			break;

		case WM_SETFOCUS :		/* we need to free the caret */
			if (ScrollLock)
				break;
			if (Opened) {
				InFocus = TRUE;
				SettingFocus();
			}
			break;

		case WM_HSCROLL :					/* Scroll bar request from user */
			if (ScrollLock && PatternBright)			/* put matched pattern to normal */
					DisplayPattern (FALSE);
			ScrollX (wParam, (BYTE) lParam);
			if (ScrollLock)
				ResetSearchPtr();
			break;

		case WM_VSCROLL :					/* Scroll bar request from user */
			if (ScrollLock) {
				if (PatternBright)			/* put matched pattern to normal */
					DisplayPattern (FALSE);
				ScrollInY (wParam, (WORD) lParam);
				ResetSearchPtr();
			} else {
				ScrollY (wParam, (BYTE) lParam);
			}
			break;

		case WM_KEYDOWN :

		if (ScrollLock) {
			if (PatternBright)			/* put matched pattern to normal */
				DisplayPattern (FALSE);
			ScrollTimeKeyHandler(hWnd, (BYTE) wParam);
			return TRUE;
		}

		return DefWindowProc( hWnd, message, wParam, lParam );

		case WM_DESTROY:
			/* if specified when created, PostQuitMessage should be called */
			/* when the window is destroyed. */
//			if(bStdioQuit)
//				PostQuitMessage(0);
//			break;
		case WM_CLOSE:				
/* if doing scroll ExitScroll */
//	ExitScrollMode();
			/* go ahead and close down */
			wclose();

			/* -- fall through to default -- */
		default:
			return DefWindowProc( hWnd, message, wParam, lParam );
    }
    return(0L);
}

/*---------------------------   Last  Line   ----------------------------- */

