/* LZVX.C -- LZW De-compression with variable number of bits.
** From code by Hemanth M.C. (Windows 95 Remote Control Project)
*/

#include "types.h"
#include "lzv.h"
#include "lzvx.h"

/* Globals */
static unsigned next_code;
static unsigned next_bump_code;
static unsigned current_code_bits;
static unsigned char decode_stack[TABLE_SIZE];
static struct dictionary *dict_array[TABLE_BANKS];

/* Local Prototypes */
unsigned InputBits(struct BitBufIndex *buffer, unsigned bit_length);
int InitializeStorage(void);
void ReleaseStorage(void);
void InitializeDictionary(void);
unsigned decode_string(unsigned count, unsigned code);
WORD LZ15VExpandBuf(BYTE *lpCompData, WORD wCompLen, BYTE *lpOutputData, WORD wOutputLen);


unsigned InputBits(struct BitBufIndex *buffer, unsigned bit_length)
{
	unsigned bits, byte_length;

	bits = 0;
	while(buffer->bit_offset + bit_length > 8) {
		byte_length = 8 - buffer->bit_offset;
		bits <<= byte_length;
		bits |= *buffer->index & ( (1 << byte_length) - 1 );
		bit_length -= byte_length;
		buffer->bit_offset = 0;
		buffer->index++;
		buffer->length--;
	}
	bits <<= bit_length;
	bits |= (*buffer->index >> (8 - bit_length));
	buffer->bit_offset += bit_length;
	return bits;
}


int InitializeStorage()
{
	unsigned i;
	
	for (i = 0; i < TABLE_BANKS; i++) {
		dict_array[i] = (struct dictionary *)calloc(256, sizeof(struct dictionary));
		if (dict_array[i] == (struct dictionary *) 0)
			return 0;
	}
	return 1;
}


void ReleaseStorage()
{
	unsigned i;
	
	for (i = 0; i < TABLE_BANKS; i++)
		free(dict_array[i]);
}


void InitializeDictionary()
{
	unsigned i;

	for (i = 0; i < TABLE_SIZE; i++)
		DICT(i).code_value = UNUSED;
	next_code = FIRST_CODE;

	current_code_bits = 9;
	next_bump_code = 511;
}

unsigned decode_string(unsigned count, unsigned code)
{
	while (code > 255) {
		decode_stack[count++] = DICT(code).character;
		code = DICT(code).parent_code;
	}
	decode_stack[count++] = (unsigned char)code;
	return count;
}

WORD LZ15VExpandBuf(BYTE *lpCompData, WORD wCompLen, BYTE *lpOutputData, WORD wOutputLen)
{
	const char fn_name[] = "LZ15VExpandBuf";
	unsigned new_code;
	unsigned old_code;
	unsigned character;
	unsigned count;
	struct BitBufIndex input_buf;
	unsigned char *output_ix;
	unsigned output_count;

	output_ix = lpOutputData;
	output_count = wOutputLen;
	input_buf.index = (unsigned char *)lpCompData;
	input_buf.length= wCompLen;
	input_buf.bit_offset = 0;
	
	if (InitializeStorage() == 0)
		return 0;

	for (;;) {
		InitializeDictionary();
		old_code = InputBits(&input_buf, current_code_bits);
		if (old_code == END_OF_STREAM)
			goto ReturnOutputLength;
		character = old_code;
		*output_ix++ = (unsigned char)old_code;
		output_count--;
		for (;;) {
			new_code = InputBits(&input_buf, current_code_bits);
			if (new_code == END_OF_STREAM)
				goto ReturnOutputLength;
			if (new_code == FLUSH_CODE)
				break;
			if (new_code == BUMP_CODE) {
				current_code_bits++;
				continue;
			}
			if (new_code >= next_code) {
				decode_stack[0] = (unsigned char)character;
				count = decode_string(1, old_code);
			} else
				count = decode_string(0, new_code);
			character = decode_stack[count - 1];
			while (count > 0 && output_count > 0) {
				*output_ix++ = decode_stack[--count];
				output_count--;
			}
			if (count > 0) {
				goto ReturnError;
			}
			DICT(next_code).parent_code = old_code;
			DICT(next_code).character   = (unsigned char)character;
			next_code++;
			old_code = new_code;
		}
	}

ReturnError:
	ReleaseStorage();
	return 0;

ReturnOutputLength:
	ReleaseStorage();
	return wOutputLen - output_count;
}

