/* ------------------------------------------------------------------------
	File		:	DLGBOX.C
	Synopsis	:	Modifies the dialog control colors and draws borders
					for Listbox, Edit controls and group box.
	Author		:	Srinivasan S / Prasad KVS
	Date		:	20 Apr 1993
 ----------------------------------------------------------------------- */

#include <windows.h>
#include <memory.h>

#include "dlgbox.h"

/* ********************************************************************** */
/* *** Internal Structures to support Ownerdraw Radio and Check boxes *** */
/* ********************************************************************** */

typedef	struct {
	WORD	ID;								// Control id.
	BYTE	State;							// Button state.
} CONTROLINFO;

typedef	CONTROLINFO	*PCTLINFO;

typedef	struct tagDIALOGINFO {
	HWND			hDlg;					// Dialog handle.
	int				NumCtls;				// Number of owner draw controls.
	struct 			tagDIALOGINFO *next;	// Next dialog info.
	CONTROLINFO		CtlArray[1];			// Space for 1 control.
} DIALOGINFO;

typedef	DIALOGINFO	*PDLGINFO;

/* ************************* Internal Globals *************************** */

HINSTANCE	hDLGInst;
HBRUSH		hBkGndBrush;
HPEN		hNullPen;
HBRUSH		hBorderBrush;
PDLGINFO	pHead;
HPEN		hWhitePen;	  			// white pen handle
HPEN		hGreyPen;	  			// Grey pen handle
HBRUSH		hGreyBrush;				// grey brush handle
int			ShadeHt = 4;
RECT		DlgRect;
HWND		hDlgHandle;

/* -------------------------------------------------------------------------
	Function	:	DLGInit(HINSTANCE hInst)
	Synopsis	:	Inits the Dlg module.
	Input		:	Instance handle.
	Output		:	TRUE on success, FALSE otherwise.
 ------------------------------------------------------------------------ */
BOOL FAR PASCAL
DLGInit(HINSTANCE hInst, int BackColor, int BorderColor, int ShadowHt)
{
	COLORREF	BkColor, BordColor;

	/* Init Globals */
	hDLGInst = hInst;
	pHead = NULL;

	switch (BackColor) {
		case DLG_BLACK:
			BkColor = RGB(0x00, 0x00, 0x00);
			break;

		case DLG_WHITE:
			BkColor = RGB(0xFF, 0xFF, 0xFF);
			break;

		case DLG_LTGRAY:
			BkColor = RGB(0xC0, 0xC0, 0xC0);
			break;

		case DLG_DKGRAY:
			BkColor = RGB(0x80, 0x80, 0x80);
			break;

		default:
			return (FALSE);
	}

	switch (BorderColor) {
		case DLG_BLACK:
			BordColor = RGB(0x00, 0x00, 0x00);
			break;

		case DLG_WHITE:
			BordColor = RGB(0xFF, 0xFF, 0xFF);
			break;

		case DLG_LTGRAY:
			BordColor = RGB(0xC0, 0xC0, 0xC0);
			break;

		case DLG_DKGRAY:
			BordColor = RGB(0x80, 0x80, 0x80);
			break;

		default:
			return (FALSE);
	}

	/* Create the brush and pen of required color and width */
	hBorderBrush = CreateSolidBrush(BordColor);
	hBkGndBrush = CreateSolidBrush(BkColor);
	hWhitePen			= CreatePen (PS_SOLID, 1, RGB(0xFF, 0xFF, 0xFF));
	hGreyPen			= CreatePen (PS_SOLID, 1, RGB(0x80, 0x80, 0x80));
	hGreyBrush 			= CreateSolidBrush(RGB(0x80, 0x80, 0x80));
	ShadeHt = ShadowHt;
	hNullPen			= CreatePen (PS_NULL,  1, RGB(   0,    0,    0));
	return (TRUE);
}

/* -------------------------------------------------------------------------
	Function	:	DLGClose(void)
	Synopsis	:	Closes the Dlg module by freeing up local resources.
	Input		:	None.
	Output		:	None.
 ------------------------------------------------------------------------ */
void FAR PASCAL
DLGClose(void)
{
	DeleteObject(hNullPen);
	DeleteObject(hBorderBrush);
	DeleteObject(hBkGndBrush);
	DeleteObject(hWhitePen);
	DeleteObject(hGreyPen);
	DeleteObject(hGreyBrush);
}

/* -------------------------------------------------------------------------
	Function	:	DLGShowFileOpenDlgBox(OPENFILENAME Ofn)
	Synopsis	:	Displays the File open common dialog box after installing
					a hook procedure to modify the colors.
	Input		:	Pointer to OPENFILENAME structure.
	Output		:	TRUE on success, FALSE otherwise.  If the function is
					successful then the OPENFILENAME structure has the
					file chosen.
 ------------------------------------------------------------------------ */
BOOL FAR PASCAL
DLGShowFileOpenDlgBox(LPOPENFILENAME lpOfn)
{
	BOOL	RetVal;
	FARPROC	CtlColorProc;

	/* Install a hook procedure */
	lpOfn->Flags |= OFN_ENABLEHOOK;
	CtlColorProc = MakeProcInstance((FARPROC) DLGCtlColorFunc, hDLGInst);
	lpOfn->lpfnHook = CtlColorProc;

	/* Invoke the dialog box */
	RetVal = GetOpenFileName(lpOfn);

	/* free the hook function */
	FreeProcInstance(CtlColorProc);

	return (RetVal);
}

/* -------------------------------------------------------------------------
	Function	:	DLGShowSaveAsDlgBox(LPOPENFILENAME lpOfn)
	Synopsis	:	Displays the Save as common dialog box after installing
					a hook procedure to modify the colors.
	Input		:	Pointer to OPENFILENAME structure.
	Output		:	TRUE on success, FALSE otherwise.  If the function is
					successful then the OPENFILENAME structure has the
					file chosen.
 ------------------------------------------------------------------------ */
BOOL FAR PASCAL
DLGShowSaveAsDlgBox(LPOPENFILENAME lpOfn)
{
	BOOL	RetVal;
	FARPROC	CtlColorProc;

	/* Install a hook procedure */
	lpOfn->Flags |= OFN_ENABLEHOOK;
	CtlColorProc = MakeProcInstance((FARPROC) DLGCtlColorFunc, hDLGInst);
	lpOfn->lpfnHook = CtlColorProc;

	/* Invoke the dialog box */
	RetVal = GetSaveFileName(lpOfn);

	/* free the hook function */
	FreeProcInstance(CtlColorProc);

	return (RetVal);
}

/* -------------------------------------------------------------------------
	Function	:	DLGShowPrintDlgBox(LPPRINTDLG lpPd)
	Synopsis	:	Displays the Print common dialog box.
	Input		:	Pointer to PRINTDLG structure.
	Output		:	TRUE on success, FALSE otherwise.  On success the
					chosen printer details are available in the
					PRINTDLG structure.
 ------------------------------------------------------------------------ */
BOOL FAR PASCAL
DLGShowPrintDlgBox(LPPRINTDLG lpPd)
{
	BOOL	RetVal;
	FARPROC	CtlColorProc;
	
	/* Install a hook procedure */
	lpPd->Flags |= (PD_ENABLEPRINTHOOK|PD_ENABLESETUPHOOK);
	CtlColorProc = MakeProcInstance((FARPROC) DLGCtlColorFunc, hDLGInst);
	lpPd->lpfnPrintHook = CtlColorProc;
	lpPd->lpfnSetupHook = CtlColorProc;

	/* Invoke the dialog box */
//	RetVal = PrintDlg(lpPd);

	/* free the hook function */
	FreeProcInstance(CtlColorProc);

	return(RetVal);
}

/* -------------------------------------------------------------------------
	Function	:	DLGShowFontDlgBox(LPCHOOSEFONT lpCf)
	Synopsis	:	Displays the Font common dialog box.
	Input		:	Pointer to CHOOSEFONT structure.
	Output		:	TRUE on success, FALSE otherwise.  On success the
					chosen printer details are available in the
					LOGFONT item in the CHOOSEFONT structure.
 ------------------------------------------------------------------------ */
BOOL FAR PASCAL
DLGShowFontDlgBox(LPCHOOSEFONT lpCf)
{
	BOOL	RetVal;
	FARPROC	CtlColorProc;
	
	/* Install a hook procedure */
	lpCf->Flags |= CF_ENABLEHOOK;
	CtlColorProc = MakeProcInstance((FARPROC) DLGCtlColorFunc, hDLGInst);
	lpCf->lpfnHook = CtlColorProc;

	/* Invoke the dialog box */
//	RetVal = ChooseFont(lpCf);

	/* free the hook function */
	FreeProcInstance(CtlColorProc);

	return(RetVal);
}

void	DrawShadow(HDC hDC, LPRECT Rect)
{
	HBRUSH	hOldBrush;
	HPEN	hOldPen;

	/* Draw 2 rectangles for shade */
	/* select null pen */
	hOldPen = SelectObject(hDC, hNullPen);
	hOldBrush = SelectObject(hDC, hBorderBrush);

	/* Draw bottom side shade */
	Rectangle (hDC, Rect->left + ShadeHt, Rect->bottom,
			Rect->right + ShadeHt, Rect->bottom + ShadeHt+ShadeHt/3);

	/* Draw right side shade */
	Rectangle (hDC, Rect->right, Rect->top + ShadeHt,
			Rect->right + ShadeHt, Rect->bottom + ShadeHt);
	SelectObject(hDC, hOldPen);
	SelectObject(hDC, hOldBrush);
}

/* -------------------------------------------------------------------------
	Function	:	DLGModifyColors(WPARAM, LPARAM)
	Synopsis	:	Sets the dialog colors to user defined colors.
	Input		:	wParam, lParam
	Output		:	BRUSH used to paint the dialog items.
 ------------------------------------------------------------------------ */
HBRUSH FAR PASCAL
DLGModifyColors(WPARAM wParam, LPARAM lParam)
{
	int		CtlID, Width, Height, ItemHt;
	WORD	CtlCode;
	HDC		hDC;
	HWND	hCtlWnd;
	RECT	TmpRect;
	HPEN	hPen, hOldPen;
	HBRUSH	hOldBrush;
	TEXTMETRIC	tm;

	char	WndClass[25];
	
	if (hDlgHandle && DlgRect.bottom) {
		if ((hDC = GetDC(hDlgHandle)) == NULL)
			goto Cont;

		// Draw a white line first
		hPen = CreatePen(PS_SOLID, 2, RGB(0xff, 0xff, 0xff));
		hOldPen = SelectObject(hDC, hPen);
		MoveTo(hDC, DlgRect.left, DlgRect.bottom);
		LineTo(hDC, DlgRect.left, DlgRect.top);
		LineTo(hDC, DlgRect.right, DlgRect.top);
		SelectObject(hDC, hOldPen);
		DeleteObject(hPen);

		// Now draw a Dark grey line.
		hPen = CreatePen(PS_SOLID, 2, RGB(0x80, 0x80, 0x80));
		hOldPen = SelectObject(hDC, hPen);
		LineTo(hDC, DlgRect.right, DlgRect.bottom);
		LineTo(hDC, DlgRect.left, DlgRect.bottom);
		SelectObject(hDC, hOldPen);
		DeleteObject(hPen);
		ReleaseDC (hDlgHandle, hDC);
Cont:
		memset(&DlgRect, 0, sizeof(RECT));
		hDlgHandle = NULL;
	}
	/* Get the DC */
	hDC = (HDC) wParam;
	CtlCode = HIWORD(lParam);
	hCtlWnd = (HWND) LOWORD(lParam);


	switch(CtlCode) {
		case CTLCOLOR_BTN:
			CtlID = GetDlgCtrlID(hCtlWnd);
			if (CtlID >= RECT_BASE && CtlID <= RECT_END) {
				/* Leave space for the text on top */
				GetTextMetrics(hDC, (LPTEXTMETRIC) &tm);
				/* Get the coordinates */
				GetClientRect(hCtlWnd, (LPRECT) &TmpRect);
				TmpRect.top += tm.tmHeight/2;

				hOldBrush = SelectObject(hDC, hGreyBrush);
				Rectangle(hDC, TmpRect.left, TmpRect.top,
							TmpRect.right, TmpRect.bottom);
				SelectObject(hDC, hOldBrush);

				DrawShadow(hDC, &TmpRect);
				return hGreyBrush;
			}
			if (CtlID >= GROUP_BOX_BASE && CtlID <= GROUP_BOX_END) {
				/* Leave space for the text on top */
				GetTextMetrics(hDC, (LPTEXTMETRIC) &tm);
				/* Get the coordinates */
				GetClientRect(hCtlWnd, (LPRECT) &TmpRect);
				TmpRect.top += tm.tmHeight/2;

				DrawShadow(hDC, &TmpRect);
			}
			SetBkColor((HDC) wParam, RGB(0xC0, 0xC0, 0xC0));
			return hBkGndBrush;

		case CTLCOLOR_STATIC:
			/* Mode is to be set to prevent the background from being */
			/* erased when windows draws the text */
			SetBkMode(hDC, TRANSPARENT);
			return (hBkGndBrush);

		case CTLCOLOR_DLG:
			hDlgHandle = hCtlWnd;
			GetClientRect(hCtlWnd, (LPRECT) &DlgRect);
			DlgRect.top += 1;
			DlgRect.left += 1;
			DlgRect.right -= 2;
			DlgRect.bottom -= 2;
			return (hBkGndBrush);

		case CTLCOLOR_MSGBOX:
			return (hBkGndBrush);

		case CTLCOLOR_LISTBOX:
		case CTLCOLOR_EDIT:

			GetWindowRect(hCtlWnd, (LPRECT) &TmpRect);
			Width  = TmpRect.right - TmpRect.left;
			Height  = TmpRect.bottom - TmpRect.top;
			ScreenToClient(hCtlWnd, (LPPOINT) &TmpRect);
			TmpRect.right = TmpRect.left + Width;
			TmpRect.bottom= TmpRect.top+ Height;

			GetClassName(hCtlWnd, (LPSTR) WndClass, sizeof(WndClass));
			if (!lstrcmpi((LPSTR) WndClass, (LPSTR) "listbox")) {

				/* Make sure the size received is the integral multiple */
				/* of the total number of strings that can be shown in the */
				/* list box, before drawing the border */

				ItemHt = (int) SendMessage(hCtlWnd, LB_GETITEMHEIGHT, 0, 0L);
				if ((Height - 2) % ItemHt)
					TmpRect.bottom -= ((Height - 2) % ItemHt);
			}
			DrawShadow(hDC, &TmpRect);

 		default:
			return ((HBRUSH) NULL);
 	}
}

/* -------------------------------------------------------------------------
	Function	:	DLGCtlColorFunc()
	Synopsis	:	Callback function for the hook for common dialog box.
	Input		:	Window, message, wParam, lParam.
	Output		:	Depends on the message processed.
 ------------------------------------------------------------------------ */
UINT CALLBACK
DLGCtlColorFunc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
		case WM_INITDIALOG:		/* !!! Don't change focus !!! */
			return (TRUE);

		case WM_CTLCOLOR:		/* Modify colors and return brush handle */
			return DLGModifyColors(wParam, lParam);

		default:				/* Let the CommDlgProc process */
			return (NULL);
	}
}

/* ----------------------------------------------------------------------- */
/*    This portion implements the functions to support owner drawn radio   */
/*    buttons and check boxes                                              */
/* ----------------------------------------------------------------------- */

/* -------------------------------------------------------------------------
	Function	:	DLGInitDialog(HWND hDlg, char *Buf)
	Synopsis	:	Initialises the internal LIFO struct with all the 
					control information.
	Input		:	Dialog handle and pointer to buf containing the
					number of owner draw controls in the first WORD
					followed by their ids.
	Output		:	None.
 ------------------------------------------------------------------------ */
void FAR PASCAL
DLGInitDialog(HWND hDlg, WORD FAR *Buf)
{
	int			i, NumCtls;
	PDLGINFO	pTmp;

	NumCtls = (int) *Buf++;
	pTmp = (PDLGINFO) LocalAlloc (LPTR, sizeof(DIALOGINFO) +
						(NumCtls - 1) * sizeof(CONTROLINFO));
	pTmp->hDlg = hDlg;
	pTmp->NumCtls = NumCtls;

	for (i=0; i < NumCtls; i++) {
		pTmp->CtlArray[i].ID = *Buf++;
		pTmp->CtlArray[i].State = DLG_BTN_UNCHKED;
	}

	if (pHead) {
		pTmp->next = pHead;
	}
	else {
		pTmp->next = NULL;
	}
	pHead = pTmp;
}

/* -------------------------------------------------------------------------
	Function	:	DLGEndDialog(void)
	Synopsis	:	Deletes the information stored for the last shown
					dialog from the internal LIFO structure.
	Input		:	None.
	Output		:	None.
 ------------------------------------------------------------------------ */
void FAR PASCAL
DLGEndDialog(void)
{
	PDLGINFO	pNext;

	if (pHead) {
		pNext = pHead->next;
		LocalFree ((HANDLE)pHead);
		pHead = pNext;
	}
}

/* -------------------------------------------------------------------------
	Function	:	DLGCheckDlgButton(HWND hDlg, WORD ID, UINT uChecked)
	Synopsis	:	Changes the state of the owner draw check box depending
					on uChecked, 0 => Unchecked, 1=> Checked and 2=> Grayed.
	Input		:	Dialog handle, Control id and new state.
	Output		:	None.
 ------------------------------------------------------------------------ */
void FAR PASCAL
DLGCheckDlgButton(HWND hDlg, WORD ID, UINT uChecked)
{
	WORD		i;
	PDLGINFO	pNode;

	pNode = pHead;
	while ((pNode->hDlg != hDlg) && pNode)
		pNode = pNode->next;

	if (!pNode)
		return;

	for (i=0; i < (WORD) pNode->NumCtls; i++) {
		if (pNode->CtlArray[i].ID == ID) {
			pNode->CtlArray[i].State = (BYTE)uChecked;
			InvalidateRect (GetDlgItem(pNode->hDlg, ID), NULL, NULL);
			UpdateWindow (GetDlgItem(pNode->hDlg, ID));
			return;
		}
	}
}

/* -------------------------------------------------------------------------
	Function	:	DLGIsDlgButtonChecked(HWND hDlg, WORD idCtl)
	Synopsis	:	Returns the current state of the owner draw check box.
	Input		:	Window handle and Control id.
	Output		:	0 => Unchecked, 1 => Checked, 2 => Grayed, -1 => Error.
 ------------------------------------------------------------------------ */
int FAR PASCAL
DLGIsDlgButtonChecked(HWND hDlg, WORD idCtl)
{
	int	i;
	PDLGINFO	pNode;

	pNode = pHead;
	while ((pNode->hDlg != hDlg) && pNode)
		pNode = pNode->next;

	if (!pNode)
		return (-1);

	for (i=0; i < pNode->NumCtls; i++) {
		if (pNode->CtlArray[i].ID == idCtl) {
			return (pNode->CtlArray[i].State);
		}
	}
	return (-1);
}

/* -------------------------------------------------------------------------
	Function	:	DLGCheckRadioButton(HWND hDlg, WORD StartID, 
												WORD EndID, WORD CurID)
	Synopsis	:	Checks the chosen owner draw radio button and unchecks
					all the other buttons in the group.
	Input		:	Window handle, Start ID, End ID and ID to check.
	Output		:	None.
 ------------------------------------------------------------------------ */
void FAR PASCAL
DLGCheckRadioButton(HWND hDlg, WORD StartID, WORD EndID, WORD CurID)
{
	int		i;
	WORD	j;
	PDLGINFO	pNode;

	pNode = pHead;
	while ((pNode->hDlg != hDlg) && pNode)
		pNode = pNode->next;

	if (!pNode)
		return;

	for (i=0; i < pNode->NumCtls; i++) {
		if (pNode->CtlArray[i].ID == StartID) {
			for (j=StartID; j <= EndID; j++,i++) {
				if (j == CurID) {
					if (pNode->CtlArray[i].State)
						return;
					else {
						pNode->CtlArray[i].State = DLG_BTN_CHKED;
						InvalidateRect (GetDlgItem(pNode->hDlg, CurID), NULL, NULL);
						UpdateWindow (GetDlgItem(pNode->hDlg, CurID));
					}
				}
				else {
					if (pNode->CtlArray[i].State) {
						/* Make it normal only if not disabled */
						if (pNode->CtlArray[i].State != DLG_BTN_GRAYED)
							pNode->CtlArray[i].State = DLG_BTN_UNCHKED;
						InvalidateRect (GetDlgItem(pNode->hDlg, j), NULL, NULL);
						UpdateWindow (GetDlgItem(pNode->hDlg, j));
					}
				}
			}
			return;
		}
	}
}

/* -------------------------------------------------------------------------
	Function	:	DLGEnableRadioButton(HWND, int, BOOL)
	Synopsis	:	Enables or disables a radio button.
	Input		:	Window handle, Control id, State
	Output		:	None.
 ------------------------------------------------------------------------ */
void FAR PASCAL
DLGEnableRadioButton(HWND hDlg, WORD CtlID, BOOL State)
{
	int		i;
	PDLGINFO	pNode;

	pNode = pHead;
	while ((pNode->hDlg != hDlg) && pNode)
		pNode = pNode->next;

	if (!pNode)
		return;

	for (i=0; i < pNode->NumCtls; i++) {
		if (pNode->CtlArray[i].ID == CtlID) {
			pNode->CtlArray[i].State = (BYTE) ((State) ?
								DLG_BTN_UNCHKED : DLG_BTN_GRAYED);
			InvalidateRect (GetDlgItem(pNode->hDlg, CtlID), NULL, NULL);
			UpdateWindow (GetDlgItem(pNode->hDlg, CtlID));
			return;
		}
	}
}

void	MyDrawEntireItem(HWND hDlg, WORD idCtl, LPDRAWITEMSTRUCT lpInfo)
{
	int 	ResourceID, chkState, bmptop;
	HDC 	hMemDC;
	char	temp[50];
	HBITMAP hbm, hOldbm;
	HBRUSH	hBr, hOldBr;
	HPEN	hPen, hOldPen;
	RECT	TmpRect;
	COLORREF	OldClr;

	if ((idCtl >= RADIO_BTN_BASE) && (idCtl <= RADIO_BTN_END)) {
		chkState = DLGIsDlgButtonChecked(hDlg, idCtl);
		ResourceID = IDR_RADIO_OFF + chkState;
	} else if ((idCtl >= CHECK_BOX_BASE) && (idCtl <= CHECK_BOX_END)) {
		chkState = DLGIsDlgButtonChecked(hDlg, idCtl);
		ResourceID = IDR_CHECK_OFF + chkState;
	} else
		return;

    if (! (hbm = LoadBitmap(hDLGInst, MAKEINTRESOURCE(ResourceID))))
		return;

	/* draw the whole button */
    hMemDC = CreateCompatibleDC(lpInfo->hDC);
    hOldbm = SelectObject(hMemDC, hbm);
	if (hOldbm) {
		/* Brush */
		hBr = GetStockObject(LTGRAY_BRUSH);
		hOldBr = SelectObject(lpInfo->hDC, hBr);
		/* Pen */
		hPen = GetStockObject(NULL_PEN);
		hOldPen = SelectObject(lpInfo->hDC, hPen);

		Rectangle(lpInfo->hDC, lpInfo->rcItem.left, lpInfo->rcItem.top,
					lpInfo->rcItem.right+2, lpInfo->rcItem.bottom+2);
		SelectObject(lpInfo->hDC, hOldBr);
		SelectObject(lpInfo->hDC, hOldPen);
		bmptop = (lpInfo->rcItem.bottom - lpInfo->rcItem.top)/2;
		bmptop += lpInfo->rcItem.top;
		bmptop -= 10;	// this makes sure of vertical centering of bmp

    	BitBlt(lpInfo->hDC, 
                   (lpInfo->rcItem).left,
                   bmptop,
				   20, 20,
                   hMemDC,
                   0, 0,
                   SRCCOPY);
		SelectObject(hMemDC, hOldbm);
		GetDlgItemText(hDlg, idCtl, (LPSTR) temp, 50);
		SetBkMode(lpInfo->hDC, TRANSPARENT);
		CopyRect((LPRECT) &TmpRect, (LPRECT) &(lpInfo->rcItem));
		TmpRect.left += 22;
//		TmpRect.top += 2;
		if (chkState == DLG_BTN_GRAYED)
			OldClr = SetTextColor(lpInfo->hDC, RGB(0x80, 0x80, 0x80));
		DrawText(lpInfo->hDC, (LPSTR) temp, -1, (LPRECT) &TmpRect,
					DT_SINGLELINE | DT_VCENTER | DT_LEFT | DT_NOCLIP);
		if (chkState == DLG_BTN_GRAYED)
			SetTextColor(lpInfo->hDC, OldClr);
	}
	DeleteDC(hMemDC);
	DeleteObject(hbm); 
}

/* -------------------------------------------------------------------------
	Function	:	DLGDrawItemControl(HWND, WORD, LPDRAWITEMSTRUCT)
	Synopsis	:	Draws the bitmaps and text for radio buttons and
					check boxes.
	Input		:	Window handle, Control Id and pointer to DRAWITEMSTRUCT.
	Output		:	None.
 ------------------------------------------------------------------------ */
void FAR PASCAL
DLGDrawItemControl(HWND hDlg, WORD idCtl, LPDRAWITEMSTRUCT lpInfo)
{
	RECT	frect;

	if (lpInfo->CtlType != ODT_BUTTON)
		return;

	if (lpInfo->itemAction == ODA_DRAWENTIRE) {
		MyDrawEntireItem(hDlg, idCtl, lpInfo);
		if (lpInfo->itemState & ODS_FOCUS)
			goto DrawFocus;
	}
	if (lpInfo->itemAction == ODA_FOCUS) {
DrawFocus:
			frect = lpInfo->rcItem;
			frect.left += DLG_FOCUS_MARGIN + 18;	/* avoid bit map */
			frect.top += DLG_FOCUS_MARGIN;
			frect.right -= DLG_FOCUS_MARGIN;
			frect.bottom -= DLG_FOCUS_MARGIN;
			DrawFocusRect(lpInfo->hDC, (LPRECT)&frect);
	}

}

/* -------------------------  End of File  -------------------------- */
