/*----------------------------------------------------------------------------
*	File		:	WINDISP.C
*	Purpose		:	Terminal emulation and cursor related routines
*	Package		:	MultiExpress (Windows) - Version 1.00
*	Authors		:	Vidy
*	Date		:	21st December 1991.
*---------------------------------------------------------------------------*/

#include	"windows.h"

#include	"main.h"
#include	"session.h"
#include	"wstdio.h"
#include	"windisp.h"
#include	"emlncons.h"
#include	"emlntype.h"
#include	"emlnfuns.h"
#include	"emlndata.h"

#include	"scroll.h"

WORD	CurHideCnt = 0;
BYTE	CurOff = FALSE;

void	DisplayCursor (void);
void	RemoveCursor (void);
BYTE	ToAdvanceCursor = FALSE;


/***************************************************************************
*	Routine	:	ClearLine
*	Input	:	pointer to the start of a line
*	Return	:	None
*	Synopsis:	Clears a line by filling with blanks
***************************************************************************/
void	ClearLine(char *lineptr)
{
	WORD	indx;
	for (indx = 0; indx < CellsInLine; indx++, lineptr++) {
		*lineptr = ' ';
		*(lineptr + ScrnBufSize) = CurAttrib;
	}
}

/***************************************************************************
*	Routine	:	DoBackSpace
*	Input	:	None
*	Return	:	None
*	Synopsis:	Interpret the effect of back space char accoring to the
*				current terminal Settings
***************************************************************************/
void	DoBackSpace(void)
{
	BYTE	*line_ptr;

	if (nCurX) {
		--nCurX;			/* We are not at the beginning of the line */
	} else {				
		if (PbkEntry.TrmParams.WrapBackSpace && nCurY) {
			nCurY--;
			nCurX = MaxClmIndx;
		}
	}
	if (PbkEntry.TrmParams.BackSpaceBlank) {
		line_ptr = GetStdioLine(nCurY);
		*(line_ptr + nCurX) = ' ';
		*(line_ptr + nCurX + ScrnBufSize) = NormAttrib;

		SetBkColor(hTermDC, ColorTable[BACK_GROUND(NormAttrib)]);
		SetTextColor(hTermDC, ColorTable[FORE_GROUND(NormAttrib)]);
		CharOutWin(' ');
		SetBkColor(hTermDC, ColorTable[BACK_GROUND(CurAttrib)]);
		SetTextColor(hTermDC, ColorTable[FORE_GROUND(CurAttrib)]);

	}
}
/****************************************************************************
Synop	:	When the terminal font changes to a smaller size, characters
			Written at right side of the lines may not be erased. So do
			a text out with old font itself of blank lines
****************************************************************************/
void	PaintTermBackGnd(void)
{
	int		i;
	for (i = 0; i < (int)LinesInScreen; i++)
		TextOut(hTermDC, 0, i * nStdioCharHeight, BlankLine, CharsInLine);
}
/***************************************************************************
*	Routine	:	SoundBell
*	Input	:	None
*	Return	:	None
*	Synopsis:	Sounds the Bell as done by a terminal
***************************************************************************/
void	SoundBell(void)
{	
	MessageBeep(0);
}

/***************************************************************************
*	Routine	:	ScrlDispBuff
*	Input	:	None
*	Return	:	None
*	Synopsis:	Updates the sDispOrig to point to the next line. Also calls
*				The Window Function to Scroll the Screen One line.
***************************************************************************/
void	ScrlDispBuff(void)
{
	if (ScrollTopMargin == 0 && ScrollBottomMargin == MaxRowIndx) {
		SaveLineToScrlBuff(sDispOrig);
		ClearLine(sDispOrig);			// clears disp mem and attribute mem area
		/* first update the display data structs */
		sDispOrig += CellsInLine;
		if (sDispOrig > sScreenBuffLast)
			sDispOrig = sScreenBuff;			/* Display origin crosses over */
	
		/* Now call window routines to update the window */
		ScrollWindow(hTermWnd, 0, -(nStdioCharHeight),
					(LPRECT) NULL, (LPRECT) &TermWindow);
		UpdateWindow(hTermWnd);
	} else {
		EmlnScrollWinUp( 1, ScrollTopMargin, ScrollBottomMargin);
	}
}

/***************************************************************************
*	Routine	:	WriteChar
*	Input	:	The character to be written
*	Return	:	None
*	Synopsis:	Displays the character at the current Cursor position
*				and advances the cursor position
***************************************************************************/
void	WriteChar (BYTE ch)
{
	char *line_ptr;

	if ( ch == BELL_ASCII ) {
		SoundBell();
		goto ExitEcho;
	}
	EraseCursor();					/* we now do a char out */
	if (ch == CR_ASCII ) {
		ToAdvanceCursor = FALSE;
		nCurX = 0;
		goto DrawCurAndExit;
	}
	if ( ch == LF_ASCII ) {
		ToAdvanceCursor = FALSE;
		if (nCurY == ScrollBottomMargin) 
			ScrlDispBuff();
		else 
			nCurY++;
		goto DrawCurAndExit;
	}
	if ( PbkEntry.TrmParams.BackSpace )	{	/* True if back space char is remapped */
		if ( ch == BackSpaceChar ) {
			ToAdvanceCursor = FALSE;
			DoBackSpace();
			goto DrawCurAndExit;
		}
	} else if (ch == BKSPACE_ASCII) {
		ToAdvanceCursor = FALSE;
		DoBackSpace();
		goto DrawCurAndExit;
	}
	/* Now That we have taken care all the special chars
	it is time for the actual chars */
	if (ToAdvanceCursor && nCurX >= MaxClmIndx) {
		AdvanceCursor();
		EraseCursor();
	}
	ToAdvanceCursor = FALSE;
	line_ptr = GetStdioLine(nCurY);
	*(line_ptr + nCurX) = ch;
	*(line_ptr + nCurX + ScrnBufSize) = CurAttrib;

	/* We have to do  a text out for displaying the char */
	/* This is a complex routine which does the char output */
	CharOutWin(ch);

	/* Update the cursor position */
	nCurX++;
	if (nCurX <= MaxClmIndx) {
		ToAdvanceCursor = FALSE;
	} else {
		nCurX = MaxClmIndx;
		if (PbkEntry.TrmParams.AutoMargin)
			ToAdvanceCursor = TRUE;
	}
DrawCurAndExit:
	CurOff = FALSE;
	ReDrawCursor();

ExitEcho:
	return;
}

/***************************************************************************
*	Routine	:	
*	Input	:
*	Return	:
*	Synopsis:
***************************************************************************/
void	SettingFocus(void)
{
	/* create a caret for the cursor management */
	/* A Block caret of one character height and width */
	CreateCaret(hTermWnd, NULL, nStdioCharWidth, nStdioCharHeight);
	ReDrawCursor();
}

/***************************************************************************
*	Routine	:	
*	Input	:
*	Return	:
*	Synopsis:
***************************************************************************/
void	LoosingFocus(void)
{
	DestroyCaret();
}

/***************************************************************************
*	Routine	:	
*	Input	:
*	Return	:
*	Synopsis:	Before wring any character using TextOut we need to hide
*				the cursor if it is a caret. The cursor is a caret only when
*				the terminal window is in focus. 
*	Note	:	the RedrawCursor call make the corresponding ShowCaret call.
***************************************************************************/
void	EraseCursor(void)
{
	CurOff = TRUE;
	if (InFocus)
		RemoveCursor();
}

/***************************************************************************
*	Routine	:	ReDrawCursor
*	Input	:	None
*	Return	:	None
*	Synopsis:	Draws the Cursor at the current nCurX - nCurY position
***************************************************************************/
void	ReDrawCursor(void)
{
	if (CurOff)					/* Emulation cursor On Flag */
		return;
	if (ScrollLock)
		return;
	if ( PbkEntry.TrmParams.FollowCursor)
		if (!CursorVisible())
			AdjustWin();
	if (InFocus)
		DisplayCursor();
}

/***************************************************************************
*	Routine	:	DrawCursorIfInView(void)
*	Input	:	None
*	Return	:	None
*	Synopsis:	Draws the Cursor at the current nCurX - nCurY position
***************************************************************************/
void	DrawCursorIfInView(void)
{
	if (CurOff || ScrollLock || ( ! CursorVisible()) )	
		return;
	if (InFocus)
		DisplayCursor ();
}

/**************************************************************************
*	Function:	SetCursorPosition
*	Input	:	The position word, upper byte is row (ie Y displacement)
*	Synop	:	Sets the caret at the specified position on the logical
*				coordinates. Updates the current cursor position CurPos
**************************************************************************/
void	SetCursorPosition (WORD position)
{
	nCurX = (BYTE) (position & 0xFF);	/* Update cursor positions */
	nCurY = (BYTE) ((position >> 8) & 0xFF);
	nCurX = min (nCurX, MaxClmIndx);
	nCurY = min (nCurY, MaxRowIndx);
	RemoveCursor();
	if (CursorVisible())
		DisplayCursor();
}

/***************************************************************************
*	routine	:	GetCursorPosition
*
*	Synop	: return the current cursor position in the YX format
****************************************************************************/
WORD	GetCursorPosition (void)
{
	return ( ((nCurY << 8) & 0xFF00) | (nCurX & 0xFF) );	
}

/***************************************************************************
*	Routine	:	CreateTermCursor
*	Input	:	None
*	Return	:	None
*	Synopsis:	Creates a caret and displays it at 0, 0. This is the
*				emulation cursor
***************************************************************************/
void	CreateTermCursor(WORD position)
{
	 /* create a caret for the cursor management */
	CreateCaret(hTermWnd, NULL, nStdioCharWidth, nStdioCharHeight);
	 								/* A Block caret of one 
	 								character height and width */
	SetCursorPosition(position); 	/* This function sets the caret at logical */
									/* co-ordinates 0, 0 */
}

/***************************************************************************
*	Routine	:	DestroyTermCursor
*	Input	:	None
*	Return	:	None
*	Synopsis:	The Caret created for the cursor is deleted
***************************************************************************/
void	DestroyTermCursor(void)
{
	DestroyCaret();
}

void	CreateCursorTimer (void);
WORD 	FAR PASCAL CursorTimer (HWND hWnd, WORD wMsg, int nIDEvent, DWORD dwTime);

int		CursorTimerID = 0;
BOOL	ResetCursorTimer = FALSE;
FARPROC	CursorTimerFn;

void
DisplayCursor (void)
{
	BYTE	xpos = nCurX - WinOrgX;
	BYTE	ypos = nCurY - WinOrgY;
	if ((xpos>=ClmsInWin) || (ypos >= RowsInWin))
		return;

	if (CursorTimerID) {
		ResetCursorTimer = TRUE;
		return;
	}
	SetCaretPos(xpos * nStdioCharWidth,
				ypos * nStdioCharHeight);/* position the caret */
	for ( ; CurHideCnt; CurHideCnt--)
		ShowCaret(hTermWnd);		/* display the caret */
	ShowCaret(hTermWnd);
}

void
RemoveCursor (void)
{
	if (CurHideCnt)
		return;
	CurHideCnt++;			/* increment the hide count */
	HideCaret(hTermWnd);	/* caret is the cursor. so hide it */
	CreateCursorTimer ();
}

void
CreateCursorTimer (void)
{
	if (! (CursorTimerID = SetTimer(NULL, NULL, 100, CursorTimerFn)))
		FreeProcInstance (CursorTimerFn);
}

WORD FAR PASCAL
CursorTimer (HWND hWnd, WORD wMsg, int nIDEvent, DWORD dwTime)
{
	if (ResetCursorTimer) {
		ResetCursorTimer = FALSE;
		return TRUE;
	}
	KillTimer (NULL, CursorTimerID);
	CursorTimerID = 0;
	DisplayCursor ();
	return TRUE;
}
/***************************************************************************
*	Routine	:	CharVisible
*	Input	:	None
*	Return	:	TRUE or FALSE
*	Synopsis:	Check if the Cursor is visible in the window
***************************************************************************/
int		CharVisible(BYTE CharX, BYTE CharY)
{
	if (CharX >= WinOrgX && CharX < (WinOrgX + ClmsInWin)
		&& CharY >= WinOrgY &&	CharY < (WinOrgY + RowsInWin) )
		return TRUE;
	return FALSE;
}

/***************************************************************************
*	Routine	:	DispChar(BYTE ch, BYTE attrib, WORD cnt)
*	Input	:	Character to display, attribute and count
*	Return	:	
*	Synopsis:	Display the character starting from current cursor position.
*				The cursor position is not affected
*				doesn't wrap around the display.
*				The attrib parameter is not used at present
***************************************************************************/
void	DispChar(BYTE ch, BYTE attrib, WORD cnt)
{
	BYTE	tCurX = nCurX;			// changed from word 2 byte
	BYTE	*line_ptr;

	if ( IN_SCROLL)
		return;

	/* Get a pointer to the line */
	line_ptr = GetStdioLine(nCurY);

	/* Hide the caret before any text out calls */
	HideCaret(hTermWnd);

	/* Update the terminal buffer with the chars */
	while (cnt-- && (tCurX <= MaxClmIndx)) {
		*(line_ptr + tCurX) = ch;
		if (CharVisible(tCurX, nCurY))
			 TextOut ( hTermDC, (tCurX - WinOrgX) * nStdioCharWidth,
		 					(nCurY - WinOrgY) * nStdioCharHeight, &ch, 1);
		 tCurX++;
	}
	ShowCaret(hTermWnd);			/* display the caret */
}

/***************************************************************************
*	Routine	:	DispStr(BYTE ch, BYTE attrib
*	Input	:	Str ptr to display, attribute
*	Return	:	
*	Synopsis:	Display the character string from current cursor position.
*				The cursor Updated. Doesn't wrap around the display
*				The attrib parameter is not used at present
***************************************************************************/
void	DispStr(BYTE *str, BYTE attrib)
{
	BYTE	*line_ptr;

	if ( IN_SCROLL)
		return;

	/* Get a pointer to the line */
	line_ptr = GetStdioLine(nCurY);

	/* Hide the caret befor any text out calls */
	HideCaret(hTermWnd);

	/* Update the terminal buffer with the chars */
	while ( *str && (nCurX <= MaxClmIndx)) {
		*(line_ptr + nCurX) = *str;
		if (CharVisible(nCurX, nCurY))
			TextOut ( hTermDC, (nCurX - WinOrgX) * nStdioCharWidth,
		 					(nCurY - WinOrgY) * nStdioCharHeight, str, 1);
		 str++;
		 if (nCurX)
			 nCurX++;
	}
	ShowCaret(hTermWnd);			/* display the caret */
}

/*--------------------------------------------------------------------------*/
/*	Name		:	ShowMesg (LPSTR err_str)								*/
/*	Input		:	str, error string to be shown.							*/
/*	Output		:	None													*/
/*	Synopsis	:	Displays the error message and waits for OK				*/
/*--------------------------------------------------------------------------*/
void	ShowMesg (LPSTR err_str)
{
	/* don't send any messages to the screen if we are in scroll mode */
	if ( IN_SCROLL)
		return;
	while (*err_str) {
		if (*err_str == '\n')
			WriteChar('\r');
		WriteChar (*err_str++);
	}
}

/*--------------------------------------------------------------------------*/
/*	Name		:	ShowNmesg (LPSTR err_str, int count)								*/
/*	Input		:	str, error string to be shown.							*/
/*	Output		:	None													*/
/*	Synopsis	:	Displays the error message and waits for OK				*/
/*--------------------------------------------------------------------------*/
void	ShowNmesg (LPSTR err_str, int count)
{
	if ( IN_SCROLL)
		return;
	while (count--) {
		if (*err_str == '\n')
			WriteChar('\r');
		WriteChar (*err_str++);
	}
}
/*--------------------------   Last  Line   ------------------------------ */
