/* AGUTIL.C -- Has utility routines.
** By: Sanjay
** Start: 12, July, 1996
** Done: 30, August, 1996
*/

#include "rtrstd.h"

#include "kag.h"

#include "vagstr.h"
#include "vag.h"

#include "agpkttyp.h"
#include "agutil.h"

#include <serial.h>
#include <udb.h>
#include <wanmgr.h>

/* -- CODE ------------------------------------------------------------- */

/* ag_get_port_availability()
**	This function determines if a port can be used by a dialout requestor.
**	Basically if a WAN port is configured as disabled or non-async or if the
**	WAN port is currently owned by someone else, the port is not available.
*/
enum AG_PORT_AVAILABILITY ag_get_port_availability(USHORT port_number)
{
	if (get_wan_port_owner(port_number) != OWNED_BY_NONE)
		return AG_PORT_IN_USE;

	if ((is_wan_enabled(port_number) == FALSE) || 
		(is_wan_port_ag_enabled(port_number) == FALSE)) 
		return AG_PORT_DISABLED;

	if (is_wan_async(port_number) == FALSE)
		return AG_PORT_NOT_ASYNC;

	return AG_PORT_AVAILABLE;
}

/* change_endian()
**	The packets received from clients have info in little-endian (intel)
**	format. So we need to convert WORD's and DWORD's to big-endian (motorola)
**	format before use. Likewise we need to convert from big-endian to
**	little-endian on packets sent to the clients from here. This has to
**	be done only on the UNION_AG_PACKET class of packets.
**  Note that the 'sptr_ag_packet' passed should represent a fully filled
**	packet.
*/
void change_endian(UNION_AG_PACKET *sptr_ag_packet)
{
	switch (sptr_ag_packet->packet_class)
	{
	/* ------------------------------------------------------------------- */
	/* Request cases fisrt */
	case AG_NAME_VALIDATE_REQ:
		/* Nothing to convert */
		break;
	case AG_PASSWORD_VALIDATE_REQ:
		/* Nothing to convert */
		break;
	case AG_CONNECT_REQ:
		/* Nothing to convert */
		break;
	case AG_STATUS_REQ:
		/* Nothing to convert */
		break;
	case AG_DISCONNECT_REQ:
		/* Nothing to convert */
		break;
	case AG_CHANGE_CONNECT_PARAM:
		little_to_big_endian_ushort(&sptr_ag_packet->packet_type.change_parameters_request.connect_vars_info.line_parameters.baud_divisor);

		little_to_big_endian_ushort(&sptr_ag_packet->packet_type.change_parameters_request.connect_vars_info.max_packet_size);
		little_to_big_endian_ushort(&sptr_ag_packet->packet_type.change_parameters_request.connect_vars_info.max_inter_packet_delay);
		little_to_big_endian_ushort(&sptr_ag_packet->packet_type.change_parameters_request.connect_vars_info.idle_time);

		break;
	case AG_RC_REQ_PKT:
		/* Done in-place on-need */
		break;
	case AG_IN_CONNECT_REQ:
		/* Nothing to convert */
		break;
	case AG_DATA_PACKET:
		little_to_big_endian_ushort(&sptr_ag_packet->packet_type.data_packet.packet_size);

		break;
	case AG_CONTROL_PACKET:
		/* Nothing to convert */
		break;
	/* ------------------------------------------------------------------- */
	/* The following are response codes */
	case AG_CONNECT_ABORTED:
		/* Nothing to convert */
		break;
	case AG_FLOW_CONTROL:
		/* Unlike the DOS AG, we do not need XON/XOFF, so we will never 
		** handle this kind of packets.
		*/
		break;
	case AG_NAME_VALIDATE_RESP:
		/* Nothing to convert */
		break;
	case AG_PASSWORD_VALIDATE_RESP:
		/* Nothing to convert */
		break;
	case AG_CONNECT_RESP:
	case AG_INBOUND_CONNECT_RESP:
		big_to_little_endian_ushort(&sptr_ag_packet->packet_type.connect_response.connect_vars_info.line_parameters.baud_divisor);
		
		big_to_little_endian_ushort(&sptr_ag_packet->packet_type.connect_response.connect_vars_info.max_packet_size);
		big_to_little_endian_ushort(&sptr_ag_packet->packet_type.connect_response.connect_vars_info.max_inter_packet_delay);
		big_to_little_endian_ushort(&sptr_ag_packet->packet_type.connect_response.connect_vars_info.idle_time);

		break;
	case AG_STATUS_RESP:
		/* This is done at the packet building point as we at this stage
		** cannot determine how many elements need to be modified.
		*/
		break;
	case AG_DISCONNECT_RESP:
		/* Nothing to convert */
		break;
	case AG_BROADCAST_PACKET:
		/* This type is not used at all in DOS AG */
		break;
	case AG_RC_RESPONSE_PACKET:
		/* Done in-place on-need */
		break;
	}
}

/* little_to_big_endian_ushort()
** little_to_big_endian_ulong()
** big_to_little_endian_ushort()
** big_to_little_endian_ulong()
**		These are endian conversion routines that do in-place change of
**		the byte ordering of USHORT and ULONG variables.
*/

USHORT little_to_big_endian_ushort(USHORT *little_endian_ushort)
{
	USHORT big_endian_ushort;

	big_endian_ushort = (*little_endian_ushort >> 8) | (*little_endian_ushort << 8);
	*little_endian_ushort = big_endian_ushort;	/* In-place conversion */
	return big_endian_ushort;
}

ULONG little_to_big_endian_ulong(ULONG *little_endian_ulong)
{
	ULONG big_endian_ulong;

	big_endian_ulong = (*little_endian_ulong << 24) | 
						(*little_endian_ulong >> 24) |
						((*little_endian_ulong & 0x0000FF00) << 8) |
						((*little_endian_ulong & 0x00FF0000) >> 8);
	*little_endian_ulong = big_endian_ulong;	/* In-place conversion */
	return big_endian_ulong;
}

USHORT big_to_little_endian_ushort(USHORT *big_endian_ushort)
{
	/* Actually same as other way, but this different name is convenient */
	USHORT little_endian_ushort;

	little_endian_ushort = (*big_endian_ushort >> 8) | (*big_endian_ushort << 8);
	*big_endian_ushort = little_endian_ushort;	/* In-place conversion */
	return little_endian_ushort;
}

ULONG big_to_little_endian_ulong(ULONG *big_endian_ulong)
{
	/* Actually same as other way, but this different name is convenient */
	ULONG little_endian_ulong;

	little_endian_ulong = (*big_endian_ulong << 24) | 
						(*big_endian_ulong >> 24) |
						((*big_endian_ulong & 0x0000FF00) << 8) |
						((*big_endian_ulong & 0x00FF0000) >> 8);
	*big_endian_ulong = little_endian_ulong;	/* In-place conversion */
	return little_endian_ulong;
}

void ag_printf(enum AG_PRINTF_GROUPS printf_group, const char *cptr_format, ...)
{
	va_list argptr;
	BYTE_ENUM(BOOLEAN) print_string;

	va_start(argptr, cptr_format);
	if (ag.print_class.ag_print_switch_on == FALSE)
	{
		va_end(argptr);
		return;
	}

	switch (printf_group)
	{
	case AG_INIT_PRINTF:
		print_string = ag.print_class.init_printing_enabled;
		break;

	case AG_ALARM_PRINTF:
		print_string = ag.print_class.alarm_printing_enabled;
		break;

	case AG_SESS_PRINTF:
		print_string = ag.print_class.session_printing_enabled;
		break;

	case AG_MCSI_IN_PRINTF:
		print_string = ag.print_class.mcsi_in_printing_enabled;
		break;

	case AG_MCSI_OUT_PRINTF:
		print_string = ag.print_class.mcsi_out_printing_enabled;
		break;

	case AG_CONNECT_PRINTF:
		print_string = ag.print_class.connect_printing_enabled;
		break;

	case AG_DIAGNOSTIC_PRINTF:
		print_string = ag.print_class.diagnostic_printing_enabled;
		break;

	case AG_LINE_PRINTF:
		print_string = ag.print_class.line_printing_enabled;
		break;

	case AG_INBOUND_PRINTF:
		print_string = ag.print_class.inbound_printing_enabled;
		break;

	default:
		print_string = FALSE;
		break;
	}

	if (print_string == TRUE)
	{
		vprintf(cptr_format, argptr);
	}

	va_end(argptr);
}

void update_line_lan_tx_statistics(BYTE line_id, USHORT send_size)
{
	LINE_INFO_ENTRY *sptr_line_info;

	if (line_id == UNKNOWN_LINE)
		return;

	sptr_line_info = &ag.line_info_array[line_id];
	if (ag.statistics_timer != 0)
		sptr_line_info->stats.lan_tx_bytes += send_size;
	sptr_line_info->stats.current_tx_data_packets++;
	sptr_line_info->stats.log_tx_data_packets++;
	sptr_line_info->stats.total_tx_data_packets++;
	sptr_line_info->stats.current_lan_tx_bytes += send_size - sizeof(DATA_TYPE) + sizeof(BYTE);
	sptr_line_info->stats.log_lan_tx_bytes += send_size - sizeof(DATA_TYPE) + sizeof(BYTE);
	sptr_line_info->stats.total_lan_tx_bytes += send_size - sizeof(DATA_TYPE) + sizeof(BYTE);
}

void update_line_lan_rx_statistics(BYTE line_id, USHORT receive_size)
{
	LINE_INFO_ENTRY *sptr_line_info;


	if (line_id == UNKNOWN_LINE)
		return;

	sptr_line_info = &ag.line_info_array[line_id];
	if (ag.statistics_timer != 0)
		sptr_line_info->stats.lan_rx_bytes += receive_size;
	sptr_line_info->stats.current_rx_data_packets++;
	sptr_line_info->stats.log_rx_data_packets++;
	sptr_line_info->stats.total_rx_data_packets++;
	sptr_line_info->stats.current_lan_rx_bytes += receive_size - sizeof(DATA_TYPE) + sizeof(BYTE);
	sptr_line_info->stats.log_lan_rx_bytes += receive_size - sizeof(DATA_TYPE) + sizeof(BYTE);
	sptr_line_info->stats.total_lan_rx_bytes += receive_size - sizeof(DATA_TYPE) + sizeof(BYTE);
}

void update_line_line_tx_statistics(BYTE line_id, USHORT send_size)
{
	LINE_INFO_ENTRY *sptr_line_info;


	if (line_id == UNKNOWN_LINE)
		return;

	sptr_line_info = &ag.line_info_array[line_id];
	if (ag.statistics_timer != 0)
		sptr_line_info->stats.line_tx_bytes += send_size;
	sptr_line_info->stats.current_line_tx_bytes += send_size;
	sptr_line_info->stats.log_line_tx_bytes += send_size;
	sptr_line_info->stats.total_line_tx_bytes += send_size;
}

void update_line_line_rx_statistics(BYTE line_id, USHORT receive_size)
{
	LINE_INFO_ENTRY *sptr_line_info;


	if (line_id == UNKNOWN_LINE)
		return;

	sptr_line_info = &ag.line_info_array[line_id];
	if (ag.statistics_timer != 0)
		sptr_line_info->stats.line_rx_bytes += receive_size;
	sptr_line_info->stats.current_line_rx_bytes += receive_size;
	sptr_line_info->stats.log_line_rx_bytes += receive_size;
	sptr_line_info->stats.total_line_rx_bytes += receive_size;
}

void reset_line_current_statistics(LINE_INFO_ENTRY *sptr_line_info)
{
	memset(&sptr_line_info->stats.connect_time, 0, sizeof(DATE_TIME));

	sptr_line_info->stats.pending_tx_bytes = 0;
	sptr_line_info->stats.pending_rx_bytes = 0;

	sptr_line_info->stats.current_tx_data_packets = 0;
	sptr_line_info->stats.current_rx_data_packets = 0;

	sptr_line_info->stats.current_line_tx_bytes = 0;
	sptr_line_info->stats.current_line_rx_bytes = 0;
	sptr_line_info->stats.current_lan_tx_bytes = 0;
	sptr_line_info->stats.current_lan_rx_bytes = 0;
}

/* NOTE: That we are returning a structure that can be assigned */
void get_current_date_time(DATE_TIME *sptr_date_time)
{
	/* Since we do not have a RTC in the box, we currently return 0.
	** When we get a version of the box with a RTC, this function can
	** be suitably coded and all other parts will work.
	*/

	memset(sptr_date_time, 0, sizeof(DATE_TIME));
}

/* Added By Naveen to support TELNET developed by Sudhir ... */
void  ag_get_caller_name(USHORT port_number, BYTE* user_name)
{
	LINE_INFO_ENTRY *sptr_line_info;

	sptr_line_info = &ag.line_info_array[port_number];
	if (!(sptr_line_info->line_status == AGLS_LINE_FREE))
	{
		get_ras_user_name(sptr_line_info->sptr_session_entry->user_entry, user_name);
		user_name[USER_NAME_LENGTH] = 0;
	}
	else
	{
		/* Port is not occupied by any user. */
		user_name[0] = 0;
	}
}	
BYTE ag_get_call_mode(USHORT port_number)
{
	LINE_INFO_ENTRY *sptr_line_info;

	sptr_line_info = &ag.line_info_array[port_number];
	if (sptr_line_info->line_status == AGLS_LINE_FREE)
	{
		/* This port is not taken by AG module */
		if (get_wan_port_owner(port_number) == OWNED_BY_NONE)
			return 2;
		else
			return 3;
	}
	else
	{
		/* This port is taken by AG for either INBOUND or for OUTBOUND */
		return (sptr_line_info->connect_mode );
	}

#if 0
	if (sptr_line_info->connect_mode == OUTBOUND_MODEM)
	  	return 0;
	 else
	 if (sptr_line_info->connect_mode == INBOUND_MODEM)
	  	return 1;
	 else
	 if (get_wan_port_owner(port_number) == OWNED_BY_NONE)
	  	return 2;
	 else
	   return 3;
#endif
}


ULONG ag_get_baud_rate(USHORT port_number)
{
	LINE_INFO_ENTRY *sptr_line_info;

	sptr_line_info = &ag.line_info_array[port_number];
	return sptr_line_info->line_vars.current_baud_rate;
}
/* ... End Of Addition by Naveen */


/* sudhir 26/6/97 
	This function is called only from shell.c in wandrvr */

int ag_get_number_of_inbound_users (USHORT port_number)
{
   if (ag.enabled != TRUE)
      return (0) ;
   return (ag.line_info_array[port_number].number_of_inbound_users) ;
}

#if 0

/* On 22/2/99 for flow control */

/* calculate_wait_time()
**	When we send an XOFF packet to a client, we also send a timer count
**	in PC timer ticks estimating the duration that the client should stop 
**	sending before attempting to transmit again. This function calculates 
**	this timer count as a function of the number of bytes from the client 
**	waiting to go out on the serial side and the current baud rate.
** NOTE: In our case (as compare to a driver on a PC), it is diffcult to
**	give such a time estimate due to the WAY the SCC uses buffers rather than
**	bytes.
*/
USHORT calculate_wait_time(LINE_INFO_ENTRY *sptr_line_info)
{
	USHORT wait_time;
	USHORT port_number = (USHORT)sptr_line_info->line_id;


	wait_time = serial_get_tx_time_estimate(port_number);
	sptr_line_info->xoff_timer = wait_time;
	ag_printf(AG_LINE_PRINTF, "AG: WAN %u: XOFF period = %u\n\r", port_number, wait_time);
	return wait_time;
}
#endif



/* -- END CODE -------------------------------------------------------- */
