#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <string.h>
#include "dhcps.h"
#include "..\..\store\boot.h"
#include <flashmgr.h>



STATIC void transfer_dhcp_database_to_flash () ;
STATIC DHCP_DATABASE_ENTRY *get_database_entry (BYTE chaddr[], BYTE hlen)
{
   DHCP_DATABASE_ENTRY *sptr_next_node = dhcp_server.sptr_dhcp_database ;

   while (sptr_next_node)
   {
      if (memcmp (&chaddr[0], &sptr_next_node->client_hardware_address[0], (int)hlen) == 0)
	 return (sptr_next_node) ;

      sptr_next_node = sptr_next_node->sptr_next_entry ;
   }
   return NULL ;
}


int add_entry_to_database (BYTE chaddr[], BYTE hlen, ULONG ip_address, ULONG lease_period, enum DHCP_DATABASE_ENTRY_STATE state)
{
   DHCP_DATABASE_ENTRY *sptr_new_node ;

   sptr_new_node = get_database_entry (chaddr, hlen) ;
   if (sptr_new_node == NULL)
   {
      sptr_new_node = (DHCP_DATABASE_ENTRY *) malloc (sizeof(DHCP_DATABASE_ENTRY)) ;
      if (sptr_new_node == NULL)
	 return (FAIL) ;
      sptr_new_node->sptr_next_entry = dhcp_server.sptr_dhcp_database ;
      dhcp_server.sptr_dhcp_database = sptr_new_node ;
      dhcp_server.number_of_database_entries++ ;
      dhcp_server.database_length += (ULONG) (hlen + sizeof(BYTE) + sizeof(ULONG)) ;
      dhcp_server.db_changed = TRUE ;
   }
   sptr_new_node->state = state ;
   sptr_new_node->hardware_address_length = hlen ;
   sptr_new_node->lease_timer = lease_period ;
   sptr_new_node->allocated_ip_address = ip_address ;
   memcpy (&sptr_new_node->client_hardware_address[0], &chaddr[0], (int)hlen) ;

   return (PASS) ;
}


int is_ip_address_free (ULONG ip_address)
{
   DHCP_DATABASE_ENTRY *sptr_next_node = dhcp_server.sptr_dhcp_database ;

   while (sptr_next_node)
   {
      if (ip_address == sptr_next_node->allocated_ip_address)
	 return (FAIL) ;

      sptr_next_node = sptr_next_node->sptr_next_entry ;
   }

   return (PASS) ;
}



ULONG get_ip_address_from_mac_address (BYTE *mac_address, BYTE mac_address_length)
{
   DHCP_DATABASE_ENTRY *sptr_database_entry = get_database_entry (mac_address, mac_address_length) ;

   if (sptr_database_entry)
      return sptr_database_entry->allocated_ip_address ;

   return (0L) ;
}



ULONG get_first_dirty_ip_address ()
{
   DHCP_DATABASE_ENTRY *sptr_next_node = dhcp_server.sptr_dhcp_database ;

   while (sptr_next_node)
   {
      if (sptr_next_node->state == DIRTY)
	 return (sptr_next_node->allocated_ip_address) ;

      sptr_next_node = sptr_next_node->sptr_next_entry ;
   }
   return 0L ;
}


void delete_entry_given_ip_address (ULONG ip_address)
{
   DHCP_DATABASE_ENTRY *sptr_prev_node= dhcp_server.sptr_dhcp_database, *sptr_next_node ;
   
   if (dhcp_server.sptr_dhcp_database->allocated_ip_address == ip_address)
   {
      sptr_next_node = dhcp_server.sptr_dhcp_database ;
      dhcp_server.sptr_dhcp_database = dhcp_server.sptr_dhcp_database->sptr_next_entry ;

      dhcp_server.number_of_database_entries-- ;
      dhcp_server.database_length -= (sptr_next_node->hardware_address_length + sizeof(BYTE) + sizeof(ULONG)) ;

      free (sptr_next_node) ;
      dhcp_server.db_changed = TRUE ;
      return ;
   }

   sptr_next_node = sptr_prev_node->sptr_next_entry ;

   while (sptr_next_node)
   {
      if (sptr_next_node->allocated_ip_address == ip_address)
      {
	 sptr_prev_node->sptr_next_entry = sptr_next_node->sptr_next_entry ;

	 dhcp_server.number_of_database_entries-- ;
	 dhcp_server.database_length -= (sptr_next_node->hardware_address_length + sizeof(BYTE) + sizeof(ULONG)) ;

	 free (sptr_next_node) ;
	 return ;
      }
      sptr_prev_node = sptr_next_node ;
      sptr_next_node = sptr_next_node->sptr_next_entry ;
   }
}


void paint_entry_dirty (ULONG ip_address)
{
   DHCP_DATABASE_ENTRY *sptr_next_node = dhcp_server.sptr_dhcp_database ;
   
   while (sptr_next_node)
   {
      if (sptr_next_node->allocated_ip_address == ip_address)
      {
	 sptr_next_node->state = DIRTY ;
	 return ;
      }
      sptr_next_node = sptr_next_node->sptr_next_entry ;
   }
   return ;
}


int is_ip_address_offered (ULONG ip_address, BYTE chaddr[], BYTE hlen)
{
   DHCP_DATABASE_ENTRY *sptr_next_node = dhcp_server.sptr_dhcp_database ;
   
   while (sptr_next_node)
   {
      if ((sptr_next_node->allocated_ip_address == ip_address) &&
	  (sptr_next_node->state == OFFERED) &&
	  (memcmp (&sptr_next_node->client_hardware_address[0], &chaddr[0], hlen) == 0))

	  return (PASS) ;

      sptr_next_node = sptr_next_node->sptr_next_entry ;
   }
   return (FAIL) ;
}

void dhcp_server_age_database_entries (void)
{
   DHCP_DATABASE_ENTRY *sptr_prev_node, *sptr_next_node = dhcp_server.sptr_dhcp_database ;

   while (sptr_next_node)
   {
      switch (sptr_next_node->state)
      {
	 case (OFFERED) :
	    if (sptr_next_node->lease_timer)
	       sptr_next_node->lease_timer-- ;
	    else
	    break ;
	       sptr_next_node->state = DELETE ;
	    
	 case (LEASED) :
	    if (sptr_next_node->lease_timer)
	       sptr_next_node->lease_timer-- ;
	    else
	       sptr_next_node->state = DIRTY ;
	    break ;

	 case (DIRTY) :
	    break ;
      }
      sptr_prev_node = sptr_next_node ;
      sptr_next_node = sptr_next_node->sptr_next_entry ;
      if (sptr_prev_node->state == DELETE)
	 delete_entry_given_ip_address (sptr_prev_node->allocated_ip_address) ;
   }
}


void dhcp_server_timer ()
{
   /* Age the database entries */
   if (dhcp_server.clock_ticks_to_be_passed)
      dhcp_server.clock_ticks_to_be_passed-- ;
   else
   {
      dhcp_server_age_database_entries () ;
      dhcp_server.clock_ticks_to_be_passed = dhcp_server.clock_ticks_per_second ;
   }

   /* Update the database to the flash */
   if (dhcp_server.db_counter)
      dhcp_server.db_counter-- ;
   else
   {
      dhcp_server.db_counter = dhcp_server.db_update_period ;

      if (dhcp_server.db_changed)
      {
	      transfer_dhcp_database_to_flash () ;
      }
   }
}


#define DHCP_DATABASE_DESCRIPTION "DHCP DATABASE"
#define DHCP_DATABASE_VSRSION "0.99"
#define MAX_DHCP_DATABASE_LENGTH (21 * 4096)
/* To accomodate 4K maximum sized entries (the hardware address cannot
be more than  16 bytes and the other 5 bytes are for IP address (5
bytes) and hardware length (1 byte) */

extern USHORT update_crc (USHORT acc_crc, BYTE *packet, ULONG length) ;


STATIC void clean_up_dhcp_database ()
{
   DHCP_SERVER_DATABASE_HEADER dhcp_header ;
   
   memset (&dhcp_header, 0, sizeof (DHCP_SERVER_DATABASE_HEADER)) ;

   dhcp_header.magic_number = 0xABCD ;
   dhcp_header.crc = 0xFFFF ;
   strcpy (dhcp_header.description, DHCP_DATABASE_DESCRIPTION) ;

	c_write_to_flash ((BYTE *)&dhcp_header, (BYTE *)FL_DHCP_HDR, 512) ;

   return ;
}


STATIC dhcp_db_write_complete ()
{
   dhcp_server.db_changed = FALSE ;
   dhcp_server.db_counter = dhcp_server.db_update_period ;
}


STATIC void transfer_dhcp_database_to_flash ()
{
   DHCP_SERVER_DATABASE_HEADER dhcp_header ;
   BootConfigType *fl_boot_hdr = (BootConfigType *)FL_BOOT_HDR ;
   int page_number, number_of_512byte_pages ;
   BYTE *bptr_new_block ;
   int bytes_transfered = 0, bytes_to_be_transfered, bytes_transfered_from_last_block ;
   DHCP_DATABASE_ENTRY *sptr_next_node = dhcp_server.sptr_dhcp_database ;
   BYTE dhcp_database_entry[16 + 4 + 1 + 25] ; /* 16 : chaddr, 4 : IP addr, 1 : hlen, 25 : extra */
   
   /* Write it at the end of the last Flash PROM */
   BYTE *bptr_destination =  (BYTE *) (((FL_ROMM_START + fl_boot_hdr->FlashPromSize) / 512) * 512) ;

   printf ("DHCP SERVER : Transfering DHCP Database to Flash.........\n") ;
   memset (&dhcp_header, 0, sizeof (DHCP_SERVER_DATABASE_HEADER)) ;
   dhcp_server.db_changed = FALSE ;
   dhcp_server.db_counter = dhcp_server.db_update_period ;

   number_of_512byte_pages = dhcp_server.database_length / 512 ;

   if (dhcp_server.database_length % 512)
      number_of_512byte_pages++ ;

   bptr_destination -= (number_of_512byte_pages * 512) ;
   dhcp_header.crc = 0xFFFF ;
   dhcp_header.down_load_address = (ULONG) bptr_destination ;

   bytes_transfered_from_last_block = 0 ;
   for (page_number = 0 ; page_number < number_of_512byte_pages ; page_number++)
   {
      bytes_to_be_transfered = 512 ;
      bptr_new_block = malloc (512) ;
      if (bptr_new_block == NULL)
      {
	      printf ("DHCP SERVER : No memory to transfer database to Flash\n") ;
	      return ;
      }

      if (bytes_transfered_from_last_block)
      {
	      memcpy (&dhcp_database_entry[0], (BYTE *)&sptr_next_node->allocated_ip_address, sizeof(ULONG)) ;
	      memcpy (&dhcp_database_entry[4], (BYTE *)&sptr_next_node->hardware_address_length, sizeof(BYTE)) ;
	      memcpy (&dhcp_database_entry[5], (BYTE *)&sptr_next_node->client_hardware_address[0], sptr_next_node->hardware_address_length) ;

	      memcpy (bptr_new_block,
	         &dhcp_database_entry[0], 
	         sizeof(ULONG)+sizeof(BYTE)+sptr_next_node->hardware_address_length-bytes_transfered_from_last_block) ;

	      dhcp_header.crc = update_crc (dhcp_header.crc,
	         &dhcp_database_entry[0], 
	         sizeof(ULONG)+sizeof(BYTE)+sptr_next_node->hardware_address_length-bytes_transfered_from_last_block) ;

	      bytes_to_be_transfered += sizeof(ULONG)+sizeof(BYTE)+sptr_next_node->hardware_address_length-bytes_transfered_from_last_block ;
	      bytes_transfered_from_last_block = 0 ;

	      sptr_next_node = sptr_next_node->sptr_next_entry ;
      }

      while (sptr_next_node)
      {
	      memcpy (&dhcp_database_entry[0], (BYTE *)&sptr_next_node->allocated_ip_address, sizeof(ULONG)) ;
	      memcpy (&dhcp_database_entry[4], (BYTE *)&sptr_next_node->hardware_address_length, sizeof(BYTE)) ;
	      memcpy (&dhcp_database_entry[5], (BYTE *)&sptr_next_node->client_hardware_address[0], sptr_next_node->hardware_address_length) ;

	      if (bytes_to_be_transfered >= (sizeof(ULONG)+sizeof(BYTE)+sptr_next_node->hardware_address_length))
	      {
	         memcpy (bptr_new_block+512-bytes_to_be_transfered, &dhcp_database_entry[0], (sizeof(ULONG)+sizeof(BYTE)+sptr_next_node->hardware_address_length)) ;
	         dhcp_header.crc = update_crc (dhcp_header.crc,
	            &dhcp_database_entry[0], (sizeof(ULONG)+sizeof(BYTE)+sptr_next_node->hardware_address_length)) ;

	         bytes_to_be_transfered -= (sizeof(ULONG)+sizeof(BYTE)+sptr_next_node->hardware_address_length) ;
	      }
	      else
	      {
	         bytes_transfered_from_last_block = (sizeof(ULONG)+sizeof(BYTE)+sptr_next_node->hardware_address_length) - bytes_to_be_transfered ;
	         memcpy (bptr_new_block+512-bytes_to_be_transfered, &dhcp_database_entry[0], bytes_to_be_transfered) ;
	         dhcp_header.crc = update_crc (dhcp_header.crc,
	            &dhcp_database_entry[0], bytes_to_be_transfered) ;
	         break ;
	      }
	 
	      sptr_next_node = sptr_next_node->sptr_next_entry ;
	      bytes_transfered_from_last_block = 0 ;
      }
      /* Write to Flash */
      bytes_transfered += 512 ;
      if (bytes_transfered >= dhcp_server.database_length)
	      schedule_flash_write (bptr_new_block, bptr_destination, 512, dhcp_db_write_complete, bptr_new_block) ;
      else
	      schedule_flash_write (bptr_new_block, bptr_destination, 512, NULL, bptr_new_block) ;
	 
      bptr_destination += 512 ;
   }
   
   dhcp_header.magic_number = 0xABCD ;
   strcpy (dhcp_header.description, DHCP_DATABASE_DESCRIPTION) ;
   dhcp_header.no_of_records = dhcp_server.number_of_database_entries ;
   dhcp_header.length = dhcp_server.database_length ;

   c_write_to_flash ((BYTE *)&dhcp_header, (BYTE *)FL_DHCP_HDR, 512) ;

   dhcp_server.db_changed = FALSE ;
   dhcp_server.db_counter = dhcp_server.db_update_period ;
}



STATIC int test_dhcp_database_integrity ()
{
   DHCP_SERVER_DATABASE_HEADER *sptr_dhcp_header = (DHCP_SERVER_DATABASE_HEADER *)FL_DHCP_HDR ;
   USHORT accumulated_crc ;

	if ((sptr_dhcp_header->magic_number != 0xABCD) ||
       (strcmpi (sptr_dhcp_header->description, DHCP_DATABASE_DESCRIPTION)) ||
       (sptr_dhcp_header->length > MAX_DHCP_DATABASE_LENGTH))
   {
      printf ("DHCP SERVER : DHCP Server Database not found, cleaning up\n") ;
      clean_up_dhcp_database () ;
      return (0) ;
   }
   else
   {
      accumulated_crc = update_crc (0xFFFF, sptr_dhcp_header->down_load_address, sptr_dhcp_header->length) ;
      if (accumulated_crc != sptr_dhcp_header->crc)
      {
         printf ("DHCP SERVER : DHCP Server Database Checksum failure, cleaning up\n") ;
	      clean_up_dhcp_database () ;
	      return (0) ;
      }
   }
   return (1) ;
}


typedef struct
{
   ULONG ip_address ;
   BYTE hlen ;
   BYTE chaddr[1] ;
} DATABASE_ENTRY ;

void initialize_dhcp_server_database ()
{
   DHCP_SERVER_DATABASE_HEADER *sptr_dhcp_db_hdr = (DHCP_SERVER_DATABASE_HEADER *)FL_DHCP_HDR ;
   BYTE *bptr_dhcp_database = (BYTE *)sptr_dhcp_db_hdr->down_load_address ;
   ULONG database_length = (ULONG)sptr_dhcp_db_hdr->length ;
   DATABASE_ENTRY *sptr_database_entry ;
   int range_index, number_of_records = 0 ;

   /* Transfer all the records from the Flash to the DRAM.
   Make sure that each entry is checked for any static bindings before it
   gets into the database */

   if (test_dhcp_database_integrity ())
   {
      while (database_length)
      {
	      sptr_database_entry = (DATABASE_ENTRY *)bptr_dhcp_database ;

         range_index = get_range_index (sptr_database_entry->ip_address) ;
         if (range_index != -1)
         {
            if (is_ip_address_excluded (sptr_database_entry->ip_address, range_index) == 0)
            {
               if ((is_mac_address_bound (sptr_database_entry->chaddr, sptr_database_entry->hlen) == 0) &&
                   (is_ip_address_bound (sptr_database_entry->ip_address) == 0))
               {
                  add_entry_to_database (sptr_database_entry->chaddr, sptr_database_entry->hlen, sptr_database_entry->ip_address, 0L, DIRTY) ;
                  number_of_records++ ;
               }
            }
         }
         database_length -= (sizeof(BYTE) + sizeof(ULONG) + sptr_database_entry->hlen) ;
	      bptr_dhcp_database += (sizeof(BYTE) + sizeof(ULONG) + sptr_database_entry->hlen) ;
      }
      printf ("DHCP SERVER : Transfering Database (%d %s) from Flash\n",
         number_of_records,
         number_of_records-1 ? "records" : "record") ;
   }
}




STATIC char *states[] =
{
   "OFFERED",
   "LEASED ",
   "DIRTY  "
} ;


void print_dhcp_database ()
{
   DHCP_DATABASE_ENTRY *sptr_dhcp_next_node = dhcp_server.sptr_dhcp_database ;
   char one_byte[5], hardware_address[20], ip_address[20] ;
   int i, x ;

   buffer_to_print[0] = 0 ;
   sprintf (buffer_to_print, "\nDHCP Server Database..........\n\
|--------------------------|-----------------|-----------|------------|\n\
|     Hardware Address     |    IP Address   |   State   | Lease Time |\n\
|--------------------------|-----------------|-----------|------------|\n") ;
   printf ("%s", buffer_to_print) ;

   buffer_to_print[0] = 0 ;
   while (sptr_dhcp_next_node)
   {
      hardware_address[0] = 0 ;
      for (i = 0 ; i < (int)sptr_dhcp_next_node->hardware_address_length ; i++)
      {
	 x = (int) sptr_dhcp_next_node->client_hardware_address[i] ;
	 sprintf (&one_byte[0], "%02X", x) ;
	 strcat (&hardware_address[0], &one_byte[0]) ;
      }
      ulong_to_dot_format (&ip_address[0], sptr_dhcp_next_node->allocated_ip_address) ;
      sprintf (temp_buffer, "|       %s       | %s |  %s  |  %08LX  |\n",
	 &hardware_address[0], &ip_address[0], states[sptr_dhcp_next_node->state], sptr_dhcp_next_node->lease_timer) ;

      strcat (buffer_to_print, temp_buffer) ;
      if (strlen (buffer_to_print) >= (MAX_PRINT_BUFFER_SIZE - 100))
      {
	 printf ("%s", buffer_to_print) ;
	 buffer_to_print[0] = 0 ;
      }

      sptr_dhcp_next_node = sptr_dhcp_next_node->sptr_next_entry ;
   }
   printf ("%s", buffer_to_print) ;
   printf ("|--------------------------|-----------------|-----------|------------|\n") ;
}

