/* 
	File		:	cmnat.c
	Author		:	Ramana Murthy
	Date  		:	Feb 26th, 1998
	Synopsis	:	Contains configuration manager support functions:
					    insert        remove        edit
						get_first     get_next      get_previous
					for the following items relating to NAT:
						- Global Address List
						- Static NAT Mappings
						- Dynamic NAT Mappings
					These support functions are invoked from Telnet and
					Web modules with cm_insert, cm_remove, etc. calls.
*/

#include "defs.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <kstart.h>
#include <v8022str.h>
#include <vethstr.h>
#include <lslproto.h>
#include <socklib.h>
#include <stddef.h>
#include "\rtrware\store\boot.h"
#include "incall.h"
#include "cfgmgr.h"
#include <flashmgr.h>

/* ---- Constants  ---- */
#define	NUM_GLOBAL_ADDRESSES_PER_NODE		3

/* ---- Extern Declerations ---- */
extern PARAMETER_NODE
*match_the_lhs_string_in_the_linked_list_and_return_rhs_string
					(BYTE* lhs_string, PARAMETER_NODE* search_to_start_at, 
					PARAMETER_NODE* start_of_next_section, BYTE* rhs_string, 
					USHORT port_number, BYTE* port_number_matched);
/*extern void convert_to_lower_case (BYTE* the_string);*/
extern PARAMETER_NODE *insert_missing_string_without_port
					(BYTE section_number, BYTE* lhs_string, BYTE* rhs_string);
extern void update_count_in_ini (BYTE *lhs_string, BYTE flag,USHORT port_num);

extern USHORT get_next_application_port_which_needs_re_entry(USHORT first_port_number);
extern BYTE is_re_entry_needed_for_application_port(USHORT port_number);

/* ---- Local Prototype Declaration ---- */
BYTE search_for_global_address (BYTE *global_addr_entry,
								BYTE *bptr_lhs_string);
BYTE the_lastly_given_global_address_in_the_current_node(BYTE new_value, 
								BYTE is_to_save);
BYTE total_number_of_global_addresses_in_the_current_node(PARAMETER_NODE *node);
BYTE search_for_static_mapping (STRUCT_NAT_STATIC_ENTRY *nat_static_entry,
								BYTE *bptr_lhs_string);
void fill_static_mapping_struct_from_rhs (BYTE *rhs_str, 
								STRUCT_NAT_STATIC_ENTRY *nat_static_entry);
BYTE search_for_dynamic_mapping (STRUCT_NAT_DYNAMIC_ENTRY *nat_dynamic_entry,
								BYTE *bptr_lhs_string);
void fill_dynamic_mapping_struct_from_rhs (BYTE *rhs_str, 
								STRUCT_NAT_DYNAMIC_ENTRY *nat_dynamic_entry);

/* ---- Strings in INI file ---- */
BYTE global_address_list_in_ini[]    = "Proxy Server Global Address List";
BYTE mapped_address_list_in_ini[]    = "Proxy Server Mapped Address List";
BYTE local_internet_servers_in_ini[] = "Proxy Server Local Internet Servers";
BYTE num_global_addresses_in_ini[]  = "Proxy Server Number of Global Addresses";
BYTE num_static_mappings_in_ini[]   = "Proxy Server Number of Static Address";
BYTE num_dynamic_servers_in_ini[]   = "Proxy Server Number of Dynamic Servers";

/* ------------------------------------------------------------------------ */
/*          Functions relating to : NAT - GLOBAL ADDRESS LIST               */
/* ------------------------------------------------------------------------ */

PARAMETER_NODE
*insert_nat_global_address (enum CM_SECTIONS section_to_which_it_belongs_to, 
							BYTE *bptr_lhs_string, 
							void *vptr_rhs)
{
	BYTE			*nat_global_address,temp_entry[16];
	BYTE			rhs_string[100];
	PARAMETER_NODE	*node;
   
	nat_global_address = (BYTE *)vptr_rhs;

	/* Validation */
	if (! is_an_ip_address(nat_global_address))
	{
		printf("Not a Valid IP Address\n");
		return NULL;
	}

	/* Check if the given global address already exists */
	if (search_for_global_address (nat_global_address, bptr_lhs_string) == TRUE)
	{
		printf("Cannot add Duplicate NAT Global Address .......\n");
		return NULL;
	}

	/* 
	 * Check if a node having less than NUM_GLOBAL_ADDRESSES_PER_NODE is
	 * already present. If so, append the given global address to it;
	 * or else create a new node and write into it.
	*/
	node = get_first_nat_global_address (section_to_which_it_belongs_to, 
								bptr_lhs_string, (void*)temp_entry);

	while (node)
	{
		if (total_number_of_global_addresses_in_the_current_node(node) < 
											NUM_GLOBAL_ADDRESSES_PER_NODE)
		{
			/* Already a node is present....just append the address to it */
			node->parameter[strlen(node->parameter)-1] = 0; /* Remove '\n' */
			strcat(node->parameter, ",");
			strcat(node->parameter, nat_global_address);
			strcat(node->parameter, "\n");

         if (node != NULL) {
      		update_count_in_ini (num_global_addresses_in_ini, CM_INCREMENT_COUNT,CM_NO_PORT_PARAMETER_PRESENT);
	      }

			return node;

		}
		node = get_next_nat_global_address (section_to_which_it_belongs_to, 
											node, (void*)temp_entry);
	}
	/* Node not present - create a new one and insert the new global address */
	sprintf(rhs_string, "%s\0\n", nat_global_address);
	node = insert_missing_string_without_port(CM_PROXY_SECTION, 
							global_address_list_in_ini,  rhs_string);
	if (node != NULL) {
		update_count_in_ini (num_global_addresses_in_ini, CM_INCREMENT_COUNT,CM_NO_PORT_PARAMETER_PRESENT);
	}
	return (node);
}

enum CM_REMOVE_STRING_RESULTS
remove_nat_global_address (enum CM_SECTIONS section_to_which_it_belongs_to, 
							BYTE *bptr_lhs_string, 
							void *vptr_rhs)
{
	BYTE			*nat_global_address;
	BYTE			temp_entry[100], temp_buffer[100];
	BYTE			main_index, sub_index, saved_index;
	PARAMETER_NODE	*node;
	BYTE			entries_available_in_the_same_buffer = FALSE;

	nat_global_address = (BYTE *)vptr_rhs;
	strcpy (temp_entry, nat_global_address);

	if ((node = get_first_nat_global_address(section_to_which_it_belongs_to, 
							bptr_lhs_string, (void*)temp_entry)) == NULL)
	{
		printf("Get First Entry Failed to get the required string to remove\n");
		return CM_RHS_STRING_NOT_FOUND_WHILE_REMOVING;
	}

	do {
		main_index = 0;
		while (node->parameter[main_index++] != '=');	/* skip LHS, '=' */
		main_index++; 									/* skip space */

		do {
			sub_index = 0;
			saved_index = main_index;

			/* Read a global address into a buffer */
			while ( node->parameter[main_index] != ',' &&
					node->parameter[main_index] != '\0' && 
					node->parameter[main_index] != '\n')
				temp_buffer[sub_index++] = node->parameter[main_index++];
			temp_buffer[sub_index] = '\0';

			/* Check if the global addresses match */
			if (! strcmpi(temp_buffer, nat_global_address))
			{
				/* Blank out from saved_index to main_index */
				if ((node->parameter[main_index] == 0 || 
					node->parameter[main_index] == '\n') && 
					(total_number_of_global_addresses_in_the_current_node
												(node) == 1))
				{
					/* Free the node as there are no entries now */
					node->previous->next = node->next;
					node->next->previous = node->previous;
					free(node); 
					update_count_in_ini (num_global_addresses_in_ini,
											CM_DECREMENT_COUNT,CM_NO_PORT_PARAMETER_PRESENT);
					return CM_REMOVING_STRING_SUCCESSFUL;
				}

				/* There are some more entries in this node. So leave it */

				if (node->parameter[main_index] == ',')
					main_index++;
				else {
					/* Last entry in the node ! */
					node->parameter[saved_index-1] = '\n';
					node->parameter[saved_index] = 0;
					update_count_in_ini (num_global_addresses_in_ini,
											CM_DECREMENT_COUNT,CM_NO_PORT_PARAMETER_PRESENT);
					return CM_REMOVING_STRING_SUCCESSFUL;
				}
				while (node->parameter[main_index] != 0)
					node->parameter[saved_index++] = 
											node->parameter[main_index++];
	   			node->parameter[saved_index] = 0;

				update_count_in_ini (num_global_addresses_in_ini,
											CM_DECREMENT_COUNT,CM_NO_PORT_PARAMETER_PRESENT);
				return CM_REMOVING_STRING_SUCCESSFUL;
			}
			else
			{
				/* 
				 * The given entry does not match with the read entry.
				 * Continue reading if there are more entries in the node. 
				*/
				if (node->parameter[main_index] == 0)		
					entries_available_in_the_same_buffer = FALSE;
				else
				{
					main_index ++; 			/* Skip the ',' */
					entries_available_in_the_same_buffer = TRUE;
				}
			}
		}
		while (entries_available_in_the_same_buffer);

		/* Read the next line/node and repeat the check for matching entries */
		node = get_next_nat_global_address (CM_PROXY_SECTION, 
											node, (void*)&temp_entry);
	} while (node);

	printf("Failed to search node for removing entry \n");
	return CM_RHS_STRING_NOT_FOUND_WHILE_REMOVING;
}

enum CM_EDIT_STRING_RESULTS 
edit_nat_global_address (enum CM_SECTIONS section_to_which_it_belongs_to, 
							BYTE *bptr_lhs_string, 
							void *vptr_old_rhs,
							void *vptr_new_rhs)
{
	enum CM_REMOVE_STRING_RESULTS delete_result;
	BYTE			*old_nat_global_address;
	BYTE			*new_nat_global_address;

	old_nat_global_address = (BYTE *)vptr_old_rhs;
	new_nat_global_address = (BYTE *)vptr_new_rhs;

	/* Validation */
	if (! is_an_ip_address(new_nat_global_address))
	{
		printf("Not a Valid IP Address\n");
		return CM_EDITING_FAILED_ON_VALIDATION;
	}

	/* Check if the specified old NAT global address exists */
	if (search_for_global_address (old_nat_global_address,
												bptr_lhs_string) == FALSE)
	{
		printf("The given NAT Global Address does not exist.......\n");
		return CM_EDITING_FAILED_ON_VALIDATION;
	}

	/* Check if the specified new static NAT mapping already exists */
	if (search_for_global_address (new_nat_global_address,
												bptr_lhs_string) == TRUE)
	{
		printf("The modified NAT Global Address is a duplicate entry.......\n");
		return CM_EDITING_FAILED_ON_VALIDATION;
	}

	/* Remove the old global address and insert the new one */
	delete_result = remove_nat_global_address(section_to_which_it_belongs_to,
												bptr_lhs_string, vptr_old_rhs);
	if (delete_result != CM_REMOVING_STRING_SUCCESSFUL)
		return CM_RHS_STRING_NOT_FOUND_WHILE_EDIT;

	if (insert_nat_global_address(section_to_which_it_belongs_to, 
								bptr_lhs_string, vptr_new_rhs) == NULL)
		return CM_RHS_STRING_NOT_FOUND_WHILE_EDIT;

	return CM_STRING_EDIT_SUCCESSFUL; 
}


void fill_the_buffer_with_nat_global_address (BYTE *address_ptr, BYTE *buffer)        
{
	BYTE j = 0, k = 0;

	while (buffer[j] != '=') ++j;
	j += 2;/* Skip '=' sign  and Space */

	while ( buffer[j] != ',' && 
			buffer[j] != '\n' && 
			buffer[j] != '\r' && 
			buffer[j] != '\0')
		address_ptr[k++] = buffer[j++];   
   
   address_ptr[k] = 0;
}

PARAMETER_NODE
*get_first_nat_global_address(enum CM_SECTIONS section_to_which_it_belongs_to, 
							  BYTE *bptr_lhs_string, 
							  void *vptr_rhs)
{
	PARAMETER_NODE	*search_to_start_at;
	BYTE			port_number = CM_NO_PORT_PARAMETER_PRESENT, 
					port_number_matched, buffer[200];
	BYTE			*ptr_to_global_address;

	search_to_start_at = sptr_to_each_section_in_the_configuration
											[section_to_which_it_belongs_to];
   
	search_to_start_at = 
		match_the_lhs_string_in_the_linked_list_and_return_rhs_string(
				bptr_lhs_string, search_to_start_at, 
				sptr_to_each_section_in_the_configuration
											[section_to_which_it_belongs_to+1],
				buffer, port_number, &port_number_matched);
   
	if (search_to_start_at == NULL)
		return NULL;	

	if (vptr_rhs == NULL)
	   return search_to_start_at;

	ptr_to_global_address = (BYTE *) vptr_rhs;
      
	fill_the_buffer_with_nat_global_address (ptr_to_global_address, 
											search_to_start_at->parameter);
	the_lastly_given_global_address_in_the_current_node(1, TRUE);
	return search_to_start_at;
}

PARAMETER_NODE
*get_next_nat_global_address (enum CM_SECTIONS section_to_which_it_belongs_to,
							PARAMETER_NODE *current_node, 
							void *vptr_rhs)
{

	BYTE	i = 0, *nat_global_address, prev_count, total,main_index, sub_index;
	BYTE	port_number = CM_NO_PORT_PARAMETER_PRESENT, port_number_matched,
			buffer[200];
	PARAMETER_NODE	*search_to_start_at;

	if (!current_node) return NULL;
   
	nat_global_address = (BYTE *)vptr_rhs;
   
	prev_count=the_lastly_given_global_address_in_the_current_node(0, FALSE);
	total = total_number_of_global_addresses_in_the_current_node(current_node);
	if (prev_count < total)
	{
		/* 
		 * One more entry can be read from the current node itself.
		 * Skip first prev_count entries.
		*/
		main_index = 0;
		for (i = 0; i < prev_count; i++) 
			while (current_node->parameter[main_index++] != ','); 
   
		sub_index = 0;
		while (	current_node->parameter[main_index] != ',' && 
				current_node->parameter[main_index] != '\0' && 
				current_node->parameter[main_index] != '\n')
			nat_global_address[sub_index++] = 
									current_node->parameter[main_index++];
		nat_global_address[sub_index] = 0;
     
		the_lastly_given_global_address_in_the_current_node(prev_count+1, TRUE);
		return current_node;
	}   

	search_to_start_at = current_node;
	search_to_start_at = 
		match_the_lhs_string_in_the_linked_list_and_return_rhs_string(
				global_address_list_in_ini, search_to_start_at,
				sptr_to_each_section_in_the_configuration
											[section_to_which_it_belongs_to+1], 
				buffer, port_number, &port_number_matched);
	if (search_to_start_at == NULL)
		return NULL;	
   
	if (vptr_rhs == NULL)
	   return search_to_start_at;
   
	fill_the_buffer_with_nat_global_address (nat_global_address,
											search_to_start_at->parameter);

	the_lastly_given_global_address_in_the_current_node(1, TRUE);
	return search_to_start_at;
}

PARAMETER_NODE *get_previous_nat_global_address 
							(enum CM_SECTIONS section_to_which_it_belongs_to,
							PARAMETER_NODE *current_node, 
							void *vptr_rhs)
{
	return NULL;
}

BYTE 
search_for_global_address (BYTE *nat_global_address, BYTE *bptr_lhs_string)
{
	BYTE			temp_entry[100], temp_buffer[100];
	BYTE			main_index, sub_index;
	PARAMETER_NODE	*node;
	BYTE			entries_available_in_the_same_buffer = FALSE;

	if ((node = get_first_nat_global_address(CM_PROXY_SECTION, 
						bptr_lhs_string, (void *)temp_entry)) == NULL)
		return FALSE;

	do {
		main_index = 0;
		while (node->parameter[main_index++] != '=');	/* skip LHS, '=' */
		main_index++; 									/* skip space */

		do {
			sub_index = 0;

			/* Read a global address into a buffer */
			while ( node->parameter[main_index] != ',' &&
					node->parameter[main_index] != '\0' && 
					node->parameter[main_index] != '\n')
				temp_buffer[sub_index++] = node->parameter[main_index++];
			temp_buffer[sub_index] = '\0';

			/* Check if the global addresses match */
			if (! strcmpi(temp_buffer, nat_global_address))
            	return TRUE;
			else
			{
				/* 
				 * The given entry does not match with the read entry.
				 * Continue reading if there are more entries in the node. 
				*/
				if (node->parameter[main_index] == 0)		
					entries_available_in_the_same_buffer = FALSE;
				else
				{
					main_index ++; 			/* Skip the ',' */
					entries_available_in_the_same_buffer = TRUE;
				}
			}
		}
		while (entries_available_in_the_same_buffer);

		/* Read the next line/node and repeat the check for matching entries */
		node = get_next_nat_global_address (CM_PROXY_SECTION, 
											node, (void*)&temp_entry);
	}
	while (node);
	return FALSE;
}

BYTE the_lastly_given_global_address_in_the_current_node (BYTE new_value,
															BYTE is_to_save)
{
	static BYTE global_addr_value;

	if (is_to_save)
		global_addr_value = new_value;
	else
		return global_addr_value;
}

BYTE total_number_of_global_addresses_in_the_current_node(PARAMETER_NODE *node)
{
	BYTE	i, num_entries,len = 0;

	num_entries = 0;
	len = strlen(node->parameter);
	for (i = 0; i < len; i++) {
		if (node->parameter[i] == ',')
			num_entries++;
	}
	return (num_entries+1);
}	

/* ------------------------------------------------------------------------ */
/*          Functions relating to : NAT - STATIC MAPPING                    */
/* ------------------------------------------------------------------------ */

PARAMETER_NODE
*insert_nat_static_mapping (enum CM_SECTIONS section_to_which_it_belongs_to, 
							BYTE *bptr_lhs_string, 
							void *vptr_rhs)
{
	STRUCT_NAT_STATIC_ENTRY	*nat_static_entry;
	BYTE			rhs_string[100];
	PARAMETER_NODE	*node;
   
	nat_static_entry = (STRUCT_NAT_STATIC_ENTRY *)vptr_rhs;

	/* Validation */
	if ((! is_an_ip_address(nat_static_entry->global_addr)) ||
		(! is_an_ip_address(nat_static_entry->local_addr)))
	{
		printf("Not a Valid IP Address\n");
		return NULL;
	}

	/* Check if the given static mapping already exists */
	if (search_for_static_mapping (nat_static_entry, bptr_lhs_string) == TRUE)
	{
		printf("Cannot add Duplicate NAT Static Mapping Entry .......\n");
		return NULL;
	}

	/* static mapping not present - create a new one */
	sprintf(rhs_string, "%s,%s\0", 
			nat_static_entry->global_addr, nat_static_entry->local_addr);
	node = insert_missing_string_without_port(CM_PROXY_SECTION, 
						mapped_address_list_in_ini,  rhs_string);
	
	if (node != NULL) {
		update_count_in_ini (num_static_mappings_in_ini, CM_INCREMENT_COUNT,
                        CM_NO_PORT_PARAMETER_PRESENT);
	}
	return (node);
}

enum CM_REMOVE_STRING_RESULTS
remove_nat_static_mapping (enum CM_SECTIONS section_to_which_it_belongs_to, 
							BYTE *bptr_lhs_string, 
							void *vptr_rhs)
{
	STRUCT_NAT_STATIC_ENTRY	*nat_static_entry;
	BYTE			buffer[200], string_ptr[200];
	BYTE			port_number = CM_NO_PORT_PARAMETER_PRESENT;
	BYTE			port_number_matched;
	PARAMETER_NODE	*search_to_start_at;

	nat_static_entry = (STRUCT_NAT_STATIC_ENTRY *)vptr_rhs;
	sprintf(buffer, "%s,%s", 
			nat_static_entry->global_addr, nat_static_entry->local_addr);

	search_to_start_at = sptr_to_each_section_in_the_configuration
											[section_to_which_it_belongs_to];
	do {
		search_to_start_at = 
			match_the_lhs_string_in_the_linked_list_and_return_rhs_string(
				mapped_address_list_in_ini, search_to_start_at,
				sptr_to_each_section_in_the_configuration
										[section_to_which_it_belongs_to+1],
				string_ptr, port_number, &port_number_matched);

		if (search_to_start_at == NULL)
			return CM_RHS_STRING_NOT_FOUND_WHILE_REMOVING;	
	} while(strcmpi(buffer, string_ptr) != 0);

	search_to_start_at->previous->next = search_to_start_at->next;
	search_to_start_at->next->previous = search_to_start_at->previous;
	free(search_to_start_at);
	update_count_in_ini (num_static_mappings_in_ini, CM_DECREMENT_COUNT,
         CM_NO_PORT_PARAMETER_PRESENT);
	return CM_REMOVING_STRING_SUCCESSFUL;
}

enum CM_EDIT_STRING_RESULTS 
edit_nat_static_mapping (enum CM_SECTIONS section_to_which_it_belongs_to, 
							BYTE *bptr_lhs_string, 
							void *vptr_old_rhs,
							void *vptr_new_rhs)
{
	STRUCT_NAT_STATIC_ENTRY	*old_nat_entry, *new_nat_entry;
	BYTE			buffer[200], string_ptr[200];
	BYTE			port_number = CM_NO_PORT_PARAMETER_PRESENT;
	BYTE			port_number_matched;
	PARAMETER_NODE	*search_to_start_at;
   
	old_nat_entry = (STRUCT_NAT_STATIC_ENTRY *)vptr_old_rhs;
	new_nat_entry = (STRUCT_NAT_STATIC_ENTRY *)vptr_new_rhs;

	/* Validate the new static NAT mapping specified */
	if ((! is_an_ip_address(new_nat_entry->global_addr)) ||
		(! is_an_ip_address(new_nat_entry->local_addr)))
	{
		printf("Not a Valid IP Address\n");
		return CM_EDITING_FAILED_ON_VALIDATION;
	}

	/* Check if the specified old static NAT mapping exists */
	if (search_for_static_mapping (old_nat_entry, bptr_lhs_string) == FALSE)
	{
		printf("The given Static NAT Mapping does not exist.......\n");
		return CM_EDITING_FAILED_ON_VALIDATION;
	}

	/* Check if the specified new static NAT mapping already exists */
	if (search_for_static_mapping (new_nat_entry, bptr_lhs_string) == TRUE)
	{
		printf("The new Static NAT Mapping already exists........\n");
		return CM_EDITING_FAILED_ON_VALIDATION;
	}

	/* Search the linked list for the matching entry */
	sprintf(buffer, "%s,%s", 
			old_nat_entry->global_addr, old_nat_entry->local_addr);

   search_to_start_at = sptr_to_each_section_in_the_configuration
											[section_to_which_it_belongs_to];

	do {
		search_to_start_at = 
			match_the_lhs_string_in_the_linked_list_and_return_rhs_string(
				mapped_address_list_in_ini, search_to_start_at,
				sptr_to_each_section_in_the_configuration
										[section_to_which_it_belongs_to+1],
				string_ptr, port_number, &port_number_matched);

		if (search_to_start_at == NULL)	/* Unlikely, as search had found ! */
			return CM_RHS_STRING_NOT_FOUND_WHILE_EDIT;	

	} while(strcmpi(buffer, string_ptr) != 0);

	strcpy(search_to_start_at->parameter,mapped_address_list_in_ini );
	strcat(search_to_start_at->parameter, " = ");
	strcat(search_to_start_at->parameter,new_nat_entry->global_addr);
	strcat(search_to_start_at->parameter,",");
	strcat(search_to_start_at->parameter,new_nat_entry->local_addr);
	strcat(search_to_start_at->parameter, "\n");

	return CM_STRING_EDIT_SUCCESSFUL;
}

PARAMETER_NODE
*get_first_nat_static_mapping(enum CM_SECTIONS section_to_which_it_belongs_to, 
							  BYTE *bptr_lhs_string, 
							  void *vptr_rhs)
{
	STRUCT_NAT_STATIC_ENTRY	*nat_static_entry;
	BYTE			buffer[200], temp_buffer[200];
	BYTE			port_number = CM_NO_PORT_PARAMETER_PRESENT;
	BYTE			port_number_matched, i, j;
	PARAMETER_NODE	*search_to_start_at;

	nat_static_entry = (STRUCT_NAT_STATIC_ENTRY *)vptr_rhs;

	search_to_start_at = sptr_to_each_section_in_the_configuration
											[section_to_which_it_belongs_to];

	search_to_start_at = 
		match_the_lhs_string_in_the_linked_list_and_return_rhs_string(
				mapped_address_list_in_ini, search_to_start_at, 
				sptr_to_each_section_in_the_configuration
											[section_to_which_it_belongs_to+1],
				buffer, port_number, &port_number_matched);

	if (search_to_start_at == NULL)
   		return NULL;	

	/* copy the first part of the mapping till the comma into global addr */
	i = j = 0;
  	while (buffer[i] != ',' && buffer[i] !=  '\0' && buffer[i] !=  '\n')
		temp_buffer[j++] = buffer[i++];
	temp_buffer[j] = '\0';
	strcpy (nat_static_entry->global_addr, temp_buffer);

	/* copy the second part of the mapping from the comma into local addr */
	j = 0;	i++;
  	while (buffer[i] != ',' && buffer[i] !=  '\0' && buffer[i] !=  '\n')
		temp_buffer[j++] = buffer[i++];
	temp_buffer[j] = '\0';
	strcpy (nat_static_entry->local_addr, temp_buffer);

	return search_to_start_at;
}

PARAMETER_NODE
*get_next_nat_static_mapping (enum CM_SECTIONS section_to_which_it_belongs_to,
							PARAMETER_NODE *current_node, 
							void *vptr_rhs)
{
	STRUCT_NAT_STATIC_ENTRY	*nat_static_entry;
	BYTE			current_node_lhs_string[100], lhs_string[100];
	BYTE			i;
	PARAMETER_NODE	*current_temp_node, *next_node;

	if (! current_node)
		return NULL;

	i = 0;
	while (current_node->parameter[i] != '=')
	{
		lhs_string[i] = current_node->parameter[i];
		i++;
	}
	lhs_string[i-1] = 0;

	current_temp_node = current_node;
	while 
		(
		current_temp_node->next && 
		current_temp_node->next != sptr_to_each_section_in_the_configuration
											[section_to_which_it_belongs_to+1]
		)
	{
		next_node = current_temp_node->next;
		i = 0;
		while (next_node->parameter[i] != '=')
		{
			current_node_lhs_string[i] = next_node->parameter[i];
			i++;
		}
		current_node_lhs_string[i-1] = 0;
/*		convert_to_lower_case(current_node_lhs_string);*/
/*		convert_to_lower_case(lhs_string);*/
		if (strcmpi(lhs_string, current_node_lhs_string) == 0)
		{
			if (vptr_rhs == NULL)
				return next_node;
			nat_static_entry = (STRUCT_NAT_STATIC_ENTRY *)vptr_rhs;

			/* fill static mapping structure with rhs in parameter node */
			fill_static_mapping_struct_from_rhs (next_node->parameter,
													nat_static_entry);
			return next_node;
		}
		current_temp_node = current_temp_node->next;
	}
	return NULL;
}

PARAMETER_NODE
*get_previous_nat_static_mapping (
							enum CM_SECTIONS section_to_which_it_belongs_to,
							PARAMETER_NODE *current_node,
							void *vptr_rhs)
{
	STRUCT_NAT_STATIC_ENTRY	*nat_static_entry;
	BYTE			current_node_lhs_string[100], lhs_string[100];
	BYTE			i;
	PARAMETER_NODE	*current_temp_node, *previous_node;

	if (! current_node)
		return NULL;

	i = 0;
	while (current_node->parameter[i] != '=')
	{
		lhs_string[i] = current_node->parameter[i];
		i++;
	}
	lhs_string[i-1] = 0;

	current_temp_node = current_node;
	while
		(
		current_temp_node != sptr_to_each_section_in_the_configuration[0] &&
		current_temp_node->previous != sptr_to_each_section_in_the_configuration
										[section_to_which_it_belongs_to - 1]
		)
	{
		previous_node = current_temp_node->previous;
		i = 0;
		while (previous_node->parameter[i] != '=')
		{
			current_node_lhs_string[i] = previous_node->parameter[i];
			i++;
		}
		current_node_lhs_string[i-1] = 0;
/*		convert_to_lower_case(current_node_lhs_string);*/
/*		convert_to_lower_case(lhs_string);*/
		if (strcmpi(lhs_string, current_node_lhs_string) == 0)
		{
			if (vptr_rhs == NULL)
				return previous_node;

			nat_static_entry = (STRUCT_NAT_STATIC_ENTRY *)vptr_rhs;

			/* fill static mapping structure with rhs in parameter node */
			fill_static_mapping_struct_from_rhs (previous_node->parameter,
													nat_static_entry);
			return previous_node;
		}
		current_temp_node = current_temp_node->previous;
	}
	return NULL;
}

BYTE 
search_for_static_mapping (	STRUCT_NAT_STATIC_ENTRY *nat_static_entry,
							BYTE *bptr_lhs_string)
{
	STRUCT_NAT_STATIC_ENTRY		temp_entry;
	BYTE						index,sub_index,
								temp_buffer1[100], temp_buffer2[100];
	PARAMETER_NODE				*node;

	sprintf(temp_buffer2, "%s,%s\0", 
			nat_static_entry->global_addr, nat_static_entry->local_addr);

	if ((node = get_first_nat_static_mapping (CM_PROXY_SECTION, 
							bptr_lhs_string, (void*)&temp_entry)) == NULL)
		return FALSE;

	do {
      sub_index = 0;
		index = 0;
		while (node->parameter[index++] != '=');/* skip  LHS string , '=' */
      index++; /* Skip space */


		while ( node->parameter[index] != '\0' && 
				node->parameter[index] != '\n')
			temp_buffer1[sub_index++] = node->parameter[index++];

		temp_buffer1[sub_index] = '\0';

		/* Are the entries matching */
		if (!strcmpi(temp_buffer1, temp_buffer2))
            return TRUE;

		/* Try to read a next line/node that matches */
		node = get_next_nat_static_mapping (CM_PROXY_SECTION, 
											node, (void*)&temp_entry);
	}
	while (node);
	return FALSE;
}

void
fill_static_mapping_struct_from_rhs (BYTE *rhs_str, 
						STRUCT_NAT_STATIC_ENTRY *nat_static_entry)
{
	BYTE	*sptr, temp_buffer[100], j;

	sptr = rhs_str;
	while (*sptr && *sptr != '=') sptr++;	/* skip till '=' */
	sptr++;									/* skip the '=' */
	while (*sptr == ' ') sptr++;			/* skip spaces */

	/* copy the first part of the mapping into global address */
	j = 0;
	while (*sptr && *sptr != ',' && *sptr != '\0' && *sptr != '\n')
		temp_buffer[j++] = *sptr++;
	temp_buffer[j] = '\0';
	strcpy (nat_static_entry->global_addr, temp_buffer);
	
	/* copy the second part of the mapping into local address */
	j = 0;	sptr++;
	while (*sptr && *sptr != ',' && *sptr != '\n')
		temp_buffer[j++] = *sptr++;
	temp_buffer[j] = '\0';
	strcpy (nat_static_entry->local_addr, temp_buffer);
}

/* ------------------------------------------------------------------------ */
/*          Functions relating to : NAT - DYNAMIC MAPPING                   */
/* ------------------------------------------------------------------------ */

PARAMETER_NODE
*insert_nat_dynamic_mapping (enum CM_SECTIONS section_to_which_it_belongs_to, 
							BYTE *bptr_lhs_string, 
				  			void *vptr_rhs)
{
	STRUCT_NAT_DYNAMIC_ENTRY	*nat_dynamic_entry;
	BYTE			rhs_string[100];
	PARAMETER_NODE	*node;
	static BYTE		re_entry = 0;
   
	nat_dynamic_entry = (STRUCT_NAT_DYNAMIC_ENTRY *)vptr_rhs;

	/* For FTP(20,21), SSL-FTP(989,990), Q931(900,902), RTP(5004,5006),
		RTCP(5005,5007) ports */

	if ((is_re_entry_needed_for_application_port(nat_dynamic_entry->port) == TRUE)
		&& (re_entry == 0)) 
	{
		re_entry = 1;
		nat_dynamic_entry->port = get_next_application_port_which_needs_re_entry(nat_dynamic_entry->port);
		
		insert_nat_dynamic_mapping (section_to_which_it_belongs_to,
									bptr_lhs_string, vptr_rhs);
		nat_dynamic_entry->port = get_next_application_port_which_needs_re_entry(nat_dynamic_entry->port);
		re_entry = 0;
	}

	/* Validation */
	if (! is_an_ip_address(nat_dynamic_entry->local_addr))
	{
		printf("Not a Valid IP Address\n");
		return NULL;
	}

	/* Check if the given dynamic mapping already exists */
	if (search_for_dynamic_mapping (nat_dynamic_entry, bptr_lhs_string) == TRUE)
	{
		printf("Cannot add Duplicate NAT Dynamic Mapping Entry .......\n");
		return NULL;
	}

	/* dynamic mapping not present - create a new one */
	sprintf(rhs_string, "%s,%04X,%04X\0", 
			nat_dynamic_entry->local_addr, 
			(! nat_dynamic_entry->protocol ? 0x0006 : 0x0011), 
			nat_dynamic_entry->port);
	node = insert_missing_string_without_port(CM_PROXY_SECTION, 
						local_internet_servers_in_ini,  rhs_string);
	if (node != NULL) {
		update_count_in_ini (num_dynamic_servers_in_ini, CM_INCREMENT_COUNT,
            CM_NO_PORT_PARAMETER_PRESENT);
	}
	return (node);
}

enum CM_REMOVE_STRING_RESULTS
remove_nat_dynamic_mapping (enum CM_SECTIONS section_to_which_it_belongs_to, 
							BYTE *bptr_lhs_string, 
							void *vptr_rhs)
{
	STRUCT_NAT_DYNAMIC_ENTRY	*nat_dynamic_entry;
	BYTE			buffer[200], string_ptr[200];
	BYTE			port_number = CM_NO_PORT_PARAMETER_PRESENT;
	BYTE			port_number_matched;
	PARAMETER_NODE	*search_to_start_at;
	static BYTE		re_entry = 0;

	nat_dynamic_entry = (STRUCT_NAT_DYNAMIC_ENTRY *)vptr_rhs;

	/* For FTP(20,21), SSL-FTP(989,990), Q931(900,902), RTP(5004,5006),
		RTCP(5005,5007) ports */

	if ((is_re_entry_needed_for_application_port(nat_dynamic_entry->port) == TRUE)
		&& (re_entry == 0)) 
	{
		re_entry = 1;
		nat_dynamic_entry->port = get_next_application_port_which_needs_re_entry(nat_dynamic_entry->port);
		
		remove_nat_dynamic_mapping (section_to_which_it_belongs_to,
									bptr_lhs_string, vptr_rhs);
		nat_dynamic_entry->port = get_next_application_port_which_needs_re_entry(nat_dynamic_entry->port);
		re_entry = 0;
	}

	sprintf(buffer, "%s,%04X,%04X\0", 
			nat_dynamic_entry->local_addr, 
			(! nat_dynamic_entry->protocol ? 0x0006 : 0x0011), 
			nat_dynamic_entry->port);
   
	search_to_start_at = sptr_to_each_section_in_the_configuration
											[section_to_which_it_belongs_to];

	do {
		search_to_start_at = 
			match_the_lhs_string_in_the_linked_list_and_return_rhs_string(
				local_internet_servers_in_ini, search_to_start_at,
				sptr_to_each_section_in_the_configuration
										[section_to_which_it_belongs_to+1],
				string_ptr, port_number, &port_number_matched);

		if (search_to_start_at == NULL)
			return CM_RHS_STRING_NOT_FOUND_WHILE_REMOVING;	
	} while(strcmpi(buffer, string_ptr) != 0);

	search_to_start_at->previous->next = search_to_start_at->next;
	search_to_start_at->next->previous = search_to_start_at->previous;
	free(search_to_start_at);
	update_count_in_ini (num_dynamic_servers_in_ini, CM_DECREMENT_COUNT,
         CM_NO_PORT_PARAMETER_PRESENT);
	return CM_REMOVING_STRING_SUCCESSFUL;
}

enum CM_EDIT_STRING_RESULTS 
edit_nat_dynamic_mapping (enum CM_SECTIONS section_to_which_it_belongs_to, 
							BYTE *bptr_lhs_string, 
							void *vptr_old_rhs,
							void *vptr_new_rhs)
{
	STRUCT_NAT_DYNAMIC_ENTRY	*old_nat_entry, *new_nat_entry;
	BYTE			buffer[200], string_ptr[200],rhs_string[100];
	BYTE			port_number = CM_NO_PORT_PARAMETER_PRESENT;
	BYTE			port_number_matched;
	PARAMETER_NODE	*search_to_start_at;
	enum CM_REMOVE_STRING_RESULTS delete_result;

	old_nat_entry = (STRUCT_NAT_DYNAMIC_ENTRY *)vptr_old_rhs;
	new_nat_entry = (STRUCT_NAT_DYNAMIC_ENTRY *)vptr_new_rhs;

	/* Validate the new dynamic NAT mapping specified */
	if (! is_an_ip_address(new_nat_entry->local_addr))
	{
		printf("Not a Valid IP Address\n");
		return CM_EDITING_FAILED_ON_VALIDATION;
	}

	/* Check if the specified old dynamic NAT mapping exists */
	if (search_for_dynamic_mapping (old_nat_entry, bptr_lhs_string) == FALSE)
	{
		printf("The given Dynamic NAT Mapping does not exist.......\n");
		return CM_EDITING_FAILED_ON_VALIDATION;
	}

	/* Check if the specified new dynamic NAT mapping already exists */
	if (search_for_dynamic_mapping (new_nat_entry, bptr_lhs_string) == TRUE)
	{
		printf("The new Dynamic NAT Mapping already exists........\n");
		return CM_EDITING_FAILED_ON_VALIDATION;
	}

	delete_result = 
		remove_nat_dynamic_mapping(section_to_which_it_belongs_to, 
											bptr_lhs_string, vptr_old_rhs);
	if (delete_result != CM_REMOVING_STRING_SUCCESSFUL)
		return CM_RHS_STRING_NOT_FOUND_WHILE_EDIT;

	if (insert_nat_dynamic_mapping(section_to_which_it_belongs_to, 
									bptr_lhs_string, vptr_new_rhs) == NULL)
		return CM_RHS_STRING_NOT_FOUND_WHILE_EDIT;

	return CM_STRING_EDIT_SUCCESSFUL; 
}

PARAMETER_NODE
*get_first_nat_dynamic_mapping (enum CM_SECTIONS section_to_which_it_belongs_to,
							BYTE *bptr_lhs_string, 
							void *vptr_rhs)
{
	STRUCT_NAT_DYNAMIC_ENTRY	*nat_dynamic_entry;
	BYTE			buffer[200], temp_buffer[200];
	BYTE			port_number = CM_NO_PORT_PARAMETER_PRESENT;
	BYTE			port_number_matched, i, j;
	PARAMETER_NODE	*search_to_start_at;
	ULONG			application_port=0;

	nat_dynamic_entry = (STRUCT_NAT_DYNAMIC_ENTRY *)vptr_rhs;
   
	search_to_start_at = sptr_to_each_section_in_the_configuration
											[section_to_which_it_belongs_to];

	search_to_start_at = 
		match_the_lhs_string_in_the_linked_list_and_return_rhs_string(
				local_internet_servers_in_ini, search_to_start_at, NULL, 
				buffer, port_number, &port_number_matched);

	if (search_to_start_at == NULL)
   		return NULL;	

	/* copy the first part of the mapping till the comma into local addr */
	i = j = 0;
  	while (buffer[i] != ',' && buffer[i] !=  '\0' && buffer[i] !=  '\n')
		temp_buffer[j++] = buffer[i++];
	temp_buffer[j] = '\0';
	strcpy (nat_dynamic_entry->local_addr, temp_buffer);

	/* copy the second part of the mapping till the next comma into protocol */
	j = 0;	i++;
  	while (buffer[i] != ',' && buffer[i] !=  '\0' && buffer[i] !=  '\n')
		temp_buffer[j++] = buffer[i++];
	temp_buffer[j] = '\0';

	/* Sudha 05 Mar 1998 */
	if (!strcmpi(temp_buffer, "0006"))
	{
		nat_dynamic_entry->protocol = 0x0006;
	}
	else
	{
		nat_dynamic_entry->protocol = 0x0011;
	}
	/* Sudha 05 Mar 1998 */

	/* copy the third part of the mapping from the comma into port */
	j = 0;	i++;
  	while (buffer[i] != ',' && buffer[i] !=  '\0' && buffer[i] !=  '\n')
		temp_buffer[j++] = buffer[i++];
	temp_buffer[j] = '\0';
	sscanf (temp_buffer, "%x", &application_port);
	nat_dynamic_entry->port = (USHORT)application_port;

	return search_to_start_at;
}

PARAMETER_NODE
*get_next_nat_dynamic_mapping (enum CM_SECTIONS section_to_which_it_belongs_to,
							PARAMETER_NODE *current_node, 
							void* vptr_rhs)
{
	STRUCT_NAT_DYNAMIC_ENTRY	*nat_dynamic_entry;
	BYTE			current_node_lhs_string[100], lhs_string[100];
	BYTE			i;
	PARAMETER_NODE	*current_temp_node, *next_node;

	if (! current_node)
		return NULL;

	i = 0;
	while (current_node->parameter[i] != '=')
	{
		lhs_string[i] = current_node->parameter[i];
		i++;
	}
	lhs_string[i-1] = 0;

	current_temp_node = current_node;
	while 
		(
		current_temp_node->next && 
		current_temp_node->next != sptr_to_each_section_in_the_configuration
											[section_to_which_it_belongs_to+1]
		)
	{
		next_node = current_temp_node->next;
		i = 0;
		while (next_node->parameter[i] != '=')
		{
			current_node_lhs_string[i] = next_node->parameter[i];
			i++;
		}
		current_node_lhs_string[i-1] = 0;
/*		convert_to_lower_case(current_node_lhs_string);*/
/*		convert_to_lower_case(lhs_string);*/
		if (strcmpi(lhs_string, current_node_lhs_string) == 0)
		{
			if (vptr_rhs == NULL)
				return next_node;
			nat_dynamic_entry = (STRUCT_NAT_DYNAMIC_ENTRY *)vptr_rhs;

			/* fill dynamic mapping structure with rhs in parameter node */
			fill_dynamic_mapping_struct_from_rhs (next_node->parameter,
													nat_dynamic_entry);
			return next_node;
		}
		current_temp_node = current_temp_node->next;
	}
	return NULL;
}

PARAMETER_NODE
*get_previous_nat_dynamic_mapping (
							enum CM_SECTIONS section_to_which_it_belongs_to,
							PARAMETER_NODE *current_node,
							void *vptr_rhs)
{
	STRUCT_NAT_DYNAMIC_ENTRY	*nat_dynamic_entry;
	BYTE			current_node_lhs_string[100], lhs_string[100];
	BYTE			i;
	PARAMETER_NODE	*current_temp_node, *previous_node;

	if (! current_node)
		return NULL;

	i = 0;
	while (current_node->parameter[i] != '=')
	{
		lhs_string[i] = current_node->parameter[i];
		i++;
	}
	lhs_string[i-1] = 0;

	current_temp_node = current_node;
	while
		(
		current_temp_node != sptr_to_each_section_in_the_configuration[0] &&
		current_temp_node->previous != sptr_to_each_section_in_the_configuration
										[section_to_which_it_belongs_to - 1]
		)
	{
		previous_node = current_temp_node->previous;
		i = 0;
		while (previous_node->parameter[i] != '=')
		{
			current_node_lhs_string[i] = previous_node->parameter[i];
			i++;
		}
		current_node_lhs_string[i-1] = 0;
/*		convert_to_lower_case(current_node_lhs_string);*/
/*		convert_to_lower_case(lhs_string);*/
		if (strcmpi(lhs_string, current_node_lhs_string) == 0)
		{
			if (vptr_rhs == NULL)
				return previous_node;

			nat_dynamic_entry = (STRUCT_NAT_DYNAMIC_ENTRY *)vptr_rhs;

			/* fill dynamic mapping structure with rhs in parameter node */
			fill_dynamic_mapping_struct_from_rhs (previous_node->parameter,
													nat_dynamic_entry);
			return previous_node;
		}
		current_temp_node = current_temp_node->previous;
	}
	return NULL;
}

BYTE 
search_for_dynamic_mapping (STRUCT_NAT_DYNAMIC_ENTRY *nat_dynamic_entry,
							BYTE *bptr_lhs_string)
{
	STRUCT_NAT_DYNAMIC_ENTRY	temp_entry;
	BYTE						index,sub_index,
								temp_buffer1[100], temp_buffer2[100];
	PARAMETER_NODE				*node;

	sprintf(temp_buffer2, "%s,%04X,%04X\0", 
			nat_dynamic_entry->local_addr, 
			(! nat_dynamic_entry->protocol ? 0x0006 : 0x0011), 
			nat_dynamic_entry->port);

	if ((node = get_first_nat_dynamic_mapping (CM_PROXY_SECTION, 
							bptr_lhs_string, (void*)&temp_entry)) == NULL)
		return FALSE;

	do {
		index = 0;
		sub_index = 0;
		while (node->parameter[index++] != '=');/* skip  LHS string , '=' */
		index++; 								/* skip space */

		while (	node->parameter[index] != '\0' && 
				node->parameter[index] != '\n')
			temp_buffer1[sub_index++] = node->parameter[index++];

		temp_buffer1[sub_index] = '\0';

		/* Are the entries matching */
		if (!strcmpi(temp_buffer1, temp_buffer2))
            return TRUE;

		/* Try to read a next line/node that matches */
		node = get_next_nat_dynamic_mapping (CM_PROXY_SECTION, 
											node, (void*)&temp_entry);
	}
	while (node);
	return FALSE;
}

void
fill_dynamic_mapping_struct_from_rhs (BYTE *rhs_str, 
						STRUCT_NAT_DYNAMIC_ENTRY *nat_dynamic_entry)
{
	BYTE	*sptr, temp_buffer[100], j;
	ULONG	application_port = 0;

	sptr = rhs_str;
	while (*sptr && *sptr != '=') sptr++;	/* skip till '=' */
	sptr++;									/* skip the '=' */
	while (*sptr == ' ') sptr++;			/* skip spaces */

	/* copy the first part of the mapping into local address */
	j = 0;
	while (*sptr && *sptr != ',' && *sptr != '\n')
		temp_buffer[j++] = *sptr++;
	temp_buffer[j] = '\0';
	strcpy (nat_dynamic_entry->local_addr, temp_buffer);
	
	/* copy the second part of the mapping into protocol */
	j = 0;	sptr++;
	while (*sptr && *sptr != ',' && *sptr != '\n')
		temp_buffer[j++] = *sptr++;
	temp_buffer[j] = '\0';

	if (!strcmpi(temp_buffer, "0006"))
	{
		nat_dynamic_entry->protocol = 0x0006;
	}
	else
	{
		nat_dynamic_entry->protocol = 0x0011;
	}

	/* copy the third part of the mapping into port */
	j = 0;	sptr++;
	while (*sptr && *sptr != ',' && *sptr != '\n')
		temp_buffer[j++] = *sptr++;
	temp_buffer[j] = '\0';
	sscanf (temp_buffer, "%x", &application_port);
	nat_dynamic_entry->port = (USHORT)application_port;

	return;
}

