/* 
	File Name : PPPDBMAP.C	- Dial Backup MAPping file
	Contents  : Routines related to PPP-Dial Back up Station & WAN port mapping
	Author 	 : Chidananda Murthy R  (Courtesy : Sachin S Desai)
	Date	 : 1st October 1996
*/
#include "defs.h"
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include "ppp.h"


/*
	Routine : station_to_wan_port()
	Purpose : Returns the number of the wan port (0-relative) being actively
					used for a specified station.
				 If the station is not enabled or if there is no WAN port for the 
				 station, then returns special value MAXIMUM_NUMBER_OF_PPP_PORTS.
	Input parameters 		: station-id  
	Output parameters 		:
	Input-Output parameters :	
*/
USHORT station_to_wan_port (USHORT station_id)
{
   /* Handling for sake of other applications when PPP is disabled */
   if (ppp.station[station_id].enable == FALSE)
      return station_id;

	if ((ppp.station[station_id].enable == TRUE) && (ppp.station[station_id].active_link_id != MAXIMUM_NUMBER_OF_PPP_LINKS))
		return (ppp.station[station_id].link[ppp.station[station_id].active_link_id].wan_port);
	else
	{
		if (ppp.station[station_id].enable == FALSE)
			ppp_printf(PPP_ALARM_PRINTF, "PPP:station_to_wan_port() called for disabled station");
/*
		else
	  		ppp_printf (PPP_ALARM_PRINTF,"PPP:station_to_wan_port() could not map station");
*/
		return MAXIMUM_NUMBER_OF_PPP_PORTS;
	}
}

/*
	Routine : wan_port_to_station()
	Purpose : Returns the station no. to which this port is currently 
				connected to.
					Returns special value MAXIMUM_NUMBER_OF_PPP_STATIONS 
				if this port is not connected to any station (or if the
 				link thru' this port is down).
	Input parameters 			: wan_port - ID of the WAN port  
	Output parameters 		:
	Input-Output parameters :	
*/
USHORT wan_port_to_station (USHORT wan_port)
{
	USHORT i;

	/* Traverse thru' the structures for all stations 
		to check if there is an enabled station with active link thru' this port.
	*/
	for (i = 0 ; i < MAXIMUM_NUMBER_OF_PPP_STATIONS ; i++)
	{
		if ((ppp.station[i].enable == TRUE) && 
			(ppp.station[i].active_link_id != MAXIMUM_NUMBER_OF_PPP_LINKS) &&
			(ppp.station[i].link[ppp.station[i].active_link_id].wan_port == wan_port))
		{
			return i;
		}
	}
/*
  	ppp_printf (PPP_ALARM_PRINTF,"PPP:wan_port_to_station() could not map wan port");
*/
	return MAXIMUM_NUMBER_OF_PPP_STATIONS;
}

/*
	Routine : station_to_primary_port()
	Purpose : Given station no, it returns the WAN port corresponding to
					Primary link.
				 Returns MAXIMUM_NUMBER_OF_PPP_PORTS if station is not enabled.
	Input parameters 			: station_id - ID of the specified station  
	Output parameters 		:
	Input-Output parameters :	
*/
USHORT station_to_primary_port (USHORT station_id)
{
	if (ppp.station[station_id].enable == TRUE) 
		return (ppp.station[station_id].link[ppp.station[station_id].link_id_list[0]].wan_port);
	else
		return MAXIMUM_NUMBER_OF_PPP_PORTS;
}

/*
	Routine : primary_port_to_station()
	Purpose : If the given WAN port corresponds to Primary link of a station,
					it returns the station-id; otherwise it returns 
					MAXIMUM_NUMBER_OF_STATIONS.
	Input parameters 			: wan_port - ID of the specified station  
	Output parameters 		:
	Input-Output parameters :	
*/
USHORT primary_port_to_station (USHORT wan_port)
{
	USHORT i;

	/* Traverse thru' the structures for all stations 
		to check if there is an enabled station with primary link thru' this port.
	*/
	for (i = 0 ; i < MAXIMUM_NUMBER_OF_PPP_STATIONS ; i++)
	{
		if ((ppp.station[i].enable == TRUE) && 
			(ppp.station[i].link[ppp.station[i].link_id_list[0]].wan_port == wan_port))
		{
			return i;
		}
	}
	return MAXIMUM_NUMBER_OF_PPP_STATIONS;
}



USHORT station_to_backup_port (USHORT station_id)
{
	USHORT backup_link_no = ppp.station[station_id].link_id_list[1];
	
	if (ppp.station[station_id].enable == TRUE && ppp.station[station_id].link[backup_link_no].enable == TRUE) 
		return (ppp.station[station_id].link[backup_link_no].wan_port);
	else
		return MAXIMUM_NUMBER_OF_PPP_PORTS;
}

enum BOOLEAN DLBK_is_wan_port_mapped (USHORT wan_port_no)
{
	/* Given a WAN port no, look into PPP configuration & tell whether
		the wan port is mapped to some link of station or not.
		If it is mapped, return TRUE; Otherwise return FALSE. 
	*/
	USHORT i,j, link_no ;

	for (i = 0 ; i < MAXIMUM_NUMBER_OF_PPP_STATIONS ; i++)
	{
		if (ppp.station[i].enable == TRUE)
		{
			for (j = 0 ; j <= 1 ; j++)
			{
				if ((link_no = ppp.station[i].link_id_list[j]) != MAXIMUM_NUMBER_OF_PPP_LINKS)
				{
					if (ppp.station[i].link[link_no].enable == TRUE && 
						ppp.station[i].link[link_no].wan_port == wan_port_no)
							return TRUE ;
				}
			}
		}
	}
	return FALSE ;
}

enum BOOLEAN DLBK_is_backup_port (USHORT wan_port_no, USHORT *pstation_no, USHORT *plink_no)
{
	/* Given a WAN port no, look into PPP configuration & tell whether
		the wan port is backup port of an unique station.
		If yes, returns TRUE and sets *pstation_no and *plink_no; 
		Otherwise returns FALSE. 
	*/
	USHORT i, link_no ;
	enum BOOLEAN found = FALSE;

	*pstation_no = MAXIMUM_NUMBER_OF_PPP_STATIONS;
	for (i = 0 ; i < MAXIMUM_NUMBER_OF_PPP_STATIONS ; i++)
	{
		if (ppp.station[i].enable == TRUE)
		{
			if ((link_no = ppp.station[i].link_id_list[1]) != MAXIMUM_NUMBER_OF_PPP_LINKS)
			{
				if (ppp.station[i].link[link_no].enable == TRUE && 
					ppp.station[i].link[link_no].wan_port == wan_port_no)
				{
					if (found == FALSE)
					{
						found = TRUE;
						*pstation_no = i;
						*plink_no = link_no;
					}
					else
					{
						/* WAN port not mapped to an unique station */
						*pstation_no = MAXIMUM_NUMBER_OF_PPP_STATIONS;
						return (FALSE);
					}
				}
			}
		}
	}
	return found;
}


enum BOOLEAN DLBK_is_uniquely_mapped (USHORT wan_port_no, USHORT *pstation_no, USHORT *plink_no)
{
	/* Given a WAN port no, look into PPP configuration & tell whether
		the wan port is mapped to an unique station.
		If yes, returns TRUE and sets *pstation_no and *plink_no; 
		Otherwise returns FALSE. 
	*/
	USHORT i,j, link_no ;
	enum BOOLEAN found = FALSE;

	*pstation_no = MAXIMUM_NUMBER_OF_PPP_STATIONS;
	for (i = 0 ; i < MAXIMUM_NUMBER_OF_PPP_STATIONS ; i++)
	{
		if (ppp.station[i].enable == TRUE)
		{
			for (j = 0 ; j < 2; j++)
			{
				if ((link_no = ppp.station[i].link_id_list[j]) != MAXIMUM_NUMBER_OF_PPP_LINKS)
				{
					if (ppp.station[i].link[link_no].enable == TRUE && 
							ppp.station[i].link[link_no].wan_port == wan_port_no)
					{
						if (found == FALSE)
						{
							found = TRUE;
							*pstation_no = i;
							*plink_no = link_no;
						}
						else
						{
							/* WAN port not mapped to an unique station */
							*pstation_no = MAXIMUM_NUMBER_OF_PPP_STATIONS;
							return (FALSE);
						}
					}
				}
			}
		}
	}
	return found;
}

