#include <stdlib.h>
#include <string.h>
#include <kstart.h>

#include "\rtrware\store\boot.h"
#include "scrtdefs.h"
#include "scrttype.h"
#include "scrtenum.h"
#include "scrtfuns.h"
#include "scrtcons.h"
#include "scrtstr.h"
#include "vscrtstr.h" /* sudha 01 April 1999 */

extern WORD CalculateCRC (BYTE *Buf, WORD BufLen) ;
ScriptHeader ScriptInfo ;

/*----------------------------------------------------------------------------
	Name		:	AddSymbol												
	Input		:	name -> symbol name,										
					sym_type -> symbol type (integer/real/string),			
					n_ptr -> pointer to intialize for node added.				
	Output	:	0 if symbol added, else -1.								
	Synopsis	:	Adds the symbol to the symbol table at the specified	
				 	level. If the symbol is already there n_ptr is set to	
				 	the node, and -1 is returned. If out of memory 0 is	
				 	returned with n_ptr set to null.					
----------------------------------------------------------------------------*/
int AddSymbol (BYTE *name, int sym_type, SymType **n_ptr, BYTE PortNum)
{
	SymType **ptr ;
	SymType *sym_ptr ;
	int result ;

	if (ScriptPortClass[PortNum].Scope >= MAX_LEVELS) 
	{
		*n_ptr = 0 ;	/*	No such level possible.	*/
		script_printf (ALARM_SCRIPT_PRINTF,
				"Maximum number of levels for port %d\n\r", PortNum) ;
		goto error_exit ;
	}

	sym_ptr = 
		ScriptPortClass[PortNum].Display[ScriptPortClass[PortNum].Scope] ;

	while (sym_ptr) 
	{
		result = strcmp (name, sym_ptr->Name) ;
		if (!result) 
		{
			*n_ptr = sym_ptr ;	/*	Symbol already found.	*/
			script_printf (ALARM_SCRIPT_PRINTF,
					"Symbol already found for port %d\n\r", PortNum) ;
			goto error_exit ;
		}
		if (result < 0) 
		{
			if (sym_ptr->Llink)
				sym_ptr = sym_ptr->Llink ;	 /*	Look in left subtree.	*/
			else 
			{
				ptr = &sym_ptr->Llink ;	/*	Insert here.	*/
				break ;
			}
		}
		else 
		{
			if (sym_ptr->Rlink)
				sym_ptr = sym_ptr->Rlink ;	/*	Look in right subtree.	*/
			else 
			{
				ptr = &sym_ptr->Rlink ;	/*	Insert here.	*/
				break ;
			}
		}
	}
	if (!sym_ptr)
		ptr = &(ScriptPortClass[PortNum].Display[ScriptPortClass[PortNum].Scope]) ;
	sym_ptr = (SymType *) malloc (sizeof (SymType)) ;
	if (!sym_ptr) 
	{
		*n_ptr = 0 ;
		script_printf (ALARM_SCRIPT_PRINTF,
					"Not enough memory to add symbols for port %d\n\r", PortNum) ;
		goto error_exit ;
	}
	sym_ptr->Name = (BYTE *) malloc (strlen (name) + 1) ;
	if (!sym_ptr->Name)
	{
		free ((BYTE *) sym_ptr) ;
		*n_ptr = 0 ;
		script_printf (ALARM_SCRIPT_PRINTF,
					"Not enough memory to add symbol names for port %d\n\r", PortNum) ;
		goto error_exit ;
	}
	*n_ptr = *ptr = sym_ptr ;

	sym_ptr->SymClass = (BYTE) sym_type ;
	sym_ptr->SymType = 0 ;
	strcpy (sym_ptr->Name, name) ;
	sym_ptr->Llink = sym_ptr->Rlink = 0;
	return 0 ;

error_exit :
	return -1 ;
}



/*----------------------------------------------------------------------------
	Name		:	AddBuiltInCommands										
	Input		:	None													
	Output	:	0 if successful, else -1.								
	Synopsis	:	Adds the built-in commands to the symbol table.			
----------------------------------------------------------------------------*/
int AddBuiltInCommands (BYTE PortNum)
{
	WORD idx ;
	SymType *sym_ptr ;

	ScriptPortClass[PortNum].Scope = 0;		/*	Set context level.	*/

	for (idx = 0 ; idx < NoOfCommands ; idx ++) 
	{
		/*	Add built-in functions to symbol table.	*/
		if (AddSymbol (BuiltInCommands[idx].Name, Q_PROC, &sym_ptr, PortNum))
			goto error_exit ;
		else 
		{
			sym_ptr->SymType = BUILT_IN_FUNCTION ;	/*	Built-in function.	*/
			sym_ptr->Handler = BuiltInCommands[idx].Hdlr ;	/*	Fn handler.	*/
		}
	}

	return 0 ;

error_exit :
	script_printf (ALARM_SCRIPT_PRINTF,
			"Error in adding Built in commands, Port %d\n\r", PortNum) ;
	return -1 ;
}


/*----------------------------------------------------------------------------
	Name		:	ReadAndDecode											
	Input		:	None													
	Output	:	0 if successful, else -1.								
	Synopsis	:	Reads the code file and builds instruction table.		
----------------------------------------------------------------------------*/
int ReadAndDecode (BYTE PortNum)
{
	SymType *sym_ptr ;
	int param_count ;
	int var_count ;
	int len ;
	WORD address ;
	BYTE operator ;
	BYTE type1 ;
	BYTE type2 ;
	WORD next_code ;	/*	Offset to put next instruction.	*/
	ArgType *a_ptr ;

	/*	Assume most probable error.	*/
	ScriptPortClass[PortNum].LastCodeIdx = 0 ;
	ScriptPortClass[PortNum].CodeIdx = 0 ;
	ScriptPortClass[PortNum].Scope = 0 ;		/*	Set context level.	*/
	/*	Add the predefined globals and get count of variables added.	*/
	var_count = AddPreDefinedVariables (PortNum) ;

	if (var_count < 0) 
	{
		script_printf (ALARM_SCRIPT_PRINTF,
				"Script file execution internal error, Port %d\n\r", PortNum) ;
		goto error_exit ;
	}

	if (GetScriptLocation (PortNum) < 0)
		goto error_exit ;

	next_code = 0 ;
	while (1) 
	{
		len = ScriptPortRead (PortNum, &operator, sizeof (BYTE)) ;

		if (len == -1)
		{
			script_printf (ALARM_SCRIPT_PRINTF,
					"Error reading script for port %d\n\r", PortNum) ;
			goto error_exit ;
		}

		if (!len)
			break ;	/*	End of file.	*/

		ScriptPortClass[PortNum].AddrTable[ScriptPortClass[PortNum].CodeIdx] = next_code ;
		ScriptPortClass[PortNum].CodeBuffer[next_code++] = operator ;

		switch (operator) 
		{
			case Q_PLUS :
			case Q_MINUS :
			case Q_TIMES :
			case Q_DIV :
			case Q_LESS_OR_EQUAL :
			case Q_GREATER_OR_EQUAL :
			case Q_LESS :
			case Q_GREATER :
			case Q_IS_EQUAL :
			case Q_IS_NOT_EQUAL :
			case Q_AND :
			case Q_OR :
				/*	Get first operand type and save it.	*/
				if (ScriptPortRead (PortNum, &type1, sizeof (BYTE)) != 
																			sizeof (BYTE))
				{
					goto error_exit ;
				}

				if (ReadArgument (type1, 
					(BYTE *) &(ScriptPortClass[PortNum].CodeBuffer[next_code]), 
							PortNum) < 0)
				{
					goto error_exit ;
				}

				next_code += sizeof (ArgType) ;
				/*	Get second operand type and save it.	*/
				if (ScriptPortRead (PortNum, &type2, sizeof (BYTE)) != 
																			sizeof (BYTE))
				{
					goto error_exit;
				}

				if (ReadArgument (type2, 
					(BYTE *) &(ScriptPortClass[PortNum].CodeBuffer[next_code]),
								PortNum) < 0)
				{
					goto error_exit;
				}

				next_code += sizeof(ArgType);
				/*	Read off the dummy argument type.	*/
				if (ScriptPortRead (PortNum, &type2, sizeof (BYTE)) != 
																			sizeof (BYTE))
				{
					goto error_exit ;
				}

				if (ReadArgument (type2, 
					(BYTE *) &(ScriptPortClass[PortNum].CodeBuffer[next_code]),
						PortNum) < 0)
				{
					goto error_exit;
				}
				next_code += sizeof(ArgType);
				break;

			case Q_NOT :
			case Q_UNARY_MINUS :
				if (ScriptPortRead (PortNum, &type1, sizeof (BYTE)) != 
																			sizeof (BYTE))
				{
					goto error_exit;
				}

				if (ReadArgument (type1,
					(BYTE *) &(ScriptPortClass[PortNum].CodeBuffer[next_code]),
							PortNum) < 0)
				{
					goto error_exit;
				}

				next_code += sizeof (ArgType) ;
				/*	Read off the dummy type.	*/
				if (ScriptPortRead (PortNum, &type1, sizeof (BYTE)) != 
																			sizeof (BYTE))
				{
					goto error_exit;
				}

				if (ReadArgument (type1, 
					(BYTE *) &(ScriptPortClass[PortNum].CodeBuffer[next_code]),
						PortNum) < 0)
				{
					goto error_exit;
				}
				next_code += sizeof(ArgType);
				break;

			case Q_MOV :
				if (ScriptPortRead (PortNum, &type1, sizeof (BYTE)) != 
																			sizeof (BYTE))
				{
					goto error_exit;
				}

				if (ReadArgument (type1,
			   	 (BYTE *) &(ScriptPortClass[PortNum].CodeBuffer[next_code]),
							PortNum) < 0)
				{
					goto error_exit;
				}

				next_code += sizeof(ArgType) ;
				if (ScriptPortRead (PortNum, &type2, sizeof (BYTE)) != sizeof (BYTE))
				{
					goto error_exit;
				}

				if (ReadArgument (type2,
					(BYTE *) &(ScriptPortClass[PortNum].CodeBuffer[next_code]),
						PortNum) < 0)
				{
					goto error_exit;
				}

				next_code += sizeof(ArgType);
				break;

			case Q_LABEL_DEFN :	/*	Label declaration.	*/
				if (ReadArgument (E_STRING_CONST, 
					(BYTE *) &(ScriptPortClass[PortNum].CodeBuffer[next_code]),
						PortNum) < 0)
				{
					goto error_exit;
				}

				next_code += sizeof(ArgType);
				break;

			case Q_JMP :	/*	Unconditional jump.	*/
				/*	Read the jump label offset.	*/
				if (ScriptPortRead (PortNum, (BYTE *) &address,
									sizeof (address)) != sizeof (address))
				{
					goto error_exit;
				}

				address = scrt_change_endian (address) ;

				*(WORD *)(ScriptPortClass[PortNum].CodeBuffer + next_code) = (WORD)address;
				next_code += sizeof(address);

				/*	Read jump label name.	*/
				if (ReadArgument(E_STRING_CONST, (BYTE *)&ScriptPortClass[PortNum].CodeBuffer[next_code],PortNum) < 0)
				{
					goto error_exit;
				}

				next_code += sizeof(ArgType);
				break;

			case Q_JZ :		/*	Conditional jump.	*/
			case Q_JNZ :	/*	Conditional jump.	*/
				if (ScriptPortRead(PortNum, &type1, sizeof(BYTE)) !=
																			sizeof(BYTE))
				{
					goto error_exit;
				}

				if (ReadArgument(type1, (BYTE *)&ScriptPortClass[PortNum].CodeBuffer[next_code],PortNum) < 0)
				{
					goto error_exit;
				}

				next_code += sizeof(ArgType);
				/*	Read the jump label offset.	*/
				if (ScriptPortRead(PortNum, (BYTE *) &address,
									sizeof(address)) != sizeof(address))
				{
					goto error_exit;
				}

				address = scrt_change_endian (address) ;

				*(WORD *)(ScriptPortClass[PortNum].CodeBuffer + next_code) = (WORD)address;
				next_code += sizeof(address);

				/*	Read jump label name.	*/
				if (ReadArgument(E_STRING_CONST, (BYTE *)&(ScriptPortClass[PortNum].CodeBuffer[next_code]),PortNum) < 0)
				{
					goto error_exit;
				}

				next_code += sizeof(ArgType);
				break;

			case Q_JEQ :	/*	Conditional jump.	*/
			case Q_JNEQ :	/*	Conditional jump.	*/
				if (ScriptPortRead(PortNum, &type1, sizeof(BYTE)) !=
														sizeof(BYTE))
				{
					goto error_exit;
				}

				if (ReadArgument(type1, (BYTE *)&(ScriptPortClass[PortNum].CodeBuffer[next_code]),PortNum) < 0)
				{
					goto error_exit;
				}

				next_code += sizeof(ArgType);

				if (ScriptPortRead(PortNum, &type2, sizeof(BYTE)) !=
																			sizeof(BYTE))
				{
					goto error_exit;
				}

				if (ReadArgument(type2, (BYTE *)&(ScriptPortClass[PortNum].CodeBuffer[next_code]),PortNum) < 0)
				{
					goto error_exit;
				}

				next_code += sizeof(ArgType);
				/*	Read the jump label offset.	*/
				if (ScriptPortRead(PortNum, (BYTE *) &address,
									sizeof(address)) != sizeof(address))
				{
					goto error_exit;
				}

				address = scrt_change_endian (address) ;

				*(WORD *)(ScriptPortClass[PortNum].CodeBuffer + next_code) = (WORD)address;
				next_code += sizeof(address);

				/*	Read jump label name.	*/
				if (ReadArgument(E_STRING_CONST, (BYTE *)&(ScriptPortClass[PortNum].CodeBuffer[next_code]),PortNum) < 0)
				{
					goto error_exit;
				}

				next_code += sizeof(ArgType);
				break;

			case Q_PUSH :
				if (ScriptPortRead (PortNum, &type1, sizeof (BYTE)) !=
																		sizeof (BYTE))
				{
					goto error_exit;
				}

				if (ReadArgument (type1, 
						(BYTE *) &(ScriptPortClass[PortNum].CodeBuffer[next_code]),
							PortNum) < 0)
				{
					goto error_exit;
				}
				next_code += sizeof (ArgType) ;
				break;

			case Q_CALL :
				if (ReadArgument (E_STRING_CONST,
						(BYTE *) &(ScriptPortClass[PortNum].CodeBuffer[next_code]),
								PortNum) < 0)
				{
					goto error_exit;
				}

				next_code += sizeof (ArgType) ;
				break;

			case Q_RET :
				break;

			case Q_LINENO :	/*	Psuedocode, line number information.	*/
				if (ScriptPortRead (PortNum, (BYTE *) &address,
									sizeof (address)) != sizeof (address))
				{
					goto error_exit;
				}

				address = scrt_change_endian (address) ;

				*(WORD *)(ScriptPortClass[PortNum].CodeBuffer + next_code) = (WORD)address;
				next_code += sizeof (address) ;
				break;

			case Q_PROCEDURE :	/*	Procedure beginning.	*/
				a_ptr = (ArgType *) &(ScriptPortClass[PortNum].CodeBuffer[next_code]) ;
				if (ReadArgument (E_STRING_CONST, 
					(BYTE *)&ScriptPortClass[PortNum].CodeBuffer[next_code],
						PortNum) < 0)
				{
					goto error_exit;
				}

				if (AddSymbol (a_ptr->StrVal, Q_PROC, &sym_ptr,PortNum))
				{
					goto error_exit;
				}
				else 
				{
					free (a_ptr->StrVal);
					/*	Remember where procedure code starts.	*/
					a_ptr->IntVal = ScriptPortClass[PortNum].AddrTable[ScriptPortClass[PortNum].CodeIdx];
					a_ptr->Handler = 0;	/*	Put a null for handler.	*/
					sym_ptr->SymType = 0;	/*	User defined function.	*/
					sym_ptr->IntVal = ScriptPortClass[PortNum].CodeIdx;	/*	Instn seq no.	*/
				}
				next_code += sizeof(ArgType);
				ScriptPortClass[PortNum].Scope++;
				param_count = 0;
				var_count = 0;	/*	Start counting local variables.	*/
				break;

			case Q_ENDPROCEDURE :	/*	Procedure end.	*/
				if (ReadArgument(E_STRING_CONST, (BYTE *)&(ScriptPortClass[PortNum].CodeBuffer[next_code]),PortNum) < 0)
				{
					goto error_exit;
				}
				next_code += sizeof (ArgType) ;
				PurgeSymbols ((SymType *) ScriptPortClass[PortNum].Display[ScriptPortClass[PortNum].Scope]) ; 
				ScriptPortClass[PortNum].Display[ScriptPortClass[PortNum].Scope] = 0 ;
				ScriptPortClass[PortNum].Scope--;
				break;

			case Q_PARAM :	/*	Procedure parameter.	*/
				if (ScriptPortRead(PortNum, &type1, sizeof(BYTE)) !=
																			sizeof(BYTE))
				{
					goto error_exit;
				}

				a_ptr = (ArgType *)&(ScriptPortClass[PortNum].CodeBuffer[next_code]);
				if (ReadArgument(E_STRING_CONST, (BYTE *)&(ScriptPortClass[PortNum].CodeBuffer[next_code]),PortNum) < 0)
				{
					goto error_exit;
				}

				ScriptPortClass[PortNum].CodeBuffer[next_code] = type1;	/*	Patch back type.	*/
				if (AddSymbol(a_ptr->StrVal, type1, &sym_ptr,PortNum))
				{
					goto error_exit;
				}
				else 
				{
					sym_ptr->SymType = VAR_PARAM;
					sym_ptr->Status = param_count++;
				}
				next_code += sizeof(ArgType);
				break;

			case Q_DECLARE :	/*	Variable declaration.	*/
				if (ScriptPortRead (PortNum, &type1, sizeof (BYTE)) !=
																			sizeof (BYTE))
				{
					goto error_exit;
				}

				a_ptr = (ArgType *) &(ScriptPortClass[PortNum].CodeBuffer[next_code]) ;
				if (ReadArgument (E_STRING_CONST,
					(BYTE *) &ScriptPortClass[PortNum].CodeBuffer[next_code],
							PortNum) < 0)
				{
					goto error_exit;
				}

				ScriptPortClass[PortNum].CodeBuffer[next_code] = type1 ;	/*	Patch back type.	*/
				if (AddSymbol (a_ptr->StrVal, type1, &sym_ptr, PortNum))
				{
					goto error_exit ;
				}
				else 
				{
					sym_ptr->SymType = (BYTE) (!ScriptPortClass[PortNum].Scope ? VAR_GLOBAL : VAR_LOCAL) ;
					sym_ptr->SymClass = type1 ;
					sym_ptr->Status = var_count ++ ;
					switch (type1) 
					{
						case E_INTEGER :
							sym_ptr->IntVal = 0 ;
							break ;

						case E_REAL :
							break ;

						case E_STRING :
							sym_ptr->StrVal = 0 ;
							break ;
					}
				}
				next_code += sizeof (ArgType) ;
				break ;

			case Q_TEMPCOUNT :	/*	Max temporaries used in fn body.	*/
				if (ScriptPortRead (PortNum, (BYTE *) &address,
									sizeof (address)) != sizeof (address))
				{
					goto error_exit;
				}

				address = scrt_change_endian (address) ;

				*(WORD *)(ScriptPortClass[PortNum].CodeBuffer + next_code) = (WORD) address ;
				next_code += sizeof (address) ;
				break;

			case Q_KILL :	/*	Kill temporary.	*/
				if (ScriptPortRead (PortNum, (BYTE *) &address,
									sizeof (address)) != sizeof (address))
				{
					goto error_exit;
				}

				address = scrt_change_endian (address) ;

				*(WORD *)(ScriptPortClass[PortNum].CodeBuffer + next_code) = (WORD)address;
				next_code += sizeof(address);
				break;

			default :
				goto error_exit;
		}
		ScriptPortClass[PortNum].CodeIdx++;		/*	Next instruction to be read.	*/
	}

	ScriptPortClass[PortNum].LastCodeIdx = ScriptPortClass[PortNum].CodeIdx;	/*	Remember the last instruction seq no.	*/
	ScriptPortClass[PortNum].AddrTable[ScriptPortClass[PortNum].CodeIdx] = (WORD) next_code ;	/*	Code ends and stack begins here.	*/
	/*	Resolve call addresses for user defined procedures and handlers	*/
	/*	for built-in fucntions.	*/
	for (next_code = 0 ; next_code < ScriptPortClass[PortNum].CodeIdx ; next_code ++) 
	{
		/*	Get the offset of the instruction beginning from addr table.	*/
		address = ScriptPortClass[PortNum].AddrTable[next_code] ;

		if (ScriptPortClass[PortNum].CodeBuffer[address] == Q_CALL) 
		{
			a_ptr = (ArgType *) &(ScriptPortClass[PortNum].CodeBuffer[address + 1]) ;

			sym_ptr = FindSymbol (a_ptr->StrVal, PortNum) ;
			if (!sym_ptr) 
			{
				goto error_exit ;
			}
			if (sym_ptr->SymClass != Q_PROC) 
			{
				goto error_exit ;
			}
			/*	Release the procedure name string.	*/
			if (a_ptr->StrVal) 
			{
				free (a_ptr->StrVal) ;
				a_ptr->StrVal = 0 ;
			}
			/*	If procedure is a built-in function take the handler	*/
			/*	address from the symbol table; otherwise take the	*/
			/*	procedure beginning address in the code buffer.	*/
			if (sym_ptr->SymType & BUILT_IN_FUNCTION) 
			{
				a_ptr->Handler = sym_ptr->Handler ;
				a_ptr->StorageClass = BUILT_IN_FUNCTION ;
			}
			else 
			{
				a_ptr->IntVal = sym_ptr->IntVal ;
				a_ptr->StorageClass = 0 ;
			}
		}
	}
	return 0 ;

error_exit :
	ScriptPortClass[PortNum].LastCodeIdx = ScriptPortClass[PortNum].CodeIdx ;	/*	Remember the last instruction seq no.	*/
	/*	Release procedure name strings.	*/
	for (next_code = 0; next_code < ScriptPortClass[PortNum].CodeIdx; next_code++) 
	{
		/*	Get the offset of the instruction beginning from addr table.	*/
		address = ScriptPortClass[PortNum].AddrTable[next_code];
		if (ScriptPortClass[PortNum].CodeBuffer[address] == Q_CALL) 
		{
			a_ptr = (ArgType *) &(ScriptPortClass[PortNum].CodeBuffer[address + 1]);
			/*	Release the procedure name string.	*/
			if ( a_ptr->StorageClass != BUILT_IN_FUNCTION )
			{
				if (a_ptr->StrVal) 
				{
					free (a_ptr->StrVal);
					a_ptr->StrVal = 0;
				}
			}
		}
	}

	PurgeNamesAndLabels (PortNum) ;
	return -1 ;
}


/*----------------------------------------------------------------------------
	Name		:	AddPreDefinedVariables									
	Input		:	None													
	Output	:	Number of predefined variables added, -1 if error.		
	Synopsis	:	Adds the predefined variables to the symbol table.		
----------------------------------------------------------------------------*/
int AddPreDefinedVariables (BYTE PortNum)
{
	SymType *sym_ptr ;
	int var_count ;

	for (var_count = 0 ; var_count < NoOfPreDefVars ; var_count ++) 
	{
		if (AddSymbol (PreDefVars[var_count].VarName,
							PreDefVars[var_count].VarType, &sym_ptr, PortNum))
			goto error_exit ;
		else 
		{
			sym_ptr->SymType = VAR_GLOBAL ;
			sym_ptr->Status = var_count ;	/*	Symbol offset.	*/
			sym_ptr->IntVal = 0 ;
		}
	}
	return var_count ;

error_exit :
	script_printf (ALARM_SCRIPT_PRINTF,
			"Error in adding predefined variables, Port %d\n\r", PortNum) ;
	return -1 ;
}


/*----------------------------------------------------------------------------
	Name		:	FindSymbol												
	Input		:	name, symbol name.										
	Output 	:	Pointer to symbol node if found, else null.				
	Synopsis	:	Looks up the symbol in the symbol table at level 0 and	
			 		returns a pointer to the node if found, else null.	
----------------------------------------------------------------------------*/
SymType *FindSymbol (BYTE *name, BYTE PortNum)
{
	SymType *sym_ptr ;
	int result ;

	sym_ptr = ScriptPortClass[PortNum].Display[0] ;
	while (sym_ptr) 
	{
		result = strcmp (name, sym_ptr->Name) ;
		if (!result)
			return sym_ptr ;	/*	Symbol found.	*/
		if (result < 0)
			sym_ptr = sym_ptr->Llink ;
		else
			sym_ptr = sym_ptr->Rlink ;
	}
	return 0 ;	/*	Symbol not found.	*/
}


/*----------------------------------------------------------------------------
	Name		:	ReadArgument											
	Input		:	arg_type, type of argument to read,						
					buffer, buffer to read argument into.					
	Output  	:	0 if read successfully, else -1.						
	Synopsis	:	Reads the argument into the buffer.						
----------------------------------------------------------------------------*/
int ReadArgument (BYTE arg_type, BYTE *buffer, BYTE PortNum)
{
	ArgType *a_ptr ;
	SymType *sym_ptr ;
	BYTE arg[MAX_TOKEN_LEN] ;
	BYTE class ;		/*	Storage class of variable.	*/
	WORD address ;	   /*	Address, offset into symbol table.	*/
	BYTE *name ;
	int sign ;
	int len ;
	long int_value ;

	len = 0 ;
	a_ptr = (ArgType *) buffer ;
	a_ptr->ArgType = arg_type ;
	switch (arg_type) 
	{
		case E_INT_CONST :
		case E_REAL_CONST :
		case E_STRING_CONST :
			name = arg ;
			while (1) 
			{
				if (ScriptPortRead (PortNum, name, sizeof (BYTE)) 
																		!= sizeof (BYTE))
				{
					goto error_exit ;
				}
				len ++ ;
				if (!*name)
					break ;	/*	Null found, done.	*/
				name ++ ;
			}

			if (arg_type == E_STRING_CONST) 
			{
				if (! (name = (BYTE *) malloc (len)))
				{
					goto error_exit ;
				}
				strcpy (name, arg) ;
				a_ptr->StrVal = name ;
				break ;
			}
			name = arg ;
			int_value = 0 ;
			sign = 1 ;
			if (*name == '+' || *name == '-') 
			{
				if (*name == '-')
					sign = -1 ;
				name ++ ;
			}
			while (*name >= '0' && *name <= '9')
				int_value = int_value * 10 + *name++ - '0' ;
			if (arg_type == E_INT_CONST) 
			{
				if (sign == -1)
					int_value = -int_value ;
				a_ptr->IntVal = int_value ;
				break ;
			}

		case E_INTEGER :
		case E_REAL :
		case E_STRING :
			if (ScriptPortRead (PortNum, &class, sizeof (class)) !=
														sizeof (class))
			{
				goto error_exit;
			}
			len += sizeof (class) ;
			switch (class) 
			{
				case VAR_GLOBAL :
					a_ptr->StorageClass = VAR_GLOBAL;
					if (ScriptPortRead(PortNum, (BYTE *) &address,
									sizeof(address)) != sizeof(address))
					{
						goto error_exit;
					}
					len += sizeof(address);

					address = scrt_change_endian (address) ;

					if (GetSymbolName(ScriptPortClass[PortNum].Display[0], VAR_GLOBAL, address,
																&sym_ptr))
						a_ptr->SymVal = sym_ptr;
					else
					{
						goto error_exit;	/*	Symbol not found, error !	*/
					}
					break;

				case VAR_PARAM :
				case VAR_LOCAL :
					a_ptr->StorageClass = class ;
					if (ScriptPortRead (PortNum, (BYTE *) &address,
									sizeof (address)) != sizeof (address))
					{
						goto error_exit ;
					}
					len += sizeof (address) ;

					address = scrt_change_endian (address) ;
	
					if (GetSymbolName (ScriptPortClass[PortNum].Display[1], class, address, &sym_ptr))
						a_ptr->IntVal = address ;
					else
					{
						goto error_exit ;	/*	Symbol not found, error !	*/
					}
					break;

				case VAR_TEMPORARY :
					a_ptr->StorageClass = VAR_TEMPORARY;
					if (ScriptPortRead (PortNum, (BYTE *) &address,
											sizeof (address)) != sizeof (address))
					{
						goto error_exit;
					}
					len += sizeof(address);

					address = scrt_change_endian (address) ;

					a_ptr->IntVal = address;
					break;

				case VAR_CONDITION :
					a_ptr->StorageClass = VAR_CONDITION;
					if (sym_ptr =
							FindSymbol(PreDefVars[CONDN_VARIABLE].VarName,PortNum))
						a_ptr->SymVal = sym_ptr;
					else
					{
						goto error_exit;	/*	Symbol not found, error !	*/
					}
					break;

				case VAR_RETURN :
					a_ptr->StorageClass = VAR_RETURN;
					if (sym_ptr =
							FindSymbol(PreDefVars[RETURN_VARIABLE].VarName,PortNum))
						a_ptr->SymVal = sym_ptr;
					else
					{
						goto error_exit;	/*	Symbol not found, error !	*/
					}
					break;
			}
			break ;
	}
	return len ;

error_exit :
	*arg = 0 ;
	return -1 ;
}


void PurgeSymbols (SymType *n_ptr)
{
	if (n_ptr) 
	{
		PurgeSymbols (n_ptr->Llink) ;
		PurgeSymbols (n_ptr->Rlink) ;

		free (n_ptr->Name) ;
		if (n_ptr->SymClass == E_STRING && n_ptr->StrVal) 
		{
			free (n_ptr->StrVal) ;
			n_ptr->StrVal = 0 ;
		}

		free ((SymType *) n_ptr) ;
	}
}



int ScriptPortRead (BYTE PortNum, BYTE *buffer, int len)
{
	DWORD FileEndPos ;

	FileEndPos = ScriptInfo.Ports[PortNum].StartAddr +
						ScriptInfo.Ports[PortNum].SrcLen +
								ScriptInfo.Ports[PortNum].CodeLen ;

	if (ScriptCodeIndex[PortNum] == FileEndPos)
		return 0 ;

	if ((ScriptCodeIndex[PortNum] + len) > FileEndPos)
	{
		return -1 ;
	}
		
	memcpy ((BYTE *) buffer, (BYTE *) ScriptCodeIndex[PortNum], len) ;
	ScriptCodeIndex[PortNum] += len ;

	return len ;
}


/*----------------------------------------------------------------------------
	Name		:	PurgeNamesAndLabels										
	Input		:	None													
	Output	:	0 if successful, else -1.								
	Synopsis	:	Releases the memory occupied by label names.			
----------------------------------------------------------------------------*/

int PurgeNamesAndLabels (BYTE PortNum)
{
	WORD idx ;
	WORD address ;
	BYTE *ptr ;
	ArgType *a_ptr ;

	for (idx = 0 ; idx < ScriptPortClass[PortNum].LastCodeIdx ; idx ++)
	{
		/*	Get the offset of the instruction beginning from addr table.	*/
		address = ScriptPortClass[PortNum].AddrTable[idx] ;
		ptr = (BYTE *) &(ScriptPortClass[PortNum].CodeBuffer[address + 1]) ;
		switch (ScriptPortClass[PortNum].CodeBuffer[address]) 
		{
			case Q_MOV :
				/*	If there is a string constant in the instruction,	*/
				/*	release the memory. Only the first argument can	*/
				/*	have a string constant, but not second argument.	*/
				a_ptr = (ArgType *) ptr ;
				if (a_ptr->ArgType == E_STRING_CONST && a_ptr->StrVal) 
				{
					free (a_ptr->StrVal) ;
					a_ptr->StrVal = 0 ;
				}
				break ;

			case Q_LABEL_DEFN :	/*	Label declaration.	*/
				a_ptr = (ArgType *) ptr ;
				if (a_ptr->StrVal) 
				{
					free (a_ptr->StrVal) ;
					a_ptr->StrVal = 0 ;
				}
				break ;

			case Q_JMP :	/*	Unconditional jump.	*/
				/*	Release the jump label name.	*/
				a_ptr = (ArgType *) (ptr + sizeof (WORD)) ;
				if (a_ptr->StrVal) 
				{
					free (a_ptr->StrVal) ;
					a_ptr->StrVal = 0 ;
				}
				break ;

			case Q_JZ :	/*	Conditional jump.	*/
			case Q_JNZ :	/*	Conditional jump.	*/
				/*	Release the jump label name.	*/
				a_ptr = (ArgType *) (ptr + sizeof (WORD) +
														sizeof (ArgType)) ;
				if (a_ptr->StrVal) 
				{
					free (a_ptr->StrVal) ;
					a_ptr->StrVal = 0 ;
				}
				break ;

			case Q_JEQ :	/*	Jump on equal.	*/
			case Q_JNEQ :	/*	Jump on not equal.	*/
				/*	Release the first argument if it is a string constant.	*/
				a_ptr = (ArgType *) ptr ;
				if (a_ptr->ArgType == E_STRING_CONST && a_ptr->StrVal) 
				{
					free (a_ptr->StrVal) ;
					a_ptr->StrVal = 0 ;
				}

				/*	Release the second argument if it is a string constant.	*/
				a_ptr = (ArgType *) (ptr + sizeof (ArgType)) ;
				if (a_ptr->ArgType == E_STRING_CONST && a_ptr->StrVal) 
				{
					free (a_ptr->StrVal) ;
					a_ptr->StrVal = 0 ;
				}

				/*	Release the jump label name.	*/
				a_ptr = (ArgType *) (ptr + sizeof (WORD) +
													2 * sizeof (ArgType)) ;
				if (a_ptr->StrVal) 
				{
					free (a_ptr->StrVal) ;
					a_ptr->StrVal = 0 ;
				}
				break ;

			case Q_ENDPROCEDURE :	/*	Procedure end.	*/
			case Q_PARAM :	/*	Procedure parameter.	*/
			case Q_DECLARE :	/*	Variable declaration.	*/
				a_ptr = (ArgType *) ptr ;
				if (a_ptr->StrVal) 
				{
					free (a_ptr->StrVal) ;
					a_ptr->StrVal = 0 ;
				}
				break ;
		}
	}
	ScriptPortClass[PortNum].LastCodeIdx = 0 ;
	return 0 ;
}


/*----------------------------------------------------------------------------
	Name		:	GetSymbolName											
	Input		:	sym_class, root of symbol table subtree,				
					class, symbol storage class,							
					address, symbol address,								
					n_ptr, pointer to pointer to symbol node.				
	Output	:	1 if symbol found else 0.						
	Synopsis	:	Looks up the symbol table and returns a pointer to the	
					symbol name (depth first search).					
----------------------------------------------------------------------------*/
int GetSymbolName (SymType *sym_ptr, BYTE class, int address, SymType **n_ptr)
{
/*	Times ++ ; */
	if (!sym_ptr)
		return 0 ;
	if (sym_ptr->SymType == class && sym_ptr->Status == address) 
	{
		*n_ptr = sym_ptr ;
		return 1 ;
	}
	if (sym_ptr->Llink && GetSymbolName(sym_ptr->Llink, class, address,
															n_ptr))
		return 1 ;
	if (sym_ptr->Rlink && GetSymbolName(sym_ptr->Rlink, class, address,
															n_ptr))
		return 1 ;
	return 0 ;
}


int GetScriptLocation (BYTE PortNum)
{
	WORD CRC ;

	memcpy ((BYTE *) &ScriptInfo, (BYTE *) FL_SCRIPT_HDR, 
														sizeof (ScriptInfo)) ;
	ScriptCodeIndex[PortNum] =
		ScriptInfo.Ports[PortNum].StartAddr +
				ScriptInfo.Ports[PortNum].SrcLen ;

	/* see to it that cod read does not wander */
	if ((ScriptInfo.Ports[PortNum].StartAddr < SCRIPT_ONE_DNLD_ADDR)
		|| (ScriptInfo.Ports[PortNum].StartAddr > (UDB_DNLD_ADDR - 1))
			|| (ScriptInfo.Ports[PortNum].SrcLen + ScriptInfo.Ports[PortNum].CodeLen > SINGLE_SCR_SIZE)
				|| (ScriptInfo.Ports[PortNum].StartAddr + ScriptInfo.Ports[PortNum].SrcLen + ScriptInfo.Ports[PortNum].CodeLen > (UDB_DNLD_ADDR - 1))
					|| ScriptInfo.MagicNumber != MAGIC_NUM)

	{
		script_printf (ALARM_SCRIPT_PRINTF,
			"Error in location of script for port %d\n\r", PortNum) ;
		return -1 ;
	}

	CRC = CalculateCRC ((BYTE *) ScriptInfo.Ports[PortNum].StartAddr, 
		ScriptInfo.Ports[PortNum].SrcLen + ScriptInfo.Ports[PortNum].CodeLen) ;
	if (CRC != ScriptInfo.Ports[PortNum].CRC)
	{
		script_printf (ALARM_SCRIPT_PRINTF,
			"CRC error for script, port %d", PortNum) ;
		return -1 ;
	}
	return 0 ;
}


WORD scrt_change_endian (WORD Num)
{

	WORD Temp ;
	BYTE *NumPtr, *TempPtr ;

	NumPtr = (BYTE *) &Num ;
	TempPtr = (BYTE *) &Temp ;

	TempPtr[0] = NumPtr[1] ;
	TempPtr[1] = NumPtr[0] ;

	return Temp ;
}

WORD GetPortScrCodLen (USHORT PortNum)
{
	memcpy ((BYTE *) &ScriptInfo, (BYTE *) FL_SCRIPT_HDR, 
														sizeof (ScriptInfo)) ;
	if (ScriptInfo.MagicNumber != MAGIC_NUM)
		return 0 ;
	return (ScriptInfo.Ports[PortNum].SrcLen +
					ScriptInfo.Ports[PortNum].CodeLen) ;
}

