/* SERRX.C -- Code handling the receive part
** By: Sanjay
** Start: 22, July, 1996
** Done: 22, July, 1996
** Modifications : sudha 15-Oct-1999. Introduced a global var disable_rts_sent
**						 to keep track of when rts is disabled. In serial_get_a_free
**						 _wan_rx_descriptor() function, changed the free buffers 
**						 marking level to disable rts from 1/4th to 1/3rd as well as
**						 setting disable_rts_sent global var. In serial_rx_complete()
**						 function, changed the free buffers marking level to enable
**						 rts from 3/4th to 2/3rd as well as enable_rts_on_scc() will 
**						 be called only when disable_rts_sent flag is set. After 
**						 enabling rts, disable_rts_sent is reset.								 	
*/

/* General Notes:
** 1. Re: RTS Flow Control: The current method is inefficient (but is okay
**	  for now). What we do is, if the number free buffers is 1/4 the
**	  max, we de-assert RTS. When the number free buffers is 3/4 the max,
**	  we re-assert RTS. In the latter case, we also take into account that
**	  8 (as of now) buffers are always locked up with the SCC and are
**	  actually free. This is also why the method is "inefficient" -- when
**	  we actually de-assert RTS, there is always 8 buffers with the SCC
**	  that potentially could have been used.
*/

#include <stdio.h>
#include "rtrstd.h"
#include "serial.h"
#include "vserstr.h"
#include "serasm.h"

/* Global variable */
enum BOOLEAN stop_passing_rx_buffer_to_application = FALSE; 

/* sudha 15-Oct-1999. Added this var, to know, whether rts is disabled.
This var will be set, whenever disable_rts_on_scc() is called. And will
be reset, when enable_rts_on_scc() is called. */
enum BOOLEAN disable_rts_sent = FALSE;

/* Global Prototypes */
void *serial_get_a_free_wan_rx_descriptor(int port_number);
void serial_rxed_packet(int port_number, int data_length);
void serial_rx_error_adjust_sch_rx_list(int port_number);
void serial_rx_complete(USHORT port_number, BYTE *sptr_buffer, USHORT send_status);

/* Local Prototypes */
void enable_rts_on_scc(USHORT port_number);
void disable_rts_on_scc(int port_number);
#ifdef DEBUG
STATIC enum TEST check_if_rx_descriptor_valid(USHORT port_number, SERIAL_RXTX_DESCRIPTOR *sptr_rx_descriptor);
#endif /* DEBUG */

/* Externals */
extern SERIAL_CLASS serial;


/* -- CODE ---------------------------------------------------------------- */

/* serial_get_a_free_wan_rx_descriptor()
**	Called to obtain a free buffer for receive purposes. This functions
**	de-asserts RTS (flow control) if it is running short of buffers.
*/
void *serial_get_a_free_wan_rx_descriptor(int port_number)
{
	SERIAL_RXTX_DESCRIPTOR *sptr_free_rx_descriptor;
	int actual_free ;	/* 01/02/97 */

	sptr_free_rx_descriptor = (SERIAL_RXTX_DESCRIPTOR *) get_entry_from_list(&serial.free_rx_list[port_number]);

	if (sptr_free_rx_descriptor == NULL) 
	{
		/* If coding (and perhaps timing) is right, this function would never 
		** have been called if there were no buffers as we manipulate RTS.
		*/

#ifdef DEBUG
		printf("SERIAL: No free buf for rx, num buf = %d\r\n", serial.num_free_rx_buf[port_number]);
#endif
		disable_rts_on_scc(port_number); /* Arrange again to stop receiving */
/* sudha 15-Oct-1999. Setting the var disable_rts_sent. */
		disable_rts_sent = TRUE;

		return NULL;
	}

	add_entry_to_list(&serial.scheduled_for_rx[port_number], (LINK *)&sptr_free_rx_descriptor->links);
	serial.num_scheduled_for_rx[port_number] += 1;
	serial.num_free_rx_buf[port_number] -= 1;

	actual_free = serial.num_free_rx_buf[port_number] +
						serial.num_scheduled_for_rx[port_number];

/* sudha 15-Oct-1999. Changed the marking level to disable rts from 1/4th to
1/3rd.Also setting the var disable_rts_sent... */

	if (actual_free <= (MAX_SERIAL_RX_DESCRIPTORS / 3))
	{
/*		printf("SERIAL: actual free is = %d\r\n", actual_free);*/
		disable_rts_on_scc(port_number);
		disable_rts_sent = TRUE;
	}
/* ...sudha 15-Oct-1999 */

	return (sptr_free_rx_descriptor->sptr_buffer);
}

/* serial_rx_error_adjust_sch_rx_list()
**	This function is called when errors occur on the receive side. This just
**	arranges to reuse the buffer on which the error occured. Re-use for us
**	means pick up buffer from head of scheduled_for_rx queue and put it at 
**	the tail. Called with interrupts disabled.
*/
void serial_rx_error_adjust_sch_rx_list(int port_number)
{
	SERIAL_RXTX_DESCRIPTOR *sptr_rx_descriptor;


	sptr_rx_descriptor = (SERIAL_RXTX_DESCRIPTOR *)get_entry_from_list(&serial.scheduled_for_rx[port_number]);
	if (sptr_rx_descriptor == NULL)
	{
#ifdef DEBUG
		printf("SERIAL: Strange! Rx error occured though nothing received\n\r");
#endif /* DEBUG */
		return;
	}
	add_entry_to_list(&serial.scheduled_for_rx[port_number], (LINK *)&sptr_rx_descriptor->links);
}

/* serial_rxed_packet()
**	This is called when a packet is received with no errors. The rxed packet
**	is simply put on the received queue. The foreground process will regularly
**	check this received queue. Called with interrupts disabled. When called
**	the packet is actually at the head of the scheduled_for_rx queue.
*/
void serial_rxed_packet(int port_number, int data_length)
{
	SERIAL_RXTX_DESCRIPTOR *sptr_rx_descriptor;


	sptr_rx_descriptor = (SERIAL_RXTX_DESCRIPTOR *)get_entry_from_list(&serial.scheduled_for_rx[port_number]);
	if (sptr_rx_descriptor == NULL)
	{
#ifdef DEBUG
		printf("SERIAL: Strange! Rx interrupt occured though nothing received\n\r");
#endif /* DEBUG */
		return;
	}
	serial.num_scheduled_for_rx[port_number] -= 1;

	sptr_rx_descriptor->length = data_length;

	add_entry_to_list(&serial.current_rx_list[port_number], (LINK *)&sptr_rx_descriptor->links);
}


/* serial_rx_complete()
**	Called from upper layer when it is done with the current RX buffer.
**	This will arrange for re-use of the buffer for future receives. Also
**	asserting of the RTS signal if it had been de-asserted due to shortage
**	of receive buffers is done here.
*/
void serial_rx_complete(USHORT port_number, BYTE *sptr_buffer, USHORT send_status)
{
	USHORT temp_sr;
	int actual_free;
	SERIAL_RXTX_DESCRIPTOR *sptr_rx_descriptor;

/* 9 March 1999.... */
/* This check is included as a part of flow control
   for AG module */

	if (send_status == 0)
		stop_passing_rx_buffer_to_application = TRUE;
	else
		stop_passing_rx_buffer_to_application = FALSE;

	if (sptr_buffer == NULL)
	{
		return;
	}

/* ....9 March 1999 */

		
	temp_sr = _GPL();
	_SPL(7);

	sptr_rx_descriptor = *(SERIAL_RXTX_DESCRIPTOR **)(sptr_buffer - RX_BUFFER_HEADER_SPACE);
#ifdef DEBUG
	if (check_if_rx_descriptor_valid(port_number, sptr_rx_descriptor) == FAIL)
	{
		printf("AG: Receive complete function passed a corrupt descriptor pointer.!!!\n\r");
		return;
	}
#endif
	add_entry_to_list(&serial.free_rx_list[port_number], (LINK *)&sptr_rx_descriptor->links);

	/* Re-enable RTS if there is more than 3/4 buffer space. Since atleast
	** 8 of these will be with the SCC, we add six to the free number
	** variable.
	*/
	serial.num_free_rx_buf[port_number] += 1;
	actual_free = serial.num_free_rx_buf[port_number] + serial.num_scheduled_for_rx[port_number];

/* sudha 15-Oct-1999. Changed the marking level to enable rts from 3/4th to
2/3rd & enabling rts only if it is disabled. Also resetting the var 
disable_rts_sent... */

	if ((actual_free >= ((2 * MAX_SERIAL_RX_DESCRIPTORS) / 3))
		&& (disable_rts_sent == TRUE))
	{
		enable_rts_on_scc(port_number);
		disable_rts_sent = FALSE;
	}
/* ...sudha 15-Oct-1999 */

	_SPL(temp_sr);
}

void enable_rts_on_scc(USHORT port_number)
{
	switch (port_number)
	{
	case 0:
		turn_on_scc2_rts();
		break;
	case 1:
		turn_on_scc3_rts();
		break;
	case 2:
		turn_on_scc4_rts();
		break;
	}
}

void disable_rts_on_scc(int port_number)
{
	switch (port_number)
	{
	case 0:
		turn_off_scc2_rts();
		break;
	case 1:
		turn_off_scc3_rts();
		break;
	case 2:
		turn_off_scc4_rts();
		break;
	}
}

#ifdef DEBUG
STATIC enum TEST check_if_rx_descriptor_valid(USHORT port_number, SERIAL_RXTX_DESCRIPTOR *sptr_rx_descriptor)
{
	int i;


	for (i = 0; i < MAX_SERIAL_RX_DESCRIPTORS; i++)
	{
		if (&serial.rx_descriptor[port_number][i] == sptr_rx_descriptor)
			return PASS;
	}

	return FAIL;
}
#endif /* DEBUG */

/* -- END CODE ------------------------------------------------------------ */

