/*---------------------------------------------------------------------------
File        : FLSH.C
Author      : Sachin S Desai
Date        : 8th May 1996
Description : This is the FLASH WRITE SCHEDULER
	Any application wanting to write to the flash simply calls
	schedule_flash_write () with the source address pointer, destination
	address pointer, write-complete function pointer, etc.. The info. gets
	queued in a queue, maintained by the scheduler. From the 10 ms timer
	another function in the scheduler, write_next_page() is called. It is up
	to this function to write the next page, update the queue, etc..
		There are a few problems here. If the application queues up a buffer,
	larger than the page size, the write_next_page () routine has to take care
	to see that only one page is written. But it needs to update the pointers
	and the length feilds and NOT update head and tail indecies of the queue.
	It is only after the last page is written out of the block that the buffer
	is freed and the corresponding write_complete function is called.
		Then, there is the other side of the problem. If an application wants
	to write a buffer with size smaller than the flash-page-size, we have again
	run into hot waters. It is possible if we have flashes of page size 1K...
	We have two solutions in such cases. We could write only the queued up
	buffer (copy the rest of the page into a temporary buffer, complete it with
	the buffer to be written and then write the page. Another solution which
	has lot of headaches is to wait till the buffer builds upto the size of
	a page and then write. Visibly, this will have lot of problems like maintaining
	pointers to all buffers, decision on how long to wait for all buffers that
	comprise a flash page to come, etc..

	change : chetan : 21st July 1997
		Checking for Invalid memory locations
		get_flash_number_from_address changed
---------------------------------------------------------------------------*/
#define FLASH_MANAGER_DEBUG 1

unsigned short crctab[256] ; 
#define updcrc(cp, crc) (crctab[((crc >> 8) & 0xFF)] ^ (crc << 8) ^ ((cp) & 0xFF))

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <kstart.h>
#include <flashmgr.h>
#include "..\store\boot.h"

struct FLASH_WRITE_SCHEDULE_CLASS flash_write_scheduler ;
char flash_write_scheduler_inited = 0 ;
enum FLASH_ACCESSIBILITY flash_accessibility[NUMBER_OF_FLASH_PROMS] = {FLASH_ACCESSIBLE, FLASH_ACCESSIBLE} ;
int flash_page_size[NUMBER_OF_FLASH_PROMS] ;

extern unsigned short Fl1ProId, Fl2ProId ;

/* sudha 21 Dec 1998.
Introduced a new variable -> buswidth of ulong type to indicate whether boot
is 8 bit or 16 bit, based on which new_write_to_flash can be done accordingly
as byte write or word write to flash. */

extern void new_write_to_flash (char *cptr_src, char *cptr_dest, int number_of_bytes, ULONG BusWidth) ;


static int get_flash_number_from_address (unsigned char *address)
{
/* NEW_BOOT */
#if 1
	return ((int) ((unsigned long) address - (unsigned long) FL_ROMM_START) / (unsigned long) FL_PROM_SIZE) ;
#else
	return ((int) ((unsigned long)address / (unsigned long)0x80000)) ;
#endif
/* NEW_BOOT */
}


/*
	Note :
		This is written to work around the problem of having to disable interrupts
		during flash_write. It is VERY, VERY important that this routine be called
		from the 10 ms timer only. The difference between write_to_flash and
		new_write_to_flash is that in new_write_to_flash, we do not wait for 10 ms
		after loading the device, hoping that there is no flash access for the next
		10 ms. Since FLASH-write is done only from here, so far so good. What is
		to be kept in mind is that, when FLASH-write is going on, there cannot be a
		FLASH-read at all.
			It has been arranged to see that new_c_write_to_flash() is never called
		with number_of_bytes > page-size. So the extra code can be carefully taken
		out. No other process has any business to call this routine except
		write_next_page(). This is very, very, very... important. Besides, it cannot
		be called with block length greater than the page size.
*/
static int new_c_write_to_flash (char *cptr_src, char *cptr_dest, int number_of_bytes)
{
	unsigned long to_flash ;
	unsigned short flash_prod_id ;

/* sudha 21 Dec 1998...*/
	ULONG BusWidth = 0;
	BYTE boot_ver[10];

	BootConfigType *BootHeader;

	BootHeader = (BootConfigType *) (FL_BOOT_HDR);
	memcpy(boot_ver,BootHeader->Version, 10);

/*	printf("\n\rBoot Version is %s",boot_ver); */
	if ((memcmp(boot_ver,"2.00.8",6) < 0) && (memcmp(boot_ver,"2.00.16",7) < 0))
		BusWidth = 1;
	else
		BusWidth = BootHeader->BootBusWidth;
/* printf("\n\rFLASHMGR : Boot bus width is %lu", BusWidth);*/
/* ...sudha 21 Dec 1998 */

/* NEW_BOOT */
#if 1
	if ((unsigned long) cptr_dest < (unsigned long) FL_CODE_HDR
			|| (unsigned long) cptr_dest + (unsigned long) number_of_bytes > (unsigned long) FL_PROM_END+1)
	{
#if FLASH_MANAGER_DEBUG
		printf ("Alarm : Attempt to write before the code header\n") ;
#endif
		return (0) ;
	}
#else
	if ((unsigned long)cptr_dest < (unsigned long)FL_CODE_HDR)
	{
#if FLASH_MANAGER_DEBUG
		printf ("Alarm : Attempt to write before the code header\n") ;
#endif
		return (0) ;
	}
#endif
/* NEW_BOOT */

	if (number_of_bytes > 512)
		return (0) ;            /* does not write more than 512 bytes in one shot */

	to_flash = (unsigned long) cptr_dest ;
	to_flash &= 0x80000 ;

	if (to_flash)
		flash_prod_id = Fl2ProId ;
	else
		flash_prod_id = Fl1ProId ;

	if (flash_prod_id == 0x1FA4)    /* AT29C040A, page size = 256 */
	{
/* sudha taken from Chris 28 Dec 1998 for 16 bit hardware... */
        if (BusWidth == PROM_WIDTH_16BITS) 
		    new_write_to_flash (cptr_src, cptr_dest, number_of_bytes,BusWidth) ;
        else 
/* Sudha.22 March 1999.Removed the if check for PROM_WIDTH_8BITS in this else 
condition.Because if boot ver is 2.00 & not 2.00.8, buswidth var won't be 
there in the bootheader.And so, script,udb etc., won't get saved thru tftp 
with this prom of boot ver 2.00 even after upgrading the latest bin which 
supports both 8bit & 16bit. */
		  { 
/* ...sudha taken from Chris 28 Dec 1998 for 16 bit hardware */
			if (number_of_bytes > 256)
			{
				new_write_to_flash (cptr_src, cptr_dest, 256,BusWidth) ;
				new_write_to_flash (cptr_src + 256, cptr_dest + 256, number_of_bytes - 256,BusWidth) ;
			}
			else
				new_write_to_flash (cptr_src, cptr_dest, number_of_bytes,BusWidth) ;
		}
	}
	else                            /* AT29C040, page size = 512 */
		new_write_to_flash (cptr_src, cptr_dest, number_of_bytes,BusWidth) ;

	return (number_of_bytes) ;
}



/*
	This function is to be called before scheduling any block to be written
	to the second flash. It initializes the queue-head and tail, flash page
	size, etc.
*/
void init_flash_write_scheduler (void)
{
	int flash_number ;
	BYTE boot_ver[10];
	BootConfigType *ptr_boot_config = (BootConfigType*) FL_BOOT_HDR; /* sudha taken from chris 04 Jan 1999 */

	memcpy(boot_ver,ptr_boot_config->Version,10);
/*	printf("\n\rBoot Version is %s",boot_ver); */

	flash_write_scheduler.head_of_Q = 0 ;   /* Place to put the new block */
	flash_write_scheduler.tail_of_Q = -1 ;  /* +1 gives the place to take
	                                           next block from */

	for (flash_number = 0 ; flash_number < NUMBER_OF_FLASH_PROMS ; flash_number++)
	{
		flash_accessibility[flash_number] = FLASH_ACCESSIBLE ;
	}

	flash_write_scheduler_inited = 1 ;

	if (Fl1ProId == 0x1FA4)
	{
/* sudha taken from chris 04 Jan 1999... */
        if ((ptr_boot_config->BootBusWidth == PROM_WIDTH_16BITS) &&
		  		(strcmp(boot_ver,"2.00")))
		    flash_page_size[0] = 512 ;/* AT29C040A - 16 bit*/
        else
		    flash_page_size[0] = 256 ;/* AT29C040A - 8 bit*/
   }
/* ...sudha taken from chris 04 Jan 1999 */
	else
		flash_page_size[0] = 512 ; /* AT29C040 */

	if (Fl2ProId == 0x1FA4)
	{
/* sudha taken from chris 04 Jan 1999... */
        if ((ptr_boot_config->BootBusWidth == PROM_WIDTH_16BITS) &&
		  		(strcmp(boot_ver,"2.00")))
		    flash_page_size[1] = 512 ;/* AT29C040A - 16 bit*/
        else
		    flash_page_size[1] = 256 ;/* AT29C040A - 8 bit*/
    }
/* ...sudha taken from chris 04 Jan 1999 */
	else
		flash_page_size[1] = 512 ; /* AT29C040 */

printf("\n\rFLASHMGR: fl1proid is %x",Fl1ProId);
printf("\n\rFLASHMGR: fl2proid is %x",Fl2ProId);

printf("\n\rFLASHMGR: flash page size[0] is %d",flash_page_size[0]);
printf("\n\rFLASHMGR: flash page size[1] is %d",flash_page_size[1]);

	return ;
}




int schedule_flash_write (char *src_ptr, char *dest_ptr, unsigned short block_length,
                          void (*fptr_block_write_complete)(), char *buffer_to_free)
{
	short place_to_write_new_block ;
	FLASH_WRITE_BLOCKS *ptr_flash_write_block ;

	if (flash_write_scheduler_inited == 0)
	{
		printf ("Flash write scheduler not yet initialized\n") ;
		return (-1) ;
	}

#if FLASH_MANAGER_DEBUG

/* NEW_BOOT */
#if 1
	if ((block_length > 512) 
		|| ((unsigned long) dest_ptr < (unsigned long) FL_CODE_HDR)
			|| (unsigned long) (dest_ptr + (unsigned long) block_length) > (unsigned long) (FL_PROM_END+1))
	{
		printf ("Request for block length of %d bytes to be written at %08X\n",
			block_length, (unsigned long)dest_ptr) ;
		return (-1) ;
	}
#else
	if ((block_length > 512) || ((unsigned long)dest_ptr < (unsigned long)FL_CODE_HDR))
	{
		printf ("Request for block length of %d bytes to be written at %08X\n",
			block_length, (unsigned long)dest_ptr) ;
		return (-1) ;
	}
#endif
/* NEW_BOOT */

#endif

  	place_to_write_new_block = flash_write_scheduler.head_of_Q ;
	if (place_to_write_new_block == flash_write_scheduler.tail_of_Q)
	{
		printf ("\nQueue full, schedule_flash_write failed") ;
		return (-1) ;  /* Queue full */
	}

	ptr_flash_write_block = &flash_write_scheduler.flash_write_blocks[place_to_write_new_block] ;
	ptr_flash_write_block->dest_ptr = dest_ptr ;
	ptr_flash_write_block->src_ptr = src_ptr ;
	ptr_flash_write_block->block_length = block_length ;
	ptr_flash_write_block->buffer_to_free = buffer_to_free ;
	ptr_flash_write_block->fptr_block_write_complete = fptr_block_write_complete ;

	if (place_to_write_new_block == (MAX_FLASH_WRITES_PENDING - 1))
		place_to_write_new_block = 0 ;
	else
		place_to_write_new_block++ ;
	flash_write_scheduler.head_of_Q = place_to_write_new_block ;

#if 0
	printf ("Block of length %d scheduled for write at %08X\n",
	         block_length, dest_ptr) ;
#endif

	return (0) ;
}



/*
	There is a problem here. If a block is queued for flash write and if it
	happens to have a length greater than the page size of the flash, we are
	in with problems. So what this routine then does is simple. It writes a
	single page to the flash and just adjusts the dest-pointer, src-pointer
	and length. When the last of such pages in that block is written,
		(a). it frees the source buffer (if need be)
		(b). it calls the flash_write_complete routine.
*/
extern enum BOOLEAN user_database_updated ;
extern int tftp_udb_write_complete ;
int write_next_page (void)
{
	short next_item = flash_write_scheduler.tail_of_Q ;
	FLASH_WRITE_BLOCKS *ptr_block ;
	int flash_number = get_flash_number_from_address ((unsigned char *)FL_UDB_HDR) ;

	if (flash_write_scheduler_inited == 0)
		return (-1) ;

	if (flash_accessibility[flash_number] == FLASH_WRITE_COMPLETE)
	{
		if (tftp_udb_write_complete == 1)
		{
			tftp_udb_write_complete = 0 ;
			user_database_updated = TRUE ;
			printf ("\nUser Database updated") ;
		}
	}

	/*
		If control comes here, it means it has been atleast 10 ms since
		the flash page write for ANY flash prom was done. So mark
		the accessibility of all flashes as ACCESSIBLE.
	*/
	for (flash_number = 0 ; flash_number < NUMBER_OF_FLASH_PROMS ; flash_number++)
		flash_accessibility[flash_number] = FLASH_ACCESSIBLE ;
	
	if (next_item == (MAX_FLASH_WRITES_PENDING - 1))
		next_item = 0 ;
	else
		next_item++ ;

	if (next_item == flash_write_scheduler.head_of_Q)
	{
		return (-1) ;  /* Queue empty */
	}

	ptr_block = &flash_write_scheduler.flash_write_blocks[next_item] ;
	flash_number = get_flash_number_from_address (ptr_block->dest_ptr) ;

	if (flash_number >= NUMBER_OF_FLASH_PROMS)
	{
		if (flash_number >= NUMBER_OF_FLASH_PROMS)
#if FLASH_MANAGER_DEBUG
		{
			printf ("Alarm : Attempt to write to flash number %d at %08X\n",
			         flash_number, ptr_block->dest_ptr) ;
		}
#endif

		/*
			Free up the block, call the completion
			routine and update the queue tail index
		*/

		if (ptr_block->buffer_to_free)
			free (ptr_block->buffer_to_free) ;

		flash_write_scheduler.tail_of_Q = next_item ;

		if (ptr_block->fptr_block_write_complete)
			(*ptr_block->fptr_block_write_complete)() ;

		return (-1) ;
	}

	flash_accessibility[flash_number] = FLASH_WRITE_IN_PROGRESS ;
	new_c_write_to_flash (ptr_block->src_ptr, ptr_block->dest_ptr, flash_page_size[flash_number]) ;

	if (ptr_block->block_length > flash_page_size[flash_number])
	{
		ptr_block->src_ptr += flash_page_size[flash_number] ;
		ptr_block->dest_ptr += flash_page_size[flash_number] ;
		ptr_block->block_length -= flash_page_size[flash_number] ;
		return (0) ;
		/* During the next 10 ms timer the next page is written */
	}

	if (ptr_block->buffer_to_free)
		free (ptr_block->buffer_to_free) ;

	flash_write_scheduler.tail_of_Q = next_item ;

	if (ptr_block->fptr_block_write_complete)
		(*ptr_block->fptr_block_write_complete)() ;

	return (0) ;
}

/* Sachin 16/12/1996 */

#define P 0x8408
void	calculate_crc_table (void)
{
	register	unsigned short b, v;
	register unsigned short i;

	for (b = 0 ; ; )
	{
		v = b ;
		for (i = 8 ; i-- ; )
			v = v & 1  ?  (v >> 1) ^ P : v >> 1 ;

		crctab[b] = v & 0xffff ;
		if (++b == 256)
			break ;
	}
}


unsigned short update_crc(unsigned short accumulated_crc, unsigned char *packet, unsigned long packet_length)
{
	unsigned long i;
	for (i = 0 ; i < packet_length ; i++)
		accumulated_crc = updcrc(0xFF & packet[i], accumulated_crc);

	return (accumulated_crc) ;
}

/* Sachin 16/12/1996 */

