#include	"defs.h"
/*---------------------------------------------------------------------------
	file	:	log.c
	Auth	:	vidy
	Date	:	30th Jan 98
	Synop	:	Contains the routines for the log abstract data type
	Assumptions :
	1. Log can log only message strings with a null termination.
	2. This is for logging events which are not-too-fast. SO it
	assumes that the log read is a much faster operation than log write.
	If this is not the case the read pointers may become invalid during
	a read process as the circular buffer eats up the head
	3. log functions are not to be used from ISRs
---------------------------------------------------------------------------*/
#include	<stdlib.h>
#include <incall.h>
#include	<string.h>
#include	"..\applicat\telnet\menusup.h"	/* has get_time_in_hms() */

#include "log.h"
#include <logif.h>

unsigned long log_size;			/* log file size */
int	log_enabled = 1;			/* flag to turn on/off logging dynamic */
int	time_stamp_log = 1;

static log_descriptor_type log_descriptors[LOG_MAX_OPENS];
static char *log_buffer;		/* pointer to the log file area start */
static char *log_head_ptr;		/* current log file head pointer */
static char *log_tail_ptr;		/* current log file tail pointer */
static unsigned long log_number_of_msgs;		/* number of log msgs */

/*---------------------------------------------------------------------------
allocate memory of log_file_size. init the area, head and tail pointers.
return true if everything is fine. otherwise false. Also make all
log_descriptors invalid
To be called only once during init time
---------------------------------------------------------------------------*/
int	init_log(void)
{
	int index;

	if (!log_size)
		log_size = LOG_MIN_FILE_SIZE;

	log_buffer = (char *)malloc(log_size);
	if (log_buffer == NULL)
	{
		log_enabled = FALSE;		/* we can't have logging */
		return FALSE;
	}
	log_head_ptr = log_tail_ptr = log_buffer;
	log_number_of_msgs = 0;		/* number of log msgs */

	for (index = 0; index < LOG_MAX_OPENS; index++)
	{
		log_descriptors[index].valid = 0;
	}
	return TRUE;
}

/*---------------------------------------------------------------------------
this will just rest the log head and tail pointers and invalidate the
log_descriptors
---------------------------------------------------------------------------*/
void clean_log(void)
{
	int index;

	log_head_ptr = log_tail_ptr = log_buffer;
	log_number_of_msgs = 0;		/* number of log msgs */
	for (index = 0; index < LOG_MAX_OPENS; index++)
	{
		log_descriptors[index].valid = 0;
	}
}

/*---------------------------------------------------------------------------
open_log: open log file for read. returns the index of log_descriptor
sets the read_ptr to log_head_ptr
---------------------------------------------------------------------------*/
int open_log(void)
{
	int	index = 0;

	if (!log_enabled)
		return LOG_NOT_LOGGING;

	for(index = 0; index < LOG_MAX_OPENS; index++)
	{
		if (log_descriptors[index].valid == FALSE)
		{
			log_descriptors[index].valid = TRUE;
			log_descriptors[index].read_ptr = log_head_ptr;
			return index;
		}
	}
	return LOG_ERR_MAX_OPEN;
}

/*---------------------------------------------------------------------------
just make the descriptor invalid
---------------------------------------------------------------------------*/
int close_log(int log_descriptor)
{
	if (!log_enabled)
		return LOG_NOT_LOGGING;
	log_descriptors[log_descriptor].valid = FALSE;
}

/*---------------------------------------------------------------------------
get the time stamp to be put into the log
---------------------------------------------------------------------------*/
static int get_log_time_stamp(char *buffer)
{
	char tbuf[40];

	get_time_in_hms(lsl.system_mib.sysUpTime/100, tbuf);
	sprintf(buffer, "%c%s-", CHAR_NEW_LINE, tbuf);
	return strlen(buffer);

}


/*---------------------------------------------------------------------------
make some space in log_buffer by advancing the log_head_pointer. Makesure
that all the read pointers in between are invalidated
---------------------------------------------------------------------------*/
static void make_space_in_log_buffer(void)
{
	char *old_head_ptr;
	int index;

	old_head_ptr = log_head_ptr;	/* keep track of the old_head_ptr */

#if 0
	log_head_ptr += LOG_FREE_UP_QUANTUM;			/* freeup so much */
	if (log_head_ptr >= (log_buffer + log_size)) /* take care of wrap */
		log_head_ptr -= log_size;
#else
	for (index = 0; index < LOG_FREE_UP_QUANTUM; index++)
	{
		if (*log_head_ptr++ == CHAR_NEW_LINE)
			log_number_of_msgs--;

		if (log_head_ptr >= (log_buffer + log_size)) /* take care of wrap */
			log_head_ptr -= log_size;
	}
#endif

	/* avoid partial messages */
	index = 0;	/* just limit max to 200 skips */
	while(*log_head_ptr != CHAR_NEW_LINE && (index++ < 200))
	{
		log_head_ptr++;
		if (log_head_ptr >= (log_buffer + log_size)) /* take care of wrap */
			log_head_ptr -= log_size;
	}

	/* check if some read descriptors became invalid in the process */
	if (old_head_ptr < log_head_ptr)	/* there was no fold back */
	{
		for (index = 0; index < LOG_MAX_OPENS; index++)
		{
			if (log_descriptors[index].valid == TRUE)
				if ((log_descriptors[index].read_ptr >= old_head_ptr) &&
						(log_descriptors[index].read_ptr < log_head_ptr))
					log_descriptors[index].valid = FALSE;
		}
	}
	else
	{	/* the log_head_ptr has folded back */
		for (index = 0; index < LOG_MAX_OPENS; index++)
		{
			if (log_descriptors[index].valid == TRUE)
			{
				if ((log_descriptors[index].read_ptr >= old_head_ptr) ||
						(log_descriptors[index].read_ptr < log_head_ptr))
					log_descriptors[index].valid = FALSE;
			}
		}
	}
}

/*---------------------------------------------------------------------------
memcpy the given buffer to the log buffer. take care of the wrap around
update the log_tail_ptr. (Space availablity is assumed)
---------------------------------------------------------------------------*/
static void copy_buffer_to_log(char *buffer, int copy_len)
{
	int first_copy_len;

	if (!copy_len)
		return;

	if ((log_tail_ptr + copy_len) <= (log_buffer + log_size))
	{
		memcpy(log_tail_ptr, buffer, copy_len);
	}
	else
	{
		first_copy_len = log_buffer + log_size - log_tail_ptr;
		memcpy(log_tail_ptr, buffer, first_copy_len);
		memcpy(log_buffer, buffer + first_copy_len, copy_len - first_copy_len);
	}

	/* increment and adjust the log_tail_ptr */
	log_tail_ptr += copy_len;
	if (log_tail_ptr >= (log_buffer + log_size))
		log_tail_ptr -= log_size;

}


/*---------------------------------------------------------------------------
write_log: write a null terminated string to log file
algo:
	get the length of the message with strlen.
	if wraps around the end of storage
		use 2 memcpys to copy the message into log_buffer.
	else
		use one memcopy to copy the message into log_buffer.
	if tail pointer cross the head pointer
		advance head pointer
		if head surpasses any log_descriptor pointers
			invalidate the log_descriptor
---------------------------------------------------------------------------*/
void write_log(char *message)
{
	int	msg_len;
	int	free_size, time_stamp_len;
	char	time_buffer[30];
	char	*str_ptr;

	if (!log_enabled)
		return;
	if ((str_ptr = strchr(message, CHAR_NEW_LINE)))
		*str_ptr = NULL;

	if ((str_ptr = strchr(message, '\r')))
		*str_ptr = NULL;

	msg_len = strlen(message);
	if (msg_len <= 0)
		return;

	if (msg_len > (log_size - LOG_MIN_FREE_SIZE))		/* just a check */
		return;
														 
	log_number_of_msgs++;		/* number of log msgs */

	free_size = (log_tail_ptr >= log_head_ptr) ?
					(log_size - (log_tail_ptr - log_head_ptr)) :
					(log_head_ptr - log_tail_ptr);
					/* in fact free_size is less by one */

	/* if free space is not enough, get some more */
	if (free_size < (msg_len + LOG_MIN_FREE_SIZE))
		make_space_in_log_buffer();

	/* prepend time if needed here only */
	if (time_stamp_log)
	{
		time_stamp_len = get_log_time_stamp(time_buffer);
		copy_buffer_to_log(time_buffer, time_stamp_len);
	}

	/* now copy the message into the buffer */
	copy_buffer_to_log(message, msg_len);

}


/*---------------------------------------------------------------------------
read_log: read from the log file "count" number of bytes
input :	int log_desc - log descriptor
			char * buffer - buffer to copy read info
			count - max read count;
Algo
	derrive from the write_log
	returns count of chars copied into buffer
	advances the read pointer (in log_descriptor) by number of chars read.
---------------------------------------------------------------------------*/
int read_log(int log_descriptor, char *buffer, int count)
{
	int log_count;
	char *read_ptr;

	if (!log_enabled)
		return LOG_NOT_LOGGING;

	if (!log_descriptors[log_descriptor].valid)
		return LOG_INVALID_DESCRIPTOR;

	read_ptr = log_descriptors[log_descriptor].read_ptr;

	/* find the maximum readable count */
	if (read_ptr <= log_tail_ptr)
		log_count = log_tail_ptr - read_ptr;
	else
		log_count = log_size - (read_ptr - log_tail_ptr);

	if (count > log_count)
		count = log_count;
	
	if (!count)
		return 0;	/* can't read anything */

	/* memcpy the buffer to the destination buffer */
	if ((read_ptr + count) <= (log_buffer + log_size))
	{
		memcpy(buffer, read_ptr, count);
	}
	else
	{
		int first_copy_count;

		first_copy_count = log_buffer + log_size - read_ptr;
		memcpy(buffer, read_ptr, first_copy_count);
		memcpy(buffer + first_copy_count, log_buffer, count - first_copy_count);
	}
	/* advance and foldback the read pointer for this descriptor */
	read_ptr += count;
	if (read_ptr >= log_buffer + log_size)
		read_ptr -= log_size;
	log_descriptors[log_descriptor].read_ptr = read_ptr;

	return count;
}

/*---------------------------------------------------------------------------
will return the approximate number of bytes and number of lines
present in the log at the moment
---------------------------------------------------------------------------*/
int get_info_log(int *approx_total_size, int *approx_number_of_msgs)
{
	if (!log_enabled)
		return LOG_NOT_LOGGING;

	*approx_number_of_msgs = log_number_of_msgs;

	if (log_tail_ptr >= log_head_ptr)
		*approx_total_size = (log_tail_ptr - log_head_ptr);
	else
		*approx_total_size = (log_size - (log_head_ptr - log_tail_ptr));

	return TRUE;
}
/***********************     last line     ***********************/
