#include	"defs.h"
/*
 * $Log: /IP/RIPREPPR.C $
 * 
 * 1     1/23/96 10:52p Titus
 * Revision corresponds to IP Release 1.31
 */
/************************************************************************/
/*	Copyright (C) 1994 RouterWare, Inc.												*/
/*	Unpublished - rights reserved under the Copyright Laws of the			*/
/*	United States.  Use, duplication, or disclosure by the 					*/
/*	Government is subject to restrictions as set forth in 					*/
/*	subparagraph (c)(1)(ii) of the Rights in Technical Data and 			*/
/*	Computer Software clause at 252.227-7013.										*/
/*	RouterWare, Inc., 3961 MacArthur Blvd. Suite 212, Newport Beach Ca   */
/************************************************************************/
#include <string.h>
#include "ip.h"
/****************************************************************************/
static enum BOOLEAN check_if_another_route_with_a_lesser_metric_already_exists_to_that_network (
	RIP_ROUTE_ENTRY *sptr_rip_entry);
static enum BOOLEAN check_if_rip_entry_is_route_to_itself (ULONG sender_ip_address, RIP_ROUTE_ENTRY *sptr_packet_route_entry);
static void	add_incoming_link_cost_to_metric (USHORT rx_port_number, RIP_ROUTE_ENTRY *sptr_packet_route_entry);
static USHORT add_entry_to_routing_table (USHORT rx_port_number, ULONG sender_ip_address,
	RIP_ROUTE_ENTRY *sptr_packet_route_entry, BYTE bits);
static USHORT process_current_route (USHORT rx_port_number, ULONG sender_ip_address, RIP_ROUTE_ENTRY *sptr_packet_route_entry,
	IP_ROUTE_ENTRY *sptr_route_entry);
static USHORT process_rip_entry_when_current_route_is_through_same_gateway (USHORT rx_port_number, ULONG sender_ip_address,
	RIP_ROUTE_ENTRY *sptr_packet_route_entry, IP_ROUTE_ENTRY *sptr_route_entry);
static USHORT process_rip_entry_when_current_route_is_through_a_different_gateway (USHORT rx_port_number,
	ULONG sender_ip_address, RIP_ROUTE_ENTRY *sptr_packet_route_entry, IP_ROUTE_ENTRY *sptr_route_entry);
static ULONG get_network_broadcast_address_for_target (ULONG target);
/****************************************************************************/
/* return: 1  if cause route table change.
			  0  no change on route table. */
USHORT process_rip_entry (USHORT rx_port_number, ULONG sender_ip_address, RIP_ROUTE_ENTRY *sptr_rip_entry)
{
	BYTE bits;
	USHORT return_value;
	IP_ROUTE_ENTRY *sptr_route_entry;
	ULONG mask;
	ULONG target_network;

	return_value = 0x0000;

	/* printf ("Port %d, sender %08X, target %08X\n", rx_port_number, sender_ip_address, sptr_rip_entry->target) ; */

	if (rip_check_if_target_is_a_valid_host_or_network_address (sptr_rip_entry->target) == FALSE)
		{
		/* printf ("rip_check_if_target_is_a_valid_host_or_network_address() failed\n") ; */
		return (return_value);
		}

	if (check_if_rip_entry_is_route_to_itself (sender_ip_address, sptr_rip_entry) == TRUE)
		{
		/* printf ("check_if_rip_entry_is_a_route_to_itself() succeeded\n") ; */
		return (return_value);
		}

	sptr_route_entry = rip_search_route_table_with_bits (sptr_rip_entry->target, 32);

	if (sptr_route_entry == NULL)
		{
		if (ip.rip.port[rx_port_number].route_summarization_enabled == FALSE)
			{
			mask = ip_get_network_mask (sptr_rip_entry->target);

			target_network = sptr_rip_entry->target & mask;

			if (sptr_rip_entry->target != target_network)
				{
				bits = ip_convert_mask_to_bits (mask);

				sptr_route_entry = rip_search_route_table_with_bits (target_network, bits);
				}
			}

		if (sptr_route_entry == NULL)
			{
			bits = ip_determine_number_of_bits_in_subnet_mask (&sptr_rip_entry->target, rx_port_number,
				(BYTE) (sptr_rip_entry->metric + ip.port[rx_port_number].config.metric));

			if ((ip.rip.port[rx_port_number].route_summarization_enabled == TRUE) && ((bits == SUBNET_MASK_NOT_KNOWN) ||
				(bits == SEND_ADDRESS_MASK_REQUEST) || (bits == AWAITING_ADDRESS_MASK_REPLY)))
				{
				return (return_value);
				}

			sptr_route_entry = rip_search_route_table_with_bits (sptr_rip_entry->target, bits);

			if (sptr_route_entry == NULL)
				{
				if (check_if_another_route_with_a_lesser_metric_already_exists_to_that_network (sptr_rip_entry) == TRUE)
					{
					/* printf ("check_if_another_route_with_lesser_metric_al...() succeeded\n") ; */
					return (return_value);
					}

				if (sptr_rip_entry->metric < INFINITY_METRIC_VALUE)
					{
					add_incoming_link_cost_to_metric (rx_port_number, sptr_rip_entry);

					return_value = add_entry_to_routing_table (rx_port_number, sender_ip_address, sptr_rip_entry, bits);

					/* printf ("Added route with target %08X, port %d, bits %d\n", */
					         /* sptr_rip_entry->target, rx_port_number, bits) ; */
					}

				return (return_value);
				}
			}
		}

	if ((sptr_route_entry->flags.private_route == TRUE) || (sptr_route_entry->flags.local_interface_route == TRUE))
		{
		return (return_value);
		}

	return_value = process_current_route (rx_port_number, sender_ip_address, sptr_rip_entry, sptr_route_entry);

	return (return_value);
}
/****************************************************************************/
static enum BOOLEAN check_if_another_route_with_a_lesser_metric_already_exists_to_that_network (
	RIP_ROUTE_ENTRY *sptr_rip_entry)
{
	ULONG new_target;
	ULONG netmask;
	IP_ROUTE_ENTRY *sptr_ip_route_entry;

	new_target = sptr_rip_entry->target;

	netmask = ip_get_network_mask (new_target);

#ifdef ADD_IP_DEFAULT_ROUTE_TO_ROUTE_LIST
	sptr_ip_route_entry = get_first_ip_route_entry ();
#else
	sptr_ip_route_entry = (IP_ROUTE_ENTRY *) get_pointer_to_first_entry_in_list ((LINK *) &ip.route_list);
#endif

	while (sptr_ip_route_entry != NULL)
		{
/* Sachin 12/06/1996 */
		if ((sptr_ip_route_entry->target & netmask) == new_target)
/* Sachin 12/06/1996 */
		/* if ((sptr_ip_route_entry->target & netmask) == new_target) */
		if ((sptr_ip_route_entry->target & sptr_ip_route_entry->mask) == new_target)
			{
			if ((sptr_ip_route_entry->flags.local_interface_route == TRUE) && (sptr_ip_route_entry->flags.route_down == FALSE))
				{
				return (TRUE);
				}
			}

#ifdef ADD_IP_DEFAULT_ROUTE_TO_ROUTE_LIST
		sptr_ip_route_entry = get_next_ip_route_entry (sptr_ip_route_entry);
#else
		sptr_ip_route_entry = (IP_ROUTE_ENTRY *) get_pointer_to_next_entry_in_list ((LINK *) sptr_ip_route_entry);
#endif
		}

	return (FALSE);
}
/****************************************************************************/
static enum BOOLEAN check_if_rip_entry_is_route_to_itself (ULONG sender_ip_address, RIP_ROUTE_ENTRY *sptr_rip_entry)
{
	/* Don't ever add a route to oneself through another router */

#ifndef __IP_DEBUG__
	PARAMETER_NOT_USED (sender_ip_address);
#endif /* __IP__DEBUG__ */

	if (ip_match_full_address_to_port (sptr_rip_entry->target) != NO_SUCH_PORT)
		{
	#ifdef __IP_DEBUG__
		ip_printf (IP_RIP_PRINTF, "IP/RIP: response packet has route to myself %s from %s metric %d\n",
			convert_ip_address_to_dot_format (&ip.print_buffer[0], sptr_rip_entry->target),
			convert_ip_address_to_dot_format (&ip.print_buffer_2[0], sender_ip_address), sptr_rip_entry->metric);
	#endif /* __IP__DEBUG__ */

		return (TRUE);
		}

	return (FALSE);
}
/****************************************************************************/
static void	add_incoming_link_cost_to_metric (USHORT rx_port_number, RIP_ROUTE_ENTRY *sptr_packet_route_entry)
{
	/* add metric by incoming link cost */

	sptr_packet_route_entry->metric += ip.port[rx_port_number].config.metric;

	if (sptr_packet_route_entry->metric > INFINITY_METRIC_VALUE)
		{
		sptr_packet_route_entry->metric = INFINITY_METRIC_VALUE;
		}
}
/****************************************************************************/
/* check if we can reach the destination through the same gateway. If yes, there is no need to add an entry.
 *	return: 1  if cause route table change.
 *			  0  no change on route table. */

static USHORT add_entry_to_routing_table (USHORT rx_port_number, ULONG sender_ip_address,
	RIP_ROUTE_ENTRY *sptr_packet_route_entry, BYTE bits)
{
	IP_ROUTE_ENTRY *sptr_route_entry;
	ROUTE_FLAGS flags;

	memset (&flags, 0, sizeof (ROUTE_FLAGS));

	sptr_route_entry = add_ip_route_entry (sptr_packet_route_entry->target, bits, sender_ip_address, rx_port_number,
		sptr_packet_route_entry->metric, &flags);

	if (sptr_route_entry != NULL)
		{
		sptr_route_entry->flags.route_changed = TRUE;

		sptr_route_entry->ipRouteProto = RIP_PROTOCOL_TYPE;
		}

	return (0x0001);
}
/****************************************************************************/
static USHORT process_current_route (USHORT rx_port_number, ULONG sender_ip_address, RIP_ROUTE_ENTRY *sptr_packet_route_entry,
	IP_ROUTE_ENTRY *sptr_route_entry)
{
	USHORT return_value;

	return_value = 0x0000;

	add_incoming_link_cost_to_metric (rx_port_number, sptr_packet_route_entry);

	if ((ip.rip.port[rx_port_number].route_holddown_enabled == TRUE) && (sptr_route_entry->metric == INFINITY_METRIC_VALUE))
		{
		/* ignore the route in hold-down */

	#ifdef __IP_DEBUG__
		ip_printf (IP_RIP_PRINTF, "IP/RIP: existing route is in hold-down: %s %lu\n",
			convert_ip_address_to_dot_format (&ip.print_buffer[0], sptr_packet_route_entry->target),
			sptr_packet_route_entry->metric);
	#endif /* __IP__DEBUG__ */

		return (return_value);
		}
	else if ((sptr_route_entry->gateway == sender_ip_address) && (sptr_route_entry->port_number == rx_port_number))
		{
		return_value = process_rip_entry_when_current_route_is_through_same_gateway (rx_port_number, sender_ip_address,
			sptr_packet_route_entry, sptr_route_entry);
		}
	else
		{
		return_value = process_rip_entry_when_current_route_is_through_a_different_gateway (rx_port_number, sender_ip_address,
			sptr_packet_route_entry, sptr_route_entry);
		}

	sptr_route_entry->ipRouteProto = RIP_PROTOCOL_TYPE;

	return (return_value);
}
/****************************************************************************/
static USHORT process_rip_entry_when_current_route_is_through_same_gateway (USHORT rx_port_number, ULONG sender_ip_address,
	RIP_ROUTE_ENTRY *sptr_packet_route_entry, IP_ROUTE_ENTRY *sptr_route_entry)
{
	ULONG mask;

	/* this route was received from the same gateway before */

	if (sptr_route_entry->metric != sptr_packet_route_entry->metric)
		{
	#ifdef __IP_DEBUG__
		ip_printf (IP_RIP_PRINTF, "IP/RIP: route to %s change from same gateway, metric: %lu -> %lu\n",
			convert_ip_address_to_dot_format (&ip.print_buffer[0], sptr_packet_route_entry->target),
			sptr_route_entry->metric, sptr_packet_route_entry->metric);
	#endif /* __IP__DEBUG__ */

		change_route_entry (sptr_route_entry, sender_ip_address, sptr_packet_route_entry->metric, rx_port_number);

		return (0x0001);
		}
	else
		{
		sptr_route_entry->aging_timer = 0x0000;
		}

	if (sptr_route_entry->flags.no_subnet_mask == TRUE)
		{
		mask = ip_get_network_mask (sptr_packet_route_entry->target);

		sptr_route_entry->number_of_subnet_mask_bits = ip_convert_mask_to_bits (mask);

		sptr_route_entry->target = sptr_route_entry->target & mask;
		sptr_route_entry->mask = mask;
		sptr_route_entry->flags.no_subnet_mask = FALSE;
		sptr_route_entry->flags.route_down = FALSE;
		}

	return (0x0000);
}
/****************************************************************************/
static USHORT process_rip_entry_when_current_route_is_through_a_different_gateway (USHORT rx_port_number,
	ULONG sender_ip_address, RIP_ROUTE_ENTRY *sptr_packet_route_entry, IP_ROUTE_ENTRY *sptr_route_entry)
{
	/* Entry is from a different gateway than the current route */

	if ((sptr_packet_route_entry->metric < sptr_route_entry->metric) ||
		((sptr_packet_route_entry->metric == sptr_route_entry->metric) &&
		/* (sptr_route_entry->aging_timer > (ip.rip.config.route_aging_timeout_value / 2)) && */
/* Vidy 040696. Due to DOD route_aging_timeout_value could be zero */
		(sptr_route_entry->aging_timer >= (ip.rip.config.route_aging_timeout_value / 2)) &&
		(sptr_packet_route_entry->metric < INFINITY_METRIC_VALUE)))
		{
		/* Switch to a new gateway */

	#ifdef __IP_DEBUG__
		ip_printf (IP_RIP_PRINTF, "IP/RIP: changed route to %s: gateway %s -> %s metric %lu -> %lu\n",
			convert_ip_address_to_dot_format (&ip.print_buffer[0], sptr_route_entry->target),
			convert_ip_address_to_dot_format (&ip.print_buffer_2[0], sptr_route_entry->gateway),
			convert_ip_address_to_dot_format (&ip.print_buffer_3[0], sender_ip_address),
			sptr_route_entry->metric, sptr_packet_route_entry->metric);
	#endif /* __IP__DEBUG__ */

		change_route_entry (sptr_route_entry, sender_ip_address, sptr_packet_route_entry->metric, rx_port_number);

		return (0x0001);
		}

	return (0x0000);
}
/****************************************************************************/
void rip_send_icmp_address_mask_request (void)
{
	IP_ROUTE_ENTRY *sptr_route_entry;
	ULONG target_netmask;
	ULONG network_mask;
	IP_ROUTE_ENTRY *sptr_route_entry_next;
	IP_PARAMETERS ip_parameters;
	UNION_ICMP_PARAMETER icmp_option_data;

#ifdef ADD_IP_DEFAULT_ROUTE_TO_ROUTE_LIST
	sptr_route_entry = get_first_ip_route_entry ();
#else
	sptr_route_entry = (IP_ROUTE_ENTRY *) get_pointer_to_first_entry_in_list ((LINK *) &ip.route_list);
#endif

	while (sptr_route_entry != NULL)
		{
		if ((sptr_route_entry->number_of_subnet_mask_bits == SUBNET_MASK_NOT_KNOWN) &&
			(sptr_route_entry->flags.no_subnet_mask == TRUE))
			{
			target_netmask = ip_get_network_mask (sptr_route_entry->target);

#ifdef ADD_IP_DEFAULT_ROUTE_TO_ROUTE_LIST
			sptr_route_entry_next = get_next_ip_route_entry (sptr_route_entry);
#else
			sptr_route_entry_next = (IP_ROUTE_ENTRY *) get_pointer_to_next_entry_in_list ((LINK *) sptr_route_entry);
#endif

			while (sptr_route_entry_next != NULL)
				{
				if ((sptr_route_entry_next->number_of_subnet_mask_bits == SUBNET_MASK_NOT_KNOWN) &&
					(sptr_route_entry_next->flags.no_subnet_mask == TRUE))
					{
					network_mask = ip_get_network_mask (sptr_route_entry_next->target);

					if (target_netmask == network_mask)
						{
						sptr_route_entry_next->number_of_subnet_mask_bits = AWAITING_ADDRESS_MASK_REPLY;
						sptr_route_entry_next->mask = 0x00000000L;
						}
					}

#ifdef ADD_IP_DEFAULT_ROUTE_TO_ROUTE_LIST
				sptr_route_entry_next = get_next_ip_route_entry (sptr_route_entry_next);
#else
				sptr_route_entry_next = (IP_ROUTE_ENTRY *) get_pointer_to_next_entry_in_list ((LINK *) sptr_route_entry_next);
#endif
				}

			memset (&ip_parameters, 0x00, sizeof (IP_PARAMETERS));

			setup_ip_parameters_and_icmp_option_data (sptr_route_entry, &ip_parameters, &icmp_option_data);

			send_icmp_request_message (sptr_route_entry->port_number, &ip_parameters, ICMP_ADDRESS_MASK_TYPE, 0x00,
				&icmp_option_data);

			sptr_route_entry->number_of_subnet_mask_bits = SEND_ADDRESS_MASK_REQUEST;
			sptr_route_entry->mask = 0x00000000L;
			}

#ifdef ADD_IP_DEFAULT_ROUTE_TO_ROUTE_LIST
		sptr_route_entry = get_next_ip_route_entry (sptr_route_entry);
#else
		sptr_route_entry = (IP_ROUTE_ENTRY *) get_pointer_to_next_entry_in_list ((LINK *) sptr_route_entry);
#endif
		}

	ip.icmp.periodic_mask_request_timer = 0x0000;
}
/****************************************************************************/
void setup_ip_parameters_and_icmp_option_data (IP_ROUTE_ENTRY *sptr_route_entry,
	IP_PARAMETERS *sptr_ip_parameters, UNION_ICMP_PARAMETER *uptr_icmp_parameters)
{
	sptr_ip_parameters->version = IP_VERSION;
	sptr_ip_parameters->header_length = MINIMUM_IP_HEADER_LENGTH;
	sptr_ip_parameters->total_length = MINIMUM_IP_HEADER_LENGTH;
	sptr_ip_parameters->time_to_live = (BYTE) ip.mib.ipDefaultTTL;
	sptr_ip_parameters->protocol = ICMP_PROTOCOL;
	sptr_ip_parameters->rx_port_number = sptr_route_entry->port_number;
	sptr_ip_parameters->source_address = ip.port[sptr_route_entry->port_number].config.ip_address;

	sptr_ip_parameters->destination_address = get_network_broadcast_address_for_target (sptr_route_entry->target);

	sptr_ip_parameters->gateway = sptr_route_entry->gateway;

	uptr_icmp_parameters->address_mask = sptr_ip_parameters->destination_address;
}
/****************************************************************************/
static ULONG get_network_broadcast_address_for_target (ULONG target)
{
	ULONG network_mask;

	network_mask = ip_get_network_mask (target);

	target = target &	network_mask;

	target = target | ~network_mask;

	return (target);
}

