/*
** TCPALIVE.C -- Code related to the connection alive timer
*/

/* General Notes:
**		Implementation follow's Richard Stevens' explanation in Illustrated
**			TCP/IP, Vol. 1. There is a initial connection idle timeout of
**			(default) 15 minutes. Following this probes will be sent at 75
**			second intervals.
*/

#include "rtrstd.h"
#include "all.h"
#include "tcpextrn.h"

/*
** reset_tcp_alive_timer()
**		Can be called from anywhere to prevent the connection from being
**		timed out.
** Params:
**		Ptr to connection record for connection
*/
void
reset_tcp_alive_timer(TCP_PER_CONN *ptr_conn_info)
{
	ptr_conn_info->conn_alive_timer = tcp.conn_idle_time;
	ptr_conn_info->conn_probe_retries = 0;
}


/*
** send_keep_alive_probe()
**		Sends a probe segment to the other end of connection and resets the
**		alive timer for 75 seconds into the future. For a give connection,
**		this keep-alive probing can be retried 10 times after which the
**		connection is terminated.
** Params:
**		Ptr to connection record for connection info
*/
void
send_keep_alive_probe(TCP_PER_CONN *ptr_conn_info)
{
	if (ptr_conn_info->conn_probe_retries >= MAX_ALIVE_PROBE_RETRIES)
	{
		tcp_printf(TCP_ALARM_PRINTF, "TCP: No activity on connection for timeout period -- terminating connection\n");
		tcp_abort(ptr_conn_info, ABORT_ERROR);
		return;
	}

#if defined(DEBUG)
	tcp_printf(TCP_ALARM_PRINTF, "TCP: Keep alive probe # %d\n",
			ptr_conn_info->conn_probe_retries + 1);
#endif /* DEBUG */

	/* Send a segment with an already acknowledged sequence number. The
	** other end will attempt ACK of such segments.
	*/
	send_tcp_control_seg(ptr_conn_info, ptr_conn_info->send_nxt - 1,
			ptr_conn_info->recv_nxt, ACK);

	ptr_conn_info->conn_probe_retries++;
	ptr_conn_info->conn_alive_timer = ALIVE_PROBE_INTERVAL;
}
