/*
** TCPRECV.C -- Handles TCP receive portion
** Modifications : sudha 15-Oct-1999. In receive_tcp_packet() function &
**						 in tcp_options() function, put in fixes taken from 
**						 ras tcp code.
*/

#include "rtrstd.h"
#include "all.h"
#include "tcpextrn.h"

/* Local Prototypes */
TCP_PER_CONN *get_tcp_conn_info(TCP_HEADER *, IP_PARAMETERS *);
void tcp_options(TCP_HEADER *, TCP_PER_CONN *, BYTE *, USHORT);
void tcp_state_switch(TCP_PER_CONN *, TCP_HEADER *);


/*
** General Notes:
**		Connection Record - Entry in the connection information table. There
**			is one record for each connection and gives the current state
**			and progress of the connection. One TCP_PER_CONN entry.
*/

/* 
** receive_tcp_packet()
**		Called by IP code when it detects a TCP packet. This routine is
**		registered with IP on startup.
** Params:
**		Ptr to some IP parameters
**		Ptr to the received IP packet
**		Size of the total TCP packet (including header)
** NOTE:
**		This routine is passed a ptr to a receive packet (buffer). This
**		buffer is released by the IP layer on return from the function. So
**		if needed, the received packet should be copied onto a seperately
**		allocated buffer and queued up waiting for the TCP socket receive.
*/
void 
receive_tcp_packet(USHORT rx_port_number, IP_PARAMETERS *sptr_ip_parameters, 
		BYTE *uptr_ip_rx_packet, USHORT tcp_packet_size)
{
	IP_INFO ip_info;
   TCP_HEADER tcp_header;
	TCP_HEADER *sptr_tcp_pdu;
	TCP_PER_CONN *ptr_conn_info;


	PARAMETER_NOT_USED(rx_port_number);
#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Rx TCP pkt source=%s, destination=%s, size=%u\n",
			convert_ip_address_to_dot_format(ip.print_buffer, 
				sptr_ip_parameters->source_address), 
			convert_ip_address_to_dot_format(&ip.print_buffer[64],
				sptr_ip_parameters->destination_address),
			tcp_packet_size);
#endif /* DEBUG */

	++tcp.mib.tcpInSegs;

	if (tcp_packet_size < sizeof(TCP_HEADER))
	{
		++tcp.mib.tcpInErrs;
		tcp_printf(TCP_ALARM_PRINTF, "TCP: Received runt segment\n");
		return;
	}

	sptr_tcp_pdu = (TCP_HEADER *) ((ULONG) uptr_ip_rx_packet + 
			sizeof(UNION_MAC_HEADER) + sptr_ip_parameters->header_length);

	if (verify_tcp_checksum(sptr_ip_parameters, sptr_tcp_pdu, 
								tcp_packet_size) != 0x0000)
	{
		++tcp.mib.tcpInErrs;
		tcp_printf (TCP_ALARM_PRINTF,"TCP: Received segment with bad checksum\n");
		return;
	}

	tcp_hdr_ntoh(&tcp_header, sptr_tcp_pdu);

	if ((tcp_header.code_and_hlen.code_and_hdrlen.hlen * 4) < 
			sizeof(TCP_HEADER) || 
			(tcp_header.code_and_hlen.code_and_hdrlen.hlen * 4) > 
			tcp_packet_size)
	{
		++tcp.mib.tcpInErrs;
		tcp_printf(TCP_ALARM_PRINTF, "TCP: Received badly sized segment\n");
		return;
	}

	ptr_conn_info = get_tcp_conn_info(&tcp_header, sptr_ip_parameters);
	if (ptr_conn_info == NULL)
	{
		/* If there's no connection info, as per TCP RFC's send a reset packet
		*/
		ip_info.source_address = sptr_ip_parameters->source_address;
		ip_info.destination_address = sptr_ip_parameters->destination_address;

/* sudha 15-Oct-1999.Taken from ras tcp code... */
/* Satish. 14th October 1998
Fix for landmine program. The actual fix would be check why tcp loopback is not working.
But here we are fixing the symptom instead of the actual problem itself
Why ?? I dont know
*/
		if (ip_info.source_address != ip_info.destination_address)
			tcp_reset(&tcp_header, tcp_packet_size, &ip_info);
/* ...sudha 15-Oct-1999.Taken from ras tcp code */

		tcp_printf(TCP_ALARM_PRINTF, "TCP: Received segment for non-existent socket\n");
		return;
	}

	/* Since a packet was received on the connection, reset the alive timer.
	** No harm in reseting the timer for even bad states.
	*/
	reset_tcp_alive_timer(ptr_conn_info);

	/* Init current segment related info in the connection record */
	/* so that info can be passed down to the other functions without */
	/* having to worry about function parameters */

/* sudhir 28/2/97  address is reversed*/

	ptr_conn_info->seg_ip_params.source_address = 
			sptr_ip_parameters->destination_address;
	ptr_conn_info->seg_ip_params.destination_address = 
			sptr_ip_parameters->source_address;
	tcp_options(&tcp_header, ptr_conn_info, (BYTE *)sptr_tcp_pdu, 
			tcp_packet_size);

	tcp_state_switch(ptr_conn_info, &tcp_header);
}


/*
** get_tcp_conn_info()
**		Returns a pointer to a connection information structure. Error 
**		conditions are checked for (bad info for current state)
** Params:
**		Ptr to the TCP header info (host order)
**		Ptr to IP info
** Returns:
**		Ptr to a the connection info structure if all is well
**		NULL otherwise
** NOTES:
**		This routine tries to find a connection entry in the connection table
**		that matches with the info in the incoming packet for the current
**		state of the connection. Connection entries in the LISTEN state are 
**		special case where only the destination address and port number in
**		the incoming packet are checked.
*/
TCP_PER_CONN *
get_tcp_conn_info(TCP_HEADER *tcp_header, IP_PARAMETERS *sptr_ip_parameters)
{
	USHORT i;
	TCP_PER_CONN *ptr_conn_info, *maybe_conn_info;

	
	maybe_conn_info = NULL;
	ptr_conn_info = tcp.tcp_conn_table;
	for (i = 0; i < tcp.max_tcp_connection; i++, ptr_conn_info++)
	{
		if (ptr_conn_info->tcp_state == FREE)
			continue;

		/* Search for a full match (src port, src ip, dst port, dst ip) first.
		** Later try partial matches.
		*/
		if (ptr_conn_info->local_port == tcp_header->dst_port &&
			ptr_conn_info->remote_port == tcp_header->src_port &&
			ptr_conn_info->local_addr == 
					sptr_ip_parameters->destination_address &&
			ptr_conn_info->remote_addr == 
						sptr_ip_parameters->source_address)
		{
			/* Packet for a valid connection found */
			break;
		}
		if (ptr_conn_info->local_port == tcp_header->dst_port &&
			ptr_conn_info->local_addr == 
											sptr_ip_parameters->destination_address &&
			ptr_conn_info->tcp_state == LISTEN)
		{
			/* Return this as a best effort in case a true match is not found */
			maybe_conn_info = ptr_conn_info;

			/* But continue searching in case there is a better match */
		}
	}

	if (i >= tcp.max_tcp_connection)
	{
		if (maybe_conn_info && IS_SYN_ON(tcp_header))
		{
#if defined(DEBUG)
			tcp_printf(TCP_DEBUG_PRINTF, "TCP: Incoming packet on a LISTEN state found\n");
#endif /* DEBUG */
			return maybe_conn_info;
		}
		else
			return NULL;
	}

	return ptr_conn_info;
}

/*
** tcp_options()
**		Handle options in the received segment (if any). Also set some
**		ptrs and info in the connection record so as to pass on info to
**		other functions.
**	Params:
**		Ptr to the TCP header in host format
**		Ptr to connection record to use
**		Ptr to the received tcp segment
**		Length of tcp packet received
*/
void 
tcp_options(TCP_HEADER *tcp_header, TCP_PER_CONN *ptr_conn_info, 
		BYTE *tcp_rx_seg, USHORT tcp_packet_len)
{
	BYTE *opt_ptr;
	USHORT temp_mss;
	USHORT num_opt_bytes;
	TCP_OPTIONS *ptr_tcp_opt;


	ptr_conn_info->seg_header = tcp_rx_seg;
	ptr_conn_info->temp_mss = DEF_TCP_MSS;

	num_opt_bytes = tcp_header->code_and_hlen.code_and_hdrlen.hlen * 4 - 
								sizeof(TCP_HEADER);
	ptr_conn_info->seg_data = tcp_rx_seg + sizeof(TCP_HEADER) + num_opt_bytes;
	ptr_conn_info->seg_len = tcp_packet_len - sizeof(TCP_HEADER) -
														num_opt_bytes;
		
	if (num_opt_bytes == 0)
	{
		/* no options received */
		ptr_conn_info->seg_options = NULL;
	}
	else
	{
#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Some options received in packet\n");
#endif /* DEBUG */

		opt_ptr = ptr_conn_info->seg_options = tcp_rx_seg + sizeof(TCP_HEADER);

		/* process the options */
		while (num_opt_bytes)
		{
			ptr_tcp_opt = (TCP_OPTIONS *)opt_ptr;
			switch (ptr_tcp_opt->kind)
			{
			case KIND_EOO:
				/* process no more */
				num_opt_bytes = 0;
				break;
			case KIND_NOP:
				opt_ptr += sizeof(ptr_tcp_opt->kind);
				num_opt_bytes -= sizeof(ptr_tcp_opt->kind);
				break;
			case KIND_MSS:
				if (ptr_tcp_opt->opt_union.opt_MSS.len != (sizeof(TCP_MSS_OPTION) + 1))
				{
					tcp_printf(TCP_ALARM_PRINTF, "TCP: Bad option length received on a connection -- resetting connection\n");
					tcp_abort(ptr_conn_info, BAD_LENGTH);
					num_opt_bytes = 0;
					break;
				}
				if (ptr_tcp_opt->opt_union.opt_MSS.len == (sizeof(TCP_MSS_OPTION) + 1))
				{
					temp_mss = net_to_host_short(
											ptr_tcp_opt->opt_union.opt_MSS.max_seg_size);
					if (IS_SYN_ON(tcp_header))
					{
/* sudha 15-Oct-1999. Taken from ras tcp code... */
/* This check was primarly there because IP fragmentation was not proper	*/
/* IP fragmentation now seems to be fixed. SO remove this			*/
/* max_send_mss variable itself is not required though it is maintained for	*/
/* historical reasons								*/
/*
						if (temp_mss > tcp.max_send_mss)
							ptr_conn_info->temp_mss = tcp.max_send_mss;
						else
- Satish. 11th Feb 1997 */
/* ...sudha 15-Oct-1999. Taken from ras tcp code */
							ptr_conn_info->temp_mss = temp_mss;
					}
				}
				opt_ptr += ptr_tcp_opt->opt_union.opt_MSS.len;
				num_opt_bytes -= ptr_tcp_opt->opt_union.opt_MSS.len;
				break;
			default:
				/* ignore all other options for now */
				num_opt_bytes = 0;
				break;
			}
		}
	}

	return;
}

/* 
** tcp_state_switch()
**		Do some front end processing and dispatch to individual state 
**		handlers.
** Params:
**		Ptr to the connection table entry for this packet
**		Ptr to the host format TCP header
*/
void 
tcp_state_switch(TCP_PER_CONN *ptr_conn_info, TCP_HEADER *tcp_header)
{
	TCPSEQ wlast, slast;
	USHORT seglen, recv_wnd;


#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Packet Rx in %s state...",
			get_state_string(ptr_conn_info->tcp_state));
	if (IS_URG_ON(tcp_header))
		tcp_printf(TCP_DEBUG_PRINTF, " URG");
	if (IS_ACK_ON(tcp_header))
		tcp_printf(TCP_DEBUG_PRINTF, " ACK");
	if (IS_PSH_ON(tcp_header))
		tcp_printf(TCP_DEBUG_PRINTF, " PSH");
	if (IS_RST_ON(tcp_header))
		tcp_printf(TCP_DEBUG_PRINTF, " RST");
	if (IS_SYN_ON(tcp_header))
		tcp_printf(TCP_DEBUG_PRINTF, " SYN");
	if (IS_FIN_ON(tcp_header))
		tcp_printf(TCP_DEBUG_PRINTF, " FIN");
	if (IS_FIN_ON(tcp_header) || IS_RST_ON(tcp_header))
		tcp_printf(TCP_DEBUG_PRINTF, ", FIN/RST Seq no. = %d", tcp_header->seq_num);
	tcp_printf(TCP_DEBUG_PRINTF, "\n");	 
#endif /* DEBUG */
	
	switch (ptr_conn_info->tcp_state) 
	{
	case CLOSED:
		/* CLOSED states should not be receiving packets */
		tcp_closed(ptr_conn_info, tcp_header);
		return;
		break;
	case LISTEN:
		/* special processing for listening connections */
		tcp_listen(ptr_conn_info, tcp_header);
		return;
		break;
	case SYN_SENT:
		/* move into the connection established state if possible */
		tcp_syn_sent(ptr_conn_info, tcp_header);
		return;
		break;
	}

	/*---------------------------------------------------------------*/
	/* For remaining states check if the incoming sequence number */
	/* falls within range of the receive window. */
	seglen = ptr_conn_info->seg_len;
	
	/* As per RFC, segment length includes SYN and FIN bits as individual
	** bytes for the purpose of sequence number generation
	*/
	if (IS_SYN_ON(tcp_header))
		seglen++;
	if (IS_FIN_ON(tcp_header))
		seglen++;

	recv_wnd = ptr_conn_info->recv_buf_size - ptr_conn_info->recv_filled;

	/* Code below does sequence number checks for the SEGMENT ARRIVES state 
	** processing as mentioned in the RFC 793. All the if's and else's
	** is so that we do not need goto's....structured programming.
	*/
	if (recv_wnd == 0)
	{
		/* if the receive window is 0, the incoming segment is actually
		** not acceptable -- yet, certain exceptions...
		*/
		if (seglen == 0)
		{
			if (tcp_header->seq_num != ptr_conn_info->recv_nxt)
			{
				if (!IS_RST_ON(tcp_header))
				{
					send_tcp_control_seg(ptr_conn_info, ptr_conn_info->send_nxt, 
							ptr_conn_info->recv_nxt, ACK);
				}
#if defined(DEBUG)
				tcp_printf(TCP_DEBUG_PRINTF, "TCP: Packet discarded -- recv_wnd = 0, seg_len = 0\n");
#endif /* DEBUG */
				return;
			}
		}
		else	/* seglen > 0 */
		{
			/* Adjust the packet length so that the packet can be atleast
			** processed for ACK and RST bits 
			*/
			ptr_conn_info->seg_len = 0;
			/* Send an ACK in response to allow for window probes */
			if (!IS_RST_ON(tcp_header))
			{
				send_tcp_control_seg(ptr_conn_info, ptr_conn_info->send_nxt, 
						ptr_conn_info->recv_nxt, ACK);
			}
		}
	}
	else			/* recv_wnd > 0 */
	{
		/* hopefully the below conditional is coded okay... */

		wlast = ptr_conn_info->recv_nxt + recv_wnd;		/* last wnd seq num */
		slast = tcp_header->seq_num + seglen - 1;			/* last seg seq num */
		if (!(
			(
			/* if segment length is 0, incoming sequence number should be
			** within the receive window
			*/
			((ptr_conn_info->recv_nxt - tcp_header->seq_num) <= 0) &&
			((tcp_header->seq_num - wlast) < 0)
			)
			||
			(
			/* if segment length is non-zero, atleast part of the incoming 
			** sequence number space should be within the receive window
			*/
			((ptr_conn_info->recv_nxt - slast) <= 0) &&
			((slast - wlast) < 0)
			)
			))
		{
			if (!IS_RST_ON(tcp_header))
			{
				send_tcp_control_seg(ptr_conn_info, ptr_conn_info->send_nxt, 
						ptr_conn_info->recv_nxt, ACK);
			}
#if defined(DEBUG)
			tcp_printf(TCP_DEBUG_PRINTF, "TCP: Out of range pack, seq no = %d, ack no = %d, expected seq no = %d\n",
					tcp_header->seq_num, tcp_header->ack_num, ptr_conn_info->recv_nxt);
#endif /* DEBUG */
			return;
		}
	}
	/* Done with checking for sequence numbers ranging */
	/*---------------------------------------------------------------*/

	/* Do some common processing for the remaining states first */
	/* Basically handle RST and SYN segments */
	if (IS_RST_ON(tcp_header))
	{
		tcp_printf(TCP_ALARM_PRINTF, "TCP: Received a RST segment when in %s state -- connection will be reset\n",
						get_state_string(ptr_conn_info->tcp_state));
		switch (ptr_conn_info->tcp_state)
		{
		case SYN_RCVD:
			if (ptr_conn_info->tcp_open == PASSIVE_OPEN)
			{
				/* Deallocate this connection record -- the original LISTEN
				** record will continue.
				*/
/* If a RESET segment comes when we have finished one half of a passive open clear the CONN_PENDING flag of the
** listen record so that we continue to accept connections on that socket
** Satish. Feb 10th 1997.
*/
				if (ptr_conn_info->parent_conn_rec_ptr)
					ptr_conn_info->parent_conn_rec_ptr->conn_flags = 0;
										/* clear all the flags though we are	*/
										/* intereseted only in CONN_PENDING	*/

				tcp_quiet_abort(ptr_conn_info);
				++tcp.mib.tcpAttemptFails;
			}
			else	/* active open */
			{
				/* Need to pass an error message to the application */
				/* ?? Following needs an accept() call to be implemented in the
				** TCP stack. But since this is required only for client ends, we
				** will ignore this for now. ??
				*/
				tcp_abort(ptr_conn_info, CONNECTION_REFUSED_ERROR);
			}
			break;
		case ESTAB:
		case CLOSE_WAIT:
			++tcp.mib.tcpEstabResets;					/* !! Fall thro' !! */
		case FIN_WAIT_1:
		case FIN_WAIT_2:
			tcp_abort(ptr_conn_info, ABORT_ERROR);
			break;
		case CLOSING:
		case LAST_ACK:
		case TIME_WAIT:
			tcp_quiet_abort(ptr_conn_info);
			break;
		default:
#if defined(DEBUG)
			tcp_printf(TCP_DEBUG_PRINTF, "TCP: Pkt received in unknown state -- %d\n",
									ptr_conn_info->tcp_state);
#endif /* DEBUG */
			break;
		}
		return;
	}

	if (IS_SYN_ON(tcp_header))
	{
		tcp_printf(TCP_ALARM_PRINTF, "TCP: Received a SYN segment when in %s state -- connection will be reset\n", 
						get_state_string(ptr_conn_info->tcp_state));

		/* SYN is not acceptable -- abort connection entirely for all states */
		if (ptr_conn_info->tcp_state == ESTAB ||
				ptr_conn_info->tcp_state == CLOSE_WAIT)
		{
			++tcp.mib.tcpEstabResets;
		}
		tcp_abort(ptr_conn_info, ABORT_ERROR);
		return;
	}

	/* For the states handled here, every segment should have an ACK. Drop
	** packets not having an ACK.
	*/
	if (!IS_ACK_ON(tcp_header))
	{
		tcp_printf(TCP_ALARM_PRINTF, "TCP: Dropping segment without ACK in %s state\n", 
				get_state_string(ptr_conn_info->tcp_state));
		return;
	}

#if defined(DEBUG)
	if (IS_FIN_ON(tcp_header))
	{
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: FIN seen in incoming packet...may terminate connection\n");
	}
#endif /* DEBUG */
	tcp_switch[ptr_conn_info->tcp_state](ptr_conn_info, tcp_header);
}

