/*
** TCPSTATE.C -- Code that handles the TCP state machine.
** Modifications : sudha 15-Oct-1999. Put in fixes taken from ras tcp code in
**						 tcp_syn_rcvd() function.
*/

/* General Notes:
**		All functions here handle the received TCP packet in the manner as 
**			specified in RFC 793.
**		Connection Entry - Same as Connection Record; One TCP_PER_CONN
**			entry (see RECV.C).
**		The only app we have now (for version 1.00) is a TELNET server. For
**			a server, we need to make sure that the states and state transitions
**			of CLOSED, LISTEN, SYN_RCVD, ESTAB, CLOSE_WAIT and LAST_ACK are
**			pucca. The initial and final transistion (CLOSED to LISTEN and
**			CLOSE_WAIT to LAST_ACK) is app initiated.
**		Functions to handle states other than LISTEN, CLOSED and SYN_SENT
**			assume that they will not be called if RST and SYN bits are set or
**			if ACK is NOT set.
*/

#include <string.h>
#include <stdlib.h>

#include "rtrstd.h"
#include "all.h"
#include "tcpextrn.h"

/* 
** tcp_closed()
**		Processing for packets rx-ed on connections in the CLOSED state.
**		A RST packet is sent in reply.
**	Params:
**		Ptr to connection entry for current connection
**		Ptr to host format tcp header for segment that came in
*/
void 
tcp_closed(TCP_PER_CONN *ptr_conn_info, TCP_HEADER *tcp_header)
{
#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Packet received in the CLOSED state\n");
#endif /* DEBUG */

	/* Send a RST packet to sender of RST packet */
	if (IS_RST_ON(tcp_header))
		return;

	if (IS_ACK_ON(tcp_header))
	{
		send_tcp_control_seg(ptr_conn_info, tcp_header->ack_num, 0, RST);
	}
	else
	{
		send_tcp_control_seg(ptr_conn_info, 0, 
				tcp_header->seq_num + ptr_conn_info->seg_len, RST | ACK);
	}

#if defined(DEBUG)
	tcp_printf(TCP_PRINTF, "TCP: Packet discarded\n");
#endif /* DEBUG */
	return;
}

/*
** tcp_listen()
**		Processing for packets rx-ed on connections in the LISTEN state.
**		Reaction depends on the code bits in the rx-ed packet. See RFC 793.
**	Params:
**		Ptr to connection entry for current connection
**		Ptr to host format tcp header for segment that came in
*/
void 
tcp_listen(TCP_PER_CONN *ptr_conn_info, TCP_HEADER *tcp_header)
{
	TCP_PER_CONN *listen_conn_record;

#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Pkt Rx in LISTEN state\n");
#endif /* DEBUG */


	if (IS_RST_ON(tcp_header))
	{
		/* RST on listen's should be ignored */
		return;
	}

	if (IS_ACK_ON(tcp_header))
	{
#if defined(DEBUG)
		tcp_printf(TCP_ALARM_PRINTF,"TCP: ACK in a pack received in LISTEN state -- sending RST\n");
#endif /* DEBUG */
		send_tcp_control_seg(ptr_conn_info, tcp_header->ack_num, 0, RST);
		return;
	}

	if (IS_SYN_ON(tcp_header))
	{
		/* Normally this is expected to complete the handshake process */
		/* "passive open" end init */

		/* Fail this listen if the socket record associated with this is
		** in use
		*/
		if (ptr_conn_info->socket_record_ptr->valid_ready_socket == TRUE
			||
			(ptr_conn_info->conn_flags & TCPF_CONNPENDING) != 0)
		{
			/* A previous listen hand-shaking didn't complete. So simply
			** ignore this. If we had implemented backlog queueing, we could
			** have queued this and looked it up later.
			*/
			tcp_printf(TCP_PRINTF, "TCP: Connect request from %s rejected\n", 
					convert_ip_address_to_dot_format(tcp.print_buffer, 
					ptr_conn_info->seg_ip_params.source_address));
#if defined(DEBUG)
			tcp_printf(TCP_DEBUG_PRINTF, "TCP: SYN packet on LISTEN rejected as previous LISTEN busy\n");
#endif /* DEBUG */
			return;
		}
		/* Acquire a new connection record and work from that record so
		** that the LISTEN record can continue listening.
		*/
		listen_conn_record = ptr_conn_info;
		ptr_conn_info = get_new_conn_record(ptr_conn_info);
		if (ptr_conn_info == NULL)
		{
			/* No free records. Treat as soft error and do nothing. Other
			** end will timeout on connect request.
			*/
			tcp_printf(TCP_PRINTF, "TCP: Connect request from %s rejected\n", 
					convert_ip_address_to_dot_format(tcp.print_buffer, 
					ptr_conn_info->seg_ip_params.source_address));
#if defined(DEBUG)
			tcp_printf(TCP_DEBUG_PRINTF, "TCP: SYN packet on LISTEN as conn records full\n");
#endif /* DEBUG */
			return;
		}
		
#if defined(DEBUG)
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: Opening a connection\n");
#endif /* DEBUG */

		/* Prevent use of this listen record until futher notice */
		listen_conn_record->conn_flags |= TCPF_CONNPENDING;

		ptr_conn_info->init_recv_seq_num = tcp_header->seq_num;
		ptr_conn_info->recv_nxt = ptr_conn_info->init_recv_seq_num + 1;

		ptr_conn_info->init_send_seq_num = get_current_isn_to_use();
		ptr_conn_info->send_nxt = ptr_conn_info->init_send_seq_num + 1;
		ptr_conn_info->send_una = ptr_conn_info->init_send_seq_num;

		/* Enter remote socket info in connection record */
		ptr_conn_info->remote_port = tcp_header->src_port;
		ptr_conn_info->local_port = tcp_header->dst_port;
/* sudhir Address was reverse */
		ptr_conn_info->remote_addr = 
				ptr_conn_info->seg_ip_params.destination_address;
		ptr_conn_info->local_addr = 
				ptr_conn_info->seg_ip_params.source_address;

		ptr_conn_info->send_mss = ptr_conn_info->temp_mss;

		ptr_conn_info->send_lwseq = tcp_header->seq_num;
		ptr_conn_info->send_lwack = ptr_conn_info->init_send_seq_num;
#if defined(DEBUG)
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: Init advert win_size = %d\n",
									tcp_header->win_size);
#endif /* DEBUG */
		ptr_conn_info->send_wnd = tcp_header->win_size;
		if (tcp_header->win_size > ptr_conn_info->send_max_advert)
		{
			ptr_conn_info->send_max_advert = tcp_header->win_size;
		}

		/* Handle any data that may have come with the SYN packet -- */
		/* TCP allows such a thing */
		TURN_FIN_OFF(tcp_header);					/* ?? Our style ?? */
		tcp_data(ptr_conn_info, tcp_header);

		/* Change state to SYN_RCVD and send a SYN + ACK as part of the */
		/* handshake */
		ptr_conn_info->tcp_state = SYN_RCVD;
		++tcp.mib.tcpPassiveOpens;

		tcp_printf(TCP_STATES_PRINTF, "TCP: Moved from LISTEN state to SYN_RCVD state\n");

		send_tcp_control_seg(ptr_conn_info, 
				ptr_conn_info->init_send_seq_num, 
				ptr_conn_info->recv_nxt, SYN | ACK);
		add_to_tcp_rtx_queue(ptr_conn_info, 
				ptr_conn_info->init_send_seq_num, 0);

		/* Start a timer on this connection */
		reset_tcp_alive_timer(ptr_conn_info);

		return;
	}

	return;
}

/* 
** tcp_syn_rcvd()
**		Processing for packets received when in the SYN_RCVD state.
**	Params:
**		Ptr to connection entry for current connection
**		Ptr to host format tcp header for segment that came in
*/
void 
tcp_syn_rcvd(TCP_PER_CONN *ptr_conn_info, TCP_HEADER *tcp_header)
{
	BYTE *bptr;
	SOCKADDR_IN peer_address;
	USER_SOCKET *new_sock;


#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Rx pkt in SYN_RCVD state\n");
#endif /* DEBUG */

	/* This is what we actually want */
	if (((ptr_conn_info->send_una - tcp_header->ack_num) < 0) &&
			((tcp_header->ack_num - ptr_conn_info->send_nxt) <= 0))
	{
		/* Acceptable ACK */

		tcp_ack(ptr_conn_info, tcp_header);
		ptr_conn_info->tcp_state = ESTAB;
		tcp_printf(TCP_STATES_PRINTF, "TCP: Moved from SYN_RCVD state to ESTABLISHED state\n");

		/* Do some socket jugglery because this successful connection 
		** should return a new socket record to the socket accept()
		** caller. The original socket is to be retained for the
		** pending LISTEN.
		*/
		new_sock = (USER_SOCKET *)tcp.socket_fptrs.fptr_socket(AF_INET, 
																SOCK_STREAM, TYPE_TCP);
		if ((int) new_sock == -1)
		{
			/* Cannot succeed, so send a reset packet and abort connection */
			send_tcp_control_seg(ptr_conn_info, tcp_header->ack_num, 0, RST);

			/* Allow LISTEN record to continue listening */
			ptr_conn_info->parent_conn_rec_ptr->conn_flags &= ~TCPF_CONNPENDING;

			tcp_quiet_abort(ptr_conn_info);
			tcp_printf(TCP_PRINTF, "TCP: Connect just established with %s "
				"aborted -- insufficient sockets\n", 
				convert_ip_address_to_dot_format(tcp.print_buffer, 
				ptr_conn_info->seg_ip_params.source_address));

			++tcp.mib.tcpAttemptFails;
			return;
		}

		/* Bind this new socket to the same socket address as the listen
		** socket record.
		*/
		new_sock->address_length = ptr_conn_info->parent_conn_rec_ptr->
				socket_record_ptr->address_length;
		new_sock->vptr_address = (BYTE *)table_malloc(1, 
				new_sock->address_length);
		if (new_sock->vptr_address == NULL)
		{
			tcp_printf(TCP_ALARM_PRINTF, "TCP: Unable to allocate memory for internal socket record\n");

			tcp.socket_fptrs.fptr_closesocket((int)new_sock);

			/* Cannot succeed, so send a reset packet and abort connection */
			send_tcp_control_seg(ptr_conn_info, tcp_header->ack_num, 0, RST);

			/* Allow LISTEN record to continue listening */
			ptr_conn_info->parent_conn_rec_ptr->conn_flags &= ~TCPF_CONNPENDING;

			tcp_quiet_abort(ptr_conn_info);
			tcp_printf(TCP_PRINTF, "TCP: Connect just established with %s "
				"aborted -- insufficient sockets\n", 
				convert_ip_address_to_dot_format(tcp.print_buffer, 
				ptr_conn_info->seg_ip_params.source_address));

			++tcp.mib.tcpAttemptFails;
			return;
		}
		memcpy(new_sock->vptr_address, ptr_conn_info->parent_conn_rec_ptr->
				socket_record_ptr->vptr_address, new_sock->address_length);

		memset(&peer_address, 0, sizeof(peer_address));
		peer_address.sin_family = AF_INET;
		peer_address.sin_port = ptr_conn_info->remote_port;
		peer_address.sin_addr.s_addr = 
				(ULONG) ptr_conn_info->remote_addr;

		bptr = (BYTE *)table_malloc(1, sizeof(peer_address));
		if (bptr == NULL)
		{
			new_sock->vptr_peer_address = NULL;
			new_sock->peer_address_length = 0;
		}
		else
		{
			memcpy(bptr, &peer_address, sizeof(peer_address));
			new_sock->vptr_peer_address = (void *)bptr;
			new_sock->peer_address_length = sizeof(peer_address);
		}
		/* Couple the new socket to its associated connection record */
		ptr_conn_info->socket_record_ptr = new_sock;
		new_sock->vptr_protocol_control_block = ptr_conn_info;

		/* Inform the listening socket app that there is a connected socket */
		ptr_conn_info->parent_conn_rec_ptr->socket_record_ptr->ready_socket = 
				(ULONG) new_sock;
		ptr_conn_info->parent_conn_rec_ptr->socket_record_ptr->
				valid_ready_socket = TRUE;
		ptr_conn_info->parent_conn_rec_ptr->conn_flags &= ~TCPF_CONNPENDING;


		/* Inform app so that app's socket "accept" call will go thro' */
		if (ptr_conn_info->socket_record_ptr->do_not_block == FALSE)
		{
			tcp.socket_fptrs.fptr_wakeup_those_waiting_on_socket_event(
				ptr_conn_info->socket_record_ptr, CONNECTION_REQUEST_EVENT);
		}

		/* Process data arriving with the segment */
		tcp_data(ptr_conn_info, tcp_header);

		if (IS_FIN_ON(tcp_header))
		{
			ptr_conn_info->conn_flags |= TCPF_RECVPSH; /* PUSH data */
			ptr_conn_info->recv_nxt += 1;	/* FIN has sequence space */
			send_tcp_control_seg(ptr_conn_info, 
					ptr_conn_info->send_nxt, 
					ptr_conn_info->recv_nxt, ACK);
		
			ptr_conn_info->tcp_state = CLOSE_WAIT;
			tcp_printf(TCP_STATES_PRINTF, "TCP: Moved from SYN_RCVD state to ESTAB to CLOSE_WAIT state\n");
			ptr_conn_info->conn_flags |= TCPF_CONNFIN;
		}
	}
	else
	{
		send_tcp_control_seg(ptr_conn_info, tcp_header->ack_num, 0, RST);

/* sudha 15-Oct-1999.Taken from ras tcp code... */
		if (ptr_conn_info->parent_conn_rec_ptr)
			ptr_conn_info->parent_conn_rec_ptr->conn_flags &= ~TCPF_CONNPENDING;
		tcp_quiet_abort(ptr_conn_info);
		++tcp.mib.tcpAttemptFails;
/* ...sudha 15-Oct-1999.Taken from ras tcp code */
	
	}

	return;
}

/* 
** tcp_syn_sent()
**		Processing for packets received when in the SYN_SENT state.
**	Params:
**		Ptr to connection entry for current connection
**		Ptr to host format tcp header for segment that came in
*/
void 
tcp_syn_sent(TCP_PER_CONN *ptr_conn_info, TCP_HEADER *tcp_header)
{
#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Pkt Rx in SYN_SENT state\n");
#endif /* DEBUG */
	if (IS_ACK_ON(tcp_header))
	{
		if (((tcp_header->ack_num - ptr_conn_info->init_send_seq_num) <= 0) ||
				((tcp_header->ack_num - ptr_conn_info->send_nxt) > 0))
		{
			if (!IS_RST_ON(tcp_header))
			{
				send_tcp_control_seg(ptr_conn_info, tcp_header->ack_num, 0, 
						RST);
			}
			return;
		}
		if (((ptr_conn_info->send_una - tcp_header->ack_num) < 0) &&
				((tcp_header->ack_num - ptr_conn_info->send_nxt) <= 0))
		{
			/* Acceptable ack */
			;
		}
		else
			return;
	}

	if (IS_RST_ON(tcp_header))
	{
		if (IS_ACK_ON(tcp_header))
		{
			/* Signal the user and abort the connection */
/* sudhir To inform App layers */
         ptr_conn_info->socket_record_ptr->valid_ready_socket =
						CONNECTION_REFUSED_ERROR;
			tcp_abort(ptr_conn_info, ABORT_ERROR);
		}
		return;
	}

	if (IS_SYN_ON(tcp_header))
	{
		ptr_conn_info->init_recv_seq_num = tcp_header->seq_num;
		ptr_conn_info->recv_nxt = ptr_conn_info->init_recv_seq_num + 1;

		ptr_conn_info->send_mss = ptr_conn_info->temp_mss;
		ptr_conn_info->send_lwseq = tcp_header->seq_num;
		ptr_conn_info->send_lwack = tcp_header->ack_num;

		ptr_conn_info->send_wnd = tcp_header->win_size;
		if (tcp_header->win_size > ptr_conn_info->send_max_advert)
		{
			ptr_conn_info->send_max_advert = tcp_header->win_size;
		}

		/* Two scenarios - Just a SYN (simultaneous open situation) or a SYN
		** with an ACK (active open successful)
		*/
		if (IS_ACK_ON(tcp_header))
		{
			/* "active open" success */
			/* If ACK were on, we would have earlier chucked out segments
			** with bad ack's - so only good stuff comes here.
			*/
			tcp_ack(ptr_conn_info, tcp_header);
			ptr_conn_info->tcp_state = ESTAB;

/*
Put back the actual retransmit timeout.
We would have freakout with it during
the connect call
Satish, 10th March 1997
*/
         ptr_conn_info->rtx_timeout = DEF_RTO;
			tcp_printf(TCP_STATES_PRINTF, "TCP: Moved from SYN_SENT state to ESTAB state\n");
			send_tcp_control_seg(ptr_conn_info, ptr_conn_info->send_nxt, 
					ptr_conn_info->recv_nxt, ACK);

			/* Inform app so that app's socket "connect" call will go thro' */
			if (ptr_conn_info->socket_record_ptr->do_not_block == FALSE)
			{
				tcp.socket_fptrs.fptr_wakeup_those_waiting_on_socket_event(
					ptr_conn_info->socket_record_ptr, 
					CONNECTION_ESTABLISHED_EVENT);
			}
/* sudhir */
         else 
         {
            ptr_conn_info->socket_record_ptr->valid_ready_socket = TRUE;
         }

			/* Process URG stuff, if any */
/* ??			tcp_urgent(ptr_conn_info, tcp_header); ?? */
		}
		else
		{
			/* "simultaneous open" requested */
			ptr_conn_info->tcp_state = SYN_RCVD;
			send_tcp_control_seg(ptr_conn_info, 
					ptr_conn_info->init_send_seq_num,
					ptr_conn_info->recv_nxt, SYN | ACK);

			/* ?? RFC says, queue the packet for URG processing once the conn
			** enters ESTAB state. SINCE OUR TCP WILL ONLY HANDLE TELNET
			** SERVER APP, WE WILL NOT DO WHAT THE RFC SAYS AND INGORE THIS.
			** MAYBE LATER IF NEEDED, THIS CAN BE DONE. ??
			*/
		}
		return;
	}

	return;
}

/* 
** tcp_estab()
**		Processing for packets received when in the SYN_SENT state.
**	Params:
**		Ptr to connection entry for current connection
**		Ptr to host format tcp header for segment that came in
*/
void 
tcp_estab(TCP_PER_CONN *ptr_conn_info, TCP_HEADER *tcp_header)
{
#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Rx pkt in ESTAB state\n");
#endif /* DEBUG */

	if (tcp_ack(ptr_conn_info, tcp_header) == FAIL)
	{
		return;
	}

	/* Now we come to the actual passing of the data part in the
	** TCP packet to the upper layer buffers
	*/
	tcp_data(ptr_conn_info, tcp_header);
	
	if (IS_FIN_ON(tcp_header))
	{
		ptr_conn_info->conn_flags |= TCPF_RECVPSH; /* PUSH data */

		ptr_conn_info->recv_nxt += 1;	/* FIN has sequence space */
		send_tcp_control_seg(ptr_conn_info, ptr_conn_info->send_nxt,
				ptr_conn_info->recv_nxt, ACK);
		
		ptr_conn_info->tcp_state = CLOSE_WAIT;
		tcp_printf(TCP_STATES_PRINTF, "TCP: Moved from ESTABLISHED state to CLOSE_WAIT state\n");
		ptr_conn_info->conn_flags |= TCPF_CONNFIN;
	}

	return;
}

/*
** tcp_fin_wait_1()
**		Processing of packets when connection is in the FIN_WAIT_1 state.
**	Params:
**		Ptr to connection entry for current connection
**		Ptr to host format tcp header for segment that came in
*/
void 
tcp_fin_wait_1(TCP_PER_CONN *ptr_conn_info, TCP_HEADER *tcp_header)
{
#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Rx pkt in FIN_WAIT_1 state\n");
#endif /* DEBUG */

	if (tcp_ack(ptr_conn_info, tcp_header) == FAIL)
	{
		return;
	}

	/* Now we come to the actual passing of the data part in the
	** TCP packet to the upper layer buffers
	*/
	tcp_data(ptr_conn_info, tcp_header);	/* The data received is useless */
	
	/* Also move into the other wait state if the final FIN (all sends)
	** is ack-ed 
	*/
	if (check_if_rtx_queue_empty(ptr_conn_info))
	{
/*
Satish. 8th March 1997.
If a tcp connection gets into FIN_WAIT_2 state and does not receive a FIN from the remote,
the connection gets struck. No timers are maintained when in this state. The RFC does not
clearly say anything about this. We are getting TCP connections struck in this state with
out WEB browser connections. As of now I dont know why this problem is happenning at all.
So solve the symptom of the problem instead of the problem itself, we maintain the 2msl
timer which is actually to be maintained in TIME_WAIT state.
*/
      tcp_start_2msl_timer(ptr_conn_info);
		ptr_conn_info->tcp_state = FIN_WAIT_2;
		tcp_printf(TCP_STATES_PRINTF, "TCP: Moved from FIN_WAIT_1 state to FIN_WAIT_2 state\n");
		if (IS_FIN_ON(tcp_header))
		{
			ptr_conn_info->conn_flags |= TCPF_RECVPSH; /* PUSH data */
			ptr_conn_info->recv_nxt += 1;	/* FIN has sequence space */
			send_tcp_control_seg(ptr_conn_info, 
					ptr_conn_info->send_nxt, ptr_conn_info->recv_nxt,
					ACK);
			tcp_start_2msl_timer(ptr_conn_info);
			ptr_conn_info->tcp_state = TIME_WAIT;
			tcp_printf(TCP_STATES_PRINTF, "TCP: Moved from FIN_WAIT_1 state to FIN_WAIT_2 to TIME_WAIT state\n");
		}
	}
	else
	{
		if (IS_FIN_ON(tcp_header))
		{
			ptr_conn_info->conn_flags |= TCPF_RECVPSH; /* PUSH data */
			ptr_conn_info->recv_nxt += 1;	/* FIN has sequence space */
			send_tcp_control_seg(ptr_conn_info, 
					ptr_conn_info->send_nxt, ptr_conn_info->recv_nxt,
					ACK);
			ptr_conn_info->tcp_state = CLOSING;
			tcp_printf(TCP_STATES_PRINTF, "TCP: Moved from FIN_WAIT_1 state to CLOSING state\n");
		}
	}

	return;
}

/*
** tcp_fin_wait_2()
**		Processing for connections in the FIN_WAIT_2 state
**	Params:
**		Ptr to connection entry for current connection
**		Ptr to host format tcp header for segment that came in
*/
void 
tcp_fin_wait_2(TCP_PER_CONN *ptr_conn_info, TCP_HEADER *tcp_header)
{
#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Rx pkt in FIN_WAIT_2 state\n");
#endif /* DEBUG */

	/* Inspite of what the RFC says, the way we handle stuff, we should
	** be expecting only a FIN segment in this particular state 
	*/
	if (tcp_ack(ptr_conn_info, tcp_header) == FAIL)
	{
		return;
	}

	/* Now we come to the actual passing of the data part in the
	** TCP packet to the upper layer buffers
	*/
	tcp_data(ptr_conn_info, tcp_header); /* The data is useless */
	
	if (IS_FIN_ON(tcp_header))
	{
		ptr_conn_info->conn_flags |= TCPF_RECVPSH; /* PUSH data */
		ptr_conn_info->recv_nxt += 1;	/* FIN has sequence space */
		send_tcp_control_seg(ptr_conn_info, 
				ptr_conn_info->send_nxt, ptr_conn_info->recv_nxt,
				ACK);
		ptr_conn_info->tcp_state = TIME_WAIT;
		tcp_printf(TCP_STATES_PRINTF, "TCP: Moved from FIN_WAIT_2 state to TIME_WAIT state\n");
		tcp_start_2msl_timer(ptr_conn_info);
	}
}


/*
** tcp_closing()
**		Processing for connections in the CLOSING state
**	Params:
**		Ptr to connection entry for current connection
**		Ptr to host format tcp header for segment that came in
*/
void 
tcp_closing(TCP_PER_CONN *ptr_conn_info, TCP_HEADER *tcp_header)
{
#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Rx pkt in CLOSING state\n");
#endif /* DEBUG */

	if (tcp_ack(ptr_conn_info, tcp_header) == FAIL)
	{
		return;
	}

	if (check_if_rtx_queue_empty(ptr_conn_info))
	{
		/* ACK is for our FIN */

		/* Inform socklib that close was successful */
		ptr_conn_info->conn_error = NOT_CONNECTED_ERROR;

		ptr_conn_info->tcp_state = TIME_WAIT;
		tcp_printf(TCP_STATES_PRINTF, "TCP: Moved from CLOSING state to TIME_WAIT state\n");
		tcp_start_2msl_timer(ptr_conn_info);
	}

	return;
}

/*
** tcp_time_wait()
**		Processing for connections in the TIME_WAIT state
**	Params:
**		Ptr to connection entry for current connection
**		Ptr to host format tcp header for segment that came in
*/
void 
tcp_time_wait(TCP_PER_CONN *ptr_conn_info, TCP_HEADER *tcp_header)
{
#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Rx pkt in TIME_WAIT state\n");
#endif /* DEBUG */

	if (IS_FIN_ON(tcp_header))
	{
		/* Remote guy lost our ACK and resent the FIN, so ACK it again */
		send_tcp_control_seg(ptr_conn_info, 
					ptr_conn_info->send_nxt, ptr_conn_info->recv_nxt,
					ACK);

		/* Restart the 2MSL timeout */
		tcp_start_2msl_timer(ptr_conn_info);
	}
	return;
}

/*
** tcp_close_wait()
**		Processing for connections in the CLOSE_WAIT state
**	Params:
**		Ptr to connection entry for current connection
**		Ptr to host format tcp header for segment that came in
*/
void 
tcp_close_wait(TCP_PER_CONN *ptr_conn_info, TCP_HEADER *tcp_header)
{
#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Rx pkt in CLOSE_WAIT state\n");
#endif /* DEBUG */

	if (tcp_ack(ptr_conn_info, tcp_header) == FAIL)
	{
		return;
	}

	return;
}

/*
** tcp_last_ack()
**		Processing for connections in the LAST_ACK state
**	Params:
**		Ptr to connection entry for current connection
**		Ptr to host format tcp header for segment that came in
*/
void 
tcp_last_ack(TCP_PER_CONN *ptr_conn_info, TCP_HEADER *tcp_header)
{
#if defined(DEBUG)
	tcp_printf(TCP_ALARM_PRINTF, "TCP: Rx pkt in LAST_ACK state\n");
#endif /* DEBUG */

	if (tcp_ack(ptr_conn_info, tcp_header) == FAIL)
		return;

	if (ptr_conn_info->send_una == ptr_conn_info->send_nxt)
	{
#if defined(DEBUG)
		tcp_printf(TCP_ALARM_PRINTF, "TCP: Closing a connection\n");
#endif /* DEBUG */

		/* ACK is for our FIN */
		tcp_quiet_abort(ptr_conn_info);
		return;
	}
	return;
}

/*
** tcp_free()
**		Dummy function. Should never be called.
*/
void
tcp_free(TCP_PER_CONN *ptr_conn_info, TCP_HEADER *tcp_header)
{
	PARAMETER_NOT_USED(ptr_conn_info);
	PARAMETER_NOT_USED(tcp_header);

#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: B A D   S T A T E   C A L L\n\r");
#endif /* DEBUG */	
}

