/*
** TCPUTL.C -- Some utility routines related to TCP stack code
** Modifications : sudha 15-Oct-1999. Put in fixes taken from ras tcp code
**						 in register_urg_func_with_tcp() function.	
*/

#include <stdarg.h>
#include <string.h>
#include <stdlib.h>

#include "rtrstd.h"
#include "all.h"
#include "tcpextrn.h"



USHORT 	tcp_current_normal_port = TCP_PORT_NUMBER;
USHORT	tcp_current_reserved_port = TCP_RESERVED_PORT_NUMBER;

/* Local Prototypes */
void init_misc_fields(TCP_PER_CONN *);
enum TEST init_recv_info(TCP_PER_CONN *);
enum TEST init_send_info(TCP_PER_CONN *);
void init_rtt_estimates(TCP_PER_CONN *);
void init_timers(TCP_PER_CONN *);
void set_tcp_checksum (BYTE *uptr_ip_rx_packet, IP_PARAMETERS *sptr_ip_parameters, USHORT checksum); /* sudha 20 Nov 1998 */

/*
** free_tcp_conn_record()
**		Releases a connection record to the available list.
** Params:
**		Ptr to connection record to free
*/
void
free_tcp_conn_record(TCP_PER_CONN *ptr_conn_info)
{
	/* Remove any retransmit buffers if left */
	clear_rtx_queue(ptr_conn_info);

	/* Free the send and receive window buffers */
	free_window_buffers(ptr_conn_info);

	tcp_printf(TCP_STATES_PRINTF, "TCP: Moved from %s to FREE state\n", 
			get_state_string(ptr_conn_info->tcp_state));
	ptr_conn_info->tcp_state = FREE;
}

/*
** get_free_tcp_conn_record()
**		Returns a pointer to a free connection record or null if there
**		is no free record.
** Params:
**		None
** Returns:
**		Ptr to a connection record or null (if there are no free records)
** NOTE:
**		The way this code is, it will return the first free record which may
**			possibly have just been released from a recent connection. From
**			a debugging point of view, it may be better to follow a simple
**			LFU strategy.
*/

TCP_PER_CONN *
get_free_tcp_conn_record()
{
	USHORT i;
	TCP_PER_CONN *ptr_conn_info;


	ptr_conn_info = tcp.tcp_conn_table;
	for (i = 0; i < tcp.max_tcp_connection; i++, ptr_conn_info++)
	{
		if (ptr_conn_info->tcp_state == FREE)
		{
			/* Do some standard init-ing of the fields and return pointer to
			** record.
			*/
			init_misc_fields(ptr_conn_info);
			if (init_recv_info(ptr_conn_info) == FAIL)
				return NULL;
			if (init_send_info(ptr_conn_info) == FAIL)
			{
				free_window_buffers(ptr_conn_info);
				return NULL;
			}
			init_rtt_estimates(ptr_conn_info);
			init_timers(ptr_conn_info);
			return ptr_conn_info;
		}
	}
	return NULL;
}

void
init_misc_fields(TCP_PER_CONN *ptr_conn_info)
{
	ptr_conn_info->local_addr = 0;
	ptr_conn_info->remote_addr = 0;
	ptr_conn_info->local_port = 0;
	ptr_conn_info->remote_port = 0;
	ptr_conn_info->socket_record_ptr = NULL;
	ptr_conn_info->parent_conn_rec_ptr = NULL;
	ptr_conn_info->tcp_open = UNKNOWN_OPEN;

	ptr_conn_info->conn_error = 0;
	ptr_conn_info->conn_flags = 0;
	ptr_conn_info->conn_alive_timer = 0;
	ptr_conn_info->conn_probe_retries = 0;

	ptr_conn_info->rtx_counter = 0;
	ptr_conn_info->rtx_retries = 0;

#if defined(DEBUG)

	if (ptr_conn_info->rtx_queue != NULL)
	{
		/* Hang on, some error from historical connections */
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: !@!@!@! A free connection record has unreleased retransmit info\n");
	}

#endif /* DEBUG */

	ptr_conn_info->rtx_queue = NULL;
}

enum TEST
init_recv_info(TCP_PER_CONN *ptr_conn_info)
{
	BYTE *pbuf;


	pbuf = (BYTE *)table_malloc(1, tcp.recv_win_size);
	if (pbuf == NULL)
	{
		tcp_printf(TCP_ALARM_PRINTF, "TCP: Unable to allocate connection RECEIVE window buffer\n");
		return FAIL;
	}
	ptr_conn_info->recv_buf = pbuf;
	ptr_conn_info->recv_buf_size = tcp.recv_win_size;
	ptr_conn_info->recv_last_advert = tcp.recv_win_size;

	ptr_conn_info->recv_first = ptr_conn_info->recv_filled = 0;
	ptr_conn_info->recv_nxt = 0;
	return PASS;
}

enum TEST
init_send_info(TCP_PER_CONN *ptr_conn_info)
{
	BYTE *pbuf;


	pbuf = (BYTE *)table_malloc(1, tcp.send_win_size);
	if (pbuf == NULL)
	{
		tcp_printf(TCP_ALARM_PRINTF, "TCP: Unable to allocate connection SEND window buffer\n");
		return FAIL;
	}
	ptr_conn_info->send_buf = pbuf;

	ptr_conn_info->send_una = ptr_conn_info->send_nxt = 0;
	ptr_conn_info->send_una_amount = ptr_conn_info->send_filled = 0;
	ptr_conn_info->send_bstart = ptr_conn_info->send_bnext = 0;
	ptr_conn_info->send_buf_size = ptr_conn_info->send_wnd = tcp.send_win_size;
	ptr_conn_info->send_max_advert = DEF_TCP_MSS;
	ptr_conn_info->send_lwseq = ptr_conn_info->send_lwack = 0;
	ptr_conn_info->send_mss = DEF_TCP_MSS;

	return PASS;
}

void
init_rtt_estimates(TCP_PER_CONN *ptr_conn_info)
{
	ptr_conn_info->srtt = 0;
	ptr_conn_info->rtx_timeout = DEF_RTO;

	ptr_conn_info->srtt_deviation = (ptr_conn_info->rtx_timeout - 
					ptr_conn_info->srtt) >> 1;
}

void
init_timers(TCP_PER_CONN *ptr_conn_info)
{
	/* Init connection idle timer */
	ptr_conn_info->conn_alive_timer = DEF_CONN_IDLE_TIME;
}

/*
** get_new_conn_record()
** 	Called when a SYN segment is got in the LISTEN state to acquire
**		a connection record and copy the LISTEN's connection record over
**		it. This is done to retain the LISTEN record in the LISTEN state.
** Params:
**		Ptr to the record in the LISTEN state
** Returns:
**		Ptr to new connection record which is a copy of the record in the
**			LISTEN state or NULL if no free record is available.
*/
TCP_PER_CONN *
get_new_conn_record(TCP_PER_CONN *old_conn_record)
{
	TCP_PER_CONN *ptr_conn_info;
	

	ptr_conn_info = get_free_tcp_conn_record();
	if (ptr_conn_info == NULL)
	{
		return NULL;
	}
	/* Init fields of this new connection record from the listen record */
	/* NOTE: Don't use a memcpy() as pointers to the buffers for connection */
	/*			will get lost.	*/
	ptr_conn_info->local_addr = old_conn_record->local_addr;
	ptr_conn_info->local_port = old_conn_record->local_port;
	ptr_conn_info->parent_conn_rec_ptr = old_conn_record;
	ptr_conn_info->socket_record_ptr = old_conn_record->socket_record_ptr;
	ptr_conn_info->tcp_state = old_conn_record->tcp_state;
	ptr_conn_info->tcp_open = old_conn_record->tcp_open;
	ptr_conn_info->conn_flags = old_conn_record->conn_flags;
	ptr_conn_info->seg_ip_params = old_conn_record->seg_ip_params;
	ptr_conn_info->seg_header = old_conn_record->seg_header;
	ptr_conn_info->seg_options = old_conn_record->seg_options;
	ptr_conn_info->seg_data = old_conn_record->seg_data;
	ptr_conn_info->seg_len = old_conn_record->seg_len;
	ptr_conn_info->temp_mss = old_conn_record->temp_mss;
	ptr_conn_info->last_seg_sent_tick = old_conn_record->last_seg_sent_tick;
	ptr_conn_info->seg_for_rtt_estimate = old_conn_record->seg_for_rtt_estimate;
	ptr_conn_info->rtx_timeout = old_conn_record->rtx_timeout;
	ptr_conn_info->srtt = old_conn_record->srtt;
	ptr_conn_info->srtt_deviation = old_conn_record->srtt_deviation;
	

	return ptr_conn_info;
}

/*
** tcp_recv_wnd()
**		Called to get value of receive window size to advertise to sending
**		end. Implements "Silly Window Syndrome" avoidance.
** Params:
** 	Ptr to connection record for connection
** NOTES:
**		Implementation follows Richard Stevens' description in TCP/IP
**			Illustrated, Vol. 1.
*/
USHORT
tcp_recv_wnd(TCP_PER_CONN *ptr_conn_info)
{
	USHORT win_size;


	/* True window size is as below */
	win_size = ptr_conn_info->recv_buf_size - ptr_conn_info->recv_filled;
	if (ptr_conn_info->tcp_state < ESTAB || 
		win_size <= ptr_conn_info->recv_last_advert)
	{
		/* The 2nd check in the conditional causes use of a small window
		** size if it is lesser than previous advertisement.
		*/
		ptr_conn_info->recv_last_advert = win_size;
		return win_size;
	}

	/* If execution has come here, the window size has grown since the
	** last advertisement. Do not allow advertisement of this new size
	** unless it advertises space greater than a segment size or greater
	** than half the buffer space available. This is done if the last
	** advertised size is 0.
	*/
	if (ptr_conn_info->recv_last_advert == 0 &&
			(win_size * 2 < ptr_conn_info->recv_buf_size || 
			win_size < tcp.max_seg_size))
	{
		/* Advertise a 0 window size faking no receive space, until we have
		** a sufficiently large amount of space. 
		*/
		ptr_conn_info->recv_last_advert = 0;
		return 0;
	}

	ptr_conn_info->recv_last_advert = win_size;
	return win_size;
}

/*
** tcp_recv_wnd_ver2()
**		Called to get value of receive window size to advertise to sending
**		end. This is another version of the function above (tcp_recv_wnd())
**		which can be used to just determine the window size that probably
**		would be advertised. Does not update the 'recv_last_advert' field.
** Params:
** 	Ptr to connection record for connection
** NOTES:
**		Implementation follows Richard Stevens' description in TCP/IP
**			Illustrated, Vol. 1.
*/
USHORT
tcp_recv_wnd_ver2(TCP_PER_CONN *ptr_conn_info)
{
	USHORT win_size;


	/* True window size is as below */
	win_size = ptr_conn_info->recv_buf_size - ptr_conn_info->recv_filled;
	if (ptr_conn_info->tcp_state < ESTAB || 
		win_size <= ptr_conn_info->recv_last_advert)
	{
		/* The 2nd check in the conditional causes use of a small window
		** size if it is lesser than previous advertisement.
		*/
		return win_size;
	}

	/* If execution has come here, the window size has grown since the
	** last advertisement. Do not allow advertisement of this new size
	** unless it advertises space greater than a segment size or greater
	** than half the buffer space available. This is done if the last
	** advertised size is 0.
	*/
	if (ptr_conn_info->recv_last_advert == 0 &&
			(win_size * 2 < ptr_conn_info->recv_buf_size || 
			win_size < tcp.max_seg_size))
	{
		/* Advertise a 0 window size faking no receive space, until we have
		** a sufficiently large amount of space. 
		*/
		return 0;
	}

	return win_size;
}

/*
** tcp_hdr_ntoh()
**		Convert all the info in the tcp packet received from net to host
**		byte order and fill into a local structure. Later only the local
**		structure will be used to extract information from.
** Params:
**		Ptr to structure to place info in
**		Ptr to the tcp header
*/

void 
tcp_hdr_ntoh(TCP_HEADER *tcp_header, TCP_HEADER *sptr_tcp_pdu)
{
	tcp_header->src_port = net_to_host_short(sptr_tcp_pdu->src_port);
	tcp_header->dst_port = net_to_host_short(sptr_tcp_pdu->dst_port);
	tcp_header->seq_num = net_to_host_long(sptr_tcp_pdu->seq_num);
	tcp_header->ack_num = net_to_host_long(sptr_tcp_pdu->ack_num);
	tcp_header->code_and_hlen.code_and_hdrlen_word = net_to_host_short(
									sptr_tcp_pdu->code_and_hlen.code_and_hdrlen_word);
	tcp_header->win_size = net_to_host_short(sptr_tcp_pdu->win_size);
	tcp_header->chk_sum = net_to_host_short(sptr_tcp_pdu->chk_sum);
	tcp_header->urg_ptr = net_to_host_short(sptr_tcp_pdu->urg_ptr);
}

/*
** tcp_hdr_hton()
**		Convert all info the passed tcp packet from host to network byte
**		order.
** Params:
**		Ptr to structure to place info in
**		Ptr to structure convert from
** NOTE:
**		Checksum needs to be calculated after this is called.
*/

void
tcp_hdr_hton(TCP_HEADER *tcp_net_buf, TCP_HEADER *tcp_host_buf)
{
	tcp_net_buf->src_port = host_to_net_short(tcp_host_buf->src_port);
	tcp_net_buf->dst_port = host_to_net_short(tcp_host_buf->dst_port);
	tcp_net_buf->seq_num = host_to_net_long(tcp_host_buf->seq_num);
	tcp_net_buf->ack_num = host_to_net_long(tcp_host_buf->ack_num);
	tcp_net_buf->code_and_hlen.code_and_hdrlen_word = host_to_net_short(
						tcp_host_buf->code_and_hlen.code_and_hdrlen_word);
	tcp_net_buf->win_size = host_to_net_short(tcp_host_buf->win_size);
	tcp_net_buf->chk_sum = host_to_net_short(tcp_host_buf->chk_sum);
	tcp_net_buf->urg_ptr = host_to_net_short(tcp_host_buf->urg_ptr);
}


/*
** tcp_printf()
**		Classes of TCP related messages printing is done here
** Params:
**		Printf group
**		Message to printf 
**		... other args for printf
*/
void
tcp_printf(enum TCP_PRINTF_GROUPS printf_group, const char *cptr_format, ...)
{
	va_list argptr;
	BYTE_ENUM(BOOLEAN) print_string;


	va_start(argptr, cptr_format);
	if (tcp.print_class.tcp_print_switch_on == FALSE)
	{
		va_end(argptr);
		return;
	}

	switch (printf_group)
	{
		case TCP_PRINTF:
			print_string = tcp.print_class.tcp_printing_enabled;
			break;
		case TCP_ALARM_PRINTF:
			print_string = tcp.print_class.alarm_printing_enabled;
			break;
		case TCP_DATA_PRINTF:
			print_string = tcp.print_class.data_printing_enabled;
			break;
		case TCP_DEBUG_PRINTF:
			print_string = tcp.print_class.debug_printing_enabled;
			break;
		case TCP_STATES_PRINTF:
			print_string = tcp.print_class.state_printing_enabled;
			break;
		default:
			print_string = FALSE;
			break;
	}

	if (print_string == TRUE)
	{
		vprintf(cptr_format, argptr);
	}

	va_end(argptr);
}

/*
** free_window_buffers()
**		Called to release allocated window memory in a connection record
** Params:
**		Ptr to connection record
*/
void
free_window_buffers(TCP_PER_CONN *ptr_conn_info)
{
	if (ptr_conn_info->send_buf != NULL)
	{
		table_free((void *)ptr_conn_info->send_buf);
		ptr_conn_info->send_buf = NULL;
	}
	if (ptr_conn_info->recv_buf != NULL)
	{
		table_free((void *)ptr_conn_info->recv_buf);
		ptr_conn_info->recv_buf = NULL;
	}
}

/*
** send_tcp_packet()
**		Call to pass a full prepared TCP packet to the IP layer for 
**		transmission on the network.
** Params:
**		Ptr to connection record for connection
**		Ptr to the TCP packet (header + data)
**		Length of the TCP packet
*/
void
send_tcp_packet(IPADDR local_addr, IPADDR remote_addr, BYTE type_of_service,
		BYTE *ptcpbuf, USHORT tcp_pkt_len, enum BOOLEAN retrans)
{
	BYTE *pipbuf;
	USHORT ip_pkt_len;
	IP_UPPER_LAYER_PARAMETERS ip_params;

/* sudhir for testing */
	memset (&ip_params, 0x00, sizeof (IP_UPPER_LAYER_PARAMETERS));
/* sudhir for testing */

	/* compute checksum including the pseudo stuff etc. */
	tcp_checksum(local_addr, remote_addr, ptcpbuf, tcp_pkt_len);

	/* Prepare IP parameters to send out */
	ip_params.source_address = local_addr;
	ip_params.destination_address = remote_addr;
/*	ip_params.do_not_fragment_flag = TRUE; */
	ip_params.do_not_fragment_flag = FALSE;
/*	(*(BYTE *)&ip_params.type_of_service) = type_of_service; */
	(*(BYTE *)&ip_params.type_of_service) = 0;
	ip_params.sequence_id = 0;
	ip_params.time_to_live = ((remote_addr & 0x000000ffL) == 
				0x000000ffL) ? 2 : 0;
	ip_params.protocol = TCP_PROTOCOL;
	ip_params.option_length = 0;
	ip_params.vptr_cached_route = NULL;
	ip_params.virtual_port_number = NO_SUCH_PORT;

	ip_pkt_len = tcp_pkt_len + sizeof(IP_HEADER) + sizeof(MAC_HEADER);
	pipbuf = (BYTE *)(ptcpbuf - sizeof(IP_HEADER) - 
																		sizeof(MAC_HEADER));

	if (retrans == FALSE)
		++tcp.mib.tcpOutSegs;
	else
		++tcp.mib.tcpRetransSegs;

	send_ip_packet_from_upper_layer(&ip_params, FALSE, (IP_PACKET *)pipbuf, 
			ip_pkt_len, tcp_tx_completion);
}

/*
** verify_tcp_checksum()
**		Performs a checksum on the arrived tcp packet
** Params:
** 	Ptr to some IP info for pseudo header
**		Ptr to the TCP packet
**		Size of TCP packet
** Returns:
**		0 if the checksum is okay, else a non-zero value
*/
USHORT 
verify_tcp_checksum(IP_PARAMETERS *sptr_ip_parameters, 
		TCP_HEADER *sptr_tcp_pdu, USHORT tcp_packet_size)
{
	PSEUDO_IP_PARAMETERS pseudo_header;


	pseudo_header.length = host_to_net_short(tcp_packet_size);
	pseudo_header.source_address = 
			host_to_net_long(sptr_ip_parameters->source_address);
	pseudo_header.destination_address = 
			host_to_net_long(sptr_ip_parameters->destination_address);
	pseudo_header.zero_field = 0x00;
	pseudo_header.protocol = TCP_PROTOCOL;

	return (calculate_ip_checksum(&pseudo_header, (BYTE *)sptr_tcp_pdu, 
			tcp_packet_size));
}

/*
** tcp_checksum()
**		Computes the checksum for a tcp packet prior to sending it and puts
**		the checksum in the tcp header.
** Params:
**		Local IP address
**		Remote IP address
**		Ptr to TCP packet
**		Size of TCP packet
*/
void
tcp_checksum(IPADDR local_address, IPADDR remote_address, BYTE *tcp_pkt, 
		USHORT tcp_packet_size)
{
	USHORT chksum;
	PSEUDO_IP_PARAMETERS pseudo_header;


	pseudo_header.length = host_to_net_short(tcp_packet_size);
	pseudo_header.source_address = host_to_net_long(local_address);
	pseudo_header.destination_address = host_to_net_long(remote_address);
	pseudo_header.zero_field = 0x00;
	pseudo_header.protocol = TCP_PROTOCOL;

	((TCP_HEADER *)tcp_pkt)->chk_sum = 0;

	/* calculate_ip_checksum() gives checksum in network byte order itself */
	chksum = calculate_ip_checksum(&pseudo_header, tcp_pkt, tcp_packet_size);

	/*	All zeros and all ones is equivalent in one's complement arithmetic. 
	** The spec requires us to change zeros into ones to distinguish an 
	** all-zero checksum from no checksum at all.
	*/
	if (chksum == 0x0000)
		chksum = 0xffff;
	((TCP_HEADER *)tcp_pkt)->chk_sum = chksum;
}

/*
** get_state_string()
**		Returns ptr to a string describing the state
** Params:
**		Current state
*/
char *
get_state_string(USHORT tcp_state)
{
	switch (tcp_state)
	{
	case CLOSED:
		return "CLOSED";
		break;
	case LISTEN:
		return "LISTEN";
		break;
	case SYN_RCVD:
		return "SYN_RECEIVED";
		break;
	case SYN_SENT:				 	
		return "SYN_SENT";
		break;
	case ESTAB:
		return "ESTABLISHED";
		break;
	case FIN_WAIT_1:
		return "FIN-WAIT-1";
		break;
	case FIN_WAIT_2:
		return "FIN-WAIT-2";
		break;
	case CLOSING:
		return "CLOSING";
		break;
	case TIME_WAIT:
		return "TIME-WAIT";
		break;
	case CLOSE_WAIT:
		return "CLOSE-WAIT";
		break;
	case LAST_ACK:
		return "LAST-ACK";
		break;
	default:
		return "UNKNOWN";
		break;
	}
}

/*
** register_urg_func_with_tcp()
**		This func. is globally callable by an app after it has made the 
**		socket() and bind() calls to register a callback function to call
**		when urgent data comes in. The callback should increment a counter
**		whenever it is called. If this counter is non-zero, the app should
**		treat all data in subsequent read()'s as urgent data until it reaches
**		a app specific end-of-urgent-data indicator.
** Params:
**		Socket descriptor
**    ULONG 			sudha 15-Oct-1999.Taken from ras tcp code.
**    USHORT			sudha 15-Oct-1999.Taken from ras tcp code.
*/

/* sudha 15-Oct-1999. Added 2 more parameters in function pointer
urg_data_handler() */

int
register_urg_func_with_tcp(int socket_descriptor, 
														void (*urg_data_handler)(USER_SOCKET *,ULONG offset,USHORT length))
{
	TCP_PER_CONN *ptr_conn_info;
	USER_SOCKET *sptr_user_socket = (USER_SOCKET *)socket_descriptor;

	if (sptr_user_socket == NULL)
	{
		tcp.socket_fptrs.fptr_set_socket_class_error_number(BAD_SOCKET_HANDLE);
		return -1;
	}
	if (sptr_user_socket->vptr_protocol_control_block == NULL)
	{
		tcp.socket_fptrs.fptr_set_socket_class_error_number(NOT_CONNECTED_ERROR);
		return -1;
	}

	ptr_conn_info = (TCP_PER_CONN *)sptr_user_socket->vptr_protocol_control_block;

	if (is_valid_conn_ptr(ptr_conn_info) == TRUE)
	{
		ptr_conn_info->urgent_data_handler = urg_data_handler;
		return 0;
	}
	else
	{
#if defined(DEBUG)
		tcp_printf(TCP_DEBUG_PRINTF, "TCP: Bad socket descriptor passed to register_urg_func_with_tcp()\n");
#endif /* DEBUG */
		return -1;
	}
}

/*
** is_valid_conn_ptr()
**		Checks is the connection ptr passed is a valid one.
** Params:
**		Ptr to conn record
*/
BYTE_ENUM(BOOLEAN)
is_valid_conn_ptr(TCP_PER_CONN *suspect_conn_ptr)
{
	USHORT i;
	TCP_PER_CONN *ptr_conn_info;

	ptr_conn_info = tcp.tcp_conn_table;
	for (i = 0; i < tcp.max_tcp_connection; i++, ptr_conn_info++)
	{
		if (ptr_conn_info == suspect_conn_ptr &&
				ptr_conn_info->tcp_state != FREE)
			return TRUE;
	}
#if defined(DEBUG)
	tcp_printf(TCP_DEBUG_PRINTF, "TCP: Bad connection record ptr passed to call\n");
#endif /* DEBUG */

	return FALSE;
}


/*
Satish. 17th March 1997.
Searches the connection table for a free port. The port numbers used are from 1024 to 65535 in case of NORMAL_PORTS
and 900 to 1023 in case of reserved ports.
Instead of the linear search used here, we can have a better implementation of the server.
*/
USHORT get_tcp_port_number(BYTE port_type)
{
	TCP_PER_CONN *ptr_conn_info;
	USHORT tcp_port_number;
	USHORT max_tcp_number_of_ports;
	BYTE	free_port_found;
	BYTE	i;
	USHORT	j;

	if (port_type == NORMAL_PORT) {
		tcp_port_number = tcp_current_normal_port;
		if (tcp_port_number == MAX_TCP_PORT_NUMBER)
			tcp_port_number = TCP_PORT_NUMBER;			/* start from the beginning */
		max_tcp_number_of_ports = MAX_TCP_PORT_NUMBER - TCP_PORT_NUMBER;
	    				
	}
	else {
		tcp_port_number = tcp_current_reserved_port;
		max_tcp_number_of_ports = TCP_PORT_NUMBER - TCP_RESERVED_PORT_NUMBER;
		if (tcp_port_number == TCP_PORT_NUMBER)
			tcp_port_number = TCP_RESERVED_PORT_NUMBER;		/* start the beginning	*/
	}

	ptr_conn_info = tcp.tcp_conn_table;

	for (j = 0; j < max_tcp_number_of_ports; j++)
	{
	    	free_port_found = TRUE;
		for (i = 0; i < tcp.max_tcp_connection; i++, ptr_conn_info++)
		{
			if (ptr_conn_info->tcp_state != FREE)
			{
				/* Search all the non free records and get a free port number
				*/
				if (ptr_conn_info->local_port == tcp_port_number) {
					free_port_found = FALSE;
					break;						/* Already used	*/
				}

			}
		}
		if (free_port_found)
			break;
		tcp_port_number++;
	    	if (port_type == NORMAL_PORT) {
		    if (tcp_port_number == MAX_TCP_PORT_NUMBER)
		    	tcp_port_number = TCP_PORT_NUMBER;			/* start from the beginning */
		}
		else {
		    if (tcp_port_number == TCP_PORT_NUMBER)
		    	tcp_port_number = TCP_RESERVED_PORT_NUMBER;		/* start the beginning	*/
		}
	}
	if (free_port_found) {

	    	if (port_type == NORMAL_PORT) 
			tcp_current_normal_port = tcp_port_number + 1;			/* take the next port */
	
		else 
			tcp_current_reserved_port = tcp_port_number + 1;		/* take the next port */
		
		return(tcp_port_number);
	}
    	else
		return(0);
}



/* sudhir  11/7/97 for proxy server */


USHORT get_tcp_src_port_from_ip_packet (BYTE *uptr_ip_rx_packet, IP_PARAMETERS *sptr_ip_parameters)
{
	TCP_HEADER *sptr_tcp_pdu;

	sptr_tcp_pdu = (TCP_HEADER *) ((ULONG) uptr_ip_rx_packet + 
		sizeof(UNION_MAC_HEADER) + sptr_ip_parameters->header_length);
	return sptr_tcp_pdu->src_port;	
}



USHORT get_tcp_dst_port_from_ip_packet (BYTE *uptr_ip_rx_packet, IP_PARAMETERS *sptr_ip_parameters)
{
	TCP_HEADER *sptr_tcp_pdu;

	sptr_tcp_pdu = (TCP_HEADER *) ((ULONG) uptr_ip_rx_packet + 
		sizeof(UNION_MAC_HEADER) + sptr_ip_parameters->header_length);
	return sptr_tcp_pdu->dst_port;	
}

USHORT get_tcp_port_number_from_ip_packet (BYTE *uptr_ip_rx_packet, IP_PARAMETERS *sptr_ip_parameters, USHORT *destination_port)
{
	TCP_HEADER *sptr_tcp_pdu;

	sptr_tcp_pdu = (TCP_HEADER *) ((ULONG) uptr_ip_rx_packet + 
		sizeof(UNION_MAC_HEADER) + sptr_ip_parameters->header_length);
	*destination_port = sptr_tcp_pdu->src_port;
	return sptr_tcp_pdu->dst_port;	
}

USHORT set_tcp_src_port_number (BYTE *uptr_ip_rx_packet, IP_PARAMETERS *sptr_ip_parameters, USHORT new_port)
{
	TCP_HEADER *sptr_tcp_pdu;
	USHORT actual_port=0;
	
	sptr_tcp_pdu = (TCP_HEADER *) ((ULONG) uptr_ip_rx_packet + 
		sizeof(UNION_MAC_HEADER) + sptr_ip_parameters->header_length);
	
	actual_port = sptr_tcp_pdu->src_port;
	sptr_tcp_pdu->src_port = new_port;
	return actual_port;
}

void set_tcp_dest_port_number (BYTE *uptr_ip_rx_packet, IP_PARAMETERS *sptr_ip_parameters, USHORT new_port)
{
	TCP_HEADER *sptr_tcp_pdu;

	sptr_tcp_pdu = (TCP_HEADER *) ((ULONG) uptr_ip_rx_packet + 
		sizeof(UNION_MAC_HEADER) + sptr_ip_parameters->header_length);
	
	sptr_tcp_pdu->dst_port = new_port;
	return;
}



BYTE *get_ptr_to_tcp_header (BYTE *uptr_ip_rx_packet, IP_PARAMETERS *sptr_ip_parameters)
{
	BYTE *sptr_tcp_pdu;

	sptr_tcp_pdu = (BYTE *) ((ULONG) uptr_ip_rx_packet + sizeof(UNION_MAC_HEADER) 
							+ sptr_ip_parameters->header_length);

	return sptr_tcp_pdu;
}



BYTE *get_ptr_to_tcp_packet (BYTE *uptr_ip_rx_packet, IP_PARAMETERS *sptr_ip_parameters)
{
	BYTE *sptr_tcp_pdu;

	sptr_tcp_pdu = (BYTE *) ((ULONG) uptr_ip_rx_packet + sizeof(UNION_MAC_HEADER) 
							+ sptr_ip_parameters->header_length + sizeof (TCP_HEADER));

	return sptr_tcp_pdu;
}

enum BOOLEAN is_close_connection_initiated (BYTE *uptr_ip_rx_packet, IP_PARAMETERS *sptr_ip_parameters)
{
	
	TCP_HEADER *sptr_tcp_pdu;

	sptr_tcp_pdu = (TCP_HEADER *) ((ULONG) uptr_ip_rx_packet + 
		sizeof(UNION_MAC_HEADER) + sptr_ip_parameters->header_length);

	if (IS_RST_ON (sptr_tcp_pdu) || IS_FIN_ON (sptr_tcp_pdu))
		return TRUE;
	else
		return FALSE;
}

#if 0
void update_tcp_checksum_in_proxy_packet (UNION_IP_PACKET *uptr_ip_rx_packet, IP_PARAMETERS *sptr_ip_parameters,
	ULONG actual_src_address, USHORT actual_source_port, USHORT mapped_port)	
{
	TCP_HEADER *sptr_tcp_pdu;
	USHORT actual_checksum, new_checksum, result =0 ;


	sptr_tcp_pdu = (TCP_HEADER *) ((ULONG) uptr_ip_rx_packet + 
		sizeof(UNION_MAC_HEADER) + sptr_ip_parameters->header_length);
			
	actual_checksum = sptr_tcp_pdu->chk_sum;

/* adjust port and address checksum in two steps */

	new_checksum = actual_checksum + mapped_port - actual_source_port;
	printf ("TCP Checksum is %4x\n",new_checksum);

	new_checksum = new_checksum - ((*(USHORT *) &actual_src_address) +(*(USHORT *) (&actual_src_address+2))) 
						+(*(USHORT *) (&sptr_ip_parameters->source_address))+ (*(USHORT *) ((&sptr_ip_parameters->source_address)+2));

	result = end_around_carry ((USHORT)new_checksum);

	sptr_tcp_pdu->chk_sum = result;
	printf ("TCP Check sum is %4x\n",sptr_tcp_pdu->chk_sum);
	return;
}
#endif

USHORT get_tcp_checksum (BYTE *uptr_ip_rx_packet, IP_PARAMETERS *sptr_ip_parameters)
{
	TCP_HEADER *sptr_tcp_pdu;

	sptr_tcp_pdu = (TCP_HEADER *) ((ULONG) uptr_ip_rx_packet + 
		sizeof(UNION_MAC_HEADER) + sptr_ip_parameters->header_length);

	return sptr_tcp_pdu->chk_sum;
}

void set_tcp_checksum (BYTE *uptr_ip_rx_packet, IP_PARAMETERS *sptr_ip_parameters, USHORT checksum)
{
	TCP_HEADER *sptr_tcp_pdu;

	sptr_tcp_pdu = (TCP_HEADER *) ((ULONG) uptr_ip_rx_packet + 
		sizeof(UNION_MAC_HEADER) + sptr_ip_parameters->header_length);

	sptr_tcp_pdu->chk_sum = checksum;
	printf ("Final Check sum is %4x\n",sptr_tcp_pdu->chk_sum);
	return;
}


void modify_tcp_acknowledge_field (BYTE *uptr_ip_rx_packet, IP_PARAMETERS *sptr_ip_parameters,int offset)
{
	TCP_HEADER *sptr_tcp_pdu;

	sptr_tcp_pdu = (TCP_HEADER *) ((ULONG) uptr_ip_rx_packet + 
		sizeof(UNION_MAC_HEADER) + sptr_ip_parameters->header_length);
	sptr_tcp_pdu->ack_num -= offset;
	return;
}

int get_tcp_seq_number_from_ip_packet (BYTE *uptr_ip_rx_packet, IP_PARAMETERS *sptr_ip_parameters)
{
	TCP_HEADER *sptr_tcp_pdu;

	sptr_tcp_pdu = (TCP_HEADER *) ((ULONG) uptr_ip_rx_packet + 
		sizeof(UNION_MAC_HEADER) + sptr_ip_parameters->header_length);
	
	return (sptr_tcp_pdu->seq_num);
}


int get_tcp_ack_number_from_ip_packet (BYTE *uptr_ip_rx_packet, IP_PARAMETERS *sptr_ip_parameters)
{
	TCP_HEADER *sptr_tcp_pdu;

	sptr_tcp_pdu = (TCP_HEADER *) ((ULONG) uptr_ip_rx_packet + 
		sizeof(UNION_MAC_HEADER) + sptr_ip_parameters->header_length);
	
	return (sptr_tcp_pdu->ack_num);
}

void modify_tcp_seq_number_field	(BYTE *uptr_ip_rx_packet, IP_PARAMETERS *sptr_ip_parameters,int offset)
{
	TCP_HEADER *sptr_tcp_pdu;

	sptr_tcp_pdu = (TCP_HEADER *) ((ULONG) uptr_ip_rx_packet + 
		sizeof(UNION_MAC_HEADER) + sptr_ip_parameters->header_length);

	sptr_tcp_pdu->seq_num += offset;
	return;
}
/* sudhir  11/7/97 for proxy server */
