/***************************************************************************
	File	: main.c
	Synop	: This is the source file for the MC68360 boot prom
	Modi	:	15/05/96 Vidy added the S/W watchdog timer reset
				19/07/96 Vidy added delay after response for next command to be ok
				Chetan added modifications for memory relocation
****************************************************************************/
#include	"..\include\mctypes.h"
#include	"..\include\boot.h"
#include	"..\include\downldr.h"
#include	"..\include\quote.h"
#include	"..\include\utils.h"

byte	ReceiveBuffer[RCV_BUF_SIZE];	/* quoted buffer */
dword SMCRcvBufSize ; /* Used so that asm files also look into this */
word	ReceiveIndex;
word	ReceiveHead;
word	ReceiveTail;
byte	ReceiveState;
byte	DownloadType;

/* all variables used in smc.68k */
byte	WaitForXON;
byte	NumOfXONs;
byte	NumOfXOFFs;
dword	NextBytePtr;
word	NumBytesLeft;
byte	ModemPresentAndOffline;		/* Modem responds OK in this case */

HeaderType *CodeHeaderPtr;
BootConfigType	*BootCfgPtr;
HeaderType *CfgHeaderPtr;

/* Essential hardware already inited */
extern	void	InitSMC(void);
extern	void	InitTimer(void);
extern	void	TransmitCnt(byte *, word);
extern	void	Reset(void);
extern	void	determine_flash_types(void);
extern BOOL CheckCRC (byte *Pkt, dword Length, word *CRCPtr) ;

extern void Execute (byte *loc) ;
extern int TestSCC1 (void) ;
extern int TestSCC2 (byte index) ;
extern int TestSCC3 (byte index) ;
extern int TestSCC4 (byte index) ;
extern void higher_write_to_flash (char *src, char *dest, word size) ;
extern void StartTimer (word count) ;
extern void StopTimer (void) ;
extern void Break (void) ;

byte BootHeader[FL_PAGE_SIZE] ;

void Delay (void)
{
	word i, j ;
	for (i = 0 ; i < 4 ; i ++)
		for (j = 0 ; j < 0xffff ; j ++) ;
}


#pragma separate	BootConfig segment BootSeg
BootConfigType	BootConfig = {
	"2.00.16",		/* Booter Version */ /* 2.xx and above will size memory */
	0xFFFF,			/* first word of ethernet address */
	0xFFFFFFFF,		/* Second dword of ethernet address */
	"",
	4,					/* Number of Hardware Ports (1LAN + 3WAN) */
	0,					/* x MEG  - Bank 1 Base address */
	0,					/* x MEG  - Bank 1 size */
	0,					/* x MEG  - Bank 2 Base Address */
	0,					/* x MEG  - Bank 2 size */

	CLOCK_33344MHZ,		/* Clock Speed */
	PROM_WIDTH_16BITS,	/* it is 16 bits wide PROM */
	MODEL_MTSR1_120,		/* model is to be filled during production */

	"",				/* 8 bytes reserved for alignment and future use */
	BOOT_HEADER_MAGIC,	/* This magic number is just to check sanity */
	0x100000,		/* Flash Prom size */
	0x00000,		/* RAM size */
	0x00000,		/* RAM start */
} ;

extern word	Fl1ProId ;  
extern word	Fl2ProId ;

main()
{
	int count ;

	word DoWrite = 0 ;
	dword *DRAMSizePtr ;
	/* These locations contains details of memory detection.
		(1C00) DRAMSizePtr -> Bank1Base
		(1C04) DRAMSizePtr + 1 -> Bank1Size
		(1C08) DRAMSizePtr + 2 -> Bank2Base
		(1C0C) DRAMSizePtr + 3 -> Bank2Size  */
	DRAMSizePtr = 0x00001C00 ; 

	SMCRcvBufSize = RCV_BUF_SIZE  ; /* Done here b'cos we have to deal with idata */
											  /* But we r not dealing with it */
#if 0
	determine_flash_types() ;
	if (Fl1ProId != 0x1FA4)		/* if Id is not 29040A assume 29040 */
		Fl1ProId = 0x1F5B ;
	if (Fl2ProId != 0x1FA4)
		Fl2ProId = 0x1F5B ;
#else
		Fl1ProId = 0x1FA4 ;		/* assume both flashes are 29040A */
		Fl2ProId = 0x1FA4 ;
#endif

	InitData() ;
	InitSMC() ;				/* initialize the BOOT port */

	/* In case of WARM Boot do not need to write about DRAM results */
	/* DoWrite helps us know whether to get rid of redundant writes
		about the DRAM results to the Boot Header area in FLASH PROM */
	if (*((byte *) BOOT_MODE_LOCATION_IN_RAM) != WARM_BOOT_MODE)
	{
		memcpy ((byte *) &BootHeader, (byte *) FL_BOOT_HDR, sizeof (BootHeader)) ;

		if (((BootConfigType *) &BootHeader)->RamSize
				!= (dword) (*(DRAMSizePtr + 1) + *(DRAMSizePtr + 3)))
		{
			((BootConfigType *) &BootHeader)->RamSize = (dword) (*(DRAMSizePtr + 1) + *(DRAMSizePtr + 3)) ;
			DoWrite = 1 ;
		}
		if (((BootConfigType *) &BootHeader)->RamStart
														!= (dword) *DRAMSizePtr)
		{
			((BootConfigType *) &BootHeader)->RamStart = (dword) *DRAMSizePtr ;
			DoWrite = 1 ;
		}
		if (((BootConfigType *) &BootHeader)->Base1 
							!= (byte) (*DRAMSizePtr / (1024 * 1024)))
		{
			((BootConfigType *) &BootHeader)->Base1 = (byte) (*DRAMSizePtr / (1024 * 1024)) ;
			DoWrite = 1 ;
		}
		if (((BootConfigType *) &BootHeader)->Size1 
						!= (byte) (*(DRAMSizePtr + 1) / (1024 * 1024)))
		{
			((BootConfigType *) &BootHeader)->Size1 = (byte) (*(DRAMSizePtr + 1) / (1024 * 1024)) ;
			DoWrite = 1 ;
		}
		if (((BootConfigType *) &BootHeader)->Base2 
						!= (byte) (*(DRAMSizePtr + 2) / (1024 * 1024)))
		{
			((BootConfigType *) &BootHeader)->Base2 = (byte) (*(DRAMSizePtr + 2) / (1024 * 1024)) ;
			DoWrite = 1 ;
		}
		if (((BootConfigType *) &BootHeader)->Size2 
							!= (byte) (*(DRAMSizePtr + 3) / (1024 * 1024)))
		{
			((BootConfigType *) &BootHeader)->Size2 = (byte) (*(DRAMSizePtr + 3) / (1024 * 1024)) ;
			DoWrite = 1 ;
		}

		if (DoWrite)
			higher_write_to_flash ((byte *) &BootHeader, (byte *) FL_BOOT_HDR, sizeof (BootHeader)) ;
	}

	InitTimer() ;			/* Initialise PIT */

	if (*((byte *) BOOT_MODE_LOCATION_IN_RAM) == WARM_BOOT_MODE)
	{
		*((byte *) BOOT_MODE_LOCATION_IN_RAM) = 0x00 ;	/* reset this flag */
		TransmitStr ("Warm Boot\r\n") ;
		if (!CheckCRC ((byte *) CfgHeaderPtr->DnLdAddr,
							CfgHeaderPtr->CodeLength, (word *) &CfgHeaderPtr->CRC))
		{
			TransmitStr ("Setup Checksum error\r\n") ;
			goto WaitDownLoad ;
		}
		Execute ((byte *) CodeHeaderPtr->StartAddr) ;
	}

	/* Check if there is any modem */
	/* AT&FS0=1 ie. factory defaults, auto answer */
	/* Since SMC has no DTR line, the modem will not respond to any command.
		So either the modem has to be set for DTR always high or DTR is to be
		made high externally from a Breakout Box. So no factory defaults. */

	TransmitStr ("AT") ;
	Delay() ;

	if (SendModemStr ("S0=1&D0\r\n"))
	{	/* ignore DTR (no DTR on SMC) */
		/* modem initiated XON/XOFF and echo */
		SendModemStr ("AT&E5E0\r\n") ;
	}

	TransmitStr("Boot Version: ") ; /* Vidy 01/08/97 */
	TransmitStr (BootConfig.Version);

	TransmitStr ("\n\rMemory: Bank1 = ");
	if(BootConfig.Size1 == 4)
		TransmitStr ("4");
	else if(BootConfig.Size1 == 16)
		TransmitStr ("16");
	else
		TransmitStr ("0");
	TransmitStr("MB, Bank2 = ");
	if(BootConfig.Size2 == 4)
		TransmitStr ("4");
	else if (BootConfig.Size2 == 16)
		TransmitStr ("16");
	else
		TransmitStr ("0");
	TransmitStr("MB. CPU speed = ");
	if (BootConfig.ClockSpeed == CLOCK_250000MHZ )
		TransmitStr("25");
	else if (BootConfig.ClockSpeed == CLOCK_258048MHZ )
		TransmitStr("25.8048");
	else if (BootConfig.ClockSpeed == CLOCK_330000MHZ)
		TransmitStr("33");
	else if (BootConfig.ClockSpeed == CLOCK_33344MHZ)
		TransmitStr("33.344");
	else
		TransmitStr("--");
	TransmitStr("MHz.\n\r");

	/* If Magic number is set and DownloadCode is not true,
		calculate CRC of the Bridge code else Bridge code has
		not been loaded or has to be updated */
	if (CodeHeaderPtr->CodeLength 
				&& CodeHeaderPtr->MagicNum == MAGIC_NUM 
						&& *((byte *) BOOT_MODE_LOCATION_IN_RAM) != DOWNLOAD_MODE)
	{
		/* It is better to validate the field DnLdAddr before checking */
		if (CheckCRC ((byte *) CodeHeaderPtr->DnLdAddr,
						CodeHeaderPtr->CodeLength, (word *) &CodeHeaderPtr->CRC))
		{
			/* There may be a problem here.
				Say the boot has changed (for sake of changing the mem
				locations of headers or relocation itself).
				The DnLdAddr field will be pointing to some other location
				which confuses the booter.Depending on the situation
				we may have Setup chksum error or a bus error itself */
			if (!CheckCRC ((byte *) CfgHeaderPtr->DnLdAddr,
							CfgHeaderPtr->CodeLength, (word *) &CfgHeaderPtr->CRC))
			{
				TransmitStr ("Setup Checksum error\r\n") ;
				goto WaitDownLoad ;
			}
			ReceiveHead = ReceiveTail = 0 ;
			*ReceiveBuffer = 0 ;
			TransmitStr ("Type 'd' to download (Setup or Code).\r\n") ;
			for (count = 0 ; count < 7 ; count ++)
			{
				Delay() ;
				if ((*ReceiveBuffer == 'd') || (*ReceiveBuffer == 'D'))
					goto WaitDownLoad ;
			}
			TransmitStr ("Booting...\r\n") ;
			*((byte *) BOOT_MODE_LOCATION_IN_RAM) = 0 ;
			ExpandAndLocate() ;
		}
		else
		{
			TransmitStr ("Code Checksum error\r\n") ;
		}
	}
	else
	{
		if (CodeHeaderPtr->MagicNum == NEW_BOOT_MAGIC)
		{
			TransmitStr ("Please download Router firmware\r\n") ;
		}
		else
		{
			TransmitStr ("Download mode or Wrong Magic number\r\n") ;
		}
	}
WaitDownLoad:
	TransmitStr ("Waiting download.\n\r");
	*((byte *) BOOT_MODE_LOCATION_IN_RAM) = 0 ;
	/* If CRC fails invoke Command interpreter */
	while (1)
	{
		if (GetPacket())
			ProcessCommand() ;
	}
}


BOOL GetPacket (void)
{
	char ch ;

	ResetSWSR() ;		/* reset the software watchdog timer */

	if (ReceiveTail != ReceiveHead)
	{
		ch = ReceiveBuffer[ReceiveHead] ;
		INC_CIR_IDX (ReceiveHead, RCV_BUF_SIZE) ;
		ReceiverEnabled = 1 ;
		switch (ReceiveState)
		{
			case RECV_7E :
				if (ch == FRAME_DELIM)
				{
					ReceiveState = RECV_DATA ;
					ReceiveIndex = 0 ;
				}
				break ;

			case RECV_DATA :
				if (ch == FRAME_DELIM)
				{
					if (ReceiveIndex >= MIN_PKT_SIZE)
					{
						ReceiveState = RECV_7E ;
						return TRUE ;
					}
					else
					{
						/* stay in the RECV_DATA state . But start a new packet */
						ReceiveIndex = 0 ;
						break ;
					}
				}
				else
					if (ch == 0x7d)
						ReceiveState = RECV_QUOTE ;
					else
						if (ReceiveIndex < RCV_BUF_SIZE)
							ProcessBuffer[ReceiveIndex++] = ch ; 
						else 
							ReceiveState = RECV_7E ;
				break ;

			case RECV_QUOTE :
				if (ReceiveIndex < RCV_BUF_SIZE)
				{
					ProcessBuffer[ReceiveIndex++] = (ch ^ 0x20) ;
					ReceiveState = RECV_DATA ; 
				}
				else 
					ReceiveState = RECV_7E ;
				break ;
		}			
	}
	return FALSE ;
}


void ProcessCommand (void)
{
 	TargetCmdType *TrgtCmd ;
	ResultType *ResultPkt ;
	byte *PktPtr ;
	word PktLen ;
	byte ResetSystem = FALSE ;

	DownloadType = NO_DNLD ;
	ResultPkt = (ResultType *) (TmpBuffer + sizeof (SMCPktHeadType)) ;
	ResultPkt->Ptype = RES_TYPE ;
	ResultPkt->Errcode = 0 ;

	/* Process the Command */
	if (!CheckCRC (ProcessBuffer, (dword) ReceiveIndex, 0))
	{
		ReceiveState = RECV_7E ;
		ResultPkt->Errcode = ERR_CRC ;
		goto sendpkt ;
	} 
	
	/* Actual Pkt starts at an offset of BridgeId(4) */
	PktPtr = ProcessBuffer + sizeof (SMCPktHeadType) ;
	switch (PktPtr[0])
	{
		case PASSWORD_TYPE :
			if (!CheckPassword (((PasswordPktType *) PktPtr)->Password))
				/* send NOK pkt */
				ResultPkt->Errcode = ERR_PASSWORD ;
			break ;

		case TRGT_CMND_TYPE :
			TrgtCmd = (TargetCmdType *) PktPtr ;
			switch (TrgtCmd->TargetAction)
			{
				case DNLOAD_CODE_TYPE :
					/* send back a result type pkt and wait for download pkts */ 
					DownloadType = DNLD_CODE ;
					break ;

				case HW_DIAG_TYPE :
					switch (TrgtCmd->Port)
					{
						case SCC1 :
							if (!TestSCC1())
								ResultPkt->Errcode = ERR_SCC1 ;
							break ;

						case SCC2 :
							if (!TestSCC2 (TrgtCmd->BaudIndex))
								ResultPkt->Errcode = ERR_SCC2 ;
							break ;

						case SCC3 :
							if (!TestSCC3 (TrgtCmd->BaudIndex))
								ResultPkt->Errcode = ERR_SCC3 ;
							break ;
							
						case SCC4 :
							if (!TestSCC4 (TrgtCmd->BaudIndex))
								ResultPkt->Errcode = ERR_SCC4 ;
							break ;
					}
					break ;

				case DNLOAD_CFG_TYPE:
					DownloadType = DNLD_CFG ;
					break ;
			}
			break ;

		case READ_REQ_TYPE :
		{
			ReadLocReqType *RRqT ;
			ReadLocRespType *RRspT ;

			RRqT = (ReadLocReqType *) PktPtr ;
			if (RRqT->Length > 128)
				RRqT->Length = 128 ;

			/* send back the response pkt */
			RRspT = (ReadLocRespType *) (TmpBuffer + sizeof (SMCPktHeadType)) ;
			RRspT->Ptype = READ_RESP_TYPE ;
			if ((RRqT->Address + RRqT->Length) >= (128 * 1024 * 1024)
					&& (RRqT->Address + RRqT->Length) < ((128 * 1024 * 1024) + (512 * 1024)))
			{
				RRspT->ErrCode = 0 ;
				RRspT->Address = RRqT->Address ;
				RRspT->Length = RRqT->Length ;
				memcpy ((byte *) RRspT->buffer,
							(byte *) RRqT->Address, RRqT->Length) ;
			}
			else
				RRspT->ErrCode = ERR_INV_MEM_RANGE ;
			PktLen = sizeof (ReadLocRespType) + sizeof (SMCPktHeadType) ;		
			goto sendresp ;
		}

		case WRITE_REQ_TYPE :
		{
			WriteLocReqType *WRqT ;

			WRqT = (WriteLocReqType *) PktPtr ;
			if (WRqT->Length > FL_PAGE_SIZE)
				WRqT->Length = FL_PAGE_SIZE ;

			/* DO not call WriteToFlash b'cos it does not check
				for 29C040 or 29C040A and just tries to program.
				In case the size > 256 bytes for 29C040A will fail */
			higher_write_to_flash ((char *) WRqT->Buffer,
						(char *) WRqT->Address, WRqT->Length) ;
			break ;
		}
			
		case REBOOT_TYPE :
			/* move reset value into the reset location and the system 
			will reboot */
			*((byte *) BOOT_MODE_LOCATION_IN_RAM) = PktPtr[1] ;		/* Store the boot type */
			ResetSystem = TRUE ;
			break ;

		default :
			/* send back a reply with ErrCode = BridgeNotup and request
						password if present */
			if (BootCfgPtr->PassWord) 
				ResultPkt->Errcode = ERR_NO_BRD_GET_PASSWORD ;
			else
				ResultPkt->Errcode = ERR_NO_BRD_NO_PASSWORD ;
			break ;
	}

sendpkt :
	PktLen =	sizeof (ResultType) + sizeof (SMCPktHeadType) ;
sendresp :
	AddCRC (TmpBuffer, (dword) PktLen) ;
	/* do not send PktLen but (Pktlen + 2) b'cos
		2 extra zeroes are appended while adding CRC */
	PktLen = Quote ((byte *) TmpBuffer,
								(byte *) ResponseBuffer + 1, PktLen + 2) ;
	ResponseBuffer[0] = ResponseBuffer[PktLen + 1] = FRAME_DELIM ;
	PktLen += 2 ;
	TransmitCnt (ResponseBuffer, PktLen) ;
	if (DownloadType)
		DownLoadNewInfo() ;
	else
		if (ResetSystem)
			Reset() ;
}

BOOL CheckPassword (char *PW)
{
	if (!strncmp (BootCfgPtr->PassWord, PW, strlen (BootCfgPtr->PassWord)))
		return TRUE ;
	else
		return FALSE ;
}

void ClearRxBuffer (void)
{
	ReceiveTail = ReceiveHead = 0 ;
}

word SearchRxBuffer (char *Str)
{
	word i, j, Len ;

	if (!ReceiveTail)
		return FALSE ;
	Len = strlen (Str) ;
	if (ReceiveTail < Len)
		return FALSE ;
	for (i = ReceiveHead ; i <= (ReceiveTail - Len) ; i ++)
	{
		for (j = 0 ; j < Len ; j ++)
		{
			if (ReceiveBuffer[i+j] != Str[j])
				break ;
		}
		if (j == Len)
			return TRUE ;
	}
	return FALSE ;
}

word SendModemStr (byte *Str)
{
	ClearRxBuffer() ;
	TransmitCnt (Str, strlen (Str)) ;		/* TransmitStr may block */
	Delay() ;
	StartTimer (MDM_RESP_TIMEOUT) ;
	while (!TimeoutOccured)
	{
		if (SearchRxBuffer ("OK"))
		{
			Delay() ;
			Delay() ;				/* Vidy 19/07/96 - for next command */
			ModemPresentAndOffline = 1 ;	/* Modem responded OK in this case */
			StopTimer() ;
			return TRUE ;
		}
	}
	StopTimer() ;
	return FALSE ;
}

