/****************************************************************************
	file	:	boot.h
	header file for Flash PROM boot module
****************************************************************************/
#define	MAGIC_NUM	0xABCD		/* Magic number in CodeHeader and Config recs */
#define	NEW_BOOT_MAGIC	0x1234	/* means new boot, need to download code again */
#define	DOWNLOAD_MODE	0xA5		/* Number for download mode */
#define	WARM_BOOT_MODE	0x5A		/* Number for warm boot (after new cnf) mode */
#define	BOOT_HEADER_MAGIC	0xa55aa55aL	/* just for the sake of boot config */

#define	FRAME_DELIM	0x7e

#define	LedPort	*(dword *)(0xff000010)

#define	FL_PG_SIZE_10	128			/* Flash PROM page size */
#define	FL_PG_SIZE_20	256			/* Flash PROM page size */
#define	FL_PG_SIZE_40	512			/* Flash PROM page size */

#define	FL_PAGE_SIZE	FL_PG_SIZE_40

#define	PART_29C040		$5B		/* type of flash chip */
#define	PART_29C040A	$A4

#define	PKTS_PER_PAGE	((word)(FL_PAGE_SIZE/DNLD_PKT_SIZE))

#define	BOOT_CODE_SIZE	(68*1024)	/* $11000 */

/* assuming 512 bytes as FLASH PROM page length */
#define FL_BASE_1 0x08000000 /* 128 MB */
#define FL_ROMM_START FL_BASE_1  /*	 68K for boot code */
#define	FL_BOOT_HDR		(FL_ROMM_START + BOOT_CODE_SIZE) /* 1 page for boot cfg hdr 0801 1000 */
#define	FL_CODE_HDR		(FL_BOOT_HDR   + FL_PAGE_SIZE)	/* 1 page for code hdr  0801 1200 */
#define	FL_CFG_HDR		(FL_CODE_HDR   + FL_PAGE_SIZE)	/* 1 page for system cfg hdr 0801 1400 */
#define	FL_SYSTEM_CFG	(FL_CFG_HDR    + FL_PAGE_SIZE)	/* 64K for system cfg  0801 1600 */
#define	FL_CODE_START	(FL_SYSTEM_CFG + 0x00010000l)	 /*  0802 1600 */

#define	RECV_7E		0
#define	RECV_DATA	1
#define	RECV_QUOTE	2

#define	ERR_CRC			12		/* part of UI err codes */
#define	ERR_PASSWORD	13
#define	ERR_SCC1			14
#define	ERR_SCC2			15
#define	ERR_SCC3			16
#define	ERR_SCC4			17

#define	ERR_NO_BRD_GET_PASSWORD		18
#define	ERR_NO_BRD_NO_PASSWORD		19
#define	ERR_INV_MEM_RANGE				20
#define	ERR_ROUTER_UP					21

#define	TRGT_CMND_TYPE	0	/* part of UI commands */
#define	RES_TYPE			12
#define	PASSWORD_TYPE	30
#define	REBOOT_TYPE		22
#define	READ_REQ_TYPE	31
#define	READ_RESP_TYPE	32
#define	WRITE_REQ_TYPE	33

/* target command types */
#define	DNLOAD_CODE_TYPE		0
#define	HW_DIAG_TYPE			1
#define	DNLOAD_CFG_TYPE		3

/* download types */
#define	NO_DNLD		0
#define	DNLD_CODE	1
#define	DNLD_CFG		2

/* Ports */
#define	SCC1		1
#define	SCC2		2
#define	SCC3		3
#define	SCC4		4

/* #defines for different speeds */
#define	CLOCK_UNKNOWN			0
#define	CLOCK_250000MHZ		1
#define	CLOCK_258048MHZ		2
#define	CLOCK_330000MHZ		3
#define	CLOCK_33344MHZ			4

/* a byte to denote the hardware model */
#define	MODEL_UNKNOWN			0	/* unknown hardware type */
#define	MODEL_MTSR3				1	/* one LAN, 3 RS232/V.35 WAN ports */
#define	MODEL_MTSR1_100		2	/* one LAN, 1 RS232/V.35 WAN port */
#define	MODEL_MTSR1_200		3	/* one LAN, one 33.6K modem port */
#define	MODEL_MTSR3_200		4	/* one LAN, three modems port */
#define	MODEL_MTSR1_202ST		5	/* one LAN, one ISDN ST port */
#define	MODEL_MTSR1_202NT		6	/* one LAN, one ISDN NT port */
#define	MODEL_MTSR3_201		7	/* one LAN, one DSU and one RS232/V.35 WAN ports */
#define	MODEL_MTSR1_120		8	/* 2 LAN and one RS232/V.35 WAN ports */

/* Boot prom width */
#define	PROM_WIDTH_UNKNOWN	0
#define	PROM_WIDTH_8BITS		1
#define	PROM_WIDTH_16BITS		2
#define	PROM_WIDTH_32BITS		3



#define	MIN_PKT_SIZE	4	/* DnLdPktHead(2 Abort Pkt) + CRC(2) */

#define	INC_CIR_IDX(seqnum, maxval)	seqnum++; seqnum &= (maxval -1)

/* Packet format of all Target Command packets 
   +----+------+------+----+----+---+-----+----+----+---...----
   | 7e |  SMC header | Pkt data... |    CRC   | 7e |	
   +----+------+------+----+----+---------+----+----+---...----
	  1         4                          2      1
CRC is calculated from Type  */

/* WARNING :
DO NOT INSERT ANY MEMBERS OR CHANGE THEIR SIZES. IF ADDITION IS REQUIRED
YOU MAY DO SO AT THE END OF THE STRUCTURE ONLY.
EVEN ROM MONITOR DEPENDS ON THE SIZES OF THE ELEMENTS DURING WARM BOOT
*/
typedef	struct {
	byte	Version[10];		/* version number of this record ASCIIZ */
	word	EthernetAddr_H;	/* Ethernet address of bridge */
	dword	EthernetAddr_L;	/* Ethernet address of bridge */
	byte	PassWord[20];		/* password for Boot config */
	byte	NumberOfPorts;
	byte	Base1;
	byte	Size1;
	byte	Base2;
	byte	Size2;

	byte	ClockSpeed;
	byte	BootBusWidth;
	byte	ModelNumber;

	byte Reserved[8] ;       /* for future use */   
	dword MagicNumber;		/* 0xA55AA55A to indicate valid config */
	dword	FlashPromSize;		/* System related information */
	dword	RamSize;
	dword	RamStart;
}BootConfigType;

/* Generic Header Pkt for all downloads. Each download type will use some 
or all of these fields */
typedef	struct {
	byte	Version[10];		/* 0 .. 9 	version number as ASCIIZ */
	byte	DateStamp[25];		/* 10.. 34	date as ASCIIZ */
	byte	Dmy;					/* 35 */
	word	MagicNum;			/* 36.. 37 */
	word	CRC;					/* 38.. 39	16bit CRC value */
	dword	CodeLength;			/* 40.. 43	Valid length of compressed code */
	dword DnLdAddr;			/* 52.. 55	Address where compressed code is put */
	dword LoadAddr;			/* 44.. 47	Address to start loading at */
	dword	StartAddr;			/* 48.. 51	Address to start execution at */
	byte	Reserved[20];		/* 56.. 75	for future use */
}HeaderType;

typedef struct
{
	byte BridgeNumber[4] ;
} SMCPktHeadType ;

typedef struct {
	byte	Ptype;				// 0
	char	Password[11];		// 1 .. 11
}	PasswordPktType;

typedef struct
{
	byte Ptype ;		/* 0 			packet type */
	byte Dmy0 ;			/* 1 			Dummy to align on word boundary */
	word Errcode ;		/* 2 .. 3 	Zero if OK else error code of the error */
} ResultType ;

typedef struct		{
	byte	Ptype;				// 0
	byte	TargetAction;		// 1
	byte	Port;					// 2	  These 2 fields are for 
	byte	BaudIndex;			// 3	  Hardware Diagnostics
}	TargetCmdType;

typedef struct {
	byte	Ptype;				// 0
#if 1
	byte  Length;				// 1
	byte	Dmy[2];				// 2..3
#else
	word  Length;				// 1..2
	byte	Dmy[1];				// 3
#endif
	dword Address;				// 4..7
}	ReadLocReqType;

#if 1
typedef struct {
	byte	Ptype;				// 0
	byte	ErrCode;				// 1
	byte	Length;				// 2
	byte	Dmy[1];				// Vidy added Dmy and Address
	dword Address;				// 3..6
	byte	buffer[128];		// 7..134
}  ReadLocRespType;
#else
typedef struct
{
	byte Ptype;					/* 0 */
	byte ErrCode;				/* 1 */
	word Length;				/* 2 3 */
	dword Address;				/* 4 5 6 7 */
	byte buffer[1024] ;		/* 8....1031 */
}  ReadLocRespType ;
#endif

typedef	struct {
	byte	Ptype;				// 0
	byte	Dmy;					// 1
	word	Length;				// 2..3
	dword Address;				// 4..7
	byte	Buffer[FL_PAGE_SIZE];	// 8..519
} WriteLocReqType;

extern	word	ReceiveIndex;
extern	byte	ReceiveState;
extern	byte	ReceiveBuffer[];	/* quoted buffer */
extern	word	ReceiveHead;
extern	word	ReceiveTail;
extern	byte	DownloadType;

/* all variables used in smc.68k */
extern	byte	WaitForXON;
extern	byte	NumOfXONs;
extern	byte	NumOfXOFFs;
extern	dword	NextBytePtr;
extern	word	NumBytesLeft;

extern	HeaderType	*CodeHeaderPtr;
extern	BootConfigType	*BootCfgPtr;
extern	HeaderType	*CfgHeaderPtr;

void	ClearRxBuffer(void);
void	ProcessCommand(void);
BOOL	GetPacket(void);
BOOL	CheckPassword(char *);
word	SearchRxBuffer(char	*);
word	SendModemStr(char *);

