/****************************************************************************
	file	:	boot.h
	header file for Flash PROM boot module
****************************************************************************/
#define	MAGIC_NUM		0xABCD	/* Magic number in CodeHeader and Config recs */
#define	DOWNLOAD_MODE	0xA5		/* Number for download mode */
#define	WARM_BOOT_MODE	0x5A		/* Number for warm boot (after new cnf) mode */

#define	FRAME_DELIM		0x7e

#define	BOOT_CODE_SIZE	(16*1024L)
#define	FL_BOOT_CFG_CODE_HDR_SIZE		0x40	/* all headers to be < 65 bytes */


/*		Flash prom sitting from 0 to 512
+--------------------------------+
|	16K - Reset Vec + Boot Code
|
|
|
+--------------------------------+
|	2K - Headers (Boot, code, config,....
|  6K - Scripts for 3 ports (2K each)
+--------------------------------+
|	8K - Configuration
|
+--------------------------------+
|	32K - Firmware starts here
|	...
|
|
|
|
|
|
+--------------------------------+
|
.
.
.
+--------------------------------+

SRAM from 512K to (512+256)K
+--------------------------------+
|
|
|
|
.
.
.
+--------------------------------+

*/

/* headers section starts at 16K boundary and is 8K long*/
#define	FL_BASE			0x0
#define	FL_SIZE			(512 * 1024L)	/* 512 KB */
#define	FL_ROMM_START	FL_BASE			/* 16K for boot code and constants */
#define	FL_HDR_SECT_START		(FL_ROMM_START + BOOT_CODE_SIZE)
#define	FL_HDR_SECT_SIZE		(8*1024)

/* Following structs are located in cmd.loc file during linking */
#define	FL_BOOT_HDR		FL_HDR_SECT_START
#define	FL_CODE_HDR		(FL_BOOT_HDR + FL_BOOT_CFG_CODE_HDR_SIZE)
#define	FL_CFG_HDR		(FL_CODE_HDR + FL_BOOT_CFG_CODE_HDR_SIZE)
#define	FL_SCRIPT_HDR	(FL_CFG_HDR + FL_BOOT_CFG_CODE_HDR_SIZE)

#define	RECV_7E		0
#define	RECV_DATA	1
#define	RECV_QUOTE	2

#define	ERR_CRC			12		/* part of UI err codes */
#define	ERR_PASSWORD	13
#define	ERR_SCC1			14
#define	ERR_SCC2			15
#define	ERR_SCC3			16
#define	ERR_SCC4			17

#define	ERR_NO_BRD_GET_PASSWORD		18
#define	ERR_NO_BRD_NO_PASSWORD		19
#define	ERR_INV_MEM_RANGE				20
#define	ERR_ROUTER_UP					21

#define	TRGT_CMND_TYPE	0	/* part of UI commands */
#define	RES_TYPE			12
#define	PASSWORD_TYPE	30
#define	REBOOT_TYPE		22
#define	READ_REQ_TYPE	31
#define	READ_RESP_TYPE	32
#define	WRITE_REQ_TYPE	33

/* target command types */
#define	DNLOAD_CODE_TYPE		0
#define	HW_DIAG_TYPE			1
#define	DNLOAD_CFG_TYPE		3

/* download types */
#define	NO_DNLD		0
#define	DNLD_CODE	1
#define	DNLD_CFG		2

/* Ports */
#define	SCC1		1
#define	SCC2		2
#define	SCC3		3
#define	SCC4		4

/* #defines for different speeds */
#define	CLOCK_UNKNOWN			0
#define	CLOCK_250000MHZ		1
#define	CLOCK_258048MHZ		2
#define	CLOCK_330000MHZ		3
#define	CLOCK_33344MHZ			4

/* a byte to denote the hardware model */
#define	MODEL_UNKNOWN			0	/* unknown hardware type */
#define	MODEL_MTSR3				1	/* one LAN, 3 RS232/V.35 WAN ports */
#define	MODEL_MTSR1_100		2	/* one LAN, 1 RS232/V.35 WAN port */
#define	MODEL_MTSR1_200		3	/* one LAN, one 33.6K modem port */
#define	MODEL_MTSR3_200		4	/* one LAN, three modems port */
#define	MODEL_MTSR1_202ST		5	/* one LAN, one ISDN ST port */
#define	MODEL_MTSR1_202NT		6	/* one LAN, one ISDN NT port */
#define	MODEL_MTSR3_201		7	/* one LAN, one DSU and one RS232/V.35 WAN ports */
#define	MODEL_MTSR1_120		8	/* 2 LAN and one RS232/V.35 WAN ports */
#define	MODEL_MTSR3_100		9	/* one LAN, three modems port */

/* Boot prom width */
#define	PROM_WIDTH_UNKNOWN	0
#define	PROM_WIDTH_8BITS		1
#define	PROM_WIDTH_16BITS		2
#define	PROM_WIDTH_32BITS		3

#define	MIN_PKT_SIZE	4	/* DnLdPktHead(2 Abort Pkt) + CRC(2) */

#define	INC_CIR_IDX(seqnum, maxidx)	seqnum++; seqnum &= (maxidx -1)
#define	PREV_SEQ_NUM(seqnum)	((seqnum) ? (seqnum - 1) : (MAX_SEQ_NUM - 1))

/* Packet format of all Target Command packets 
   +----+------+------+----+----+---+-----+----+----+---...----
   | 7e |  SMC header | Pkt data... |    CRC   | 7e |	
   +----+------+------+----+----+---------+----+----+---...----
	  1         4                          2      1
CRC is calculated from Type  */

/* WARNING :
DO NOT INSERT ANY MEMBERS OR CHANGE THEIR SIZES. IF ADDITION IS REQUIRED
YOU MAY DO SO AT THE END OF THE STRUCTURE ONLY. MAX is 64
*/
typedef	struct {
	byte	MajorVersion;		/* major version number */
	byte	MinorVersion;		/* minor version number */
	word	EthernetAddr_H;	/* Ethernet address of bridge */
	dword	EthernetAddr_L;	/* Ethernet address of bridge */
	byte	PassWord[8];		/* password for Boot config */		/* 16 */

	byte	NumberOfPorts;
	byte	ClockSpeed;
	byte	BootBusWidth;
	byte	ModelNumber;														/* 20 */

	word MagicNumber;			/* 0xABCD to indicate valid config */
	word	dmy;					/* to align the rest long words */
	dword	FlashPromSize;		/* System related information */
	dword	RamSize;
	dword	RamStart;															/* 34 */
}BootConfigType;

/* Following structure is to communicate parameters between
the boot and the firware */

typedef struct {
	byte	BootMode;		/* is it warm boot, with download etc */
	byte	Reserved[15];
}BootBinIOPBType;


/* Command and response structures used for communication between
the target and roucon */

/* Generic Header Pkt for all downloads. Each download type will use some 
or all of these fields */
typedef	struct {
	byte	Version[10];		/* 0 .. 9 	version number as ASCIIZ */
	byte	DateStamp[25];		/* 10.. 34	date as ASCIIZ */
	byte	Dmy;					/* 35 */
	word	MagicNum;			/* 36.. 37 */
	word	CRC;					/* 38.. 39	16bit CRC value */
	dword	CodeLength;			/* 40.. 43	Valid length of compressed code */
	dword DnLdAddr;			/* 52.. 55	Address where compressed code is put */
	dword LoadAddr;			/* 44.. 47	Address to start loading at */
	dword	StartAddr;			/* 48.. 51	Address to start execution at */
}HeaderType;

typedef struct
{
	byte BridgeNumber[4] ;
} SMCPktHeadType ;

typedef struct {
	byte	Ptype;				// 0
	char	Password[11];		// 1 .. 11
}	PasswordPktType;

typedef struct
{
	byte Ptype ;		/* 0 			packet type */
	byte Dmy0 ;			/* 1 			Dummy to align on word boundary */
	word Errcode ;		/* 2 .. 3 	Zero if OK else error code of the error */
} ResultType ;

typedef struct		{
	byte	Ptype;				// 0
	byte	TargetAction;		// 1
	byte	Port;					// 2	  These 2 fields are for 
	byte	BaudIndex;			// 3	  Hardware Diagnostics
}	TargetCmdType;

typedef struct {
	byte	Ptype;				// 0
	byte  Length;				// 1
	byte	Dmy[2];				// 2..3
	dword Address;				// 4..7
}	ReadLocReqType;

typedef struct {
	byte	Ptype;				// 0
	byte	ErrCode;				// 1
	byte	Length;				// 2
	byte	Dmy[1];				// 3
	dword Address;				// 4..7
	byte	buffer[128];		// 8..
}  ReadLocRespType;

typedef	struct {
	byte	Ptype;				// 0
	byte	Dmy;					// 1
	word	Length;				// 2..3
	dword Address;				// 4..7
	byte	Buffer[512];		// 8..519  /* total size was 512 */
} WriteLocReqType;

extern	word	ReceiveIndex;
extern	byte	ReceiveState;
extern	byte	ReceiveBuffer[];	/* quoted buffer */
extern	word	ReceiveHead;
extern	word	ReceiveTail;
extern	byte	DownloadType;

/* following variables are used in smc.68k */
extern	byte	WaitForXON;
extern	byte	NumOfXONs;
extern	byte	NumOfXOFFs;
extern	dword	NextBytePtr;
extern	word	NumBytesLeft;

void	ClearRxBuffer(void);
void	ProcessCommand(void);
BOOL	GetPacket(void);
BOOL	CheckPassword(char *);
word	SearchRxBuffer(char	*);
word	SendModemStr(char *);

