/***************************************************************************
	DOWNLDR.C - file handling the download funtions
***************************************************************************/

#include	"..\include\mctypes.h"
#include	"..\include\boot.h"
#include	"..\include\downldr.h"
#include	"..\include\quote.h"
#include	"..\include\utils.h"

#define	MAX_SEQ_NUM		128		/* number of PROM pages */

/* Error counters and timeout flags */
word	TotErrorCnt;		/* Cumulative error count */
byte	TimeoutOccured;
byte	GblTimeOut;			/* a global incremented in timeout PIT isr */
								/* when exceeds MAX val download is aborted */

word (*StateHandler[2])();	/* DownLoader state handlers */
byte	DownLdrState;
byte	CurSeqNum;
word	NextCodePage ;		/* offset of next code page to burn .*/
word	NumPktsSaved;		/* number of pkts saved in the Flash Buffer */

HeaderType	HeaderPkt;
SectorStatusType	SectStat;		/* stores info on sector being downloaded */

byte	ReceiverEnabled ;		/* To stop the buffer overwriting */

byte	ProcessBuffer[RCV_BUF_SIZE>>1];	/* unquoted receive buffer */
byte	TmpBuffer[RCV_BUF_SIZE>>1];		/* unquoted response buffer */
byte	ResponseBuffer[RCV_BUF_SIZE];		/* quoted response buffer */

/* the following is in a separate segment which is readonly sitting
just after the code */
byte	FlashPromBuf[64*1024];			/* buffer to hold downloads */
extern BootBinIOPBType	BootBinIOPB;
extern	volatile int x;

#pragma separate	SectorBaseAddress segment SectBaseSeg
dword SectorBaseAddress[] =		/* AMD 29F400AB flash page sizes table */
{
	0l,					/* PAGE_16K */
	16*1024l,			/* PAGE_8K */
	24*1024l,			/* PAGE_8K */
	32*1024l,			/* PAGE_32K */
	64*1024l,			/* PAGE_64K */
	128*1024l,			/* PAGE_64K */
	192*1024l,	 		/* PAGE_64K */
	256*1024l,			/* PAGE_64K */
	320*1024l,			/* PAGE_64K */
	384*1024l,			/* PAGE_64K */
	448*1024l			/* PAGE_64K */
};

extern void TransmitCnt (byte *, word) ;
/* extern WriteToFlash (char *, char *, word) ; */
extern void Execute (byte *) ;
extern CopyToSector(word *SectorAddress, word *Source, dword length);

extern void StartTimer (word timeout) ;
extern void StopTimer (void) ;
extern void ResetTimer (word timeout) ;

void SmallDelay (void) 
{
	int i = 512 ;
	while (i)
		i -- ;
}

void DownLoadNewInfo (void)
{
	word ret_val ;

	GlobalCRC = 0xffff;
	TotErrorCnt = 0 ;			/* reset the counters */
	GblTimeOut = 0 ;
	CurSeqNum = 0 ;
	NumPktsSaved = 0 ;
	NextCodePage = 0 ;
	DownLdrState = STATE_WT_HDR ;
/* kprint("Dnld: New dnld\n\r"); */

	StartTimer (TMOUT_VAL) ;
	while (GblTimeOut <= 3)
	{
		if (GetPacket())
		{
/* kprint("Dnld: Got pkt\n\r"); */
			StopTimer() ;				/* We got a packet so stop timer */
			GblTimeOut = 0 ;			/* packet has come,reset timeout counter */

			if (!CheckCRC (ProcessBuffer, (dword) ReceiveIndex, 0))
			{ 	/* Is the packet CRC OK */
kprint("Dnld: CRC error\n\r");
				if (ProcessErrorRcv() == STOP_DNLD)
				{	/* if not OK send NACK */
					return ;				/* if err count too much stop */
				}
				else
				{
					StartTimer (TMOUT_VAL) ;
					continue ;
				}
			}
			/* If the previous packet comes again, send an ACK since it has 
			already been processed */
			if (((DnLdPktHdrType *) ProcessBuffer)->SeqNum == PREV_SEQ_NUM(CurSeqNum))
			{
/* kprint("Dnld: Old pkt\n\r"); */
				CurSeqNum = PREV_SEQ_NUM(CurSeqNum);
				SendNACKorACK (PT_ACK) ;
			}
			else
			{
/* kprint("Dnld: to handler\n\r"); */
				ret_val = (*StateHandler[DownLdrState])() ;
				if (ret_val == STOP_DNLD)
					return ;
			}
			StartTimer (TMOUT_VAL) ;
		}
		else
			if (TimeoutOccured)
			{
				/* clear the receive buffer to receive a new packet */
				ReceiveState = RECV_7E ;
				SendNACKorACK (PT_NACK) ;
				ResetTimer (TMOUT_VAL) ;	  
			}
	}
	ResetTimer (TMOUT_VAL) ;
	while (!TimeoutOccured) ;
	StopTimer() ;
	if (GblTimeOut) 
    	AbortDownLoad (ERR_TMOUT) ;

	return ;
}

word ProcessErrorRcv(void)
{
	TotErrorCnt++;
	if (TotErrorCnt >= MAX_TOT_ERR)
	{
    	AbortDownLoad (ERR_MAX_ERR) ;
		return STOP_DNLD ;
	}
	SendNACKorACK (PT_NACK) ;
	return CONT_DNLD ;
}

dword GetSectorSizeOrBaseAddress(dword BaseAddress, int GetBaseAddress)
{
	int	index;
	int	MaximumSectorNumber;

	index = 1;
	if (BaseAddress >= (512*1024))		/* First non-addressable location */
		return 0xFFFFFFFF;					/* return invalid */
	MaximumSectorNumber = sizeof(SectorBaseAddress) /
														sizeof(SectorBaseAddress[0]);
	/* find the first baseaddress above the passed in address */
	for (;index < MaximumSectorNumber; index++)
		if (SectorBaseAddress[index] > BaseAddress)
			break;

	if (GetBaseAddress == GET_BASE_ADDR)
		return SectorBaseAddress[index-1];

	/* return the previous sector base address */
	if (index > 5)
		return (64l*1024);
	return (SectorBaseAddress[index] - SectorBaseAddress[index-1]);
}

void	SetSectorStatus(dword BaseAddress)
{

	SectStat.BaseAddress = BaseAddress;		/* absolute base address */
	SectStat.CopyOffset = 0;
	SectStat.Size = GetSectorSizeOrBaseAddress(BaseAddress, GET_SIZE);
}

/* This is state handler 0 */
word ProcessHeader (void)		
{

	DnLdHeadType *Header ;

/* kprint("Dnld: ProcessHeader\r\n"); */

	/* Has the proper packet come */
	switch (((DnLdPktHdrType *) ProcessBuffer)->PacketType)
	{
		case PT_HEADER :  
			if (((DnLdPktHdrType *) ProcessBuffer)->SeqNum == CurSeqNum)
			{
				/* Some of the fields come in the Header & rest in EOF struct */
				Header = (DnLdHeadType *) (ProcessBuffer + sizeof (DnLdPktHdrType)) ;
				if (Header->DnLdAddr < FL_BOOT_HDR)
				{
					AbortDownLoad (ERR_CODE_SAVE) ;
					return STOP_DNLD ;
				}
				memset((char *)&HeaderPkt, 0, sizeof(HeaderPkt));
				memcpy (HeaderPkt.Version, Header->Version, 10) ;
				memcpy (HeaderPkt.DateStamp, Header->DateStamp, 25) ;
				HeaderPkt.MagicNum = Header->MagicNum ;
				HeaderPkt.DnLdAddr = Header->DnLdAddr ;
				HeaderPkt.LoadAddr = Header->LoadAddr ;
				HeaderPkt.StartAddr = Header->StartAddr ;
				DownLdrState = STATE_WT_DATA ;
				SetSectorStatus(Header->DnLdAddr);

				SendNACKorACK (PT_ACK) ;
				return CONT_DNLD ;
			}
			break ;

		case PT_ABORT :
			if (((DnLdPktHdrType *) ProcessBuffer)->SeqNum == CurSeqNum)
			{
	     		StopDownLoad() ;
				return STOP_DNLD ;
			}
			break ;

		default :
  			AbortDownLoad (ERR_INV_PKT) ;
			return STOP_DNLD ;
	}
  	AbortDownLoad (ERR_SEQ_NUM) ;
	return STOP_DNLD ;
}

int	OverWriteFlashSector (byte *from, byte *to, dword len)
{

	CopyToSector((word *)to, (word *)from, len);

	SetSectorStatus((dword) (to + len));	/* for next sector */
	return 1;
}


int	CopyHeaderToFlash (byte *from, dword len, int codecnf)
{
	byte	*to;

	/* first read the header sector into flashprombuf */
	memcpy(FlashPromBuf, (byte *)FL_HDR_SECT_START, FL_HDR_SECT_SIZE);

	/* based on code header or config header fix destination address
	in FlashPromBuf to copy the new header*/
	if (codecnf == DNLD_CODE)
		to = FlashPromBuf + (FL_CODE_HDR - FL_HDR_SECT_START);
	else if (codecnf == DNLD_CFG)
		to = FlashPromBuf + (FL_CFG_HDR - FL_HDR_SECT_START);
	else
		return 0;		/* this is an error */

	/* copy the new header info */
	memcpy(to, from, len);

	/* replace the header sector with new one in FlashPromBuf */
	OverWriteFlashSector(FlashPromBuf, (byte *) FL_HDR_SECT_START,  FL_HDR_SECT_SIZE);
	return 1;
}

word ProcessData (void)
{
	byte *PktPtr ;
	word PktLen ;
	int i ;
/* byte tmpstr[10]; */

	switch (((DnLdPktHdrType *) ProcessBuffer)->PacketType)
	{
		case PT_DATA : 
			if (((DnLdPktHdrType *) ProcessBuffer)->SeqNum == CurSeqNum)
			{
				PktPtr = (byte *) (ProcessBuffer + sizeof (DnLdPktHdrType)) ;
				PktLen = ReceiveIndex - (word)sizeof(DnLdPktHdrType) - 2 ; /* 2 for CRC */
				
				UpdateGlobalCRC (PktPtr, (dword) PktLen) ;

				/* if the packetLength + total received is < sector size just
				copy the data in to FlashPromBuf, send ACK and return */
				if ((PktLen + SectStat.CopyOffset) < SectStat.Size)
				{
/*
kprint("Dnld: memcpy- Len=");
ultoh(tmpstr,PktLen);
kprint(tmpstr);
kprint(" Offset=");
ultoh(tmpstr,SectStat.CopyOffset);
kprint(tmpstr);
kprint(" Size=");
ultoh(tmpstr,SectStat.Size);
kprint(tmpstr);
kprint("\n\r");
*/
					memcpy ((byte *) (FlashPromBuf+SectStat.CopyOffset),
								PktPtr, PktLen) ;
					SectStat.CopyOffset += PktLen;

					/* wait here for some time and then send ACK */
					for (i = 0 ; i < 8 ; i ++)
						SmallDelay() ;
					SendNACKorACK (PT_ACK) ;
					return CONT_DNLD ;	
				}
				else if ((PktLen + SectStat.CopyOffset) != SectStat.Size)
				{
kprint("Dnld: sector size err\r\n");
					TransmitStr("Sector Size error.");
					StopDownLoad() ;
					return STOP_DNLD ;
				}
				memcpy ((byte *) (FlashPromBuf + SectStat.CopyOffset),
								PktPtr, PktLen) ;
				SectStat.CopyOffset += PktLen;
/*
kprint("SectWrite: From=");
ultoh(tmpstr,(dword)FlashPromBuf);
kprint(tmpstr);
kprint(" To=");
ultoh(tmpstr,(dword)SectStat.BaseAddress);
kprint(tmpstr);
kprint(" Count=");
ultoh(tmpstr,(dword)SectStat.Size);
kprint(tmpstr);
kprint("\n\r");
*/
				if (OverWriteFlashSector (FlashPromBuf,
								(byte *)SectStat.BaseAddress, SectStat.Size))
				{
					SendNACKorACK (PT_ACK) ;
					return CONT_DNLD ;
				}
				else
				{
   				AbortDownLoad (ERR_CODE_SAVE) ;
					return STOP_DNLD ;
				}		
			}
			break ;

		case PT_EOF :
			if (((DnLdPktHdrType *) ProcessBuffer)->SeqNum == CurSeqNum)
			{
				ProcessEOF() ;
				return STOP_DNLD ;
			}
			break ;

		case PT_ABORT :
			if (((DnLdPktHdrType *) ProcessBuffer)->SeqNum == CurSeqNum)
			{
				StopDownLoad() ;
				return STOP_DNLD ;
			}
			break ;

		default :
    		AbortDownLoad (ERR_INV_PKT) ;
			return STOP_DNLD ;
	}
   AbortDownLoad (ERR_SEQ_NUM) ;
	return STOP_DNLD ;
}

void ProcessEOF (void)
{
	DnLdEOFType	*EndOF ;

	EndOF = (DnLdEOFType *) (ProcessBuffer + sizeof (DnLdPktHdrType)) ;
	HeaderPkt.CodeLength = EndOF->CodeLength ;
	HeaderPkt.CRC = EndOF->CRC ;

	/* The received code CRC does not match what we calculated so don't save */
	GlobalCRC = updcrc ((EndOF->CRC & 0xFF00) >> 8, GlobalCRC) ;
	GlobalCRC = updcrc ((EndOF->CRC & 0x00FF), GlobalCRC) ;
	if (GlobalCRC)
	{
kprint("ProcEOF: down load CRC Fails\n\r");
	  	AbortDownLoad (ERR_DNLD) ;
		return ;
	}
	/* if some pkts are waiting to be written, write them */
	if (SectStat.CopyOffset)
		if (!OverWriteFlashSector (FlashPromBuf,
									(byte *)SectStat.BaseAddress, SectStat.CopyOffset))
			AbortDownLoad (ERR_CODE_SAVE) ;

	/* Burn the header and clear boot mode location */
	if (CopyHeaderToFlash ((byte *) &HeaderPkt, sizeof (HeaderPkt), DownloadType))
	{
		SendNACKorACK (PT_ACK) ;
		BootBinIOPB.BootMode = 0;
	}
	else
	  	AbortDownLoad (ERR_CODE_SAVE) ;
}

/* When an invalid packet comes from UI, target Aborts the download */
void AbortDownLoad (byte ErrVal)
{
	StopTimer() ;
	if (ErrVal)
		SendErrorAbort (ErrVal) ;
}

/* When an ABORT is inititated from the UI */
void StopDownLoad (void)
{
	StopTimer() ;
}

void SendNACKorACK (byte NackOrAck)
{
	word PktLen ;

	/* Form the packet. Calculate CRC. Quote and send */
	((DnLdPktHdrType *) TmpBuffer)->PacketType = NackOrAck ; /* PT_ACK or PT_NACK */
	((DnLdPktHdrType *) TmpBuffer)->SeqNum = CurSeqNum ;
	PktLen = sizeof (DnLdPktHdrType) ;
	AddCRC (TmpBuffer, (dword) PktLen) ;
	PktLen = Quote ((byte *) TmpBuffer, (byte *) ResponseBuffer + 1,
																				PktLen + 2) ;
	ResponseBuffer[0] = ResponseBuffer[PktLen+1] = FRAME_DELIM ;
	PktLen += 2 ;
	TransmitCnt (ResponseBuffer, PktLen) ;

	/* If ACK */
	/* Wait for next sequence num */
	/* else if NACK */
	/* wait for the same sequence num */
	if (NackOrAck == PT_ACK)
		INC_CIR_IDX (CurSeqNum, MAX_SEQ_NUM) ;
}

void SendErrorAbort (byte ErrVal)
{
	word PktLen ;

	/* Form the packet. Calculate CRC. Quote and send */
	((DnLdPktHdrType *) TmpBuffer)->PacketType = PT_ABORT ;
	((DnLdPktHdrType *) TmpBuffer)->SeqNum = CurSeqNum ;
	PktLen = sizeof (DnLdPktHdrType) ;
	*(TmpBuffer + PktLen) = ErrVal ;		/*  In Pkt Data */
	AddCRC(TmpBuffer, (dword) (PktLen + 1)) ;
	PktLen = Quote ((byte *) TmpBuffer, (byte *) ResponseBuffer + 1,
																				PktLen + 2 + 1) ;
	ResponseBuffer[0] = ResponseBuffer[PktLen+1] = FRAME_DELIM ;
	PktLen += 2 ;
	TransmitCnt (ResponseBuffer, PktLen) ;
	/* Wait for next sequence num */
	INC_CIR_IDX(CurSeqNum, MAX_SEQ_NUM) ;	
}

/***************************   Last  Line   ********************/
